#!/bin/bash
# Set MTU on network interfaces based on /rw/config/network-settings.json config file.
# The json config file should have the following syntax: 
# {
#     "Interfaces": {
#         "enp2s0": {
#             "MTU": 1400
#             ... (other interface settings)
#         },
#         "enp3s0": {
#             "MTU": 1500
#             ... (other interface settings)
#         },
#         ... (other interfaces)
#     }
#     ... (other global settings)
# }
config="/rw/config/network-settings.json"

# Test dependencies.
test -f $config || exit 1
hash jq || exit 1

# For each network interfaces listed in the json config file...
i=0
while jq -e ".Interfaces|keys[$i]" $config > /dev/null 2>&1; do

    # Note: the -r is important to remove quotes around the fields, if any
    ifname=$(jq -r ".Interfaces|keys[$i]" $config)
    # Case-insensitive query for 'MTU' key. We should find a way to do this for all queries to the json.
    mtu=$(jq -r ".Interfaces.$ifname | with_entries(select(.key | match(\"mtu\";\"i\"))) | to_entries[].value" $config)

    if [ "$ifname" != "null" ] && [ -n "$ifname" ] && [ "$mtu" != "null" ] && [ -n "$mtu" ]; then

        if find /sys/class/net/$ifname -type l > /dev/null 2>&1; then

            echo "Setting MTU $mtu on network interface $ifname..."

            ifconfig $ifname mtu $mtu
    
            if [ "$?" != "0" ]; then
                echo "MTU setting failed."
            fi

        else
            echo "Network interface $ifname not found."
        fi
    fi

    let i++
done 

exit 0

