#!/bin/sh
# This script configures the network bonding driver that has been loaded beforehand by modprobe:
#   1) Detect the presence of the bonding kernel driver and the bond0 interface
#   2) Detect the presence of two network interfaces whose names start with "enp".
#   3) Disable ethernet through connman before enslaving the devices 
#   4) Configure the bonding driver to use active-backup MII-monitoring mode
#   5) Enslave the two enp* interfaces to the bonding interface 
#   6) Re-enable ethernet through connman.

# Validate that the bonding driver is loaded.
/bin/lsmod | grep bonding > /dev/null 2>&1
driver_ret=$?

/bin/ls /sys/class/net/bond0 > /dev/null 2>&1
interface_ret=$?

if [ "$driver_ret" != "0" ] || [ "$interface_ret" != "0" ]; then
    echo "Error: the network bonding driver is not loaded or the bonding interface does not exist."
    exit 1
fi

# Find ethernet interfaces
read -d '\n' -r -a interfaces <<< $(ls -l /sys/class/net/ | grep enp | awk '{ print $9 }')
if_count=${#interfaces[@]}

if [ "$if_count" == "2" ]; then
    echo "Found ethernet interfaces: ${interfaces[@]}"
else
    echo "Error: did not find two ethernet interfaces, found $if_count."
    exit 1
fi

/usr/bin/connmanctl disable ethernet
ret=$?

if [ $ret != 0 ]; then
    echo "Error: can't disable ethernet: $ret"
    exit $ret
fi

# Configure the driver
echo "Set active-backup mode..."
echo "active-backup" > /sys/class/net/bond0/bonding/mode
mode_ret=$?

echo "Set ${interfaces[0]} as primary interface..."
echo "${interfaces[0]}" > /sys/class/net/bond0/bonding/primary
primary_ret=$?

echo "Set MII monitoring interval to 1000 ms..."
echo "1000" > /sys/class/net/bond0/bonding/miimon
miimon_ret=$?

if [ $mode_ret != 0 ] || [ $primary_ret != 0 ] || [ $miimon_ret != 0 ]; then
    echo "Error: failed to configure the driver."
    exit 1
fi

# Reorder network interfaces according to settings before enslaving. The MAC address of the first 
# enslaved interface is used by the bonding interface.
firstslave=$(cat /rw/config/enable_network_bonding | grep "interface=" | cut -d '=' -f 2)
secondslave=""

if [ -n "$firstslave" ]; then

    # /sys/class/net contains symbolic links, tested with -h
    if [ -h "/sys/class/net/$firstslave" ]; then

        for if in ${interfaces[@]}; do
            if [ "$if" != "$firstslave" ]; then
                secondslave="$if"
            fi
        done

        interfaces[0]="$firstslave"
        interfaces[1]="$secondslave"

    else
        echo "Error: slave \"$firstslave\" does not exist, using default slave MAC address."
    fi
fi

# Enslave both interfaces to the bonding interface
for if in ${interfaces[@]}; do

    echo "Enslaving interface $if..."
    echo "+$if" > /sys/class/net/bond0/bonding/slaves
    ret=$?
    
    if [ $ret != 0 ]; then
        echo "Error: failed to enslave interface $if: return code $ret"
        break
    fi

done

# Re-enable ethernet
if [ $ret == 0 ]; then

    /usr/bin/connmanctl enable ethernet
    ret=$?

    if [ $ret == 0 ]; then
        echo "Ethernet re-enabled."
    else
        echo "Error: failed to re-enable ethernet: return code $ret"
    fi
fi

exit $ret

