/******************************************************************************\

Module Name:    SampleNetStreamer.c

Description:    LNetStreamer sample application.

References:

BSD 2-Clause License

Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this 
   list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation 
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\******************************************************************************/

/* Detect operating system */
#if defined(WIN32) || defined(_WIN32) || defined(__WIN32) || defined(__WIN32__)
#define LNETSTREAMER_CONFIG_OS_WIN32
#else /* #if defined(WIN32) */
#define LNETSTREAMER_CONFIG_OS_UNIX
#endif /* #if defined(WIN32) */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#if defined(LNETSTREAMER_CONFIG_OS_UNIX)

#include <unistd.h>
#include <fcntl.h>
#include <termios.h>
#include <pthread.h>
#include <sys/time.h>

#else

#include <windows.h>
#include <pthread.h>

#endif

#include "LNetStreamer.h"

#define FILE_SINK_PACKET_TYPE_VIDEO_FRAME       ((MUINT8)(0x00u))
#define FILE_SINK_PACKET_TYPE_AUDIO_FRAME       ((MUINT8)(0x01u))

#define DUMP_STATISTICS_PERIOD                  1000u

struct SampleNetStreamerOptions;

struct SampleNetStreamerOptionsList;

struct SampleNetStreamerThreadContext
{
    struct SampleNetStreamerOptions*            poSampleOptions;
    FILE*                                       fpMedia;
    LNetStreamer_MediaType                      eMediaType;
};

struct SampleNetStreamerFileSink
{
    struct SampleNetStreamerOptions*            poSampleOptions;
    pthread_mutex_t                             oLock;
    FILE*                                       fpMedia;
    MUINT8*                                     paucCacheData;
    MUINT32                                     uiCacheTimestamp;
    MUINT32                                     uiCacheDataLength;
    MUINT32 volatile                            uiFirstTimestamp;
    MBOOL8 volatile                             bFirstTimestamp;
};

struct SampleNetStreamerOptions
{
    MUINT64                                     uiVideoBytesPeriod;
    MUINT64                                     uiAudioBytesPeriod;
    char*                                       pacLocation;
    char*                                       pacDestination;
    char*                                       pacRtspFolder;
    char*                                       pacNetInterface;
    char*                                       pacServerMediaFile;
    char*                                       pacClientMediaFile;
    char*                                       pacSdp;
    struct SampleNetStreamerOptionsList*        poSampleOptionsList;
    LNetStreamer_Handle                         hSession;
    pthread_t                                   oVideoThread;
    pthread_t                                   oAudioThread;
    pthread_mutex_t                             oSessionMutex;
    struct SampleNetStreamerThreadContext       oVideoContext;
    struct SampleNetStreamerThreadContext       oAudioContext;
    struct SampleNetStreamerFileSink            oFileSink;
    unsigned long int                           uiVideoPort;
    unsigned long int                           uiAudioPort;
    unsigned long int                           uiRtspPort;
    unsigned long int                           uiRtcpFrequency;
    MUINT32                                     uiReferenceTimeMsec;
    MUINT                                       uiSessionIndex;
    LNetStreamer_Service                        eService;
    LNetStreamer_Protocol                       eProtocol;
    int                                         iRtspServer;
    int                                         iRtmpServer;
    MBOOL8                                      bEnableVideo;
    MBOOL8                                      bEnableAudio;
    MBOOL8                                      bDisableFrameSkipDetection;
    MBOOL8                                      bQuerySessionConfig;
    MBOOL8 volatile                             bWaitForAudio;
    MBOOL8 volatile                             bWaitForVideo;
    MBOOL8                                      bValidSession;
    MUINT8                                      aucAudioHeader[2u];
    char                                        acRtspFolderDefault[16u];
    MBOOL8 volatile                             bContinueStreaming;
    MBOOL8 volatile                             bReferenceTimeDone;
};

struct SampleNetStreamerOptionsList
{
    struct SampleNetStreamerOptions*            paoSampleOptions;
    pthread_t                                   oStatisticsThread;
    MUINT                                       uiSessionMaxCount;
    MUINT                                       uiCurrentSessionIndex;
    MBOOL8 volatile                             bContinueStatThread;
};

static void SampleNetStreamer_usleep(unsigned long int usec)
{
#if defined(LNETSTREAMER_CONFIG_OS_UNIX)
    usleep((useconds_t)(usec));
#else
    Sleep(usec/1000u);
#endif
}

static MUINT32 SampleNetStreamer_GetTimeMsec(void)
{
#if defined(LNETSTREAMER_CONFIG_OS_UNIX)
    struct timespec                             oTime;

    clock_gettime(CLOCK_MONOTONIC, &oTime);

    return (MUINT32)(oTime.tv_sec * 1000 + (oTime.tv_nsec / 1000 / 1000));
#else
    /* Note: could also use GetTickCount() or GetSystemTimeAsFileTime() */
    return (MUINT32)(timeGetTime());
#endif
}

static void SampleNetStreamer_InitFileSink(
    struct SampleNetStreamerFileSink*           poFileSink,
    struct SampleNetStreamerOptions*            poSampleOptions)
{
    poFileSink->poSampleOptions = poSampleOptions;

    pthread_mutex_init(&poFileSink->oLock, NULL);

    if (poSampleOptions->pacClientMediaFile)
    {
        poFileSink->fpMedia = fopen(poSampleOptions->pacClientMediaFile, "wb");

        poFileSink->paucCacheData = malloc(3ul * 1024ul * 1024ul);
    }
}

static void SampleNetStreamer_BuildFileSinkPacketHeader(
    MUINT8*                                     aucPacketHeader,
    MUINT8                                      uiPacketType,
    MUINT32                                     uiDataLength,
    MUINT32                                     uiTimestampPacket)
{
    aucPacketHeader[0u] = uiPacketType;

    aucPacketHeader[1u] = (MUINT8)((uiDataLength >> 0) & 0xFFu);

    aucPacketHeader[2u] = (MUINT8)((uiDataLength >> 8) & 0xFFu);

    aucPacketHeader[3u] = (MUINT8)((uiDataLength >> 16) & 0xFFu);

    aucPacketHeader[4u] = (MUINT8)((uiTimestampPacket >> 0) & 0xFFu);

    aucPacketHeader[5u] = (MUINT8)((uiTimestampPacket >> 8) & 0xFFu);

    aucPacketHeader[6u] = (MUINT8)((uiTimestampPacket >> 16) & 0xFFu);

    aucPacketHeader[7u] = (MUINT8)((uiTimestampPacket >> 24) & 0xFFu);
}

static void SampleNetStreamer_WriteFileSinkPacket(
    struct SampleNetStreamerFileSink*           poFileSink,
    MUINT8                                      uiPacketType,
    MUINT8*                                     paucData,
    MUINT32                                     uiDataLength,
    MUINT32                                     uiTimestamp)
{
    MUINT32                                     uiTimestampPacket;
    MUINT8                                      aucPacketHeader[8u];

    if (poFileSink->fpMedia)
    {
        pthread_mutex_lock(&poFileSink->oLock);

        if (poFileSink->fpMedia)
        {
            uiTimestampPacket = ((uiTimestamp - poFileSink->uiFirstTimestamp) * 1000u);

            SampleNetStreamer_BuildFileSinkPacketHeader(
                aucPacketHeader,
                uiPacketType,
                uiDataLength,
                uiTimestampPacket);

            fwrite(aucPacketHeader, 8, 1, poFileSink->fpMedia);

            fwrite(paucData, uiDataLength, 1, poFileSink->fpMedia);
        }

        pthread_mutex_unlock(&poFileSink->oLock);
    }
}

static void SampleNetStreamer_FlushFileSinkVideoCache(
    struct SampleNetStreamerFileSink*           poFileSink)
{
    if (poFileSink->fpMedia && poFileSink->uiCacheDataLength)
    {
        SampleNetStreamer_WriteFileSinkPacket(
            poFileSink,
            FILE_SINK_PACKET_TYPE_VIDEO_FRAME,
            poFileSink->paucCacheData,
            poFileSink->uiCacheDataLength,
            poFileSink->uiCacheTimestamp);

        poFileSink->uiCacheDataLength = 0u;
    }
}

static void SampleNetStreamer_WriteFileSink(
    struct SampleNetStreamerFileSink*           poFileSink,
    LNetStreamer_MediaType                      eMediaType,
    MUINT8*                                     paucData,
    MUINT32                                     uiDataLength,
    MUINT32                                     uiTimestamp)
{
    MUINT8*                                     paucNaluType;

    if (poFileSink->fpMedia)
    {
        if (!poFileSink->bFirstTimestamp)
        {
            poFileSink->uiFirstTimestamp = uiTimestamp;

            poFileSink->bFirstTimestamp = MTRUE;
        }

        if ((MUINT32)(uiTimestamp - poFileSink->uiFirstTimestamp) < (MUINT32)(0xFF000000ul))
        {
            // Save data to a file...
            if (LNetStreamer_MediaType_VIDEO == eMediaType)
            {
                // copy all data into cache
                memcpy(poFileSink->paucCacheData + poFileSink->uiCacheDataLength, paucData, uiDataLength);
                poFileSink->uiCacheDataLength += uiDataLength;
                poFileSink->uiCacheTimestamp = uiTimestamp;

                // Detect the slice type
                paucNaluType = paucData;
                while (*paucNaluType == 0x00u)
                {
                    paucNaluType ++;
                }
                if (*paucNaluType == 0x01u)
                {
                    paucNaluType ++;
                }
                if (((*paucNaluType & 0x1Fu) != 0x07u) && ((*paucNaluType & 0x1Fu) != 0x08u))
                {
                    SampleNetStreamer_FlushFileSinkVideoCache(poFileSink);
                }
            }
            else
            {
                SampleNetStreamer_WriteFileSinkPacket(
                    poFileSink,
                    (MUINT8)((LNetStreamer_MediaType_VIDEO == eMediaType) ?
                        FILE_SINK_PACKET_TYPE_VIDEO_FRAME :
                        FILE_SINK_PACKET_TYPE_AUDIO_FRAME),
                    paucData,
                    uiDataLength,
                    uiTimestamp);
            }

            // callback server sessions that are linked
        }
    }
}

static void SampleNetStreamer_CleanupFileSink(
    struct SampleNetStreamerFileSink*           poFileSink)
{
    if (poFileSink->fpMedia)
    {
        SampleNetStreamer_FlushFileSinkVideoCache(poFileSink);

        free(poFileSink->paucCacheData);

        poFileSink->paucCacheData = MNULL;

        if (poFileSink->fpMedia)
        {
            fclose(poFileSink->fpMedia);

            poFileSink->fpMedia = MNULL;
        }
    }
}

static LNetStreamer_Config* SampleNetStreamer_GetConfig(
    LNetStreamer_Handle                         hSession)
{
    LStatus                                     eLiberatusStatus;
    MUINT32                                     uiSessionConfigLength;
    LNetStreamer_ConfigType*                    pvSessionConfigBuffer;
    LNetStreamer_Config*                        poSessionConfigV1;

    poSessionConfigV1 = (LNetStreamer_Config*)(0);
    uiSessionConfigLength = (MUINT32)(12345u);
    eLiberatusStatus = LNetStreamer_GetConfigLength(hSession,
        LNetStreamer_ConfigType_STANDARD, &uiSessionConfigLength);
    if (LSTATUS_IS_SUCCESS(eLiberatusStatus))
    {
        pvSessionConfigBuffer = (LNetStreamer_ConfigType*)(malloc(
                uiSessionConfigLength));
        if (pvSessionConfigBuffer)
        {
            *pvSessionConfigBuffer = LNetStreamer_ConfigType_STANDARD;
            eLiberatusStatus = LNetStreamer_GetConfig(hSession,
                pvSessionConfigBuffer, uiSessionConfigLength);
            if (LSTATUS_IS_SUCCESS(eLiberatusStatus))
            {
                poSessionConfigV1 = (LNetStreamer_Config*)pvSessionConfigBuffer;
            }
        }
    }

    return poSessionConfigV1;
}

static void SampleNetStreamer_FreeConfig(
    LNetStreamer_Config*                        poSessionConfigV1)
{
    free((void*)(poSessionConfigV1));
}

static LStatus SampleNetStreamer_QueryConfig(
    struct SampleNetStreamerOptions*            poSampleOptions)
{
    LStatus                                     eLiberatusStatus;
    LNetStreamer_Config*                        poSessionConfigV1;

    if (poSampleOptions->bValidSession && poSampleOptions->hSession)
    {
        poSessionConfigV1 = SampleNetStreamer_GetConfig(poSampleOptions->hSession);
        if (poSessionConfigV1)
        {
            if (LNetStreamer_ConfigFlags_LOCATION & poSessionConfigV1->uiConfigFlags)
            {
                printf("%s %3u: URL %s\n",
                    (LNetStreamer_Service_SERVER == poSampleOptions->eService) ? "server" : "client",
                    poSampleOptions->uiSessionIndex,
                    poSessionConfigV1->pacLocation);
            }

            if ((LNetStreamer_Protocol_RTP == poSampleOptions->eProtocol) ||
                (LNetStreamer_Protocol_TS == poSampleOptions->eProtocol))
            {
                if (LNetStreamer_ConfigFlags_SDP & poSessionConfigV1->uiConfigFlags)
                {
                    printf("%s %3u: SDP vvv\n%sSDP ^^^\n",
                        (LNetStreamer_Service_SERVER == poSampleOptions->eService) ? "server" : "client",
                        poSampleOptions->uiSessionIndex,
                        poSessionConfigV1->pacSdp);
                }
            }

            eLiberatusStatus = LStatus_OK;

            SampleNetStreamer_FreeConfig(poSessionConfigV1);
        }
        else
        {
            eLiberatusStatus = LStatus_FAIL;
        }
    }
    else
    {
        eLiberatusStatus = LStatus_OK;
    }

    return eLiberatusStatus;
}

static void SampleNetStreamer_AccumulateStatistics(
    struct SampleNetStreamerThreadContext*      poGenericThreadContext,
    LNetStreamer_MediaType                      eMediaType,
    MUINT32                                     uiLength)
{
    pthread_mutex_lock(&poGenericThreadContext->poSampleOptions->oSessionMutex);

    if (LNetStreamer_MediaType_VIDEO == eMediaType)
    {
        poGenericThreadContext->poSampleOptions->uiVideoBytesPeriod += uiLength;
    }

    if (LNetStreamer_MediaType_AUDIO == eMediaType)
    {
        poGenericThreadContext->poSampleOptions->uiAudioBytesPeriod += uiLength;
    }

    pthread_mutex_unlock(&poGenericThreadContext->poSampleOptions->oSessionMutex);
}

static void* SampleNetStreamer_ClientThread(
    void*                                       pvClientThreadContext)
{
    LStatus                                     eLiberatusStatus;
    LNetStreamer_Media                          oBufferInfo;
    struct SampleNetStreamerThreadContext*      poClientThreadContext;

    poClientThreadContext = (struct SampleNetStreamerThreadContext*)(pvClientThreadContext);

    poClientThreadContext->poSampleOptions->bWaitForAudio =
        poClientThreadContext->poSampleOptions->bEnableAudio;

    poClientThreadContext->poSampleOptions->bWaitForVideo =
        poClientThreadContext->poSampleOptions->bEnableVideo;

    poClientThreadContext->poSampleOptions->bQuerySessionConfig = MTRUE;

    while (poClientThreadContext->poSampleOptions->bContinueStreaming)
    {
        oBufferInfo.eMediaType = poClientThreadContext->eMediaType;
        eLiberatusStatus = LNetStreamer_GetMedia(poClientThreadContext->poSampleOptions->hSession,
            &oBufferInfo.eMediaType);
        if (LSTATUS_IS_SUCCESS(eLiberatusStatus))
        {
            SampleNetStreamer_AccumulateStatistics(
                poClientThreadContext,
                oBufferInfo.eMediaType,
                oBufferInfo.uiDataLength);

            if (LNetStreamer_MediaType_VIDEO == oBufferInfo.eMediaType)
            {
                poClientThreadContext->poSampleOptions->bWaitForVideo = MFALSE;
            }

            if (LNetStreamer_MediaType_AUDIO == oBufferInfo.eMediaType)
            {
                poClientThreadContext->poSampleOptions->bWaitForAudio = MFALSE;
            }

            /* Get headers and store into file sink */
            if (!poClientThreadContext->poSampleOptions->bWaitForAudio &&
                !poClientThreadContext->poSampleOptions->bWaitForVideo)
            {
                SampleNetStreamer_WriteFileSink(
                    &poClientThreadContext->poSampleOptions->oFileSink,
                    poClientThreadContext->eMediaType,
                    oBufferInfo.paucData,
                    oBufferInfo.uiDataLength,
                    oBufferInfo.uiPTS);
            }

            eLiberatusStatus = LNetStreamer_ReleaseMedia(poClientThreadContext->poSampleOptions->hSession,
                &oBufferInfo.eMediaType);
        }

    }

    return MNULL;
}

static void* SampleNetStreamer_ServerThread(
    void*                                       pvServerThreadContext)
{
    struct SampleNetStreamerThreadContext*      poServerThreadContext;
    LNetStreamer_Media                          oBufferInfo;
    MUINT32                                     uiLength;
    MUINT32                                     uiTimestampFirst;
    MUINT32                                     uiTimestampNext;
    MUINT32                                     uiTimestampPacket;
    MUINT32                                     uiCurrentTimeMsec;
    LStatus                                     eLiberatusStatus;
    LStatus                                     eTempLiberatusStatus;
    MUINT8                                      ucType;
    MUINT8                                      aucPacketHeader[8u];
    MBOOL8                                      bEndOfFile;
    MBOOL8                                      bUseFrame;
    MBOOL8                                      bReleaseMedia;

    poServerThreadContext = (struct SampleNetStreamerThreadContext*)(pvServerThreadContext);

    eLiberatusStatus = LStatus_OK;

    uiTimestampFirst = 0u;
    uiTimestampNext = 0u;

    while (LSTATUS_IS_SUCCESS(eLiberatusStatus) &&
        poServerThreadContext->poSampleOptions->bContinueStreaming)
    {
        uiTimestampFirst += uiTimestampNext;
        uiTimestampNext = 0ul;

        fseek(poServerThreadContext->fpMedia, 0l, SEEK_SET);
        bEndOfFile = MFALSE;
        while (!bEndOfFile &&
            poServerThreadContext->poSampleOptions->bContinueStreaming &&
            (1 == fread(aucPacketHeader, sizeof(aucPacketHeader), 1, poServerThreadContext->fpMedia)))
        {
            ucType = aucPacketHeader[0u];

            uiLength = (MUINT32)((
                    (MUINT32)(aucPacketHeader[1u]) << 0u)
                    | ((MUINT32)(aucPacketHeader[2u]) << 8u)
                    | ((MUINT32)(aucPacketHeader[3u]) << 16u));

            if (2u > ucType)
            {
                uiTimestampPacket = (MUINT32)(
                    ((MUINT32)(aucPacketHeader[4u]) << 0u)
                    | ((MUINT32)(aucPacketHeader[5u]) << 8u)
                    | ((MUINT32)(aucPacketHeader[6u]) << 16u)
                    | ((MUINT32)(aucPacketHeader[7u]) << 24u));

                uiTimestampPacket = (MUINT32)(uiTimestampPacket / 1000ul);

                if ((uiTimestampPacket + 33u) > uiTimestampNext)
                {
                    uiTimestampNext = (uiTimestampPacket + 33u);
                }
            }
            else
            {
                uiTimestampPacket = 0u;
            }

            uiTimestampPacket = (MUINT32)(uiTimestampFirst + uiTimestampPacket);

            bUseFrame = MFALSE;
            if ((0x00u == ucType) && (poServerThreadContext->eMediaType == LNetStreamer_MediaType_VIDEO))
            {
                bUseFrame = MTRUE;
            }
            else if ((0x01u == ucType) && (poServerThreadContext->eMediaType == LNetStreamer_MediaType_AUDIO))
            {
                bUseFrame = MTRUE;
            }

            if (bUseFrame)
            {
                uiCurrentTimeMsec = SampleNetStreamer_GetTimeMsec();
                if ((uiCurrentTimeMsec - poServerThreadContext->poSampleOptions->uiReferenceTimeMsec) < uiTimestampPacket)
                {
                    SampleNetStreamer_usleep(
                        (uiTimestampPacket - (uiCurrentTimeMsec - poServerThreadContext->poSampleOptions->uiReferenceTimeMsec)) * 1000);
                }

                bReleaseMedia = MFALSE;
                do
                {
                    memset(&oBufferInfo, 0x00u, sizeof(oBufferInfo));
                    oBufferInfo.eMediaType = poServerThreadContext->eMediaType;
                    eLiberatusStatus = LNetStreamer_GetMedia(
                        poServerThreadContext->poSampleOptions->hSession, &oBufferInfo.eMediaType);
                    if (LSTATUS_IS_SUCCESS(eLiberatusStatus))
                    {
                        bReleaseMedia = MTRUE;
                    }
                    else
                    {
                        SampleNetStreamer_usleep(100ul);
                    }
                }
                while (LSTATUS_IS_FAIL(eLiberatusStatus) && poServerThreadContext->poSampleOptions->bContinueStreaming);

                if ((uiLength <= oBufferInfo.uiDataMaxLength) &&
                    (1 == fread(oBufferInfo.paucData, uiLength, 1, poServerThreadContext->fpMedia)))
                {
                    oBufferInfo.uiDataLength = uiLength;
                    oBufferInfo.uiPTS = (uiTimestampPacket - poServerThreadContext->poSampleOptions->uiReferenceTimeMsec);
                    oBufferInfo.uiDTS = oBufferInfo.uiPTS;

                    SampleNetStreamer_AccumulateStatistics(
                        poServerThreadContext,
                        oBufferInfo.eMediaType,
                        uiLength);
                }
                else
                {
                    bEndOfFile = MTRUE;
                }

                if (bReleaseMedia)
                {
                    bReleaseMedia = MFALSE;

                    eLiberatusStatus = LNetStreamer_ReleaseMedia(
                        poServerThreadContext->poSampleOptions->hSession, &oBufferInfo.eMediaType);
                }

                if (LStatus_OK == eLiberatusStatus)
                {
                    if (poServerThreadContext->poSampleOptions->bQuerySessionConfig)
                    {
                        pthread_mutex_lock(&poServerThreadContext->poSampleOptions->oSessionMutex);

                        if (poServerThreadContext->poSampleOptions->bQuerySessionConfig)
                        {
                            eTempLiberatusStatus = SampleNetStreamer_QueryConfig(
                                poServerThreadContext->poSampleOptions);

                            if (LSTATUS_IS_SUCCESS(eTempLiberatusStatus))
                            {
                                poServerThreadContext->poSampleOptions->bQuerySessionConfig = 0;
                            }
                        }

                        pthread_mutex_unlock(&poServerThreadContext->poSampleOptions->oSessionMutex);
                    }
                }
            }
            else
            {
                fseek(poServerThreadContext->fpMedia, (signed long int)(uiLength), SEEK_CUR);
            }
        }
    }

    return MNULL;
}

static LStatus SampleNetStreamer_EventCallback(
    void*                                       pvEventContext,
    LNetStreamer_Handle                         hNetStreamer,
    LNetStreamer_EventType*                     peEventType)
{
    struct SampleNetStreamerOptions*            poSampleOptions;

    (void)(hNetStreamer);

    poSampleOptions = (struct SampleNetStreamerOptions*)(pvEventContext);

    printf("%s %3u: %s event\n",
        (LNetStreamer_Service_SERVER == poSampleOptions->eService) ? "server" : "client",
        (unsigned int)(poSampleOptions->uiSessionIndex),
        ((*peEventType) == LNetStreamer_EventType_START) ? "START" :
        ((*peEventType) == LNetStreamer_EventType_STOP) ? " STOP" : " SKIP");

    return LStatus_OK;
}

static void SampleNetStreamer_ConvertOptions(
    struct SampleNetStreamerOptions*            poSampleOptions,
    LNetStreamer_Config*                        poSessionConfigV1)
{
    memset(poSessionConfigV1, 0x00u, sizeof(*poSessionConfigV1));
    poSessionConfigV1->eConfigType = LNetStreamer_ConfigType_STANDARD;
    poSessionConfigV1->eService = poSampleOptions->eService;
    poSessionConfigV1->eProtocol = poSampleOptions->eProtocol;
    poSessionConfigV1->bEnableVideo = poSampleOptions->bEnableVideo;
    poSessionConfigV1->bEnableAudio = poSampleOptions->bEnableAudio;
    if (poSampleOptions->pacLocation)
    {
        poSessionConfigV1->uiConfigFlags |= LNetStreamer_ConfigFlags_LOCATION;
        poSessionConfigV1->pacLocation = poSampleOptions->pacLocation;
        poSessionConfigV1->uiLocationLength = (MUINT32)(strlen(poSessionConfigV1->pacLocation));
    }
    if (poSampleOptions->pacSdp)
    {
        poSessionConfigV1->uiConfigFlags |= LNetStreamer_ConfigFlags_SDP;
        poSessionConfigV1->pacSdp = poSampleOptions->pacSdp;
        poSessionConfigV1->uiSdpLength = (MUINT32)(strlen(poSessionConfigV1->pacSdp));
    }
    if (poSampleOptions->pacDestination)
    {
        poSessionConfigV1->uiConfigFlags |= LNetStreamer_ConfigFlags_DESTINATION;
        poSessionConfigV1->pacDestination = poSampleOptions->pacDestination;
        poSessionConfigV1->uiDestinationLength = (MUINT32)(strlen(poSessionConfigV1->pacDestination));
    }
    if (poSampleOptions->pacRtspFolder)
    {
        poSessionConfigV1->uiConfigFlags |= LNetStreamer_ConfigFlags_RTSP_FOLDER;
        poSessionConfigV1->pacRtspFolder = poSampleOptions->pacRtspFolder;
        poSessionConfigV1->uiRtspFolderLength = (MUINT32)(strlen(poSessionConfigV1->pacRtspFolder));
    }
    else
    {
        sprintf(poSampleOptions->acRtspFolderDefault, "S%d", poSampleOptions->uiSessionIndex);
        poSessionConfigV1->uiConfigFlags |= LNetStreamer_ConfigFlags_RTSP_FOLDER;
        poSessionConfigV1->pacRtspFolder = poSampleOptions->acRtspFolderDefault;
        poSessionConfigV1->uiRtspFolderLength = (MUINT32)(strlen(poSessionConfigV1->pacRtspFolder));
    }
    if (poSampleOptions->pacNetInterface)
    {
        poSessionConfigV1->uiConfigFlags |= LNetStreamer_ConfigFlags_NET_INTERFACE;
        poSessionConfigV1->pacNetInterface = poSampleOptions->pacNetInterface;
        poSessionConfigV1->uiNetInterfaceLength = (MUINT32)(strlen(poSessionConfigV1->pacNetInterface));
    }
    if (poSampleOptions->uiVideoPort)
    {
        poSessionConfigV1->uiConfigFlags |= LNetStreamer_ConfigFlags_VIDEO_PORT;
        poSessionConfigV1->uiVideoPort = (MUINT32)(poSampleOptions->uiVideoPort);
    }
    else
    {
        if (LNetStreamer_Service_CLIENT == poSampleOptions->eService)
        {
            poSessionConfigV1->uiConfigFlags |= LNetStreamer_ConfigFlags_VIDEO_PORT;
            poSessionConfigV1->uiVideoPort = 13000 + (poSampleOptions->uiSessionIndex * 8u);
        }
        else
        {
            poSessionConfigV1->uiConfigFlags |= LNetStreamer_ConfigFlags_VIDEO_PORT;
            poSessionConfigV1->uiVideoPort = 15000 + (poSampleOptions->uiSessionIndex * 8u);
        }
    }
    if (poSampleOptions->uiAudioPort)
    {
        poSessionConfigV1->uiConfigFlags |= LNetStreamer_ConfigFlags_AUDIO_PORT;
        poSessionConfigV1->uiAudioPort = (MUINT32)(poSampleOptions->uiAudioPort);
    }
    else
    {
        if (LNetStreamer_Service_CLIENT == poSampleOptions->eService)
        {
            poSessionConfigV1->uiConfigFlags |= LNetStreamer_ConfigFlags_AUDIO_PORT;
            poSessionConfigV1->uiAudioPort = 13004 + (poSampleOptions->uiSessionIndex * 8u);
        }
        else
        {
            poSessionConfigV1->uiConfigFlags |= LNetStreamer_ConfigFlags_AUDIO_PORT;
            poSessionConfigV1->uiAudioPort = 15004 + (poSampleOptions->uiSessionIndex * 8u);
        }
    }
    if (poSampleOptions->uiRtspPort)
    {
        poSessionConfigV1->uiConfigFlags |= LNetStreamer_ConfigFlags_RTSP_PORT;
        poSessionConfigV1->uiRtspPort = (MUINT32)(poSampleOptions->uiRtspPort);
    }

    poSessionConfigV1->uiVideoBufferCount = LNETSTREAMER_VIDEO_BUFFER_COUNT_DEFAULT;
    poSessionConfigV1->uiVideoBufferLength = 3ul * 1024ul * 1024ul;

    poSessionConfigV1->uiConfigFlags |= LNetStreamer_ConfigFlags_EVENT;
    poSessionConfigV1->pvEventContext = poSampleOptions;
    poSessionConfigV1->pfnEventCallback = &SampleNetStreamer_EventCallback;

    if (poSampleOptions->iRtspServer >= 0)
    {
        poSessionConfigV1->uiConfigFlags |= LNetStreamer_ConfigFlags_RTSP_SERVER;
        poSessionConfigV1->bEnableRtspServer = (poSampleOptions->iRtspServer > 0);
    }

    if (poSampleOptions->iRtmpServer >= 0)
    {
        poSessionConfigV1->uiConfigFlags |= LNetStreamer_ConfigFlags_RTMP_SERVER;
        poSessionConfigV1->bEnableRtmpServer = (poSampleOptions->iRtmpServer > 0);
    }

    if (poSampleOptions->uiRtcpFrequency)
    {
        poSessionConfigV1->uiConfigFlags |= LNetStreamer_ConfigFlags_RTCP_FREQUENCY;
        poSessionConfigV1->uiRtcpFrequency = (MUINT32)(poSampleOptions->uiRtcpFrequency);
    }

    if (poSampleOptions->bDisableFrameSkipDetection)
    {
        poSessionConfigV1->uiConfigFlags |= LNetStreamer_ConfigFlags_NO_SKIP;
    }
}

static void* SampleNetStreamer_GenericThread(
    void*                                       pvGenericThreadContext)
{
    struct SampleNetStreamerThreadContext*      poGenericThreadContext;
    void*                                       pvGenericThreadResult;

    poGenericThreadContext = (struct SampleNetStreamerThreadContext*)(pvGenericThreadContext);

    if (LNetStreamer_Service_CLIENT == poGenericThreadContext->poSampleOptions->eService)
    {
        pvGenericThreadResult = SampleNetStreamer_ClientThread(pvGenericThreadContext);
    }
    else if (LNetStreamer_Service_SERVER == poGenericThreadContext->poSampleOptions->eService)
    {
        pvGenericThreadResult = SampleNetStreamer_ServerThread(pvGenericThreadContext);
    }
    else
    {
        pvGenericThreadResult = MNULL;
    }

    return pvGenericThreadResult;
}

static LStatus SampleNetStreamer_StartSession(
    struct SampleNetStreamerOptions* poSampleOptions)
{
    LStatus                                     eLiberatusStatus;
    LNetStreamer_Config                         oSessionOptions;

    if (poSampleOptions->bValidSession && !poSampleOptions->hSession && (0 != poSampleOptions->eService))
    {
        SampleNetStreamer_ConvertOptions(poSampleOptions, &oSessionOptions);

        eLiberatusStatus = LNetStreamer_Create(&oSessionOptions.eConfigType, &poSampleOptions->hSession);

        if (LSTATUS_IS_SUCCESS(eLiberatusStatus))
        {
            if (LNetStreamer_Service_CLIENT == poSampleOptions->eService)
            {
                SampleNetStreamer_InitFileSink(&poSampleOptions->oFileSink, poSampleOptions);
            }

            if (LStatus_OK == eLiberatusStatus)
            {
                if (LNetStreamer_Service_SERVER == poSampleOptions->eService)
                {
                    poSampleOptions->bQuerySessionConfig = MTRUE;
                }
            }
            else
            {
                poSampleOptions->bQuerySessionConfig = MTRUE;
            }

            pthread_mutex_init(&poSampleOptions->oSessionMutex, NULL);

            poSampleOptions->uiVideoBytesPeriod = 0u;

            poSampleOptions->uiAudioBytesPeriod = 0u;

            poSampleOptions->uiReferenceTimeMsec = SampleNetStreamer_GetTimeMsec();

            poSampleOptions->bReferenceTimeDone = MTRUE;

            poSampleOptions->bContinueStreaming = MTRUE;

            if (poSampleOptions->bEnableVideo)
            {
                poSampleOptions->oVideoContext.poSampleOptions = poSampleOptions;

                poSampleOptions->oVideoContext.eMediaType = LNetStreamer_MediaType_VIDEO;

                if (LNetStreamer_Service_SERVER == poSampleOptions->eService)
                {
                    if (poSampleOptions->pacServerMediaFile)
                    {
                        poSampleOptions->oVideoContext.fpMedia = fopen(poSampleOptions->pacServerMediaFile, "rb");
                    }
                }

                pthread_create(
                    &(poSampleOptions->oVideoThread),
                    NULL,
                    &SampleNetStreamer_GenericThread,
                    (void*)(&poSampleOptions->oVideoContext));
            }

            if (poSampleOptions->bEnableAudio)
            {
                poSampleOptions->oAudioContext.poSampleOptions = poSampleOptions;

                poSampleOptions->oAudioContext.eMediaType = LNetStreamer_MediaType_AUDIO;

                if (LNetStreamer_Service_SERVER == poSampleOptions->eService)
                {
                    if (poSampleOptions->pacServerMediaFile)
                    {
                        poSampleOptions->oAudioContext.fpMedia = fopen(poSampleOptions->pacServerMediaFile, "rb");
                    }
                }

                pthread_create(
                    &(poSampleOptions->oAudioThread),
                    NULL,
                    &SampleNetStreamer_GenericThread,
                    (void*)(&poSampleOptions->oAudioContext));
            }
        }
        else
        {
            printf("unable to create network stream\n");
        }
    }
    else
    {
        eLiberatusStatus = LStatus_OK;
    }

    return eLiberatusStatus;
}

static void SampleNetStreamer_StopSession(
    struct SampleNetStreamerOptions* poSampleOptions)
{
    void*                                       pvVideoThreadResult;
    void*                                       pvAudioThreadResult;

    if (poSampleOptions->hSession)
    {
        poSampleOptions->bContinueStreaming = MFALSE;

        if (poSampleOptions->bEnableAudio)
        {
            pthread_join(
                poSampleOptions->oAudioThread,
                &pvAudioThreadResult);

            if (poSampleOptions->oAudioContext.fpMedia)
            {
                fclose(poSampleOptions->oAudioContext.fpMedia);

                poSampleOptions->oAudioContext.fpMedia = MNULL;
            }
        }

        if (poSampleOptions->bEnableVideo)
        {
            pthread_join(
                poSampleOptions->oVideoThread,
                &pvVideoThreadResult);

            if (poSampleOptions->oVideoContext.fpMedia)
            {
                fclose(poSampleOptions->oVideoContext.fpMedia);

                poSampleOptions->oVideoContext.fpMedia = MNULL;
            }
        }

        poSampleOptions->bReferenceTimeDone = MFALSE;

        poSampleOptions->uiReferenceTimeMsec = 0u;

        SampleNetStreamer_CleanupFileSink(&poSampleOptions->oFileSink);

        LNetStreamer_Destroy(poSampleOptions->hSession);

        poSampleOptions->hSession = MNULL;
    }
}

static struct SampleNetStreamerOptions* SampleNetStreamer_NewSession(
    struct SampleNetStreamerOptionsList*        poSampleOptionsList,
    unsigned int                                uiNextSessionIndex)
{
    struct SampleNetStreamerOptions*            poSampleOptions;

    if (uiNextSessionIndex < poSampleOptionsList->uiSessionMaxCount)
    {
        poSampleOptions = &(poSampleOptionsList->paoSampleOptions[uiNextSessionIndex]);

        if (!poSampleOptions->bValidSession)
        {
            memset(poSampleOptions, 0x00, sizeof(*poSampleOptions));

            poSampleOptions->eProtocol = LNetStreamer_Protocol_RTSP;

            poSampleOptions->iRtspServer = -1;

            poSampleOptions->iRtmpServer = -1;

            poSampleOptions->bEnableVideo = MTRUE;

            poSampleOptions->bEnableAudio = MFALSE;

            poSampleOptions->poSampleOptionsList = poSampleOptionsList;

            poSampleOptions->uiSessionIndex = uiNextSessionIndex;

            poSampleOptions->bValidSession = MTRUE;
        }

        poSampleOptionsList->uiCurrentSessionIndex = uiNextSessionIndex;
    }
    else
    {
        poSampleOptionsList->uiCurrentSessionIndex = ~0u;

        poSampleOptions = MNULL;
    }

    return poSampleOptions;
}

static void SampleNetStreamer_StartSessionList(
    struct SampleNetStreamerOptionsList*        poSampleOptionsList)
{
    MUINT                                       uiSessionIndex;
    struct SampleNetStreamerOptions*            poSampleOptions;

    for (uiSessionIndex = 0u; uiSessionIndex < poSampleOptionsList->uiSessionMaxCount; uiSessionIndex ++)
    {
        poSampleOptions = &(poSampleOptionsList->paoSampleOptions[uiSessionIndex]);

        if (poSampleOptions->bValidSession && !poSampleOptions->hSession)
        {
            SampleNetStreamer_StartSession(poSampleOptions);
        }
    }
}

static void SampleNetStreamer_StopSessionList(
    struct SampleNetStreamerOptionsList*        poSampleOptionsList)
{
    MUINT                                       uiSessionIndex;
    struct SampleNetStreamerOptions*            poSampleOptions;

    for (uiSessionIndex = 0u; uiSessionIndex < poSampleOptionsList->uiSessionMaxCount; uiSessionIndex ++)
    {
        poSampleOptions = &(poSampleOptionsList->paoSampleOptions[poSampleOptionsList->uiSessionMaxCount - uiSessionIndex - 1u]);

        SampleNetStreamer_StopSession(poSampleOptions);
    }
}

static void SampleNetStreamer_InitSessionList(
    struct SampleNetStreamerOptionsList*        poSampleOptionsList)
{
    poSampleOptionsList->uiSessionMaxCount = 128u;

    poSampleOptionsList->uiCurrentSessionIndex = ~0u;

    poSampleOptionsList->bContinueStatThread = MFALSE;

    poSampleOptionsList->paoSampleOptions = (struct SampleNetStreamerOptions*)(malloc(
            sizeof(struct SampleNetStreamerOptions) * poSampleOptionsList->uiSessionMaxCount));

    memset(poSampleOptionsList->paoSampleOptions, 0x00u, sizeof(struct SampleNetStreamerOptions) * poSampleOptionsList->uiSessionMaxCount);
}

static void SampleNetStreamer_PrintUsage(void)
{
    printf(
        "\n"
        "Optional arguments (common):\n"
        "    <idx>         Select session index (0-127)\n"
        "    ,             Select next session index\n"
        "    -c            Select client service\n"
        "    -s            Select server service\n"
        "    -p            Select protocol rtsp\n"
        "    -e            Select protocol rtp\n"
        "    -t            Select protocol mpeg2-ts\n"
        "    -j            Select protocol rtmp\n"
        "    -v            Enable video only track\n"
        "    -a            Enable audio only track\n"
        "    -V            Enable video and audio tracks\n"
        "    -d <ip>       Select destination ip address ex.: 11.12.13.14\n"
        "    -w <port>     Select video port (1-65535)\n"
        "    -b <port>     Select audio port (1-65535)\n"
        "    -r <port>     Select rtsp server port (1-65535)\n"
        "    -i <eth>      Select net interface ex.: eth0\n"
        "    -P            Enable RTSP server\n"
        "    -J            Enable RTMP server\n"
        "    -E <msec>     Delay between RTCP reports\n"
        "\n"
        "Optional arguments (server):\n"
        "    -f <name>     Select rtsp folder name ex.: test\n"
        "    <file>        Select a media file (full paths) ex.: ./dump.bin\n"
        "    -m <file>     Select a media file ex.: ./dump.bin\n"
        "\n"
        "Optional arguments (client):\n"
        "    <uri>         Select rtsp or rtmp <uri> ex.: rtsp://11.12.13.14:554/test\n"
        "    -l <uri>      Select rtsp or rtmp <uri> ex.: rtsp://11.12.13.14:554/test\n"
        "    -o <file>     Select a dump file ex.: dump.bin\n"
        "    -k            Disable frame skip detection\n"
        "\n"
        "Examples:\n"
        "  Save video stream from a unicast RTSP server to a file:\n"
        "    LNetStreamerSample rtsp://11.12.13.14:554/test -o ./dump.bin\n"
        "  Create a unicast RTSP server streaming encoded video from a file:\n"
        "    LNetStreamerSample ./dump.bin\n"
        "\n");
}

static int SampleNetStreamer_ParseSessionListOptions(
    struct SampleNetStreamerOptionsList*        poSampleOptionsList,
    int                                         argc,
    char**                                      argv)
{
    struct SampleNetStreamerOptions*            poSampleOptions;
    int                                         argi;
    int                                         iValidOptions;

    if (poSampleOptionsList->uiCurrentSessionIndex < poSampleOptionsList->uiSessionMaxCount)
    {
        poSampleOptions = poSampleOptionsList->paoSampleOptions + poSampleOptionsList->uiCurrentSessionIndex;
        if (!poSampleOptions->bValidSession)
        {
            poSampleOptions = MNULL;
        }
    }
    else
    {
        poSampleOptions = MNULL;
    }

    argi = 1;
    iValidOptions = 1;

    while (argi < argc)
    {
        if ((0 == strcmp(argv[argi], "-h")) ||
            (0 == strcmp(argv[argi], "--help")) ||
            (0 == strcmp(argv[argi], "-?")) ||
            ('h' == argv[argi][0]))
        {
            SampleNetStreamer_PrintUsage();
            iValidOptions = 0;
            argi ++;
        }
        else if (('0' <= argv[argi][0]) && ('9' >= argv[argi][0]))
        {
            unsigned int uiNextSessionIndex;
            sscanf(argv[argi], "%u", &(uiNextSessionIndex));
            poSampleOptions = SampleNetStreamer_NewSession(poSampleOptionsList, uiNextSessionIndex);
            argi ++;
        }
        else if (',' == argv[argi][0])
        {
            poSampleOptions = SampleNetStreamer_NewSession(poSampleOptionsList, poSampleOptionsList->uiCurrentSessionIndex + 1);
            argi ++;
        }
        else
        {
            if (!poSampleOptions)
            {
                poSampleOptions = SampleNetStreamer_NewSession(poSampleOptionsList, 0u);
            }

            if (poSampleOptions)
            {
                if (0 == strcmp(argv[argi], "-c"))
                {
                    poSampleOptions->eService = LNetStreamer_Service_CLIENT;
                    argi ++;
                }
                else if (0 == strcmp(argv[argi], "-s"))
                {
                    poSampleOptions->eService = LNetStreamer_Service_SERVER;
                    argi ++;
                }
                else if ((0 == strcmp(argv[argi], "-l")) ||
                    (0 == strncmp(argv[argi], "v=", 2)) ||
                    (0 == strncmp(argv[argi], "rtsp://", 7)) ||
                    (0 == strncmp(argv[argi], "rtmp://", 7)))
                {
                    if (0 == strcmp(argv[argi], "-l"))
                    {
                        argi ++;
                    }
                    if (argi < argc)
                    {
                        if (0 == strncmp(argv[argi], "v=", 2))
                        {
                            poSampleOptions->eService = LNetStreamer_Service_CLIENT;
                            poSampleOptions->eProtocol = LNetStreamer_Protocol_RTP;
                            poSampleOptions->pacLocation = 0;
                            poSampleOptions->pacSdp = strdup(argv[argi]);
                        }
                        else if (0 == strncmp(argv[argi], "rtsp://", 7))
                        {
                            poSampleOptions->eService = LNetStreamer_Service_CLIENT;
                            poSampleOptions->eProtocol = LNetStreamer_Protocol_RTSP;
                            poSampleOptions->pacLocation = strdup(argv[argi]);
                            poSampleOptions->pacSdp = 0;
                        }
                        else if (0 == strncmp(argv[argi], "rtmp://", 7))
                        {
                            poSampleOptions->eService = LNetStreamer_Service_CLIENT;
                            poSampleOptions->eProtocol = LNetStreamer_Protocol_RTMP;
                            poSampleOptions->pacLocation = strdup(argv[argi]);
                            poSampleOptions->pacSdp = 0;
                        }
                        else
                        {
                            poSampleOptions->eService = LNetStreamer_Service_CLIENT;
                            poSampleOptions->pacLocation = strdup(argv[argi]);
                            poSampleOptions->pacSdp = 0;
                        }
                        argi ++;
                    }
                }
                else if (0 == strcmp(argv[argi], "-p"))
                {
                    poSampleOptions->eProtocol = LNetStreamer_Protocol_RTSP;
                    argi ++;
                }
                else if (0 == strcmp(argv[argi], "-j"))
                {
                    poSampleOptions->eProtocol = LNetStreamer_Protocol_RTMP;
                    argi ++;
                }
                else if (0 == strcmp(argv[argi], "-e"))
                {
                    poSampleOptions->eProtocol = LNetStreamer_Protocol_RTP;
                    argi ++;
                }
                else if (0 == strcmp(argv[argi], "-t"))
                {
                    poSampleOptions->eProtocol = LNetStreamer_Protocol_TS;
                    argi ++;
                }
                else if (0 == strcmp(argv[argi], "-v"))
                {
                    poSampleOptions->bEnableAudio = MFALSE;
                    poSampleOptions->bEnableVideo = MTRUE;
                    argi++;
                }
                else if (0 == strcmp(argv[argi], "-V"))
                {
                    poSampleOptions->bEnableAudio = MTRUE;
                    poSampleOptions->bEnableVideo = MTRUE;
                    argi++;
                }
                else if (0 == strcmp(argv[argi], "-a"))
                {
                    poSampleOptions->bEnableVideo = MFALSE;
                    poSampleOptions->bEnableAudio = MTRUE;
                    argi++;
                }
                else if (0 == strcmp(argv[argi], "-d"))
                {
                    argi++;
                    if (argi < argc)
                    {
                        poSampleOptions->pacDestination = strdup(argv[argi]);
                        argi ++;
                    }
                }
                else if (0 == strcmp(argv[argi], "-w"))
                {
                    argi++;
                    if (argi < argc)
                    {
                        sscanf(argv[argi], "%lu", &poSampleOptions->uiVideoPort);
                        argi ++;
                    }
                }
                else if (0 == strcmp(argv[argi], "-b"))
                {
                    argi++;
                    if (argi < argc)
                    {
                        sscanf(argv[argi], "%lu", &poSampleOptions->uiAudioPort);
                        argi ++;
                    }
                }
                else if (0 == strcmp(argv[argi], "-r"))
                {
                    argi++;
                    if (argi < argc)
                    {
                        sscanf(argv[argi], "%ld", &poSampleOptions->uiRtspPort);
                        argi ++;
                    }
                }
                else if (0 == strcmp(argv[argi], "-f"))
                {
                    argi++;
                    if (argi < argc)
                    {
                        poSampleOptions->pacRtspFolder = strdup(argv[argi]);
                        argi ++;
                    }
                }
                else if (0 == strcmp(argv[argi], "-i"))
                {
                    argi++;
                    if (argi < argc)
                    {
                        poSampleOptions->pacNetInterface = strdup(argv[argi]);
                        argi ++;
                    }
                }
                else if ((0 == strcmp(argv[argi], "-m")) || ('.' == argv[argi][0]) || ('/' == argv[argi][0]))
                {
                    if (0 == strcmp(argv[argi], "-m"))
                    {
                        argi ++;
                    }
                    if (argi < argc)
                    {
                        poSampleOptions->eService = LNetStreamer_Service_SERVER;
                        poSampleOptions->pacServerMediaFile = strdup(argv[argi]);
                        argi ++;

                        while (argi < argc)
                        {
                            if (('.' == argv[argi][0]) || ('/' == argv[argi][0]))
                            {
                                poSampleOptions = SampleNetStreamer_NewSession(poSampleOptionsList, poSampleOptionsList->uiCurrentSessionIndex + 1);
                                if (poSampleOptions)
                                {
                                    poSampleOptions->eService = LNetStreamer_Service_SERVER;
                                    poSampleOptions->pacServerMediaFile = strdup(argv[argi]);
                                }
                                argi ++;
                            }
                            else
                            {
                                break;
                            }
                        }
                    }
                }
                else if (0 == strcmp(argv[argi], "-o"))
                {
                    argi ++;
                    if (argi < argc)
                    {
                        poSampleOptions->pacClientMediaFile = strdup(argv[argi]);
                        argi ++;
                    }
                }
                else if (0 == strcmp(argv[argi], "-k"))
                {
                    argi++;
                    poSampleOptions->bDisableFrameSkipDetection = MTRUE;
                }
                else if (0 == strcmp(argv[argi], "-P"))
                {
                    argi++;
                    poSampleOptions->iRtspServer = 1;
                }
                else if (0 == strcmp(argv[argi], "-J"))
                {
                    argi++;
                    poSampleOptions->iRtmpServer = 1;
                }
                else if (0 == strcmp(argv[argi], "-E"))
                {
                    argi++;
                    if (argi < argc)
                    {
                        sscanf(argv[argi], "%lu", &poSampleOptions->uiRtcpFrequency);
                        argi++;
                    }
                }
                else
                {
                    fprintf(stderr, "unknown command line option\n");
                    argi++;
                    iValidOptions = 0;
                }
            }
        }
    }

    return iValidOptions;
}

static void SampleNetStreamer_KeyboardEventLoop(void)
{
    static char                                 acUserInputChunk[16u];
    MBOOL                                       bQuitKeyboardLoop;

    bQuitKeyboardLoop = MFALSE;

    while (!bQuitKeyboardLoop)
    {
        puts("press <Enter> to exit");

        if (fgets(acUserInputChunk, sizeof(acUserInputChunk), stdin))
        {
            if (('\r' == acUserInputChunk[0u]) ||
                ('\n' == acUserInputChunk[0u]))
            {
                bQuitKeyboardLoop = MTRUE;
            }
        }
        else
        {
            bQuitKeyboardLoop = MTRUE;
        }
    }
}

static void SampleNetStreamer_PrintSessionStatistics(
    struct SampleNetStreamerOptions*            poSampleOptions)
{
    MUINT64                                     uiVideoBytesPeriod;
    MUINT64                                     uiAudioBytesPeriod;

    if (poSampleOptions->bValidSession && poSampleOptions->bReferenceTimeDone)
    {
        pthread_mutex_lock(&poSampleOptions->oSessionMutex);

        uiVideoBytesPeriod = poSampleOptions->uiVideoBytesPeriod;

        uiAudioBytesPeriod = poSampleOptions->uiAudioBytesPeriod;

        poSampleOptions->uiVideoBytesPeriod = 0;

        poSampleOptions->uiAudioBytesPeriod = 0;

        pthread_mutex_unlock(&poSampleOptions->oSessionMutex);

        if (poSampleOptions->bEnableVideo && poSampleOptions->bEnableAudio)
        {
            printf("%s %3u: %7lu Kbps video, %7lu Kbps audio\n",
                (LNetStreamer_Service_SERVER == poSampleOptions->eService) ? "server" : "client",
                (unsigned int)(poSampleOptions->uiSessionIndex),
                (unsigned long int)((uiVideoBytesPeriod * 8000) / (DUMP_STATISTICS_PERIOD * 1000)),
                (unsigned long int)((uiAudioBytesPeriod * 8000) / (DUMP_STATISTICS_PERIOD * 1000)));
        }
        else if (poSampleOptions->bEnableVideo)
        {
            printf("%s %3u: %7lu Kbps video\n",
                (LNetStreamer_Service_SERVER == poSampleOptions->eService) ? "server" : "client",
                (unsigned int)(poSampleOptions->uiSessionIndex),
                (unsigned long int)((uiVideoBytesPeriod * 8000) / (DUMP_STATISTICS_PERIOD * 1000)));
        }
        else if (poSampleOptions->bEnableAudio)
        {
            printf("%s %3u: %7lu Kbps audio\n",
                (LNetStreamer_Service_SERVER == poSampleOptions->eService) ? "server" : "client",
                (unsigned int)(poSampleOptions->uiSessionIndex),
                (unsigned long int)((uiAudioBytesPeriod * 8000) / (DUMP_STATISTICS_PERIOD * 1000)));
        }
    }
}

static void SampleNetStreamer_PrintSessionListStatistics(
    struct SampleNetStreamerOptionsList*        poSampleOptionsList)
{
    MUINT                                       uiSessionIndex;

    for (uiSessionIndex = 0u; uiSessionIndex < poSampleOptionsList->uiSessionMaxCount; uiSessionIndex ++)
    {
        SampleNetStreamer_PrintSessionStatistics(
            poSampleOptionsList->paoSampleOptions + uiSessionIndex);
    }
}

static void SampleNetStreamer_StatisticsLoop(
    struct SampleNetStreamerOptionsList*        poSampleOptionsList)
{
    MUINT32                                     uiLastReportTime;
    MUINT32                                     uiCurrentTime;

    uiLastReportTime = SampleNetStreamer_GetTimeMsec();

    while (poSampleOptionsList->bContinueStatThread)
    {
        uiCurrentTime = SampleNetStreamer_GetTimeMsec();

        SampleNetStreamer_usleep((100ul - (uiCurrentTime % 100ul)) * 1000ul);

        uiCurrentTime = SampleNetStreamer_GetTimeMsec();

        if ((MUINT32)(uiCurrentTime - uiLastReportTime) >= DUMP_STATISTICS_PERIOD)
        {
            SampleNetStreamer_PrintSessionListStatistics(poSampleOptionsList);

            uiLastReportTime += (MUINT32)(DUMP_STATISTICS_PERIOD);
        }
    }
}

static void* SampleNetStreamer_StatisticsThread(
    void*                                       pvStatisticsThreadContext)
{
    struct SampleNetStreamerOptionsList*        poSampleOptionsList;

    poSampleOptionsList = (struct SampleNetStreamerOptionsList*)(pvStatisticsThreadContext);

    SampleNetStreamer_StatisticsLoop(poSampleOptionsList);

    return (void*)(0);
}

static void SampleNetStreamer_StartStatisticsThread(
    struct SampleNetStreamerOptionsList*        poSampleOptionsList)
{
    if (!poSampleOptionsList->bContinueStatThread)
    {
        poSampleOptionsList->bContinueStatThread = MTRUE;

        pthread_create(
            &poSampleOptionsList->oStatisticsThread,
            NULL,
            &SampleNetStreamer_StatisticsThread,
            poSampleOptionsList);
    }
}

static void SampleNetStreamer_StopStatisticsThread(
    struct SampleNetStreamerOptionsList*        poSampleOptionsList)
{
    void*                                       pvThreadResult;

    if (poSampleOptionsList->bContinueStatThread)
    {
        poSampleOptionsList->bContinueStatThread = MFALSE;

        pthread_join(
            poSampleOptionsList->oStatisticsThread,
            &pvThreadResult);
    }
}

static void SampleNetStreamer_Main(
    int                                         argc,
    char**                                      argv)
{
    struct SampleNetStreamerOptionsList         oSampleOptionsList;

    SampleNetStreamer_InitSessionList(&oSampleOptionsList);

    /* start statistics loop */
    SampleNetStreamer_StartStatisticsThread(&oSampleOptionsList);

    if (SampleNetStreamer_ParseSessionListOptions(&oSampleOptionsList, argc, argv))
    {
        /* start all sessions */
        SampleNetStreamer_StartSessionList(&oSampleOptionsList);

        /* keyboard loop... */
        SampleNetStreamer_KeyboardEventLoop();

        /* stop all sessions */
        SampleNetStreamer_StopSessionList(&oSampleOptionsList);
    }

    /* stop statistics loop */
    SampleNetStreamer_StopStatisticsThread(&oSampleOptionsList);
}

int main(
    int                                         argc,
    char **                                     argv)
{
#if defined(LNETSTREAMER_CONFIG_OS_WIN32)
    timeBeginPeriod(1);
#endif

    SampleNetStreamer_Main(argc, argv);

#ifdef LNETSTREAMER_CONFIG_OS_WIN32
    timeEndPeriod(1);
#endif //LNETSTREAMER_CONFIG_OS_WIN32

    return 0;
}
