/******************************************************************************\

Module Name:    LEncoder.c

Description:    LEncoder application main implementation file.
 
                In this sample we use two CPU threads: one to push the pictures 
                to encode to the encoder and one to pull the encoded data from 
                the encoder and sent it to a file, a pipe or a socket. In this
                implementation we write to a file which may be a regular file or
                a named pipe created by the mkfifo command.

References:     Revision 2.0
                xxxxx-xxx-xxxx-xxxx
                Matrox Liberatus specifications

Copyright (c) 2015, Matrox Graphics Inc.
All Rights Reserved.

BSD 2-Clause License

Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this 
   list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation 
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 
\******************************************************************************/

// -----------------------------------------------------------------------------
// I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------
#include "Liberatus.h"
#include "LBoard.h"
#include "LVideoIn.h"
#include "LBuffer.h"
#include "LDeviceThread.h"
#include "LVideoProcessor.h"
#include "LH264VideoCodec.h"
#include "LBlit.h"
#include <assert.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <time.h>
#include <malloc.h>
#include <string.h>

#include <pthread.h>

#if _WIN32
#include <windows.h>
#else
#include <unistd.h>
#include <poll.h>
#include <getopt.h>
#include <sys/signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#endif

//#define TRACE_NALU_SIZE

// -----------------------------------------------------------------------------
// N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------

#define NB_CAPTURE_BUFFER 32

typedef struct CPUTHREAD_OBJECT* CpuThread_Handle;
typedef LStatus (*CpuThread_Function)(void* pvRef);
typedef void* (*PosixThread_Function)(void* pvRef);

typedef struct tagCpuThread_Data
{
    volatile MBOOL32 bThreadRunning;
    LH264E_Handle    hEncoder;
             
} CpuThread_Data;

// -----------------------------------------------------------------------------
// S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------

MBOOL   bSkipWrite = MFALSE;
MCHAR8* szFileName = "/dev/null";
MUINT32 uiTestDuration = 0;
MUINT64 uiTestStartTime = 0;

// -----------------------------------------------------------------------------
// C O D E
// -----------------------------------------------------------------------------

static MUINT32 LEncoder_GetTimeSec(void)
{
  #if _WIN32
    return (MUINT32)(timeGetTime()) / 1000;
  #else
    struct timespec oTime;

    clock_gettime(CLOCK_MONOTONIC, &oTime);

    return (MUINT32)(oTime.tv_sec * 1000 + (oTime.tv_nsec / 1000 / 1000)) / 1000;
  #endif
}

/******************************************************************************\

Function:       show_usage

Description:    Show the usage of this program

Parameters: 

Return Value:   

Comments:       
 
\******************************************************************************/
void show_usage()
{
    fprintf(stderr, "Usage:\n");
    fprintf(stderr, "LEncoder [options] [filename]\n\n");
    fprintf(stderr, "Options:\n");
    fprintf(stderr, "-h, --help: This usage information\n\n");
    fprintf(stderr, "-t N, --test N: Duration in seconds of the test\n\n");
    fprintf(stderr, "Examples:\n");
    fprintf(stderr, "LEncoder\n");
    fprintf(stderr, "\tencode to /dev/null\n");
    fprintf(stderr, "LEncoder video0.h264\n");
    fprintf(stderr, "\tencode to the file video0.h264\n");
    fprintf(stderr, "mkfifo /tmp/video0.fifo; LEncoder /tmp/video0.fifo\n");
    fprintf(stderr, "\tencode to a named pipe (fifo) /tmp/video0.fifo\n");
    fprintf(stderr, "\n");
}

/******************************************************************************\

Function:       get_szstatusstr

Description:    Utility function to get the LStatus string.

Parameters:     eStatus     LStatus to convert to string.

Return Value:   None

Comments:       
 
\******************************************************************************/
const MCHAR8* get_szstatusstr(
                        LStatus eStatus)
{
    switch (eStatus)
    {
        case LStatus_SEQUENCE_RESET:                     return "SEQUENCE_RESET";
        case LStatus_DISORDERED_RELEASE_COUNT:           return "DISORDERED_RELEASE_COUNT";
        case LStatus_CRITICAL_END_OF_STREAM:             return "CRITICAL_END_OF_STREAM";
        case LStatus_STREAM_NOT_INITIALIZED:             return "STREAM_NOT_INITIALIZED";
        case LStatus_INCOMPLETE_COMMAND:                 return "INCOMPLETE_COMMAND";
        case LStatus_REORDERED_SEQUENCE_PENDING:         return "REORDERED_SEQUENCE_PENDING";
        case LStatus_END_OF_STREAM:                      return "END_OF_STREAM";
        case LStatus_INFO_NOT_AVAILABLE:                 return "INFO_NOT_AVAILABLE";
        case LStatus_TEMPORARY_LACK_OF_RESOURCES:        return "TEMPORARY_LACK_OF_RESOURCES";
        case LStatus_HARDWARE_MALFUNCTION:               return "HARDWARE_MALFUNCTION";
        case LStatus_CONNECTION_LOST:                    return "CONNECTION LOST";
        case LStatus_COMMAND_PENDING:                    return "COMMAND PENDING";
        case LStatus_CANCELED:                           return "CANCELED";
        case LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL:    return "DEVICETHREAD_COMMAND_QUEUE_FULL";
        case LStatus_UNSUPPORTED:                        return "UNSUPPORTED";
        case LStatus_ACCESS_DENIED:                      return "ACCESS_DENIED";
        case LStatus_RESOURCES_BUSY:                     return "RESOURCES_BUSY";
        case LStatus_OUT_OF_RESOURCES:                   return "OUT_OF_RESOURCES";
        case LStatus_OUT_OF_MEMORY:                      return "OUT_OF_MEMORY";
        case LStatus_NO_MORE_DATA:                       return "NO_MORE_DATA";
        case LStatus_TIMEOUT:                            return "TIMEOUT";
        case LStatus_INVALID_PARAM:                      return "INVALID_PARAM";
        case LStatus_FAIL:                               return "FAIL";
        case LStatus_OK:                                 return "OK";
        case LStatus_NOT_OPTIMAL:                        return "NOT_OPTIMAL";
        case LStatus_OK_INCOMPLETE_REORDERED_SEQUENCE:   return "INCOMPLETE_REORDERED_SEQUENCE";
        case LStatus_OK_MISSING_FIRST_SPS:               return "LStatus_OK_MISSING_FIRST_SPS";
        case LStatus_OK_STREAM_NOT_LOWLATENCY:           return "LStatus_OK_STREAM_NOT_LOWLATENCY";
    }
    return " UNKNOWN";
}

/******************************************************************************\

Function:       get_options

Description:    Utility function to get the command line options into global 
                variables

Parameters:     argc,
                argv

Return Value:   None

Comments:       
 
\******************************************************************************/
void get_options(int argc, char** argv)
{
  #if _WIN32
    char* optarg;
  #else
    static struct option long_options[] =
    {
        {"help", no_argument,       0, 'h'},
        {"test", required_argument, 0, 't'},
        {0, 0, 0, 0}
    };
  #endif  

    int argf = 0;
    int argi = 1;
    int c;

    do
    {
      #if _WIN32
        c = -1;

        while (argi < argc)
        {
            if (strcmp(argv[argi], "-h") == 0)
            {
                c = 'h';
                argi++;
                break;
            }
            else if (strcmp(argv[argi], "-t") == 0)
            {
                c = 't';
                argi++;
                optarg = argi < argc ? argv[argi] : "0";
                argi++;
                break;
            }
            else
            {
                c = -1;
                argi++;
                break;
            }
        }
      #else
        c = getopt_long(argc, argv, "ht:", long_options, &argi);
      #endif
        
        if (c != -1)
        {
            argf++;
            
            switch (c)
            {
                case ':':
                case '?':
                case 'h':
                default:
                    // help or an unknown option
                    show_usage();
                    exit(0);
                    
                case 't':
                    uiTestDuration = atoi(optarg); argf++;
                    break;
            }
        }                    

    } while (c != -1);

    if (argc == 2 + argf)
    {
        szFileName = argv[1 + argf];
    }
}

/******************************************************************************\

Function:       is_stop_required

Description:    Utility function to return MTRUE if 'Enter' is pressed on the
                keyboard and a line of data is available from stdin.

Parameters: 

Return Value:   MTRUE, MFALSE

Comments:       Pressing 'Enter' will stop encoding.
 
\******************************************************************************/
MBOOL is_stop_required()
{
  #if _WIN32
    if (uiTestDuration != 0 && LEncoder_GetTimeSec() > uiTestStartTime + uiTestDuration)
    {
        return MTRUE;
    }

    return _kbhit();
  #else
    int           poll_status;
    struct pollfd poll_info;

    if (uiTestDuration != 0 && LEncoder_GetTimeSec() > uiTestStartTime + uiTestDuration)
    {
        return MTRUE;
    }
    
    poll_info.fd = 0;
    poll_info.events = POLLIN;

    poll_status = poll(&poll_info, 1, 0);
    
    if (poll_status > 0)
    {
        return MTRUE;
    }
    else
    {        
        return MFALSE;
    }
  #endif
}

/******************************************************************************\

Function:       CpuThread_Create

Description:    Utility function to create a CPU thread using the OS services.

Parameters:     fnThread
                pvRef
                phThread

Return Value:   LStatus_OK, LStatus_FAIL

Comments:

\******************************************************************************/
LStatus CpuThread_Create(

    CpuThread_Function fnThread, 
    void*              pvRef, 
    CpuThread_Handle*  phThread)
{
    pthread_t* poThread = (pthread_t*) malloc(sizeof(pthread_t));

    if (poThread != MNULL)
    {
        if (pthread_create(poThread, MNULL, (PosixThread_Function)fnThread, pvRef) != 0)
        {
            free(poThread);
            poThread = MNULL;
        }
    }

    *phThread = (CpuThread_Handle)poThread;

    return *phThread ? LStatus_OK : LStatus_FAIL;
}

/******************************************************************************\

Function:       CpuThread_Destroy

Description:    Utility function to destroy a CPU thread using the OS services.

Parameters: 

Return Value:   LStatus_OK, LStatus_FAIL

Comments:       

\******************************************************************************/
LStatus CpuThread_Destroy(CpuThread_Handle hThread)
{
    if (hThread != MNULL)
    {
        if (pthread_join(*((pthread_t*)hThread), MNULL) != 0)
        {
            return LStatus_FAIL;
        }

        free(hThread);

        return LStatus_OK;
    }
    else
    {
        return LStatus_FAIL;
    }
}

/******************************************************************************\

Function:       open_fifo_output

Description:    Open the output file which may be a named pipe or a regular file

Parameters:     name

Return Value:   -1 on error, file descriptor >= 0 otherwise

Comments: 

\******************************************************************************/
MINT64 open_fifo_output(char* name)
{
  #if _WIN32
    HANDLE  hFipe;

    hFipe = CreateFile(name, GENERIC_WRITE, 0, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_OVERLAPPED, NULL);

    return (MINT64)hFipe;
  #else
    int fd   = -1;
    int fdrw = -1;
    
    struct sigaction sa = { { 0 } };
    sa.sa_handler = SIG_IGN;
    sa.sa_flags = 0;
    
    if (sigaction(SIGPIPE, &sa, 0) == -1) 
    {
        return -1;
    }

    fdrw = open(name, O_RDWR | O_NONBLOCK | O_SYNC | O_CREAT | O_TRUNC, S_IRWXU | S_IRWXG | S_IRWXO);

    if (fdrw == -1)
    {
        return -1;
    }

    fd = open(name, O_WRONLY | O_NONBLOCK | O_SYNC, S_IRWXU | S_IRWXG | S_IRWXO);
    
    if (fd == -1)
    {
        return -1;
    }

    close(fdrw);

    return fd;
  #endif
}

/******************************************************************************\

Function:       close_fifo_output

Description:    Close the output file

Parameters:     fd

Return Value:   

Comments: 

\******************************************************************************/
void close_fifo_output(MINT64 fd)
{
  #if _WIN32
    CloseHandle((HANDLE)fd);
  #else
    if (fd >= 0)
    {
        close(fd);
    }
  #endif
}

/******************************************************************************\

Function:       write_fifo_output

Description:    Write a packet of data to the output

Parameters:     fd
                packet_size
                packet_data

Return Value:   -1 on error, 0 otherwise

Comments: 

\******************************************************************************/
int write_fifo_output(MINT64 fd, size_t packet_size, MUINT8* packet_data)
{
  #if _WIN32
    int    status = 0;
    DWORD  write_size = 0;
    BOOL   bSuccess;

    while (!bSkipWrite)
    {
        bSuccess = WriteFile((HANDLE)fd, packet_data, packet_size, &write_size, NULL);

        if (write_size == 0 || !bSuccess)
        {
            status = -1;
            break;
        }
        else
        {
            packet_data += write_size;
            packet_size -= write_size;

            if (packet_size == 0)
            {
                break;
            }
        }
    }

    return status;
  #else
    int    status     = 0;
    size_t write_size = 0;
    
    while (!bSkipWrite)
    {
        write_size = write(fd, packet_data, packet_size);

        if (write_size == -1)
        {
            if (errno == EPIPE || errno == EAGAIN)
            {
                int poll_status;
                
                struct pollfd poll_info;
                poll_info.fd = fd;
                poll_info.events = POLLOUT;

                poll_status = poll(&poll_info, 1, 1);

                assert(poll_status == 0 || poll_status == 1 || poll_status == -1);

                if (poll_status == -1)
                {
                    status = -1;
                    break;
                }
            }
            else
            {
                status = -1;
                break;
            }
        }
        else
        {
            packet_data += write_size;
            packet_size -= write_size;
            
            if (packet_size == 0)
            {
                break;
            }
        }
    }
    
    return status;
  #endif
}

/******************************************************************************\

Function:       TransmitNalu

Description:    

Parameters:     fd_fifo_output
                poNaluInfo

Return Value:

Comments: 

\******************************************************************************/
LStatus TransmitNalu(MINT64 fd_fifo_output, LH264E_NaluInfo* poNaluInfo)
{
    LStatus eStatus       = LStatus_OK;
    size_t  uiPacketSize  = poNaluInfo->uiSize;
    MUINT8* puiPacketData = MNULL;

    eStatus = LBuffer_BeginAccess((LBuffer_Handle)poNaluInfo->hBuffer, 0, 1, &puiPacketData);
    
    if (eStatus != LStatus_OK)
    {
        return eStatus;
    }
    
    puiPacketData += poNaluInfo->uiStartOffset;
    
    if (write_fifo_output(fd_fifo_output, uiPacketSize, puiPacketData) == -1)
    {
        eStatus = LStatus_FAIL;
    }
    else
    {
        eStatus = LStatus_OK;
    }
    
    LBuffer_EndAccess((LBuffer_Handle)poNaluInfo->hBuffer);
    
    return eStatus;
}

/******************************************************************************\

Function:       StreamingThread

Description:    CPU thread responsible to pull the encoded data from the 
                encoder and to transmit it.

Parameters:     pvRef: pointer to thread's data

Return Value:

Comments:       Thread end with on the last captured picture from source

\******************************************************************************/
LStatus StreamingThread(void* pvRef)
{
    LStatus           eStatus        = LStatus_OK;
    CpuThread_Data*   poThreadData   = (CpuThread_Data*)pvRef;
    LH264E_NaluInfo   oNaluInfo      = { LH264E_NaluInfoTypeHeader_STANDARD };
    MBOOL             bDone          = MFALSE;
    MINT64            fd_fifo_output = -1;

    poThreadData->bThreadRunning     = MTRUE;

    fprintf(stderr, "LEncoder streaming thread starting\n");
    
    fd_fifo_output = open_fifo_output(szFileName);
    
    if (fd_fifo_output == -1)
    {
        poThreadData->bThreadRunning = MFALSE;
        return LStatus_FAIL;
    }

    while (poThreadData->bThreadRunning && !bDone)
    {
        LStatus eExecStatus = LStatus_OK;
        
        // Get the next encoded data.
        eStatus = LH264E_GetNextNalu(
            poThreadData->hEncoder,
            (LH264E_NaluInfoTypeHeader*)&oNaluInfo.eType,
            0);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            if ((oNaluInfo.flNaluStatusFlags & LH264E_NaluStatusFlag_FATALFIRMWAREERROR) ||
                (oNaluInfo.flNaluStatusFlags & LH264E_NaluStatusFlag_FATALSTREAMERROR)   ||
                (oNaluInfo.flNaluStatusFlags & LH264E_NaluStatusFlag_MISSINGPARAMETERS)  ||
                (oNaluInfo.flNaluStatusFlags & LH264E_NaluStatusFlag_UNSUPPORTEDPARAMETERS))
            {
                fprintf(stderr, "LH264E_GetNextNalu returned error flags 0x%x\n", oNaluInfo.flNaluStatusFlags);
                break;
            }
            
            if ((oNaluInfo.flNaluStatusFlags & LH264E_NaluStatusFlag_PICTURESSKIPPED))
            {
                fprintf(stderr, "LH264E_GetNextNalu returned that pictures have been skipped\n");
            }
       
            if (oNaluInfo.hBuffer == 0 || oNaluInfo.uiSize == 0)
            {
                if ((oNaluInfo.flNaluStatusFlags & LH264E_NaluStatusFlag_AUFENCE) == 0)
                {
                    fprintf(stderr, "LH264E_GetNextNalu returned without data\n");
                }
            }
            else
            {
                #if defined(TRACE_NALU_SIZE)
                fprintf(stderr, "nalu %d\n", oNaluInfo.uiSize);
                #endif
                
                eExecStatus = TransmitNalu(fd_fifo_output, &oNaluInfo);
                
                if (eExecStatus != LStatus_OK)
                {
                    fprintf(
                        stderr,
                        "TransmitNalu failed %d (%s)\n",
                        eExecStatus,
                        get_szstatusstr(eExecStatus));
                    bDone = MTRUE;
                }
            }
            
            // The current encoded data must be released as soon
            // as it isn't required anymore and in the same order
            // as it was received. This warns the encoder that
            // it can reuse the memory for new encoded data.            
            do
            {
                eExecStatus = LH264E_ReleaseNalu(
                    poThreadData->hEncoder, 
                    (LH264E_NaluInfoTypeHeader*)&oNaluInfo.eType);

            } while (eExecStatus == LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL);
            
            if (eExecStatus != LStatus_OK)
            {
                fprintf(
                    stderr,
                    "LH264E_ReleaseNalu failed %d (%s)\n",
                    eExecStatus,
                    get_szstatusstr(eExecStatus));
                break;
            }
        }
        else if (eStatus == LStatus_END_OF_STREAM)
        {
            // This means the encoder pipeline has been emptied.
            fprintf(stderr, "LEncoder streaming thread END-OF-STREAM\n");
            break;
        }
        else if ((eStatus != LStatus_TIMEOUT) && (eStatus != LStatus_STREAM_NOT_INITIALIZED))
        {
            fprintf(
                stderr,
                "LH264E_GetNextNalu failed %d (%s)\n",
                eExecStatus,
                get_szstatusstr(eExecStatus));
            break;
        }
    }

    close_fifo_output(fd_fifo_output);

    fprintf(stderr, "LEncoder streaming thread terminating\n");

    poThreadData->bThreadRunning = MFALSE;

    return LStatus_OK;
}

/******************************************************************************\

Function:       main

Description:    LEncoder application entry point.

Parameters:     None.

Return Value:   0 on success, -1 otherwise.

Comments:       None.

\******************************************************************************/
MINT32 main(int argc, char** argv)
{
    MINT32                  iResult           = -1;
    MUINT32                 uiIndex           = 0;
    LStatus                 eStatus           = LStatus_FAIL;
    LDevice_Handle          hDevice           = MNULL;
    LVideoIn_Handle         hVideoIn          = MNULL;
    MBOOL32                 bFoundValidSource = MFALSE;
    MUINT32                 uiConnectorIndex  = 0;
    MUINT32                 uiSourceWidth     = 0;
    MUINT32                 uiSourceHeight    = 0;
    MUINT32                 uiBufferSelect    = 0;
    LBoard_ConnectorInfo    oConnectorInfo;

    get_options(argc, argv);
            
    fprintf(stderr, "LEncoder starting\n");
    fprintf(stderr, "... press 'Enter' to quit\n");
    fprintf(stderr, "... encoding to %s\n", szFileName);

    if (uiTestDuration != 0)
    {
        fprintf(stderr, "... for %d seconds\n", uiTestDuration);
    }

    Liberatus_Load();

    // Search for the first video input connector with a 60 Hz progressive 
    // source detected on the primary Liberatus handle.
    hDevice = Liberatus_GetDevice(0);

    memset(&oConnectorInfo, 0, sizeof(oConnectorInfo));
    
    do
    {
        oConnectorInfo.uiSize = sizeof(oConnectorInfo);
        
        eStatus = LBoard_GetConnectorInfo( 
            hDevice,
            uiConnectorIndex++,
            &oConnectorInfo);

        if (LSTATUS_IS_SUCCESS(eStatus) == MFALSE )
        {
            break; // No valid source detected
        }
        else if (oConnectorInfo.bVideoInput)
        {
            eStatus = LVideoIn_GetHandle(
                hDevice, 
                oConnectorInfo.uiVideoInIndex,
                LAccessMode_READWRITE_EXCLUSIVE,
                &hVideoIn);

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                MBOOL32 bIsCapturable = MFALSE;
                MBOOL32 bIsDetected   = MFALSE;
                
                eStatus = LVideoIn_DetectSource(  
                    hVideoIn,
                    &bIsCapturable,
                    &bIsDetected,
                    &uiSourceWidth,
                    &uiSourceHeight);

                if (bIsCapturable)
                {
                    LVideo_VidParam oVidParam = {0};
                    
                    eStatus = LVideoIn_GetDetectedVideoParameters(  
                        hVideoIn,
                        &oVidParam);
                        
                    if (LSTATUS_IS_SUCCESS(eStatus))
                    {
                        bFoundValidSource=((oVidParam.bInterlaced == MFALSE)  &&
                                           (oVidParam.uiVRate_Hz  == 60));
                    }
                }

                if (!bFoundValidSource)
                {
                    LVideoIn_ReleaseHandle(hVideoIn);
                }
            }
        }
        
    } while (!bFoundValidSource);

    // At this point, we've selected a valid source, and we're now ready to 
    // capture incoming frames.

    // Create and initialize all engines and device threads
    
    // The encoder needs two device threads: a first one to send frames to the encoder and a 
    // second one to retrieve encoded data. We need to create as many buffers for the video 
    // processor output as we have buffers for the capture. We program the encoder parameters 
    // with common values. For more information, see the Matrox Liberatus H.264 Video Codec 
    // specification. A new CPU thread is started to retrieve encoded data from a bytestream 
    // container.
    
    // Start the capture
    if (bFoundValidSource)
    {
        // This device thread is used to send the encode commands to the encoder.
        LDeviceThread_Handle    hDTInputEncoder               = MNULL; 

        // This device thread is used to release the encoded data of the encoder.
        LDeviceThread_Handle    hDTOutputEncoder              = MNULL; 

        LPixelFormat            ePixelFormat                  = LPixelFormat_INVALID;
        LVPE_Handle             hVpe                          = MNULL;
        CpuThread_Handle        hStreamingThread              = MNULL;
        LBlit_Handle            hBlit_Handle                  = MNULL;
        LBuffer_Handle          hVpeBuffer[NB_CAPTURE_BUFFER] = {0};

        CpuThread_Data          oThreadData;
        LVP_ColorSpace          oColorSpace;
        LBuffer_VideoAttributes oVideoAttributes;
        LH264E_CreateOpt        oEncoderCreateOpt;
        
        // Standard sequence data structure.
        LH264_StreamInfoEncoderSequenceData oSequenceData;

        // Standard subGOP data structure.
        LH264_StreamInfoEncoderSubSequenceData oSubSequenceData;

        // Standard encode structure.
        LH264E_EncodeOpt    oEncodeOpt; 

        LVP_Rect            oVpeRect;
        LColor              oColor;

        fprintf(stderr, "Capturing source w:%d, h:%d, Hz:%d\n", uiSourceWidth, uiSourceHeight, 60);

        memset(&oThreadData, 0, sizeof(oThreadData));

        LVideoIn_EnumSupportedPixelFormat( 
            hVideoIn,
            0,                  // First supported pixel format
            &ePixelFormat);

        LVideoIn_CreateBuffers(
            hVideoIn,
            uiSourceWidth,
            uiSourceHeight,
            ePixelFormat,
            NB_CAPTURE_BUFFER);

        LVideoIn_StartCapture(hVideoIn);
       
        // Create device threads.
        LDeviceThread_Create(hDevice, MNULL, &hDTInputEncoder);
        LDeviceThread_Create(hDevice, MNULL, &hDTOutputEncoder);
   
        // Create VPE target buffer.
        // This buffer is also used by the encoder which requires a buffer
        // aligned on a multiple of 16 lines to properly support the 16x16
        // h264 macro block encoding.
        memset(&oVideoAttributes, 0, sizeof(oVideoAttributes));
        
        oVideoAttributes.eAttributeType = LBuffer_Type_VIDEO;
        oVideoAttributes.ePixelFormat   = LPixelFormat_MP_Y8_U8V8_420;
        oVideoAttributes.uiWidth        = 1920;
        oVideoAttributes.uiHeight       = 1088; // 16x16 H264 Macro block

        LBlit_Create(hDevice, hDTInputEncoder, MNULL, &hBlit_Handle);

        for (uiIndex = 0; uiIndex < NB_CAPTURE_BUFFER; uiIndex++)
        {
            LBuffer_Create(
                hDevice,
                &oVideoAttributes.eAttributeType,
                &hVpeBuffer[uiIndex]);    

            // Fill with Y'CbCr black.
            memset(&oColor, 0, sizeof(oColor));
            
            oColor.ePixelFormat = LPixelFormat_MP_Y8_U8V8_420;
            oColor.uiY          = 16;
            oColor.uiU          = 128;
            oColor.uiV          = 128;

            LBlit_SolidFill(
                hBlit_Handle, 
                hVpeBuffer[uiIndex], 
                &oColor, 
                MNULL, 
                MNULL, 
                LBlit_Bop_S, 
                MNULL);
        }

        // Create the VPE.
        // Parameters are set to their default values at creation:
        // These default values are set for progressive buffers.
        // Source and target color spaces are set for HD buffers (Buffers
        // family can be R'G'B' with gamma based on Rec. ITU-R BT.709-5 or
        // Y'CbCr with transfer matrix coefficients based on Rec. ITU-R
        // BT.709-5).
        // The scaler filtering is in automatic mode (chosen based on the
        // scaling factor).
        // The bayer filter, deinterlacer module, and advanced enhancement
        // filtering are disabled by default.
        LVPE_Create(hDevice, hDTInputEncoder, MNULL, &hVpe);
       
        memset(&oColorSpace, 0, sizeof(oColorSpace));
        
        // Set source and target color space
        // R'G'B' with gamma based on Rec. ITU-R BT.709-5 (default).
        oColorSpace.eFunction       = LVP_TransferFunction_BT709; 
        // Y'CbCr transfer matrix with coefficients based on
        // Rec. ITU-R BT.709-5 (HD)(default).
        oColorSpace.eMatrix         = LVP_TransferMatrix_BT709;   
        // No user defined matrices.
        oColorSpace.poUserMatrix    = MNULL;

        LVPE_SetSourceParam(
            hVpe,
            LVPE_Source_COLOR_SPACE,
            sizeof(LVP_ColorSpace),
            &oColorSpace);

        LVPE_SetTargetParam(
            hVpe,
            LVPE_Target_COLOR_SPACE,
            sizeof(LVP_ColorSpace),
            &oColorSpace);
       
        // Source rectangle uses the default value (disable) and doesn't need to
        // be set. This means that the entire source is used based on the size
        // specified in the source buffer.
        // Enable the target rectangle and set it to 1080 lines to avoid
        // scaling the source to 1088 lines which is the VPE buffer size.
        // The entire source will be scaled to 1920 x 1080.
        memset(&oVpeRect, 0, sizeof(oVpeRect));
        
        oVpeRect.bEnable       = MTRUE;
        oVpeRect.oRect.iLeft   = 0;
        oVpeRect.oRect.iRight  = 1920;
        oVpeRect.oRect.iTop    = 0;
        oVpeRect.oRect.iBottom = 1080;

        LVPE_SetSourceParam(
            hVpe,
            LVPE_Source_DESTINATION_RECT,
            sizeof(oVpeRect),
            &oVpeRect);
            
        oVpeRect.bEnable       = MTRUE;
        oVpeRect.oRect.iLeft   = 0;
        oVpeRect.oRect.iRight  = 1920;
        oVpeRect.oRect.iTop    = 0;
        oVpeRect.oRect.iBottom = 1080;

        LVPE_SetTargetParam(
            hVpe,
            LVPE_Target_RECT,   
            sizeof(oVpeRect),
            &oVpeRect);
           
        // Create and initialize the encoder
        // (see the Matrox Liberatus H.264 Video Codec specification).

        memset(&oEncoderCreateOpt, 0, sizeof(oEncoderCreateOpt));
        oEncoderCreateOpt.eType = LH264E_CreateOptTypeHeader_STANDARD;
        
        // Create input device thread, used to send commands to the encoder.
        oEncoderCreateOpt.hDTInput        = (MUINT64)hDTInputEncoder;                            
        // Create output device thread, used to release encoded data.
        oEncoderCreateOpt.hDTOutput       = (MUINT64)hDTOutputEncoder;                           
        // The encoder will generate the HRD parameters into the VUI, 
        // the buffering period SEI, and the CPB removal delay and dpb 
        // output delay in the picture timing SEI.
        oEncoderCreateOpt.flCreateOptFlags = 
        // Use Context Adaptive Binary Arithmetic Coding.
            LH264E_CreateOptFlag_CABAC;

        // Set the rate control to normal mode.
        oEncoderCreateOpt.eEncoderMode     = LH264E_EncoderMode_RATECONTROL;  
        // Set the normal priority to this stream.
        oEncoderCreateOpt.ePriority        = LH264_Priority_NONREALTIME;         
        // Set profile to HIGH
        oEncoderCreateOpt.eProfile         = LH264_Profile_HIGH;
        // Set the encoding level high enough to do 1080p60.
        oEncoderCreateOpt.eLevel           = LH264_Level_5_2;                           

        // Define the start codes written as specified in the H.264 specification.
        oEncoderCreateOpt.eStartCodeMode   = LH264E_StartCodeMode_NORMAL;

        // At the encoder creation, an encoder handle is returned to identify
        // the current stream.
        
        LH264E_Create(
            hDevice,
            (LH264E_CreateOptTypeHeader*)&oEncoderCreateOpt.eType,
            &oThreadData.hEncoder);

        // Set the parameters used for the whole sequence.

        // Set the parameters related to the picture.
        memset(&oSequenceData, 0, sizeof(oSequenceData));

        oSequenceData.eType = LH264_StreamInfoTypeHeader_ENCODERSEQUENCEDATA;

        // Set width, always aligned on a multiple of 16.
        oSequenceData.uiPictureWidth              = 1920;  
        // Set height, always aligned on a multiple of 16.
        oSequenceData.uiPictureHeight             = 1088;  
        // NV12
        oSequenceData.ePixelFormat                = LPixelFormat_MP_Y8_U8V8_420;  
        oSequenceData.eScanMode                   = LH264_ScanMode_PROGRESSIVE;
        oSequenceData.oFrameRate.uiNumerator      = 60000;
        oSequenceData.oFrameRate.uiDenominator    = 1000;
        oSequenceData.fFAQStrength                = 0.75;

        oSequenceData.flSequenceDataFlags = 
          LH264_StreamInfoEncoderSequenceDataFlag_PICTUREINFOPRESENT |
          // Set the parameters related to the rate control.
          LH264_StreamInfoEncoderSequenceDataFlag_RATECONTROLPRESENT |
          // Set the control flags to crop the stream since the source is
          // 1920x1080, but encoding is done on a 1920x1088 buffer.
          LH264_StreamInfoEncoderSequenceDataFlag_PICTURECROPPEDRECTANGLEPRESENT;
          
        // Set target bit rate for rate control.
        oSequenceData.uiTargetBitRate               = 12000000;                      
        // Set a tight drift to the rate control to reduce the rate variation.
        oSequenceData.fCorrectionTensor             = 1.0; 
        // Set minimum Quantization Parameter (QP) allowed before the frequence
        // adaptive quantization.
        oSequenceData.fQPMin                        = 10;                
        // Set maximum QP allowed before the frequency adaptive quantization.
        oSequenceData.fQPMax                        = 48;
        // Set a CBR HRD.
        oSequenceData.eHRDMode                      = LH264_HRDMode_CBR; 
        // Set maximum Peak rate.
        oSequenceData.uiCPBMaxBitRate               = 24000000;          
        // Set buffering time to 33 ms
        oSequenceData.uiCPBSizeInBits               = (oSequenceData.uiCPBMaxBitRate * 33) / 1000;
        // Set the size of the source to 1920x1080.
        oSequenceData.oPictureCroppedRect.iLeft     = 0;
        oSequenceData.oPictureCroppedRect.iRight    = 0;
        oSequenceData.oPictureCroppedRect.iTop      = 0;
        oSequenceData.oPictureCroppedRect.iBottom   = 4;

        // Send the sequence parameters to the encoder.
        LH264E_UpdateStreamInfo(
            oThreadData.hEncoder,
            (LH264_StreamInfoTypeHeader*)&oSequenceData.eType);

        // Set the parameters related to the subgop.
        memset(&oSubSequenceData, 0, sizeof(oSubSequenceData));
        
        oSubSequenceData.eType = LH264_StreamInfoTypeHeader_ENCODERSUBSEQUENCEDATA;
        oSubSequenceData.flSubSequenceDataFlags =
            // Set the reference parameters.
            LH264_StreamInfoEncoderSubSequenceDataFlag_REFERENCEPRESENT;
            
        // Set the number of reference used for P.
        oSubSequenceData.uiNbRefForP               =  2;
        // Set the reference mode level to have no B pictures.
        oSubSequenceData.eRefMode                  = LH264_ReferenceMode_AUTOMATIC; 
        // Set when an I picture is an Idr. One Idr every 4x I pictures.
        oSubSequenceData.uiIdrPeriod               = 1;                     
        // Set GOP set to 1 second.
        oSubSequenceData.uiIPeriod                 = 90;                    
        // Since low latency doesn't allow B pictures, every picture that isn't
        // an I picture is a P picture
        oSubSequenceData.uiPPeriod                 = 1;                     

        // Send the subgop parameters to the encoder.
        LH264E_UpdateStreamInfo(
            oThreadData.hEncoder,
            (LH264_StreamInfoTypeHeader*)&oSubSequenceData.eType);

        memset(&oEncodeOpt, 0, sizeof(oEncodeOpt));
        oEncodeOpt.eType = LH264E_EncodeOptTypeHeader_STANDARD;

        // Picture type remains unspecified because we're in manual mode.
        oEncodeOpt.ePictureType = LH264_PictureType_UNSPECIFIED;
        oEncodeOpt.flPictureFlags = 0;

        // Start streaming thread.
        CpuThread_Create( 
            StreamingThread,
            (void*)&oThreadData,
            &hStreamingThread);

        uiBufferSelect = 0;
 
        uiTestStartTime = LEncoder_GetTimeSec();
        
        do
        {
            MUINT64         uiTag            = 0;
            MUINT64         uiFieldCounter   = 0;
            MUINT64         uiTickRefCounter = 0;
            MBOOL32         bIsTopField      = 0;
            LBuffer_Handle  hCapturedBuffer  = MNULL;
            LStatus         eExecStatus      = LStatus_OK;
            
            if (is_stop_required())
            {
                fprintf(stderr, "exit request in progress ...\n");
                bSkipWrite = MTRUE;
                goto do_exit;
            }

            eStatus = LVideoIn_GetNextBuffer(   
                hVideoIn,
                MTRUE,              // Wait for a Ready buffer
                0,                  // in ms
                &hCapturedBuffer,
                &uiFieldCounter,
                &uiTickRefCounter,
                &bIsTopField);

            if (eStatus == LStatus_TIMEOUT)
            {
                eStatus = LStatus_OK;
                continue;
            }

            if (eStatus != LStatus_OK)
            {
                fprintf(
                    stderr,
                    "LVideoIn_GetNextBuffer failed %d (%s)\n",
                    eExecStatus,
                    get_szstatusstr(eExecStatus));
                return -1;
            }

            // Start the execution of the video processor engine.
            eExecStatus = LVPE_ExecuteFrame( 
                hVpe,
                hVpeBuffer[uiBufferSelect],
                LVPE_CurrentFrameOrField_FRAME,  // Picture is frame based.
                MNULL,                           // previous source
                hCapturedBuffer,                 // current  source
                MNULL);                          // next     source

            if (eExecStatus != LStatus_OK)
            {
                fprintf(stderr, "LVPE_ExecuteFrame failed %d counter %d\n", eExecStatus, (MUINT32)uiFieldCounter);
                return -1;
            }

            // Schedule a release for hCapturedBuffer after ExecuteFrame is complete.
            eExecStatus = LVideoIn_ScheduleReleaseBuffer( 
                hVideoIn,
                hCapturedBuffer, 
                hDTInputEncoder,
                1000); // 1 sec

            if (eExecStatus != LStatus_OK)
            {
                fprintf(
                    stderr,
                    "LVideoIn_ScheduleReleaseBuffer failed %d (%s)\n",
                    eExecStatus,
                    get_szstatusstr(eExecStatus));
                return -1;
            }
            
            // Encode VPE output. Set the new buffer to encode.
            oEncodeOpt.hPicture = (MUINT64)hVpeBuffer[uiBufferSelect];  

            do
            {
                // Send the encode command to the encoder.
                eExecStatus = LH264E_Encode(
                    oThreadData.hEncoder,
                    (LH264E_EncodeOptTypeHeader*)&oEncodeOpt.eType);

                if (eExecStatus == LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL && is_stop_required())
                {
                    fprintf(stderr, "exit request in progress ...\n");
                    bSkipWrite = MTRUE;
                    goto do_exit;
                }                    
                    
            } while (eExecStatus == LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL);

            if (eExecStatus != LStatus_OK && eExecStatus != LStatus_OK_INCOMPLETE_REORDERED_SEQUENCE)
            {
                fprintf(
                    stderr,
                    "LH264E_Encode failed %d (%s) counter %d\n",
                    eExecStatus,
                    get_szstatusstr(eExecStatus),
                    (MUINT32)uiFieldCounter);
                return -1;
            }

            // Save completion tag.
            LDeviceThread_Handle hEncInternalDT = MNULL;
            LH264E_GetDTInputLastSubmissionTag(oThreadData.hEncoder, &uiTag, &hEncInternalDT);
            MUINT64 uiEncInternalDTLuid = LDEVICETHREAD_DEFAULT_LUID;
            LDeviceThread_GetLUID(hEncInternalDT, &uiEncInternalDTLuid);

            LBuffer_SetPrivateData(
                hVpeBuffer[uiBufferSelect],
                0, // index 0 is used to set the tag
                uiTag);
            LBuffer_SetPrivateData(
                hVpeBuffer[uiBufferSelect],
                1, // index 1 is used to set the DT LUID
                uiEncInternalDTLuid);
            // Select next buffer.
            uiBufferSelect = (uiBufferSelect + 1) % NB_CAPTURE_BUFFER;

            // Wait for buffer not in use.
            LBuffer_GetPrivateData (hVpeBuffer[uiBufferSelect], 0, &uiTag);
            LBuffer_GetPrivateData (hVpeBuffer[uiBufferSelect], 1, &uiEncInternalDTLuid);

            if (uiTag != 0)
            {
                eExecStatus = LDeviceThread_WaitCompletionTag(
                    hDTInputEncoder,
                    uiEncInternalDTLuid, 
                    uiTag, 
                    1000); // 1 sec

                if (eExecStatus != LStatus_OK)
                {
                    fprintf(
                        stderr,
                        "LDeviceThread_WaitCompletionTag failed %d (%s) \n",
                        eExecStatus,
                        get_szstatusstr(eExecStatus));
                    return -1;
                }
            }

        } while (LSTATUS_IS_SUCCESS(eStatus));
        
do_exit:

        LH264E_EndStream(oThreadData.hEncoder);

        while (oThreadData.bThreadRunning)
        {
            // The output thread will run until it gets a END_OF_STREAM return
            // value, which means the encoder pipeline is now empty.
        }

        CpuThread_Destroy(hStreamingThread);

        eStatus = LDeviceThread_WaitCompletionTag(
            hDTInputEncoder,
            LDEVICETHREAD_DEFAULT_LUID, 
            LDEVICETHREAD_DEFAULT_TAG, 
            2000); // 2 sec

        if (eStatus != LStatus_OK)
        {
            fprintf(stderr, "Encoder input device-thread not empty after 2 sec\n");
        }

        eStatus = LDeviceThread_WaitCompletionTag(
            hDTOutputEncoder,
            LDEVICETHREAD_DEFAULT_LUID, 
            LDEVICETHREAD_DEFAULT_TAG, 
            2000); // 2 sec

        if (eStatus != LStatus_OK)
        {
            fprintf(stderr, "Encoder output device-thread not empty after 2 sec\n");
        }

        // Destroy the encoder.
        LH264E_Destroy(oThreadData.hEncoder);
        
        // Stop and release the capture.
        LVideoIn_StopCapture(hVideoIn);
        LVideoIn_DestroyBuffers(hVideoIn);
        LVideoIn_ReleaseHandle(hVideoIn);

        LVPE_Destroy(hVpe);

        for(uiIndex = 0; uiIndex < NB_CAPTURE_BUFFER; uiIndex++)
        {
            LBuffer_Destroy(hVpeBuffer[uiIndex]);
        }

        LBlit_Destroy(hBlit_Handle);

        LDeviceThread_Destroy(hDTInputEncoder);
        LDeviceThread_Destroy(hDTOutputEncoder);

        iResult = 0; // Success
    }

    fprintf(stderr, "LEncoder terminating\n");

    Liberatus_UnLoad();

    return iResult;
}
