/******************************************************************************\

Module Name:    LDecoder.c

Description:    LDecoder application main implementation file.

                In this sample we use two CPU threads: one to pull the encoded 
                data from a file, a pipe or a socket and send it to the decoder
                and one to pull the pictures to display from the decoder.

References:     Revision 2.0
                Matrox Liberatus specifications.
                xxxxx-xxx-xxxx-xxxx

    Copyright (c) 2015, Matrox Graphics Inc.
    All Rights Reserved.

BSD 2-Clause License

Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this 
   list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation 
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON 
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\******************************************************************************/

// -----------------------------------------------------------------------------
// I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------
#include "Liberatus.h"
#include "LBoard.h"
#include "LVideoOut.h"
#include "LBuffer.h"
#include "LDeviceThread.h"
#include "LVideoProcessor.h"
#include "LBlit.h"
#include "LH264VideoCodec.h"

#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <fcntl.h>
#include <stdio.h>
#include <errno.h>
#include <time.h>
#include <malloc.h>
#include <string.h>
#include <pthread.h>

#if _WIN32
#include <windows.h>
#else
#include <unistd.h>
#include <poll.h>
#include <getopt.h>
#include <sys/signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#endif

//#define TRACE_NALU_SIZE

// -----------------------------------------------------------------------------
// N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------

#define NB_DISPLAY_BUFFER         2
#define NB_STREAM_BUFFER          8
#define NB_STREAM_BUFFER_SIZE  4096

typedef struct CPUTHREAD_OBJECT* CpuThread_Handle;
typedef LStatus (*CpuThread_Function)(void* pvRef);
typedef void* (*PosixThread_Function)(void* pvRef);

typedef struct tagCpuThread_Data
{
    volatile MBOOL32     bThreadRunning;
    LDevice_Handle       hDevice;
    LH264D_Handle        hDecoder;
    LDeviceThread_Handle hDTInputDecoder;
             
} CpuThread_Data;

// -----------------------------------------------------------------------------
// S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------

MCHAR8* szFileName  = "/dev/null";
MBOOL   bFileIsFifo = MFALSE;
MUINT32 uiTestDuration = 0;
MUINT32 uiTestStartTime = 0;

// -----------------------------------------------------------------------------
// C O D E
// -----------------------------------------------------------------------------

static MUINT32 LDecoder_GetTimeSec(void)
{
  #if _WIN32
    return (MUINT32)(timeGetTime()) / 1000;
  #else
    struct timespec oTime;

    clock_gettime(CLOCK_MONOTONIC, &oTime);

    return (MUINT32)(oTime.tv_sec * 1000 + (oTime.tv_nsec / 1000 / 1000)) / 1000;
  #endif
}

/******************************************************************************\

Function:       show_usage

Description:    Show the usage of this program

Parameters: 

Return Value:   

Comments:       
 
\******************************************************************************/
void show_usage()
{
    fprintf(stderr, "Usage:\n");
    fprintf(stderr, "LDecoder [options] filename\n\n");
    fprintf(stderr, "Options:\n");
    fprintf(stderr, "-h, --help: This usage information\n\n");
    fprintf(stderr, "-t N, --test N: Duration in seconds of the test\n\n");
    fprintf(stderr, "Examples:\n");
    fprintf(stderr, "LDecoder video0.h264\n");
    fprintf(stderr, "\tdecode from the file video0.h264\n");
    fprintf(stderr, "LDecoder /tmp/video0.fifo\n");
    fprintf(stderr, "\tdecode from a named pipe (fifo) /tmp/video0.fifo\n");
    fprintf(stderr, "\n");
}

/******************************************************************************\

Function:       get_szstatusstr

Description:    Utility function to get the LStatus string.

Parameters:     eStatus     LStatus to convert to string.

Return Value:   None

Comments:       
 
\******************************************************************************/
const MCHAR8* get_szstatusstr(
                        LStatus eStatus)
{
    switch (eStatus)
    {
        case LStatus_SEQUENCE_RESET:                     return "SEQUENCE_RESET";
        case LStatus_DISORDERED_RELEASE_COUNT:           return "DISORDERED_RELEASE_COUNT";
        case LStatus_CRITICAL_END_OF_STREAM:             return "CRITICAL_END_OF_STREAM";
        case LStatus_STREAM_NOT_INITIALIZED:             return "STREAM_NOT_INITIALIZED";
        case LStatus_INCOMPLETE_COMMAND:                 return "INCOMPLETE_COMMAND";
        case LStatus_REORDERED_SEQUENCE_PENDING:         return "REORDERED_SEQUENCE_PENDING";
        case LStatus_END_OF_STREAM:                      return "END_OF_STREAM";
        case LStatus_INFO_NOT_AVAILABLE:                 return "INFO_NOT_AVAILABLE";
        case LStatus_TEMPORARY_LACK_OF_RESOURCES:        return "TEMPORARY_LACK_OF_RESOURCES";
        case LStatus_HARDWARE_MALFUNCTION:               return "HARDWARE_MALFUNCTION";
        case LStatus_CONNECTION_LOST:                    return "CONNECTION LOST";
        case LStatus_COMMAND_PENDING:                    return "COMMAND PENDING";
        case LStatus_CANCELED:                           return "CANCELED";
        case LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL:    return "DEVICETHREAD_COMMAND_QUEUE_FULL";
        case LStatus_UNSUPPORTED:                        return "UNSUPPORTED";
        case LStatus_ACCESS_DENIED:                      return "ACCESS_DENIED";
        case LStatus_RESOURCES_BUSY:                     return "RESOURCES_BUSY";
        case LStatus_OUT_OF_RESOURCES:                   return "OUT_OF_RESOURCES";
        case LStatus_OUT_OF_MEMORY:                      return "OUT_OF_MEMORY";
        case LStatus_NO_MORE_DATA:                       return "NO_MORE_DATA";
        case LStatus_TIMEOUT:                            return "TIMEOUT";
        case LStatus_INVALID_PARAM:                      return "INVALID_PARAM";
        case LStatus_FAIL:                               return "FAIL";
        case LStatus_OK:                                 return "OK";
        case LStatus_NOT_OPTIMAL:                        return "NOT_OPTIMAL";
        case LStatus_OK_INCOMPLETE_REORDERED_SEQUENCE:   return "INCOMPLETE_REORDERED_SEQUENCE";
        case LStatus_OK_MISSING_FIRST_SPS:               return "LStatus_OK_MISSING_FIRST_SPS";
        case LStatus_OK_STREAM_NOT_LOWLATENCY:           return "LStatus_OK_STREAM_NOT_LOWLATENCY";
    }
    return " UNKNOWN";
}

/******************************************************************************\

Function:       get_options

Description:    Utility function to get the command line options into global 
                variables

Parameters:     argc,
                argv

Return Value:   None

Comments:       
 
\******************************************************************************/
void get_options(int argc, char** argv)
{
  #if _WIN32
    char* optarg;
  #else
    static struct option long_options[] =
    {
        {"help", no_argument,       0, 'h'},
        {"test", required_argument, 0, 't'},
        {0, 0, 0, 0}
    };
  #endif  

    int argf = 0;
    int argi = 1;
    int c;

    do
    {
      #if _WIN32
        c = -1;

        while (argi < argc)
        {
            if (strcmp(argv[argi], "-h") == 0)
            {
                c = 'h';
                argi++;
                break;
            }
            else if (strcmp(argv[argi], "-t") == 0)
            {
                c = 't';
                argi++;
                optarg = argi < argc ? argv[argi] : "0";
                argi++;
                break;
            }
            else
            {
                c = -1;
                argi++;
                break;
            }
        }
      #else
        c = getopt_long(argc, argv, "ht:", long_options, &argi);
      #endif
        
        if (c != -1)
        {
            argf++;
            
            switch (c)
            {
                case ':':
                case '?':
                case 'h':
                default:
                    // help or an unknown option
                    show_usage();
                    exit(0);
                    
                case 't':
                    uiTestDuration = atoi(optarg); argf++;
                    break;
            }
        }                    

    } while (c != -1);

    if (argc == 2 + argf)
    {
        szFileName = argv[1 + argf];
    }
}

/******************************************************************************\

Function:       is_stop_required

Description:    Utility function to return MTRUE if 'Enter' is pressed on the
                keyboard and a line of data is available from stdin.

Parameters: 

Return Value:   MTRUE, MFALSE

Comments:       Pressing 'Enter' will stop encoding.
 
\******************************************************************************/
MBOOL is_stop_required()
{
  #if _WIN32
    if (uiTestDuration != 0 && LDecoder_GetTimeSec() > uiTestStartTime + uiTestDuration)
    {
        return MTRUE;
    }

    return _kbhit();
  #else
    int           poll_status;
    struct pollfd poll_info;

    if (uiTestDuration != 0 && LDecoder_GetTimeSec() > uiTestStartTime + uiTestDuration)
    {
        return MTRUE;
    }
    
    poll_info.fd = 0;
    poll_info.events = POLLIN;

    poll_status = poll(&poll_info, 1, 0);
    
    if (poll_status > 0)
    {
        return MTRUE;
    }
    else
    {        
        return MFALSE;
    }
  #endif
}

/******************************************************************************\

Function:       CpuThread_Create

Description:    Utility function to create a CPU thread using the OS services.

Parameters:     fnThread
                pvRef
                phThread

Return Value:   LStatus_OK, LStatus_FAIL

Comments:

\******************************************************************************/
LStatus CpuThread_Create(

    CpuThread_Function fnThread, 
    void*              pvRef, 
    CpuThread_Handle*  phThread)
{
    pthread_t* poThread = (pthread_t*) malloc(sizeof(pthread_t));

    if (poThread != MNULL)
    {
        if (pthread_create(poThread, MNULL, (PosixThread_Function)fnThread, pvRef) != 0)
        {
            free(poThread);
            poThread = MNULL;
        }
    }

    *phThread = (CpuThread_Handle)poThread;

    return *phThread ? LStatus_OK : LStatus_FAIL;
}

/******************************************************************************\

Function:       CpuThread_Destroy

Description:    Utility function to destroy a CPU thread using the OS services.

Parameters: 

Return Value:   LStatus_OK, LStatus_FAIL

Comments:       

\******************************************************************************/
LStatus CpuThread_Destroy(CpuThread_Handle hThread)
{
    if (hThread != MNULL)
    {
        if (pthread_join(*((pthread_t*)hThread), MNULL) != 0)
        {
            return LStatus_FAIL;
        }

        free(hThread);

        return LStatus_OK;
    }
    else
    {
        return LStatus_FAIL;
    }
}

/******************************************************************************\

Function:       open_fifo_input

Description:    Open the input file which may be a named pipe or a regular file

Parameters:     name

Return Value:   -1 on error, file descriptor >= 0 otherwise

Comments:       

\******************************************************************************/
int open_fifo_input(char* name)
{
  #if _WIN32
    HANDLE  hFipe;

    hFipe = CreateFile(name, GENERIC_READ, 0, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL | FILE_FLAG_OVERLAPPED, NULL);

    return (MINT64)hFipe;
  #else
    int fd = -1;
    
    struct sigaction sa = { { 0 } };
    sa.sa_handler = SIG_IGN;
    sa.sa_flags = 0;
    
    if (sigaction(SIGPIPE, &sa, 0) == -1) 
    {
        return -1;
    }

    fd = open(name, O_RDONLY | O_NONBLOCK | O_SYNC);

    if (fd == -1)
    {
        return -1;
    }

    struct stat st;
    
    if (fstat(fd, &st) == 0)
    {
        bFileIsFifo = S_ISFIFO(st.st_mode);

        if (bFileIsFifo )
        {
            fprintf(stderr, "... decoding from a named pipe (fifo)\n");
        }
    }

    return fd;
  #endif
}

/******************************************************************************\

Function:       close_fifo_input

Description:    Close the input file

Parameters:     fd

Return Value:   

Comments: 

\******************************************************************************/
void close_fifo_input(MINT64 fd)
{
  #if _WIN32
    CloseHandle((HANDLE)fd);
  #else
    if (fd >= 0)
    {
        close(fd);
    }
  #endif
}

/******************************************************************************\

Function:       read_fifo_input

Description:    Read a packet of data from the input

Parameters:     fd
                packet_size
                packet_data

Return Value:   -1 on error, 1 on end-of-stream, 0 otherwise

Comments: 

\******************************************************************************/
int read_fifo_input(MINT64 fd, size_t* packet_size, MUINT8* packet_data)
{
  #if _WIN32
    int    status = 0;
    DWORD  read_size = 0;
    DWORD  current_packet_size = (*packet_size); // to read
    DWORD  initial_packet_size = (*packet_size); // to keep
    BOOL   bSuccess;

    while (MTRUE)
    {
        bSuccess = ReadFile((HANDLE)fd, packet_data, *packet_size, &read_size, NULL);
       
        if (!bSuccess)
        {
            status = -1;
            break;
        }
        else if (read_size == 0 && !bFileIsFifo)
        {
            status = 1;
            break;
        }
        else
        {
            packet_data         += read_size;
            current_packet_size -= read_size;

            break;  // return as soon as some data is available
        }
    }

    (*packet_size) = initial_packet_size - current_packet_size;

    return status;
  #else
    int    status              = 0;
    size_t read_size           = 0;
    size_t current_packet_size = (*packet_size); // to read
    size_t initial_packet_size = (*packet_size); // to keep

    while (MTRUE)
    {
        read_size = read(fd, packet_data, current_packet_size);

        if (read_size == -1)
        {
            if (errno == EPIPE || errno == EAGAIN)
            {
                int poll_status;
                
                struct pollfd poll_info;
                poll_info.fd = fd;
                poll_info.events = POLLIN;

                poll_status = poll(&poll_info, 1, 1);

                if (is_stop_required())
                {
                    fprintf(stderr, "exit request in progress ...\n");
                    status = 1;
                    break;
                }
                
                assert(poll_status == 0 || poll_status == 1 || poll_status == -1);

                if (poll_status == -1)
                {
                    status =  -1;
                    break;
                }
            }
            else
            {
                status =  -1;
                break;
            }
        }
        else if (read_size == 0 && !bFileIsFifo)
        {
            status = 1;
            break;
        }
        else
        {
            packet_data         += read_size;
            current_packet_size -= read_size;
            
            break;  // return as soon as some data is available
        }
    }

    (*packet_size) = initial_packet_size - current_packet_size;

    return status;
  #endif
}

/******************************************************************************\

Function:       ReceiveNalu

Description:    

Parameters:     fd_fifo_input
                poNaluInfo

Return Value:

Comments: 

\******************************************************************************/
LStatus ReceiveNalu(MINT64 fd_fifo_input, LH264E_NaluInfo* poNaluInfo)
{
    LStatus eStatus = LStatus_OK;

    size_t  uiPacketSize = poNaluInfo->uiSize;
    MUINT8* puiPacketData = MNULL;

    eStatus = LBuffer_BeginAccess((LBuffer_Handle)poNaluInfo->hBuffer, 0, 1, &puiPacketData);
    
    if (eStatus != LStatus_OK)
    {
        return eStatus;
    }
    
    puiPacketData += poNaluInfo->uiStartOffset;
    
    int error = read_fifo_input(fd_fifo_input, &uiPacketSize, puiPacketData);

    if (error == -1)
    {
        eStatus = LStatus_FAIL;
    }
    else if (error == 1)
    {
        eStatus = LStatus_END_OF_STREAM;
    }
    else
    {
        eStatus = LStatus_OK;
    }

    poNaluInfo->uiSize = uiPacketSize;
    
    LBuffer_EndAccess((LBuffer_Handle)poNaluInfo->hBuffer);

    return eStatus;
}

/******************************************************************************\

Function:       StreamingThread

Description:    CPU thread responsible to pull the encoded data from the 
                encoder fifo and to send it to the decoder.

Parameters: pvRef pointer to Decoder's handle

Return Value:


Comments: Thread end with on the last packet from the decoder
 
\******************************************************************************/
LStatus StreamingThread(void *pvRef)
{
    LStatus           eStatus                       = LStatus_OK;
    MINT64            fd_fifo_input                 = -1;
    LBuffer_Handle    ahBuffer[NB_STREAM_BUFFER]    = { MNULL };
    MUINT64           ahBufferTag[NB_STREAM_BUFFER] = {   0   };
    MUINT64           ahBufferLUID[NB_STREAM_BUFFER] = {   0   };
    MUINT             uiCurrentBuffer               = 0;
    CpuThread_Data*   poThreadData                  = (CpuThread_Data*)pvRef;
    MUINT             i                             = 0;

    LH264E_NaluInfo   oNaluInfo;
    LH264D_DecodeOpt  oDecodeOpt;

    poThreadData->bThreadRunning                    = MTRUE;

    fprintf(stderr, "LDecoder streaming thread starting\n");

    fd_fifo_input = open_fifo_input(szFileName);
    
    if (fd_fifo_input == -1)
    {
        // We got an error so we end the stream
        do
        {
            eStatus = LH264D_EndStream(poThreadData->hDecoder);
            
        } while (eStatus == LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL);

        if (eStatus != LStatus_OK)
        {
            fprintf(
                stderr,
                "LH264D_EndStream failed %d (%s)\n",
                eStatus,
                get_szstatusstr(eStatus));
            exit(1);
        }
        poThreadData->bThreadRunning = MFALSE;
        return LStatus_FAIL;
    }

    // Create the buffers for receiving the NALU packets
    for (i = 0; i < NB_STREAM_BUFFER; i++)
    {
        ahBufferLUID[i] = LDEVICETHREAD_DEFAULT_LUID;
        
        eStatus = LBuffer_CreateLinear(poThreadData->hDevice, NB_STREAM_BUFFER_SIZE, &ahBuffer[i]);

        if (eStatus != LStatus_OK)
        {
            close_fifo_input(fd_fifo_input);

            // We got an error so we end the stream
            do
            {
                eStatus = LH264D_EndStream(poThreadData->hDecoder);
                
            } while (eStatus == LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL);

            if (eStatus != LStatus_OK)
            {
                fprintf(
                    stderr,
                    "LH264D_EndStream failed %d (%s)\n",
                    eStatus,
                    get_szstatusstr(eStatus));
                    exit(1);
            }
            poThreadData->bThreadRunning = MFALSE;
            return LStatus_FAIL;
        }
    }

    while (poThreadData->bThreadRunning)
    {
        if (is_stop_required())
        {
            fprintf(stderr, "exit request in progress ...\n");
            break;
        }

        if (ahBufferLUID[uiCurrentBuffer] != LDEVICETHREAD_DEFAULT_LUID)
        {
            eStatus = LDeviceThread_WaitCompletionTag(
                poThreadData->hDTInputDecoder,
                ahBufferLUID[uiCurrentBuffer],
                ahBufferTag[uiCurrentBuffer],
                10000); // 10 sec
                        
            if (eStatus != LStatus_OK)
            {
                fprintf(
                    stderr,
                    "LDeviceThread_WaitCompletionTag failed %d (%s)\n",
                    eStatus,
                    get_szstatusstr(eStatus));
                break;
            }
        }
        
        // Get a new NALU from the network.
        memset(&oNaluInfo, 0, sizeof(oNaluInfo));
        oNaluInfo.eType = LH264E_NaluInfoTypeHeader_STANDARD;
        oNaluInfo.hBuffer = (MUINT64)ahBuffer[uiCurrentBuffer];
        oNaluInfo.uiStartOffset = 0;
        oNaluInfo.uiSize = NB_STREAM_BUFFER_SIZE;

        eStatus = ReceiveNalu(
            fd_fifo_input,
            &oNaluInfo);

        // Post a decode command.
        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            if (oNaluInfo.uiSize != 0)
            {
                #if defined(TRACE_NALU_SIZE)
                fprintf(stderr, "nalu %d\n", oNaluInfo.uiSize);
                #endif
            
                MUINT32 uiBytesConsumed    = 0;
                MUINT32 uiSumBytesConsumed = 0;
                
                memset(&oDecodeOpt, 0, sizeof(oDecodeOpt));
                oDecodeOpt.eType = LH264D_DecodeOptTypeHeader_STANDARD;
                    
                do
                {
                    // Set a NALU blob.
                    oDecodeOpt.hBuffer       = oNaluInfo.hBuffer;
                    // Set the offset where the encoded data begins.
                    oDecodeOpt.uiStartOffset = oNaluInfo.uiStartOffset + uiSumBytesConsumed;
                    // Set the size of the current NALU blob.
                    oDecodeOpt.uiSize        = oNaluInfo.uiSize - uiSumBytesConsumed;

                    // Decode picture.
                    eStatus = LH264D_DecodeNALBlob(
                        poThreadData->hDecoder,
                        (LH264D_DecodeOptTypeHeader*)&oDecodeOpt.eType,
                        &uiBytesConsumed);
                        
                    if (eStatus == LStatus_INCOMPLETE_COMMAND ||
                        eStatus == LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL)
                    {
                        uiSumBytesConsumed += uiBytesConsumed;
                    }
                    else if (eStatus == LStatus_OK ||
                             eStatus == LStatus_OK_MISSING_FIRST_SPS)
                    {
                        uiSumBytesConsumed += uiBytesConsumed;
                    }
                    else if (eStatus == LStatus_END_OF_STREAM)
                    {
                        break;
                    }
                    
                } while (uiSumBytesConsumed != oNaluInfo.uiSize);

                LDeviceThread_Handle hDecInternalDT = MNULL;
                eStatus = LH264D_GetDTInputLastSubmissionTag(
                                        poThreadData->hDecoder,
                                        &ahBufferTag[uiCurrentBuffer],
                                        &hDecInternalDT);
                LDeviceThread_GetLUID(hDecInternalDT, &ahBufferLUID[uiCurrentBuffer]);

                uiCurrentBuffer = (uiCurrentBuffer + 1) % NB_STREAM_BUFFER;            
            }
        }
        else if (eStatus == LStatus_END_OF_STREAM)
        {
            // We reach the end of the stream, then we must send the
            // EndStream command to empty the decode pipeline.
            break;
        }
        else
        {
            // We got an error so we end the stream
            break;
        }
    }

    for (i = 0; i < NB_STREAM_BUFFER; i++)
    {
        eStatus = LDeviceThread_WaitCompletionTag(
                    poThreadData->hDTInputDecoder,
                    ahBufferLUID[i],
                    ahBufferTag[i],
                    1000); // 1 sec

        if (eStatus != LStatus_OK)
        {
            fprintf(
                stderr,
                "LDeviceThread_WaitCompletionTag failed %d (%s)\n",
                eStatus,
                get_szstatusstr(eStatus));
            break;
        }
                        
        LBuffer_Destroy(ahBuffer[i]);
    }

    do
    {
        eStatus = LH264D_EndStream(poThreadData->hDecoder);
        
    } while (eStatus == LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL);

    if (eStatus != LStatus_OK)
    {
        fprintf(
            stderr,
            "LH264D_EndStream failed %d (%s)\n",
            eStatus,
            get_szstatusstr(eStatus));
        exit(1);
    }

    close_fifo_input(fd_fifo_input);

    fprintf(stderr, "LDecoder streaming thread terminating\n");

    poThreadData->bThreadRunning = MFALSE;

    return LStatus_OK;
}

// Decoder

// This sample demonstrates how to decode an incoming stream and display it. 
// For simplification, we assume infinite input buffering and data are always 
// available from the byte-stream. 

// The code uses two CPU threads. One handles the decoder input and the other 
// the decoder output. This is required to avoid potential deadlock because 
// input and output aren't necessary one to one match (i.e. we can have more 
// NALU blobs than pictures, or the opposite). Also, both the input and the 
// output aren't synchronized, such that more than one NALU can be sent to 
// the decoder before it returns a single decoded picture.

// Detect display

// This is how we get information about connectors and connected monitors. 
// We must first detect all monitors before getting information. This code 
// displays incoming pictures on the first connector with a connected monitor.

/******************************************************************************\

Function:       main

Description:    LDecoder application entry point.

Parameters:     None.

Return Value:   0 on success, -1 otherwise.

Comments:                       

\******************************************************************************/
MINT32 main(int argc, char** argv)
{
    MINT32                  iResult             = -1;
    LStatus                 eStatus             = LStatus_FAIL;
    LDevice_Handle          hDevice             = MNULL;
    LVideoOut_Handle        hVideoOut           = MNULL;
    MBOOL32                 bFoundValidOutput   = MFALSE;
    MUINT32                 uiConnectorIndex    = 0;
    CpuThread_Handle        hStreamingThread    = MNULL;    
    MUINT                   i                   = 0;
    MUINT                   uiPictureErrors     = 0;
    
    LBoard_ConnectorInfo    oConnectorInfo;
    LBoard_MonitorInfo      oMonitorInfo;
    CpuThread_Data          oThreadData;
    LVideo_VidParam         oVidParam;

    get_options(argc, argv);

    fprintf(stderr, "LDecoder starting\n");
    fprintf(stderr, "... press 'Enter' to quit\n");
    fprintf(stderr, "... decoding from %s\n", szFileName);
    
    if (uiTestDuration != 0)
    {
        fprintf(stderr, "... for %d seconds\n", uiTestDuration);
    }
    
    Liberatus_Load();

    memset(&oThreadData, 0, sizeof(oThreadData));
    
    hDevice = Liberatus_GetDevice(0);

    LBoard_DetectAllMonitors(hDevice);
   
    // Search for the first video output connector.
    do
    {
        memset(&oConnectorInfo, 0, sizeof(oConnectorInfo));
        oConnectorInfo.uiSize = sizeof(oConnectorInfo);
        
        eStatus = LBoard_GetConnectorInfo( 
            hDevice,
            uiConnectorIndex,
            &oConnectorInfo);

        if (LSTATUS_IS_SUCCESS(eStatus) == MFALSE)
        {
            break; // No more connector
        }
        else if (oConnectorInfo.bVideoOutput)
        {
            memset(&oMonitorInfo, 0, sizeof(oMonitorInfo));
            oMonitorInfo.uiSize = sizeof(oMonitorInfo);
            
            LBoard_GetMonitorInfo(hDevice, uiConnectorIndex, &oMonitorInfo);

            eStatus = LVideoOut_GetHandle( 
                hDevice,
                oConnectorInfo.uiVideoOutIndex,
                LAccessMode_READWRITE_EXCLUSIVE,
                &hVideoOut);

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                bFoundValidOutput = MTRUE;
            }
            
            // Optimal video parameters required
            memset(&oVidParam, 0, sizeof(oVidParam));
            
            eStatus = LVideoOut_GetOptimalVideoParameters(
                hVideoOut,
                LVideo_SignalType_USEDEFAULT,
                &oVidParam);

            if (LSTATUS_IS_FAIL(eStatus))
            {
                fprintf(
                    stderr,
                    "LVideoOut_GetOptimalVideoParameters failed %d (%s)\n",
                    eStatus,
                    get_szstatusstr(eStatus));
                return -1;
            }
        }
        
        ++uiConnectorIndex;
           
    } while (!bFoundValidOutput);

    // Initialize display and engines
    
    // This code applies 1080p60 video parameters with discrete timing. The 
    // decoder uses two (2) device threads: a first one to send NALUs to the 
    // decoder, and a second one to process the decoded picture and release 
    // the decoder buffers. The example below doesn't demonstrate how to 
    // retrieve the EDID. For more information on EDID retrieval, see the 
    // GetEDIDSize() and GetEDID() API functions in the Matrox Liberatus Board 
    // Module specification.
    
    // Enable display.
    if (bFoundValidOutput)
    {
        MUINT32                  uiSelectBuffer                = 0;
        // This device thread is used to send the decode commands to the decoder.
        LDeviceThread_Handle     hDTInputDecoder               = MNULL; 
        // This device thread is used to release the decoded pictures of the decoder.
        LDeviceThread_Handle     hDTOutputDecoder              = MNULL; 
        LVPE_Handle              hVpe                          = MNULL;
        LBuffer_Handle           hVpeBuffer[NB_DISPLAY_BUFFER] = {MNULL};
        LH264D_Handle            hDecoder                      = MNULL;

        LVP_ColorSpace           oColorSpace;
        LBuffer_VideoAttributes  oVideoAttributes;
        LVP_Rect                 oVpeRect;
        LH264D_CreateOpt         oCreateOpt;
        LH264D_PictureInfo       oPictureInfo;
        
        fprintf(stderr, "Display parameters being used w:%d, h:%d, Hz:%d\n", oVidParam.uiHActive, oVidParam.uiVActive, oVidParam.uiVRate_Hz);
        
        // Create device-threads
        LDeviceThread_Create(hDevice, MNULL, &hDTInputDecoder);
        LDeviceThread_Create(hDevice, MNULL, &hDTOutputDecoder);

        // Create the VPE.
        // Parameters are set to their default values at creation.
        // These default values are set for progressive buffers.
        // Source and target color spaces are set for HD buffers (Buffers
        // family can be R'G'B' with gamma based on Rec. ITU-R BT.709-5 or
        // Y'CbCr with transfer matrix coefficients based on Rec. ITU-R
        // BT.709-5).
        // The scaler filtering is in automatic mode (chosen based on the
        // scaling factor).
        // The bayer filter, deinterlacer module, and advanced enhancement
        // filtering are disabled by default.
        LVPE_Create(hDevice, hDTOutputDecoder, MNULL, &hVpe);

        memset(&oColorSpace, 0, sizeof(oColorSpace));
        
        // Set source and target color space.
        // R'G'B' with gamma based on Rec. ITU-R BT.709-5 (default).
        oColorSpace.eFunction = LVP_TransferFunction_BT709;
        // Y'CbCr transfer matrix with coefficients based on
        // Rec. ITU-R BT.709-5 (HD)(default).
        oColorSpace.eMatrix   = LVP_TransferMatrix_BT709;
        // No user defined matrices.
        oColorSpace.poUserMatrix = MNULL;

        LVPE_SetSourceParam(
            hVpe,
            LVPE_Source_COLOR_SPACE,
            sizeof(LVP_ColorSpace),
            &oColorSpace);

        LVPE_SetTargetParam(
            hVpe,
            LVPE_Target_COLOR_SPACE,
            sizeof(LVP_ColorSpace),
            &oColorSpace);

        // Enable the source rectangle and crop it to 1080 lines to remove
        // unused lines caused by 16x16 H264 macro block alignment constrain.
        // Target rectangle is using default value (disable) and doesn't need to
        // be set. This means that a specified source is scaled to fit the size
        // specified in the target buffer.
        memset(&oVpeRect, 0, sizeof(oVpeRect));
        
        oVpeRect.bEnable            = MTRUE;
        oVpeRect.oRect.iLeft        = 0;
        oVpeRect.oRect.iRight       = 1920;
        oVpeRect.oRect.iTop         = 0;
        oVpeRect.oRect.iBottom      = 1080;

        LVPE_SetSourceParam(
            hVpe,
            LVPE_Source_SOURCE_RECT,
            sizeof(oVpeRect),
            &oVpeRect);

        // Init decoder.
        memset(&oCreateOpt, 0, sizeof(oCreateOpt));
        
        // We use standard options for the creation of the decoder context.
        oCreateOpt.eType = LH264D_CreateOptTypeHeader_STANDARD;

        // Create input device thread, used to send commands to the decoder.
        oCreateOpt.hDTInput         = (MUINT64)hDTInputDecoder; 
        // Create output device thread, used to release the pictures to the decoder.
        oCreateOpt.hDTOutput        = (MUINT64)hDTOutputDecoder;
        // Optimize the decoder for normal latency.
        oCreateOpt.eDecoderMode     = LH264D_DecoderMode_NORMAL;       
        // Set the normal priority.
        oCreateOpt.ePriority        = LH264_Priority_NONREALTIME;   

        // At the decoder creation a decoder handle is returned to identify
        // the current stream.
        eStatus = LH264D_Create(
            hDevice,
            (LH264D_CreateOptTypeHeader*)&oCreateOpt.eType,
            &hDecoder);

        if (eStatus != LStatus_OK)
        {
            fprintf(
                stderr,
                "LH264D_Create failed %d (%s)\n",
                eStatus,
                get_szstatusstr(eStatus));
            return -1;
        }

        oThreadData.hDecoder        = hDecoder;
        oThreadData.hDevice         = hDevice;
        oThreadData.hDTInputDecoder = hDTInputDecoder;

        memset(&oVideoAttributes, 0, sizeof(oVideoAttributes));
        oVideoAttributes.eAttributeType = LBuffer_Type_VIDEO;
        oVideoAttributes.ePixelFormat   = LPixelFormat_B8G8R8A8;
        oVideoAttributes.uiWidth        = oVidParam.uiHActive;
        oVideoAttributes.uiHeight       = oVidParam.uiVActive;

        for (i = 0; i < NB_DISPLAY_BUFFER; i++)
        {
            eStatus = LBuffer_Create(
                hDevice,
                &oVideoAttributes.eAttributeType,
                &hVpeBuffer[i]);
                
            if (eStatus != LStatus_OK)
            {
                fprintf(
                    stderr,
                    "LBuffer_Create failed %d (%s)\n",
                    eStatus,
                    get_szstatusstr(eStatus));
                return -1;
            }
        }

        // Fill hVpeBuffer[NB_DISPLAY_BUFFER-1], the first display buffer, with black.
        {
            LBlit_Handle         hBlit_Handle   = MNULL;
            LColor               oColor;

            memset(&oColor, 0, sizeof(oColor));
            oColor.ePixelFormat = oVideoAttributes.ePixelFormat;
            oColor.uiRed        = 0;
            oColor.uiGreen      = 0;
            oColor.uiBlue       = 0;
            oColor.uiAlpha      = 0;

            LBlit_Create (hDevice, hDTInputDecoder, MNULL, &hBlit_Handle);

            LBlit_SolidFill(
                hBlit_Handle,
                hVpeBuffer[NB_DISPLAY_BUFFER-1],
                &oColor,
                MNULL,
                MNULL,
                LBlit_Bop_S,
                MNULL);

            LDeviceThread_WaitCompletionTag(
                hDTInputDecoder,
                LDEVICETHREAD_DEFAULT_LUID,
                LDEVICETHREAD_DEFAULT_TAG,
                LINFINITE_TIMEOUT);
                
            LBlit_Destroy(hBlit_Handle);
        }

        // Init video output.
        LVideoOut_Enable(
            hVideoOut,
            LVideo_SignalType_USEDEFAULT,
            &oVidParam,
            hVpeBuffer[NB_DISPLAY_BUFFER-1],
            LVideoOut_PowerState_ON);
                            
        // Start streaming thread.
        CpuThread_Create( 
            StreamingThread,
            (void*)&oThreadData,
            &hStreamingThread);                            
            
        // Start decoding
        
        // This reads incoming encoded data from the byte-stream for decoding and 
        // displaying. At the beginning of the stream, we should get information 
        // (such as cropping information) to initialize the video processor.

        // Decoding will stop after we decode an end of stream or the 'Enter' key is pressed.

        uiSelectBuffer     = 0;
        eStatus            = LStatus_OK;
 
        uiTestStartTime = LDecoder_GetTimeSec();
        
        do
        {
            LStatus eExecStatus = LStatus_OK;

            /********************************
             *   DECODER OUTPUT HANDLING    *
             *******************************/

            // Fetch a decoded picture.
            memset(&oPictureInfo, 0, sizeof(oPictureInfo));
            
            // We use standard structure for picture information.
            oPictureInfo.eType = LH264D_PictureInfoTypeHeader_STANDARD;
            
            eStatus = LH264D_GetNextPicture(
                hDecoder, 
                (LH264D_PictureInfoTypeHeader*)&oPictureInfo.eType, 
                0);

            if (eStatus != LStatus_OK && 
                eStatus != LStatus_TIMEOUT && 
                eStatus != LStatus_STREAM_NOT_INITIALIZED &&
                eStatus != LStatus_END_OF_STREAM)
            {
                fprintf(
                    stderr,
                    "LH264D_GetNextPicture failed %d (%s)\n",
                    eStatus,
                    get_szstatusstr(eStatus));
                return -1;
            }
            
            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                // Check errors from picture info
                if ((oPictureInfo.flPictureStatusFlags & LH264D_PictureStatusFlag_FATALFIRMWAREERROR) ||
                    (oPictureInfo.flPictureStatusFlags & LH264D_PictureStatusFlag_FATALSTREAMERROR)   ||
                    (oPictureInfo.flPictureStatusFlags & LH264D_PictureStatusFlag_MISSINGDATA)        ||
                    (oPictureInfo.flPictureStatusFlags & LH264D_PictureStatusFlag_CORRUPTEDDATA)      ||
                    (oPictureInfo.flPictureStatusFlags & LH264D_PictureStatusFlag_UNSUPPORTEDINPUT))
                {
                    fprintf(stderr, "LH264D_GetNextPicture returned fatal error flags 0x%x\n", oPictureInfo.flPictureStatusFlags);
                    
                    uiPictureErrors++;
                    
                    if (uiPictureErrors >= 4)
                    {
                        return -1;
                    }
                }

                if ((oPictureInfo.flPictureStatusFlags & LH264D_PictureStatusFlag_PICTURESSKIPPED))
                {
                    fprintf(stderr, "LH264D_GetNextPicture returned that pictures have been skipped\n");
                }

                if (oPictureInfo.hBuffer == 0)
                {
                    fprintf(stderr, "LH264D_GetNextPicture returned without picture\n");
                    
                    do
                    {
                        eExecStatus = LH264D_ReleasePicture(
                            hDecoder, 
                            (LH264D_PictureInfoTypeHeader*)&(oPictureInfo.eType));
                        
                    } while (eExecStatus == LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL);

                    if (eExecStatus != LStatus_OK)
                    {
                        fprintf(
                            stderr,
                            "LH264D_ReleasePicture failed %d (%s)\n",
                            eExecStatus,
                            get_szstatusstr(eExecStatus));
                        return -1;
                    }
                    
                    continue;
                }

                // Read the sequence of information (such as cropping parameters)
                // at the beginning of the stream.
                if ((oPictureInfo.flPictureInfoFlags & LH264D_PictureInfoFlag_STREAMINFOFLAGSPRESENT) &&
                    (oPictureInfo.flStreamInfoFlags  & LH264D_StreamInfoUpdateFlag_GENDATA))
                {
                    LH264_StreamInfoDecoderGenData oDecoderGenData;
                    
                    memset(&oDecoderGenData, 0, sizeof(oDecoderGenData));
                    oDecoderGenData.eType = LH264_StreamInfoTypeHeader_DECODERGENDATA;
                    
                    // Fetch the data for the cropping to set the right source
                    // rectangle during the picture processing.
                    if (LSTATUS_IS_SUCCESS(
                            LH264D_GetStreamInfo(
                                hDecoder,
                                (LH264_StreamInfoTypeHeader*)&oDecoderGenData.eType)) &&
                            (oDecoderGenData.flDecoderGenDataFlags &
                                LH264_StreamInfoDecoderGenDataFlag_PICTURECROPPEDRECTANGLEPRESENT))
                    {
                        oVpeRect.bEnable       = MTRUE;
                        oVpeRect.oRect.iLeft   = 2*oDecoderGenData.oPictureCroppedRect.iLeft;
                        oVpeRect.oRect.iRight  = oDecoderGenData.uiPictureWidth - 2*oDecoderGenData.oPictureCroppedRect.iRight;
                        oVpeRect.oRect.iTop    = 2*oDecoderGenData.oPictureCroppedRect.iTop;
                        oVpeRect.oRect.iBottom = oDecoderGenData.uiPictureHeight - 2*oDecoderGenData.oPictureCroppedRect.iBottom;

                        LVPE_SetSourceParam(
                            hVpe,
                            LVPE_Source_SOURCE_RECT,
                            sizeof(oVpeRect),
                            &oVpeRect);
                    }
                }

                // Start the execution of the video processor engine to convert
                // to displayable buffer.
                eExecStatus = LVPE_ExecuteFrame(
                    hVpe,
                    hVpeBuffer[uiSelectBuffer],
                    LVPE_CurrentFrameOrField_FRAME,         // Picture is frame based.
                    MNULL,                                  // previous source
                    (LBuffer_Handle)oPictureInfo.hBuffer,   // current  source
                    MNULL);                                 // next     source

                if (eExecStatus != LStatus_OK)
                {
                    fprintf(
                        stderr,
                        "LVPE_ExecuteFrame failed %d (%s)\n",
                        eExecStatus,
                        get_szstatusstr(eExecStatus));
                    return -1;
                }

                // The current picture must be released after we've finished
                // with it. This returns the memory to the decoder to be reused
                // for new decoded pictures.
                do
                {
                    eExecStatus = LH264D_ReleasePicture(
                        hDecoder, 
                        (LH264D_PictureInfoTypeHeader*)&(oPictureInfo.eType));

                } while (eExecStatus == LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL);

                if (eExecStatus != LStatus_OK)
                {
                    fprintf(
                        stderr,
                        "LH264D_ReleasePicture failed %d (%s)\n",
                        eExecStatus,
                        get_szstatusstr(eExecStatus));
                    return -1;
                }

                // Display
                // Wait for this buffer to start displaying.
                eExecStatus = LVideoOut_ScheduleSetBuffer(
                    hVideoOut,
                    hVpeBuffer[uiSelectBuffer],
                    MTRUE,   // Top field
                    hDTOutputDecoder,
                    10000,   // 10sec
                    LVideoOut_FlipType_NEXT_FIELD_WAIT);

                if (eExecStatus != LStatus_OK)
                {
                    fprintf(
                        stderr,
                        "LVideoOut_ScheduleSetBuffer failed %d (%s)\n",
                        eExecStatus,
                        get_szstatusstr(eExecStatus));
                    return -1;
                }

                // Select the next buffer.
                uiSelectBuffer = (uiSelectBuffer + 1) % NB_DISPLAY_BUFFER;
            }
            
        } while (eStatus != LStatus_END_OF_STREAM); 
                
        // When this condition is reached, it means there's no new data and the
        // decoder pipeline has been emptied.

        // Release and destroy
        
        while (oThreadData.bThreadRunning)
        {
            // The input thread will run until it gets a END_OF_STREAM return
            // value, which means the encoder pipeline is now empty.
        }
        
        CpuThread_Destroy(hStreamingThread);

        eStatus = LDeviceThread_WaitCompletionTag(
            hDTInputDecoder,
            LDEVICETHREAD_DEFAULT_LUID, 
            LDEVICETHREAD_DEFAULT_TAG, 
            2000); // 2 sec

        if (eStatus != LStatus_OK)
        {
            fprintf(stderr, "Decoder input device-thread not empty after 2 sec\n");
        }

        eStatus = LDeviceThread_WaitCompletionTag(
            hDTOutputDecoder,
            LDEVICETHREAD_DEFAULT_LUID, 
            LDEVICETHREAD_DEFAULT_TAG, 
            2000); // 2 sec

        if (eStatus != LStatus_OK)
        {
            fprintf(stderr, "Decoder input device-thread not empty after 2 sec\n");
        }

        // Destroy the decoder.
        LH264D_Destroy(hDecoder);

        // Disable VideoOut
        LVideoOut_Disable(hVideoOut);
        LVideoOut_ReleaseHandle(hVideoOut);

        // Destroy VPE.
        LVPE_Destroy(hVpe);
        
        for (i = 0; i < NB_DISPLAY_BUFFER; i++)
        {
            LBuffer_Destroy(hVpeBuffer[i]);
        }
        
        LDeviceThread_Destroy(hDTInputDecoder);
        LDeviceThread_Destroy(hDTOutputDecoder);

        iResult = 0; // Success
    }

    fprintf(stderr, "LDecoder terminating\n");

    Liberatus_UnLoad();
    
    return iResult;
}
