/******************************************************************************\

Module Name:    LAudioOut.c

Description:    LAudioOut sample implementation file.

References:     LAudioOut.doc.
                Matrox Liberatus specifications

    Copyright (c) 2015, Matrox Graphics Inc.
    All Rights Reserved.

BSD 2-Clause License

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\******************************************************************************/

// --------------------------------------------------------------------------------------
//                        I N C L U D E S   A N D   U S I N G S
// --------------------------------------------------------------------------------------
#ifdef UNIX
#include <termios.h>
#include <unistd.h>
#endif

#include "Liberatus.h"
#include "LBuffer.h"
#include "LAudioOut.h"
#include "LBoard.h"

#include "stdio.h"
#include <fcntl.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

// -----------------------------------------------------------------------------------------------------------
//                                   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

typedef struct
{
    const char* szInputFilePath;
    MUINT32     uiOutputIdx;
    MUINT32     uiSamplingRate;
    MUINT32     uiNbChannels;
    MUINT32     uiSampleSize;
    MUINT32     uiSamplesPerBuf;

} MainArgs;

// --------------------------------------------------------------------------------------
//                                         C O D E
// --------------------------------------------------------------------------------------

/************************************************************************************************************\

Function:       ParseArgs

Description:    Parse application arguments.

Comments:       None.

\************************************************************************************************************/
void ParseArgs(int argc, char** argv, MainArgs* poArgs)
{
    MBOOL32 bInvalidArg = MFALSE;
    MUINT i = 0;

    poArgs->szInputFilePath     = MNULL;
    poArgs->uiOutputIdx         = 0;
    poArgs->uiSamplingRate      = 48000;
    poArgs->uiNbChannels        = 2;
    poArgs->uiSampleSize        = 16;
    poArgs->uiSamplesPerBuf     = 1024;


    for (i = 1; i < argc; i++)
    {
        if ((strcmp(argv[i], "-i") == 0) && ((i+1) < argc))
        {
            i++;
            poArgs->szInputFilePath = argv[i];
        }
        else if ((strcmp(argv[i], "-o") == 0) && ((i+1) < argc))
        {
            i++;
            poArgs->uiOutputIdx = strtoul(argv[i], MNULL, 10);
        }
        else if ((strcmp(argv[i], "-r") == 0) && ((i+1) < argc))
        {
            i++;
            poArgs->uiSamplingRate = strtoul(argv[i], MNULL, 10);
        }
        else if ((strcmp(argv[i], "-c") == 0) && ((i+1) < argc))
        {
            i++;
            poArgs->uiNbChannels = strtoul(argv[i], MNULL, 10);
        }
        else if ((strcmp(argv[i], "-s") == 0) && ((i+1) < argc))
        {
            i++;
            poArgs->uiSampleSize = strtoul(argv[i], MNULL, 10);
        }
        else if ((strcmp(argv[i], "-spb") == 0) && ((i+1) < argc))
        {
            i++;
            poArgs->uiSamplesPerBuf = strtoul(argv[i], MNULL, 10);
        }
        else
        {
            bInvalidArg = MTRUE;
            break;
        }
    }

    if(bInvalidArg
       || (poArgs->szInputFilePath == MNULL))
    {
        printf("Usage: %s <-i input_file> [-o n] [-r n] [-c n] [-s n] [-spb n]\n", argv[0]);
        printf("Mandatory arguments:\n");
        printf("\t-i str        : Input file path.\n");
        printf("Optional arguments:\n");
        printf("\t-o n          : Audio output device index (default=0).\n");
        printf("\t-r n          : Sample rate (default=48000).\n");
        printf("\t-c n          : Number of channels (default=2).\n");
        printf("\t-s n          : Sample size (bits per sample, default=16).\n");
        printf("\t-spb n        : Samples per buffer (default=1024).\n");
        exit(1);
    }
}

/************************************************************************************************************\

Function:       GetChar

Description:    Non blocking getch.

Comments:       None.

\************************************************************************************************************/
int GetChar()
{
#ifdef _WIN32
    int ch = _getch();

    /*In linux, a keypress enter is a LFeed with a value of 10. In Windows,
    The enter key calls a carriage return which has an ASCII value of 13 instead
    To avoid having to modify it at each call of getchar we go it once here so that
    it's consistant in what it expects between Windows and Linux*/
    if (ch == 13)
        ch = 10;
    return ch;
#else
    struct termios oOldAttrib;
    struct termios oNewAttrib;

    tcgetattr(STDIN_FILENO, &oOldAttrib);
    oNewAttrib = oOldAttrib;
    oNewAttrib.c_lflag &= ~(ICANON | ECHO);
    tcsetattr(STDIN_FILENO, TCSANOW, &oNewAttrib);

    int iOldFlags = fcntl(STDIN_FILENO, F_GETFL, 0);
    fcntl(STDIN_FILENO, F_SETFL, iOldFlags | O_NONBLOCK);

    int ch = getchar();

    tcsetattr(STDIN_FILENO, TCSANOW, &oOldAttrib);
    fcntl(STDIN_FILENO, F_SETFL, iOldFlags);

    return ch;
#endif

}

/************************************************************************************************************\

Function:       PrintDescriptor

Description:    Prints LAudioFormat_Descriptor.

Comments:       None.

\************************************************************************************************************/
void PrintDescriptor(LAudioFormat_Descriptor* poDescriptor)
{
    MUINT32 i = 0;

    printf("AudioFormat:        0x%08x\n"
           "Sampling rate:      %u\n"
           "Sample size:        %u\n"
           "Number of channels: %u\n"
           "Number of plans:    %u\n"
           "PCM:                %u\n",
           poDescriptor->eAudioFormat,
           poDescriptor->uiSampleRate,
           poDescriptor->uiSampleSize,
           poDescriptor->uiNumberOfChannels,
           poDescriptor->uiNumberOfPlans,
           poDescriptor->bPcmData);

    printf("Layout:");
    for(i=0; i < poDescriptor->uiNumberOfPlans; i++)
    {
        printf("             [%d]:[%08x] ", i, poDescriptor->aflPlansChannelLayout[i]);
    }
    printf("\n");
}

/************************************************************************************************************\

Function:       PrintFormat

Description:    Print LAudioFormat descriptor.

Comments:       None.

\************************************************************************************************************/
LStatus PrintFormat(LAudioFormat eAudioFormat)
{
    LAudioFormat_Descriptor oDescriptor = {LAudioFormat_INVALID};

    LStatus eStatus = LAudioFormat_GetDescriptor(eAudioFormat, &oDescriptor);

    if((eStatus == LStatus_OK)
       && (eAudioFormat == oDescriptor.eAudioFormat))
    {
        PrintDescriptor(&oDescriptor);
        eStatus = LStatus_OK;
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       GetAudioFormat

Description:    Find a supported audio format associated with the audio output device.

Comments:       None.

\************************************************************************************************************/
LStatus GetAudioFormat(
    LAudioOut_Handle    hAudioOut,
    MUINT32             uiSampleRate,
    MUINT32             uiNbChannels,
    MUINT32             uiSampleSize,
    LAudioFormat*       peFormat)
{
    LStatus         eStatus = LStatus_OK;
    LAudioFormat    eFormat = LAudioFormat_INVALID;
    *peFormat               = LAudioFormat_INVALID;

    MUINT i;
    for(i=0; (*peFormat == LAudioFormat_INVALID) && LSTATUS_IS_SUCCESS(eStatus); i++)
    {
        eStatus = LAudioOut_EnumSupportedAudioFormat(hAudioOut, i, &eFormat);

        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            LAudioFormat_Descriptor oDescriptor;
            eStatus = LAudioFormat_GetDescriptor(eFormat, &oDescriptor);
            if(LSTATUS_IS_SUCCESS(eStatus))
            {
                if((oDescriptor.uiSampleRate == uiSampleRate)
                    && (oDescriptor.uiNumberOfChannels == uiNbChannels)
                    && (oDescriptor.uiSampleSize == uiSampleSize))
                {
                    *peFormat = eFormat;
                }
            }
        }
    }

    if(eStatus == LStatus_NO_MORE_DATA)
    {
        eStatus = LStatus_OK;
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       ReadBuffer

Description:    Read a buffer of uiSize bytes from hFile in the hBuffer.

Comments:       None.

\************************************************************************************************************/
size_t ReadBuffer(FILE* hFile, LBuffer_Handle hBuffer, MUINT32 uiSize)
{
    LStatus eStatus     = LStatus_OK;
    MUINT8* pbyBuffer   = MNULL;
    size_t  uiReadSize  = 0;

    eStatus = LBuffer_BeginAccess(hBuffer, 0, 1, &pbyBuffer);

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        uiReadSize = fread(pbyBuffer, 1, uiSize, hFile);

        LBuffer_EndAccess(hBuffer);
    }

    return uiReadSize;
}

/************************************************************************************************************\

Function:       main

Description:    LAudioOut sample entry point.

Comments:

\************************************************************************************************************/
int main(int argc, char** argv)
{
    LStatus                 eStatus         = LStatus_OK;
    LDevice_Handle          hDevice         = MNULL;
    LAudioOut_Handle        hAudioOut       = MNULL;
    LAudioFormat            eFormat         = LAudioFormat_INVALID;
    MUINT32                 uiBufferSize    = 0;
    LBuffer_Handle          ahBuffers[2]    = {MNULL};
    MainArgs                oArgs           = {0};
    FILE*                   hInFile         = MNULL;
    MUINT32                 i               = 0;
    MBOOL                   bStarted        = MFALSE;    
    MUINT32                 uiTickRefFreq   = 0;

    // ============================================
    // Parse main arguments.
    ParseArgs(argc, argv, &oArgs);

    Liberatus_Load();

    // ============================================
    // Get Liberatus handle.
    hDevice = Liberatus_GetDevice(0);
    if(hDevice == MNULL)
    {
        eStatus = LStatus_FAIL;
    }

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LBoard_GetTickReferenceFrequency(hDevice, &uiTickRefFreq);
        printf("Tick reference frequency: %u\n", uiTickRefFreq);
    }

    // ============================================
    // Open input file.
    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        hInFile = fopen(oArgs.szInputFilePath, "rb");
        if(hInFile == MNULL)
        {
            eStatus = LStatus_INVALID_PARAM;
        }
    }

    // ============================================
    // Get LAudioOut handle.
    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LAudioOut_GetHandle(
                        hDevice,
                        oArgs.uiOutputIdx,
                        LAccessMode_READWRITE_EXCLUSIVE,
                        &hAudioOut);
    }

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        eFormat = LAudioFormat_INVALID;
        eStatus = GetAudioFormat(
                        hAudioOut,
                        oArgs.uiSamplingRate,
                        oArgs.uiNbChannels,
                        oArgs.uiSampleSize,
                        &eFormat);

        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LAudioFormat_ComputeBufferSizeBasedOnFrame(
                            eFormat,
                            oArgs.uiSamplesPerBuf,
                            &uiBufferSize);
        }
    }

    // ============================================
    // Create LAudioOut input buffers.
    for(i=0; (i<2) && LSTATUS_IS_SUCCESS(eStatus); i++)
    {
        Declare_LBuffer_LinearAttributes(oBufferAttr);
        oBufferAttr.uiSize = uiBufferSize;

        eStatus = LBuffer_Create(
                        hDevice,
                        &oBufferAttr.eAttributeType,
                        &ahBuffers[i]);
    }

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LAudioOut_SetBufferAttributes(
                        hAudioOut,
                        eFormat,
                        uiBufferSize);
    }

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        LAudioFormat eConfigFormat = LAudioFormat_INVALID;
        MUINT32      uiConfigBuffersSize = 0;

        eStatus = LAudioOut_GetBufferAttributes(
                        hAudioOut,
                        &eConfigFormat,
                        &uiConfigBuffersSize);

        printf("Buffer attributes: format=%08x, size=%u\n", eConfigFormat, uiConfigBuffersSize);
    }

    // ===========================================
    // Set a first buffer before starting playback.
    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LStatus_FAIL;
        if(ReadBuffer(hInFile, ahBuffers[0], uiBufferSize) > 0)
        {
            eStatus = LAudioOut_SetBuffer(hAudioOut, ahBuffers[0], uiBufferSize, MFALSE);
        }
    }

    // ============================================
    // Start audio capture.
    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        printf("Starting playback, hit 'q' to exit...\n");
        eStatus = LAudioOut_StartPlayback(hAudioOut);
        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            bStarted = MTRUE;
        }
    }

    MBOOL       bExit           = MFALSE;
    MUINT       uiBufIdx        = 1;
    MUINT64     uiLastTickRef   = 0;
    MFLOAT64    fTotalTime      = 0;
    MUINT32     uiNbBuffers     = 1;    // Count the init buffer

    while(!bExit && LSTATUS_IS_SUCCESS(eStatus))
    {
        while (!bExit)
        {
            if(ReadBuffer(hInFile, ahBuffers[uiBufIdx], uiBufferSize) == 0)
            {
                break;
            }

            eStatus = LAudioOut_SetBuffer(
                            hAudioOut,
                            ahBuffers[uiBufIdx],
                            uiBufferSize,
                            MFALSE);

            if(LSTATUS_IS_SUCCESS(eStatus))
            {
                MUINT64 uiTickRef = 0;
                MUINT64 uiCounter = 0;

                eStatus = LAudioOut_WaitForBufferChange(hAudioOut, 1000, &uiCounter, &uiTickRef);

                if(LSTATUS_IS_SUCCESS(eStatus))
                {
                    if(uiLastTickRef != 0)
                    {
                        MFLOAT64 fTime = (MFLOAT64)(uiTickRef - uiLastTickRef) / uiTickRefFreq;
                        fTotalTime += fTime;
                    }

                    uiLastTickRef = uiTickRef;
                    uiNbBuffers++;
                }
                else
                {
                    printf("Error in LAudioOut_WaitForBufferChange: %d\n", eStatus);
                }
            }

            if(uiBufIdx == 0)
            {
                uiBufIdx = 1;
            }
            else
            {
                uiBufIdx = 0;
            }

            if(GetChar() == 'q')
            {
                bExit = MTRUE;
            }
        }

        if(!bExit)
        {
            printf("Restart file.\n");
            fseek(hInFile, 0, SEEK_SET);
        }

        eStatus = LStatus_OK;
    }

    if(bStarted)
    {
        printf("Stopping playback...\n");
        LStatus eStopStatus = LAudioOut_StopPlayback(hAudioOut);
        if(LSTATUS_IS_FAIL(eStopStatus))
        {
            printf("Failed to stop playback: %d\n", eStopStatus);
        }
    }

    printf("Average buffer time: %f seconds, %u buffers\n",
           fTotalTime / uiNbBuffers, uiNbBuffers);

    // ============================================
    // Cleanup.

    for(i=0; i<2; i++)
    {
        if(ahBuffers[i] != MNULL)
        {
            LBuffer_Destroy(ahBuffers[i]);
            ahBuffers[i] = MNULL;
        }
    }

    if(hInFile != MNULL)
    {
        fclose(hInFile);
        hInFile = MNULL;
    }

    if(hAudioOut != MNULL)
    {
        LAudioOut_ReleaseHandle(hAudioOut);
        hAudioOut = MNULL;
    }

    Liberatus_UnLoad();

    if(LSTATUS_IS_FAIL(eStatus))
    {
        printf("Failed with status code: %d\n", eStatus);
        return 1;
    }
    else
    {
        return 0;
    }
}
