/******************************************************************************\

Module Name:    LAudioEncoder.c

Description:    LAudioEncoder sample implementation file.

                This sample captures a number of buffers from an audio capture
                device, encodes them in AAC and writes the resulting AAC stream
                to an output file. To simplity the sample, all the operations
                are done in a single thread.

References:     Revision 2.0
                Matrox Liberatus specifications.
                xxxxx-xxx-xxxx-xxxx

    Copyright (c) 2015, Matrox Graphics Inc.
    All Rights Reserved.

BSD 2-Clause License

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\******************************************************************************/

// --------------------------------------------------------------------------------------
//                        I N C L U D E S   A N D   U S I N G S
// --------------------------------------------------------------------------------------

#include "Liberatus.h"
#include "LBuffer.h"
#include "LAudioIn.h"
#include "LAudioCodec.h"
#include "stdio.h"
#include <stdlib.h>
#include <string.h>

#if defined(_WIN32)
#include <windows.h>
#else
#include <unistd.h>
#endif

// -----------------------------------------------------------------------------------------------------------
//                                   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

typedef struct
{
    const MCHAR8* szOutputFilePath;
    MUINT32 uiInputIdx;
    MUINT32 uiNbBuffers;
} MainArgs;

// --------------------------------------------------------------------------------------
//                                         C O D E
// --------------------------------------------------------------------------------------

/************************************************************************************************************\

Function:       ParseArgs

Description:    Parse application arguments.

Comments:       None.

\************************************************************************************************************/
void ParseArgs(int argc, char** argv, MainArgs* poArgs)
{
    MBOOL32 bInvalidArg = MFALSE;
    MUINT i = 0;

    poArgs->szOutputFilePath = MNULL;
    poArgs->uiInputIdx = 0;

    for(i=1; i<argc; i++)
    {
        if((strcmp(argv[i], "-o") == 0) && ((i+1)<argc))
        {
            i++;
            poArgs->szOutputFilePath = argv[i];
        }
        else if((strcmp(argv[i], "-in") == 0) && ((i+1)<argc))
        {
            i++;
            poArgs->uiInputIdx = atoi(argv[i]);
        }
        else if((strcmp(argv[i], "-buf") == 0) && ((i+1)<argc))
        {
            i++;
            MINT32 iNbBufs = atoi(argv[i]);
            if(iNbBufs > 0)
            {
                poArgs->uiNbBuffers = (MUINT32) iNbBufs;
            }
            else
            {
                bInvalidArg = MTRUE;
            }
        }
        else
        {
            bInvalidArg = MTRUE;
            break;
        }
    }

    if((poArgs->szOutputFilePath == MNULL)
        || (poArgs->uiInputIdx > 3))
    {
        bInvalidArg = MTRUE;
    }

    if(bInvalidArg)
    {
        printf("Usage: %s <-o output_file> [-in n] [-buf n]\n", argv[0]);
        printf("Optional arguments:\n");
        printf("\t-o str        : Output file path\n");
        printf("\t-in n         : Audio input index (0 to 3, default=0)\n");
        printf("\t-buf n        : The number of buffers to capture and encode (default=500)\n");
        exit(1);
    }
}

/********************************************************************\

Function:       GetSamplingRate

Description:    Get LAudioCodec sampling rate enum from LAudioFormat.

Comments:       None.

\********************************************************************/
LAudioCodec_SamplingRate GetSamplingRate(LAudioFormat eFormat)
{
    MUINT32 uiSamplingRate = LAudioFormat_GetSampleRate(eFormat);

    switch(uiSamplingRate)
    {
        case 8000:  return LAudioCodec_SamplingRate_8000Hz;
        case 11025: return LAudioCodec_SamplingRate_11025Hz;
        case 12000: return LAudioCodec_SamplingRate_12000Hz;
        case 16000: return LAudioCodec_SamplingRate_16000Hz;
        case 22050: return LAudioCodec_SamplingRate_22050Hz;
        case 24000: return LAudioCodec_SamplingRate_24000Hz;
        case 32000: return LAudioCodec_SamplingRate_32000Hz;
        case 44100: return LAudioCodec_SamplingRate_44100Hz;
        case 48000: return LAudioCodec_SamplingRate_48000Hz;
        case 88200: return LAudioCodec_SamplingRate_88200Hz;
        case 96000: return LAudioCodec_SamplingRate_96000Hz;
        default:    return LAudioCodec_SamplingRate_UNKNOWN;
    }
}

/********************************************************************\

Function:       GetChannelConfig

Description:    Get LAudioCodec channel config enum from LAudioFormat.

Comments:       None.

\********************************************************************/
LAudioCodec_ChannelCfg GetChannelConfig(LAudioFormat eFormat)
{
    MUINT32 uiNbChannels = LAudioFormat_GetNumberOfChannel(eFormat);

    switch(uiNbChannels)
    {
        case 1:     return LAudioCodec_ChannelCfg_MONO;
        case 2:     return LAudioCodec_ChannelCfg_STEREO;
        default:    return LAudioCodec_ChannelCfg_UNKNOWN;
    }
}

/********************************************************************\

Function:       main

Description:    LAudioEncoder sample entry point.

Comments:       None.

\********************************************************************/
int main(int argc, char** argv)
{
    LStatus                 eStatus         = LStatus_INVALID_PARAM;
    LDevice_Handle          hDevice         = MNULL;
    LAudioIn_Handle         hAudioIn        = MNULL;
    LAudioEncoder_Handle    hEncoder        = MNULL;
    LAudioFormat            eFormat         = LAudioFormat_INVALID;
    LBuffer_Handle          hPCMBuffer      = MNULL;
    MainArgs                oArgs           = {0};
    FILE*                   hOutFile        = MNULL;
    MBOOL32                 bBuffersCreated = MFALSE;
    MBOOL32                 bStarted        = MFALSE;
    MUINT32                 uiBufferSize    = 0;

    /* Parse main arguments. */
    ParseArgs(argc, argv, &oArgs);

    /* Open output file. */
    hOutFile = fopen(oArgs.szOutputFilePath, "wb+");
    if(hOutFile == MNULL)
    {
        printf("Error, can't open output file %s\n", oArgs.szOutputFilePath);
        goto cleanup;
    }

    eStatus = LStatus_FAIL;

    /*  Get Liberatus handle. */
    Liberatus_Load();
    hDevice = Liberatus_GetDevice(0);
    if(hDevice == MNULL)
    {
        printf("Error, can't get LDevice_Handle 0\n");
        goto cleanup;
    }

    /* ============================================= *
     *              Initialize LAudioIn              *
     * ============================================= */

    printf("Getting LAudioIn handle...\n");

    /* Get LAudioIn handle. */
    eStatus = LAudioIn_GetHandle(hDevice,
                           oArgs.uiInputIdx,
                           LAccessMode_READWRITE_EXCLUSIVE,
                           &hAudioIn);

    /* Detect audio input signal. */
    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        MBOOL32 bIsDetected = MFALSE;
        MBOOL32 bIsCapturable = MFALSE;

        eStatus = LAudioIn_DetectSource(hAudioIn, &bIsCapturable, &bIsDetected);

        printf("Audio input signal detected:%s, capturable:%s\n",
               bIsDetected   ? "yes" : "no",
               bIsCapturable ? "yes" : "no");
    }

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        eFormat = LAudioFormat_INVALID;
        eStatus = LAudioIn_EnumSupportedAudioFormat(hAudioIn, 0, &eFormat);
    }

    /* Create LAudioIn input buffers. */
    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LAudioFormat_ComputeBufferSizeBasedOnFrame(eFormat, 1024, &uiBufferSize);
        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LAudioIn_CreateBuffers(hAudioIn, eFormat, uiBufferSize, 4);
            bBuffersCreated = LSTATUS_IS_SUCCESS(eStatus) ? MTRUE : MFALSE;
        }
    }

    /* Start audio capture */
    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        printf("Starting audio capture...\n");
        eStatus = LAudioIn_StartCapture(hAudioIn);
        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            bStarted = MTRUE;
        }
    }

    /* ============================================= *
     *            Initialize LAudioEncoder           *
     * ============================================= */
    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        LAudioCodec_SamplingRate eSamplingRate = LAudioCodec_SamplingRate_UNKNOWN;
        LAudioCodec_ChannelCfg   eChannelCfg   = LAudioCodec_ChannelCfg_UNKNOWN;

        /* Validate capture parameters */
        if((LAudioFormat_GetSampleSize(eFormat) != 16)
            || ((eSamplingRate = GetSamplingRate(eFormat)) == LAudioCodec_SamplingRate_UNKNOWN)
            || ((eChannelCfg = GetChannelConfig(eFormat)) == LAudioCodec_ChannelCfg_UNKNOWN))
        {
            eStatus = LStatus_INVALID_PARAM;
        }

        /* Create AAC encoder */
        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            LAudioEncoder_AACOptions oAacOpt                = {LAudioEncoder_CodecOptTypeHeader_AAC};
            oAacOpt.eProfile                                = LAudioCodec_AACProfile_LC;
            oAacOpt.eQualityLevel                           = LAudioCodec_AACQualityLevel_2;
            oAacOpt.eStreamFormat                           = LAudioCodec_AACStreamFormat_ADTS;
            oAacOpt.flEncodingFlags                         = LAudioEncoder_AACFlags_USE_TNS;
            oAacOpt.uiBitRate                               = 128000;

            LAudioEncoder_CreateOpt oCreateOpt              = {LAudioEncoder_CreateOptTypeHeader_STANDARD};
            oCreateOpt.hDevice                              = hDevice;
            oCreateOpt.oPCMAudioAttributes.eType            = LAudioCodec_AudioAttributeTypeHeader_STANDARD;
            oCreateOpt.oPCMAudioAttributes.eBitDepth        = LAudioCodec_BitDepth_16;
            oCreateOpt.oPCMAudioAttributes.eChannelCfg      = eChannelCfg;
            oCreateOpt.oPCMAudioAttributes.ePCMMode         = LAudioCodec_PCMMode_LINEAR;
            oCreateOpt.oPCMAudioAttributes.eSamplingRate    = eSamplingRate;
            oCreateOpt.oPCMAudioAttributes.flFlags          = 0;
            oCreateOpt.uiNbOutputBuffers                    = 5;
            oCreateOpt.eOutputBufferType                    = LBuffer_Type_SYSTEM_LINEAR;
            oCreateOpt.peEncoderCodecOptions                = &oAacOpt.eType;

            eStatus = LAudioEncoder_Create(&oCreateOpt.eType, &hEncoder);
        }
    }

    /* ============================================= *
     *          Capture and encode stream            *
     * ============================================= */
    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        printf("Capture and encode stream...\n");

        MUINT32 uiNbCapturedBuffers = 0;
        MUINT32 uiNbEncodedBuffers  = 0;
        MUINT32 uiNbWrittenBuffers  = 0;

        while (uiNbCapturedBuffers < oArgs.uiNbBuffers)
        {
            /* Get captured PCM buffer */
            eStatus = LAudioIn_GetNextBuffer(hAudioIn, MTRUE, 100, &hPCMBuffer, MNULL, MNULL, MNULL);

            if(LSTATUS_IS_SUCCESS(eStatus))
            {
                /* Copy captured PCM buffer in audio encoder */
                LAudioCodec_BufferInfo oBufInfo = { LAudioCodec_BufferInfoTypeHeader_STANDARD };
                oBufInfo.hBuffer = hPCMBuffer;
                oBufInfo.uiSize = uiBufferSize;
                oBufInfo.uiStartOffset = 0;

                eStatus = LAudioEncoder_PutData(hEncoder, &oBufInfo.eType, 0);

                /* Encode a PCM buffer */
                if(LSTATUS_IS_SUCCESS(eStatus))
                {
                    LAudioEncoder_EncodeOpt oEncodeOpt = {LAudioEncoder_EncodeOptTypeHeader_STANDARD};
                    oEncodeOpt.bReplaceOldestBuffer = MFALSE;

                    eStatus = LAudioEncoder_EncodeBuffer(hEncoder, &oEncodeOpt.eType);
                    if(LSTATUS_IS_SUCCESS(eStatus))
                    {
                        uiNbEncodedBuffers++;
                    }
                }

                /* Release PCM buffer back to LAudioIn */
                LAudioIn_ReleaseBuffer(hAudioIn, hPCMBuffer);

                LAudioCodec_BufferInfo oBufferInfo = {LAudioCodec_BufferInfoTypeHeader_STANDARD};
                MUINT8* puiOutBuffer = MNULL;

                /* Get Encoded AAC buffer */
                if(LSTATUS_IS_SUCCESS(eStatus))
                {
                    eStatus = LAudioEncoder_GetNextBuffer(hEncoder, &oBufferInfo.eType, 100, MNULL, MNULL);
                }

                /* Write encoded AAC buffer to output file */
                if(LSTATUS_IS_SUCCESS(eStatus))
                {
                    eStatus = LBuffer_BeginAccess(oBufferInfo.hBuffer, 0, 1, &puiOutBuffer);
                    if(LSTATUS_IS_SUCCESS(eStatus))
                    {
                        fwrite(puiOutBuffer, 1, oBufferInfo.uiSize, hOutFile);
                        uiNbWrittenBuffers++;
                        LBuffer_EndAccess(oBufferInfo.hBuffer);
                    }

                    /* Release encoded AAC buffer back to LAudioEncoder */
                    LAudioEncoder_ReleaseBuffer(hEncoder, oBufferInfo.hBuffer);
                }

                uiNbCapturedBuffers++;

                printf("Captured buffer: %u\r", uiNbCapturedBuffers);
                fflush(stdout);
            }
        }

        printf("\nTotal Captured buffers: %u\nTotal Encoded buffers: %u\nTotal Written buffers: %u\n",
               uiNbCapturedBuffers, uiNbEncodedBuffers, uiNbWrittenBuffers);
    }

cleanup:

    /* ============================================= *
     *                    Cleanup                    *
     * ============================================= */
    if(LSTATUS_IS_FAIL(eStatus)
        && (eStatus != LStatus_END_OF_STREAM))
    {
        printf("Failed with status: %d\n", eStatus);
    }

    if(hOutFile != MNULL)
    {
        fclose(hOutFile);
    }

    if(hEncoder != MNULL)
    {
        LAudioEncoder_Destroy(hEncoder);
    }

    if(bStarted)
    {
        LStatus eStopStatus = LAudioIn_StopCapture(hAudioIn);
        if(LSTATUS_IS_FAIL(eStopStatus))
        {
            printf("Error in LAudioIn_StopCapture: %d\n", eStopStatus);
        }
    }

    if(bBuffersCreated)
    {
        LAudioIn_DestroyBuffers(hAudioIn);
    }

    if(hAudioIn != MNULL)
    {
        LAudioIn_ReleaseHandle(hAudioIn);
    }

    Liberatus_UnLoad();

    return LSTATUS_IS_SUCCESS(eStatus) ? 0 : 1;
}
