/******************************************************************************\

Module Name:    LAudioDecoder.c

Description:    LAudioDecoder sample implementation file.

                This sample reads an AAC input file, decodes it, and writes
                the resulting PCM stream to an output file. To simplify the
                the sample, all the operations are done in a single thread.

References:     Revision 2.0
                Matrox Liberatus specifications.
                xxxxx-xxx-xxxx-xxxx

    Copyright (c) 2015, Matrox Graphics Inc.
    All Rights Reserved.

BSD 2-Clause License

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\******************************************************************************/

// --------------------------------------------------------------------------------------
//                        I N C L U D E S   A N D   U S I N G S
// --------------------------------------------------------------------------------------

#include "Liberatus.h"
#include "LBuffer.h"
#include "LAudioCodec.h"
#include "stdio.h"
#include <stdlib.h>
#include <string.h>

// -----------------------------------------------------------------------------------------------------------
//                                   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

typedef struct
{
    const MCHAR8* szInputFilePath;
    const MCHAR8* szOutputFilePath;
    MBOOL32 bRaw;
    LAudioCodec_SamplingRate eSamplingRate;
    LAudioCodec_AACProfile eProfile;
} MainArgs;

// --------------------------------------------------------------------------------------
//                                         C O D E
// --------------------------------------------------------------------------------------

/************************************************************************************************************\

Function:       GetSamplingRateEnum

Description:    Get LAudioCodec sampling rate enum from sampling rate.

Comments:       None.

\************************************************************************************************************/
LAudioCodec_SamplingRate GetSamplingRateEnum(MUINT32 uiSamplingRate)
{
    switch(uiSamplingRate)
    {
        case 8000:  return LAudioCodec_SamplingRate_8000Hz;
        case 11025: return LAudioCodec_SamplingRate_11025Hz;
        case 12000: return LAudioCodec_SamplingRate_12000Hz;
        case 16000: return LAudioCodec_SamplingRate_16000Hz;
        case 22050: return LAudioCodec_SamplingRate_22050Hz;
        case 24000: return LAudioCodec_SamplingRate_24000Hz;
        case 32000: return LAudioCodec_SamplingRate_32000Hz;
        case 44100: return LAudioCodec_SamplingRate_44100Hz;
        case 48000: return LAudioCodec_SamplingRate_48000Hz;
        case 88200: return LAudioCodec_SamplingRate_88200Hz;
        case 96000: return LAudioCodec_SamplingRate_96000Hz;
        default:    return LAudioCodec_SamplingRate_UNKNOWN;
    }
}

/************************************************************************************************************\

Function:       ParseArgs

Description:    Parse application arguments.

Comments:       None.

\************************************************************************************************************/
void ParseArgs(int argc, char** argv, MainArgs* poArgs)
{
    MBOOL32 bInvalidArg = MFALSE;
    MUINT i = 0;

    poArgs->szInputFilePath = MNULL;
    poArgs->szOutputFilePath = MNULL;
    poArgs->eProfile = LAudioCodec_AACProfile_UNKNOWN;
    poArgs->eSamplingRate = LAudioCodec_SamplingRate_UNKNOWN;
    poArgs->bRaw = MFALSE;

    for(i=1; i<argc; i++)
    {
        if((strcmp(argv[i], "-o") == 0) && ((i+1)<argc))
        {
            i++;
            poArgs->szOutputFilePath = argv[i];
        }
        else if((strcmp(argv[i], "-i") == 0) && ((i+1)<argc))
        {
            i++;
            poArgs->szInputFilePath = argv[i];
        }
        else if(strcmp(argv[i], "-raw") == 0)
        {
            poArgs->bRaw = MTRUE;
        }
        else if((strcmp(argv[i], "-p") == 0) && ((i+1)<argc))
        {
            i++;
            if(strcmp(argv[i], "lc") == 0)
            {
                poArgs->eProfile = LAudioCodec_AACProfile_LC;
            }
            else if(strcmp(argv[i], "hev1") == 0)
            {
                poArgs->eProfile = LAudioCodec_AACProfile_HEV1;
            }
            else if(strcmp(argv[i], "hev2") == 0)
            {
                poArgs->eProfile = LAudioCodec_AACProfile_HEV2;
            }
            else
            {
                bInvalidArg = MTRUE;
            }
        }
        else if((strcmp(argv[i], "-sr") == 0) && ((i+1)<argc))
        {
            i++;
            MINT32 iSampleRate = atoi(argv[i]);
            if(iSampleRate > 0)
            {
                poArgs->eSamplingRate = GetSamplingRateEnum((MUINT32) iSampleRate);
                if(poArgs->eSamplingRate == LAudioCodec_SamplingRate_UNKNOWN)
                {
                    bInvalidArg = MTRUE;
                }
            }
        }
        else
        {
            bInvalidArg = MTRUE;
        }

        if(bInvalidArg)
        {
            break;
        }
    }

    if((poArgs->szOutputFilePath == MNULL)
        || (poArgs->szInputFilePath == MNULL)
        || (poArgs->bRaw
             && ((poArgs->eSamplingRate == LAudioCodec_SamplingRate_UNKNOWN) || (poArgs->eProfile == LAudioCodec_AACProfile_UNKNOWN))))
    {
        bInvalidArg = MTRUE;
    }

    if(bInvalidArg)
    {
        printf("Usage: %s <-o output_file> <-i input_file> [-raw] [-sr samplerate] [-p profile] \n", argv[0]);
        printf("Optional arguments:\n");
        printf("\t-o output_file    : Output file path\n");
        printf("\t-i input_file     : Input file path\n");
        printf("\t-raw              : Stream is raw\n");
        printf("\t-sr samplerate    : AAC sampling rate for raw stream. Mandatory if -raw is present, ignored otherwise.\n");
        printf("\t-p profile        : AAC profile for raw stream (lc, hev1 or hev2). Mandatory if -raw option is used, ignored otherwise.\n");
        exit(1);
    }
}

/********************************************************************\

Function:       main

Description:    LAudioDecoder sample entry point.

Comments:       None.

\********************************************************************/
int main(int argc, char** argv)
{
    LStatus                 eStatus         = LStatus_INVALID_PARAM;
    LDevice_Handle          hDevice         = MNULL;
    LAudioDecoder_Handle    hDecoder        = MNULL;
    MUINT32                 uiBufferSize    = 0;
    LBuffer_Handle          hAACBuffer      = MNULL;
    MUINT8*                 pbyBuffer       = MNULL;
    MainArgs                oArgs           = {0};
    FILE*                   hInFile         = MNULL;
    FILE*                   hOutFile        = MNULL;
    size_t                  uiReadSize      = 0;
    MUINT32                 uiConsumedBytes = 0;

    /* Parse main arguments. */
    ParseArgs(argc, argv, &oArgs);

    /* Open input file. */
    hInFile = fopen(oArgs.szInputFilePath, "rb");
    if(hInFile == MNULL)
    {
        printf("Error, can't open input file %s\n", oArgs.szInputFilePath);
        goto cleanup;
    }

    /* Open output file. */
    hOutFile = fopen(oArgs.szOutputFilePath, "wb+");
    if(hOutFile == MNULL)
    {
        printf("Error, can't open output file %s\n", oArgs.szOutputFilePath);
        goto cleanup;
    }

    eStatus = LStatus_FAIL;
    /*  Get Liberatus handle. */
    Liberatus_Load();
    hDevice = Liberatus_GetDevice(0);
    if(hDevice == MNULL)
    {
        printf("Error, can't get LDevice_Handle 0\n");
        goto cleanup;
    }

    /* ============================================= *
     *           Initialize LAudioDecoder            *
     * ============================================= */

    /* Create AAC decoder */
    printf("Create decoder...\n");
    LAudioDecoder_AACOptions oAacOpt    = { LAudioDecoder_CodecOptTypeHeader_AAC };
    oAacOpt.eStreamFormat               = oArgs.bRaw
                                            ? LAudioCodec_AACStreamFormat_RAW
                                            : LAudioCodec_AACStreamFormat_UNKNOWN;
    oAacOpt.eProfile                    = oArgs.bRaw
                                            ? oArgs.eProfile
                                            : LAudioCodec_AACProfile_UNKNOWN;
    oAacOpt.flFlags                     = 0;
    oAacOpt.bIsFramed                   = MFALSE;

    LAudioDecoder_CreateOpt oCreateOpt                  = { LAudioDecoder_CreateOptTypeHeader_STANDARD };
    oCreateOpt.hDevice                                  = hDevice;
    oCreateOpt.oEncodedAudioAttributes.eType            = LAudioCodec_AudioAttributeTypeHeader_STANDARD;
    oCreateOpt.oEncodedAudioAttributes.eBitDepth        = LAudioCodec_BitDepth_UNKNOWN;
    oCreateOpt.oEncodedAudioAttributes.eChannelCfg      = LAudioCodec_ChannelCfg_UNKNOWN;
    oCreateOpt.oEncodedAudioAttributes.eSamplingRate    = oArgs.bRaw
                                                            ? oArgs.eSamplingRate
                                                            : LAudioCodec_SamplingRate_UNKNOWN;
    oCreateOpt.oEncodedAudioAttributes.flFlags          = 0;
    oCreateOpt.peDecoderCodecOptions                    = &oAacOpt.eType;
    oCreateOpt.uiNbOutputBuffers                        = 5;
    oCreateOpt.eOutputBufferType                        = LBuffer_Type_SYSTEM_LINEAR;

    eStatus = LAudioDecoder_Create(&oCreateOpt.eType, &hDecoder);

    /* Get buffer informations */
    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        LAudioDecoder_BufferInfo oBufferInfo = { LAudioDecoder_InfoTypeHeader_BUFFER_INFO };

        eStatus = LAudioDecoder_GetInfo(hDecoder, &oBufferInfo.eType);

        uiBufferSize = oBufferInfo.uiRecommendedInputBufferSize;
    }

    /* Create LBuffer for AAC input */
    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        LBuffer_SystemLinearAttributes oBufferAttribute = { LBuffer_Type_SYSTEM_LINEAR };
        oBufferAttribute.pvSystemMemory = MNULL;
        oBufferAttribute.uiSize = uiBufferSize;
        eStatus = LBuffer_Create(hDevice, &oBufferAttribute.eAttributeType, &hAACBuffer);
    }

    /* Initialize decoder */
    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        printf("Initialize decoder...\n");
        /* Begin input buffer access */
        eStatus = LBuffer_BeginAccess(hAACBuffer, 0, 1, &pbyBuffer);

        /* Send the beginning of the stream to the decoder */
        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            uiReadSize = fread(pbyBuffer, 1, uiBufferSize, hInFile);

            LBuffer_EndAccess(hAACBuffer);

            if(uiReadSize > 0)
            {
                LAudioCodec_BufferInfo oBufInfo = { LAudioCodec_BufferInfoTypeHeader_STANDARD };
                oBufInfo.hBuffer = hAACBuffer;
                oBufInfo.uiSize = uiReadSize;
                oBufInfo.uiStartOffset = 0;
                eStatus = LAudioDecoder_PutData(hDecoder, &oBufInfo.eType, 0);
            }
        }

        /* Initialize decoder with the AAC data */
        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            LAudioDecoder_InitDecoderOpt oInitDecoderOpt = { LAudioDecoder_DecodeOptTypeHeader_INIT_DECODER };

            eStatus = LAudioDecoder_DecodeBuffer(hDecoder, &oInitDecoderOpt.eType, &uiConsumedBytes);
        }
    }

    /* ============================================= *
     *                Decode stream                  *
     * ============================================= */

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        printf("Decode stream...\n");

        MBOOL32 bDecodingDone = MFALSE;
        MUINT32 uiNbBuffers = 0;
        uiReadSize = 0;

        while (!bDecodingDone)
        {
            /* Read a chunk of AAC data */
            if(uiReadSize == 0)
            {
                eStatus = LBuffer_BeginAccess(hAACBuffer, 0, 1, &pbyBuffer);
                if(LSTATUS_IS_SUCCESS(eStatus))
                {
                    uiReadSize = fread(pbyBuffer, 1, uiBufferSize, hInFile);
                    LBuffer_EndAccess(hAACBuffer);
                }
            }

            /* Copy the chunk of AAC data to the decoder. Retry with the same chunk until success */
            if(uiReadSize > 0)
            {
                LAudioCodec_BufferInfo oBufInfo = { LAudioCodec_BufferInfoTypeHeader_STANDARD };
                oBufInfo.hBuffer = hAACBuffer;
                oBufInfo.uiSize = uiReadSize;
                oBufInfo.uiStartOffset = 0;
                eStatus = LAudioDecoder_PutData(hDecoder, &oBufInfo.eType, 0);

                if(LSTATUS_IS_SUCCESS(eStatus))
                {
                    uiReadSize = 0;
                }
            }
            else
            {
                /*
                 * We have read all of the input file, signal end of stream, LAudioDecoder_DecodeBuffer
                 * will return LStatus_END_OF_STREAM once all the copied AAC data has been decoded.
                 */
                LAudioDecoder_EndStream(hDecoder, MFALSE);
            }

            /* Decode an AAC buffer */
            LAudioDecoder_DecodeOpt oDecodeOpt = { LAudioDecoder_DecodeOptTypeHeader_STANDARD };
            oDecodeOpt.bReplaceOldestBuffer = MFALSE;
            uiConsumedBytes = 0;
            eStatus = LAudioDecoder_DecodeBuffer(hDecoder, &oDecodeOpt.eType, &uiConsumedBytes);
            if(LSTATUS_IS_SUCCESS(eStatus))
            {
                /* Retrieve the decoded buffer */
                LAudioCodec_BufferInfo oBufferInfo = { LAudioCodec_BufferInfoTypeHeader_STANDARD };
                eStatus = LAudioDecoder_GetNextBuffer(hDecoder, &oBufferInfo.eType, 0, MNULL, MNULL);
                if(LSTATUS_IS_SUCCESS(eStatus))
                {
                    /* Write the decoded buffer to output file */
                    eStatus = LBuffer_BeginAccess(oBufferInfo.hBuffer, 0, 1, &pbyBuffer);
                    if(LSTATUS_IS_SUCCESS(eStatus))
                    {
                        fwrite(pbyBuffer + oBufferInfo.uiStartOffset, 1, oBufferInfo.uiSize, hOutFile);
                        LBuffer_EndAccess(oBufferInfo.hBuffer);
                        uiNbBuffers++;
                    }

                    /* Release the decoded buffer back to the decoder */
                    LAudioDecoder_ReleaseBuffer(hDecoder, oBufferInfo.hBuffer);
                }
            }
            else if(eStatus == LStatus_END_OF_STREAM)
            {
                bDecodingDone = MTRUE;
            }
        }

        printf("Decoded %u buffers\n", uiNbBuffers);
    }

cleanup:

    /* ============================================= *
     *                   Cleanup                     *
     * ============================================= */

    if(LSTATUS_IS_FAIL(eStatus)
        && (eStatus != LStatus_END_OF_STREAM))
    {
        printf("Failed with status: %d\n", eStatus);
    }

    if(hAACBuffer != MNULL)
    {
        LBuffer_Destroy(hAACBuffer);
    }

    if(hInFile != MNULL)
    {
        fclose(hInFile);
    }

    if(hOutFile != MNULL)
    {
        fclose(hOutFile);
    }

    if(hDecoder != MNULL)
    {
        LAudioDecoder_Destroy(hDecoder);
    }

    Liberatus_UnLoad();

    return LSTATUS_IS_SUCCESS(eStatus) ? 0 : 1;
}
