/************************************************************************************************************\

Module Name:    main.c

Description:    Sample of a video pipeline using Liberatus H264 encoder and capture.

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "VinModule.h"
#include "VoutModule.h"
#ifdef LINUX
 #include "VoutGlxModule.h"
#endif
#include "EncoderModule.h"
#include "DecoderModule.h"
#include "WriteFileModule.h"
#include "RtspServerModule.h"
#include "MosaicModule.h"
#include "RotateModule.h"
#include "CommonUtils.h"
#include "AinModule.h"
#include "AinAlsaModule.h"
#include "AoutModule.h"
#include "AoutAlsaModule.h"
#include "AudioEncoderModule.h"

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

#define AVG_PERIOD          60
#define MAX_STREAM_COUNT    16
#define MAX_VIN_COUNT       32
#define MAX_VOUTMTX_COUNT   32
#ifdef LINUX
  #define MAX_VOUTGLX_COUNT 1
#else
  #define MAX_VOUTGLX_COUNT 0
#endif
#define MAX_VOUT_COUNT      (MAX_VOUTMTX_COUNT + MAX_VOUTGLX_COUNT)
#define MAX_VIN_PER_STEAM   16
#define MAX_AIN_COUNT       32
#define MAX_AOUT_COUNT      32
#define MIN_FRAME_SIZE      64
#define MAX_FRAME_SIZE      4096
#define MAX_BEZEL_SIZE      256
#define MIN_AAC_BITRATE     32000

typedef struct tagVoutParam
{
    MUINT                   uiInputCount;
    MUINT                   uiIndex;
    MUINT                   uiCycleTimeSec;
    MUINT                   uiNextCycleTimeSec;
    MUINT                   uiNextCycleInputIdx;
    LSIZE                   oSize;
    LPixelFormat            ePixFmt;
    LPixelFormat            eOutPixFmt;
    MBOOL                   bOutRgbFullYuv709;
    MBOOL                   bForceMode;
    LSIZE                   oForcedSize;
    MUINT                   uiForcedFps;
    MBOOL                   bIsMasterSync;

    // Modules
    MosaicModule            oOutMosMod;
    VoutModule              oVoutMod;
   #ifdef LINUX
    VoutGlxModule           oVoutGlxMod;
   #endif

} VoutParam;

const VoutParam g_oDefaultVoutParam =
{
    /*.uiInputCount             =*/ 0,
    /*.uiIndex                  =*/ 0,
    /*.uiCycleTimeSec           =*/ 0,
    /*.uiLastCycleTimeSec       =*/ 0,
    /*.uiNextCycleInputIdx      =*/ 0,
    /*.oSize                    =*/ {1920, 1080},
    /*.ePixFmt                  =*/ LPixelFormat_R8G8B8A8,
    /*.eOutPixFmt               =*/ LPixelFormat_R8G8B8A8,
    /*.bOutRgbFullYuv709        =*/ MTRUE,
    /*.bForceMode               =*/ MFALSE,
    /*.oForcedSize              =*/ {1920, 1080},
    /*.uiForcedFps              =*/ 60,
    /*.bIsMasterSync            =*/ MTRUE,
    /*.oMosMod                  =*/ MosaicModule_Construct,
    /*.oVoutMod                 =*/ VoutModule_Construct,
    #ifdef LINUX
    /*.oVoutGlxMod              =*/ VoutGlxModule_Construct
   #endif
};

typedef struct tagVinParam
{
    MBOOL           bEnabled;
    MUINT           uiIndex;
    LPixelFormat    ePixFmt;
    MUINT           uiInMosIdx;
    VoutParam*      poVoutParam;
    VinModule       oVinMod;
    MBOOL           bDoScaling;

} VinParam;

const VinParam g_oDefaultVinParam =
{
    /*.bEnabled         =*/ MFALSE,
    /*.uiIndex          =*/ 0,
    /*.ePixFmt          =*/ LPixelFormat_MP_Y8_U8V8_422,
    /*.uiMosInputIdx    =*/ 0,
    /*.poVoutParam      =*/ MNULL,
    /*.oVinMod          =*/ VinModule_Construct,
    /*.bDoScaling       =*/ MFALSE
};

typedef struct tagStreamParam
{
    MBOOL               bEnable;

    // Parameters
    VinParam*           apoVinParam[MAX_VIN_PER_STEAM];
    MUINT               uiVinCount;
    VoutParam*          poVoutParam;
    const char*         szDstFilePath;
    LSIZE               oFrameSize;
    LSIZE               oFrameDiv;
    LSIZE               oBezelSize;
    LSIZE               oSubFrameSize;
    LPOS                oSubFramePos;
    MBOOL               bUseVinWidth;
    MBOOL               bUseVinHeight;
    MBOOL               bFullFrame;
    MBOOL               bKeepRatio;
    MUINT               uiFrameRateDivisor;
    MUINT               uiTargetBitRateKbs;
    MUINT               uiIPeriod;
    MUINT               uiPPeriod;
    MUINT               uiQpMin;
    MUINT               uiQpMax;
    MUINT               uiSliceCount;
    LH264E_EncoderMode  eEncodeMode;
    LH264_Profile       eEncodeProfile;
    LH264_Level         eEncodeLevel;
    MBOOL               bNetServer;
    const char*         szNetAddress;
    const char*         szRtmpPushLocation;
    const char*         szUsername;
    const char*         szPassword;
    const char*         szFolderName;
    MUINT               uiNetPort;
    MUINT               uiMtu;
    LNetStreamer_Protocol eNetProtocol;
    LPixelFormat        eEncodePixFmt;
    MUINT               uiFaq;
    MBOOL               bDfEnabled;
    MBOOL               bEnableSrt;
    MUINT               uiOutMosIdx;
    LVPC_Rotation       eRotation;
    MUINT               uiHlsDuration;
    MUINT               uiHlsCount;
    const char*         szHlsFolder;
    const char*         szHlsMaster;
    const char*         szNetInterface;
    MBOOL               bEnableIpv6;
    LNetStreamer_SrtMode eSrtMode;

    // Modules
    MosaicModule        oInMosMod;
    EncoderModule       oEncMod;
    WriteFileModule     oWrFileMod;
    RtspServerModule    oRtspSrvMod;
    DecoderModule       oDecMod;
    RotateModule        oRotModA;
    RotateModule        oRotModB;

    // Statistics
    MBOOL               bSrvAddrDisplayed;
    MUINT64             uiNextStatTime;
    MUINT               uiSec;
    MUINT               uiMin;
    MUINT               uiHour;
    MUINT               uiOldAvgIdx;
    MUINT               uiNewAvgIdx;
    MUINT               auiCapturedFrameCount[AVG_PERIOD];
    MUINT               auiEncodedFrameCount[AVG_PERIOD];
    MUINT64             auiEncodedBytes[AVG_PERIOD];
    MUINT64             auiElapsedTimeUsec[AVG_PERIOD];

} StreamParam;

const StreamParam g_oDefaultStreamParam =
{
    /*.bEnable              =*/ MFALSE,
    /*.apoVinParam          =*/ { MNULL, MNULL, MNULL, MNULL,
                                  MNULL, MNULL, MNULL, MNULL,
                                  MNULL, MNULL, MNULL, MNULL,
                                  MNULL, MNULL, MNULL, MNULL },
    /*.uiVinCount           =*/ 0,
    /*.poVoutParam          =*/ MNULL,
    /*.szDstFilePath        =*/ MNULL,
    /*.oFrameSize           =*/ {0, 0},
    /*.oFrameDiv            =*/ {0, 0},
    /*.oBezelSize           =*/ {0, 0},
    /*.oSubFrameSize        =*/ {0, 0},
    /*.oSubFramePos         =*/ {0, 0},
    /*.bUseVinWidth         =*/ MTRUE,
    /*.bUseVinHeight        =*/ MTRUE,
    /*.bFullFrame           =*/ MTRUE,
    /*.bKeepRatio           =*/ MTRUE,
    /*.uiRateDivisor        =*/ 1,
    /*.uiTargetBitRateKbs   =*/ 10000,
    /*.uiIPeriod            =*/ 90,
    /*.uiPPeriod            =*/ 1,
    /*.uiQpMin              =*/ 10,
    /*.uiQpMax              =*/ 42,
    /*.uiSliceCount         =*/ 1,
    /*.eEncodeMode          =*/ LH264E_EncoderMode_RATECONTROLWINDOWED,
    /*.eEncodeProfile       =*/ LH264_Profile_HIGH444,
    /*.eEncodeLevel         =*/ LH264_Level_5_2,
    /*.bNetServer          =*/ MTRUE,
    /*.szNetAddress         =*/ MNULL,
    /*.szRtmpPushLocation   =*/ MNULL,
    /*.szUsername           =*/ MNULL,
    /*.szPassword           =*/ MNULL,
    /*.szFolderName         =*/ MNULL,
    /*.uiNetPort            =*/ 0,
    /*.uiMtu                =*/ 0,
    /*.eNetProtocol         =*/ LNetStreamer_Protocol_RTSP,
    /*.eEncodePixFmt        =*/ LPixelFormat_MP_Y8_U8V8_420,
    /*.uiFaq                =*/ 75,
    /*.bDfEnabled           =*/ MTRUE,
    /*.bEnableSrt           =*/ MFALSE,
    /*.uiOutMosIdx          =*/ 0,
    /*.eRotation            =*/ LVPC_Rotation_NONE,
    /*.uiHlsDuration        =*/ 0,
    /*.uiHlsCount           =*/ 0,
    /*.szHlsFolder          =*/ MNULL,
    /*.szHlsMaster          =*/ MNULL,
    /*.szNetInterface       =*/ MNULL,
    /*.bEnableIpv6          =*/ MFALSE,
    /*.eSrtMode             =*/ (LNetStreamer_SrtMode)(0),
    /*.oInMosMod            =*/ MosaicModule_Construct,
    /*.oEncMod              =*/ EncoderModule_Construct,
    /*.oWrFileMod           =*/ WriteFileModule_Construct,
    /*.oRtspSrvMod          =*/ RtspServerModule_Construct,
    /*.oDecMod              =*/ DecoderModule_Construct,
    /*.oRotModA             =*/ RotateModule_Construct,
    /*.oRotModB             =*/ RotateModule_Construct,
};

typedef struct tagAinParam
{
    MBOOL           bEnabled;
    MBOOL           bMtxIn;
    MUINT           uiIndex;
    LAudioFormat    eAudioFmt;
    MUINT           uiSamplingRate;
    MUINT           uiNbChannels;
    MUINT           uiSampleSize;
    AinModule       oAinMod;
    AinAlsaModule   oAinAlsaMod;

} AinParam;

const AinParam g_oDefaultAinParam =
{
    /*.bEnabled         =*/ MFALSE,
    /*.bMtxIn           =*/ MTRUE,
    /*.uiIndex          =*/ 0,
    /*.eAudioFmt        =*/ LAudioFormat_INVALID,
    /*.uiSamplingRate   =*/ 48000,
    /*.uiNbChannels     =*/ 2,
    /*.uiSampleSize     =*/ 16,
    /*.oAinMod          =*/ AinModule_Construct,
    /*.oAinAlsaMod      =*/ AinAlsaModule_Construct,
};

typedef struct tagAoutParam
{
    MBOOL               bEnabled;
    MBOOL               bMtxOut;
    MUINT               uiIndex;
    AinParam*           poAinParam;
    ModuleSyncMaster*   poModSyncMst;
    AoutModule          oAoutMod;
    AoutAlsaModule      oAoutAlsaMod;

} AoutParam;

const AoutParam g_oDefaultAoutParam =
{
    /*.bEnabled         =*/ MFALSE,
    /*.bMtxOut          =*/ MTRUE,
    /*.uiIndex          =*/ 0,
    /*.poAinParam       =*/ MNULL,
    /*.poModSyncMst     =*/ MNULL,
    /*.oAoutMod         =*/ AoutModule_Construct,
    /*.oAoutAlsaMod     =*/ AoutAlsaModule_Construct,
};

typedef struct tagAudioStreamParam
{
    MBOOL                       bEnable;

    // Parameters
    AinParam*                   poAinParam;
    const char*                 szNetAddress;
    const char*                 szRtmpPushLocation;
    const char*                 szUsername;
    const char*                 szPassword;
    const char*                 szFolderName;
    MUINT                       uiNetPort;
    MUINT                       uiMtu;
    LNetStreamer_Protocol       eNetProtocol;

    // AAC parameters
    LAudioCodec_AACStreamFormat eAacStreamFormat;
    LAudioCodec_AACProfile      eAacProfile;
    MUINT32                     uiAacBitrate;
    LAudioCodec_AACQualityLevel eAacQualityLvl;
    MBOOL                       bUseTns;
    MBOOL                       bEnableSrt;
    MUINT                       uiHlsDuration;
    MUINT                       uiHlsCount;
    const char*                 szHlsFolder;
    const char*                 szHlsMaster;
    const char*                 szNetInterface;
    MBOOL                       bEnableIpv6;
    LNetStreamer_SrtMode        eSrtMode;

    // Modules
    AudioEncModule              oAudioEncMod;
    RtspServerModule*           poRtspSrvMod;

} AudioStreamParam;

const AudioStreamParam g_oDefaultAudioStreamParam =
{
    /*.bEnable              =*/ MFALSE,
    /*.poAinParam           =*/ MNULL,
    /*.szNetAddress        =*/ MNULL,
    /*.szRtmpPushLocation  =*/ MNULL,
    /*.szUsername          =*/ MNULL,
    /*.szPassword          =*/ MNULL,
    /*.szFolderName        =*/ MNULL,
    /*.uiNetPort           =*/ 0,
    /*.uiMtu               =*/ 0,
    /*.eNetProtocol        =*/ LNetStreamer_Protocol_RTSP,
    /*.eAacStreamFormat     =*/ LAudioCodec_AACStreamFormat_ADTS,
    /*.eAacProfile          =*/ LAudioCodec_AACProfile_LC,
    /*.eAacBitrate          =*/ 96000,
    /*.eAacQualityLvl       =*/ LAudioCodec_AACQualityLevel_2,
    /*.bUseTns              =*/ MTRUE,
    /*.bEnableSrt           =*/ MFALSE,
    /*.uiHlsDuration        =*/ 0,
    /*.uiHlsCount           =*/ 0,
    /*.szHlsFolder          =*/ MNULL,
    /*.szHlsMaster          =*/ MNULL,
    /*.szNetInterface       =*/ MNULL,
    /*.bEnableIpv6          =*/ MFALSE,
    /*.eSrtMode             =*/ (LNetStreamer_SrtMode)(0),
    /*.oAudioEncMod         =*/ AudioEncModule_Construct,
    /*.poRtspSrvMod         =*/ MNULL
};

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

VinParam    g_aoVinParam[MAX_VIN_COUNT];
VoutParam   g_aoVoutParam[MAX_VOUT_COUNT];
AinParam    g_aoAinParam[MAX_AIN_COUNT];
AoutParam   g_aoAoutParam[MAX_AOUT_COUNT];
StreamParam g_aoStreamParam[MAX_STREAM_COUNT];
AudioStreamParam g_aoAudioStreamParam[MAX_STREAM_COUNT];

MBOOL g_bSigTerm = MFALSE;

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Function:       InitInMosParam

\************************************************************************************************************/
LStatus InitInMosParam(
    MUINT       uiInMosIdx,
    MUINT       uiFrameWidth,
    MUINT       uiFrameHeight,
    MUINT       uiFrameRateNum,
    MUINT       uiFrameRateDen,
    VoutParam*  poVoutParam,
    MBOOL       bDoScaling)
{
    const MUINT uiDiv = (poVoutParam->uiInputCount <  2) ? 1 :
                        (poVoutParam->uiInputCount <  5) ? 2 :
                        (poVoutParam->uiInputCount < 10) ? 3 : 4;
    LRECT32 oSrcRect;
    LRECT32 oDstRect;
    LPOS32  oPos;

    oPos.iX          = uiInMosIdx % uiDiv;
    oPos.iY          = uiInMosIdx / uiDiv;
    oDstRect.iLeft   = (oPos.iX * poVoutParam->oSize.iWidth) / uiDiv;
    oDstRect.iTop    = (oPos.iY * poVoutParam->oSize.iHeight) / uiDiv;
    oDstRect.iRight  = ((oPos.iX + 1) * poVoutParam->oSize.iWidth) / uiDiv;
    oDstRect.iBottom = ((oPos.iY + 1) * poVoutParam->oSize.iHeight) / uiDiv;

    if (!bDoScaling
        && (((oDstRect.iRight - oDstRect.iLeft) > uiFrameWidth)
            || ((oDstRect.iBottom - oDstRect.iTop) > uiFrameHeight)))
    {
        oDstRect.iRight  = oDstRect.iLeft + uiFrameWidth;
        oDstRect.iBottom = oDstRect.iTop + uiFrameHeight;
    }

    oSrcRect.iLeft   = 0;
    oSrcRect.iTop    = 0;
    oSrcRect.iRight  = uiFrameWidth;
    oSrcRect.iBottom = uiFrameHeight;

    LSIZE oFrameSize;

    oFrameSize.iWidth  = uiFrameWidth;
    oFrameSize.iHeight = uiFrameHeight;

    LStatus eStatus = MosMod_SetParameters(
                                &poVoutParam->oOutMosMod,
                                uiInMosIdx,
                                &oFrameSize,
                                &oSrcRect,
                                &oDstRect,
                                uiFrameRateNum,
                                uiFrameRateDen,
                                0,
                                MTRUE,
                                1.0f);

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        MUINT uiFpsNum;
        MUINT uiFpsDen;

        eStatus = MosMod_GetOutputFrameRate(&poVoutParam->oOutMosMod, &uiFpsNum, &uiFpsDen);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            if (poVoutParam->uiIndex < MAX_VOUTMTX_COUNT)
            {
                VoutMod_SetInputFrameRate(&poVoutParam->oVoutMod, uiFpsNum, uiFpsDen);
            }
            else if (poVoutParam->uiIndex < MAX_VOUT_COUNT)
            {
              #ifdef LINUX
                VoutGlxMod_SetInputFrameRate(&poVoutParam->oVoutGlxMod, uiFpsNum, uiFpsDen);
              #else
                assert(MFALSE);
              #endif
            }
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       SetRtspServerParams

\************************************************************************************************************/
LStatus SetRtspServerParams(
    StreamParam*        poVideoStreamParam,
    AudioStreamParam*   poAudioStreamParam,
    MUINT               uiMaxNaluSizeBytes,
    MUINT               uiMaxAudioBufSizeBytes,
    MUINT               uiStreamIdx)
{
    LStatus eStatus = LStatus_OK;

    MBOOL bVideoEnabled = poVideoStreamParam->bEnable && poVideoStreamParam->bNetServer;
    MBOOL bAudioEnabled = poAudioStreamParam->bEnable;

    if (((bVideoEnabled) || (bAudioEnabled))
           && LSTATUS_IS_SUCCESS(eStatus))
    {
        char szDefaultFolderName[10];
        const char* szFolderName;

        snprintf(szDefaultFolderName, sizeof(szDefaultFolderName), "S%u", uiStreamIdx);

        LNetStreamer_Protocol   eNetProtocol;
        const char*             szNetAddress;
        const char*             szRtmpPushLocation;
        const char*             szUsername;
        const char*             szPassword;
        MUINT                   uiVideoPort = 0;
        MUINT                   uiAudioPort = 0;
        const MUINT             uiBasePort  = (15000 + (10 * uiStreamIdx));
        MUINT                   uiMtu = 0;
        MBOOL                   bEnableSrt = MFALSE;
        MUINT                   uiHlsDuration = 0;
        MUINT                   uiHlsCount = 0;
        const char*             szHlsFolder = NULL;
        const char*             szHlsMaster = NULL;
        const char*             szNetInterface = NULL;
        MBOOL                   bEnableIpv6 = MFALSE;
        LNetStreamer_SrtMode    eSrtMode = (LNetStreamer_SrtMode)(0);

        if (bVideoEnabled)
        {
            eNetProtocol = poVideoStreamParam->eNetProtocol;
            szNetAddress = poVideoStreamParam->szNetAddress;
            szRtmpPushLocation = poVideoStreamParam->szRtmpPushLocation;
            szUsername = poVideoStreamParam->szUsername;
            szPassword = poVideoStreamParam->szPassword;
            szFolderName = poVideoStreamParam->szFolderName;
            uiMtu = poVideoStreamParam->uiMtu;
            bEnableSrt = poVideoStreamParam->bEnableSrt;
            uiHlsDuration = poVideoStreamParam->uiHlsDuration;
            uiHlsCount = poVideoStreamParam->uiHlsCount;
            szHlsFolder = poVideoStreamParam->szHlsFolder;
            szHlsMaster = poVideoStreamParam->szHlsMaster;
            szNetInterface = poVideoStreamParam->szNetInterface;
            bEnableIpv6 = poVideoStreamParam->bEnableIpv6;
            eSrtMode = poVideoStreamParam->eSrtMode;
        }
        else
        {
            eNetProtocol = poAudioStreamParam->eNetProtocol;
            szNetAddress = poAudioStreamParam->szNetAddress;
            szRtmpPushLocation = poAudioStreamParam->szRtmpPushLocation;
            szUsername = poAudioStreamParam->szUsername;
            szPassword = poAudioStreamParam->szPassword;
            szFolderName = poAudioStreamParam->szFolderName;
            uiMtu = poAudioStreamParam->uiMtu;
            bEnableSrt = poAudioStreamParam->bEnableSrt;
            uiHlsDuration = poAudioStreamParam->uiHlsDuration;
            uiHlsCount = poAudioStreamParam->uiHlsCount;
            szHlsFolder = poAudioStreamParam->szHlsFolder;
            szHlsMaster = poAudioStreamParam->szHlsMaster;
            szNetInterface = poAudioStreamParam->szNetInterface;
            bEnableIpv6 = poAudioStreamParam->bEnableIpv6;
            eSrtMode = poAudioStreamParam->eSrtMode;
        }

        if (!szFolderName)
        {
            szFolderName = szDefaultFolderName;
        }

        if ((eNetProtocol == LNetStreamer_Protocol_TS)
             || (eNetProtocol == LNetStreamer_Protocol_RTP))
        {
            if (bVideoEnabled)
            {
                if (poVideoStreamParam->uiNetPort != 0)
                {
                    uiVideoPort = poVideoStreamParam->uiNetPort;
                }
                else
                {
                    uiVideoPort = uiBasePort;
                }
            }

            if (bAudioEnabled)
            {
                if (poAudioStreamParam->uiNetPort != 0)
                {
                    uiAudioPort = poAudioStreamParam->uiNetPort;
                }
                else
                {
                    uiAudioPort = (uiVideoPort != 0)
                                   ? (uiVideoPort + 4)
                                   : (uiBasePort + 4);
                }
            }
        }

        eStatus = RtspSrvMod_SetParams(
                      &(poVideoStreamParam->oRtspSrvMod),
                      uiMaxNaluSizeBytes,
                      uiMaxAudioBufSizeBytes,
                      szFolderName,
                      szNetAddress,
                      bVideoEnabled,
                      uiVideoPort,
                      bAudioEnabled,
                      uiAudioPort,
                      bAudioEnabled ? poAudioStreamParam->oAudioEncMod.uiAudioSpecificConfig : 0,
                      eNetProtocol,
                      szRtmpPushLocation,
                      szUsername,
                      szPassword,
                      uiMtu,
                      bEnableSrt,
                      uiHlsDuration,
                      uiHlsCount,
                      szHlsFolder,
                      szHlsMaster,
                      szNetInterface,
                      bEnableIpv6,
                      eSrtMode);
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       InitStream

\************************************************************************************************************/
LStatus InitStream(LDevice_Handle hDevice, StreamParam* poStreamParam, MUINT* puiMaxNaluSizeBytes)
{
    poStreamParam->bSrvAddrDisplayed        = MFALSE;
    poStreamParam->uiNextStatTime           = 0;
    poStreamParam->uiSec                    = 0;
    poStreamParam->uiMin                    = 0;
    poStreamParam->uiHour                   = 0;
    poStreamParam->uiOldAvgIdx              = 0;
    poStreamParam->uiNewAvgIdx              = 0;
    poStreamParam->auiCapturedFrameCount[0] = 0;
    poStreamParam->auiEncodedFrameCount[0]  = 0;
    poStreamParam->auiEncodedBytes[0]       = 0;
    poStreamParam->auiElapsedTimeUsec[0]    = 0;

    if (!poStreamParam->bEnable)
    {
        return LStatus_OK;
    }

    LBuffer_Attributes  oInEncBufAttrib;
    LBuffer_Attributes  oMosBufAttrib;
    LSIZE               oMosFrameSize;
    LSIZE               oSubFrameSizePadded;
    MBOOL               bWrFile = (poStreamParam->szDstFilePath != MNULL) ? MTRUE : MFALSE;
    MBOOL               bVout   = (poStreamParam->poVoutParam != MNULL) ? MTRUE : MFALSE;
    MBOOL               bRotate = (poStreamParam->eRotation != LVPC_Rotation_NONE) ? MTRUE : MFALSE;

    EncMod_GetPaddedFrameSize(&poStreamParam->oSubFrameSize, &oSubFrameSizePadded);

    oInEncBufAttrib.eAttributeType                  = LBuffer_Type_VIDEO;
    oInEncBufAttrib.oVideoAttributes.uiWidth        = oSubFrameSizePadded.iWidth;
    oInEncBufAttrib.oVideoAttributes.uiHeight       = oSubFrameSizePadded.iHeight;
    oInEncBufAttrib.oVideoAttributes.ePixelFormat   = poStreamParam->eEncodePixFmt;

    if (bRotate)
    {
        if ((poStreamParam->eRotation == LVPC_Rotation_90)
            || (poStreamParam->eRotation == LVPC_Rotation_270)
            || (poStreamParam->eRotation == LVPC_Rotation_FLIP_90)
            || (poStreamParam->eRotation == LVPC_Rotation_FLIP_270))
        {
            oMosFrameSize.iWidth  = poStreamParam->oSubFrameSize.iHeight;
            oMosFrameSize.iHeight = poStreamParam->oSubFrameSize.iWidth;
        }
        else
        {
            oMosFrameSize = poStreamParam->oSubFrameSize;
        }

        oMosBufAttrib.eAttributeType                = LBuffer_Type_VIDEO;
        oMosBufAttrib.oVideoAttributes.uiWidth      = oMosFrameSize.iWidth;
        oMosBufAttrib.oVideoAttributes.uiHeight     = oMosFrameSize.iHeight;
        oMosBufAttrib.oVideoAttributes.ePixelFormat = LPixelFormat_R8G8B8A8;
    }
    else
    {
        oMosFrameSize = poStreamParam->oSubFrameSize;
        oMosBufAttrib = oInEncBufAttrib;
    }

    LStatus eStatus = MosMod_Init(
                        &poStreamParam->oInMosMod,
                        hDevice,
                        16,
                        &oMosBufAttrib,
                        &oMosFrameSize,
                        MNULL,
                        MTRUE,
                        MFALSE);

    LSIZE oDstFrameSize;
    LPOS  oDstFramePos;

    oDstFrameSize.iWidth    = poStreamParam->oFrameSize.iWidth  / poStreamParam->oFrameDiv.iWidth;
    oDstFrameSize.iHeight   = poStreamParam->oFrameSize.iHeight / poStreamParam->oFrameDiv.iHeight;
    oDstFramePos.iX         = 0;
    oDstFramePos.iY         = 0;

    MUINT i;
    for (i = 0; (i < poStreamParam->uiVinCount) && LSTATUS_IS_SUCCESS(eStatus); i++)
    {
        VinModule* poVinMod = &(poStreamParam->apoVinParam[i]->oVinMod);

        LRECT32 oSrcRect;
        LRECT32 oDstRect;
        LSIZE   oDstSize = poVinMod->oFrameSize;

        if (poStreamParam->bKeepRatio)
        {
            oDstSize.iHeight = (oDstSize.iHeight * oDstFrameSize.iWidth) / oDstSize.iWidth;
            oDstSize.iWidth  = oDstFrameSize.iWidth;

            if (oDstSize.iHeight > oDstFrameSize.iHeight)
            {
                oDstSize.iWidth  = (oDstSize.iWidth * oDstFrameSize.iHeight) / oDstSize.iHeight;
                oDstSize.iHeight = oDstFrameSize.iHeight;
            }
        }
        else
        {
            oDstSize = oDstFrameSize;
        }

        oSrcRect.iLeft      = 0;
        oSrcRect.iTop       = 0;
        oSrcRect.iRight     = poVinMod->oFrameSize.iWidth;
        oSrcRect.iBottom    = poVinMod->oFrameSize.iHeight;

        oDstRect.iLeft      = oDstFramePos.iX * (oDstFrameSize.iWidth + poStreamParam->oBezelSize.iWidth)
                              - poStreamParam->oSubFramePos.iX;
        oDstRect.iTop       = oDstFramePos.iY * (oDstFrameSize.iHeight + poStreamParam->oBezelSize.iHeight)
                              - poStreamParam->oSubFramePos.iY;
        oDstRect.iRight     = oDstRect.iLeft + oDstSize.iWidth;
        oDstRect.iBottom    = oDstRect.iTop  + oDstSize.iHeight;

        eStatus = MosMod_SetParameters(
                            &poStreamParam->oInMosMod,
                            i,
                            &poVinMod->oFrameSize,
                            &oSrcRect,
                            &oDstRect,
                            poVinMod->uiFrameRate1000x,
                            1000,
                            0,
                            MFALSE,
                            1.0f);

        oDstFramePos.iX++;

        if (oDstFramePos.iX >= poStreamParam->oFrameDiv.iWidth)
        {
            oDstFramePos.iX = 0;
            oDstFramePos.iY++;
        }
    }

    MUINT uiFpsNum;
    MUINT uiFpsDen;

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = MosMod_GetOutputFrameRate(&poStreamParam->oInMosMod, &uiFpsNum, &uiFpsDen);
    }

    if (bRotate && LSTATUS_IS_SUCCESS(eStatus))
    {
        LRECT32 oSrcRect;
        LPOS    oDstPos;

        oSrcRect.iLeft   = 0;
        oSrcRect.iTop    = 0;
        oSrcRect.iRight  = oMosFrameSize.iWidth;
        oSrcRect.iBottom = oMosFrameSize.iHeight;
        oDstPos.iX       = 0;
        oDstPos.iY       = 0;

        LBuffer_Attributes oRotBufAttrib = oInEncBufAttrib;

        oRotBufAttrib.oVideoAttributes.ePixelFormat = oMosBufAttrib.oVideoAttributes.ePixelFormat;

        eStatus = RotMod_Init(
                        &poStreamParam->oRotModA,
                        hDevice,
                        16,
                        &oRotBufAttrib,
                        &poStreamParam->oSubFrameSize,
                        &oSrcRect,
                        &oDstPos,
                        poStreamParam->eRotation);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            // Since rotation is not supported for all pixel formats (ex: 420 and 422), we make rotation in
            // RGB and then translate pixel format with an other module.

            oSrcRect.iLeft   = 0;
            oSrcRect.iTop    = 0;
            oSrcRect.iRight  = poStreamParam->oSubFrameSize.iWidth;
            oSrcRect.iBottom = poStreamParam->oSubFrameSize.iHeight;
            oDstPos.iX       = 0;
            oDstPos.iY       = 0;

            eStatus = RotMod_Init(
                            &poStreamParam->oRotModB,
                            hDevice,
                            16,
                            &oInEncBufAttrib,
                            &poStreamParam->oSubFrameSize,
                            &oSrcRect,
                            &oDstPos,
                            LVPC_Rotation_NONE);
        }

        poStreamParam->oRotModA.oInLink.uiRateDivisor = poStreamParam->uiFrameRateDivisor;
    }
    else
    {
        poStreamParam->oEncMod.oInLink.uiRateDivisor = poStreamParam->uiFrameRateDivisor;
    }

    MUINT uiMaxNaluSizeBytes = (poStreamParam->oSubFrameSize.iWidth
                                * poStreamParam->oSubFrameSize.iHeight
                                * LPixelFormat_Get3xBppAllPlanes(poStreamParam->eEncodePixFmt)) / (3 * 8);

    LBuffer_Attributes oOutEncBufAttrib;

    oOutEncBufAttrib.oSystemLinearAttributes.eAttributeType = LBuffer_Type_SYSTEM_LINEAR;
    oOutEncBufAttrib.oSystemLinearAttributes.uiSize         = uiMaxNaluSizeBytes;
    oOutEncBufAttrib.oSystemLinearAttributes.pvSystemMemory = MNULL;

    if(puiMaxNaluSizeBytes != MNULL)
    {
        *puiMaxNaluSizeBytes = uiMaxNaluSizeBytes;
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        uiFpsDen *= poStreamParam->uiFrameRateDivisor;

        eStatus = EncMod_Init(
                        &(poStreamParam->oEncMod),
                        hDevice,
                        16,
                        &oOutEncBufAttrib,
                        poStreamParam->eEncodeMode,
                        poStreamParam->eEncodeProfile,
                        poStreamParam->eEncodeLevel,
                        poStreamParam->uiSliceCount,
                        poStreamParam->uiFaq,
                        poStreamParam->bDfEnabled);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = EncMod_SetParam(
                        &(poStreamParam->oEncMod),
                        &poStreamParam->oSubFrameSize,
                        uiFpsNum,
                        uiFpsDen,
                        poStreamParam->uiTargetBitRateKbs,
                        poStreamParam->uiIPeriod,
                        poStreamParam->uiPPeriod,
                        poStreamParam->uiQpMin,
                        poStreamParam->uiQpMax);
        }
    }

    if (poStreamParam->bNetServer && LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = RtspSrvMod_Init(&(poStreamParam->oRtspSrvMod), hDevice);
    }

    if (bWrFile && LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = WrFileMod_Init(&(poStreamParam->oWrFileMod), hDevice, poStreamParam->szDstFilePath, MFALSE);
    }

    if (bVout && LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = DecMod_Init(&poStreamParam->oDecMod, hDevice, 24, uiFpsNum, uiFpsDen, MFALSE, MFALSE);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = InitInMosParam(
                            poStreamParam->uiOutMosIdx,
                            poStreamParam->oSubFrameSize.iWidth,
                            poStreamParam->oSubFrameSize.iHeight,
                            uiFpsNum,
                            uiFpsDen,
                            poStreamParam->poVoutParam,
                            MFALSE);
        }
    }

    // Pipeline configuration:
    //
    //  Vin--->MosaicIn--->Encode--+-->RtspSrv                      >> Optional with "-rtsp" argument.
    //                             |
    //                             +-->WrFile                       >> Optional with "-o" argument.
    //                             |
    //                             +-->Decode--->MosaicOut--->Vout  >> Optional with "-od" argument.

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        char szVin[100];
        char szBranch[200] = "-->";

        printf("***** Encode at %ux%u@%.3ffps *****\n", poStreamParam->oSubFrameSize.iWidth,
               poStreamParam->oSubFrameSize.iHeight, (MFLOAT32)uiFpsNum / uiFpsDen);

        for (i = 0; i < poStreamParam->uiVinCount - 1; i++)
        {
            printf("[Vin%u]--+\n        |\n", poStreamParam->apoVinParam[i]->uiIndex);
        }

        snprintf(szVin, sizeof(szVin),
                 "[Vin%u]--%s->[Mosaic-In]%s--->[Encode]-",
                 poStreamParam->apoVinParam[poStreamParam->uiVinCount - 1]->uiIndex,
                 (poStreamParam->uiVinCount > 1) ? "+-" : "",
                 bRotate ? "--->[Rotate]" : "");

        MUINT uiOutputCount = (poStreamParam->bNetServer ? 1 : 0)
                              + (bWrFile ? 1 : 0)
                              + (bVout ? 1 : 0);

        if (uiOutputCount > 1)
        {
            strncat_wz(szVin, "-+", sizeof(szVin));
            int iSpaces = (int)(strlen(szVin) - 1);
            snprintf(szBranch, sizeof(szBranch),
                     "%*s|\n"
                     "%*s+", iSpaces, " ", iSpaces, " ");
        }

        printf("%s", szVin);

        if (poStreamParam->bNetServer)
        {
            printf("-->[RtspSrv]\n");

            if (--uiOutputCount > 0)
            {
                printf("%s", szBranch);
            }
        }

        if (bWrFile)
        {
            printf("-->[WrFile: \"%s\"]\n", poStreamParam->szDstFilePath);

            if (--uiOutputCount > 0)
            {
                printf("%s", szBranch);
            }
        }

        if (bVout)
        {
            MUINT uiVoutIdx = poStreamParam->poVoutParam->uiIndex;
            printf("-->[Decode]--->%u:[Mosaic-Out]--->[Vout%s%u]\n", poStreamParam->uiOutMosIdx,
                   (uiVoutIdx < MAX_VOUTMTX_COUNT) ? "" : "-x",
                   (uiVoutIdx < MAX_VOUTMTX_COUNT) ? uiVoutIdx : (uiVoutIdx - MAX_VOUTMTX_COUNT));
        }
    }

    for (i = 0; (i < poStreamParam->uiVinCount) && LSTATUS_IS_SUCCESS(eStatus); i++)
    {
        eStatus = ModLnk_Connect(&(poStreamParam->apoVinParam[i]->oVinMod.oOutLink),
                                 &(poStreamParam->oInMosMod.aoInLink[i]));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        if (bRotate)
        {
            eStatus = ModLnk_Connect(&(poStreamParam->oInMosMod.oOutLink),
                                     &(poStreamParam->oRotModA.oInLink));

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                eStatus = ModLnk_Connect(&(poStreamParam->oRotModA.oOutLink),
                                         &(poStreamParam->oRotModB.oInLink));
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                eStatus = ModLnk_Connect(&(poStreamParam->oRotModB.oOutLink),
                                         &(poStreamParam->oEncMod.oInLink));
            }
        }
        else
        {
            eStatus = ModLnk_Connect(&(poStreamParam->oInMosMod.oOutLink),
                                     &(poStreamParam->oEncMod.oInLink));
        }
    }

    if (poStreamParam->bNetServer && LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = ModLnk_Connect(&(poStreamParam->oEncMod.oOutLink), &(poStreamParam->oRtspSrvMod.oVideoInLink));
    }

    if (bWrFile && LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = ModLnk_Connect(&(poStreamParam->oEncMod.oOutLink), &(poStreamParam->oWrFileMod.oInLink));
    }

    if (bVout && LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = ModLnk_Connect(&(poStreamParam->oEncMod.oOutLink), &(poStreamParam->oDecMod.oInLink));

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = ModLnk_Connect(
                            &poStreamParam->oDecMod.oOutLink,
                            &poStreamParam->poVoutParam->oOutMosMod.aoInLink[poStreamParam->uiOutMosIdx]);
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       InitAVSyncParams

\************************************************************************************************************/
void InitAVSyncParams()
{
    MUINT i;

    // Synchronize audio and video streams that come from the same input and use the same output.
    for (i = 0; (i < MAX_AIN_COUNT) && (i < MAX_VIN_COUNT); i++)
    {
        AinParam* poAinParam = &g_aoAinParam[i];
        VinParam* poVinParam = &g_aoVinParam[i];

        if (poAinParam->bEnabled
             && poAinParam->bMtxIn
             && poVinParam->bEnabled)
        {
            // Check for "-d" argument used.
            //    Vin(i)--->MosaicOut--->Vout(uiOutIdx)
            //    Ain(i)---------------->Aout(uiOutIdx)

            VoutParam* poVoutParam = poVinParam->poVoutParam;

            if (poVoutParam != MNULL)
            {
                if (poVoutParam->uiIndex < MAX_AOUT_COUNT)
                {
                    AoutParam* poAoutParam = &g_aoAoutParam[poVoutParam->uiIndex];

                    if ((poAoutParam->poAinParam == poAinParam)
                         && (poAoutParam->bMtxOut))
                    {
                        printf("A/V stream from input %u synchronized on output %u\n", i, poAoutParam->uiIndex);
                        poVoutParam->bIsMasterSync = MFALSE;
                        poAoutParam->poModSyncMst = &(poVoutParam->oVoutMod.oModSync.oModSyncMst);
                    }
                }
            }
            else
            {
                // "-d" not used, check for "-od" argument used.
                //    Vin(i)--->MosaicIn--->Encode--->Decode--->MosaicOut--->Vout(uiOutIdx)
                //    Ain(i)------------------------------------------------>Aout(uiOutIdx)
                MUINT j;
                for(j = 0; j < MAX_STREAM_COUNT; j++)
                {
                    StreamParam* poStreamParam = &g_aoStreamParam[j];
                    poVoutParam = poStreamParam->poVoutParam;

                    if(poVoutParam != MNULL)
                    {
                        AoutParam* poAoutParam = &g_aoAoutParam[poVoutParam->uiIndex];

                        MUINT k;
                        for(k = 0; k < poStreamParam->uiVinCount; k++)
                        {
                            VinParam* poStreamVinParam = poStreamParam->apoVinParam[k];

                            if((poStreamVinParam == poVinParam)
                                && (poAoutParam->poAinParam == poAinParam)
                                && (poAoutParam->bMtxOut))
                            {
                                printf("A/V streams from input %u synchronized on output %u\n", i, poAoutParam->uiIndex);
                                poVoutParam->bIsMasterSync = MFALSE;
                                poAoutParam->poModSyncMst = &(poVoutParam->oVoutMod.oModSync.oModSyncMst);
                            }
                        }
                    }
                }
            }
        }
    }
}

/************************************************************************************************************\

Function:       GetAudioCodecSamplingRate

\************************************************************************************************************/
LAudioCodec_SamplingRate GetAudioCodecSamplingRate(MUINT32 uiSamplingRate)
{
    switch(uiSamplingRate)
    {
        case 8000:  return LAudioCodec_SamplingRate_8000Hz;
        case 11025: return LAudioCodec_SamplingRate_11025Hz;
        case 12000: return LAudioCodec_SamplingRate_12000Hz;
        case 16000: return LAudioCodec_SamplingRate_16000Hz;
        case 22050: return LAudioCodec_SamplingRate_22050Hz;
        case 24000: return LAudioCodec_SamplingRate_24000Hz;
        case 32000: return LAudioCodec_SamplingRate_32000Hz;
        case 44100: return LAudioCodec_SamplingRate_44100Hz;
        case 48000: return LAudioCodec_SamplingRate_48000Hz;
        case 88200: return LAudioCodec_SamplingRate_88200Hz;
        case 96000: return LAudioCodec_SamplingRate_96000Hz;
        default:    return LAudioCodec_SamplingRate_UNKNOWN;
    }
}

/************************************************************************************************************\

Function:       GetAudioFormat

\************************************************************************************************************/
LAudioFormat GetAudioFormat(MUINT32 uiSamplingRate, MUINT32 uiSampleSize, MUINT32 uiNbChannels)
{
    LAudioFormat eFormat = LAudioFormat_INVALID;

    return LSTATUS_IS_SUCCESS(LAudioFormat_GetAudioFormat(
                                uiSamplingRate,
                                uiSampleSize,
                                uiNbChannels,
                                1,
                                &eFormat))
            ? eFormat : LAudioFormat_INVALID;
}

/************************************************************************************************************\

Function:       InitAudioStream

\************************************************************************************************************/
LStatus InitAudioStream(LDevice_Handle hDevice, MUINT uiStreamIdx)
{
    AudioStreamParam*   poAudioStreamParam  = &g_aoAudioStreamParam[uiStreamIdx];
    StreamParam*        poVideoStreamParam  = &g_aoStreamParam[uiStreamIdx];
    AinParam*           poAinParam          = poAudioStreamParam->poAinParam;

    if (!poAudioStreamParam->bEnable)
    {
        return LStatus_OK;
    }

    if (poAinParam == MNULL)
    {
        return LStatus_FAIL;
    }

    MUINT32     uiSamplingRate  = poAinParam->uiSamplingRate;
    MUINT32     uiNbChannels    = poAinParam->uiNbChannels;
    MUINT32     uiSampleSize    = poAinParam->uiSampleSize;
    LStatus     eStatus         = LStatus_OK;

    LAudioCodec_ChannelCfg eChannelCfg = LAudioCodec_ChannelCfg_UNKNOWN;

    if(uiNbChannels == 1)
    {
        eChannelCfg = LAudioCodec_ChannelCfg_MONO;
    }
    else if(uiNbChannels == 2)
    {
        eChannelCfg = LAudioCodec_ChannelCfg_STEREO;
    }
    else
    {
        printf("Error, invalid number of audio channel: %u. Supported: 1 or 2\n", uiNbChannels);
        eStatus = LStatus_INVALID_PARAM;
    }

    if(uiSampleSize != 16)
    {
        printf("Error, invalid audio sample size: %u. (supported: 16 bits per sample)\n", uiSampleSize);
        eStatus = LStatus_INVALID_PARAM;
    }

    LAudioCodec_SamplingRate eSamplingRate = GetAudioCodecSamplingRate(uiSamplingRate);
    if(eSamplingRate == LAudioCodec_SamplingRate_UNKNOWN)
    {
        printf("Error, invalid audio sampling rate: %uHz.\nSupported sampling rates:\n"
               "  8000Hz, 11025Hz, 12000Hz, 16000Hz, 22050Hz, 24000Hz\n  32000Hz, 44100Hz, 48000Hz, 88200Hz, 96000Hz\n",
               uiSamplingRate);
        eStatus = LStatus_INVALID_PARAM;
    }

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = AudioEncMod_Init(
                      &(poAudioStreamParam->oAudioEncMod),
                      hDevice,
                      5,
                      LBuffer_Type_SYSTEM_LINEAR,
                      eSamplingRate,
                      eChannelCfg,
                      poAudioStreamParam->eAacStreamFormat,
                      poAudioStreamParam->eAacProfile,
                      poAudioStreamParam->eAacQualityLvl,
                      poAudioStreamParam->bUseTns,
                      poAudioStreamParam->uiAacBitrate);
    }

    if(!(poVideoStreamParam->bEnable && poVideoStreamParam->bNetServer)
        && LSTATUS_IS_SUCCESS(eStatus))
    {
        // Must init rtsp server for audio stream alone.
        eStatus = RtspSrvMod_Init(poAudioStreamParam->poRtspSrvMod, hDevice);
    }

    // Pipeline configuration:
    //
    //  Video pipeline --------------+        >> Optional.
    //                               |
    //  Ain--+-->AudioEncode----->RtspSrv
    //       |
    //       +------------------->Aout        >> Optional.

    ModuleLink* poOutLnk = poAinParam->bMtxIn
                            ? (&(poAinParam->oAinMod.oOutLink))
                            : (&(poAinParam->oAinAlsaMod.oOutLink));

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = ModLnk_Connect(
                      poOutLnk,
                      &(poAudioStreamParam->oAudioEncMod.oInLink));

        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = ModLnk_Connect(
                        &(poAudioStreamParam->oAudioEncMod.oOutLink),
                        &(poAudioStreamParam->poRtspSrvMod->oAudioInLink));
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       StartStream

\************************************************************************************************************/
LStatus StartStream(StreamParam* poStreamParam)
{
    LStatus eStatus = poStreamParam->bEnable ? LStatus_OK : LStatus_FAIL;

    if ((poStreamParam->poVoutParam != MNULL) && LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = DecMod_Start(&(poStreamParam->oDecMod));
    }

    if (poStreamParam->bNetServer && LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = RtspSrvMod_Start(&(poStreamParam->oRtspSrvMod));
    }

    if ((poStreamParam->szDstFilePath != MNULL) && LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = WrFileMod_Start(&(poStreamParam->oWrFileMod));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = EncMod_Start(&(poStreamParam->oEncMod));
    }

    if ((poStreamParam->eRotation != LVPC_Rotation_NONE) && LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = RotMod_Start(&(poStreamParam->oRotModB));

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = RotMod_Start(&(poStreamParam->oRotModA));
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = MosMod_Start(&(poStreamParam->oInMosMod));
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       StartAudioStream

\************************************************************************************************************/
LStatus StartAudioStream(AudioStreamParam* poAudioStreamParam)
{
    LStatus eStatus = poAudioStreamParam->bEnable ? LStatus_OK : LStatus_FAIL;

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        // Only start if video is disabled on this server. Otherwise it is already started.
        if(!poAudioStreamParam->poRtspSrvMod->bEnableVideo)
        {
            eStatus = RtspSrvMod_Start(poAudioStreamParam->poRtspSrvMod);
        }

        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = AudioEncMod_Start(&poAudioStreamParam->oAudioEncMod);
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       StopStream

\************************************************************************************************************/
LStatus StopStream(StreamParam* poStreamParam)
{
    LStatus eStatus = poStreamParam->bEnable ? LStatus_OK : LStatus_FAIL;

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        MosMod_Stop(&(poStreamParam->oInMosMod));

        if (poStreamParam->eRotation != LVPC_Rotation_NONE)
        {
            RotMod_Stop(&(poStreamParam->oRotModA));
            RotMod_Stop(&(poStreamParam->oRotModB));
        }

        EncMod_Stop(&(poStreamParam->oEncMod));

        if (poStreamParam->bNetServer)
        {
            RtspSrvMod_Stop(&(poStreamParam->oRtspSrvMod));
        }

        if (poStreamParam->szDstFilePath != MNULL)
        {
            WrFileMod_Stop(&(poStreamParam->oWrFileMod));
        }

        if (poStreamParam->poVoutParam != MNULL)
        {
            DecMod_Stop(&(poStreamParam->oDecMod));
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       StopAudioStream

\************************************************************************************************************/
LStatus StopAudioStream(AudioStreamParam* poAudioStreamParam)
{
    LStatus eStatus = poAudioStreamParam->bEnable ? LStatus_OK : LStatus_FAIL;

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        AudioEncMod_Stop(&poAudioStreamParam->oAudioEncMod);

        // Only stop if video is disabled on this server.
        // Otherwise it will be stopped with the video pipeline.
        if(!poAudioStreamParam->poRtspSrvMod->bEnableVideo)
        {
            RtspSrvMod_Stop(poAudioStreamParam->poRtspSrvMod);
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       DoStats

\************************************************************************************************************/
MBOOL DoStats(
    StreamParam*    poStreamParam,
    MUINT           uiStreamIdx,
    MUINT           uiVinIdx,
    MBOOL           bShow,
    MUINT64*        puiPrevBusyTimeCpu,
    MUINT64*        puiPrevTotalTimeCpu)
{
    const MUINT uiOneSec = 1000 * 1000;
    MBOOL       bShown   = MFALSE;

    VinModule* poVinMod = &poStreamParam->apoVinParam[uiVinIdx]->oVinMod;

    if (poVinMod->uiElapsedTimeUsec >= uiOneSec)
    {
        if (poStreamParam->uiNextStatTime == 0)
        {
            poStreamParam->uiNextStatTime = GetMonoTimeUsec();
        }

        if (GetMonoTimeUsec() >= poStreamParam->uiNextStatTime)
        {
            MUINT   uiCapturedFrameCountDiff = poVinMod->uiCapturedFrameCount;
            MUINT   uiEncodedFrameCountDiff  = poStreamParam->oEncMod.uiPictureCount;
            MUINT64 uiEncodedBytesDiff       = poStreamParam->oEncMod.uiEncodedBytes;
            MUINT64 uiElapsedTimeUsecDiff    = poVinMod->uiElapsedTimeUsec;

            uiCapturedFrameCountDiff -= poStreamParam->auiCapturedFrameCount[poStreamParam->uiOldAvgIdx];
            uiEncodedFrameCountDiff  -= poStreamParam->auiEncodedFrameCount[poStreamParam->uiOldAvgIdx];
            uiEncodedBytesDiff       -= poStreamParam->auiEncodedBytes[poStreamParam->uiOldAvgIdx];
            uiElapsedTimeUsecDiff    -= poStreamParam->auiElapsedTimeUsec[poStreamParam->uiOldAvgIdx];

            poStreamParam->uiNewAvgIdx++;

            if (poStreamParam->uiNewAvgIdx == AVG_PERIOD)
            {
                poStreamParam->uiNewAvgIdx = 0;
            }

            if (poStreamParam->uiOldAvgIdx == poStreamParam->uiNewAvgIdx)
            {
                poStreamParam->uiOldAvgIdx++;

                if (poStreamParam->uiOldAvgIdx == AVG_PERIOD)
                {
                    poStreamParam->uiOldAvgIdx = 0;
                }
            }

            poStreamParam->auiCapturedFrameCount[poStreamParam->uiNewAvgIdx]
                = poVinMod->uiCapturedFrameCount;
            poStreamParam->auiEncodedFrameCount[poStreamParam->uiNewAvgIdx]
                = poStreamParam->oEncMod.uiPictureCount;
            poStreamParam->auiEncodedBytes[poStreamParam->uiNewAvgIdx]
                = poStreamParam->oEncMod.uiEncodedBytes;
            poStreamParam->auiElapsedTimeUsec[poStreamParam->uiNewAvgIdx]
                = poVinMod->uiElapsedTimeUsec;

            MFLOAT32 fAvgCaptureFps     = 0;
            MFLOAT32 fAvgEncodedFps     = 0;
            MUINT    uiAvgBitrateKbps   = 0;

            if (uiElapsedTimeUsecDiff > 0)
            {
                fAvgCaptureFps = ((MFLOAT32)uiCapturedFrameCountDiff * 1000 * 1000)
                                 / uiElapsedTimeUsecDiff;
                fAvgEncodedFps = ((MFLOAT32)uiEncodedFrameCountDiff * 1000 * 1000)
                                 / uiElapsedTimeUsecDiff;
                uiAvgBitrateKbps = (uiEncodedBytesDiff * 1000 * 8) / uiElapsedTimeUsecDiff;
            }

            if (++poStreamParam->uiSec == 60)
            {
                poStreamParam->uiSec = 0;

                if (++poStreamParam->uiMin == 60)
                {
                    poStreamParam->uiMin = 0;
                    poStreamParam->uiHour++;
                }
            }

            if (bShow)
            {
                char szEncStat[128] = "";

                if (poStreamParam->bEnable)
                {
                    snprintf(szEncStat,
                             sizeof(szEncStat),
                             "| Enc[%u]: %u nalu, %u pict, %.02f fps, %.02f Mbps ",
                             uiStreamIdx,
                             poStreamParam->oEncMod.uiNaluCount,
                             poStreamParam->oEncMod.uiPictureCount,
                             fAvgEncodedFps,
                             ((MFLOAT32)uiAvgBitrateKbps) / 1000);
                }

                MUINT uiCpuUsage10x = GetCpuUsage(puiPrevBusyTimeCpu, puiPrevTotalTimeCpu);
                MUINT uiMemUsage10x = GetMemUsage();

                printf("[%u:%02u:%02u] Vin[%u]: %u frm, %u skip, %.02f fps %s| Cpu: %.01f%% | Mem: %.01f%%\n",
                       poStreamParam->uiHour,
                       poStreamParam->uiMin,
                       poStreamParam->uiSec,
                       poStreamParam->apoVinParam[uiVinIdx]->uiIndex,
                       poVinMod->uiTotalFrameCount,
                       poVinMod->uiTotalFrameCount - poVinMod->uiCapturedFrameCount,
                       fAvgCaptureFps,
                       szEncStat,
                       (MFLOAT32)uiCpuUsage10x / 10,
                       (MFLOAT32)uiMemUsage10x / 10);

                bShown = MTRUE;
            }

            poStreamParam->uiNextStatTime += uiOneSec;
        }
    }

    return bShown;
}

/************************************************************************************************************\

Function:       ShowModuleStates

\************************************************************************************************************/
void ShowModuleStates()
{
    MUINT i;

    printf("\n");

    for (i = 0; i < MAX_VIN_COUNT; i++)
    {
        if (g_aoVinParam[i].bEnabled)
        {
            printf("Vin[%u]:\n"
                   "-------\n", i);

            ModLnk_ShowState(&g_aoVinParam[i].oVinMod.oOutLink);
        }
    }

    printf("\n");

    for (i = 0; i < MAX_STREAM_COUNT; i++)
    {
        if (g_aoStreamParam[i].bEnable)
        {
            printf("Stream[%u]:\n"
                   "----------\n", i);

            printf("Mosaic-In:\n");
            ModLnk_ShowState(&g_aoStreamParam[i].oInMosMod.oOutLink);
            printf("Rotate-A:\n");
            ModLnk_ShowState(&g_aoStreamParam[i].oRotModA.oOutLink);
            printf("Rotate-B:\n");
            ModLnk_ShowState(&g_aoStreamParam[i].oRotModB.oOutLink);
            printf("Encoder:\n");
            ModLnk_ShowState(&g_aoStreamParam[i].oEncMod.oOutLink);
            printf("Decoder:\n");
            ModLnk_ShowState(&g_aoStreamParam[i].oDecMod.oOutLink);
        }
    }

    printf("\n");

    for (i = 0; i < MAX_VOUT_COUNT; i++)
    {
        if (g_aoVoutParam[i].uiInputCount > 0)
        {
            printf("Vout[%s%u]:\n"
                   "--------\n",
                   (i < MAX_VOUTMTX_COUNT) ? "" : "x",
                   (i < MAX_VOUTMTX_COUNT) ? i : (i - MAX_VOUTMTX_COUNT));

            printf("Mosaic-Out:\n");
            ModLnk_ShowState(&g_aoVoutParam[i].oOutMosMod.oOutLink);
        }
    }

    printf("\n");
}

/************************************************************************************************************\

Function:       GetAacProfileName

\************************************************************************************************************/
const char* GetAacProfileName(LAudioCodec_AACProfile eProfile)
{
    switch(eProfile)
    {
        case LAudioCodec_AACProfile_LC:     return "LC";
        case LAudioCodec_AACProfile_HEV1:   return "HEV1";
        case LAudioCodec_AACProfile_HEV2:   return "HEV2";
        default:                            return "UNKNOWN";
    }
}

/************************************************************************************************************\

Function:       GetAacStreamFormatName

\************************************************************************************************************/
const char* GetAacStreamFormatName(LAudioCodec_AACStreamFormat eStreamFormat)
{
    switch(eStreamFormat)
    {
        case LAudioCodec_AACStreamFormat_ADTS:  return "ADTS";
        case LAudioCodec_AACStreamFormat_ADIF:  return "ADIF";
        case LAudioCodec_AACStreamFormat_RAW:   return "RAW";
        default:                                return "UNKNOWN";
    }
}

/************************************************************************************************************\

Function:       GetAacQualityLevel

\************************************************************************************************************/
MUINT32 GetAacQualityLevel(LAudioCodec_AACQualityLevel eQualityLvl)
{
    switch(eQualityLvl)
    {
        case LAudioCodec_AACQualityLevel_0:  return 0;
        case LAudioCodec_AACQualityLevel_1:  return 1;
        case LAudioCodec_AACQualityLevel_2:  return 2;
        default:                             return MUINT32_MAX;
    }
}

/************************************************************************************************************\

Function:       ShowInputKeyOptions

\************************************************************************************************************/
void ShowInputKeyOptions()
{
    printf("Input key options:\n"
           "  'q'       : Quit.\n"
           "  'r'       : Restart.\n"
           "  'n'       : Cycle to next full screen video.\n"
           "  's'       : Show modules queue states.\n"
           "  '0-9,a-f' : Show stream stats.\n"
           "  'i'       : Show next Vin stats of the selected stream.\n"
           "  'P'       : Pause/Resume all streams.\n");
}

/************************************************************************************************************\

Function:       SignalHandler

\************************************************************************************************************/
void SignalHandler(int iSigNum)
{
    printf("Received signal: %d.\n", iSigNum);

    if (iSigNum == SIGTERM)
    {
        g_bSigTerm = MTRUE;
    }
}

/************************************************************************************************************\

Function:       main

\************************************************************************************************************/
int main(int iArgc, char **argv)
{
    ModThread_SetName("Main");

    // Do normal stop on kill.
    signal(SIGTERM, SignalHandler);

    StreamParam*        poStreamParam       = MNULL;
    StreamParam*        poPrevStreamParam   = MNULL;
    MUINT               uiStreamCount       = 0;
    AudioStreamParam*   poAudioStreamParam  = MNULL;
    AudioStreamParam*   poPrevAudioStreamParam = MNULL;
    MUINT               uiAudioStreamCount  = 0;
    MBOOL               bShowStats          = MFALSE;
    MUINT               uiMsgLogLevel       = 0;
    MBOOL               bEnableVcLog        = MFALSE;
    LRECT               oWinRect            = { 0, 0, 0, 0 };
    MBOOL               bForceWinRect       = MFALSE;
    MUINT32             uiMaxVinCount       = MAX_VIN_COUNT;
    MUINT32             uiMaxVoutMtxCount   = MAX_VOUTMTX_COUNT;
    MUINT32             uiMaxAinCount       = MAX_AIN_COUNT;
    MUINT32             uiMaxMtxAinCount    = 0;
    MUINT32             uiMaxAoutCount      = MAX_AOUT_COUNT;
    MUINT32             uiMaxMtxAoutCount   = 0;
    MBOOL               bArgError           = MFALSE;
    MBOOL               bAvSync             = MFALSE;
    MBOOL               bLowLatency         = MFALSE;
    MINT                i;

    // -------------------------------------------------------------------------------------------------------
    // Load library and init device

    Liberatus_Load();

    LDevice_Handle hDevice = Liberatus_GetDevice(0);
    LStatus        eStatus = (hDevice != MNULL) ? LStatus_OK : LStatus_FAIL;

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LVideoIn_GetCount(hDevice, &uiMaxVinCount);

        if (LSTATUS_IS_FAIL(eStatus))
        {
            printf("ERROR! LVideoIn_GetCount returned %d (%s).\n", eStatus, GetStatusStr(eStatus));
        }
    }
    else
    {
        printf("ERROR! Liberatus_GetDevice returned NULL handle.\n");
    }

    uiMaxVinCount = min(uiMaxVinCount, MAX_VIN_COUNT);

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LVideoOut_GetCount(hDevice, &uiMaxVoutMtxCount);

        if (LSTATUS_IS_FAIL(eStatus))
        {
            printf("ERROR! LVideoOut_GetCount returned %d (%s).\n", eStatus, GetStatusStr(eStatus));
        }
    }

    uiMaxVoutMtxCount = min(uiMaxVoutMtxCount, MAX_VOUTMTX_COUNT);

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LAudioIn_GetCount(hDevice, &uiMaxAinCount);

        if (LSTATUS_IS_FAIL(eStatus))
        {
            printf("ERROR! LAudioIn_GetCount returned %d (%s).\n", eStatus, GetStatusStr(eStatus));
        }
    }

    uiMaxAinCount = min(uiMaxAinCount, MAX_AIN_COUNT);
    uiMaxMtxAinCount = uiMaxAinCount;

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        MUINT32 uiAlsaDevCount = 0;

        if(AinAlsaMod_EnumDevices(&uiAlsaDevCount) == LStatus_OK)
        {
            uiMaxAinCount += uiAlsaDevCount;
            uiMaxAinCount  = min(uiMaxAinCount, MAX_AIN_COUNT);
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LAudioOut_GetCount(hDevice, &uiMaxAoutCount);

        if (LSTATUS_IS_FAIL(eStatus))
        {
            printf("ERROR! LAudioOut_GetCount returned %d (%s).\n", eStatus, GetStatusStr(eStatus));
        }
    }

    uiMaxAoutCount = min(uiMaxAoutCount, MAX_AOUT_COUNT);
    uiMaxMtxAoutCount = uiMaxAoutCount;

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        MUINT32 uiAlsaDevCount = 0;

        if(AoutAlsaMod_EnumDevices(&uiAlsaDevCount) == LStatus_OK)
        {
            uiMaxAoutCount += uiAlsaDevCount;
            uiMaxAoutCount  = min(uiMaxAoutCount, MAX_AOUT_COUNT);
        }
    }

    // -------------------------------------------------------------------------------------------------------
    // Parse arguments

    for (i = 0; i < MAX_VIN_COUNT; i++)
    {
        g_aoVinParam[i]             = g_oDefaultVinParam;
        g_aoVinParam[i].uiIndex     = i;
    }

    for (i = 0; i < MAX_VOUT_COUNT; i++)
    {
        g_aoVoutParam[i]            = g_oDefaultVoutParam;
        g_aoVoutParam[i].uiIndex    = i;
    }

    for (i = 0; i < MAX_AIN_COUNT; i++)
    {
        g_aoAinParam[i]             = g_oDefaultAinParam;
        g_aoAinParam[i].uiIndex     = i;
    }

    for (i = 0; i < MAX_AOUT_COUNT; i++)
    {
        g_aoAoutParam[i]            = g_oDefaultAoutParam;
        g_aoAoutParam[i].uiIndex    = i;
    }

    for (i = 0; i < MAX_STREAM_COUNT; i++)
    {
        g_aoStreamParam[i]                      = g_oDefaultStreamParam;
        g_aoAudioStreamParam[i]                 = g_oDefaultAudioStreamParam;
        g_aoAudioStreamParam[i].poRtspSrvMod    = &(g_aoStreamParam[i].oRtspSrvMod);
    }

    for (i = 1; i < iArgc; i++)
    {
        if (strcmp(argv[i], "-s") == 0)
        {
            if (++i < iArgc)
            {
                MUINT uiStrmIdx = atoi(argv[i]);
                if (uiStrmIdx < MAX_STREAM_COUNT)
                {
                    if (poPrevStreamParam != &(g_aoStreamParam[uiStrmIdx]))
                    {
                        // Default parameters = Previous stream parameters
                        if (poPrevStreamParam != MNULL)
                        {
                            g_aoStreamParam[uiStrmIdx] = *poPrevStreamParam;
                        }

                        poPrevStreamParam           = &(g_aoStreamParam[uiStrmIdx]);
                        poPrevStreamParam->bEnable  = MTRUE;
                        poStreamParam               = poPrevStreamParam;
                    }
                    uiStreamCount++;
                    poAudioStreamParam = MNULL;
                    continue;
                }
            }
        }
        else if(strcmp(argv[i], "-as") == 0)
        {
            if (++i < iArgc)
            {
                MUINT uiStrmIdx = atoi(argv[i]);
                if (uiStrmIdx < MAX_STREAM_COUNT)
                {
                    if (poPrevAudioStreamParam != &(g_aoAudioStreamParam[uiStrmIdx]))
                    {
                        // Default parameters = Previous audio stream parameters
                        if (poPrevAudioStreamParam != MNULL)
                        {
                            RtspServerModule* poRtspSrvMod = g_aoAudioStreamParam[uiStrmIdx].poRtspSrvMod;
                            g_aoAudioStreamParam[uiStrmIdx] = *poPrevAudioStreamParam;
                            g_aoAudioStreamParam[uiStrmIdx].poRtspSrvMod = poRtspSrvMod;
                        }

                        poPrevAudioStreamParam          = &(g_aoAudioStreamParam[uiStrmIdx]);
                        poPrevAudioStreamParam->bEnable = MTRUE;
                        poAudioStreamParam              = poPrevAudioStreamParam;
                    }
                    uiAudioStreamCount++;
                    poStreamParam = MNULL;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-d") == 0)
        {
            if (++i < iArgc)
            {
                MUINT uiVinIdx    = 0;
                MUINT uiVinEndIdx = uiMaxVinCount;

                if (strcmp(argv[i], "a") != 0)
                {
                    uiVinIdx    = atoi(argv[i]);
                    uiVinEndIdx = uiVinIdx + 1;
                }

                if ((uiVinIdx < uiMaxVinCount) && (++i < iArgc))
                {
                    MUINT uiVoutIdx;

                    if (argv[i][0] == 'x')
                    {
                        uiVoutIdx = atoi(argv[i] + 1) + MAX_VOUTMTX_COUNT;
                    }
                    else
                    {
                        uiVoutIdx = atoi(argv[i]);
                    }

                    if ((uiVoutIdx < MAX_VOUT_COUNT)
                        && ((uiVoutIdx < uiMaxVoutMtxCount)
                            || (uiVoutIdx >= MAX_VOUTMTX_COUNT)))
                    {
                        for (; uiVinIdx < uiVinEndIdx; uiVinIdx++)
                        {
                            g_aoVinParam[uiVinIdx].poVoutParam  = &g_aoVoutParam[uiVoutIdx];
                            g_aoVinParam[uiVinIdx].bEnabled     = MTRUE;
                        }
                        continue;
                    }
                }
            }
        }
        else if (strcmp(argv[i], "-a") == 0)
        {
            if (++i < iArgc)
            {
                MUINT uiAinIdx = atoi(argv[i]);

                if ((++i < iArgc) && (uiAinIdx < uiMaxAinCount))
                {
                    MUINT uiAoutIdx = atoi(argv[i]);

                    if (uiAoutIdx < uiMaxAoutCount)
                    {
                        AoutParam* poAout = &g_aoAoutParam[uiAoutIdx];
                        AinParam*  poAin  = &g_aoAinParam[uiAinIdx];

                        if ((poAout->poAinParam == MNULL)
                             || (poAout->poAinParam == poAin))
                        {
                            poAout->poAinParam  = poAin;
                            poAout->bEnabled    = MTRUE;
                            poAin->bEnabled     = MTRUE;
                            continue;
                        }
                        else
                        {
                            printf("ERROR: audio output %u already connected to audio input %u\n",
                                   poAout->uiIndex, poAin->uiIndex);
                        }
                    }
                }
            }
        }
        else if (strcmp(argv[i], "-si") == 0)
        {
            if (++i < iArgc)
            {
                MUINT uiVinIdx = atoi(argv[i]);

                if (uiVinIdx < uiMaxVinCount)
                {
                    g_aoVinParam[uiVinIdx].bDoScaling = MTRUE;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-dbg") == 0)
        {
            if (++i < iArgc)
            {
                uiMsgLogLevel = atoi(argv[i]);
                continue;
            }
        }
        else if (strcmp(argv[i], "-vclog") == 0)
        {
            bEnableVcLog = MTRUE;
            continue;
        }
        else if (strcmp(argv[i], "-stat") == 0)
        {
            bShowStats = MTRUE;
            continue;
        }
        else if (strcmp(argv[i], "-cpf") == 0)
        {
            if (++i < iArgc)
            {
                MUINT uiIdx    = 0;
                MUINT uiEndIdx = uiMaxVinCount;

                if (strcmp(argv[i], "a") != 0)
                {
                    uiIdx    = atoi(argv[i]);
                    uiEndIdx = uiIdx + 1;
                }

                if ((uiIdx < uiMaxVinCount) && (++i < iArgc))
                {
                    LPixelFormat ePixFmt =
                        (strcmp(argv[i], "RGB-8")       == 0) ? LPixelFormat_R8G8B8A8               :
                        (strcmp(argv[i], "RGB-10")      == 0) ? LPixelFormat_R10G10B10A2            :
                        (strcmp(argv[i], "444-10")      == 0) ? LPixelFormat_U82Y82V82A2            :
                        (strcmp(argv[i], "YUV")         == 0) ? LPixelFormat_V8U8Y8A8               :
                        (strcmp(argv[i], "YUY2")        == 0) ? LPixelFormat_Y8U8Y8V8               :
                        (strcmp(argv[i], "420-8-2P")    == 0) ? LPixelFormat_MP_Y8_U8V8_420         :
                        (strcmp(argv[i], "422-8-2P")    == 0) ? LPixelFormat_MP_Y8_U8V8_422         :
                        (strcmp(argv[i], "444-8-2P")    == 0) ? LPixelFormat_MP_Y8_U8V8_444         :
                        (strcmp(argv[i], "420-8-3P")    == 0) ? LPixelFormat_MP_Y8_U8_V8_420        :
                        (strcmp(argv[i], "422-8-3P")    == 0) ? LPixelFormat_MP_Y8_U8_V8_422        :
                        (strcmp(argv[i], "444-8-3P")    == 0) ? LPixelFormat_MP_Y8_U8_V8_444        :
                        (strcmp(argv[i], "RGB-8-3P")    == 0) ? LPixelFormat_MP_G8_R8_B8            :
                        (strcmp(argv[i], "RGB-10-3P")   == 0) ? LPixelFormat_MP_G10_R10_B10         :
                        (strcmp(argv[i], "420-10-2P")   == 0) ? LPixelFormat_MP_Y82_U82V82_X2_420   :
                        (strcmp(argv[i], "422-10-2P")   == 0) ? LPixelFormat_MP_Y82_U82V82_X2_422   :
                        (strcmp(argv[i], "444-10-2P")   == 0) ? LPixelFormat_MP_Y82_U82V82_X2_444   :
                        (strcmp(argv[i], "420-10-3P")   == 0) ? LPixelFormat_MP_Y82_U82_V82_X2_420  :
                        (strcmp(argv[i], "422-10-3P")   == 0) ? LPixelFormat_MP_Y82_U82_V82_X2_422  :
                        (strcmp(argv[i], "444-10-3P")   == 0) ? LPixelFormat_MP_Y82_U82_V82_X2_444  :

                        // Backward compatibility.
                        (strcmp(argv[i], "MP-RGB-8")    == 0) ? LPixelFormat_MP_G8_R8_B8            : 
                        (strcmp(argv[i], "420-8")       == 0) ? LPixelFormat_MP_Y8_U8V8_420         :
                        (strcmp(argv[i], "422-8")       == 0) ? LPixelFormat_MP_Y8_U8V8_422         :
                        (strcmp(argv[i], "444-8")       == 0) ? LPixelFormat_MP_Y8_U8V8_444         :
                                                                LPixelFormat_INVALID;

                    if (ePixFmt != LPixelFormat_INVALID)
                    {
                        for (; uiIdx < uiEndIdx; uiIdx++)
                        {
                            g_aoVinParam[uiIdx].ePixFmt = ePixFmt;
                        }

                        continue;
                    }
                }
            }
        }
        else if (strcmp(argv[i], "-dpf") == 0)
        {
            if (++i < iArgc)
            {
                MUINT uiIdx    = 0;
                MUINT uiEndIdx = uiMaxVoutMtxCount;

                if (strcmp(argv[i], "a") != 0)
                {
                    uiIdx    = atoi(argv[i]);
                    uiEndIdx = uiIdx + 1;
                }

                if ((uiIdx < uiMaxVoutMtxCount) && (++i < iArgc))
                {
                    LPixelFormat ePixFmt =
                        (strcmp(argv[i], "RGB-8")       == 0) ? LPixelFormat_R8G8B8A8               :
                        (strcmp(argv[i], "RGB-10")      == 0) ? LPixelFormat_R10G10B10A2            :
                        (strcmp(argv[i], "444-10")      == 0) ? LPixelFormat_U82Y82V82A2            :
                        (strcmp(argv[i], "YUV")         == 0) ? LPixelFormat_V8U8Y8A8               :
                        (strcmp(argv[i], "YUY2")        == 0) ? LPixelFormat_Y8U8Y8V8               :
                        (strcmp(argv[i], "420-8-2P")    == 0) ? LPixelFormat_MP_Y8_U8V8_420         :
                        (strcmp(argv[i], "422-8-2P")    == 0) ? LPixelFormat_MP_Y8_U8V8_422         :
                        (strcmp(argv[i], "444-8-2P")    == 0) ? LPixelFormat_MP_Y8_U8V8_444         :
                        (strcmp(argv[i], "420-8-3P")    == 0) ? LPixelFormat_MP_Y8_U8_V8_420        :
                        (strcmp(argv[i], "422-8-3P")    == 0) ? LPixelFormat_MP_Y8_U8_V8_422        :
                        (strcmp(argv[i], "444-8-3P")    == 0) ? LPixelFormat_MP_Y8_U8_V8_444        :
                        (strcmp(argv[i], "RGB-8-3P")    == 0) ? LPixelFormat_MP_G8_R8_B8            :
                        (strcmp(argv[i], "RGB-10-3P")   == 0) ? LPixelFormat_MP_G10_R10_B10         :
                        (strcmp(argv[i], "444-8-3P")    == 0) ? LPixelFormat_MP_Y8_U8_V8_444        :
                        (strcmp(argv[i], "420-10-2P")   == 0) ? LPixelFormat_MP_Y82_U82V82_X2_420   :
                        (strcmp(argv[i], "422-10-2P")   == 0) ? LPixelFormat_MP_Y82_U82V82_X2_422   :
                        (strcmp(argv[i], "444-10-2P")   == 0) ? LPixelFormat_MP_Y82_U82V82_X2_444   :
                        (strcmp(argv[i], "420-10-3P")   == 0) ? LPixelFormat_MP_Y82_U82_V82_X2_420  :
                        (strcmp(argv[i], "422-10-3P")   == 0) ? LPixelFormat_MP_Y82_U82_V82_X2_422  :
                        (strcmp(argv[i], "444-10-3P")   == 0) ? LPixelFormat_MP_Y82_U82_V82_X2_444  :

                        // Backward compatibility.
                        (strcmp(argv[i], "MP-RGB-8")    == 0) ? LPixelFormat_MP_G8_R8_B8            :
                        (strcmp(argv[i], "420-8")       == 0) ? LPixelFormat_MP_Y8_U8V8_420         :
                        (strcmp(argv[i], "422-8")       == 0) ? LPixelFormat_MP_Y8_U8V8_422         :
                        (strcmp(argv[i], "444-8")       == 0) ? LPixelFormat_MP_Y8_U8V8_444         :
                                                                LPixelFormat_INVALID;

                    if (ePixFmt != LPixelFormat_INVALID)
                    {
                        for (; uiIdx < uiEndIdx; uiIdx++)
                        {
                            g_aoVoutParam[uiIdx].ePixFmt = ePixFmt;
                        }

                        continue;
                    }
                }
            }
        }
        else if (strcmp(argv[i], "-opf") == 0)
        {
            if (++i < iArgc)
            {
                MUINT uiIdx    = 0;
                MUINT uiEndIdx = uiMaxVoutMtxCount;

                if (strcmp(argv[i], "a") != 0)
                {
                    uiIdx    = atoi(argv[i]);
                    uiEndIdx = uiIdx + 1;
                }

                if ((uiIdx < uiMaxVoutMtxCount) && (++i < iArgc))
                {
                    LPixelFormat ePixFmt =
                        (strcmp(argv[i], "RGB-8")   == 0) ? LPixelFormat_R8G8B8A8               :
                        (strcmp(argv[i], "RGB-10")  == 0) ? LPixelFormat_R10G10B10A2            :
                        (strcmp(argv[i], "420-8")   == 0) ? LPixelFormat_MP_Y8_U8V8_420         :
                        (strcmp(argv[i], "422-8")   == 0) ? LPixelFormat_MP_Y8_U8V8_422         :
                        (strcmp(argv[i], "444-8")   == 0) ? LPixelFormat_MP_Y8_U8V8_444         :
                        (strcmp(argv[i], "420-10")  == 0) ? LPixelFormat_MP_Y82_U82V82_X2_420   :
                        (strcmp(argv[i], "422-10")  == 0) ? LPixelFormat_MP_Y82_U82V82_X2_422   :
                        (strcmp(argv[i], "444-10")  == 0) ? LPixelFormat_MP_Y82_U82_V82_X2_444  :
                                                            LPixelFormat_INVALID;

                    if (ePixFmt != LPixelFormat_INVALID)
                    {
                        if (++i < iArgc)
                        {
                            MBOOL bValidFlag = MFALSE;
                            MBOOL bFlag      = MFALSE;

                            if ((ePixFmt == LPixelFormat_R8G8B8A8)
                                || (ePixFmt == LPixelFormat_R10G10B10A2))
                            {
                                if (strcmp(argv[i], "fr") == 0)
                                {
                                    bValidFlag = MTRUE;
                                    bFlag      = MTRUE;
                                }
                                else if (strcmp(argv[i], "lr") == 0)
                                {
                                    bValidFlag = MTRUE;
                                    bFlag      = MFALSE;
                                }
                            }
                            else
                            {
                                if (strcmp(argv[i], "bt709") == 0)
                                {
                                    bValidFlag = MTRUE;
                                    bFlag      = MTRUE;
                                }
                                else if (strcmp(argv[i], "bt601") == 0)
                                {
                                    bValidFlag = MTRUE;
                                    bFlag      = MFALSE;
                                }
                            }

                            if (bValidFlag)
                            {
                                for (; uiIdx < uiEndIdx; uiIdx++)
                                {
                                    g_aoVoutParam[uiIdx].eOutPixFmt         = ePixFmt;
                                    g_aoVoutParam[uiIdx].bOutRgbFullYuv709  = bFlag;
                                }

                                continue;
                            }
                        }
                    }
                }
            }
        }
        else if (strcmp(argv[i], "-fsc") == 0)
        {
            if (++i < iArgc)
            {
                MUINT uiIdx;

                if (argv[i][0] == 'x')
                {
                    uiIdx = atoi(argv[i] + 1) + MAX_VOUTMTX_COUNT;
                }
                else
                {
                    uiIdx = atoi(argv[i]);
                }

                if ((uiIdx < MAX_VOUT_COUNT)
                    && ((uiIdx < uiMaxVoutMtxCount)
                        || (uiIdx >= MAX_VOUTMTX_COUNT)))
                {
                    if (++i < iArgc)
                    {
                        g_aoVoutParam[uiIdx].uiCycleTimeSec = atoi(argv[i]);
                        continue;
                    }
                }
            }
        }
        else if (strcmp(argv[i], "-fvp") == 0)
        {
            if (++i < iArgc)
            {
                MUINT uiIdx    = 0;
                MUINT uiEndIdx = uiMaxVoutMtxCount;

                if (strcmp(argv[i], "a") != 0)
                {
                    uiIdx    = atoi(argv[i]);
                    uiEndIdx = uiIdx + 1;
                }

                if ((uiIdx < uiMaxVoutMtxCount) && (++i < iArgc))
                {
                    LSIZE oSize;
                    MUINT uiFps;

                    if (sscanf(argv[i],
                               "%ux%u@%u",
                               &oSize.iWidth,
                               &oSize.iHeight,
                               &uiFps) == 3)
                    {
                        for (; uiIdx < uiEndIdx; uiIdx++)
                        {
                            g_aoVoutParam[uiIdx].oForcedSize = oSize;
                            g_aoVoutParam[uiIdx].uiForcedFps = uiFps;
                            g_aoVoutParam[uiIdx].bForceMode  = MTRUE;
                        }
                        continue;
                    }
                }
            }
        }
        else if (strcmp(argv[i], "-vor") == 0)
        {
            if (++i < iArgc)
            {
                if (sscanf(argv[i],
                           "%d,%d,%d,%d",
                           &oWinRect.iLeft,
                           &oWinRect.iTop,
                           &oWinRect.iRight,
                           &oWinRect.iBottom) == 4)
                {
                    if ((oWinRect.iLeft < oWinRect.iRight)
                        && (oWinRect.iTop < oWinRect.iBottom))
                    {
                        bForceWinRect = MTRUE;
                        continue;
                    }
                }
            }
        }
        else if (strcmp(argv[i], "-acp") == 0)
        {
            if(++i < iArgc)
            {
                char* szToken = strtok(argv[i], "/");

                MUINT32 uiAinIdx = strtoul(szToken, MNULL, 10);
                MUINT32 uiSamplingRate = 0;
                MUINT32 uiNbChannels = 0;
                MUINT32 uiSampleSize = 0;

                if(uiAinIdx < MAX_AIN_COUNT)
                {
                    szToken = strtok(MNULL, "/");

                    if(szToken != MNULL)
                    {
                        uiSamplingRate = strtoul(szToken, MNULL, 10);
                    }

                    szToken = strtok(MNULL, "/");

                    if(szToken != MNULL)
                    {
                        uiNbChannels = strtoul(szToken, MNULL, 10);
                    }

                    szToken = strtok(MNULL, "/");

                    if(szToken != MNULL)
                    {
                        uiSampleSize = strtoul(szToken, MNULL, 10);
                    }

                    if((uiSamplingRate > 0)
                        && (uiNbChannels > 0)
                        && (uiSampleSize > 0))
                    {
                        g_aoAinParam[uiAinIdx].uiSamplingRate = uiSamplingRate;
                        g_aoAinParam[uiAinIdx].uiNbChannels = uiNbChannels;
                        g_aoAinParam[uiAinIdx].uiSampleSize = uiSampleSize;
                        continue;
                    }
                }
            }
        }
        else if (strcmp(argv[i], "-avsync") == 0)
        {
            bAvSync = MTRUE;
            continue;
        }
        else if (strcmp(argv[i], "-ll") == 0)
        {
            bLowLatency = MTRUE;
            continue;
        }
        else if (poStreamParam == MNULL)
        {
            // Need to enable at least one stream before to set the remaining parameters.
        }
        else if (strcmp(argv[i], "-i") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->uiVinCount = 0;

                while (*argv[i] != '-')
                {
                    MUINT uiVinIdx = atoi(argv[i]);

                    if ((uiVinIdx < uiMaxVinCount)
                        && (poStreamParam->uiVinCount < MAX_VIN_PER_STEAM)
                        && (poStreamParam->uiVinCount < MOSAIC_MAX_INPUT))
                    {
                        poStreamParam->apoVinParam[poStreamParam->uiVinCount] = &g_aoVinParam[uiVinIdx];
                        poStreamParam->uiVinCount++;
                    }

                    if (++i >= iArgc)
                    {
                        break;
                    }
                }

                if (poStreamParam->uiVinCount > 0)
                {
                    i--;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-o") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->szDstFilePath = argv[i];
                if (poStreamParam->szDstFilePath != MNULL) continue;
            }
        }
        else if (strcmp(argv[i], "-od") == 0)
        {
            if (++i < iArgc)
            {
                int iVoutIdx;

                if (argv[i][0] == 'x')
                {
                    iVoutIdx = atoi(argv[i] + 1) + MAX_VOUTMTX_COUNT;
                }
                else
                {
                    iVoutIdx = atoi(argv[i]);
                }

                if (iVoutIdx == -1)
                {
                    poStreamParam->poVoutParam = MNULL;
                    continue;
                }
                else if ((iVoutIdx < MAX_VOUT_COUNT)
                         && ((iVoutIdx < uiMaxVoutMtxCount)
                             || (iVoutIdx >= MAX_VOUTMTX_COUNT)))
                {
                    poStreamParam->poVoutParam = &g_aoVoutParam[iVoutIdx];
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-rtsp") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "on") == 0)
                {
                    poStreamParam->bNetServer = MTRUE;
                    continue;
                }
                else if (strcmp(argv[i], "off") == 0)
                {
                    poStreamParam->bNetServer = MFALSE;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-proto") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "rtsp") == 0)
                {
                    poStreamParam->eNetProtocol = LNetStreamer_Protocol_RTSP;
                    continue;
                }
                else if (strcmp(argv[i], "rtp") == 0)
                {
                    poStreamParam->eNetProtocol = LNetStreamer_Protocol_RTP;
                    continue;
                }
                else if (strcmp(argv[i], "ts") == 0)
                {
                    poStreamParam->eNetProtocol = LNetStreamer_Protocol_TS;
                    continue;
                }
                else if (strcmp(argv[i], "rtmp") == 0)
                {
                    poStreamParam->eNetProtocol = LNetStreamer_Protocol_RTMP;
                    continue;
                }
                else if (strcmp(argv[i], "hls") == 0)
                {
                    poStreamParam->eNetProtocol = LNetStreamer_Protocol_HLS;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-addr") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "0") == 0)
                {
                    poStreamParam->szNetAddress = MNULL;
                }
                else
                {
                    poStreamParam->szNetAddress = argv[i];
                }
                continue;
            }
        }
        else if (strcmp(argv[i], "-port") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->uiNetPort = atoi(argv[i]);
                continue;
            }
        }
        else if (strcmp(argv[i], "-mtu") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->uiMtu = atoi(argv[i]);
                continue;
            }
        }
        else if (strcmp(argv[i], "-url") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "0") == 0)
                {
                    poStreamParam->szRtmpPushLocation = MNULL;
                }
                else
                {
                    poStreamParam->szRtmpPushLocation = argv[i];
                }
                continue;
            }
        }
        else if (strcmp(argv[i], "-name") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "0") == 0)
                {
                    poStreamParam->szFolderName = MNULL;
                }
                else
                {
                    poStreamParam->szFolderName = argv[i];
                }
                continue;
            }
        }
        else if (strcmp(argv[i], "-username") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "0") == 0)
                {
                    poStreamParam->szUsername = MNULL;
                }
                else
                {
                    poStreamParam->szUsername = argv[i];
                }
                continue;
            }
        }
        else if (strcmp(argv[i], "-password") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "0") == 0)
                {
                    poStreamParam->szPassword = MNULL;
                }
                else
                {
                    poStreamParam->szPassword = argv[i];
                }
                continue;
            }
        }
        else if (strcmp(argv[i], "-srt") == 0)
        {
            poStreamParam->bEnableSrt = MTRUE;
            continue;
        }
        else if (strcmp(argv[i], "-srtmode") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "listener") == 0)
                {
                    poStreamParam->eSrtMode = LNetStreamer_SrtMode_LISTENER;
                    continue;
                }
                else if (strcmp(argv[i], "caller") == 0)
                {
                    poStreamParam->eSrtMode = LNetStreamer_SrtMode_CALLER;
                    continue;
                }
                else if (strcmp(argv[i], "rendezvous") == 0)
                {
                    poStreamParam->eSrtMode = LNetStreamer_SrtMode_RENDEZVOUS;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-t") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->uiHlsDuration = (MUINT)(atof(argv[i]) * 1000.0f);
                continue;
            }
        }
        else if (strcmp(argv[i], "-hlscount") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->uiHlsCount = (MUINT)(atoi(argv[i]));
                continue;
            }
        }
        else if (strcmp(argv[i], "-hlsmaster") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->szHlsMaster = argv[i];
                continue;
            }
        }
        else if (strcmp(argv[i], "-neti") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->szNetInterface = argv[i];
                continue;
            }
        }
        else if (strcmp(argv[i], "-ipv6") == 0)
        {
            poStreamParam->bEnableIpv6 = MTRUE;
            continue;
        }
        else if (strcmp(argv[i], "-f") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->szHlsFolder = argv[i];
                continue;
            }
        }
        else if (strcmp(argv[i], "-w") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->oFrameSize.iWidth = atoi(argv[i]);

                if (poStreamParam->oFrameSize.iWidth == 0)
                {
                    poStreamParam->bUseVinWidth = MTRUE;
                    continue;
                }
                else if ((poStreamParam->oFrameSize.iWidth >= MIN_FRAME_SIZE)
                         && (poStreamParam->oFrameSize.iWidth <= MAX_FRAME_SIZE))
                {
                    poStreamParam->bUseVinWidth = MFALSE;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-h") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->oFrameSize.iHeight = atoi(argv[i]);

                if (poStreamParam->oFrameSize.iHeight == 0)
                {
                    poStreamParam->bUseVinHeight = MTRUE;
                    continue;
                }
                else if ((poStreamParam->oFrameSize.iHeight >= MIN_FRAME_SIZE)
                         && (poStreamParam->oFrameSize.iHeight <= MAX_FRAME_SIZE))
                {
                    poStreamParam->bUseVinHeight = MFALSE;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-sr") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "full") == 0)
                {
                    poStreamParam->bFullFrame = MTRUE;
                    continue;
                }
                else
                {
                    LRECT oSubRect;

                    if (sscanf(argv[i],
                               "%d,%d,%d,%d",
                               &oSubRect.iLeft,
                               &oSubRect.iTop,
                               &oSubRect.iRight,
                               &oSubRect.iBottom) == 4)
                    {
                        poStreamParam->oSubFramePos.iX       = oSubRect.iLeft;
                        poStreamParam->oSubFramePos.iY       = oSubRect.iTop;
                        poStreamParam->oSubFrameSize.iWidth  = oSubRect.iRight - oSubRect.iLeft;
                        poStreamParam->oSubFrameSize.iHeight = oSubRect.iBottom - oSubRect.iTop;

                        if ((poStreamParam->oSubFrameSize.iWidth >= MIN_FRAME_SIZE)
                            && (poStreamParam->oSubFrameSize.iWidth <= MAX_FRAME_SIZE)
                            && (poStreamParam->oSubFrameSize.iHeight >= MIN_FRAME_SIZE)
                            && (poStreamParam->oSubFrameSize.iWidth <= MAX_FRAME_SIZE))
                        {
                            poStreamParam->bFullFrame = MFALSE;
                            continue;
                        }
                    }
                }
            }
        }
        else if (strcmp(argv[i], "-kar") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "on") == 0)
                {
                    poStreamParam->bKeepRatio = MTRUE;
                    continue;
                }
                else if (strcmp(argv[i], "off") == 0)
                {
                    poStreamParam->bKeepRatio = MFALSE;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-nmc") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->oFrameDiv.iWidth = atoi(argv[i]);

                if ((poStreamParam->oFrameDiv.iWidth >= 0)
                    && (poStreamParam->oFrameDiv.iWidth <= uiMaxVinCount))
                {
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-rot") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->eRotation = (strcmp(argv[i], "0")    == 0) ? LVPC_Rotation_NONE      :
                                           (strcmp(argv[i], "90")   == 0) ? LVPC_Rotation_90        :
                                           (strcmp(argv[i], "180")  == 0) ? LVPC_Rotation_180       :
                                           (strcmp(argv[i], "270")  == 0) ? LVPC_Rotation_270       :
                                           (strcmp(argv[i], "0F")   == 0) ? LVPC_Rotation_FLIP_NONE :
                                           (strcmp(argv[i], "90F")  == 0) ? LVPC_Rotation_FLIP_90   :
                                           (strcmp(argv[i], "180F") == 0) ? LVPC_Rotation_FLIP_180  :
                                           (strcmp(argv[i], "270F") == 0) ? LVPC_Rotation_FLIP_270  : ~0;

                if (poStreamParam->eRotation != ~0)
                {
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-bz") == 0)
        {
            if (++i < iArgc)
            {
                if (sscanf(argv[i],
                           "%d,%d",
                           &poStreamParam->oBezelSize.iWidth,
                           &poStreamParam->oBezelSize.iHeight) == 2)
                {
                    if ((poStreamParam->oBezelSize.iWidth >= 0)
                        && (poStreamParam->oBezelSize.iWidth <= MAX_BEZEL_SIZE)
                        && (poStreamParam->oBezelSize.iHeight >= 0)
                        && (poStreamParam->oBezelSize.iWidth <= MAX_BEZEL_SIZE))
                    {
                        continue;
                    }
                }
            }
        }
        else if (strcmp(argv[i], "-div") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->uiFrameRateDivisor = atoi(argv[i]);
                if (poStreamParam->uiFrameRateDivisor > 0) continue;
            }
        }
        else if (strcmp(argv[i], "-tbr") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->uiTargetBitRateKbs = atoi(argv[i]);
                if ((poStreamParam->uiTargetBitRateKbs >= 50)
                    && (poStreamParam->uiTargetBitRateKbs <= ENC_MAX_BITRATE_KBPS)) continue;
            }
        }
        else if (strcmp(argv[i], "-ip") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->uiIPeriod = atoi(argv[i]);
                if (poStreamParam->uiIPeriod > 0) continue;
            }
        }
        else if (strcmp(argv[i], "-pp") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->uiPPeriod = atoi(argv[i]);
                if ((poStreamParam->uiPPeriod > 0)
                    && (poStreamParam->uiPPeriod <= 8)) continue;
            }
        }
        else if (strcmp(argv[i], "-qpmin") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->uiQpMin = atoi(argv[i]);
                if ((poStreamParam->uiQpMin >= 0)
                    && (poStreamParam->uiQpMin <= 51)) continue;
            }
        }
        else if (strcmp(argv[i], "-qpmax") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->uiQpMax = atoi(argv[i]);
                if ((poStreamParam->uiQpMax >= 0)
                    && (poStreamParam->uiQpMax <= 51)) continue;
            }
        }
        else if (strcmp(argv[i], "-ns") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->uiSliceCount = atoi(argv[i]);
                if ((poStreamParam->uiSliceCount > 0)
                    && (poStreamParam->uiSliceCount <= ENC_MAX_NB_SLICES)) continue;
            }
        }
        else if (strcmp(argv[i], "-rc") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "win") == 0)
                {
                    poStreamParam->eEncodeMode = LH264E_EncoderMode_RATECONTROLWINDOWED;
                    continue;
                }
                else if (strcmp(argv[i], "man") == 0)
                {
                    poStreamParam->eEncodeMode = LH264E_EncoderMode_MANUAL;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-prof") == 0)
        {
            if (++i < iArgc)
            {
                MBOOL bValid = MTRUE;

                if (strcmp(argv[i], "b")  == 0)
                {
                    poStreamParam->eEncodeProfile = LH264_Profile_BASELINE;
                }
                else if (strcmp(argv[i], "m")  == 0)
                {
                    poStreamParam->eEncodeProfile = LH264_Profile_MAIN;
                }
                else if (strcmp(argv[i], "hi")  == 0)
                {
                    poStreamParam->eEncodeProfile = LH264_Profile_HIGH;
                }
                else if (strcmp(argv[i], "hi10")  == 0)
                {
                    poStreamParam->eEncodeProfile = LH264_Profile_HIGH10;
                }
                else if (strcmp(argv[i], "hi422")  == 0)
                {
                    poStreamParam->eEncodeProfile = LH264_Profile_HIGH422;
                }
                else if (strcmp(argv[i], "hi444")  == 0)
                {
                    poStreamParam->eEncodeProfile = LH264_Profile_HIGH444;
                }
                else if (strcmp(argv[i], "cavlc")  == 0)
                {
                    poStreamParam->eEncodeProfile = LH264_Profile_CAVLC444INTRA;
                }
                else
                {
                    bValid = MFALSE;
                }

                if (bValid)
                {
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-lvl") == 0)
        {
            if (++i < iArgc)
            {
                MBOOL bValid = MTRUE;

                if      (strcmp(argv[i], "1B")  == 0)   poStreamParam->eEncodeLevel = LH264_Level_1B;
                else if (strcmp(argv[i], "1")   == 0)   poStreamParam->eEncodeLevel = LH264_Level_1;
                else if (strcmp(argv[i], "1.1") == 0)   poStreamParam->eEncodeLevel = LH264_Level_1_1;
                else if (strcmp(argv[i], "1.2") == 0)   poStreamParam->eEncodeLevel = LH264_Level_1_2;
                else if (strcmp(argv[i], "1.3") == 0)   poStreamParam->eEncodeLevel = LH264_Level_1_3;
                else if (strcmp(argv[i], "2")   == 0)   poStreamParam->eEncodeLevel = LH264_Level_2;
                else if (strcmp(argv[i], "2.1") == 0)   poStreamParam->eEncodeLevel = LH264_Level_2_1;
                else if (strcmp(argv[i], "2.2") == 0)   poStreamParam->eEncodeLevel = LH264_Level_2_2;
                else if (strcmp(argv[i], "3")   == 0)   poStreamParam->eEncodeLevel = LH264_Level_3;
                else if (strcmp(argv[i], "3.1") == 0)   poStreamParam->eEncodeLevel = LH264_Level_3_1;
                else if (strcmp(argv[i], "3.2") == 0)   poStreamParam->eEncodeLevel = LH264_Level_3_2;
                else if (strcmp(argv[i], "4")   == 0)   poStreamParam->eEncodeLevel = LH264_Level_4;
                else if (strcmp(argv[i], "4.1") == 0)   poStreamParam->eEncodeLevel = LH264_Level_4_1;
                else if (strcmp(argv[i], "4.2") == 0)   poStreamParam->eEncodeLevel = LH264_Level_4_2;
                else if (strcmp(argv[i], "5")   == 0)   poStreamParam->eEncodeLevel = LH264_Level_5;
                else if (strcmp(argv[i], "5.1") == 0)   poStreamParam->eEncodeLevel = LH264_Level_5_1;
                else if (strcmp(argv[i], "5.2") == 0)   poStreamParam->eEncodeLevel = LH264_Level_5_2;
                else
                {
                    bValid = MFALSE;
                }

                if (bValid)
                {
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-faq") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->uiFaq = atoi(argv[i]);
                if ((poStreamParam->uiFaq >= 0)
                    && (poStreamParam->uiFaq <= 100)) continue;
            }
        }
        else if (strcmp(argv[i], "-df") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "on") == 0)
                {
                    poStreamParam->bDfEnabled = MTRUE;
                    continue;
                }
                else if (strcmp(argv[i], "off") == 0)
                {
                    poStreamParam->bDfEnabled = MFALSE;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-epf") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "400-8") == 0)
                {
                    poStreamParam->eEncodePixFmt = LPixelFormat_Y8;
                    continue;
                }
                else if (strcmp(argv[i], "400-10") == 0)
                {
                    poStreamParam->eEncodePixFmt = LPixelFormat_Y82Y82Y82X2;
                    continue;
                }
                else if (strcmp(argv[i], "420-8") == 0)
                {
                    poStreamParam->eEncodePixFmt = LPixelFormat_MP_Y8_U8V8_420;
                    continue;
                }
                else if (strcmp(argv[i], "420-10") == 0)
                {
                    poStreamParam->eEncodePixFmt = LPixelFormat_MP_Y82_U82V82_X2_420;
                    continue;
                }
                else if (strcmp(argv[i], "422-8") == 0)
                {
                    poStreamParam->eEncodePixFmt = LPixelFormat_MP_Y8_U8V8_422;
                    continue;
                }
                else if (strcmp(argv[i], "422-10") == 0)
                {
                    poStreamParam->eEncodePixFmt = LPixelFormat_MP_Y82_U82V82_X2_422;
                    continue;
                }
                else if (strcmp(argv[i], "444-8") == 0)
                {
                    poStreamParam->eEncodePixFmt = LPixelFormat_MP_Y8_U8_V8_444;
                    continue;
                }
                else if (strcmp(argv[i], "444-10") == 0)
                {
                    poStreamParam->eEncodePixFmt = LPixelFormat_MP_Y82_U82_V82_X2_444;
                    continue;
                }
            }
        }

        if (i >= iArgc)
        {

        }
        else if (poAudioStreamParam == MNULL)
        {
            // Remaining are per audio stream arguments.
        }
        else if (strcmp(argv[i], "-i") == 0)
        {
            if (++i < iArgc)
            {
                MUINT uiAinIdx = atoi(argv[i]);

                if (uiAinIdx < uiMaxAinCount)
                {
                    poAudioStreamParam->poAinParam = &(g_aoAinParam[uiAinIdx]);
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-fmt") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], GetAacStreamFormatName(LAudioCodec_AACStreamFormat_RAW)) == 0)
                {
                    poAudioStreamParam->eAacStreamFormat = LAudioCodec_AACStreamFormat_RAW;
                    continue;
                }
                else if (strcmp(argv[i], GetAacStreamFormatName(LAudioCodec_AACStreamFormat_ADTS)) == 0)
                {
                    poAudioStreamParam->eAacStreamFormat = LAudioCodec_AACStreamFormat_ADTS;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-prof") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], GetAacProfileName(LAudioCodec_AACProfile_LC)) == 0)
                {
                    poAudioStreamParam->eAacProfile = LAudioCodec_AACProfile_LC;
                    continue;
                }
                else if (strcmp(argv[i], GetAacProfileName(LAudioCodec_AACProfile_HEV1)) == 0)
                {
                    poAudioStreamParam->eAacProfile = LAudioCodec_AACProfile_HEV1;
                    continue;
                }
                else if (strcmp(argv[i], GetAacProfileName(LAudioCodec_AACProfile_HEV2)) == 0)
                {
                    poAudioStreamParam->eAacProfile = LAudioCodec_AACProfile_HEV2;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-br") == 0)
        {
            if (++i < iArgc)
            {
                poAudioStreamParam->uiAacBitrate = atoi(argv[i]);
                continue;
            }
        }
        else if (strcmp(argv[i], "-q") == 0)
        {
            if (++i < iArgc)
            {
                MUINT uiLevel = atoi(argv[i]);

                if (uiLevel == GetAacQualityLevel(LAudioCodec_AACQualityLevel_0))
                {
                    poAudioStreamParam->eAacQualityLvl = LAudioCodec_AACQualityLevel_0;
                    continue;
                }
                else if (uiLevel == GetAacQualityLevel(LAudioCodec_AACQualityLevel_1))
                {
                    poAudioStreamParam->eAacQualityLvl = LAudioCodec_AACQualityLevel_1;
                    continue;
                }
                else if (uiLevel == GetAacQualityLevel(LAudioCodec_AACQualityLevel_2))
                {
                    poAudioStreamParam->eAacQualityLvl = LAudioCodec_AACQualityLevel_2;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-tns") == 0)
        {
            if (++i < iArgc)
            {
                MINT iTns = atoi(argv[i]);

                if (iTns == 0)
                {
                    poAudioStreamParam->bUseTns = MFALSE;
                    continue;
                }
                else if (iTns == 1)
                {
                    poAudioStreamParam->bUseTns = MTRUE;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-proto") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "rtsp") == 0)
                {
                    poAudioStreamParam->eNetProtocol = LNetStreamer_Protocol_RTSP;
                    continue;
                }
                else if (strcmp(argv[i], "rtp") == 0)
                {
                    poAudioStreamParam->eNetProtocol = LNetStreamer_Protocol_RTP;
                    continue;
                }
                else if (strcmp(argv[i], "ts") == 0)
                {
                    poAudioStreamParam->eNetProtocol = LNetStreamer_Protocol_TS;
                    continue;
                }
                else if (strcmp(argv[i], "rtmp") == 0)
                {
                    poAudioStreamParam->eNetProtocol = LNetStreamer_Protocol_RTMP;
                    continue;
                }
                else if (strcmp(argv[i], "hls") == 0)
                {
                    poAudioStreamParam->eNetProtocol = LNetStreamer_Protocol_HLS;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-addr") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "0") == 0)
                {
                    poAudioStreamParam->szNetAddress = MNULL;
                }
                else
                {
                    poAudioStreamParam->szNetAddress = argv[i];
                }
                continue;
            }
        }
        else if (strcmp(argv[i], "-port") == 0)
        {
            if (++i < iArgc)
            {
                poAudioStreamParam->uiNetPort = atoi(argv[i]);
                continue;
            }
        }
        else if (strcmp(argv[i], "-mtu") == 0)
        {
            if (++i < iArgc)
            {
                poAudioStreamParam->uiMtu = atoi(argv[i]);
                continue;
            }
        }
        else if (strcmp(argv[i], "-url") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "0") == 0)
                {
                    poAudioStreamParam->szRtmpPushLocation = MNULL;
                }
                else
                {
                    poAudioStreamParam->szRtmpPushLocation = argv[i];
                }
                continue;
            }
        }
        else if (strcmp(argv[i], "-name") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "0") == 0)
                {
                    poAudioStreamParam->szFolderName = MNULL;
                }
                else
                {
                    poAudioStreamParam->szFolderName = argv[i];
                }
                continue;
            }
        }
        else if (strcmp(argv[i], "-username") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "0") == 0)
                {
                    poAudioStreamParam->szUsername = MNULL;
                }
                else
                {
                    poAudioStreamParam->szUsername = argv[i];
                }
                continue;
            }
        }
        else if (strcmp(argv[i], "-password") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "0") == 0)
                {
                    poAudioStreamParam->szPassword = MNULL;
                }
                else
                {
                    poAudioStreamParam->szPassword = argv[i];
                }
                continue;
            }
        }
        else if (strcmp(argv[i], "-srt") == 0)
        {
            poAudioStreamParam->bEnableSrt = MTRUE;
            continue;
        }
        else if (strcmp(argv[i], "-srtmode") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "listener") == 0)
                {
                    poAudioStreamParam->eSrtMode = LNetStreamer_SrtMode_LISTENER;
                    continue;
                }
                else if (strcmp(argv[i], "caller") == 0)
                {
                    poAudioStreamParam->eSrtMode = LNetStreamer_SrtMode_CALLER;
                    continue;
                }
                else if (strcmp(argv[i], "rendezvous") == 0)
                {
                    poAudioStreamParam->eSrtMode = LNetStreamer_SrtMode_RENDEZVOUS;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-t") == 0)
        {
            if (++i < iArgc)
            {
                poAudioStreamParam->uiHlsDuration = (MUINT)(atof(argv[i]) * 1000.0f);
                continue;
            }
        }
        else if (strcmp(argv[i], "-hlscount") == 0)
        {
            if (++i < iArgc)
            {
                poAudioStreamParam->uiHlsCount = (MUINT)(atoi(argv[i]));
                continue;
            }
        }
        else if (strcmp(argv[i], "-hlsmaster") == 0)
        {
            if (++i < iArgc)
            {
                poAudioStreamParam->szHlsMaster = argv[i];
                continue;
            }
        }
        else if (strcmp(argv[i], "-neti") == 0)
        {
            if (++i < iArgc)
            {
                poAudioStreamParam->szNetInterface = argv[i];
                continue;
            }
        }
        else if (strcmp(argv[i], "-ipv6") == 0)
        {
            poAudioStreamParam->bEnableIpv6 = MTRUE;
            continue;
        }
        else if (strcmp(argv[i], "-f") == 0)
        {
            if (++i < iArgc)
            {
                poAudioStreamParam->szHlsFolder = argv[i];
                continue;
            }
        }

        bArgError = MTRUE;
        break;
    }

    for (i = 0; (i < MAX_VIN_COUNT) && !bArgError; i++)
    {
        if (g_aoVinParam[i].bEnabled)
        {
            if (g_aoVinParam[i].poVoutParam->uiInputCount < MOSAIC_MAX_INPUT)
            {
                g_aoVinParam[i].uiInMosIdx = g_aoVinParam[i].poVoutParam->uiInputCount;
                g_aoVinParam[i].poVoutParam->uiInputCount++;
            }
            else
            {
                bArgError = MTRUE;
            }
        }
    }

    MUINT uiEnabledStrmCnt = 0;
    MUINT uiEnabledVideoStrmCnt = 0;

    for (i = 0; (i < MAX_STREAM_COUNT) && !bArgError; i++)
    {
        if (g_aoStreamParam[i].bEnable)
        {
            if (g_aoStreamParam[i].uiVinCount == 0)
            {
                g_aoStreamParam[i].apoVinParam[0] = &g_aoVinParam[0];
                g_aoStreamParam[i].uiVinCount     = 1;
            }

            MUINT j;
            for (j = 0; j < g_aoStreamParam[i].uiVinCount; j++)
            {
                g_aoStreamParam[i].apoVinParam[j]->bEnabled = MTRUE;
            }

            if (g_aoStreamParam[i].poVoutParam != MNULL)
            {
                if (g_aoStreamParam[i].poVoutParam->uiInputCount < MOSAIC_MAX_INPUT)
                {
                    g_aoStreamParam[i].uiOutMosIdx = g_aoStreamParam[i].poVoutParam->uiInputCount;
                    g_aoStreamParam[i].poVoutParam->uiInputCount++;
                }
                else
                {
                    bArgError = MTRUE;
                }
            }
            else if (!g_aoStreamParam[i].bNetServer
                     && (g_aoStreamParam[i].szDstFilePath == MNULL))
            {
                g_aoStreamParam[i].szDstFilePath = "/dev/null";
            }

            uiEnabledVideoStrmCnt++;
        }

        if (g_aoAudioStreamParam[i].bEnable)
        {
            if (g_aoAudioStreamParam[i].poAinParam == MNULL)
            {
                g_aoAudioStreamParam[i].poAinParam = &g_aoAinParam[0];
            }

            g_aoAudioStreamParam[i].poAinParam->bEnabled = MTRUE;
        }

        if (g_aoStreamParam[i].bEnable || g_aoAudioStreamParam[i].bEnable)
        {
            uiEnabledStrmCnt++;
        }
    }

    if(uiEnabledStrmCnt > MAX_STREAM_COUNT)
    {
        bArgError = MTRUE;
    }

    MBOOL bVinEnabled = MFALSE;

    for (i = 0; i < MAX_VIN_COUNT; i++)
    {
        if (g_aoVinParam[i].bEnabled)
        {
            bVinEnabled = MTRUE;
        }
    }

    MBOOL bVoutConnected = MFALSE;

    for (i = 0; i < MAX_VOUT_COUNT; i++)
    {
        if (g_aoVoutParam[i].uiInputCount > 0)
        {
            bVoutConnected = MTRUE;
        }
    }

    MBOOL bVinOnly = (uiEnabledVideoStrmCnt == 0) && bVinEnabled;

    if (bArgError
        || (bVinOnly && !bVoutConnected)
        || (iArgc == 1))
    {
        char szVoutRange[64] = "";

        if (uiMaxVoutMtxCount > 0)
        {
            snprintf(szVoutRange, sizeof(szVoutRange), "0..%u%s",
                     uiMaxVoutMtxCount-1, (MAX_VOUTGLX_COUNT > 0) ? ", " : "");
        }

        if (MAX_VOUTGLX_COUNT > 0)
        {
            char szXRange[32];

            snprintf(szXRange, sizeof(szXRange), "x0..x%u", MAX_VOUTGLX_COUNT-1);
            strncat_wz(szVoutRange, szXRange, sizeof(szVoutRange));
        }

        printf("\nMandatory arguments (need at least one of these):\n");
        printf("  -s <N>         : Enable stream 'N': [0..%u]. All remaining non-global arguments between\n"
               "                   this argument and the next \"-s\" or \"-as\" argument will be applied to this\n"
               "                   stream. As default, this stream takes the arguments of the previous stream in\n"
               "                   the command line.\n",
                                   MAX_STREAM_COUNT-1);
        printf("  -as <N>        : Enable audio stream 'N' : [0..%u]. All remaining non-global arguments between\n"
               "                   this argument and the next \"-s\" or \"-as\" argument will be applied to this\n"
               "                   stream. As default, this audio stream takes the arguments of the previous audio\n"
               "                   stream in the command line.\n"
               "                   If a video stream with network server is enabled at the same index, this audio\n"
               "                   stream will be added to the video network server, using the video network\n"
               "                   parameters. If no video stream is enabled at this index, a network server\n"
               "                   will be created for the audio stream alone. The maximum number of enabled\n"
               "                   network servers is %u (audio, video or audio+video).\n",
                                    MAX_STREAM_COUNT-1, MAX_STREAM_COUNT);
        printf("  -d <I> <O>     : Display input 'I' to the output 'O': I= [0..%u, a (all inputs)],\n"
               "                   O= [%s] where 'x0' means X-Windows output 0.\n",
                                   uiMaxVinCount-1, szVoutRange);
        printf("  -a <I> <O>     : Playback audio input 'I' on the output 'O': I= [0..%u], O= [0..%u].\n",
                                   uiMaxAinCount-1, uiMaxAoutCount-1);
        if (uiMaxMtxAinCount < uiMaxAinCount)
        {
            printf("                   Audio inputs from %u and above are ALSA devices:\n",
                                   uiMaxMtxAinCount);

            for(i = 0; i < (uiMaxAinCount - uiMaxMtxAinCount); i++)
            {
                LSTR64 szName = {0};
                LStatus eAlsaStatus = AinAlsaMod_GetDeviceName(i, &szName);
                printf("                     Input[%u]: %s\n",
                                   uiMaxMtxAinCount + i,
                                   LSTATUS_IS_SUCCESS(eAlsaStatus) ? szName : "ERROR!");
            }
        }
        if (uiMaxMtxAoutCount < uiMaxAoutCount)
        {
            printf("                   Audio outputs from %u and above are ALSA devices:\n",
                                   uiMaxMtxAoutCount);
            for(i = 0; i < (uiMaxAoutCount - uiMaxMtxAoutCount); i++)
            {
                LSTR64 szName = {0};
                LStatus eAlsaStatus = AoutAlsaMod_GetDeviceName(i, &szName);
                printf("                     Output[%u]: %s\n",
                                   uiMaxMtxAoutCount + i,
                                   LSTATUS_IS_SUCCESS(eAlsaStatus) ? szName : "ERROR!");
            }
        }
        printf("\nOptional arguments (global):\n");
        printf("  -dbg <N>       : Set debug message level 'N': [0..], default= 0 (disabled).\n");
        printf("  -vclog         : Enable VC log: default= disabled.\n");
        printf("  -stat          : Show statistics: default= disabled.\n");
        printf("  -cpf <I> <F>   : Set captured pixel format 'F' of input 'I': I= [0..%u, a (all inputs)],\n"
               "                   F= [RGB-8, RGB-10, 444-10, YUV, YUY2, 420-8-2P, 422-8-2P (default),\n"
               "                   444-8-2P, 420-8-3P, 422-8-3P, 444-8-3P, RGB-8-3P, RGB-10-3P, \n"
               "                   420-10-2P, 422-10-2P, 444-10-2P, 420-10-3P, 422-10-3P, 444-10-3P].\n",
                                   uiMaxVinCount-1);
        printf("  -dpf <O> <F>   : Set display pixel format 'F' of output 'O': O= [0..%u, a (all outputs)],\n"
               "                   F= [RGB-8 (default), RGB-10, 444-10, YUV, YUY2, 420-8-2P, 422-8-2P,\n"
               "                   444-8-2P, 420-8-3P, 422-8-3P, 444-8-3P, RGB-8-3P, RGB-10-3P, \n"
               "                   420-10-2P, 422-10-2P, 444-10-2P, 420-10-3P, 422-10-3P, 444-10-3P].\n",
                                   uiMaxVoutMtxCount-1);
        printf("  -opf <O> <F> <M>  : Set output pixel format 'F' of output 'O' with mode 'M': O= [0..%u,\n"
               "                      a (all outputs)], F= [RGB-8 (default), RGB-10, 420-8, 422-8, 444-8,\n"
               "                      420-10, 422-10, 444-10], M (with RGB): [fr (full range, default),\n"
               "                      lr (limited range)], M (with YUV): [bt601, bt709].\n",
                                      uiMaxVoutMtxCount-1);
        printf("  -fvp <O> <WxH@HZ> : Force video out parameters on output 'O': [0..%u, a (all outputs)].\n"
               "                      W= Width, H= Height, HZ= Vertical frequency. Default= Use preferred\n"
               "                      mode of the monitor.\n", uiMaxVoutMtxCount-1);
        printf("  -fsc <O> <T>   : Set full screen cycle time 'T' on the video output 'O': O= [%s], \n"
               "                   T= Cycle time in seconds, default= 0 (disabled).\n",
                                   szVoutRange);
        printf("  -vor <L,T,R,B> : Set video out window rectangle (output 1 only). L,T,R,B= Left, top,\n"
               "                   right and bottom coordinates of the window. Default= Full screen.\n");
        printf("  -acp <N/N/N/N> : Audio capture parameters for analog inputs (ignored for digital input):\n"
               "                    <AinIndex/SamplingRate/NumberOfChannels/SampleSize>\n"
               "                      AinIndex:         Index of the audio input on which these parameters are applied.\n"
               "                      SamplingRate:     Sampling frequency of the stream, default=%u.\n"
               "                      NumberOfChannels: Number of channels of the stream, default=%u.\n"
               "                      SampleSize:       Size of an audio sample for this stream, default=%u.\n",
                                   g_oDefaultAinParam.uiSamplingRate,
                                   g_oDefaultAinParam.uiNbChannels,
                                   g_oDefaultAinParam.uiSampleSize);
        printf("  -avsync        : Enable audio/video synchronization.\n");
        printf("  -ll            : Low latency mode: Minimize latency as low as possible.\n");
        printf("  -si <I>        : Scale input 'I' to its assigned output size (use with option '-d').\n"
               "                   I= [0..%u].\n", uiMaxVinCount-1);
        printf("\nOptional arguments (per stream):\n");
        printf("  -i <N> ...     : Video input index: [0..%u], default= 0. Can specify up to %u inputs.\n",
                                   uiMaxVinCount-1, MAX_VIN_PER_STEAM);
        printf("  -w <N>         : Encode frame width: [%u..%u], default= 0 (Vin width).\n",
                                   MIN_FRAME_SIZE, MAX_FRAME_SIZE);
        printf("  -h <N>         : Encode frame height: [%u..%u], default= 0 (Vin height).\n",
                                   MIN_FRAME_SIZE, MAX_FRAME_SIZE);
        printf("  -sr <L,T,R,B>  : Encode sub-rectangle: L,T,R,B= Left, top, right and bottom coordinates\n"
               "                   of the rectangle or 'full' (default) to encode the whole frame. The\n"
               "                   minimum and maximum rectangle size are %u and %u respectively.\n",
                                   MIN_FRAME_SIZE, MAX_FRAME_SIZE);
        printf("  -kar <on|off>  : Keep frame aspect ratio on scaling. Default= %s.\n",
                                   g_oDefaultStreamParam.bKeepRatio ? "on" : "off");
        printf("  -nmc <N>       : Number 'N' of columns in the input mosaic: [0..%u], default= 0 (auto)\n",
                                   uiMaxVinCount);
        printf("  -rot <N>       : Rotate frame 'N' degrees clockwise: [0, 90, 180, 270, 0F, 90F, 180F,\n"
               "                   270F], default= 0. 'F' means flip frame horizontally before to rotate.\n");
        printf("  -bz <W,H>      : Bezel width 'W' and height 'H' between each frames in the input mosaic:\n"
               "                   [0..%u], default = %d,%d.\n", MAX_BEZEL_SIZE,
                                   g_oDefaultStreamParam.oBezelSize.iWidth,
                                   g_oDefaultStreamParam.oBezelSize.iHeight);
        printf("  -o <PATH>      : Output H.264 video file: default= \"/dev/null\" (no file).\n");
        printf("  -od <N>        : Decode and output to display 'N': [%s],\n"
               "                   default= -1 (disabled).\n", szVoutRange);
        printf("  -rtsp <on|off> : RTSP server: on= enabled (default), off= disabled.\n");
        printf("  -proto <x>     : Select network protocol: rtsp (default), rtmp, rtp, ts or hls\n");
        printf("  -addr <IP>     : Select network destination address in IPv4 form.\n");
        printf("  -port <N>      : Select network destination port.\n");
        printf("  -mtu <N>       : Select network MTU size in bytes.\n");
        printf("  -srt           : Enable SRT protocol.\n");
        printf("  -srtmode       : Select SRT connection mode: listener, caller, rendezvous.\n");
        printf("  -url <x>       : Select network destination in URL form for rtmp protocol.\n");
        printf("  -name <x>      : Select network stream name.\n");
        printf("  -username <x>  : Select username for authentication.\n");
        printf("  -password <x>  : Select password for authentication.\n");
        printf("  -t <N>         : Duration in seconds of HLS segments.\n");
        printf("  -hlscount <N>  : Number of segments in HLS playlist.\n");
        printf("  -hlsmaster <x> : Name of HLS master playlist.\n");
        printf("  -f <x>         : Path for generated HLS files.\n");
        printf("  -neti <x>      : Network interface name or address.\n");
        printf("  -ipv6          : Enable IPv6 addressing.\n");
        printf("  -div <N>       : Vin frame rate divisor: [1..], default= %u.\n",
                                   g_oDefaultStreamParam.uiFrameRateDivisor);
        printf("  -tbr <N>       : Target bit rate in kbits/sec: [100..%u], default= %u.\n",
                                   ENC_MAX_BITRATE_KBPS, g_oDefaultStreamParam.uiTargetBitRateKbs);
        printf("  -ip <N>        : I period: [1..], default= %u. Must be a multiple of -pp value.\n",
                                   g_oDefaultStreamParam.uiIPeriod);
        printf("  -pp <N>        : P period: [1..8], default= %u. Must be divisible by -ip value.\n",
                                   g_oDefaultStreamParam.uiPPeriod);
        printf("  -qpmin <N>     : Minimum QP: [0..51], default= %u. Must be < or = to -qpmax value.\n",
                                   g_oDefaultStreamParam.uiQpMin);
        printf("  -qpmax <N>     : Maximum QP: [0..51], default= %u.  Must be > or = to -qpmin value.\n",
                                   g_oDefaultStreamParam.uiQpMax);
        printf("  -ns <N>        : Number of slices: [1..%u], default= %u.\n", ENC_MAX_NB_SLICES,
                                   g_oDefaultStreamParam.uiSliceCount);
        printf("  -rc <M>        : Set rate control mode 'M'. Options: win = Windowed (default),\n"
               "                   man = Manual.\n");
        printf("  -prof <P>      : Encoding profile 'P'. P= [b, m, hi, hi10, hi422, hi444 (default),\n"
               "                   cavlc].\n");
        printf("  -lvl <L>       : Encoding level 'L'. L= [1B, 1, 1.1, 1.2, 1.3, 2, 2.1, 2.2, 3, 3.1, 3.2,\n"
               "                   4, 4.1, 4.2, 5, 5.1, 5.2 (default)].\n");
        printf("  -epf <F>       : Set encoding pixel format: F= [400-8, 400-10, 420-8 (default),\n"
               "                   420-10, 422-8, 422-10, 444-8 or 444-10].\n");
        printf("  -faq <N>       : Frequency adaptive quantization in percent: [0..100], default= %u.\n",
                                   g_oDefaultStreamParam.uiFaq);
        printf("  -df <on|off>   : Deblocking filter. Default= %s.\n",
                                   g_oDefaultStreamParam.bDfEnabled ? "on" : "off");

        printf("\nOptional arguments (per audio stream):\n");
        printf("  -i <N>         : Audio input index: [0..%u], default= 0.\n",
                                   uiMaxAinCount-1);
        printf("  -fmt <F>       : AAC stream format. Can be '%s' or '%s', default='%s'.\n",
                                   GetAacStreamFormatName(LAudioCodec_AACStreamFormat_ADTS),
                                   GetAacStreamFormatName(LAudioCodec_AACStreamFormat_RAW),
                                   GetAacStreamFormatName(g_oDefaultAudioStreamParam.eAacStreamFormat));
        printf("  -prof <P>      : AAC profile. Can be '%s', '%s' or '%s', default='%s'.\n",
                                   GetAacProfileName(LAudioCodec_AACProfile_LC),
                                   GetAacProfileName(LAudioCodec_AACProfile_HEV1),
                                   GetAacProfileName(LAudioCodec_AACProfile_HEV2),
                                   GetAacProfileName(g_oDefaultAudioStreamParam.eAacProfile));
        printf("  -br <N>        : AAC bitrate in bps, default=%u.\n"
               "                     MinBitrate:  %u.\n"
               "                     MaxBitrate:  - For %s profile:\t6 * NumberOfChannels * SamplingRate.\n"
               "                                  - For %s profile:\t3 * NumberOfChannels * SamplingRate.\n"
               "                                  - For %s profile:\t3 * SamplingRate.\n",
                                   g_oDefaultAudioStreamParam.uiAacBitrate,
                                   MIN_AAC_BITRATE,
                                   GetAacProfileName(LAudioCodec_AACProfile_LC),
                                   GetAacProfileName(LAudioCodec_AACProfile_HEV1),
                                   GetAacProfileName(LAudioCodec_AACProfile_HEV2));
        printf("  -q <N>         : AAC quality level. Can be 0, 1 or 2, default=%u\n",
                                   GetAacQualityLevel(g_oDefaultAudioStreamParam.eAacQualityLvl));
        printf("  -tns <N>       : Use temporal noise shaping. 0: disable, 1: enable, default=%d\n",
                                   g_oDefaultAudioStreamParam.bUseTns);
        printf("  -proto <x>     : Select network protocol: rtsp (default), rtmp, rtp, ts or hls. \n"
               "                   Ignored if a video network server is enabled at the same stream index.\n");
        printf("  -addr <IP>     : Select network destination address in IPv4 form.\n"
               "                   Ignored if a video network server is enabled at the same stream index.\n");
        printf("  -port <N>      : Select network destination port.\n");
        printf("  -mtu <N>       : Select network MTU size in bytes.\n");
        printf("  -srt           : Enable SRT protocol.\n");
        printf("  -srtmode       : Select SRT connection mode: listener, caller, rendezvous.\n");
        printf("  -url <x>       : Select network destination in URL form for rtmp protocol.\n");
        printf("  -name <x>      : Select network stream name.\n");
        printf("  -username <x>  : Select username for authentication.\n");
        printf("  -password <x>  : Select password for authentication.\n");
        printf("  -t <N>         : Duration in seconds of HLS segments.\n");
        printf("  -hlscount <N>  : Number of segments in HLS playlist.\n");
        printf("  -hlsmaster <x> : Name of HLS master playlist.\n");
        printf("  -f <x>         : Path for generated HLS files.\n");
        printf("  -neti <x>      : Network interface name or address.\n");
        printf("  -ipv6          : Enable IPv6 addressing.\n");
        printf("\nExamples:\n");
        printf("  Capture video from input 0 and display it to video output 0:\n"
               "    %s -d 0 0\n", argv[0]);
        printf("  Create a RTSP server streaming encoded video from input 0:\n"
               "    %s -s 0\n", argv[0]);
        printf("  Create a RTSP server streaming encoded audio from input 0:\n"
               "    %s -as 0\n", argv[0]);
        printf("  Create a RTSP server streaming encoded audio from input 0, change port value,\n"
               "  and playback audio on output 0:\n"
               "    %s -s 0 -rtsp off -port 1234 -as 0 -a 0 0\n", argv[0]);
        printf("  Create a RTSP server streaming encoded video and audio from input 0:\n"
               "    %s -s 0 -as 0\n", argv[0]);
        printf("  Create 4 RTSP servers from 4 video inputs (0 to 3) and display all on video output 1:\n"
               "    %s -s 0 -i 0 -s 1 -i 1 -s 2 -i 2 -s 3 -i 3 -d a 1\n", argv[0]);
        printf("  Capture, encode, decode and display one video stream without RTSP server:\n"
               "    %s -s 0 -od 0 -rtsp off\n", argv[0]);
        printf("\n");

        return -1;
    }

    //--------------------------------------------------------------------------------------------------------
    // Init modules

    SetMsgLogLevel(uiMsgLogLevel);

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = SetVideoCodecFwTraceLevel(hDevice, bEnableVcLog ? 0xFF : 3);
        if (!LSTATUS_IS_SUCCESS(eStatus))
        {
            printf("***** SetVideoCodecFwTraceLevel failed!!!! *****\n");
        }
    }

    if(bAvSync)
    {
        InitAVSyncParams();
    }

    MBOOL bRestart;

  Restart:
    bRestart = MFALSE;

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        printf("***** Init Vin/Vout *****\n");

        for (i = 0; i < MAX_VIN_COUNT; i++)
        {
            if (g_aoVinParam[i].bEnabled && LSTATUS_IS_SUCCESS(eStatus))
            {
                eStatus = VinMod_Init(&g_aoVinParam[i].oVinMod, hDevice, i, 16, 1, 0,
                                      g_aoVinParam[i].ePixFmt, bLowLatency);

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    printf("Vin[%u] mode: %ux%u%s@%0.3fHz%s\n",
                           i,
                           g_aoVinParam[i].oVinMod.oFrameSize.iWidth,
                           g_aoVinParam[i].oVinMod.oFrameSize.iHeight,
                           g_aoVinParam[i].oVinMod.bInterlaced ? "i" : "",
                           (MFLOAT32)(g_aoVinParam[i].oVinMod.uiFrameRate1000x) / 1000,
                           g_aoVinParam[i].oVinMod.bFailSafe ? " (No signal -> Failsafe mode)" : "");

                    if (g_aoVinParam[i].oVinMod.bInterlaced)
                    {
                        printf("Vin[%u]: Interlaced mode not supported!\n", i);
                        eStatus = LStatus_UNSUPPORTED;
                    }
                }
            }
        }

        for (i = 0; i < MAX_STREAM_COUNT; i++)
        {
            poStreamParam = &g_aoStreamParam[i];

            if (poStreamParam->bEnable)
            {
                MUINT uiMaxWidth  = 0;
                MUINT uiMaxHeight = 0;

                MUINT j;
                for (j = 0; j < poStreamParam->uiVinCount; j++)
                {
                    uiMaxWidth  = max(uiMaxWidth,  poStreamParam->apoVinParam[j]->oVinMod.oFrameSize.iWidth);
                    uiMaxHeight = max(uiMaxHeight, poStreamParam->apoVinParam[j]->oVinMod.oFrameSize.iHeight);
                }

                if (poStreamParam->oFrameDiv.iWidth == 0)
                {
                    poStreamParam->oFrameDiv.iWidth
                        = (poStreamParam->uiVinCount <= 1) ? 1 :
                          (poStreamParam->uiVinCount <= 4) ? 2 :
                          (poStreamParam->uiVinCount <= 9) ? 3 : 4;
                }
                else if (poStreamParam->oFrameDiv.iWidth > poStreamParam->uiVinCount)
                {
                    poStreamParam->oFrameDiv.iWidth = poStreamParam->uiVinCount;
                }

                poStreamParam->oFrameDiv.iHeight
                    = (poStreamParam->uiVinCount + poStreamParam->oFrameDiv.iWidth - 1)
                      / poStreamParam->oFrameDiv.iWidth;

                if (poStreamParam->bUseVinWidth)
                {
                    poStreamParam->oFrameSize.iWidth = poStreamParam->oFrameDiv.iWidth * uiMaxWidth;
                }

                if (poStreamParam->bUseVinHeight)
                {
                    poStreamParam->oFrameSize.iHeight = poStreamParam->oFrameDiv.iHeight * uiMaxHeight;
                }

                if (poStreamParam->bFullFrame)
                {
                    poStreamParam->oSubFrameSize   = poStreamParam->oFrameSize;
                    poStreamParam->oSubFramePos.iX = 0;
                    poStreamParam->oSubFramePos.iY = 0;
                }

                poStreamParam->oSubFrameSize.iWidth += poStreamParam->oBezelSize.iWidth
                                                       * (poStreamParam->oFrameDiv.iWidth - 1);

                poStreamParam->oSubFrameSize.iHeight += poStreamParam->oBezelSize.iHeight
                                                        * (poStreamParam->oFrameDiv.iHeight - 1);

                if (poStreamParam->oSubFrameSize.iWidth > MAX_FRAME_SIZE)
                {
                    poStreamParam->oSubFrameSize.iWidth = MAX_FRAME_SIZE;
                }

                if (poStreamParam->oSubFrameSize.iHeight > MAX_FRAME_SIZE)
                {
                    poStreamParam->oSubFrameSize.iHeight = MAX_FRAME_SIZE;
                }

                if ((poStreamParam->eRotation == LVPC_Rotation_90)
                    || (poStreamParam->eRotation == LVPC_Rotation_270)
                    || (poStreamParam->eRotation == LVPC_Rotation_FLIP_90)
                    || (poStreamParam->eRotation == LVPC_Rotation_FLIP_270))
                {
                    MUINT uiHeight = poStreamParam->oSubFrameSize.iWidth;
                    poStreamParam->oSubFrameSize.iWidth  = poStreamParam->oSubFrameSize.iHeight;
                    poStreamParam->oSubFrameSize.iHeight = uiHeight;
                }
            }
        }

        for (i = 0; i < MAX_VOUT_COUNT; i++)
        {
            if ((g_aoVoutParam[i].uiInputCount > 0) && LSTATUS_IS_SUCCESS(eStatus))
            {
                ModuleLinkInput* poVoutInLink = MNULL;
                MUINT            uiFreq       = 60;

                if (i < uiMaxVoutMtxCount)
                {
                    poVoutInLink = &(g_aoVoutParam[i].oVoutMod.oInLink);

                    eStatus = VoutMod_Init(
                                    &g_aoVoutParam[i].oVoutMod,
                                    hDevice,
                                    i,
                                    g_aoVoutParam[i].ePixFmt,
                                    g_aoVoutParam[i].bForceMode ? &g_aoVoutParam[i].oForcedSize : MNULL,
                                    g_aoVoutParam[i].bForceMode ? g_aoVoutParam[i].uiForcedFps  : 0,
                                    60,
                                    1,
                                    g_aoVoutParam[i].bIsMasterSync,
                                    g_aoVoutParam[i].bIsMasterSync,
                                    bLowLatency);

                    if (LSTATUS_IS_SUCCESS(eStatus))
                    {
                        eStatus = VoutMod_SetOutputFormat(
                                            &g_aoVoutParam[i].oVoutMod,
                                            g_aoVoutParam[i].eOutPixFmt,
                                            g_aoVoutParam[i].bOutRgbFullYuv709);
                    }

                    if (LSTATUS_IS_SUCCESS(eStatus))
                    {
                        g_aoVoutParam[i].oSize.iWidth  = g_aoVoutParam[i].oVoutMod.oVidParam.uiHActive;
                        g_aoVoutParam[i].oSize.iHeight = g_aoVoutParam[i].oVoutMod.oVidParam.uiVActive;
                        uiFreq                         = g_aoVoutParam[i].oVoutMod.oVidParam.uiVRate_Hz;
                    }

                }
              #ifdef LINUX
                else if (i >= MAX_VOUTMTX_COUNT)
                {
                    poVoutInLink = &(g_aoVoutParam[i].oVoutGlxMod.oInLink);

                    eStatus = VoutGlxMod_Init(
                                    &g_aoVoutParam[i].oVoutGlxMod,
                                    hDevice,
                                    bForceWinRect ? &oWinRect : MNULL,
                                    60,
                                    1,
                                    MTRUE);

                    if (LSTATUS_IS_SUCCESS(eStatus))
                    {
                        g_aoVoutParam[i].oSize = g_aoVoutParam[i].oVoutGlxMod.oWindowSize;
                    }
                }
              #endif
                else
                {
                    eStatus = LStatus_FAIL;
                }

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    MUINT uiVoutIdx = (i < MAX_VOUTMTX_COUNT) ? i : (i - MAX_VOUTMTX_COUNT);

                    printf("Vout[%s%u] mode: %ux%u@%uHz\n", (i < MAX_VOUTMTX_COUNT) ? "" : "x",
                           uiVoutIdx, g_aoVoutParam[i].oSize.iWidth, g_aoVoutParam[i].oSize.iHeight, uiFreq);

                    LBuffer_Attributes oDispBufAttrib;

                    oDispBufAttrib.eAttributeType                = LBuffer_Type_VIDEO;
                    oDispBufAttrib.oVideoAttributes.ePixelFormat = g_aoVoutParam[i].ePixFmt;
                    oDispBufAttrib.oVideoAttributes.uiWidth      = g_aoVoutParam[i].oSize.iWidth;
                    oDispBufAttrib.oVideoAttributes.uiHeight     = g_aoVoutParam[i].oSize.iHeight;

                    eStatus = MosMod_Init(
                                &g_aoVoutParam[i].oOutMosMod,
                                hDevice,
                                24,
                                &oDispBufAttrib,
                                MNULL,
                                MNULL,
                                bVinOnly,
                                bLowLatency);
                }

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    ModLnk_Connect(&g_aoVoutParam[i].oOutMosMod.oOutLink, poVoutInLink);
                }
            }
        }

        for (i = 0; i < MAX_VIN_COUNT; i++)
        {
            if (g_aoVinParam[i].bEnabled
                && (g_aoVinParam[i].poVoutParam != MNULL)
                && LSTATUS_IS_SUCCESS(eStatus))
            {
                MUINT uiInMosIdx = g_aoVinParam[i].uiInMosIdx;
                MUINT uiVoutIdx  = g_aoVinParam[i].poVoutParam->uiIndex;

                printf("[Vin%u]--->%u:[Mosaic]--->[Vout%s%u]\n",
                       i, uiInMosIdx,
                       (uiVoutIdx < MAX_VOUTMTX_COUNT) ? "" : "-x",
                       (uiVoutIdx < MAX_VOUTMTX_COUNT) ? uiVoutIdx : (uiVoutIdx - MAX_VOUTMTX_COUNT));

                eStatus = ModLnk_Connect(&g_aoVinParam[i].oVinMod.oOutLink,
                                         &g_aoVinParam[i].poVoutParam->oOutMosMod.aoInLink[uiInMosIdx]);

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    eStatus = InitInMosParam(
                                g_aoVinParam[i].uiInMosIdx,
                                g_aoVinParam[i].oVinMod.oFrameSize.iWidth,
                                g_aoVinParam[i].oVinMod.oFrameSize.iHeight,
                                g_aoVinParam[i].oVinMod.uiFrameRate1000x,
                                1000,
                                g_aoVinParam[i].poVoutParam,
                                g_aoVinParam[i].bDoScaling);
                }
            }
        }

        if (LSTATUS_IS_FAIL(eStatus))
        {
            printf(
                "***** Init Vin/Vout *****: FAILED! (status= %d - %s)\n",
                eStatus,
                GetStatusStr(eStatus));
        }

        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            printf("***** Init Ain/Aout *****\n");

            for (i = 0; i < MAX_AIN_COUNT; i++)
            {
                if (g_aoAinParam[i].bEnabled
                     && LSTATUS_IS_SUCCESS(eStatus))
                {
                    AinParam* poAinParam = &g_aoAinParam[i];

                    if (i < uiMaxMtxAinCount)
                    {
                        eStatus = AinMod_Init(
                                            &poAinParam->oAinMod,
                                            hDevice,
                                            poAinParam->uiIndex,
                                            32,
                                            1024);

                        poAinParam->uiSamplingRate  = LAudioFormat_GetSampleRate(poAinParam->oAinMod.eAudioFormat);
                        poAinParam->uiNbChannels    = LAudioFormat_GetNumberOfChannel(poAinParam->oAinMod.eAudioFormat);
                        poAinParam->uiSampleSize    = LAudioFormat_GetSampleSize(poAinParam->oAinMod.eAudioFormat);
                    }
                    else
                    {
                        eStatus = AinAlsaMod_Init(
                                            &(poAinParam->oAinAlsaMod),
                                            hDevice,
                                            32,
                                            (i - uiMaxMtxAinCount),
                                            poAinParam->uiSamplingRate,
                                            poAinParam->uiNbChannels,
                                            poAinParam->uiSampleSize,
                                            MFALSE,
                                            MFALSE,
                                            1024);

                        poAinParam->bMtxIn = MFALSE;
                    }

                    eStatus = LAudioFormat_GetAudioFormat(
                                        poAinParam->uiSamplingRate,
                                        poAinParam->uiSampleSize,
                                        poAinParam->uiNbChannels,
                                        1,
                                        &poAinParam->eAudioFmt);

                    if(LSTATUS_IS_SUCCESS(eStatus))
                    {
                        if (!poAinParam->oAinMod.bFailSafe)
                        {
                            printf("Ain%s[%u] mode: %uBit - %uCh - %uHz\n",
                                   poAinParam->bMtxIn ? "" : "(ALSA)",
                                   poAinParam->uiIndex,
                                   poAinParam->uiSampleSize,
                                   poAinParam->uiNbChannels,
                                   poAinParam->uiSamplingRate);
                        }
                        else
                        {
                            printf("Ain%s[%u] mode: No signal -> Failsafe mode.\n",
                                   poAinParam->bMtxIn ? "" : "(ALSA)",
                                   poAinParam->uiIndex);
                        }
                    }
                }
            }

            for (i = 0; i < MAX_AOUT_COUNT; i++)
            {
                if (g_aoAoutParam[i].bEnabled
                     && LSTATUS_IS_SUCCESS(eStatus)
                     && (g_aoAoutParam[i].poAinParam != MNULL))
                {
                    AoutParam*          poAoutParam     = &g_aoAoutParam[i];
                    AinParam*           poAinParam      = poAoutParam->poAinParam;
                    ModuleLinkInput*    poLnkInput      = MNULL;

                    if (i < uiMaxMtxAoutCount)
                    {
                        eStatus = AoutMod_Init(
                                            &poAoutParam->oAoutMod,
                                            hDevice,
                                            poAoutParam->uiIndex,
                                            poAinParam->eAudioFmt,
                                            1024,
                                            poAoutParam->poModSyncMst,
                                            MTRUE,
                                            MFALSE);

                        poLnkInput = &poAoutParam->oAoutMod.oInLink;
                    }
                    else
                    {
                        eStatus = AoutAlsaMod_Init(
                                            &(poAoutParam->oAoutAlsaMod),
                                            (i - uiMaxMtxAoutCount),
                                            poAinParam->uiSamplingRate,
                                            poAinParam->uiNbChannels,
                                            poAinParam->uiSampleSize,
                                            MFALSE,
                                            MFALSE,
                                            1024);

                        poLnkInput = &poAoutParam->oAoutAlsaMod.oInLink;
                        poAoutParam->bMtxOut = MFALSE;
                    }

                    if(LSTATUS_IS_SUCCESS(eStatus))
                    {
                        eStatus = ModLnk_Connect(
                                        poAinParam->bMtxIn
                                          ? &poAinParam->oAinMod.oOutLink
                                          : &poAinParam->oAinAlsaMod.oOutLink,
                                        poLnkInput);
                    }
                }
            }

            if (LSTATUS_IS_FAIL(eStatus))
            {
                printf(
                    "***** Init Ain/Aout *****: FAILED! (status= %d - %s)\n",
                    eStatus,
                    GetStatusStr(eStatus));
            }
        }
    }

    if (bVinOnly)
    {
        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = InitStream(hDevice, &g_aoStreamParam[0], MNULL);

            for (i = 0; i < MAX_VIN_COUNT; i++)
            {
                if (g_aoVinParam[i].bEnabled)
                {
                    g_aoStreamParam[0].apoVinParam[0] = &g_aoVinParam[i];
                    break;
                }
            }
        }
    }

    for (i = 0; i < MAX_STREAM_COUNT; i++)
    {
        MUINT uiMaxNaluSizeBytes = 0;

        if (g_aoStreamParam[i].bEnable && LSTATUS_IS_SUCCESS(eStatus))
        {
            printf("***** Init Stream[%u] *****\n", i);

            eStatus = InitStream(hDevice, &g_aoStreamParam[i], &uiMaxNaluSizeBytes);

            if (LSTATUS_IS_FAIL(eStatus))
            {
                printf(
                    "***** Init Stream[%u] *****: FAILED! (status= %d - %s)\n",
                    i,
                    eStatus,
                    GetStatusStr(eStatus));
            }
        }

        if(g_aoAudioStreamParam[i].bEnable && LSTATUS_IS_SUCCESS(eStatus))
        {
            printf("***** Init Audio Stream[%u] *****\n", i);

            eStatus = InitAudioStream(hDevice, i);

            if (LSTATUS_IS_FAIL(eStatus))
            {
                printf("***** Init Audio Stream[%u] *****: FAILED! (status= %d - %s)\n",
                    i,
                    eStatus,
                    GetStatusStr(eStatus));
            }
        }

        if(((g_aoStreamParam[i].bEnable && g_aoStreamParam[i].bNetServer)
             || g_aoAudioStreamParam[i].bEnable)
            && LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = SetRtspServerParams(
                          &g_aoStreamParam[i],
                          &g_aoAudioStreamParam[i],
                          uiMaxNaluSizeBytes,
                          g_aoAudioStreamParam[i].oAudioEncMod.uiMaxEncodedBufSize,
                          i);
        }
    }

    //--------------------------------------------------------------------------------------------------------
    // Start video stream

    // We must start from last to first module.
    for (i = 0; i < MAX_VOUT_COUNT; i++)
    {
        if ((g_aoVoutParam[i].uiInputCount > 0) && LSTATUS_IS_SUCCESS(eStatus))
        {
            if (i < uiMaxVoutMtxCount)
            {
                printf("***** Start Vout[%u] *****\n", i);

                eStatus = VoutMod_Start(&g_aoVoutParam[i].oVoutMod);
            }
          #ifdef LINUX
            else if (i >= MAX_VOUTMTX_COUNT)
            {
                printf("***** Start Vout[x%u] *****\n", i - MAX_VOUTMTX_COUNT);

                eStatus = VoutGlxMod_Start(&g_aoVoutParam[i].oVoutGlxMod);

            }
          #endif
            else
            {
                eStatus = LStatus_FAIL;
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                eStatus = MosMod_Start(&g_aoVoutParam[i].oOutMosMod);

                g_aoVoutParam[i].uiNextCycleTimeSec
                    = (MUINT)(GetMonoTimeUsec() / (1000 * 1000)) + g_aoVoutParam[i].uiCycleTimeSec;
            }

            if (LSTATUS_IS_FAIL(eStatus))
            {
                printf(
                    "***** Start Vout[%u] *****: FAILED! (status= %d - %s)\n",
                    i,
                    eStatus,
                    GetStatusStr(eStatus));
            }
        }
    }

    for (i = 0; i < MAX_AOUT_COUNT; i++)
    {
        if (g_aoAoutParam[i].bEnabled && LSTATUS_IS_SUCCESS(eStatus))
        {
            if (i < uiMaxMtxAoutCount)
            {
                printf("***** Start Aout[%u] *****\n", i);

                eStatus = AoutMod_Start(&g_aoAoutParam[i].oAoutMod);
            }
            else
            {
                printf("***** Start Aout-ALSA[%u] *****\n", i);

                eStatus = AoutAlsaMod_Start(&g_aoAoutParam[i].oAoutAlsaMod);
            }

            if(LSTATUS_IS_FAIL(eStatus))
            {
                printf(
                    "***** Start Aout[%u] *****: FAILED! (status= %d - %s)\n",
                    i,
                    eStatus,
                    GetStatusStr(eStatus));
            }
        }
    }

    for (i = 0; i < MAX_STREAM_COUNT; i++)
    {
        if (g_aoStreamParam[i].bEnable && LSTATUS_IS_SUCCESS(eStatus))
        {
            printf("***** Start Stream[%u] *****\n", i);

            eStatus = StartStream(&g_aoStreamParam[i]);

            if (LSTATUS_IS_FAIL(eStatus))
            {
                printf(
                    "***** Start Stream[%u] *****: FAILED (status= %d - %s)\n",
                    i,
                    eStatus,
                    GetStatusStr(eStatus));
            }
        }

        if(g_aoAudioStreamParam[i].bEnable && LSTATUS_IS_SUCCESS(eStatus))
        {
            printf("***** Start Audio Stream[%u] *****\n", i);

            eStatus = StartAudioStream(&g_aoAudioStreamParam[i]);

            if (LSTATUS_IS_FAIL(eStatus))
            {
                printf(
                    "***** Start Audio Stream[%u] *****: FAILED (status= %d - %s)\n", 
                    i, 
                    eStatus,
                    GetStatusStr(eStatus));
            }
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        for (i = 0; i < MAX_VIN_COUNT; i++)
        {
            if (g_aoVinParam[i].bEnabled && LSTATUS_IS_SUCCESS(eStatus))
            {
                printf("***** Start Vin[%u] *****\n", i);

                eStatus = VinMod_Start(&g_aoVinParam[i].oVinMod);

                if (LSTATUS_IS_FAIL(eStatus))
                {
                    printf(
                        "***** Start Vin[%u] *****: FAILED! (status= %d - %s)\n",
                        i,
                        eStatus,
                        GetStatusStr(eStatus));
                }
            }
        }
    }

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        for (i = 0; i < MAX_AIN_COUNT; i++)
        {
            if (g_aoAinParam[i].bEnabled && LSTATUS_IS_SUCCESS(eStatus))
            {
                if (i < uiMaxMtxAinCount)
                {
                    printf("***** Start Ain[%u] *****\n", i);

                    eStatus = AinMod_Start(&g_aoAinParam[i].oAinMod);
                }
                else
                {
                    printf("***** Start Ain-ALSA[%u] *****\n", i);

                    eStatus = AinAlsaMod_Start(&g_aoAinParam[i].oAinAlsaMod);
                }

                if(LSTATUS_IS_FAIL(eStatus))
                {
                    printf(
                        "***** Start Ain[%u] *****: FAILED! (status= %d - %s)\n",
                        i,
                        eStatus,
                        GetStatusStr(eStatus));
                }
            }
        }
    }

    //--------------------------------------------------------------------------------------------------------
    // Running

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        printf("***** Started *****\n");
        ShowInputKeyOptions();

        MUINT   iShowStatIdx          = ~0;
        MBOOL   bForceShow            = MFALSE;
        int     iChar                 = EOF;
        MUINT64 uiPrevBusyTimeCpu     = 0;
        MUINT64 uiPrevTotalTimeCpu    = 0;
        MBOOL   bForceFullScreenCycle = MFALSE;
        MBOOL   bPaused               = MFALSE;
        MUINT   auiVinIdx[MAX_STREAM_COUNT];

        for (i = 0; i < MAX_STREAM_COUNT; i++)
        {
            auiVinIdx[i] = 0;
        }

        while ((iChar != 'q') && !bRestart && !g_bSigTerm && LSTATUS_IS_SUCCESS(eStatus))
        {
            for (i = 0; i < MAX_VIN_COUNT; i++)
            {
                if (g_aoVinParam[i].bEnabled
                    && !g_aoVinParam[i].oVinMod.bSourceModeValid)
                {
                    printf("Vin[%u] mode changed or invalid.\n", i);
                    bRestart = MTRUE;
                }
            }

            for(i = 0; i < MAX_AIN_COUNT; i++)
            {
                if(g_aoAinParam[i].bEnabled
                    && (i < uiMaxMtxAinCount)
                    && (!g_aoAinParam[i].oAinMod.bSourceParamsValid))
                {
                    printf("Ain[%u] mode changed or invalid.\n", i);
                    bRestart = MTRUE;
                }
            }

            for (i = 0; i < MAX_VOUT_COUNT; i++)
            {
                if ((g_aoVoutParam[i].uiInputCount > 0)
                    && ((g_aoVoutParam[i].uiCycleTimeSec > 0)
                        || bForceFullScreenCycle))
                {
                    MUINT uiTimeSec = GetMonoTimeUsec() / (1000 * 1000);

                    if ((uiTimeSec >= g_aoVoutParam[i].uiNextCycleTimeSec)
                        || bForceFullScreenCycle)
                    {
                        MosMod_ShowOneInputFullScreen(
                            &g_aoVoutParam[i].oOutMosMod,
                            (g_aoVoutParam[i].uiNextCycleInputIdx < g_aoVoutParam[i].uiInputCount),
                            g_aoVoutParam[i].uiNextCycleInputIdx);

                        g_aoVoutParam[i].uiNextCycleTimeSec = uiTimeSec + g_aoVoutParam[i].uiCycleTimeSec;
                        g_aoVoutParam[i].uiNextCycleInputIdx++;

                        if (g_aoVoutParam[i].uiNextCycleInputIdx > g_aoVoutParam[i].uiInputCount)
                        {
                            g_aoVoutParam[i].uiNextCycleInputIdx = 0;
                        }
                    }
                }
            }

            bForceFullScreenCycle = MFALSE;

            if (bVinOnly)
            {
                if (DoStats(&(g_aoStreamParam[0]),
                            0,
                            0,
                            bShowStats || bForceShow,
                            &uiPrevBusyTimeCpu,
                            &uiPrevTotalTimeCpu))
                {
                    bForceShow = MFALSE;
                }
            }

            for (i = 0; i < MAX_STREAM_COUNT; i++)
            {
                const MBOOL bVideoServer = g_aoStreamParam[i].bEnable && g_aoStreamParam[i].bNetServer;
                const MBOOL bAudioServer = g_aoAudioStreamParam[i].bEnable;

                if (bVideoServer || bAudioServer)
                {
                    if (!g_aoStreamParam[i].bSrvAddrDisplayed
                        && g_aoStreamParam[i].oRtspSrvMod.bSessionStarted)
                    {
                        char szServParams[512]      = {0};
                        char szServVideoParams[256] = {0};
                        char szServAudioParams[256] = {0};
                        char szVideoPort[16]        = {0};
                        char szAudioPort[16]        = {0};

                        if (bVideoServer && (g_aoStreamParam[i].uiNetPort != 0))
                        {
                            snprintf(szVideoPort, sizeof(szVideoPort), "-port %u", g_aoStreamParam[i].uiNetPort);
                        }

                        if (bAudioServer && (g_aoAudioStreamParam[i].uiNetPort != 0))
                        {
                            snprintf(szAudioPort, sizeof(szAudioPort), "-port %u", g_aoAudioStreamParam[i].uiNetPort);
                        }

                        LNetStreamer_Protocol   eNetProtocol;
                        const char*             szNetAddress;

                        if (bVideoServer)
                        {
                            eNetProtocol = g_aoStreamParam[i].eNetProtocol;
                            szNetAddress = g_aoStreamParam[i].szNetAddress;
                        }
                        else
                        {
                            eNetProtocol = g_aoAudioStreamParam[i].eNetProtocol;
                            szNetAddress = g_aoAudioStreamParam[i].szNetAddress;
                        }

                        if ((LNetStreamer_Protocol_RTSP == eNetProtocol) ||
                            (LNetStreamer_Protocol_RTMP == eNetProtocol))
                        {
                            snprintf(szServParams, sizeof(szServParams),
                                    "Stream[%u] parameters: -i %s -addr %s",
                                    i, g_aoStreamParam[i].oRtspSrvMod.szServerLocation,
                                    szNetAddress ? szNetAddress : "0");
                        }
                        else if (LNetStreamer_Protocol_RTP == eNetProtocol)
                        {
                            snprintf(szServParams, sizeof(szServParams),
                                    "Stream[%u] parameters: -proto rtp -addr %s",
                                    i, szNetAddress ? szNetAddress : "0");
                        }
                        else if (LNetStreamer_Protocol_TS == eNetProtocol)
                        {
                            const char* szPort = bVideoServer ? szVideoPort : szAudioPort;

                            snprintf(szServParams, sizeof(szServParams),
                                    "Stream[%u] parameters: -proto ts -addr %s %s",
                                    i, szNetAddress ? szNetAddress : "0", szPort);
                        }

                        if (bVideoServer)
                        {
                            const char* szPort = (LNetStreamer_Protocol_RTP == eNetProtocol)
                                                   ? szVideoPort : "";

                            snprintf(szServVideoParams, sizeof(szServVideoParams),
                                     "  Video parameters: -w %u -h %u -fps %u/%u %s",
                                     g_aoStreamParam[i].oSubFrameSize.iWidth,
                                     g_aoStreamParam[i].oSubFrameSize.iHeight,
                                     g_aoStreamParam[i].oEncMod.oSeqData.oFrameRate.uiNumerator,
                                     g_aoStreamParam[i].oEncMod.oSeqData.oFrameRate.uiDenominator,
                                     szPort);
                        }

                        if (bAudioServer)
                        {
                            const char* szPort = (LNetStreamer_Protocol_RTP == eNetProtocol)
                                                   ? szAudioPort : "";

                            snprintf(szServAudioParams, sizeof(szServAudioParams),
                                     "  Audio parameters: -rate %u -ch %u -prof %s %s",
                                     g_aoAudioStreamParam[i].poAinParam->uiSamplingRate,
                                     g_aoAudioStreamParam[i].poAinParam->uiNbChannels,
                                     GetAacProfileName(g_aoAudioStreamParam[i].eAacProfile),
                                     szPort);
                        }

                        MUINT j;

                        MUINT uiHdrLen = max(strlen(szServParams),
                                             max(strlen(szServVideoParams), strlen(szServAudioParams)));

                        for (j = 0; j < uiHdrLen; j++)
                        {
                            printf("%c", '~');
                        }

                        printf("\n%s\n%s%s%s%s",
                               szServParams,
                               szServVideoParams,
                               (strlen(szServVideoParams) > 0) ? ("\n") : (""),
                               (strlen(szServAudioParams) > 0) ? (szServAudioParams) : (""),
                               (strlen(szServAudioParams) > 0) ? ("\n") : (""));

                        for (j = 0; j < uiHdrLen; j++)
                        {
                            printf("%c", '~');
                        }

                        printf("%s", "\n");

                        g_aoStreamParam[i].bSrvAddrDisplayed = MTRUE;
                    }
                }

                if(g_aoStreamParam[i].bEnable)
                {
                    if (iShowStatIdx == ~0)
                    {
                        iShowStatIdx = i;
                    }

                    MBOOL bShowNow = ((bShowStats || bForceShow)
                                      && (iShowStatIdx == i));

                    if (DoStats(&(g_aoStreamParam[i]), i, auiVinIdx[i], bShowNow, &uiPrevBusyTimeCpu,
                                &uiPrevTotalTimeCpu))
                    {
                        bForceShow = MFALSE;
                    }
                }
            }

            for (i = 0; i < MAX_STREAM_COUNT; i++)
            {
                if (g_aoStreamParam[i].bEnable &&
                    g_aoStreamParam[i].bNetServer &&
                    g_aoStreamParam[i].oRtspSrvMod.bUnresponsive)
                {
                    printf("Forced restart because of unresponsive network\n");
                    bRestart = MTRUE;
                }
            }

            usleep(10*1000);
            iChar = GetChar();

            if (iChar != EOF)
            {
                printf(" > ");

                if (((iChar >= '0') && (iChar <= '9'))
                    || ((iChar >= 'a') && (iChar <= 'f')))
                {
                    if ((iChar >= '0') && (iChar <= '9'))
                    {
                        i = iChar - '0';
                    }
                    else
                    {
                        i = iChar - 'a' + 10;
                    }

                    if (bVinOnly)
                    {
                        if ((i < MAX_VIN_COUNT)
                            && g_aoVinParam[i].bEnabled)
                        {
                            printf("Show statistics of Vin[%u].\n", i);
                            g_aoStreamParam[0].apoVinParam[0] = &g_aoVinParam[i];
                            bForceShow = MTRUE;
                        }
                        else
                        {
                            printf("Vin[%u] not enabled.\n", i);
                        }
                    }
                    else
                    {
                        if ((i < MAX_STREAM_COUNT)
                            && g_aoStreamParam[i].bEnable)
                        {
                            printf("Show statistics of Stream[%u].\n", i);
                            iShowStatIdx = i;
                            bForceShow   = MTRUE;
                        }
                        else
                        {
                            printf("Stream[%u] not enabled.\n", i);
                        }
                    }
                }
                else if (iChar == 'i')
                {
                    auiVinIdx[iShowStatIdx]++;

                    if (auiVinIdx[iShowStatIdx] >= g_aoStreamParam[iShowStatIdx].uiVinCount)
                    {
                        auiVinIdx[iShowStatIdx] = 0;
                    }

                    printf("Show statistics of Vin[%u].\n",
                           g_aoStreamParam[iShowStatIdx].apoVinParam[auiVinIdx[iShowStatIdx]]->uiIndex);
                }
                else if (iChar == 'r')
                {
                    bRestart = MTRUE;
                }
                else if (iChar == 'n')
                {
                    bForceFullScreenCycle = MTRUE;
                    printf("Cycle full screen video.\n");
                }
                else if (iChar == 's')
                {
                    ShowModuleStates();
                }
                else if (iChar == 'P')
                {
                    if (bPaused)
                    {
                        bPaused = MFALSE;
                        printf("Resume all streaming.\n");
                    }
                    else
                    {
                        bPaused = MTRUE;
                        printf("Pause all streaming.\n");
                    }

                    for (i = 0; i < MAX_VIN_COUNT; i++)
                    {
                        g_aoVinParam[i].oVinMod.oOutLink.bReturnAllBuffer = bPaused;
                    }

                    for (i = 0; i < MAX_VOUT_COUNT; i++)
                    {
                        g_aoVoutParam[i].oOutMosMod.oOutLink.bReturnAllBuffer = bPaused;
                    }

                    for (i = 0; i < MAX_STREAM_COUNT; i++)
                    {
                        g_aoStreamParam[i].oInMosMod.oOutLink.bReturnAllBuffer = bPaused;
                    }
                }
                else if (iChar != 'q')
                {
                    printf("Invalid key.\n");
                    ShowInputKeyOptions();
                }
            }
        }

        printf("***** Stopping... *****\n");
    }

    //--------------------------------------------------------------------------------------------------------
    // Stop streams

    // We must stop in the reverse order as the start i.e. from first to last module.

    for (i = 0; i < MAX_VIN_COUNT; i++)
    {
        if (g_aoVinParam[i].bEnabled)
        {
            VinMod_Stop(&g_aoVinParam[i].oVinMod);
        }
    }

    for (i = 0; i < MAX_AIN_COUNT; i++)
    {
        if (g_aoAinParam[i].bEnabled)
        {
            if (i < uiMaxMtxAinCount)
            {
                AinMod_Stop(&g_aoAinParam[i].oAinMod);
            }
            else
            {
                AinAlsaMod_Stop(&g_aoAinParam[i].oAinAlsaMod);
            }
        }
    }

    for (i = 0; i < MAX_STREAM_COUNT; i++)
    {
        if(g_aoAudioStreamParam[i].bEnable)
        {
            StopAudioStream(&g_aoAudioStreamParam[i]);
        }

        if (g_aoStreamParam[i].bEnable)
        {
            StopStream(&g_aoStreamParam[i]);
        }
    }

    for (i = 0; i < MAX_VOUT_COUNT; i++)
    {
        if (g_aoVoutParam[i].uiInputCount > 0)
        {
            MosMod_Stop(&g_aoVoutParam[i].oOutMosMod);

            if (i < MAX_VOUTMTX_COUNT)
            {
                VoutMod_Stop(&g_aoVoutParam[i].oVoutMod);
            }
          #ifdef LINUX
            else
            {
                VoutGlxMod_Stop(&g_aoVoutParam[i].oVoutGlxMod);
            }
           #endif
        }
    }

    for (i = 0; i < MAX_AOUT_COUNT; i++)
    {
        if (g_aoAoutParam[i].bEnabled > 0)
        {
            if (i < uiMaxMtxAoutCount)
            {
                AoutMod_Stop(&g_aoAoutParam[i].oAoutMod);
            }
            else
            {
                AoutAlsaMod_Stop(&g_aoAoutParam[i].oAoutAlsaMod);
            }
        }
    }

    printf("***** Stopped ******\n");

    //--------------------------------------------------------------------------------------------------------
    // Cleanup modules

    for (i = 0; i < MAX_VIN_COUNT; i++)
    {
        VinMod_Cleanup(&(g_aoVinParam[i].oVinMod));
    }

    for (i = 0; i < MAX_AIN_COUNT; i++)
    {
        AinMod_Cleanup(&(g_aoAinParam[i].oAinMod));
        AinAlsaMod_Cleanup(&(g_aoAinParam[i].oAinAlsaMod));
    }

    for (i = 0; i < MAX_STREAM_COUNT; i++)
    {
        AudioEncMod_Cleanup(&(g_aoAudioStreamParam[i].oAudioEncMod));

        MosMod_Cleanup(&(g_aoStreamParam[i].oInMosMod));
        RotMod_Cleanup(&(g_aoStreamParam[i].oRotModA));
        RotMod_Cleanup(&(g_aoStreamParam[i].oRotModB));
        EncMod_Cleanup(&(g_aoStreamParam[i].oEncMod));
        RtspSrvMod_Cleanup(&(g_aoStreamParam[i].oRtspSrvMod));
        WrFileMod_Cleanup(&(g_aoStreamParam[i].oWrFileMod));
        DecMod_Cleanup(&(g_aoStreamParam[i].oDecMod));
    }

    for (i = 0; i < MAX_VOUT_COUNT; i++)
    {
        MosMod_Stop(&g_aoVoutParam[i].oOutMosMod);

        if (i < MAX_VOUTMTX_COUNT)
        {
            VoutMod_Cleanup(&g_aoVoutParam[i].oVoutMod);
        }
        else
        {
          #ifdef LINUX
            VoutGlxMod_Cleanup(&g_aoVoutParam[i].oVoutGlxMod);
          #endif
        }
    }

    for (i = 0; i < MAX_AOUT_COUNT; i++)
    {
        AoutMod_Cleanup(&g_aoAoutParam[i].oAoutMod);
        AoutAlsaMod_Cleanup(&g_aoAoutParam[i].oAoutAlsaMod);
    }

    if (bRestart)
    {
        printf("***** Restarting *****\n");
        goto Restart;
    }

    Liberatus_UnLoad();

    return LSTATUS_IS_SUCCESS(eStatus) ? 0 : -1;
}
