/************************************************************************************************************\

Module Name:    main.c

Description:    Sample of a video pipeline using Liberatus H264 decoder. The source is a RTSP stream or a
                h264 video file.

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "ReadFileModule.h"
#include "VpeModule.h"
#include "VoutModule.h"
#ifdef LINUX
 #include "VoutGlxModule.h"
#endif
#include "EncoderModule.h"
#include "DecoderModule.h"
#include "WriteFileModule.h"
#include "RtspClientModule.h"
#include "MosaicModule.h"
#include "RotateModule.h"
#include "RtspServerModule.h"
#include "AudioDecoderModule.h"
#include "AoutModule.h"
#include "AoutAlsaModule.h"
#include "CommonUtils.h"

#include <signal.h>

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

#define AVG_PERIOD          60
#define MAX_STREAM_COUNT    16
#define MAX_VOUTMTX_COUNT   32
#ifdef LINUX
  #define MAX_VOUTGLX_COUNT 1
#else
  #define MAX_VOUTGLX_COUNT 0
#endif
#define MAX_VOUT_COUNT      (MAX_VOUTMTX_COUNT + MAX_VOUTGLX_COUNT)
#define MIN_FRAME_SIZE      64
#define MAX_FRAME_SIZE      4096
#define DEMO_COUNT          2

typedef struct tagVoutParam
{
    // Parameters
    MUINT                   uiInputCount;
    MBOOL                   bVoutEnabled;
    MBOOL                   bControlQueueOverflow;
    MUINT                   uiCycleTimeSec;
    MUINT                   uiNextCycleTimeSec;
    MUINT                   uiNextCycleInputIdx;
    LSIZE                   oSize;
    LPixelFormat            ePixFmt;
    LPixelFormat            eOutPixFmt;
    MBOOL                   bOutRgbFullYuv709;
    MBOOL                   bForceMode;
    LSIZE                   oForcedSize;
    MUINT                   uiForcedFps;
    MBOOL                   bNetServer;
    MBOOL                   bSrvAddrDisplayed;
    MUINT                   uiTargetBitRateKbs;
    MUINT                   uiIPeriod;
    MUINT                   uiPPeriod;
    MUINT                   uiQpMin;
    MUINT                   uiQpMax;
    MUINT                   uiSliceCount;
    LH264E_EncoderMode      eEncodeMode;
    LH264_Profile           eEncodeProfile;
    LH264_Level             eEncodeLevel;
    LPixelFormat            eEncodePixFmt;
    MUINT                   uiFaq;
    MBOOL                   bDfEnabled;
    MBOOL                   bIsMasterSync;

    // Modules
    MosaicModule            oMosMod;
    VoutModule              oVoutMod;
   #ifdef LINUX
    VoutGlxModule           oVoutGlxMod;
   #endif
    RotateModule            oRotMod;
    EncoderModule           oEncMod;
    RtspServerModule        oRtspSrvMod;

} VoutParam;

const VoutParam g_oDefaultVoutParam =
{
    /*.uiInputCount             =*/ 0,
    /*.bVoutEnabled             =*/ MFALSE,
    /*.bControlQueueOverflow    =*/ MFALSE,
    /*.uiCycleTimeSec           =*/ 0,
    /*.uiNextCycleTimeSec       =*/ 0,
    /*.uiNextCycleInputIdx      =*/ 0,
    /*.oSize                    =*/ {1920, 1080},
    /*.ePixFmt                  =*/ LPixelFormat_R8G8B8A8,
    /*.eOutPixFmt               =*/ LPixelFormat_R8G8B8A8,
    /*.bOutRgbFullYuv709        =*/ MTRUE,
    /*.bForceMode               =*/ MFALSE,
    /*.oForcedSize              =*/ {1920, 1080},
    /*.uiForcedFps              =*/ 60,
    /*.bNetServer               =*/ MFALSE,
    /*.bSrvAddrDisplayed        =*/ MFALSE,
    /*.uiTargetBitRateKbs       =*/ 10000,
    /*.uiIPeriod                =*/ 90,
    /*.uiPPeriod                =*/ 1,
    /*.uiQpMin                  =*/ 10,
    /*.uiQpMax                  =*/ 42,
    /*.uiSliceCount             =*/ 1,
    /*.eEncodeMode              =*/ LH264E_EncoderMode_RATECONTROLWINDOWED,
    /*.eEncodeProfile           =*/ LH264_Profile_HIGH444,
    /*.eEncodeLevel             =*/ LH264_Level_5_2,
    /*.eEncodePixFmt            =*/ LPixelFormat_MP_Y8_U8V8_420,
    /*.uiFaq                    =*/ 75,
    /*.bDfEnabled               =*/ MTRUE,
    /*.bIsMasterSync            =*/ MTRUE,
    /*.oMosMod                  =*/ MosaicModule_Construct,
    /*.oVoutMod                 =*/ VoutModule_Construct,
   #ifdef LINUX
    /*.oVoutGlxMod              =*/ VoutGlxModule_Construct,
   #endif
    /*.oRotMod                  =*/ RotateModule_Construct,
    /*.oEncMod                  =*/ EncoderModule_Construct,
    /*.oRtspSrvMod              =*/ RtspServerModule_Construct,
};

#define MAX_VOUT_PER_STREAM 4

typedef struct tagStreamParam
{
    MBOOL               bEnable;
    MUINT               uiStrmIdx;

    // Parameters
    const char*         szSrcFilePath;
    const char*         szDstFilePath;
    const char*         szRtspLocation;
    LSIZE               oFrameSize;
    MUINT               uiFrameRateNum;
    MUINT               uiFrameRateDen;
    MUINT               auiVoutIndex[MAX_VOUT_PER_STREAM];
    MUINT               auiMosInputIndex[MAX_VOUT_PER_STREAM];
    MUINT               uiVoutCount;
    MUINT               uiMtu;
    MBOOL               bLoop;
    MBOOL               bDecode;
    MBOOL               bReduceLocalMemUsage;
    MBOOL               bAutoDstRect;
    LRECT32             oDstRect;
    const char*         szNetAddr;
    MUINT               uiNetPort;
    LNetStreamer_Protocol eNetProtocol;
    MBOOL               bSkipToNextIdr;
    MBOOL               bEnableSrt;
    const char*         szNetInterface;
    MBOOL               bEnableIpv6;
    LNetStreamer_SrtMode eSrtMode;

    // Used for demo-B
    MFLOAT32            fAlpha;
    MINT                iAngularSpeed;
    MUINT               uiAngularPos1000x;

    // Modules
    RtspClientModule    oRtspClnMod;
    DecoderModule       oDecodeMod;
    ReadFileModule      oRdFileMod;
    WriteFileModule     oWrFileMod;

    // Source rectangle for the given Vout;
    LRECT32             aoSrcRect[MAX_VOUT_COUNT];

    // Statistics
    MUINT64             uiNextStatTime;
    MUINT               uiSec;
    MUINT               uiMin;
    MUINT               uiHour;
    MUINT               uiOldAvgIdx;
    MUINT               uiNewAvgIdx;
    MUINT               auiPictureCount[AVG_PERIOD];
    MUINT64             auiTotalSizeBytes[AVG_PERIOD];
    MUINT64             auiElapsedTimeUsec[AVG_PERIOD];

} StreamParam;

const StreamParam g_oDefaultStreamParam =
{
    /*.bEnable              =*/ MFALSE,
    /*.uiStreamIdx          =*/ 0,
    /*.szSrcFilePath        =*/ MNULL,
    /*.szDstFilePath        =*/ MNULL,
    /*.szRtspLocation       =*/ MNULL,
    /*.oFrameSize           =*/ {0, 0},
    /*.uiFrameRateNum       =*/ 60,
    /*.uiFrameRateDen       =*/ 1,
    /*.auiVoutIndex         =*/ {0, 0, 0, 0},
    /*.auiMosInputIndex     =*/ {0, 0, 0, 0},
    /*.uiVoutCount          =*/ 1,
    /*.uiMtu                =*/ 0,
    /*.bLoop                =*/ MFALSE,
    /*.bDecode              =*/ MTRUE,
    /*.bReduceLocalMemUsage =*/ MFALSE,
    /*.bAutoDstRect         =*/ MTRUE,
    /*.oDstRect             =*/ {0, 0, 0, 0},
    /*.szNetAddr            =*/ MNULL,
    /*.uiNetPort            =*/ 0,
    /*.eNetProtocol         =*/ LNetStreamer_Protocol_RTSP,
    /*.bSkipToNextIdr       =*/ MFALSE,
    /*.bEnableSrt           =*/ MFALSE,
    /*.szNetInterface       =*/ MNULL,
    /*.bEnableIpv6          =*/ MFALSE,
    /*.eSrtMode             =*/ (LNetStreamer_SrtMode)(0),
    /*.fAlpha               =*/ 1.0f,
    /*.uiAngularSpeed       =*/ 0,
    /*.uiAngularPos1000x    =*/ 0,
    /*.oRtspClnMod          =*/ RtspClientModule_Construct,
    /*.oDecodeMod           =*/ DecoderModule_Construct,
    /*.oRdFileMod           =*/ ReadFileModule_Construct,
    /*.oWrFileMod           =*/ WriteFileModule_Construct,
};

typedef struct tagAudioStreamParam
{
    MBOOL                   bEnable;

    // Parameters
    MBOOL                   bEnableOutput;
    MUINT                   uiAoutIdx;
    MBOOL                   bMtxAout;
    const char*             szRtspLocation;
    const char*             szNetAddr;
    MUINT                   uiNetPort;
    MUINT                   uiMtu;
    MBOOL                   bEnableSrt;
    const char*             szNetInterface;
    MBOOL                   bEnableIpv6;
    LNetStreamer_SrtMode    eSrtMode;
    LNetStreamer_Protocol   eNetProtocol;
    LAudioCodec_SamplingRate eSamplingRate;
    LAudioCodec_ChannelCfg  eChannelCfg;
    LAudioCodec_AACProfile  eProfile;

    // For A/V sync on Liberatus audio output.
    ModuleSyncMaster*       poModSyncMst;

    // Modules
    RtspClientModule*       poRtspClnMod;
    AudioDecModule          oAudioDecMod;
    AoutModule              oAoutMod;
    AoutAlsaModule          oAoutAlsaMod;
    WriteFileModule         oDummySinkMod;

} AudioStreamParam;

const AudioStreamParam g_oDefaultAudioStreamParam =
{
    /*.bEnable          =*/ MFALSE,
    /*.bEnableOutput    =*/ MTRUE,
    /*.uiAoutIdx        =*/ 0,
    /*.bMtxAout         =*/ MTRUE,
    /*.szRtspLocation   =*/ MNULL,
    /*.szNetAddr        =*/ MNULL,
    /*.uiNetPort        =*/ 0,
    /*.uiMtu            =*/ 0,
    /*.bEnableSrt       =*/ MFALSE,
    /*.szNetInterface   =*/ MNULL,
    /*.bEnableIpv6      =*/ MFALSE,
    /*.eSrtMode         =*/ (LNetStreamer_SrtMode)(0),
    /*.eNetProtocol     =*/ LNetStreamer_Protocol_RTSP,
    /*.uiSamplingRate   =*/ LAudioCodec_SamplingRate_UNKNOWN,
    /*.uiNbChannels     =*/ LAudioCodec_ChannelCfg_UNKNOWN,
    /*.eProfile         =*/ LAudioCodec_AACProfile_UNKNOWN,
    /*.poModSyncMst     =*/ MNULL,
    /*.poRtspClnMod     =*/ MNULL,
    /*.oAudioDecMod     =*/ AudioDecModule_Construct,
    /*.oAoutMod         =*/ AoutModule_Construct,
    /*.oAoutAlsaMod     =*/ AoutAlsaModule_Construct,
    /*.oDummySinkMod    =*/ WriteFileModule_Construct
};

static const LSIZE  g_oDemo1BackSize        = { 3840, 2160 };
static const LSIZE  g_oDemo1BorderSize      = { 8, 8 };
static const MUINT  g_uiDemo1CornerFactor   = 4;

static const LSIZE  g_oBigDecodePictureSize = { 2048, 2048 };

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

VoutParam           g_aoVoutParam[MAX_VOUT_COUNT];
StreamParam         g_aoStreamParam[MAX_STREAM_COUNT];
AudioStreamParam    g_aoAudioStreamParam[MAX_STREAM_COUNT];

MBOOL g_bSigTerm = MFALSE;

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Function:       Demo1_IncrementAngularPos

\************************************************************************************************************/
void DemoA_IncrementAngularPos(MosaicModule* poMosMod, MBOOL bEos)
{
    if (bEos)
    {
        return;
    }

    MUINT i;
    for (i = 0; i < MAX_STREAM_COUNT; i++)
    {
        StreamParam* poStreamParam = g_aoStreamParam + i;

        if (poStreamParam->bEnable
            && poStreamParam->bDecode
            && (poStreamParam->iAngularSpeed != 0))
        {
            MosaicInputParam* poParam = poMosMod->aoInParam + poStreamParam->auiMosInputIndex[0];

            MINT iAngleInc = (poStreamParam->iAngularSpeed * (MINT)poMosMod->uiOutFrameRateDen * 1000)
                             / (MINT)poMosMod->uiOutFrameRateNum;

            if (iAngleInc > 0)
            {
                poStreamParam->uiAngularPos1000x += iAngleInc;
            }
            else
            {
                poStreamParam->uiAngularPos1000x += (360*1000 + iAngleInc);
            }

            poStreamParam->uiAngularPos1000x %= (360*1000);

            LRECT32 oMovedRect = poParam->oDstRect.oRect;
            LPOS    oOffset;

            GetRectOnEllipsoidalPath(
                poStreamParam->uiAngularPos1000x,
                &g_oDemo1BackSize,
                &g_oDemo1BorderSize,
                g_uiDemo1CornerFactor,
                &oMovedRect);

            oOffset.iX = oMovedRect.iLeft - poParam->oDstRect.oRect.iLeft;
            oOffset.iY = oMovedRect.iTop - poParam->oDstRect.oRect.iTop;

            MosMod_MoveDstRect(poMosMod, poStreamParam->auiMosInputIndex[0], &oOffset);
        }
    }
}

/************************************************************************************************************\

Function:       GetAudioFormat

\************************************************************************************************************/
LAudioFormat GetAudioFormat(MUINT32 uiSamplingRate, MUINT32 uiSampleSize, MUINT32 uiNbChannels)
{
    LAudioFormat eFormat = LAudioFormat_INVALID;

    return LSTATUS_IS_SUCCESS(LAudioFormat_GetAudioFormat(
                                uiSamplingRate,
                                uiSampleSize,
                                uiNbChannels,
                                1,
                                &eFormat))
            ? eFormat : LAudioFormat_INVALID;
}

/************************************************************************************************************\

Function:       InitStream

\************************************************************************************************************/
LStatus InitStream(
    LDevice_Handle      hDevice,
    StreamParam*        poStreamParam,
    AudioStreamParam*   poAudioStreamParam,
    MUINT               uiDemoIdx,
    MUINT               uiDecBufferCount,
    MUINT               uiMaxMtxAoutCount)
{
    LBuffer_Attributes  oRdFileBufAttrib;
    LStatus             eStatus             = LStatus_OK;
    MBOOL               bReadFile           = (poStreamParam->bEnable && (poStreamParam->szSrcFilePath != MNULL))
                                                ? MTRUE : MFALSE;
    MBOOL               bWriteFile          = (poStreamParam->bEnable && (poStreamParam->szDstFilePath != MNULL))
                                                ? MTRUE : MFALSE;
    MBOOL               bVideoClnEnabled    = (poStreamParam->bEnable && !bReadFile)
                                                ? MTRUE : MFALSE;
    MBOOL               bAudioClnEnabled    = poAudioStreamParam->bEnable;

    poStreamParam->uiNextStatTime           = 0;
    poStreamParam->uiSec                    = 0;
    poStreamParam->uiMin                    = 0;
    poStreamParam->uiHour                   = 0;
    poStreamParam->uiOldAvgIdx              = 0;
    poStreamParam->uiNewAvgIdx              = 0;
    poStreamParam->auiPictureCount[0]       = 0;
    poStreamParam->auiTotalSizeBytes[0]     = 0;
    poStreamParam->auiElapsedTimeUsec[0]    = 0;

    if (bReadFile)
    {
        const MUINT uiBufferCount = MAX_BUFFER_COUNT;

        // Read file module
        oRdFileBufAttrib.eAttributeType                 = LBuffer_Type_SYSTEM_LINEAR;
        oRdFileBufAttrib.oSystemLinearAttributes.uiSize = (((poStreamParam->oFrameSize.iWidth
                                                             * poStreamParam->oFrameSize.iHeight * 2)
                                                            / uiBufferCount) + 0xFFF) & ~0xFFF;
        oRdFileBufAttrib.oSystemLinearAttributes.pvSystemMemory = MNULL;

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = RdFileMod_Init(
                        &(poStreamParam->oRdFileMod),
                        hDevice,
                        &poStreamParam->oFrameSize,
                        uiBufferCount,
                        &oRdFileBufAttrib,
                        poStreamParam->szSrcFilePath,
                        poStreamParam->bLoop,
                        0,
                        0);
        }
    }
    else if(bVideoClnEnabled || bAudioClnEnabled)
    {
        const char* szRtspLocation  = MNULL;
        const char* szNetAddr       = MNULL;
        MUINT       uiVideoPort     = 0;
        MUINT       uiAudioPort     = 0;
        const MUINT uiBasePort      = (15000 + (10 * poStreamParam->uiStrmIdx));
        LNetStreamer_Protocol eProtocol = LNetStreamer_Protocol_RTSP;
        MUINT       uiMtu           = 0;
        MBOOL       bEnableSrt      = MFALSE;
        const char* szNetInterface  = MNULL;
        MBOOL       bEnableIpv6     = MFALSE;
        LNetStreamer_SrtMode eSrtMode = (LNetStreamer_SrtMode)(0);

        if(bVideoClnEnabled)
        {
            szRtspLocation  = poStreamParam->szRtspLocation;
            szNetAddr       = poStreamParam->szNetAddr;
            eProtocol       = poStreamParam->eNetProtocol;
            uiMtu           = poStreamParam->uiMtu;
            bEnableSrt      = poStreamParam->bEnableSrt;
            szNetInterface  = poStreamParam->szNetInterface;
            bEnableIpv6     = poStreamParam->bEnableIpv6;
            eSrtMode        = poStreamParam->eSrtMode;
        }
        else
        {
            szRtspLocation  = poAudioStreamParam->szRtspLocation;
            szNetAddr       = poAudioStreamParam->szNetAddr;
            eProtocol       = poAudioStreamParam->eNetProtocol;
            uiMtu           = poAudioStreamParam->uiMtu;
            bEnableSrt      = poAudioStreamParam->bEnableSrt;
            szNetInterface  = poAudioStreamParam->szNetInterface;
            bEnableIpv6     = poAudioStreamParam->bEnableIpv6;
            eSrtMode        = poAudioStreamParam->eSrtMode;
        }

        if (bVideoClnEnabled)
        {
            if (poStreamParam->uiNetPort != 0)
            {
                uiVideoPort = poStreamParam->uiNetPort;
            }
            else
            {
                uiVideoPort = uiBasePort;
            }
        }

        if (bAudioClnEnabled)
        {
            if (poAudioStreamParam->uiNetPort != 0)
            {
                uiAudioPort = poAudioStreamParam->uiNetPort;
            }
            else
            {
                uiAudioPort = (uiVideoPort != 0)
                               ? (uiVideoPort + 4)
                               : (uiBasePort + 4);
            }
        }

        MUINT uiMaxNaluSizeBytes  = (poStreamParam->oFrameSize.iWidth
                                     * poStreamParam->oFrameSize.iHeight * 3) / 2;

        eStatus = RtspClnMod_Init(
                    &(poStreamParam->oRtspClnMod),
                    hDevice,
                    bVideoClnEnabled,
                    bAudioClnEnabled,
                    16,
                    uiMaxNaluSizeBytes,
                    16,
                    8*1024,
                    eProtocol,
                    szRtspLocation,
                    szNetAddr,
                    uiVideoPort,
                    uiAudioPort,
                    poStreamParam->bSkipToNextIdr,
                    GetSamplingRate(poAudioStreamParam->eSamplingRate),
                    uiMtu,
                    bEnableSrt,
                    szNetInterface,
                    bEnableIpv6,
                    eSrtMode);

        if (LSTATUS_IS_FAIL(eStatus))
        {
            eStatus = LStatus_CONNECTION_LOST;
        }
    }
    else
    {
        eStatus = LStatus_INVALID_PARAM;
    }

    if (bWriteFile && LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = WrFileMod_Init(&poStreamParam->oWrFileMod, hDevice, poStreamParam->szDstFilePath, MFALSE);
    }

    if (poStreamParam->bEnable && poStreamParam->bDecode && LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = DecMod_Init(
                        &poStreamParam->oDecodeMod,
                        hDevice,
                        uiDecBufferCount,
                        poStreamParam->uiFrameRateNum,
                        poStreamParam->uiFrameRateDen,
                        bReadFile,
                        poStreamParam->bReduceLocalMemUsage);
    }

    if(bAudioClnEnabled && LSTATUS_IS_SUCCESS(eStatus))
    {
        LBuffer_Type eOutBufType = (poAudioStreamParam->bEnableOutput && poAudioStreamParam->bMtxAout)
                                     ? (LBuffer_Type_LINEAR) : (LBuffer_Type_SYSTEM_LINEAR);

        if (poStreamParam->oRtspClnMod.oAudioCfg.bValid)
        {
            eStatus = AudioDecMod_InitWithConfig(
                          &(poAudioStreamParam->oAudioDecMod),
                          hDevice,
                          32,
                          eOutBufType,
                          poStreamParam->oRtspClnMod.oAudioCfg.uiCfg,
                          MTRUE);
        }
        else
        {
            eStatus = AudioDecMod_Init(
                          &(poAudioStreamParam->oAudioDecMod),
                          hDevice,
                          32,
                          eOutBufType,
                          poAudioStreamParam->eSamplingRate,
                          poAudioStreamParam->eChannelCfg,
                          poAudioStreamParam->eProfile,
                          MTRUE);
        }

        // Retrieve audio format from audio decoder parameters.
        LAudioFormat eAudioFormat = LAudioFormat_INVALID;
        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            eAudioFormat = GetAudioFormat(
                                poAudioStreamParam->oAudioDecMod.uiSamplingRate,
                                poAudioStreamParam->oAudioDecMod.uiSampleSize,
                                poAudioStreamParam->oAudioDecMod.uiNbChannels);

            if(eAudioFormat == LAudioFormat_INVALID)
            {
                eStatus = LStatus_FAIL;
            }
        }

        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            if (poAudioStreamParam->bEnableOutput)
            {
                if(poAudioStreamParam->bMtxAout)
                {
                    eStatus = AoutMod_Init(
                                  &(poAudioStreamParam->oAoutMod),
                                  hDevice,
                                  poAudioStreamParam->uiAoutIdx,
                                  eAudioFormat,
                                  1024,
                                  poAudioStreamParam->poModSyncMst,
                                  MTRUE,
                                  MTRUE);
                }
                else
                {
                    eStatus = AoutAlsaMod_Init(
                                  &(poAudioStreamParam->oAoutAlsaMod),
                                  poAudioStreamParam->uiAoutIdx - uiMaxMtxAoutCount,
                                  poAudioStreamParam->oAudioDecMod.uiSamplingRate,
                                  poAudioStreamParam->oAudioDecMod.uiNbChannels,
                                  poAudioStreamParam->oAudioDecMod.uiSampleSize,
                                  MFALSE,
                                  MFALSE,
                                  1024);
                }
            }
            else
            {
                eStatus = WrFileMod_Init(
                              &(poAudioStreamParam->oDummySinkMod),
                              hDevice,
                              "/dev/null",
                              MTRUE);
            }
        }
    }

    if(poStreamParam->bEnable && LSTATUS_IS_SUCCESS(eStatus))
    {
        //--------------------------------------------------------------------------------------------------------
        // Connect video modules

        // Video pipeline configuration:
        //
        //  RtspClient---+--->Decode--->Mosaic--->Vout
        //  (or RdFile)  |
        //              (+--->WrFile) >> Optional with "-o" or "-oo" argument
        char szSrcName[100];

        ModuleLink* poSrcLink = MNULL;

        if (bReadFile)
        {
            snprintf(szSrcName, sizeof(szSrcName), "[RdFile: \"%s\"]-", poStreamParam->szSrcFilePath);
            poSrcLink = &(poStreamParam->oRdFileMod.oOutLink);
        }
        else
        {
            snprintf(szSrcName, sizeof(szSrcName), "[RtspClient: \"%s\"]-", poStreamParam->szRtspLocation);
            poSrcLink = &(poStreamParam->oRtspClnMod.oVideoOutLink);
        }

        if (poStreamParam->bDecode && LSTATUS_IS_SUCCESS(eStatus))
        {
            char szStreamName[256];
            char szBranch[256];

            // TODO: Show all Vout.
            VoutParam* poVoutParam = g_aoVoutParam + poStreamParam->auiVoutIndex[0];

            if (bWriteFile)
            {
                strncat_wz(szSrcName, "-+", sizeof(szSrcName));
            }

            snprintf(szStreamName, sizeof(szStreamName), "%s-->[Decode]--->%u:[Mosaic]-", szSrcName,
                     poStreamParam->auiMosInputIndex[0]);

            if (poVoutParam->bNetServer && poVoutParam->bVoutEnabled)
            {
                strncat_wz(szStreamName, "-+", sizeof(szStreamName));
                int iSpaces = (int)(strlen(szStreamName) - 1);
                snprintf(szBranch, sizeof(szBranch),
                         "%*s|\n"
                         "%*s+", iSpaces, " ", iSpaces, " ");
            }

            if (poVoutParam->bVoutEnabled)
            {
                printf("%s-->[Vout%u]\n", szStreamName, poStreamParam->auiVoutIndex[0]);
            }
            else
            {
                strncpy_wz(szBranch, szStreamName, sizeof(szBranch));
            }

            if (poVoutParam->bNetServer)
            {
                printf("%s-->[Rotate]--->[Encode]--->[RtspSrv]\n", szBranch);
            }

            eStatus = ModLnk_Connect(poSrcLink, &(poStreamParam->oDecodeMod.oInLink));

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                MUINT i;
                for (i = 0; i < poStreamParam->uiVoutCount; i++)
                {
                    poVoutParam = g_aoVoutParam + poStreamParam->auiVoutIndex[i];

                    MosaicModule*    poMosMod    = &(poVoutParam->oMosMod);
                    ModuleLinkInput* poVoutInput = &(poVoutParam->oVoutMod.oInLink);
                  #ifdef LINUX
                                     poVoutInput = (poStreamParam->auiVoutIndex[i] < MAX_VOUTMTX_COUNT)
                                                   ? &(poVoutParam->oVoutMod.oInLink)
                                                   : &(poVoutParam->oVoutGlxMod.oInLink);
                  #endif

                    if (uiDemoIdx == 0)
                    {
                        MUINT j;
                        for (j = 0; (j < poVoutParam->uiInputCount) && LSTATUS_IS_SUCCESS(eStatus); j++)
                        {
                            eStatus = ModLnk_Connect(&(poStreamParam->oDecodeMod.oOutLink),
                                                     poMosMod->aoInLink + poStreamParam->auiMosInputIndex[i] + j);
                        }
                    }
                    else
                    {
                        eStatus = ModLnk_Connect(&(poStreamParam->oDecodeMod.oOutLink),
                                                 poMosMod->aoInLink + poStreamParam->auiMosInputIndex[i]);
                    }

                    if (LSTATUS_IS_SUCCESS(eStatus) && poVoutParam->bVoutEnabled)
                    {
                        eStatus = ModLnk_Connect(&(poMosMod->oOutLink), poVoutInput);
                    }

                    if (LSTATUS_IS_SUCCESS(eStatus))
                    {
                        LRECT32  aoDstRect[4];
                        MUINT    uiDstRectCount = 0;
                        LRECT32  oSrcRect;
                        LRECT32* poSrcRect = &(poStreamParam->aoSrcRect[poStreamParam->auiVoutIndex[i]]);

                        if ((poSrcRect->iLeft >= poSrcRect->iRight)
                            || (poSrcRect->iTop >= poSrcRect->iBottom))
                        {
                            oSrcRect.iLeft   = 0;
                            oSrcRect.iTop    = 0;
                            oSrcRect.iRight  = poStreamParam->oFrameSize.iWidth;
                            oSrcRect.iBottom = poStreamParam->oFrameSize.iHeight;

                            poSrcRect = &oSrcRect;
                        }

                        if (uiDemoIdx == 0)
                        {
                            aoDstRect[0].iLeft      = 0;
                            aoDstRect[0].iTop       = 0;
                            aoDstRect[0].iRight     = 3840;
                            aoDstRect[0].iBottom    = 2160;
                            aoDstRect[1]            = aoDstRect[0];
                            aoDstRect[1].iLeft      = aoDstRect[0].iRight  - 1920;
                            aoDstRect[1].iTop       = aoDstRect[0].iBottom - 1080;
                            aoDstRect[2]            = aoDstRect[0];
                            aoDstRect[2].iLeft      = aoDstRect[0].iRight  - 1280;
                            aoDstRect[2].iTop       = aoDstRect[0].iBottom - 720;
                            aoDstRect[3]            = aoDstRect[0];
                            aoDstRect[3].iLeft      = aoDstRect[0].iRight  - 854;
                            aoDstRect[3].iTop       = aoDstRect[0].iBottom - 480;

                            uiDstRectCount  = poVoutParam->uiInputCount;
                        }
                        else
                        {
                            aoDstRect[0]    = poStreamParam->oDstRect;
                            uiDstRectCount  = 1;
                        }

                        MUINT j;
                        for (j = 0; (j < uiDstRectCount) && LSTATUS_IS_SUCCESS(eStatus); j++)
                        {
                            eStatus = MosMod_SetParameters(
                                        poMosMod,
                                        poStreamParam->auiMosInputIndex[i] + j,
                                        &poStreamParam->oFrameSize,
                                        poSrcRect,
                                        &aoDstRect[j],
                                        poStreamParam->uiFrameRateNum,
                                        poStreamParam->uiFrameRateDen,
                                        poStreamParam->auiMosInputIndex[i] + j,
                                        !bReadFile,
                                        poStreamParam->fAlpha);
                        }

                        if (LSTATUS_IS_SUCCESS(eStatus) && poVoutParam->bVoutEnabled)
                        {
                            MUINT uiFpsNum;
                            MUINT uiFpsDen;

                            eStatus = MosMod_GetOutputFrameRate(poMosMod, &uiFpsNum, &uiFpsDen);

                            if (LSTATUS_IS_SUCCESS(eStatus))
                            {
                                if (poStreamParam->auiVoutIndex[i] < MAX_VOUTMTX_COUNT)
                                {
                                    VoutMod_SetInputFrameRate(&poVoutParam->oVoutMod, uiFpsNum, uiFpsDen);
                                }
                                else
                                {
                                   #ifdef LINUX
                                    VoutGlxMod_SetInputFrameRate(&poVoutParam->oVoutGlxMod, uiFpsNum, uiFpsDen);
                                   #else
                                    assert(MFALSE);
                                   #endif
                                }
                            }
                        }
                    }
                }
            }
        }

        if (bWriteFile && LSTATUS_IS_SUCCESS(eStatus))
        {
            if (poStreamParam->bDecode)
            {
                int iSpaces = (int)(strlen(szSrcName) - 1);

                printf("%*s|\n"
                       "%*s+", iSpaces, " ", iSpaces, " ");
            }
            else
            {
                printf("%s", szSrcName);
            }

            printf("-->[WrFile: \"%s\"]\n", poStreamParam->szDstFilePath);

            eStatus = ModLnk_Connect(poSrcLink, &(poStreamParam->oWrFileMod.oInLink));
        }
    }

    //--------------------------------------------------------------------------------------------------------
    // Connect audio modules

    // Audio pipeline configuration:
    //
    //  RtspClient----->AudioDecode--->Aout(Liberatus|ALSA)
    //                                 OR
    //                                 Dummy WrFile.
    if(bAudioClnEnabled && LSTATUS_IS_SUCCESS(eStatus))
    {
        char szAudioStream[256];

        snprintf(szAudioStream,
                 sizeof(szAudioStream),
                 "[RtspClient: \"%s\"]--->[AudioDecode]-",
                 poAudioStreamParam->szRtspLocation);

        eStatus = ModLnk_Connect(
                      &(poAudioStreamParam->poRtspClnMod->oAudioOutLink),
                      &(poAudioStreamParam->oAudioDecMod.oInLink));

        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            char szOutput[256];
            ModuleLinkInput* poOutputInLnk = MNULL;

            if (poAudioStreamParam->bEnableOutput)
            {
                if(poAudioStreamParam->bMtxAout)
                {
                    snprintf(szOutput,
                             sizeof(szOutput),
                             "-->[Liberatus Aout: %u]\n",
                             poAudioStreamParam->uiAoutIdx);

                    poOutputInLnk = &(poAudioStreamParam->oAoutMod.oInLink);
                }
                else
                {
                    LSTR64 szDevName = {0};
                    AoutAlsaMod_GetDeviceName(
                                    poAudioStreamParam->uiAoutIdx - uiMaxMtxAoutCount,
                                    &szDevName);

                    snprintf(szOutput,
                             sizeof(szOutput),
                             "-->[ALSA Aout: %s]\n",
                             szDevName);

                    poOutputInLnk = &(poAudioStreamParam->oAoutAlsaMod.oInLink);
                }
            }
            else
            {
                snprintf(szOutput,
                         sizeof(szOutput),
                         "-->[Dummy sink]\n");

                poOutputInLnk = &(poAudioStreamParam->oDummySinkMod.oInLink);
            }

            strncat_wz(szAudioStream, szOutput, sizeof(szAudioStream));

            printf("%s\n", szAudioStream);

            eStatus = ModLnk_Connect(
                          &(poAudioStreamParam->oAudioDecMod.oOutLink),
                          poOutputInLnk);
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       InitAVSyncParams

\************************************************************************************************************/
void InitAVSyncParams()
{
    MUINT i;

    // Synchronize audio and video streams if they come from the same session and use the same output.
    for(i = 0; i < MAX_STREAM_COUNT; i++)
    {
        StreamParam*        poStreamParam       = &g_aoStreamParam[i];
        AudioStreamParam*   poAudioStreamParam  = &g_aoAudioStreamParam[i];

        if(poStreamParam->bEnable
           && (poAudioStreamParam->bEnable)
           && (poAudioStreamParam->bEnableOutput)
           && (poAudioStreamParam->bMtxAout))
        {
            MUINT j;
            for(j = 0; j < poStreamParam->uiVoutCount; j++)
            {
                MUINT uiVoutIdx = poStreamParam->auiVoutIndex[j];
                VoutParam* poVoutParam = g_aoVoutParam + uiVoutIdx;

                if(poAudioStreamParam->uiAoutIdx == uiVoutIdx)
                {
                    printf("A/V streams from network stream %u synchronized on output %u\n", i, uiVoutIdx);
                    poVoutParam->bIsMasterSync = MFALSE;
                    poAudioStreamParam->poModSyncMst = &(poVoutParam->oVoutMod.oModSync.oModSyncMst);
                }
            }
        }
    }
}

/************************************************************************************************************\

Function:       DoStats

\************************************************************************************************************/
MBOOL DoStats(
    StreamParam*    poStreamParam,
    MUINT           uiStreamIdx,
    MBOOL           bShow,
    MUINT64*        puiPrevBusyTimeCpu,
    MUINT64*        puiPrevTotalTimeCpu)
{
    const MUINT uiOneSec = 1000 * 1000;
    MBOOL       bShown   = MFALSE;

    if (poStreamParam->oDecodeMod.uiElapsedTimeUsec >= uiOneSec)
    {
        if (poStreamParam->uiNextStatTime == 0)
        {
            poStreamParam->uiNextStatTime = GetMonoTimeUsec();
        }

        if (GetMonoTimeUsec() >= poStreamParam->uiNextStatTime)
        {
            MUINT   uiPictureCountDiff      = poStreamParam->oDecodeMod.uiPictureCount;
            MUINT64 uiTotalSizeBytesDiff    = poStreamParam->oDecodeMod.uiTotalSizeBytes;
            MUINT64 uiElapsedTimeUsecDiff   = poStreamParam->oDecodeMod.uiElapsedTimeUsec;

            uiPictureCountDiff      -= poStreamParam->auiPictureCount[poStreamParam->uiOldAvgIdx];
            uiTotalSizeBytesDiff    -= poStreamParam->auiTotalSizeBytes[poStreamParam->uiOldAvgIdx];
            uiElapsedTimeUsecDiff   -= poStreamParam->auiElapsedTimeUsec[poStreamParam->uiOldAvgIdx];

            poStreamParam->uiNewAvgIdx++;

            if (poStreamParam->uiNewAvgIdx == AVG_PERIOD)
            {
                poStreamParam->uiNewAvgIdx = 0;
            }

            if (poStreamParam->uiOldAvgIdx == poStreamParam->uiNewAvgIdx)
            {
                poStreamParam->uiOldAvgIdx++;

                if (poStreamParam->uiOldAvgIdx == AVG_PERIOD)
                {
                    poStreamParam->uiOldAvgIdx = 0;
                }
            }

            poStreamParam->auiPictureCount[poStreamParam->uiNewAvgIdx]
                = poStreamParam->oDecodeMod.uiPictureCount;
            poStreamParam->auiTotalSizeBytes[poStreamParam->uiNewAvgIdx]
                = poStreamParam->oDecodeMod.uiTotalSizeBytes;
            poStreamParam->auiElapsedTimeUsec[poStreamParam->uiNewAvgIdx]
                = poStreamParam->oDecodeMod.uiElapsedTimeUsec;

            MFLOAT32 fAvgFps    = 0;
            MFLOAT32 fAvgMbps   = 0;

            if (uiElapsedTimeUsecDiff > 0)
            {
                fAvgFps     = ((MFLOAT32)uiPictureCountDiff * 1000 * 1000)
                              / uiElapsedTimeUsecDiff;
                fAvgMbps    = ((MFLOAT32)uiTotalSizeBytesDiff * 8)
                              / uiElapsedTimeUsecDiff;
            }

            if (++poStreamParam->uiSec == 60)
            {
                poStreamParam->uiSec = 0;

                if (++poStreamParam->uiMin == 60)
                {
                    poStreamParam->uiMin = 0;
                    poStreamParam->uiHour++;
                }
            }

            if (bShow)
            {
                MUINT uiCpuUsage10x = GetCpuUsage(puiPrevBusyTimeCpu, puiPrevTotalTimeCpu);
                MUINT uiMemUsage10x = GetMemUsage();

                printf("[%u:%02u:%02u] Dec[%u]: %lu byte, %u NAL-blob, %u frm, %.02f fps, %.02f Mbps"
                       " | Cpu: %.01f%% | Mem: %.01f%%\n",
                       poStreamParam->uiHour,
                       poStreamParam->uiMin,
                       poStreamParam->uiSec,
                       uiStreamIdx,
                       poStreamParam->oDecodeMod.uiTotalSizeBytes,
                       (MUINT32)poStreamParam->oDecodeMod.uiAccessUnitId,
                       poStreamParam->oDecodeMod.uiPictureCount,
                       fAvgFps,
                       fAvgMbps,
                       (MFLOAT32)uiCpuUsage10x / 10,
                       (MFLOAT32)uiMemUsage10x / 10);

                bShown = MTRUE;
            }

            poStreamParam->uiNextStatTime += uiOneSec;
        }
    }

    return bShown;
}

/************************************************************************************************************\

Function:       SetStreamDstRect

\************************************************************************************************************/
void SetStreamDstRect(
    MUINT  uiVoutIndex,
    LSIZE* poSize)
{
    LPOS32  oPos = {0, 0};
    MUINT   uiAutoInputCount = 0;
    MUINT   i;
    MUINT   j;

    for (i = 0; i < MAX_STREAM_COUNT; i++)
    {
        if (g_aoStreamParam[i].bEnable
            && g_aoStreamParam[i].bAutoDstRect)
        {
            for (j = 0; j < g_aoStreamParam[i].uiVoutCount; j++)
            {
                if (g_aoStreamParam[i].auiVoutIndex[j] == uiVoutIndex)
                {
                    uiAutoInputCount++;
                }
            }
        }
    }

    const MUINT uiDiv = (uiAutoInputCount <  2) ? 1 :
                        (uiAutoInputCount <  5) ? 2 :
                        (uiAutoInputCount < 10) ? 3 : 4;

    for (i = 0; i < MAX_STREAM_COUNT; i++)
    {
        if (g_aoStreamParam[i].bEnable
            && g_aoStreamParam[i].bAutoDstRect)
        {
            for (j = 0; j < g_aoStreamParam[i].uiVoutCount; j++)
            {
                if (g_aoStreamParam[i].auiVoutIndex[j] == uiVoutIndex)
                {
                    g_aoStreamParam[i].oDstRect.iLeft   = (oPos.iX * poSize->iWidth)  / uiDiv;
                    g_aoStreamParam[i].oDstRect.iTop    = (oPos.iY * poSize->iHeight) / uiDiv;
                    g_aoStreamParam[i].oDstRect.iRight  = ((oPos.iX + 1) * poSize->iWidth) / uiDiv;
                    g_aoStreamParam[i].oDstRect.iBottom = ((oPos.iY + 1) * poSize->iHeight) / uiDiv;

                    // Do not scale up
                    if (((g_aoStreamParam[i].oDstRect.iRight - g_aoStreamParam[i].oDstRect.iLeft)
                         > g_aoStreamParam[i].oFrameSize.iWidth)
                        && ((g_aoStreamParam[i].oDstRect.iBottom - g_aoStreamParam[i].oDstRect.iTop)
                            > g_aoStreamParam[i].oFrameSize.iHeight))
                    {
                        g_aoStreamParam[i].oDstRect.iRight  = g_aoStreamParam[i].oDstRect.iLeft
                                                              + g_aoStreamParam[i].oFrameSize.iWidth;
                        g_aoStreamParam[i].oDstRect.iBottom = g_aoStreamParam[i].oDstRect.iTop
                                                              + g_aoStreamParam[i].oFrameSize.iHeight;
                    }

                    oPos.iX++;

                    if (oPos.iX == uiDiv)
                    {
                        oPos.iX = 0;
                        oPos.iY++;
                    }
                }
            }
        }
    }
}

/************************************************************************************************************\

Function:       ShowInputKeyOptions

\************************************************************************************************************/
void ShowInputKeyOptions()
{
    printf("Input key options:\n"
           "  'q'       : Quit.\n"
           "  'r'       : Restart.\n"
           "  'n'       : Cycle to next full screen video.\n"
           "  '0-9,a-f' : Show stream stats.\n"
           "  'P'       : Pause/Resume all streams.\n");
}

/************************************************************************************************************\

Function:       SignalHandler

\************************************************************************************************************/
void SignalHandler(int iSigNum)
{
    printf("Received signal: %d.\n", iSigNum);

    if (iSigNum == SIGTERM)
    {
        g_bSigTerm = MTRUE;
    }
}

/************************************************************************************************************\

Function:       main

\************************************************************************************************************/
int main(int iArgc, char **argv)
{
    ModThread_SetName("Main");

    // This is a patch to avoid process killing when a RTSP connection is lost.
   #ifdef LINUX
    signal(SIGPIPE, SignalHandler);
   #endif

    // Do normal stop on kill.
    signal(SIGTERM, SignalHandler);

    // -------------------------------------------------------------------------------------------------------
    // Parse arguments

    const MUINT uiDefBufferCount = 16;
    const MUINT uiMinBufferCount = 3;
    const MUINT uiMaxBufferCount = 24;

    StreamParam*        poStreamParam           = MNULL;
    StreamParam*        poPrevStreamParam       = MNULL;
    AudioStreamParam*   poAudioStreamParam      = MNULL;
    AudioStreamParam*   poPrevAudioStreamParam  = MNULL;
    MUINT               uiMsgLogLevel           = 0;
    MBOOL               bEnableVcLog            = MFALSE;
    MBOOL               bShowStats              = MFALSE;
    LRECT               oWinRect                = { 0, 0, 0, 0 };
    MBOOL               bForceWinRect           = MFALSE;
    MBOOL               bArgError               = MFALSE;
    MUINT               uiDemoIdx               = ~0;
    MINT                iDemoACount             = 0;
    MINT                iAngularSpeed           = 0;
    MUINT               uiTransparency          = 100;
    MUINT               uiCurVoutIdx            = ~0;
    MUINT32             uiMaxVoutMtxCount       = MAX_VOUTMTX_COUNT;
    MUINT32             uiMaxAoutCount          = 0;
    MUINT32             uiMaxMtxAoutCount       = 0;
    MUINT               uiBigDecodeCount        = 0;
    MUINT               uiBufferCount           = uiDefBufferCount;
    MBOOL               bAvSync                 = MFALSE;
    MINT                i;

    // -------------------------------------------------------------------------------------------------------
    // Load library and init device

    Liberatus_Load();

    LDevice_Handle hDevice = Liberatus_GetDevice(0);
    LStatus        eStatus = (hDevice != MNULL) ? LStatus_OK : LStatus_FAIL;

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LVideoOut_GetCount(hDevice, &uiMaxVoutMtxCount);

        if (LSTATUS_IS_FAIL(eStatus))
        {
            printf(
                "ERROR! LVideoOut_GetCount returned %d (%s).\n",
                eStatus,
                GetStatusStr(eStatus));
        }
    }
    else
    {
        printf("ERROR! Liberatus_GetDevice returned NULL handle.\n");
    }

    uiMaxVoutMtxCount = min(uiMaxVoutMtxCount, MAX_VOUTMTX_COUNT);


    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LAudioOut_GetCount(hDevice, &uiMaxAoutCount);

        if (LSTATUS_IS_FAIL(eStatus))
        {
            printf("ERROR! LAudioOut_GetCount returned %d (%s).\n", eStatus, GetStatusStr(eStatus));
        }
    }

    uiMaxMtxAoutCount = uiMaxAoutCount;

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        MUINT32 uiAlsaDevCount = 0;

        if(AoutAlsaMod_EnumDevices(&uiAlsaDevCount) == LStatus_OK)
        {
            uiMaxAoutCount += uiAlsaDevCount;
        }
    }

    // -------------------------------------------------------------------------------------------------------
    // Parse arguments

    for (i = 0; i < MAX_VOUT_COUNT; i++)
    {
        g_aoVoutParam[i] = g_oDefaultVoutParam;
    }

    for (i = 0; i < MAX_STREAM_COUNT; i++)
    {
        g_aoStreamParam[i] = g_oDefaultStreamParam;
        g_aoStreamParam[i].uiStrmIdx = i;

        MUINT j;
        for (j = 0; j < MAX_VOUT_COUNT; j++)
        {
            g_aoStreamParam[i].aoSrcRect[j].iLeft   = 0;
            g_aoStreamParam[i].aoSrcRect[j].iTop    = 0;
            g_aoStreamParam[i].aoSrcRect[j].iRight  = 0;
            g_aoStreamParam[i].aoSrcRect[j].iBottom = 0;
        }

        g_aoAudioStreamParam[i]                 = g_oDefaultAudioStreamParam;
        g_aoAudioStreamParam[i].poRtspClnMod    = &(g_aoStreamParam[i].oRtspClnMod);
    }

    for (i = 1; i < iArgc; i++)
    {
        if (strcmp(argv[i], "-s") == 0)
        {
            if (++i < iArgc)
            {
                MUINT uiStrmIdx = atoi(argv[i]);
                if (uiStrmIdx < MAX_STREAM_COUNT)
                {
                    if (poPrevStreamParam != MNULL)
                    {
                        g_aoStreamParam[uiStrmIdx] = *poPrevStreamParam;
                    }

                    poPrevStreamParam            = &(g_aoStreamParam[uiStrmIdx]);
                    poPrevStreamParam->bEnable   = MTRUE;
                    poPrevStreamParam->uiStrmIdx = uiStrmIdx;

                    poStreamParam = poPrevStreamParam;
                    poAudioStreamParam = MNULL;
                    continue;
                }
            }
        }
        else if(strcmp(argv[i], "-as") == 0)
        {
            if(++i < iArgc)
            {
                MUINT uiStrmIdx = atoi(argv[i]);
                if (uiStrmIdx < MAX_STREAM_COUNT)
                {
                    if (poPrevAudioStreamParam != MNULL)
                    {
                        RtspClientModule* poRtspClnMod = g_aoAudioStreamParam[uiStrmIdx].poRtspClnMod;
                        g_aoAudioStreamParam[uiStrmIdx] = *poPrevAudioStreamParam;
                        g_aoAudioStreamParam[uiStrmIdx].poRtspClnMod = poRtspClnMod;
                    }

                    poPrevAudioStreamParam            = &(g_aoAudioStreamParam[uiStrmIdx]);
                    poPrevAudioStreamParam->bEnable   = MTRUE;

                    poAudioStreamParam = poPrevAudioStreamParam;
                    poStreamParam = MNULL;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-dbg") == 0)
        {
            if (++i < iArgc)
            {
                uiMsgLogLevel = atoi(argv[i]);
                continue;
            }
        }
        else if (strcmp(argv[i], "-vclog") == 0)
        {
            bEnableVcLog = MTRUE;
            continue;
        }
        else if (strcmp(argv[i], "-stat") == 0)
        {
            bShowStats = MTRUE;
            continue;
        }
        else if (strcmp(argv[i], "-fsc") == 0)
        {
            if (++i < iArgc)
            {
                MUINT uiIdx;

                if (argv[i][0] == 'x')
                {
                    uiIdx = atoi(argv[i] + 1) + MAX_VOUTMTX_COUNT;
                }
                else
                {
                    uiIdx = atoi(argv[i]);
                }

                if ((uiIdx < MAX_VOUT_COUNT)
                    && ((uiIdx < uiMaxVoutMtxCount)
                        || (uiIdx >= MAX_VOUTMTX_COUNT)))
                {
                    if (++i < iArgc)
                    {
                        g_aoVoutParam[uiIdx].uiCycleTimeSec = atoi(argv[i]);
                        continue;
                    }
                }
            }
        }
        else if (strcmp(argv[i], "-dpf") == 0)
        {
            if (++i < iArgc)
            {
                MUINT uiIdx    = 0;
                MUINT uiEndIdx = uiMaxVoutMtxCount;

                if (strcmp(argv[i], "a") != 0)
                {
                    uiIdx    = atoi(argv[i]);
                    uiEndIdx = uiIdx + 1;
                }

                if ((uiIdx < uiMaxVoutMtxCount) && (++i < iArgc))
                {
                    LPixelFormat ePixFmt =
                        (strcmp(argv[i], "RGB-8")       == 0) ? LPixelFormat_R8G8B8A8               :
                        (strcmp(argv[i], "RGB-10")      == 0) ? LPixelFormat_R10G10B10A2            :
                        (strcmp(argv[i], "444-10")      == 0) ? LPixelFormat_U82Y82V82A2            :
                        (strcmp(argv[i], "YUV")         == 0) ? LPixelFormat_V8U8Y8A8               :
                        (strcmp(argv[i], "YUY2")        == 0) ? LPixelFormat_Y8U8Y8V8               :
                        (strcmp(argv[i], "420-8-2P")    == 0) ? LPixelFormat_MP_Y8_U8V8_420         :
                        (strcmp(argv[i], "422-8-2P")    == 0) ? LPixelFormat_MP_Y8_U8V8_422         :
                        (strcmp(argv[i], "444-8-2P")    == 0) ? LPixelFormat_MP_Y8_U8V8_444         :
                        (strcmp(argv[i], "420-8-3P")    == 0) ? LPixelFormat_MP_Y8_U8_V8_420        :
                        (strcmp(argv[i], "422-8-3P")    == 0) ? LPixelFormat_MP_Y8_U8_V8_422        :
                        (strcmp(argv[i], "444-8-3P")    == 0) ? LPixelFormat_MP_Y8_U8_V8_444        :
                        (strcmp(argv[i], "RGB-8-3P")    == 0) ? LPixelFormat_MP_G8_R8_B8            :
                        (strcmp(argv[i], "RGB-10-3P")   == 0) ? LPixelFormat_MP_G10_R10_B10         :
                        (strcmp(argv[i], "420-10-2P")   == 0) ? LPixelFormat_MP_Y82_U82V82_X2_420   :
                        (strcmp(argv[i], "422-10-2P")   == 0) ? LPixelFormat_MP_Y82_U82V82_X2_422   :
                        (strcmp(argv[i], "444-10-2P")   == 0) ? LPixelFormat_MP_Y82_U82V82_X2_444   :
                        (strcmp(argv[i], "420-10-3P")   == 0) ? LPixelFormat_MP_Y82_U82_V82_X2_420  :
                        (strcmp(argv[i], "422-10-3P")   == 0) ? LPixelFormat_MP_Y82_U82_V82_X2_422  :
                        (strcmp(argv[i], "444-10-3P")   == 0) ? LPixelFormat_MP_Y82_U82_V82_X2_444  :

                        // Backward compatibility.
                        (strcmp(argv[i], "MP-RGB-8")    == 0) ? LPixelFormat_MP_G8_R8_B8            : 
                        (strcmp(argv[i], "420-8")       == 0) ? LPixelFormat_MP_Y8_U8V8_420         :
                        (strcmp(argv[i], "422-8")       == 0) ? LPixelFormat_MP_Y8_U8V8_422         :
                        (strcmp(argv[i], "444-8")       == 0) ? LPixelFormat_MP_Y8_U8V8_444         :
                                                                LPixelFormat_INVALID;

                    if (ePixFmt != LPixelFormat_INVALID)
                    {
                        for (; uiIdx < uiEndIdx; uiIdx++)
                        {
                            g_aoVoutParam[uiIdx].ePixFmt = ePixFmt;
                        }

                        continue;
                    }
                }
            }
        }
        else if (strcmp(argv[i], "-opf") == 0)
        {
            if (++i < iArgc)
            {
                MUINT uiIdx    = 0;
                MUINT uiEndIdx = uiMaxVoutMtxCount;

                if (strcmp(argv[i], "a") != 0)
                {
                    uiIdx    = atoi(argv[i]);
                    uiEndIdx = uiIdx + 1;
                }

                if ((uiIdx < uiMaxVoutMtxCount) && (++i < iArgc))
                {
                    LPixelFormat ePixFmt =
                        (strcmp(argv[i], "RGB-8")   == 0) ? LPixelFormat_R8G8B8A8               :
                        (strcmp(argv[i], "RGB-10")  == 0) ? LPixelFormat_R10G10B10A2            :
                        (strcmp(argv[i], "420-8")   == 0) ? LPixelFormat_MP_Y8_U8V8_420         :
                        (strcmp(argv[i], "422-8")   == 0) ? LPixelFormat_MP_Y8_U8V8_422         :
                        (strcmp(argv[i], "444-8")   == 0) ? LPixelFormat_MP_Y8_U8V8_444         :
                        (strcmp(argv[i], "420-10")  == 0) ? LPixelFormat_MP_Y82_U82V82_X2_420   :
                        (strcmp(argv[i], "422-10")  == 0) ? LPixelFormat_MP_Y82_U82V82_X2_422   :
                        (strcmp(argv[i], "444-10")  == 0) ? LPixelFormat_MP_Y82_U82_V82_X2_444  :
                                                            LPixelFormat_INVALID;

                    if (ePixFmt != LPixelFormat_INVALID)
                    {
                        if (++i < iArgc)
                        {
                            MBOOL bValidFlag = MFALSE;
                            MBOOL bFlag      = MFALSE;

                            if ((ePixFmt == LPixelFormat_R8G8B8A8)
                                || (ePixFmt == LPixelFormat_R10G10B10A2))
                            {
                                if (strcmp(argv[i], "fr") == 0)
                                {
                                    bValidFlag = MTRUE;
                                    bFlag      = MTRUE;
                                }
                                else if (strcmp(argv[i], "lr") == 0)
                                {
                                    bValidFlag = MTRUE;
                                    bFlag      = MFALSE;
                                }
                            }
                            else
                            {
                                if (strcmp(argv[i], "bt709") == 0)
                                {
                                    bValidFlag = MTRUE;
                                    bFlag      = MTRUE;
                                }
                                else if (strcmp(argv[i], "bt601") == 0)
                                {
                                    bValidFlag = MTRUE;
                                    bFlag      = MFALSE;
                                }
                            }

                            if (bValidFlag)
                            {
                                for (; uiIdx < uiEndIdx; uiIdx++)
                                {
                                    g_aoVoutParam[uiIdx].eOutPixFmt         = ePixFmt;
                                    g_aoVoutParam[uiIdx].bOutRgbFullYuv709  = bFlag;
                                }

                                continue;
                            }
                        }
                    }
                }
            }
        }
        else if (strcmp(argv[i], "-fvp") == 0)
        {
            if (++i < iArgc)
            {
                MUINT uiIdx    = 0;
                MUINT uiEndIdx = uiMaxVoutMtxCount;

                if (strcmp(argv[i], "a") != 0)
                {
                    uiIdx    = atoi(argv[i]);
                    uiEndIdx = uiIdx + 1;
                }

                if ((uiIdx < uiMaxVoutMtxCount) && (++i < iArgc))
                {
                    LSIZE oSize;
                    MUINT uiFps;

                    if (sscanf(argv[i],
                               "%ux%u@%u",
                               &oSize.iWidth,
                               &oSize.iHeight,
                               &uiFps) == 3)
                    {
                        for (; uiIdx < uiEndIdx; uiIdx++)
                        {
                            g_aoVoutParam[uiIdx].oForcedSize = oSize;
                            g_aoVoutParam[uiIdx].uiForcedFps = uiFps;
                            g_aoVoutParam[uiIdx].bForceMode  = MTRUE;
                        }
                        continue;
                    }
                }
            }
        }
        else if (strcmp(argv[i], "-vor") == 0)
        {
            if (++i < iArgc)
            {
                if (sscanf(argv[i],
                           "%d,%d,%d,%d",
                           &oWinRect.iLeft,
                           &oWinRect.iTop,
                           &oWinRect.iRight,
                           &oWinRect.iBottom) == 4)
                {
                    if ((oWinRect.iLeft < oWinRect.iRight)
                        && (oWinRect.iTop < oWinRect.iBottom))
                    {
                        bForceWinRect = MTRUE;
                        continue;
                    }
                }
            }
        }
        else if (strcmp(argv[i], "-avsync") == 0)
        {
            bAvSync = MTRUE;
            continue;
        }
        else if (strcmp(argv[i], "-DEMO-A") == 0)
        {
            uiDemoIdx = 0;

            if (++i < iArgc)
            {
                iDemoACount = atoi(argv[i]);

                if ((iDemoACount > 0) && (iDemoACount <= 4)) continue;
            }
        }
        else if (strcmp(argv[i], "-DEMO-B") == 0)
        {
            uiDemoIdx = 1;

            if (++i < iArgc)
            {
                iAngularSpeed = atoi(argv[i]);

                if ((iAngularSpeed >= -360) && (iAngularSpeed <= 360))
                {
                    if (++i < iArgc)
                    {
                        uiTransparency = atoi(argv[i]);

                        if ((uiTransparency >= 0) && (uiTransparency <= 100)) continue;
                    }
                }
            }
        }
        else if (strcmp(argv[i], "-srv") == 0)
        {
            if (++i < iArgc)
            {
                if (argv[i][0] == 'x')
                {
                    uiCurVoutIdx = atoi(argv[i] + 1) + MAX_VOUTMTX_COUNT;
                }
                else
                {
                    uiCurVoutIdx = atoi(argv[i]);
                }

                if ((uiCurVoutIdx < MAX_VOUT_COUNT)
                    && ((uiCurVoutIdx < uiMaxVoutMtxCount)
                        || (uiCurVoutIdx >= MAX_VOUTMTX_COUNT)))
                {
                    g_aoVoutParam[uiCurVoutIdx].bNetServer = MTRUE;
                    continue;
                }
            }
        }
        else if (poStreamParam == MNULL)
        {
            // Need to enable at least one stream before to set the remaining parameters.
        }
        else if (strcmp(argv[i], "-i") == 0)
        {
           if (++i < iArgc)
           {
                if (strncmp("rtsp:", argv[i], 5) == 0)
                {
                    poStreamParam->szRtspLocation = argv[i];
                    poStreamParam->szSrcFilePath = MNULL;
                    poStreamParam->eNetProtocol = LNetStreamer_Protocol_RTSP;
                }
                else if (strncmp("rtmp:", argv[i], 5) == 0)
                {
                    poStreamParam->szRtspLocation = argv[i];
                    poStreamParam->szSrcFilePath = MNULL;
                    poStreamParam->eNetProtocol = LNetStreamer_Protocol_RTMP;
                }
                else
                {
                    poStreamParam->szRtspLocation = MNULL;
                    poStreamParam->szSrcFilePath = argv[i];
                }

                continue;
           }
        }
        else if (strcmp(argv[i], "-skip") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "on") == 0)
                {
                    poStreamParam->bSkipToNextIdr = MTRUE;
                    continue;
                }
                else if (strcmp(argv[i], "off") == 0)
                {
                    poStreamParam->bSkipToNextIdr = MFALSE;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-o") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "none") == 0)
                {
                    poStreamParam->szDstFilePath    = MNULL;
                    poStreamParam->bDecode          = MTRUE;
                    continue;
                }
                else
                {
                    poStreamParam->szDstFilePath    = argv[i];
                    poStreamParam->bDecode          = MTRUE;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-oo") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "none") == 0)
                {
                    poStreamParam->szDstFilePath    = MNULL;
                    poStreamParam->bDecode          = MTRUE;
                    continue;
                }
                else
                {
                    poStreamParam->szDstFilePath    = argv[i];
                    poStreamParam->bDecode          = MFALSE;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-w") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->oFrameSize.iWidth = atoi(argv[i]);
                if ((poStreamParam->oFrameSize.iWidth >= MIN_FRAME_SIZE)
                    && (poStreamParam->oFrameSize.iWidth <= MAX_FRAME_SIZE)) continue;
            }
        }
        else if (strcmp(argv[i], "-h") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->oFrameSize.iHeight = atoi(argv[i]);
                if ((poStreamParam->oFrameSize.iHeight >= MIN_FRAME_SIZE)
                    && (poStreamParam->oFrameSize.iHeight <= MAX_FRAME_SIZE)) continue;
            }
        }
        else if (strcmp(argv[i], "-fps") == 0)
        {
            if (++i < iArgc)
            {
                if (sscanf(argv[i],
                           "%u/%u",
                           &(poStreamParam->uiFrameRateNum),
                           &(poStreamParam->uiFrameRateDen)) == 2)
                {
                    if ((poStreamParam->uiFrameRateNum != 0)
                        && (poStreamParam->uiFrameRateDen != 0)) continue;
                }
            }
        }
        else if (strcmp(argv[i], "-loop") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "on") == 0)
                {
                    poStreamParam->bLoop = MTRUE;
                    continue;
                }
                else if (strcmp(argv[i], "off") == 0)
                {
                    poStreamParam->bLoop = MFALSE;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-proto") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "rtsp") == 0)
                {
                    poStreamParam->eNetProtocol = LNetStreamer_Protocol_RTSP;
                    continue;
                }
                else if (strcmp(argv[i], "rtp") == 0)
                {
                    poStreamParam->szRtspLocation = argv[i];
                    poStreamParam->eNetProtocol = LNetStreamer_Protocol_RTP;
                    continue;
                }
                else if (strcmp(argv[i], "ts") == 0)
                {
                    poStreamParam->szRtspLocation = argv[i];
                    poStreamParam->eNetProtocol = LNetStreamer_Protocol_TS;
                    continue;
                }
                else if (strcmp(argv[i], "rtmp") == 0)
                {
                    poStreamParam->eNetProtocol = LNetStreamer_Protocol_RTMP;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-addr") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "0") == 0)
                {
                    poStreamParam->szNetAddr = MNULL;
                }
                else
                {
                    poStreamParam->szNetAddr = argv[i];
                }
                continue;
            }
        }
        else if (strcmp(argv[i], "-port") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->uiNetPort = atoi(argv[i]);
                continue;
            }
        }
        else if (strcmp(argv[i], "-mtu") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->uiMtu = atoi(argv[i]);
                continue;
            }
        }
        else if (strcmp(argv[i], "-srt") == 0)
        {
            poStreamParam->bEnableSrt = MTRUE;
            continue;
        }
        else if (strcmp(argv[i], "-srtmode") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "listener") == 0)
                {
                    poStreamParam->eSrtMode = LNetStreamer_SrtMode_LISTENER;
                    continue;
                }
                else if (strcmp(argv[i], "caller") == 0)
                {
                    poStreamParam->eSrtMode = LNetStreamer_SrtMode_CALLER;
                    continue;
                }
                else if (strcmp(argv[i], "rendezvous") == 0)
                {
                    poStreamParam->eSrtMode = LNetStreamer_SrtMode_RENDEZVOUS;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-neti") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->szNetInterface = argv[i];
                continue;
            }
        }
        else if (strcmp(argv[i], "-ipv6") == 0)
        {
            poStreamParam->bEnableIpv6 = MTRUE;
            continue;
        }
        else if (strcmp(argv[i], "-d") == 0)
        {
            if (++i < iArgc)
            {
                poStreamParam->uiVoutCount = 0;

                while (*argv[i] != '-')
                {
                    MUINT uiVoutIdx = atoi(argv[i]);

                    if (argv[i][0] == 'x')
                    {
                        uiVoutIdx = atoi(argv[i] + 1) + MAX_VOUTMTX_COUNT;
                    }
                    else
                    {
                        uiVoutIdx = atoi(argv[i]);
                    }

                    if ((uiVoutIdx < MAX_VOUT_COUNT)
                        && (poStreamParam->uiVoutCount < MAX_VOUT_PER_STREAM)
                        && ((uiVoutIdx < uiMaxVoutMtxCount)
                            || (uiVoutIdx >= MAX_VOUTMTX_COUNT)))
                    {
                        poStreamParam->auiVoutIndex[poStreamParam->uiVoutCount] = uiVoutIdx;
                        poStreamParam->uiVoutCount++;
                    }

                    if (++i >= iArgc)
                    {
                        break;
                    }
                }

                if (poStreamParam->uiVoutCount > 0)
                {
                    i--;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-sr") == 0)
        {
            if (++i < iArgc)
            {
                MUINT uiVoutIdx = atoi(argv[i]);

                if (uiVoutIdx < MAX_VOUT_COUNT)
                {
                    if (++i < iArgc)
                    {
                        LRECT32* poSrcRect = &poStreamParam->aoSrcRect[uiVoutIdx];

                        if (sscanf(argv[i],
                                   "%d,%d,%d,%d",
                                   &poSrcRect->iLeft,
                                   &poSrcRect->iTop,
                                   &poSrcRect->iRight,
                                   &poSrcRect->iBottom) == 4)
                        {
                            if ((poSrcRect->iLeft < poSrcRect->iRight)
                                && (poSrcRect->iTop < poSrcRect->iBottom))
                            {
                                continue;
                            }
                        }
                    }
                }
            }
        }
        else if (strcmp(argv[i], "-dr") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "auto") == 0)
                {
                    poStreamParam->bAutoDstRect = MTRUE;
                    continue;
                }
                else
                {
                    if (sscanf(argv[i],
                               "%d,%d,%d,%d",
                               &poStreamParam->oDstRect.iLeft,
                               &poStreamParam->oDstRect.iTop,
                               &poStreamParam->oDstRect.iRight,
                               &poStreamParam->oDstRect.iBottom) == 4)
                    {
                        LSIZE oSize;

                        oSize.iWidth  = poStreamParam->oDstRect.iRight - poStreamParam->oDstRect.iLeft;
                        oSize.iHeight = poStreamParam->oDstRect.iBottom - poStreamParam->oDstRect.iTop;

                        if ((oSize.iWidth >= MIN_FRAME_SIZE)
                            && (oSize.iHeight >= MIN_FRAME_SIZE)
                            && (oSize.iWidth <= MAX_FRAME_SIZE)
                            && (oSize.iHeight <= MAX_FRAME_SIZE))
                        {
                            poStreamParam->bAutoDstRect = MFALSE;
                            continue;
                        }
                    }
                }
            }
        }
        else if (strcmp(argv[i], "-nbuf") == 0)
        {
            if (++i < iArgc)
            {
                MUINT uiCount = atoi(argv[i]);

                if ((uiCount >= 3) && (uiCount <= 24))
                {
                    uiBufferCount = uiCount;
                    continue;
                }
            }
        }

        if (i >= iArgc)
        {

        }
        else if (uiCurVoutIdx >= MAX_VOUT_COUNT)
        {
            // Need '-srv <N>' for using the next arguments.
        }
        else if (strcmp(argv[i], "-tbr") == 0)
        {
            if (++i < iArgc)
            {
                MUINT uiTargetBitRateKbs = atoi(argv[i]);

                if ((uiTargetBitRateKbs >= 100) && (uiTargetBitRateKbs <= ENC_MAX_BITRATE_KBPS))
                {
                    g_aoVoutParam[uiCurVoutIdx].uiTargetBitRateKbs = uiTargetBitRateKbs;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-ip") == 0)
        {
            if (++i < iArgc)
            {
                g_aoVoutParam[uiCurVoutIdx].uiIPeriod = atoi(argv[i]);
                if (g_aoVoutParam[uiCurVoutIdx].uiIPeriod > 0) continue;
            }
        }
        else if (strcmp(argv[i], "-pp") == 0)
        {
            if (++i < iArgc)
            {
                g_aoVoutParam[uiCurVoutIdx].uiPPeriod = atoi(argv[i]);
                if ((g_aoVoutParam[uiCurVoutIdx].uiPPeriod > 0)
                    && (g_aoVoutParam[uiCurVoutIdx].uiPPeriod <= 8)) continue;
            }
        }
        else if (strcmp(argv[i], "-qpmin") == 0)
        {
            if (++i < iArgc)
            {
                g_aoVoutParam[uiCurVoutIdx].uiQpMin = atoi(argv[i]);
                if ((g_aoVoutParam[uiCurVoutIdx].uiQpMin >= 0)
                    && (g_aoVoutParam[uiCurVoutIdx].uiQpMin <= 51)) continue;
            }
        }
        else if (strcmp(argv[i], "-qpmax") == 0)
        {
            if (++i < iArgc)
            {
                g_aoVoutParam[uiCurVoutIdx].uiQpMax = atoi(argv[i]);
                if ((g_aoVoutParam[uiCurVoutIdx].uiQpMax >= 0)
                    && (g_aoVoutParam[uiCurVoutIdx].uiQpMax <= 51)) continue;
            }
        }
        else if (strcmp(argv[i], "-ns") == 0)
        {
            if (++i < iArgc)
            {
                g_aoVoutParam[uiCurVoutIdx].uiSliceCount = atoi(argv[i]);
                if ((g_aoVoutParam[uiCurVoutIdx].uiSliceCount > 0)
                    && (g_aoVoutParam[uiCurVoutIdx].uiSliceCount <= ENC_MAX_NB_SLICES)) continue;
            }
        }
        else if (strcmp(argv[i], "-rc") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "win") == 0)
                {
                    g_aoVoutParam[uiCurVoutIdx].eEncodeMode = LH264E_EncoderMode_RATECONTROLWINDOWED;
                    continue;
                }
                else if (strcmp(argv[i], "man") == 0)
                {
                    g_aoVoutParam[uiCurVoutIdx].eEncodeMode = LH264E_EncoderMode_MANUAL;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-prof") == 0)
        {
            if (++i < iArgc)
            {
                MBOOL bValid = MTRUE;

                if (strcmp(argv[i], "b")  == 0)
                {
                    g_aoVoutParam[uiCurVoutIdx].eEncodeProfile = LH264_Profile_BASELINE;
                }
                else if (strcmp(argv[i], "m")  == 0)
                {
                    g_aoVoutParam[uiCurVoutIdx].eEncodeProfile = LH264_Profile_MAIN;
                }
                else if (strcmp(argv[i], "hi")  == 0)
                {
                    g_aoVoutParam[uiCurVoutIdx].eEncodeProfile = LH264_Profile_HIGH;
                }
                else if (strcmp(argv[i], "hi10")  == 0)
                {
                    g_aoVoutParam[uiCurVoutIdx].eEncodeProfile = LH264_Profile_HIGH10;
                }
                else if (strcmp(argv[i], "hi422")  == 0)
                {
                    g_aoVoutParam[uiCurVoutIdx].eEncodeProfile = LH264_Profile_HIGH422;
                }
                else if (strcmp(argv[i], "hi444")  == 0)
                {
                    g_aoVoutParam[uiCurVoutIdx].eEncodeProfile = LH264_Profile_HIGH444;
                }
                else if (strcmp(argv[i], "cavlc")  == 0)
                {
                    g_aoVoutParam[uiCurVoutIdx].eEncodeProfile = LH264_Profile_CAVLC444INTRA;
                }
                else
                {
                    bValid = MFALSE;
                }

                if (bValid)
                {
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-lvl") == 0)
        {
            if (++i < iArgc)
            {
                MBOOL      bValid      = MTRUE;
                VoutParam* poVoutParam = &g_aoVoutParam[uiCurVoutIdx];

                if      (strcmp(argv[i], "1B")  == 0)   poVoutParam->eEncodeLevel = LH264_Level_1B;
                else if (strcmp(argv[i], "1")   == 0)   poVoutParam->eEncodeLevel = LH264_Level_1;
                else if (strcmp(argv[i], "1.1") == 0)   poVoutParam->eEncodeLevel = LH264_Level_1_1;
                else if (strcmp(argv[i], "1.2") == 0)   poVoutParam->eEncodeLevel = LH264_Level_1_2;
                else if (strcmp(argv[i], "1.3") == 0)   poVoutParam->eEncodeLevel = LH264_Level_1_3;
                else if (strcmp(argv[i], "2")   == 0)   poVoutParam->eEncodeLevel = LH264_Level_2;
                else if (strcmp(argv[i], "2.1") == 0)   poVoutParam->eEncodeLevel = LH264_Level_2_1;
                else if (strcmp(argv[i], "2.2") == 0)   poVoutParam->eEncodeLevel = LH264_Level_2_2;
                else if (strcmp(argv[i], "3")   == 0)   poVoutParam->eEncodeLevel = LH264_Level_3;
                else if (strcmp(argv[i], "3.1") == 0)   poVoutParam->eEncodeLevel = LH264_Level_3_1;
                else if (strcmp(argv[i], "3.2") == 0)   poVoutParam->eEncodeLevel = LH264_Level_3_2;
                else if (strcmp(argv[i], "4")   == 0)   poVoutParam->eEncodeLevel = LH264_Level_4;
                else if (strcmp(argv[i], "4.1") == 0)   poVoutParam->eEncodeLevel = LH264_Level_4_1;
                else if (strcmp(argv[i], "4.2") == 0)   poVoutParam->eEncodeLevel = LH264_Level_4_2;
                else if (strcmp(argv[i], "5")   == 0)   poVoutParam->eEncodeLevel = LH264_Level_5;
                else if (strcmp(argv[i], "5.1") == 0)   poVoutParam->eEncodeLevel = LH264_Level_5_1;
                else if (strcmp(argv[i], "5.2") == 0)   poVoutParam->eEncodeLevel = LH264_Level_5_2;
                else
                {
                    bValid = MFALSE;
                }

                if (bValid)
                {
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-faq") == 0)
        {
            if (++i < iArgc)
            {
                g_aoVoutParam[uiCurVoutIdx].uiFaq = atoi(argv[i]);
                if ((g_aoVoutParam[uiCurVoutIdx].uiFaq >= 0)
                    && (g_aoVoutParam[uiCurVoutIdx].uiFaq <= 100)) continue;
            }
        }
        else if (strcmp(argv[i], "-df") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "on") == 0)
                {
                    g_aoVoutParam[uiCurVoutIdx].bDfEnabled = MTRUE;
                    continue;
                }
                else if (strcmp(argv[i], "off") == 0)
                {
                    g_aoVoutParam[uiCurVoutIdx].bDfEnabled = MFALSE;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-epf") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "400-8") == 0)
                {
                    g_aoVoutParam[uiCurVoutIdx].eEncodePixFmt = LPixelFormat_Y8;
                    continue;
                }
                else if (strcmp(argv[i], "400-10") == 0)
                {
                    g_aoVoutParam[uiCurVoutIdx].eEncodePixFmt = LPixelFormat_Y82Y82Y82X2;
                    continue;
                }
                else if (strcmp(argv[i], "420-8") == 0)
                {
                    g_aoVoutParam[uiCurVoutIdx].eEncodePixFmt = LPixelFormat_MP_Y8_U8V8_420;
                    continue;
                }
                else if (strcmp(argv[i], "420-10") == 0)
                {
                    g_aoVoutParam[uiCurVoutIdx].eEncodePixFmt = LPixelFormat_MP_Y82_U82V82_X2_420;
                    continue;
                }
                else if (strcmp(argv[i], "422-8") == 0)
                {
                    g_aoVoutParam[uiCurVoutIdx].eEncodePixFmt = LPixelFormat_MP_Y8_U8V8_422;
                    continue;
                }
                else if (strcmp(argv[i], "422-10") == 0)
                {
                    g_aoVoutParam[uiCurVoutIdx].eEncodePixFmt = LPixelFormat_MP_Y82_U82V82_X2_422;
                    continue;
                }
                else if (strcmp(argv[i], "444-8") == 0)
                {
                    g_aoVoutParam[uiCurVoutIdx].eEncodePixFmt = LPixelFormat_MP_Y8_U8_V8_444;
                    continue;
                }
                else if (strcmp(argv[i], "444-10") == 0)
                {
                    g_aoVoutParam[uiCurVoutIdx].eEncodePixFmt = LPixelFormat_MP_Y82_U82_V82_X2_444;
                    continue;
                }
            }
        }

        if (i >= iArgc)
        {

        }
        else if(poAudioStreamParam == MNULL)
        {
            // The following parameters are applied to the audio stream previously specified with "-as"
        }
        else if (strcmp(argv[i], "-i") == 0)
        {
            if (++i < iArgc)
            {
                if (strncmp("rtsp:", argv[i], 5) == 0)
                {
                    poAudioStreamParam->szRtspLocation = argv[i];
                    poAudioStreamParam->eNetProtocol = LNetStreamer_Protocol_RTSP;
                    continue;
                }
                else if (strncmp("rtmp:", argv[i], 5) == 0)
                {
                    poAudioStreamParam->szRtspLocation = argv[i];
                    poAudioStreamParam->eNetProtocol = LNetStreamer_Protocol_RTMP;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-o") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "d") == 0)
                {
                    poAudioStreamParam->bEnableOutput = MFALSE;
                    continue;
                }
                else
                {
                    MUINT uiAoutIdx = atoi(argv[i]);

                    if (uiAoutIdx < uiMaxAoutCount)
                    {
                        MUINT j;
                        MBOOL bBusyIdx = MFALSE;
                        for (j = 0; j < MAX_STREAM_COUNT; j++)
                        {
                            if ((&g_aoAudioStreamParam[j] != poAudioStreamParam)
                                && (g_aoAudioStreamParam[j].bEnable)
                                && (g_aoAudioStreamParam[j].bEnableOutput)
                                && (g_aoAudioStreamParam[j].uiAoutIdx == uiAoutIdx))
                            {
                                bBusyIdx = MTRUE;
                            }
                        }

                        if (!bBusyIdx)
                        {
                            poAudioStreamParam->bEnableOutput = MTRUE;
                            poAudioStreamParam->uiAoutIdx = uiAoutIdx;
                            continue;
                        }
                    }
                }
            }
        }
        else if (strcmp(argv[i], "-proto") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "rtsp") == 0)
                {
                    poAudioStreamParam->eNetProtocol = LNetStreamer_Protocol_RTSP;
                    continue;
                }
                else if (strcmp(argv[i], "rtp") == 0)
                {
                    poAudioStreamParam->szRtspLocation = argv[i];
                    poAudioStreamParam->eNetProtocol = LNetStreamer_Protocol_RTP;
                    continue;
                }
                else if (strcmp(argv[i], "ts") == 0)
                {
                    poAudioStreamParam->szRtspLocation = argv[i];
                    poAudioStreamParam->eNetProtocol = LNetStreamer_Protocol_TS;
                    continue;
                }
                else if (strcmp(argv[i], "rtmp") == 0)
                {
                    poAudioStreamParam->eNetProtocol = LNetStreamer_Protocol_RTMP;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-addr") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "0") == 0)
                {
                    poAudioStreamParam->szNetAddr = MNULL;
                }
                else
                {
                    poAudioStreamParam->szNetAddr = argv[i];
                }
                continue;
            }
        }
        else if (strcmp(argv[i], "-port") == 0)
        {
            if (++i < iArgc)
            {
                poAudioStreamParam->uiNetPort = atoi(argv[i]);
                continue;
            }
        }
        else if (strcmp(argv[i], "-mtu") == 0)
        {
            if (++i < iArgc)
            {
                poAudioStreamParam->uiMtu = atoi(argv[i]);
                continue;
            }
        }
        else if (strcmp(argv[i], "-srt") == 0)
        {
            poAudioStreamParam->bEnableSrt = MTRUE;
            continue;
        }
        else if (strcmp(argv[i], "-srtmode") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "listener") == 0)
                {
                    poAudioStreamParam->eSrtMode = LNetStreamer_SrtMode_LISTENER;
                    continue;
                }
                else if (strcmp(argv[i], "caller") == 0)
                {
                    poAudioStreamParam->eSrtMode = LNetStreamer_SrtMode_CALLER;
                    continue;
                }
                else if (strcmp(argv[i], "rendezvous") == 0)
                {
                    poAudioStreamParam->eSrtMode = LNetStreamer_SrtMode_RENDEZVOUS;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-neti") == 0)
        {
            if (++i < iArgc)
            {
                poAudioStreamParam->szNetInterface = argv[i];
                continue;
            }
        }
        else if (strcmp(argv[i], "-ipv6") == 0)
        {
            poAudioStreamParam->bEnableIpv6 = MTRUE;
            continue;
        }
        else if (strcmp(argv[i], "-rate") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "8000") == 0)
                {
                    poAudioStreamParam->eSamplingRate = LAudioCodec_SamplingRate_8000Hz;
                    continue;
                }
                else if (strcmp(argv[i], "11025") == 0)
                {
                    poAudioStreamParam->eSamplingRate = LAudioCodec_SamplingRate_11025Hz;
                    continue;
                }
                else if (strcmp(argv[i], "12000") == 0)
                {
                    poAudioStreamParam->eSamplingRate = LAudioCodec_SamplingRate_12000Hz;
                    continue;
                }
                else if (strcmp(argv[i], "16000") == 0)
                {
                    poAudioStreamParam->eSamplingRate = LAudioCodec_SamplingRate_16000Hz;
                    continue;
                }
                else if (strcmp(argv[i], "22050") == 0)
                {
                    poAudioStreamParam->eSamplingRate = LAudioCodec_SamplingRate_22050Hz;
                    continue;
                }
                else if (strcmp(argv[i], "24000") == 0)
                {
                    poAudioStreamParam->eSamplingRate = LAudioCodec_SamplingRate_24000Hz;
                    continue;
                }
                else if (strcmp(argv[i], "32000") == 0)
                {
                    poAudioStreamParam->eSamplingRate = LAudioCodec_SamplingRate_32000Hz;
                    continue;
                }
                else if (strcmp(argv[i], "44100") == 0)
                {
                    poAudioStreamParam->eSamplingRate = LAudioCodec_SamplingRate_44100Hz;
                    continue;
                }
                else if (strcmp(argv[i], "48000") == 0)
                {
                    poAudioStreamParam->eSamplingRate = LAudioCodec_SamplingRate_48000Hz;
                    continue;
                }
                else if (strcmp(argv[i], "88200") == 0)
                {
                    poAudioStreamParam->eSamplingRate = LAudioCodec_SamplingRate_88200Hz;
                    continue;
                }
                else if (strcmp(argv[i], "96000") == 0)
                {
                    poAudioStreamParam->eSamplingRate = LAudioCodec_SamplingRate_96000Hz;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-ch") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "1") == 0)
                {
                    poAudioStreamParam->eChannelCfg = LAudioCodec_ChannelCfg_MONO;
                    continue;
                }
                else if (strcmp(argv[i], "2") == 0)
                {
                    poAudioStreamParam->eChannelCfg = LAudioCodec_ChannelCfg_STEREO;
                    continue;
                }
            }
        }
        else if (strcmp(argv[i], "-prof") == 0)
        {
            if (++i < iArgc)
            {
                if (strcmp(argv[i], "LC") == 0)
                {
                    poAudioStreamParam->eProfile = LAudioCodec_AACProfile_LC;
                    continue;
                }
                else if (strcmp(argv[i], "HEV1") == 0)
                {
                    poAudioStreamParam->eProfile = LAudioCodec_AACProfile_HEV1;
                    continue;
                }
                else if (strcmp(argv[i], "HEV2") == 0)
                {
                    poAudioStreamParam->eProfile = LAudioCodec_AACProfile_HEV2;
                    continue;
                }
            }
        }

        bArgError = MTRUE;
        break;
    }

    MUINT uiMaxInput = (uiDemoIdx == 0) ? 1 : MOSAIC_MAX_INPUT;
    MUINT uiEnabledVideoStrmCnt = 0;
    MUINT uiEnabledAudioStrmCnt = 0;

    for (i = 0; (i < MAX_STREAM_COUNT) && !bArgError; i++)
    {
        if (g_aoStreamParam[i].bEnable)
        {
            if ((g_aoStreamParam[i].oFrameSize.iWidth == 0)
                || (g_aoStreamParam[i].oFrameSize.iHeight == 0)
                || ((g_aoStreamParam[i].szSrcFilePath == MNULL)
                    && (g_aoStreamParam[i].szRtspLocation == MNULL)))
            {
                bArgError = MTRUE;
            }
            else if (g_aoStreamParam[i].bDecode)
            {
                MUINT j;
                for (j = 0; j < g_aoStreamParam[i].uiVoutCount; j++)
                {
                    MUINT uiVoutIdx = g_aoStreamParam[i].auiVoutIndex[j];

                    if ((uiVoutIdx < MAX_VOUT_COUNT)
                        && (g_aoVoutParam[uiVoutIdx].uiInputCount < uiMaxInput))
                    {
                        g_aoStreamParam[i].auiMosInputIndex[j] = g_aoVoutParam[uiVoutIdx].uiInputCount;

                        if (uiDemoIdx == 0)
                        {
                            g_aoVoutParam[uiVoutIdx].uiInputCount = iDemoACount;
                        }
                        else
                        {
                            g_aoVoutParam[uiVoutIdx].uiInputCount++;
                        }

                        if (g_aoStreamParam[i].szRtspLocation != MNULL)
                        {
                            g_aoVoutParam[uiVoutIdx].bControlQueueOverflow = MTRUE;
                        }
                    }
                    else
                    {
                        bArgError = MTRUE;
                    }
                }

                if ((g_aoStreamParam[i].oFrameSize.iWidth >= g_oBigDecodePictureSize.iWidth)
                    && (g_aoStreamParam[i].oFrameSize.iHeight >= g_oBigDecodePictureSize.iHeight))
                {
                    uiBigDecodeCount++;
                }
            }

            uiEnabledVideoStrmCnt++;
        }
        else if(g_aoAudioStreamParam[i].bEnable)
        {
            uiEnabledAudioStrmCnt++;
        }

        if(g_aoAudioStreamParam[i].bEnable)
        {
            if(g_aoStreamParam[i].bEnable)
            {
                // Can't do audio client if the video stream at the same index has a file source.
                if(g_aoStreamParam[i].szSrcFilePath != MNULL)
                {
                    bArgError = MTRUE;
                }
                else
                {
                    // Use video network parameters for audio stream.
                    g_aoAudioStreamParam[i].szRtspLocation  = g_aoStreamParam[i].szRtspLocation;
                    g_aoAudioStreamParam[i].eNetProtocol    = g_aoStreamParam[i].eNetProtocol;
                    g_aoAudioStreamParam[i].szNetAddr       = g_aoStreamParam[i].szNetAddr;
                }
            }

            if(g_aoAudioStreamParam[i].szRtspLocation == MNULL)
            {
                bArgError = MTRUE;
            }

            if (g_aoAudioStreamParam[i].bEnableOutput
                 && (g_aoAudioStreamParam[i].uiAoutIdx >= uiMaxMtxAoutCount))
            {
                g_aoAudioStreamParam[i].bMtxAout = MFALSE;
            }
        }
    }

    if(((uiEnabledVideoStrmCnt + uiEnabledAudioStrmCnt) > MAX_STREAM_COUNT)
        || ((uiEnabledVideoStrmCnt == 0) && (uiEnabledAudioStrmCnt == 0)))
    {
        bArgError = MTRUE;
    }

    if (bArgError)
    {
        char szVoutRange[64] = "";

        if (uiMaxVoutMtxCount > 0)
        {
            snprintf(szVoutRange, sizeof(szVoutRange), "0..%u%s",
                     uiMaxVoutMtxCount-1, (MAX_VOUTGLX_COUNT > 0) ? ", " : "");
        }

        if (MAX_VOUTGLX_COUNT > 0)
        {
            char szXRange[32];

            snprintf(szXRange, sizeof(szXRange), "x0..x%u", MAX_VOUTGLX_COUNT-1);
            strncat_wz(szVoutRange, szXRange, sizeof(szVoutRange));
        }

        printf("\nMandatory arguments:\n");
        printf("  -s <N>         : Enable stream 'N': [0..%u]. All remaining non-global arguments between\n"
               "                   this argument and the next \"-s\" or \"-as\" argument will be applied\n"
               "                   to this stream. As default, this stream takes the arguments of the previous\n"
               "                   stream in the command line.\n",
                                   MAX_STREAM_COUNT-1);
        printf("  -as <N>        : Enable audio stream 'N' : [0..%u]. All remaining non-global arguments between\n"
               "                   this argument and the next \"-s\" or \"-as\" argument will be applied\n"
               "                   to this audio stream. As default, this audio stream takes the arguments of the\n"
               "                   previous audio stream in the command line.\n"
               "                   If a video stream with network client is enabled at the same index, this audio\n"
               "                   stream will be added to the video network client. If a video stream without\n"
               "                   network client is enabled at this index, failure will happen. If no video stream\n"
               "                   is enabled at this index, a network client will be created for the audio stream alone.\n"
               "                   The maximum number of enabled streams is %u (audio, video or audio+video).\n"
               "                   If the stream contains audio and video, network parameters for this audio stream,\n"
               "                   if specified, will be overwritten by those of the video stream at the same index\n",
                                    MAX_STREAM_COUNT-1, MAX_STREAM_COUNT);
        printf("  -i <ADDR/PATH> : Input server rtsp address 'ADDR' or H.264 video file 'PATH'.\n");
        printf("  -w <N>         : Frame width: [%u..%u].\n", MIN_FRAME_SIZE, MAX_FRAME_SIZE);
        printf("  -h <N>         : Frame height: [%u..%u].\n", MIN_FRAME_SIZE, MAX_FRAME_SIZE);
        printf("\nOptional arguments (per video stream):\n");
        printf("  -fps <N/D>     : Frame rate in frame per second: [1..]/[1..], default= %u/%u.\n",
                                   g_oDefaultStreamParam.uiFrameRateNum,
                                   g_oDefaultStreamParam.uiFrameRateDen);
        printf("  -o <PATH>      : Output H.264 video file: default= none.\n");
        printf("  -oo <PATH>     : Output H.264 video file ONLY (no decoding): default= none.\n");
        printf("  -loop <on|off> : Play input video file in loop: on = enabled, off= disabled (default).\n");
        printf("  -d <N> ...     : Display index: [%s] where 'x0' means X-Windows output 0. Default= 0.\n"
               "                   Can specify up to %u display index.\n", szVoutRange, MAX_VOUT_PER_STREAM);
        printf("  -sr <N> <L,T,R,B> : Source rectangle '<L,T,R,B>' for display index '<N>'. N = [%s]\n"
               "                      where 'x0' means X-Windows output 0. L,T,R,B= Left, top, right and\n"
               "                      bottom coordinates of the rectangle. Default= Whole frame.\n",
                                      szVoutRange);
        printf("  -dr <L,T,R,B>  : Display rectangle. L,T,R,B= Left, top, right and bottom coordinates of\n"
               "                   the rectangle or 'auto' (default) to set automatically in a mosaic\n"
               "                   arrangement. The minimum and maximum rectangle size are %u and %u\n"
               "                   respectively.\n", MIN_FRAME_SIZE, MAX_FRAME_SIZE);
        printf("  -proto <x>     : Select network protocol: rtsp (default), rtmp, rtp or ts.\n");
        printf("  -addr <IP>     : Select network destination address. Default= 0 (unicast).\n");
        printf("  -port <N>      : Select network destination port.\n");
        printf("  -mtu <N>       : Select network MTU size in bytes.\n");
        printf("  -srt           : Enable SRT protocol.\n");
        printf("  -srtmode <x>   : Select SRT connection mode: listener, caller, rendezvous.\n");
        printf("  -neti <x>      : Network interface name or address.\n");
        printf("  -ipv6          : Enable IPv6 addressing.\n");
        printf("  -skip <on|off> : Skip frames until the next IDR when a packet drop occurs. Default: %s.\n",
                                   g_oDefaultStreamParam.bSkipToNextIdr ? "on" : "off");
        printf("\nOptional arguments (per audio stream):\n");
        printf("  -i <ADDR>      : Input server rtsp address 'ADDR'.\n");
        printf("  -o <N>         : Enable audio output at index '<N>'. N = [0..%u], use 'd' to disable audio output.\n"
               "                   Default= 0.\n"
               "                     Indexes [0..%u] are Liberatus for devices.\n",
                                   uiMaxAoutCount-1, uiMaxMtxAoutCount-1);
        if (uiMaxAoutCount > uiMaxMtxAoutCount)
        {
            printf("                     Indexes [%u..%u] are for ALSA devices:\n",
                                       uiMaxMtxAoutCount, uiMaxAoutCount-1);
            MUINT uiAlsaIdx;
            for (uiAlsaIdx = 0; uiAlsaIdx < (uiMaxAoutCount - uiMaxMtxAoutCount); uiAlsaIdx++)
            {
                LSTR64 szAlsaDev = {0};
                AoutAlsaMod_GetDeviceName(uiAlsaIdx, &szAlsaDev);
                printf("                       [%u]: %s\n", uiAlsaIdx + uiMaxMtxAoutCount, szAlsaDev);
            }
        }
        else
        {
            printf("                    No ALSA devices.\n");
        }
        printf("  -proto <x>     : Select network protocol: rtsp (default), rtmp, rtp or ts.\n");
        printf("  -addr <IP>     : Select network destination address. Default= 0 (unicast).\n");
        printf("  -port <N>      : Select network destination port.\n");
        printf("  -mtu <N>       : Select network MTU size in bytes.\n");
        printf("  -srt           : Enable SRT protocol.\n");
        printf("  -srtmode <x>   : Select SRT connection mode: listener, caller, rendezvous.\n");
        printf("  -neti <x>      : Network interface name or address.\n");
        printf("  -ipv6          : Enable IPv6 addressing.\n");
        printf("  -rate <N>      : Specify sampling rate '<N>' (required for ts and rtp protocols).\n");
        printf("  -ch <N>        : Specify number of channels '<N>' (required for ts and rtp protocols).\n");
        printf("  -prof <P>      : Specify encoding profile '<P>' (LC, HEV1 or HEV2. required for ts and rtp protocols).\n");
        printf("\nOptional arguments (global):\n");
        printf("  -dbg <N>       : Set debug message level: [0..], default= 0 (disabled).\n");
        printf("  -vclog         : Enable VC log: default= Disabled.\n");
        printf("  -stat          : Show statistics: default= disabled.\n");
        printf("  -fsc <O> <T>   : Set full screen cycle time 'T' on the video output 'O': O= [%s],\n"
               "                 : T= Cycle time in seconds, default= 0 (disabled).\n", szVoutRange);
        printf("  -dpf <O> <F>   : Set display pixel format 'F' of output 'O': O= [0..%u, a (all outputs)],\n"
               "                   F= [RGB-8 (default), RGB-10, 444-10, YUV, YUY2, 420-8-2P, 422-8-2P,\n"
               "                   444-8-2P, 420-8-3P, 422-8-3P, 444-8-3P, RGB-8-3P, RGB-10-3P, \n"
               "                   420-10-2P, 422-10-2P, 444-10-2P, 420-10-3P, 422-10-3P, 444-10-3P].\n",
                                   uiMaxVoutMtxCount-1);
        printf("  -opf <O> <F> <M>  : Set output pixel format 'F' of output 'O' with mode 'M': O= [0..%u,\n"
               "                      a (all outputs)], F= [RGB-8 (default), RGB-10, 420-8, 422-8, 444-8,\n"
               "                      420-10, 422-10, 444-10], M (with RGB): [fr (full range, default),\n"
               "                      lr (limited range)], M (with YUV): [bt601, bt709].\n",
                                      uiMaxVoutMtxCount-1);
        printf("  -fvp <O> <WxH@HZ> : Force video out parameters on output 'O': [0..%u, a (all outputs)].\n"
               "                      W= Width, H= Height, HZ= Vertical frequency. Default= Use preferred\n"
               "                      mode of the monitor.\n", uiMaxVoutMtxCount-1);
        printf("  -vor <L,T,R,B> : Set video out window rectangle (output 1 only). L,T,R,B= Left, top,\n"
               "                   right and bottom coordinates of the window. Default= Full screen.\n");
        printf("  -avsync        : Enable audio/video synchronization.\n");
        printf("  -DEMO-A <N>    : Run demo A: Scale video 'N' times to different sizes from 4K to 480p in\n"
               "                   cascade mode. Only one stream could be displayed. N= [1..4].\n");
        printf("  -DEMO-B <S> <T>: Run demo B: Play 4 videos with different sizes from 4K to 480p with\n"
               "                   transparency and animation: S= [-360..360] Angular speed in degree/sec.\n"
               "                   T= [0..100] Transparency in percent.\n");
        printf("  -srv <N>       : Enable encoding and RTSP server on display 'N': [%s],\n"
               "                   default= disabled.\n", szVoutRange);
        printf("  -nbuf <N>      : Set the number of buffers 'N' of the decoders and Vouts. Reduce this\n"
               "                   number when there is not enough local memory. N= [%u..%u], Default= %u.\n",
                                   uiMinBufferCount, uiMaxBufferCount, uiDefBufferCount);
        printf("\nOptional arguments used with '-srv <N>' argument:\n");
        printf("  -tbr <N>       : Target bit rate in kbits/sec: [100..%u], default= %u.\n",
                                   ENC_MAX_BITRATE_KBPS, g_oDefaultVoutParam.uiTargetBitRateKbs);
        printf("  -ip <N>        : I period: [1..], default= %u. Must be a multiple of -pp value.\n",
                                   g_oDefaultVoutParam.uiIPeriod);
        printf("  -pp <N>        : P period: [1..8], default= %u. Must be divisible by -ip value.\n",
                                   g_oDefaultVoutParam.uiPPeriod);
        printf("  -qpmin <N>     : Minimum QP: [0..51], default= %u. Must be < or = to -qpmax value.\n",
                                   g_oDefaultVoutParam.uiQpMin);
        printf("  -qpmax <N>     : Maximum QP: [0..51], default= %u. Must be > or = to -qpmin value.\n",
                                   g_oDefaultVoutParam.uiQpMax);
        printf("  -ns <N>        : Number of slices: [1..%u], default= %u.\n", ENC_MAX_NB_SLICES,
                                   g_oDefaultVoutParam.uiSliceCount);
        printf("  -rc <M>        : Set rate control mode 'M'. Options: win = Windowed (default),\n"
               "                   man = Manual.\n");
        printf("  -prof <P>      : Encoding profile 'P'. P= [b, m, hi, hi10, hi422, hi444 (default),\n"
               "                   cavlc].\n");
        printf("  -lvl <L>       : Encoding level 'L'. L= [1B, 1, 1.1, 1.2, 1.3, 2, 2.1, 2.2, 3, 3.1, 3.2,\n"
               "                   4, 4.1, 4.2, 5, 5.1, 5.2 (default)].\n");
        printf("  -epf <F>       : Set encoding pixel format: F= [400-8, 400-10, 420-8 (default),\n"
               "                   420-10, 422-8, 422-10, 444-8 or 444-10].\n");
        printf("  -faq <N>       : Frequency adaptive quantization in percent: [0..100], default= %u.\n",
                                   g_oDefaultVoutParam.uiFaq);
        printf("  -df <on|off>   : Deblocking filter switch. Default= %s.\n",
                                   g_oDefaultVoutParam.bDfEnabled ? "on" : "off");
        printf("\nExamples:\n");
        printf("  Decode a 1080p60 video stream from a RTSP server and display it on video output 0:\n"
               "    %s -s 0 -i rtsp://192.168.152.84:3049/S0 -w 1920 -h 1080\n", argv[0]);
        printf("  Decode a 1080p30 video file and display it on video output 1:\n"
               "    %s -s 0 -i videotest.h264 -w 1920 -h 1080 -fps 30/1 -d 1\n", argv[0]);
        printf("  Make the two examples above at the same time:\n"
               "    %s -s 0 -i rtsp://192.168.152.84:3049/S0 -w 1920 -h 1080\n"
               "    -s 1 -i videotest.h264 -w 1920 -h 1080 -fps 30/1 -d 1\n", argv[0]);
        printf("  Decode an audio stream from a RTSP server and play it on output 0:\n"
               "    %s -as 0 -i rtsp://192.168.152.84:3049/S0 -o 0\n", argv[0]);
        printf("  Decode a 1080p60 video stream and an audio stream from a RTSP server. Display the\n"
               "  video stream on video output 0, play the audio stream on audio output 0:\n"
               "    %s -s 0 -i rtsp://192.168.152.84:3049/S0 -w 1920 -h 1080 -as 0 -o 0\n", argv[0]);
        printf("\n");

        return -1;
    }

    //--------------------------------------------------------------------------------------------------------
    // Init and connect modules

    SetMsgLogLevel(uiMsgLogLevel);

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = SetVideoCodecFwTraceLevel(hDevice, bEnableVcLog ? 0xFF : 3);
    }

    if(bAvSync)
    {
        InitAVSyncParams();
    }

    MBOOL bRestart;

  Restart:
    bRestart = MFALSE;

    for (i = 0; i < MAX_VOUT_COUNT; i++)
    {
        if ((g_aoVoutParam[i].uiInputCount > 0) && LSTATUS_IS_SUCCESS(eStatus))
        {
            MUINT uiFreq;

            printf("***** Init Vout[%s%u] *****\n",
                   (i < MAX_VOUTMTX_COUNT) ? "" : "x",
                   (i < MAX_VOUTMTX_COUNT) ? i : (i - MAX_VOUTMTX_COUNT));

            if (i < MAX_VOUTMTX_COUNT)
            {
                eStatus = VoutMod_Init(
                                &g_aoVoutParam[i].oVoutMod,
                                hDevice,
                                i,
                                g_aoVoutParam[i].ePixFmt,
                                g_aoVoutParam[i].bForceMode ? &g_aoVoutParam[i].oForcedSize : MNULL,
                                g_aoVoutParam[i].bForceMode ? g_aoVoutParam[i].uiForcedFps  : 0,
                                60,
                                1,
                                g_aoVoutParam[i].bIsMasterSync ? g_aoVoutParam[i].bControlQueueOverflow : MFALSE,
                                g_aoVoutParam[i].bIsMasterSync,
                                MFALSE);

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    eStatus = VoutMod_SetOutputFormat(
                                        &g_aoVoutParam[i].oVoutMod,
                                        g_aoVoutParam[i].eOutPixFmt,
                                        g_aoVoutParam[i].bOutRgbFullYuv709);
                }

                g_aoVoutParam[i].oSize.iWidth   = g_aoVoutParam[i].oVoutMod.oVidParam.uiHActive;
                g_aoVoutParam[i].oSize.iHeight  = g_aoVoutParam[i].oVoutMod.oVidParam.uiVActive;
                uiFreq                          = g_aoVoutParam[i].oVoutMod.oVidParam.uiVRate_Hz;
            }
            else
            {
              #ifdef LINUX
                eStatus = VoutGlxMod_Init(
                                &g_aoVoutParam[i].oVoutGlxMod,
                                hDevice,
                                bForceWinRect ? &oWinRect : MNULL,
                                60,
                                1,
                                g_aoVoutParam[i].bControlQueueOverflow);

                g_aoVoutParam[i].oSize = g_aoVoutParam[i].oVoutGlxMod.oWindowSize;
                uiFreq                 = 60;
               #else
                assert(MFALSE);
               #endif
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                g_aoVoutParam[i].bVoutEnabled = MTRUE;
            }
            else
            {
                MsgLog(0, "Cannot init Vout[%s%u]. Try to continue without...",
                          (i < MAX_VOUTMTX_COUNT) ? "" : "x",
                          (i < MAX_VOUTMTX_COUNT) ? i : (i - MAX_VOUTMTX_COUNT));

                g_aoVoutParam[i].bVoutEnabled = MFALSE;
                eStatus = LStatus_OK;

                if (g_aoVoutParam[i].bForceMode)
                {
                    g_aoVoutParam[i].oSize = g_aoVoutParam[i].oForcedSize;
                    uiFreq                 = g_aoVoutParam[i].uiForcedFps;
                }
                else
                {
                    g_aoVoutParam[i].oSize.iWidth  = 1920;
                    g_aoVoutParam[i].oSize.iHeight = 1080;
                    uiFreq                         = 60;
                }
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                LBuffer_Attributes oDispBufAttrib;

                oDispBufAttrib.eAttributeType                   = LBuffer_Type_VIDEO;
                oDispBufAttrib.oVideoAttributes.ePixelFormat    = g_aoVoutParam[i].ePixFmt;
                oDispBufAttrib.oVideoAttributes.uiWidth         = g_aoVoutParam[i].oSize.iWidth;
                oDispBufAttrib.oVideoAttributes.uiHeight        = g_aoVoutParam[i].oSize.iHeight;

                printf("Video mode: %ux%u@%uHz\n",
                       g_aoVoutParam[i].oSize.iWidth, g_aoVoutParam[i].oSize.iHeight, uiFreq);

                eStatus = MosMod_Init(
                            &g_aoVoutParam[i].oMosMod,
                            hDevice,
                            uiBufferCount,
                            &oDispBufAttrib,
                            MNULL,
                            (uiDemoIdx == 1) ? DemoA_IncrementAngularPos : MNULL,
                            MFALSE,
                            MFALSE);
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                SetStreamDstRect(i, &g_aoVoutParam[i].oSize);
            }

            if (LSTATUS_IS_FAIL(eStatus))
            {
                printf("***** Init Vout[%s%u] *****: FAILED (status= %d - %s)\n",
                       (i < MAX_VOUTMTX_COUNT) ? "" : "x",
                       (i < MAX_VOUTMTX_COUNT) ? i : (i - MAX_VOUTMTX_COUNT),
                       eStatus, GetStatusStr(eStatus));
            }
        }
    }

    MUINT uiStreamCount = 0;

    for (i = 0; i < MAX_STREAM_COUNT; i++)
    {
        if ((g_aoStreamParam[i].bEnable || g_aoAudioStreamParam[i].bEnable)
             && LSTATUS_IS_SUCCESS(eStatus))
        {
            printf("***** Init Stream[%u] *****\n", i);

            if (uiDemoIdx == 1)
            {
                const MUINT auiAngularPos[3] = {0, 180, 250};

                if (uiStreamCount == 0)
                {
                    g_aoStreamParam[i].oDstRect.iLeft   = 0;
                    g_aoStreamParam[i].oDstRect.iTop    = 0;
                    g_aoStreamParam[i].oDstRect.iRight  = g_aoStreamParam[i].oFrameSize.iWidth;
                    g_aoStreamParam[i].oDstRect.iBottom = g_aoStreamParam[i].oFrameSize.iHeight;
                }
                else
                {
                    g_aoStreamParam[i].fAlpha            = (MFLOAT32)uiTransparency / 100;
                    g_aoStreamParam[i].iAngularSpeed     = iAngularSpeed;
                    g_aoStreamParam[i].uiAngularPos1000x = auiAngularPos[(uiStreamCount - 1) % 3] * 1000;

                    GetRectOnEllipsoidalPath(
                        g_aoStreamParam[i].uiAngularPos1000x,
                        &g_oDemo1BackSize,
                        &g_oDemo1BorderSize,
                        g_uiDemo1CornerFactor,
                        &g_aoStreamParam[i].oDstRect);
                }
            }

            uiStreamCount++;

            while (MTRUE)
            {
                if ((uiBigDecodeCount > 2)
                    && (g_aoStreamParam[i].oFrameSize.iWidth >= g_oBigDecodePictureSize.iWidth)
                    && (g_aoStreamParam[i].oFrameSize.iHeight >= g_oBigDecodePictureSize.iHeight))
                {
                    g_aoStreamParam[i].bReduceLocalMemUsage = MTRUE;
                    printf("Stream[%u]: Reduced local memory usaged.\n", i);
                }

                eStatus = InitStream(
                            hDevice, 
                            g_aoStreamParam + i, 
                            g_aoAudioStreamParam + i,
                            uiDemoIdx, 
                            uiBufferCount,
                            uiMaxMtxAoutCount);

                if (eStatus == LStatus_CONNECTION_LOST)
                {
                    printf("\n");
                    printf("Cannot connect to the server. Press 'q' to stop retrying.\n");

                    MUINT s;

                    for (s = 5; s > 0; s--)
                    {
                        printf("Retry in %u seconds...\n", s);
                        usleep(1*1000*1000);

                        if ((GetChar() == 'q') || g_bSigTerm)
                        {
                            break;
                        }
                    }

                    if (s > 0)
                    {
                        break;
                    }
                }
                else
                {
                    break;
                }
            }

            if (LSTATUS_IS_FAIL(eStatus))
            {
                printf(
                    "***** Init Stream[%u] *****: FAILED! (status= %d - %s)\n",
                    i,
                    eStatus,
                    GetStatusStr(eStatus));
            }
        }
    }

    for (i = 0; (i < MAX_VOUT_COUNT) && LSTATUS_IS_SUCCESS(eStatus); i++)
    {
        VoutParam* poVoutParam = g_aoVoutParam + i;

        if (poVoutParam->bNetServer && (poVoutParam->uiInputCount > 0))
        {
            printf("***** Init rtsp server Vout[%s%u] *****\n",
                   (i < MAX_VOUTMTX_COUNT) ? "" : "x",
                   (i < MAX_VOUTMTX_COUNT) ? i : (i - MAX_VOUTMTX_COUNT));

            MUINT uiFpsNum = 0;
            MUINT uiFpsDen = 0;
            MUINT uiMaxNaluSizeBytes = (poVoutParam->oSize.iWidth
                                        * poVoutParam->oSize.iHeight
                                        * LPixelFormat_Get3xBppAllPlanes(poVoutParam->eEncodePixFmt))
                                       / (3 * 8);

            eStatus = MosMod_GetOutputFrameRate(&poVoutParam->oMosMod, &uiFpsNum, &uiFpsDen);

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                LRECT32             oSrcRect;
                LPOS                oDstPos;
                LBuffer_Attributes  oInEncBufAttrib;
                LSIZE               oPaddedSize;

                oSrcRect.iLeft   = 0;
                oSrcRect.iTop    = 0;
                oSrcRect.iRight  = poVoutParam->oSize.iWidth;
                oSrcRect.iBottom = poVoutParam->oSize.iHeight;
                oDstPos.iX       = 0;
                oDstPos.iY       = 0;

                EncMod_GetPaddedFrameSize(&poVoutParam->oSize, &oPaddedSize);

                oInEncBufAttrib.eAttributeType                = LBuffer_Type_VIDEO;
                oInEncBufAttrib.oVideoAttributes.ePixelFormat = poVoutParam->eEncodePixFmt;
                oInEncBufAttrib.oVideoAttributes.uiWidth      = oPaddedSize.iWidth;
                oInEncBufAttrib.oVideoAttributes.uiHeight     = oPaddedSize.iHeight;

                eStatus = RotMod_Init(
                                &poVoutParam->oRotMod,
                                hDevice,
                                16,
                                &oInEncBufAttrib,
                                &poVoutParam->oSize,
                                &oSrcRect,
                                &oDstPos,
                                LVPC_Rotation_NONE);
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                LBuffer_Attributes oOutEncBufAttrib;

                oOutEncBufAttrib.oSystemLinearAttributes.eAttributeType = LBuffer_Type_SYSTEM_LINEAR;
                oOutEncBufAttrib.oSystemLinearAttributes.uiSize         = uiMaxNaluSizeBytes;
                oOutEncBufAttrib.oSystemLinearAttributes.pvSystemMemory = MNULL;

                eStatus = EncMod_Init(
                                &poVoutParam->oEncMod,
                                hDevice,
                                16,
                                &oOutEncBufAttrib,
                                poVoutParam->eEncodeMode,
                                poVoutParam->eEncodeProfile,
                                poVoutParam->eEncodeLevel,
                                poVoutParam->uiSliceCount,
                                poVoutParam->uiFaq,
                                poVoutParam->bDfEnabled);

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    eStatus = EncMod_SetParam(
                                &poVoutParam->oEncMod,
                                &poVoutParam->oSize,
                                uiFpsNum,
                                uiFpsDen,
                                poVoutParam->uiTargetBitRateKbs,
                                poVoutParam->uiIPeriod,
                                poVoutParam->uiPPeriod,
                                poVoutParam->uiQpMin,
                                poVoutParam->uiQpMax);
                }
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                char szFolder[10];

                snprintf(szFolder, sizeof(szFolder), "S%u", i);

                eStatus = RtspSrvMod_Init(&poVoutParam->oRtspSrvMod, hDevice);

                if(LSTATUS_IS_SUCCESS(eStatus))
                {
                    eStatus = RtspSrvMod_SetParams(
                                    &poVoutParam->oRtspSrvMod,
                                    uiMaxNaluSizeBytes,
                                    0,
                                    szFolder,
                                    MNULL,
                                    MTRUE,
                                    0,
                                    MFALSE,
                                    0,
                                    0,
                                    LNetStreamer_Protocol_RTSP,
                                    MNULL,
                                    MNULL,
                                    MNULL,
                                    0,
                                    MFALSE,
                                    0,
                                    0,
                                    MNULL,
                                    MNULL,
                                    MNULL,
                                    MFALSE,
                                    (LNetStreamer_SrtMode)(0));
                }
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                eStatus = ModLnk_Connect(&poVoutParam->oMosMod.oOutLink, &poVoutParam->oRotMod.oInLink);
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                eStatus = ModLnk_Connect(&poVoutParam->oRotMod.oOutLink, &poVoutParam->oEncMod.oInLink);
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                eStatus = ModLnk_Connect(&poVoutParam->oEncMod.oOutLink, &poVoutParam->oRtspSrvMod.oVideoInLink);
            }
            else
            {
                printf(
                    "***** Init rtsp server Vout[%u] *****: FAILED! (status= %d - %s)\n",
                    i,
                    eStatus,
                    GetStatusStr(eStatus));
            }
        }
    }

    //--------------------------------------------------------------------------------------------------------
    // Start stream

    // We must start from last to first module.

    for (i = 0; i < MAX_VOUT_COUNT; i++)
    {
        if ((g_aoVoutParam[i].uiInputCount > 0) && LSTATUS_IS_SUCCESS(eStatus))
        {
            printf("***** Start Vout[%s%u] *****\n",
                   (i < MAX_VOUTMTX_COUNT) ? "" : "x",
                   (i < MAX_VOUTMTX_COUNT) ? i : (i - MAX_VOUTMTX_COUNT));

            if (g_aoVoutParam[i].bVoutEnabled)
            {
                if (i < MAX_VOUTMTX_COUNT)
                {
                    eStatus = VoutMod_Start(&g_aoVoutParam[i].oVoutMod);
                }
                else
                {
               #ifdef LINUX
                    eStatus = VoutGlxMod_Start(&g_aoVoutParam[i].oVoutGlxMod);
               #endif
                }
            }

            if (g_aoVoutParam[i].bNetServer && LSTATUS_IS_SUCCESS(eStatus))
            {
                eStatus = RtspSrvMod_Start(&g_aoVoutParam[i].oRtspSrvMod);

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    eStatus = EncMod_Start(&g_aoVoutParam[i].oEncMod);
                }

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    eStatus = RotMod_Start(&g_aoVoutParam[i].oRotMod);
                }
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                eStatus = MosMod_Start(&g_aoVoutParam[i].oMosMod);
            }

            if (LSTATUS_IS_FAIL(eStatus))
            {
                printf(
                    "***** Start Vout[%u] *****: FAILED (status= %d - %s)\n",
                    i,
                    eStatus,
                    GetStatusStr(eStatus));
            }
        }
    }

    for (i = 0; i < MAX_STREAM_COUNT; i++)
    {
        if ((g_aoStreamParam[i].bEnable || g_aoAudioStreamParam[i].bEnable)
                && LSTATUS_IS_SUCCESS(eStatus))
        {
            printf("***** Start Stream[%u] *****\n", i);

            // Video branch
            if (g_aoStreamParam[i].bEnable)
            {
                if ((g_aoStreamParam[i].szDstFilePath != MNULL) && LSTATUS_IS_SUCCESS(eStatus))
                {
                    eStatus = WrFileMod_Start(&(g_aoStreamParam[i].oWrFileMod));
                }

                if (g_aoStreamParam[i].bDecode && LSTATUS_IS_SUCCESS(eStatus))
                {
                    eStatus = DecMod_Start(&(g_aoStreamParam[i].oDecodeMod));
                }
            }

            // Audio branch
            if (g_aoAudioStreamParam[i].bEnable && LSTATUS_IS_SUCCESS(eStatus))
            {
                if (g_aoAudioStreamParam[i].bEnableOutput)
                {
                    if (g_aoAudioStreamParam[i].bMtxAout)
                    {
                        eStatus = AoutMod_Start(&(g_aoAudioStreamParam[i].oAoutMod));
                    }
                    else
                    {
                        eStatus = AoutAlsaMod_Start(&(g_aoAudioStreamParam[i].oAoutAlsaMod));
                    }
                }
                else
                {
                    eStatus = WrFileMod_Start(&(g_aoAudioStreamParam[i].oDummySinkMod));
                }

                if(LSTATUS_IS_SUCCESS(eStatus))
                {
                    eStatus = AudioDecMod_Start(&(g_aoAudioStreamParam[i].oAudioDecMod));
                }
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                if (g_aoStreamParam[i].szSrcFilePath != MNULL)
                {
                    eStatus = RdFileMod_Start(&(g_aoStreamParam[i].oRdFileMod));
                }
                else
                {
                    eStatus = RtspClnMod_Start(&(g_aoStreamParam[i].oRtspClnMod));
                }
            }

            if (LSTATUS_IS_FAIL(eStatus))
            {
                printf(
                    "***** Start Stream[%u] *****: FAILED (status= %d - %s)\n",
                    i,
                    eStatus,
                    GetStatusStr(eStatus));
            }
        }
    }

    //--------------------------------------------------------------------------------------------------------
    // Running

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        printf("***** Started *****\n");
        ShowInputKeyOptions();

        MUINT   iShowStatIdx          = ~0;
        MBOOL   bForceShow            = MFALSE;
        int     iChar                 = EOF;
        MUINT64 uiPrevBusyTimeCpu     = 0;
        MUINT64 uiPrevTotalTimeCpu    = 0;
        MBOOL   bRunning              = MTRUE;
        MBOOL   bForceFullScreenCycle = MFALSE;
        MBOOL   bPaused               = MFALSE;
        MBOOL   abStopped[MAX_STREAM_COUNT];
        MBOOL   abLostConnection[MAX_STREAM_COUNT];

        for (i = 0; i < MAX_STREAM_COUNT; i++)
        {
            abStopped[i]        = MFALSE;
            abLostConnection[i] = MFALSE;
        }

        while ((iChar != 'q') && !bRestart && bRunning && !g_bSigTerm)
        {
            bRunning = MFALSE;
            MBOOL bLostConnection = MFALSE;

            for (i = 0; i < MAX_VOUT_COUNT; i++)
            {
                if (g_aoVoutParam[i].uiInputCount > 0)
                {
                    if (g_aoVoutParam[i].bNetServer
                        && !g_aoVoutParam[i].bSrvAddrDisplayed
                        && g_aoVoutParam[i].oRtspSrvMod.bSessionStarted)
                    {
                        char szServAddr[512];

                        snprintf(szServAddr, sizeof(szServAddr),
                                 "Stream[%u] parameters: -i %s -w %u -h %u -fps %u/%u",
                                 i, g_aoVoutParam[i].oRtspSrvMod.szServerLocation,
                                 g_aoVoutParam[i].oSize.iWidth,
                                 g_aoVoutParam[i].oSize.iHeight,
                                 g_aoVoutParam[i].oEncMod.oSeqData.oFrameRate.uiNumerator,
                                 g_aoVoutParam[i].oEncMod.oSeqData.oFrameRate.uiDenominator);

                        MUINT j;
                        for (j = 0; j < strlen(szServAddr); j++)
                        {
                            printf("%c", '~');
                        }

                        printf("\n%s\n", szServAddr);

                        for (j = 0; j < strlen(szServAddr); j++)
                        {
                            printf("%c", '~');
                        }

                        printf("%s", "\n");

                        g_aoVoutParam[i].bSrvAddrDisplayed = MTRUE;
                    }

                    if ((uiDemoIdx >= DEMO_COUNT)
                        && ((g_aoVoutParam[i].uiCycleTimeSec > 0)
                            || bForceFullScreenCycle))
                    {
                        MUINT uiTimeSec = GetMonoTimeUsec() / (1000 * 1000);

                        if ((uiTimeSec >= g_aoVoutParam[i].uiNextCycleTimeSec)
                            || bForceFullScreenCycle)
                        {
                            MosMod_ShowOneInputFullScreen(
                                &g_aoVoutParam[i].oMosMod,
                                (g_aoVoutParam[i].uiNextCycleInputIdx < g_aoVoutParam[i].uiInputCount),
                                g_aoVoutParam[i].uiNextCycleInputIdx);

                            g_aoVoutParam[i].uiNextCycleTimeSec = uiTimeSec + g_aoVoutParam[i].uiCycleTimeSec;
                            g_aoVoutParam[i].uiNextCycleInputIdx++;

                            if (g_aoVoutParam[i].uiNextCycleInputIdx > g_aoVoutParam[i].uiInputCount)
                            {
                                g_aoVoutParam[i].uiNextCycleInputIdx = 0;
                            }
                        }
                    }
                }
            }

            bForceFullScreenCycle = MFALSE;

            for (i = 0; i < MAX_STREAM_COUNT; i++)
            {
                if (g_aoStreamParam[i].bEnable)
                {
                    if (iShowStatIdx == ~0)
                    {
                        iShowStatIdx = i;
                    }

                    MBOOL bShowNow = ((bShowStats || bForceShow) && (iShowStatIdx == i));

                    if (DoStats(g_aoStreamParam + i, i, bShowNow, &uiPrevBusyTimeCpu, &uiPrevTotalTimeCpu))
                    {
                        bForceShow = MFALSE;
                    }

                    if (g_aoStreamParam[i].bDecode)
                    {
                        ModuleThread* poVoutThread;

                        if (g_aoStreamParam[i].auiVoutIndex[0] < MAX_VOUTMTX_COUNT)
                        {
                            poVoutThread =
                                &(g_aoVoutParam[g_aoStreamParam[i].auiVoutIndex[0]].oVoutMod.oCpuThread);
                        }
                        else
                        {
                          #ifdef LINUX
                            poVoutThread =
                                &(g_aoVoutParam[g_aoStreamParam[i].auiVoutIndex[0]].oVoutGlxMod.oCpuThread);
                          #endif
                        }
   
                        if (ModThread_IsRunning(poVoutThread))
                        {
                            bRunning = MTRUE;
                        }

                        if (g_aoStreamParam[i].oRtspClnMod.bLostConnection)
                        {
                            if (!abLostConnection[i])
                            {
                                printf("Lost connection on stream[%u].\n", i);
                            }

                            bLostConnection     = MTRUE;
                            abLostConnection[i] = MTRUE;
                        }

                        if (!abStopped[i]
                            && !ModThread_IsRunning(&(g_aoStreamParam[i].oDecodeMod.oCpuThreadOut)))
                        {
                            printf("Stream[%u] stopped.\n", i);
                            abStopped[i] = MTRUE;
                        }
                    }

                    if (g_aoStreamParam[i].szDstFilePath != MNULL)
                    {
                        if (ModThread_IsRunning(&(g_aoStreamParam[i].oWrFileMod.oCpuThread)))
                        {
                            bRunning = MTRUE;
                        }
                        else if (!abStopped[i])
                        {
                            printf("Stream[%u] stopped.\n", i);
                            abStopped[i] = MTRUE;
                        }
                    }
                }
                else if(g_aoAudioStreamParam[i].bEnable)
                {
                    if (g_aoAudioStreamParam[i].bEnableOutput)
                    {
                        if(g_aoAudioStreamParam[i].bMtxAout)
                        {
                            bRunning = ModThread_IsRunning(&(g_aoAudioStreamParam[i].oAoutMod.oCpuThread));
                        }
                        else
                        {
                            bRunning = ModThread_IsRunning(&(g_aoAudioStreamParam[i].oAoutAlsaMod.oCpuThread));
                        }
                    }
                    else
                    {
                        bRunning = ModThread_IsRunning(&(g_aoAudioStreamParam[i].oDummySinkMod.oCpuThread));
                    }

                    if (g_aoAudioStreamParam[i].poRtspClnMod->bLostConnection)
                    {
                        if (!abLostConnection[i])
                        {
                            printf("Lost connection on audio stream[%u].\n", i);
                        }

                        bLostConnection     = MTRUE;
                        abLostConnection[i] = MTRUE;
                    }
                }
            }

            usleep(10*1000);
            iChar = GetChar();

            if (iChar != EOF)
            {
                printf(" > ");

                if (((iChar >= '0') && (iChar <= '9'))
                    || ((iChar >= 'a') && (iChar <= 'f')))
                {
                    if ((iChar >= '0') && (iChar <= '9'))
                    {
                        i = iChar - '0';
                    }
                    else
                    {
                        i = iChar - 'a' + 10;
                    }

                    if ((i < MAX_STREAM_COUNT)
                        && g_aoStreamParam[i].bEnable)
                    {
                        printf("Show statistics of Stream[%u].\n", i);
                        iShowStatIdx = i;
                        bForceShow   = MTRUE;
                    }
                    else
                    {
                        printf("Stream[%u] not enabled.\n", i);
                    }
                }
                else if (iChar == 'n')
                {
                    bForceFullScreenCycle = MTRUE;
                    printf("Cycle full screen video.\n");
                }
                else if (iChar == 'P')
                {
                    if (bPaused)
                    {
                        bPaused = MFALSE;
                        printf("Resume all streaming.\n");
                    }
                    else
                    {
                        bPaused = MTRUE;
                        printf("Pause all streaming.\n");
                    }

                    for (i = 0; i < MAX_STREAM_COUNT; i++)
                    {
                        g_aoStreamParam[i].oRdFileMod.oOutLink.bReturnAllBuffer = bPaused;
                        g_aoStreamParam[i].oRtspClnMod.oVideoOutLink.bReturnAllBuffer = bPaused;
                    }

                    for (i = 0; i < MAX_VOUT_COUNT; i++)
                    {
                        g_aoVoutParam[i].oMosMod.oOutLink.bReturnAllBuffer = bPaused;
                    }
                }
                else if ((iChar != 'q') && (iChar != 'r'))
                {
                    printf("Invalid key.\n");
                    ShowInputKeyOptions();
                }
            }

            bRestart = (iChar == 'r') || bLostConnection;
        }

        printf("***** Stopping... ******\n");
    }

    //--------------------------------------------------------------------------------------------------------
    // Stop modules

    // We must stop in the reverse order as the start i.e. from first to last module.

    for (i = 0; i < MAX_STREAM_COUNT; i++)
    {
        if (g_aoStreamParam[i].bEnable || g_aoAudioStreamParam[i].bEnable)
        {
            if (g_aoStreamParam[i].szSrcFilePath != MNULL)
            {
                RdFileMod_Stop(&(g_aoStreamParam[i].oRdFileMod));
            }
            else
            {
                RtspClnMod_Stop(&(g_aoStreamParam[i].oRtspClnMod));
            }

            if(g_aoStreamParam[i].bEnable)
            {
                if (g_aoStreamParam[i].bDecode)
                {
                    DecMod_Stop(&(g_aoStreamParam[i].oDecodeMod));
                }

                if (g_aoStreamParam[i].szDstFilePath != MNULL)
                {
                    WrFileMod_Stop(&(g_aoStreamParam[i].oWrFileMod));
                }
            }

            if(g_aoAudioStreamParam[i].bEnable)
            {
                AudioDecMod_Stop(&(g_aoAudioStreamParam[i].oAudioDecMod));

                if (g_aoAudioStreamParam[i].bEnableOutput)
                {
                    if(g_aoAudioStreamParam[i].bMtxAout)
                    {
                        AoutMod_Stop(&(g_aoAudioStreamParam[i].oAoutMod));
                    }
                    else
                    {
                        AoutAlsaMod_Stop(&(g_aoAudioStreamParam[i].oAoutAlsaMod));
                    }
                }
                else
                {
                    WrFileMod_Stop(&(g_aoAudioStreamParam[i].oDummySinkMod));
                }
            }
        }
    }

    for (i = 0; i < MAX_VOUT_COUNT; i++)
    {
        if (g_aoVoutParam[i].uiInputCount > 0)
        {
            MosMod_Stop(&g_aoVoutParam[i].oMosMod);

            if (g_aoVoutParam[i].bVoutEnabled)
            {
                if (i < MAX_VOUTMTX_COUNT)
                {
                    VoutMod_Stop(&g_aoVoutParam[i].oVoutMod);
                }
                else
                {
                   #ifdef LINUX
                    VoutGlxMod_Stop(&g_aoVoutParam[i].oVoutGlxMod);
                   #endif
                }
            }

            if (g_aoVoutParam[i].bNetServer)
            {
                RotMod_Stop(&g_aoVoutParam[i].oRotMod);
                EncMod_Stop(&g_aoVoutParam[i].oEncMod);
                RtspSrvMod_Stop(&g_aoVoutParam[i].oRtspSrvMod);
            }
        }
    }

    printf("***** Stopped ******\n");

    //--------------------------------------------------------------------------------------------------------
    // Cleanup modules

    for (i = 0; i < MAX_STREAM_COUNT; i++)
    {
        RdFileMod_Cleanup(&(g_aoStreamParam[i].oRdFileMod));
        RtspClnMod_Cleanup(&(g_aoStreamParam[i].oRtspClnMod));
        DecMod_Cleanup(&(g_aoStreamParam[i].oDecodeMod));
        WrFileMod_Cleanup(&(g_aoStreamParam[i].oWrFileMod));

        AoutMod_Cleanup(&(g_aoAudioStreamParam[i].oAoutMod));
        AoutAlsaMod_Cleanup(&(g_aoAudioStreamParam[i].oAoutAlsaMod));
        AudioDecMod_Cleanup(&(g_aoAudioStreamParam[i].oAudioDecMod));
        WrFileMod_Cleanup(&(g_aoAudioStreamParam[i].oDummySinkMod));
    }

    for (i = 0; i < MAX_VOUT_COUNT; i++)
    {
        MosMod_Cleanup(&g_aoVoutParam[i].oMosMod);
        RotMod_Cleanup(&g_aoVoutParam[i].oRotMod);
        EncMod_Cleanup(&g_aoVoutParam[i].oEncMod);
        RtspSrvMod_Cleanup(&g_aoVoutParam[i].oRtspSrvMod);
        VoutMod_Cleanup(&g_aoVoutParam[i].oVoutMod);
       #ifdef LINUX
        VoutGlxMod_Cleanup(&g_aoVoutParam[i].oVoutGlxMod);
       #endif
    }

    if (bRestart)
    {
        printf("***** Restarting ******\n");

        // Patch asynchronious destroy problems: If we create an other decode context, it is possible this
        // destroy is not finished yet and the decoding could fail. So, wait a second.
        usleep(1*1000*1000);

        goto Restart;
    }

    Liberatus_UnLoad();

    return LSTATUS_IS_SUCCESS(eStatus) ? 0 : -1;
}
