/************************************************************************************************************\

Module Name:    WriteFileModule.c

Description:    .

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "WriteFileModule.h"

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

static const    MCHAR8      g_szModuleNameBase[]    = "WrFile";
static          MUINT32     g_uiWrFileModCount      = 0;

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Function:       WrFileMod_Init

Description:    .

\************************************************************************************************************/
LStatus WrFileMod_Init(
            WriteFileModule* poWrFileMod,
            LDevice_Handle   hDevice,
            const char*      szFilePath,
            MBOOL            bAppend)
{
    MsgLog(2, "{...");

    WrFileMod_Cleanup(poWrFileMod);

    LStatus eStatus = ((poWrFileMod != MNULL)
                       && (hDevice != MNULL)
                       && (szFilePath != MNULL))
                      ? LStatus_OK : LStatus_INVALID_PARAM;

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        poWrFileMod->bInit   = MTRUE;
        poWrFileMod->bAppend = bAppend;

        strncpy_wz(poWrFileMod->szFilePath, szFilePath, sizeof(poWrFileMod->szFilePath));
        snprintf(
                    poWrFileMod->szModuleName,
                    sizeof(poWrFileMod->szModuleName),
                    "%s%d",
                    g_szModuleNameBase,
                    g_uiWrFileModCount);
    }
    else
    {
        WrFileMod_Cleanup(poWrFileMod);
    }

    ++g_uiWrFileModCount;

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       WrFileMod_Cleanup

Description:    .

\************************************************************************************************************/
void WrFileMod_Cleanup(WriteFileModule* poWrFileMod)
{
    MsgLog(2, "{...");

    poWrFileMod->oInLink.poModLnk = MNULL;

    MsgLog(2, "...}");
}

/************************************************************************************************************\

Function:       WrFileMod_CpuThread

Description:    .

\************************************************************************************************************/
LStatus WrFileMod_CpuThread(void* pvData)
{
    if (pvData == MNULL)
    {
        MsgLogErr("ERROR! NULL data.");
        return LStatus_INVALID_PARAM;
    }

    WriteFileModule* poWrFileMod     = (WriteFileModule*)pvData;
    FILE*            hFile           = MNULL;
    const MUINT      uiTmpBufferSize = 1*1024*1024;
    char*            puiTmpBuffer    = MNULL;
    MBOOL            bStub           = (strcmp(poWrFileMod->szFilePath, "/dev/null") == 0);

    ModThread_SetName(poWrFileMod->szModuleName);
    MsgLog(2, "Start thread %p.", pthread_self());

    if (bStub)
    {
        MsgLog(2, "NULL file -> No writting.");
    }
    else
    {
        hFile = fopen(poWrFileMod->szFilePath, poWrFileMod->bAppend ? "wab" : "wb");

        if (hFile == MNULL)
        {
            MsgLogErr("ERROR! Cannot open file: %s", poWrFileMod->szFilePath);
            poWrFileMod->oCpuThread.bKillThread = MTRUE;
        }


        puiTmpBuffer = malloc(uiTmpBufferSize);

        if (puiTmpBuffer != NULL)
        {
            if (setvbuf(hFile, puiTmpBuffer, _IOFBF, uiTmpBufferSize) != 0)
            {
                MsgLogErr("WARNING! Cannot enable file buffering. Writing performance will be low...");
            }
        }
        else
        {
            MsgLogErr("WARNING! Cannot allocate buffer file. Writing performance will be low...");
        }
    }

    while (!poWrFileMod->oCpuThread.bKillThread)
    {
        BufferInfo* poSrcBuffer;
        LStatus     eStatus
            = ModLnkIn_GetSubmittedBuffer(&(poWrFileMod->oInLink), 100, 0, MNULL, &poSrcBuffer, MNULL, MNULL);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            if (poSrcBuffer->bEndOfStream)
            {
                MsgLog(4, "END-OF-STREAM");

                poWrFileMod->oCpuThread.bKillThread = MTRUE;
            }
            else if (!bStub)
            {
                LBuffer_PlaneInfo oPlaneInfo;

                eStatus = LBuffer_GetPlaneInfo(poSrcBuffer->hBuffer, &oPlaneInfo);

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    MUINT8* apuiBuffer[LBUFFER_MAXPLANE];

                    MsgLog(4, "Write Buffer[%u] at offset %ld...", poSrcBuffer->uiId, ftell(hFile));

                    eStatus = LBuffer_BeginAccess(
                                    poSrcBuffer->hBuffer,
                                    0,
                                    oPlaneInfo.uiPlaneCount,
                                    apuiBuffer);

                    if (LSTATUS_IS_SUCCESS(eStatus))
                    {
                        MUINT uiOffset  = poSrcBuffer->uiStartOffset;
                        MUINT uiSize    = poSrcBuffer->uiSizeBytes;

                        if ((uiSize > 0) && ((uiOffset + uiSize) <= oPlaneInfo.auiPitchInBytes[0]))
                        {
                            if (fwrite(apuiBuffer[0] + uiOffset, 1, uiSize, hFile) != uiSize)
                            {
                                MsgLogErr("ERROR! Cannot write linear buffer to file!");
                            }
                        }
                        else
                        {
                            MUINT i;
                            MUINT y;

                            for (i = 0; i < oPlaneInfo.uiPlaneCount; i++)
                            {
                                MUINT8* puiBuffer = apuiBuffer[i];

                                for (y = 0; y < oPlaneInfo.auiHeight[i]; y++)
                                {
                                    if (fwrite(puiBuffer, 1, oPlaneInfo.auiWidthInBytes[i], hFile)
                                        != oPlaneInfo.auiWidthInBytes[i])
                                    {
                                        MsgLogErr("ERROR! Cannot write buffer to file.");
                                        break;
                                    }

                                    puiBuffer += oPlaneInfo.auiPitchInBytes[i];
                                }
                            }
                        }

                        LBuffer_EndAccess(poSrcBuffer->hBuffer);
                    }
                    else
                    {
                        MsgLogErr("ERROR! Cannot access the buffer.");
                    }

                    MsgLog(4, "Write Buffer[%u] done.", poSrcBuffer->uiId);
                }
            }

            ModLnkIn_ReturnBuffer(&(poWrFileMod->oInLink), poSrcBuffer, MNULL, NO_TAG);
        }

        if (LSTATUS_IS_FAIL(eStatus)
            && (eStatus != LStatus_TIMEOUT))
        {
            usleep(1000);
        }
    }

    if (hFile != MNULL)
    {
        fclose(hFile);
    }

    if (puiTmpBuffer != NULL)
    {
        free(puiTmpBuffer);
    }

    MsgLog(2, "Kill thread.");

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       WrFileMod_Start

Description:    .

\************************************************************************************************************/
LStatus WrFileMod_Start(WriteFileModule* poWrFileMod)
{
    MsgLog(2, "{...");

    LStatus eStatus = LStatus_INVALID_PARAM;

    if ((poWrFileMod != MNULL)
        && poWrFileMod->bInit)
    {
        if (poWrFileMod->oInLink.poModLnk != MNULL)
        {
            eStatus = ModThread_Start(&(poWrFileMod->oCpuThread), poWrFileMod, WrFileMod_CpuThread);
        }
        else
        {
            MsgLogErr("ERROR! Bad connection.");
            eStatus = LStatus_FAIL;
        }
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       WrFileMod_Stop

Description:    .

\************************************************************************************************************/
void WrFileMod_Stop(WriteFileModule* poWrFileMod)
{
    MsgLog(2, "{...");

    if (poWrFileMod != MNULL)
    {
        ModThread_Stop(&(poWrFileMod->oCpuThread));
    }

    MsgLog(2, "...}");
}

