/************************************************************************************************************\

Module Name:    VpeModule.c

Description:    .

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "VpeModule.h"

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

static const    MCHAR8      g_szModuleNameBase[]    = "Vpe";
static          MUINT32     g_uiVpeModCount = 0;

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Function:       VpeMod_Init

Description:    .

\************************************************************************************************************/
LStatus VpeMod_Init(
            VpeModule*              poVpeMod,
            LDevice_Handle          hDevice,
            MUINT                   uiOutBufferCount,
            LBuffer_Attributes*     poOutBufferAttributes,
            MBOOL                   bInterlaced)
{
    MsgLog(2, "{...");

    VpeMod_Cleanup(poVpeMod);

    LStatus eStatus = ((poVpeMod != MNULL)
                       && (hDevice != MNULL)
                       && (poOutBufferAttributes != MNULL)
                       && (poOutBufferAttributes->eAttributeType == LBuffer_Type_VIDEO))
                      ? LStatus_OK : LStatus_INVALID_PARAM;

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LDeviceThread_CreateAttributes oAttributes = {LDeviceThread_AttributeType_CREATE, MTRUE, MTRUE, MFALSE, "vpemod1"};

        eStatus = LDeviceThread_Create(hDevice, &oAttributes.eType, &(poVpeMod->hDevThread));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LDeviceThread_CreateAttributes oAttributes = {LDeviceThread_AttributeType_CREATE, MTRUE, MTRUE, MFALSE, "vpemod2"};

        eStatus = LDeviceThread_Create(hDevice, &oAttributes.eType, &(poVpeMod->hDevThread2));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LVPE_Create(hDevice, poVpeMod->hDevThread, MNULL, &(poVpeMod->hVpe));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LVPE_Create(hDevice, poVpeMod->hDevThread2, MNULL, &(poVpeMod->hVpe2));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LVPC_Create(hDevice, poVpeMod->hDevThread, MNULL, &(poVpeMod->hVpc));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        snprintf(
                    poVpeMod->szModuleName,
                    sizeof(poVpeMod->szModuleName),
                    "%s%d",
                    g_szModuleNameBase,
                    g_uiVpeModCount);

        eStatus = ModLnk_Init(
                        &(poVpeMod->oOutLink),
                        hDevice,
                        uiOutBufferCount,
                        poOutBufferAttributes,
                        MFALSE,
                        0,
                        poVpeMod->szModuleName);
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LVP_ColorSpace oColorSpace;

        oColorSpace.eFunction    = LVP_TransferFunction_Other;
        oColorSpace.eMatrix      = LVP_TransferMatrix_BT709;
        oColorSpace.poUserMatrix = MNULL;

        eStatus = LVPE_SetSourceParam(
                    poVpeMod->hVpe,
                    LVPE_Source_COLOR_SPACE,
                    sizeof(LVP_ColorSpace),
                    &oColorSpace);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LVPE_SetSourceParam(
                        poVpeMod->hVpe2,
                        LVPE_Source_COLOR_SPACE,
                        sizeof(LVP_ColorSpace),
                        &oColorSpace);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LVPC_SetSourceParam(
                        poVpeMod->hVpc,
                        0,
                        LVPC_Source_COLOR_SPACE,
                        sizeof(LVP_ColorSpace),
                        &oColorSpace);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LVPE_SetTargetParam(
                        poVpeMod->hVpe,
                        LVPE_Target_COLOR_SPACE,
                        sizeof(LVP_ColorSpace),
                        &oColorSpace);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LVPE_SetTargetParam(
                        poVpeMod->hVpe2,
                        LVPE_Target_COLOR_SPACE,
                        sizeof(LVP_ColorSpace),
                        &oColorSpace);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LVPC_SetTargetParam(
                        poVpeMod->hVpc,
                        LVPC_Target_COLOR_SPACE,
                        sizeof(LVP_ColorSpace),
                        &oColorSpace);

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                LVPC_CompositorColorSpace oCompColSpace;

                oCompColSpace.bYCbCrA
                    = LPixelFormat_IsRgb(poOutBufferAttributes->oVideoAttributes.ePixelFormat)
                      ? MFALSE : MTRUE;
                oCompColSpace.oColorSpace = oColorSpace;

                eStatus = LVPC_SetCompositorParam(
                            poVpeMod->hVpc,
                            LVPC_Compositor_COLOR_SPACE,
                            sizeof(LVPC_CompositorColorSpace),
                            &oCompColSpace);
            }
        }
    }

    if (bInterlaced)
    {
        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            LVPE_SourceOperation oSrcOp;

            oSrcOp.eOperationType = LVPE_OperationType_MADI;

            eStatus = LVPE_SetSourceParam(
                        poVpeMod->hVpe,
                        LVPE_Source_OPERATION,
                        sizeof(oSrcOp),
                        &oSrcOp);

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                LVP_SourceFrameFormat oSrcFrmFmt;

                oSrcFrmFmt.eFrameFormat = LVP_FrameFormat_SINGLE_TOP_FIELD;

                eStatus = LVPE_SetSourceParam(
                            poVpeMod->hVpe,
                            LVPE_Source_FRAME_FORMAT,
                            sizeof(oSrcFrmFmt),
                            &oSrcFrmFmt);
            }
        }

        poVpeMod->bDeinterlace = MTRUE;
    }
    else
    {
        poVpeMod->bDeinterlace = MFALSE;
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LColor oColor;

        oColor.ePixelFormat = LPixelFormat_B8G8R8A8;
        oColor.uiAlpha      = 0;
        oColor.uiRed        = 128;
        oColor.uiGreen      = 128;
        oColor.uiBlue       = 128;

        MUINT i;
        for (i = 0; i < poVpeMod->oOutLink.uiBufferCount; i++)
        {
            // Don't care if the clear has failed (it will with packed pixel formats).
            ClearBuffer(hDevice, poVpeMod->oOutLink.aoBufferInfo[i].hBuffer, &oColor);
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        poVpeMod->oBufferSize.iWidth    = poOutBufferAttributes->oVideoAttributes.uiWidth;
        poVpeMod->oBufferSize.iHeight   = poOutBufferAttributes->oVideoAttributes.uiHeight;
        poVpeMod->oSrcRect.bEnable      = MFALSE;
        poVpeMod->oDstRect.bEnable      = MFALSE;
        poVpeMod->oClipRect.bEnable     = MFALSE;
    }
    else
    {
        VpeMod_Cleanup(poVpeMod);
    }

    ++g_uiVpeModCount;

    MsgLog(
        2,
        "...} (status= %d - %s)",
        eStatus,
        GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       VpeMod_Cleanup

Description:    .

\************************************************************************************************************/
void VpeMod_Cleanup(VpeModule* poVpeMod)
{
    MsgLog(2, "{...");

    if (poVpeMod != MNULL)
    {
        ModLnk_Cleanup(&(poVpeMod->oOutLink));

        if (poVpeMod->hVpe != MNULL)
        {
            while (LSTATUS_IS_FAIL(LVPE_Destroy(poVpeMod->hVpe)))
            {
                MsgLog(0, "LVPE_Destroy failed! Retry...");
                usleep(1000*1000);
            }
        }

        if (poVpeMod->hVpe2 != MNULL)
        {
            while (LSTATUS_IS_FAIL(LVPE_Destroy(poVpeMod->hVpe2)))
            {
                MsgLog(0, "LVPE_Destroy failed! Retry...");
                usleep(1000*1000);
            }
        }

        if (poVpeMod->hVpc != MNULL)
        {
            while (LSTATUS_IS_FAIL(LVPC_Destroy(poVpeMod->hVpc)))
            {
                MsgLog(0, "LVPC_Destroy failed! Retry...");
                usleep(1000*1000);
            }
        }

        if (poVpeMod->hDevThread != MNULL)
        {
            LDeviceThread_Destroy(poVpeMod->hDevThread);
        }

        if (poVpeMod->hDevThread2 != MNULL)
        {
            LDeviceThread_Destroy(poVpeMod->hDevThread2);
        }

        poVpeMod->hVpe              = MNULL;
        poVpeMod->hVpe2             = MNULL;
        poVpeMod->hVpc              = MNULL;
        poVpeMod->hDevThread        = MNULL;
        poVpeMod->hDevThread2       = MNULL;
        poVpeMod->oInLink.poModLnk  = MNULL;
    }

    MsgLog(2, "...}");
}

/************************************************************************************************************\

Function:       VpeMod_SetRectangles

Description:    .

\************************************************************************************************************/
LStatus VpeMod_SetRectangles(
    VpeModule*     poVpeMod,
    const LRECT32* poSrcRect,
    const LRECT32* poDstRect,
    const LRECT32* poClipRect)
{
    LStatus eStatus = LStatus_INVALID_PARAM;

    MsgLog(2, "{...");

    if ((poVpeMod != MNULL)
        && (poVpeMod->hVpe != MNULL)
        && (poVpeMod->hVpe2 != MNULL))
    {
        eStatus = LStatus_OK;

        if (poSrcRect != MNULL)
        {
            poVpeMod->oSrcRect.bEnable = MTRUE;
            poVpeMod->oSrcRect.oRect   = *poSrcRect;
        }
        else
        {
            poVpeMod->oSrcRect.bEnable = MFALSE;
        }

        if (poDstRect != MNULL)
        {
            poVpeMod->oDstRect.bEnable = MTRUE;
            poVpeMod->oDstRect.oRect   = *poDstRect;
        }
        else
        {
            poVpeMod->oDstRect.bEnable = MFALSE;
        }

        if (poClipRect != MNULL)
        {
            poVpeMod->oClipRect.bEnable = MTRUE;
            poVpeMod->oClipRect.oRect   = *poClipRect;
        }
        else
        {
            poVpeMod->oClipRect.bEnable = MFALSE;
        }

        // Since VPC is 2x faster than VPE but cannot scale, use it if source:destination ratio is 1:1.
        // Also, split vpe if frame size is larger than 6M in order to make 4K@60fps.
        const MUINT uiSplitSizeThreshold = 6 * 1024 * 1024;

        poVpeMod->bUseVpc   = MFALSE;
        poVpeMod->bSplitVpe = MFALSE;
        poVpeMod->oSrcRect2 = poVpeMod->oSrcRect;
        poVpeMod->oDstRect2 = poVpeMod->oDstRect;

        if (poVpeMod->oSrcRect.bEnable
            && poVpeMod->oDstRect.bEnable
            && !poVpeMod->bDeinterlace)
        {
            LSIZE oSrcSize;
            LSIZE oDstSize;

            oSrcSize.iWidth  = poVpeMod->oSrcRect.oRect.iRight - poVpeMod->oSrcRect.oRect.iLeft;
            oSrcSize.iHeight = poVpeMod->oSrcRect.oRect.iBottom - poVpeMod->oSrcRect.oRect.iTop;
            oDstSize.iWidth  = poVpeMod->oDstRect.oRect.iRight - poVpeMod->oDstRect.oRect.iLeft;
            oDstSize.iHeight = poVpeMod->oDstRect.oRect.iBottom - poVpeMod->oDstRect.oRect.iTop;

            if ((oSrcSize.iWidth == oDstSize.iWidth)
                && (oSrcSize.iHeight == oDstSize.iHeight))
            {
                poVpeMod->bUseVpc = MTRUE;
            }
            else if (((oSrcSize.iWidth * oSrcSize.iHeight) > uiSplitSizeThreshold)
                     || ((oDstSize.iWidth * oDstSize.iHeight) > uiSplitSizeThreshold))
            {
                poVpeMod->bSplitVpe = MTRUE;

                poVpeMod->oSrcRect.oRect.iBottom   -= oSrcSize.iHeight / 2;
                poVpeMod->oSrcRect2.oRect.iTop      = poVpeMod->oSrcRect.oRect.iBottom;
                poVpeMod->oDstRect.oRect.iBottom   -= oDstSize.iHeight / 2;
                poVpeMod->oDstRect2.oRect.iTop      = poVpeMod->oDstRect.oRect.iBottom;
            }
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LVPE_SetSrcRect(poVpeMod->hVpe, &poVpeMod->oSrcRect);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LVPE_SetSrcRect(poVpeMod->hVpe2, &poVpeMod->oSrcRect2);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LVPC_SetSrcRect(poVpeMod->hVpc, 0, &poVpeMod->oSrcRect);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LVPE_SetDstRect(poVpeMod->hVpe, &poVpeMod->oDstRect);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LVPE_SetDstRect(poVpeMod->hVpe2, &poVpeMod->oDstRect2);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LVPC_SetDstRect(poVpeMod->hVpc, 0, &poVpeMod->oDstRect);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LVPE_SetClipRect(poVpeMod->hVpe, &poVpeMod->oClipRect);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LVPE_SetClipRect(poVpeMod->hVpe2, &poVpeMod->oClipRect);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LVPC_SetClipRect(poVpeMod->hVpc, &poVpeMod->oClipRect);
        }
    }

    MsgLog(
        2,
        "...} (Use VP%c, status= %d - %s)",
        poVpeMod->bUseVpc ? 'C' : 'E',
        eStatus,
        GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       VpeMod_CpuThread

Description:    .

\************************************************************************************************************/
LStatus VpeMod_CpuThread(void* pvData)
{
    if (pvData == MNULL)
    {
        MsgLogErr("ERROR! NULL data.");
        return LStatus_INVALID_PARAM;
    }

    VpeModule*  poVpe           = (VpeModule*)pvData;
    BufferInfo* poDstBuffer     = MNULL;
    BufferInfo* poSrcBuffer     = MNULL;
    MBOOL       bReady          = MTRUE;

    // Used with deinterlacing only
    BufferInfo* poBottomField   = MNULL;
    BufferInfo* poTopField      = MNULL;

    ModThread_SetName(poVpe->szModuleName);
    MsgLog(2, "Start thread %p.", pthread_self());

    while (!poVpe->oCpuThread.bKillThread)
    {
        LStatus eStatus = LStatus_OK;

        if (poDstBuffer == MNULL)
        {
            eStatus = ModLnk_GetReturnedBuffer(&(poVpe->oOutLink), 100, poVpe->hDevThread, &poDstBuffer);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = ModLnkIn_GetSubmittedBuffer(
                            &(poVpe->oInLink),
                            100,
                            0,
                            poVpe->hDevThread,
                            &poSrcBuffer,
                            MNULL,
                            MNULL);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            MUINT64 uiTimestampUsec = poSrcBuffer->uiTimestampUsec;

            if (poSrcBuffer->bEndOfStream)
            {
                MsgLog(4, "END-OF-STREAM");

                poDstBuffer->bEndOfStream = MTRUE;
                poVpe->oCpuThread.bKillThread = MTRUE;
            }
            else
            {
                MsgLog(4, "LVP%c_ExecuteFrame(DstBuffer[%u], SrcBuffer[%u])...",
                          poVpe->bUseVpc ? 'C' : 'E', poDstBuffer->uiId, poSrcBuffer->uiId);

                if (poVpe->bDeinterlace)
                {
                    BufferInfo* poNextBuffer = poSrcBuffer;

                    if (poNextBuffer->eFrameType == FrameType_TopField)
                    {
                        poSrcBuffer     = poTopField;
                        poTopField      = poNextBuffer;
                        bReady          = MFALSE;
                    }
                    else if ((poBottomField != MNULL)
                             && (poTopField != MNULL))
                    {
                        uiTimestampUsec = poTopField->uiTimestampUsec;

                        eStatus = LVPE_ExecuteFrame(
                                            poVpe->hVpe,
                                            poDstBuffer->hBuffer,
                                            LVPE_CurrentFrameOrField_FRAME,
                                            poBottomField->hBuffer,
                                            poTopField->hBuffer,
                                            poNextBuffer->hBuffer);

                        ModLnkIn_ReturnBuffer(&(poVpe->oInLink), poBottomField, poVpe->hDevThread, NO_TAG);

                        poBottomField   = poNextBuffer;
                        poSrcBuffer     = poTopField;
                        poTopField      = MNULL;
                        bReady          = MTRUE;
                    }
                    else
                    {
                        poSrcBuffer     = poBottomField;
                        poBottomField   = poNextBuffer;
                        bReady          = MFALSE;
                    }
                }
                else if (poVpe->bUseVpc)
                {
                    eStatus = LVPC_Execute(poVpe->hVpc, poSrcBuffer->hBuffer, poDstBuffer->hBuffer, MFALSE);
                }
                else
                {
                    eStatus = LVPE_Execute(poVpe->hVpe, poSrcBuffer->hBuffer, poDstBuffer->hBuffer);

                    if (poVpe->bSplitVpe && LSTATUS_IS_SUCCESS(eStatus))
                    {
                        eStatus = LVPE_Execute(poVpe->hVpe2, poSrcBuffer->hBuffer, poDstBuffer->hBuffer);

                        if (LSTATUS_IS_SUCCESS(eStatus))
                        {
                            eStatus = LDeviceThread_SyncDt(
                                        poVpe->hDevThread,
                                        poVpe->hDevThread2,
                                        &poVpe->oCpuThread.bKillThread);
                        }
                    }
                }

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    MsgLog(
                        4,
                        "LVP%c_ExecuteFrame done. (status= %d - %s)",
                        poVpe->bUseVpc ? 'C' : 'E',
                        eStatus,
                        GetStatusStr(eStatus));
                }
                else
                {
                    MsgLogErr(
                        "ERROR! LVP%c_ExecuteFrame returned status= %d (%s).",
                        poVpe->bUseVpc ? 'C' : 'E',
                        eStatus,
                        GetStatusStr(eStatus));
                }
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                if (bReady)
                {
                    poDstBuffer->uiTimestampUsec = uiTimestampUsec;
                    ModLnk_SubmitBuffer(&(poVpe->oOutLink), poDstBuffer, poVpe->hDevThread, NO_TAG);
                    poDstBuffer = MNULL;
                }

                if (poSrcBuffer != MNULL)
                {
                    ModLnkIn_ReturnBuffer(&(poVpe->oInLink), poSrcBuffer, poVpe->hDevThread, NO_TAG);
                    poSrcBuffer = MNULL;
                }
            }
            else
            {
                ModLnk_ReleaseBuffer(&(poVpe->oOutLink), poDstBuffer);
                poDstBuffer = MNULL;

                if (poSrcBuffer != MNULL)
                {
                    ModLnkIn_ReturnBuffer(&(poVpe->oInLink), poSrcBuffer, poVpe->hDevThread, NO_TAG);
                    poSrcBuffer = MNULL;
                }
            }
        }

        if (LSTATUS_IS_FAIL(eStatus)
            && (eStatus != LStatus_TIMEOUT))
        {
            usleep(1000);
        }
    }

    if (poDstBuffer != MNULL)
    {
        ModLnk_ReleaseBuffer(&(poVpe->oOutLink), poDstBuffer);
    }

    if (poBottomField != MNULL)
    {
        ModLnkIn_ReturnBuffer(&(poVpe->oInLink), poBottomField, poVpe->hDevThread, NO_TAG);
    }

    if (poTopField != MNULL)
    {
        ModLnkIn_ReturnBuffer(&(poVpe->oInLink), poTopField, poVpe->hDevThread, NO_TAG);
    }

    MsgLog(2, "Kill thread.");

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       VpeMod_Start

Description:    .

\************************************************************************************************************/
LStatus VpeMod_Start(VpeModule* poVpeMod)
{
    LStatus eStatus = LStatus_INVALID_PARAM;

    MsgLog(2, "{...");

    if ((poVpeMod != MNULL)
        && (poVpeMod->hVpe != MNULL))
    {
        if ((poVpeMod->oInLink.poModLnk != MNULL)
            && (poVpeMod->oOutLink.uiSubmitCount > 0))
        {
            eStatus = ModThread_Start(&(poVpeMod->oCpuThread), poVpeMod, VpeMod_CpuThread);
        }
        else
        {
            MsgLogErr("ERROR! Bad connection.");
            eStatus = LStatus_FAIL;
        }
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       RdFileMod_Stop

Description:    .

\************************************************************************************************************/
void VpeMod_Stop(VpeModule* poVpeMod)
{
    MsgLog(2, "{...");

    if (poVpeMod != MNULL)
    {
        ModThread_Stop(&(poVpeMod->oCpuThread));
    }

    MsgLog(2, "...}");
}

