/************************************************************************************************************\

Module Name:    VoutModule.c

Description:    .

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "VoutModule.h"
#include "ModuleSync.h"
#include "LBoard.h"

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

static const    MCHAR8      g_szModuleNameBase[]    = "Vout";

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

static const LVideo_VidParam g_oDefaultVidParam
        = {1920, 1080, 60, 68, 148500, 88, 44, 148, 4, 5, 36, MFALSE, MFALSE, MTRUE, MTRUE, 0};

static const LVideo_VidParam g_oDefaultVidParamWqhd
        = {2560, 1440, 60, 89, 241500, 48, 32, 80, 3, 5, 33, MFALSE, MFALSE, MTRUE, MFALSE, 0};

static const LVideo_VidParam g_oDefaultVidParam4k
        = {3840, 2160, 30, 68, 297000, 176, 88, 296, 8, 10, 72, MFALSE, MFALSE, MTRUE, MTRUE, 0};

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Function:       VoutMod_Init

Description:    .

\************************************************************************************************************/
LStatus VoutMod_Init(
    VoutModule*                 poVout,
    LDevice_Handle              hDevice,
    MUINT                       uiVoutIdx,
    LPixelFormat                ePixFmt,
    LSIZE*                      poDisplaySize,
    MUINT                       uiDisplayFps,
    MUINT                       uiInFrameRateNum,
    MUINT                       uiInFrameRateDen,
    MBOOL                       bControlQueueOverflow,
    MBOOL                       bIsMasterSync,
    MBOOL                       bLowLatency)
{
    MsgLog(2, "{...");

    VoutMod_Cleanup(poVout);

    LStatus eStatus = ((poVout != MNULL)
                       && (hDevice != MNULL))
                      ? LStatus_OK : LStatus_INVALID_PARAM;

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        snprintf(
                    poVout->szModuleName,
                    sizeof(poVout->szModuleName),
                    "%s%d",
                    g_szModuleNameBase,
                    uiVoutIdx);
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        poVout->uiVoutIdx = uiVoutIdx;
        eStatus = LVideoOut_GetHandle(hDevice, uiVoutIdx, LAccessMode_READWRITE_EXCLUSIVE, &(poVout->hVout));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        MBOOL bVideoParamDone = MFALSE;

        if (poDisplaySize != MNULL)
        {
            // Special cases.
            if ((poDisplaySize->iWidth == g_oDefaultVidParam.uiHActive)
                && (poDisplaySize->iHeight == g_oDefaultVidParam.uiVActive)
                && ((uiDisplayFps == 0)
                    || (uiDisplayFps == g_oDefaultVidParam.uiVRate_Hz)))
            {
                poVout->oVidParam = g_oDefaultVidParam;
                bVideoParamDone   = MTRUE;
            }
            else if ((poDisplaySize->iWidth == g_oDefaultVidParam4k.uiHActive)
                     && (poDisplaySize->iHeight == g_oDefaultVidParam4k.uiVActive)
                     && ((uiDisplayFps == 0)
                         || (uiDisplayFps == g_oDefaultVidParam4k.uiVRate_Hz)))
            {
                poVout->oVidParam = g_oDefaultVidParam4k;
                bVideoParamDone   = MTRUE;
            }
            else if ((poDisplaySize->iWidth == g_oDefaultVidParamWqhd.uiHActive)
                     && (poDisplaySize->iHeight == g_oDefaultVidParamWqhd.uiVActive)
                     && ((uiDisplayFps == 0)
                         || (uiDisplayFps == g_oDefaultVidParamWqhd.uiVRate_Hz)))
            {
                poVout->oVidParam = g_oDefaultVidParamWqhd;
                bVideoParamDone   = MTRUE;
            }

            MUINT uiIdx = 0;

            while (!bVideoParamDone)
            {
                LVideo_VidParam oVidParam;

                if (LSTATUS_IS_FAIL(LVideoOut_EnumVideoParameters(
                                                poVout->hVout,
                                                LVideo_SignalType_USEDEFAULT,
                                                uiIdx,
                                                &oVidParam)))
                {
                    break;
                }

                MsgLog(2, "Valid video parameters: %ux%u@%uHz", oVidParam.uiHActive,
                          oVidParam.uiVActive, oVidParam.uiVRate_Hz);

                if ((poDisplaySize->iWidth == oVidParam.uiHActive)
                    && (poDisplaySize->iHeight == oVidParam.uiVActive)
                    && ((uiDisplayFps == 0)
                        || (uiDisplayFps == oVidParam.uiVRate_Hz)))
                {
                    poVout->oVidParam = oVidParam;
                    bVideoParamDone   = MTRUE;
                }

                uiIdx++;
            }

            if (bVideoParamDone)
            {
                MsgLog(2, "Forced video parameters: %ux%u@%uHz", poVout->oVidParam.uiHActive,
                          poVout->oVidParam.uiVActive, poVout->oVidParam.uiVRate_Hz);
            }
        }

        if (!bVideoParamDone)
        {
            eStatus = LVideoOut_GetOptimalVideoParameters(
                        poVout->hVout,
                        LVideo_SignalType_USEDEFAULT,
                        &(poVout->oVidParam));

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                bVideoParamDone = MTRUE;

                MsgLog(2, "Optimal video parameters: %ux%u@%uHz", poVout->oVidParam.uiHActive,
                          poVout->oVidParam.uiVActive, poVout->oVidParam.uiVRate_Hz);
            }
            else
            {
                MsgLogErr("WARNING! Cannot get optimal video parameters (status= %d - %s). Use default mode.",
                          eStatus,
                          GetStatusStr(eStatus));

                eStatus = LStatus_OK;
            }
        }

        if (!bVideoParamDone)
        {
            poVout->oVidParam = g_oDefaultVidParam;

            MsgLog(2, "Default video parameters: %ux%u@%uHz", poVout->oVidParam.uiHActive,
                      poVout->oVidParam.uiVActive, poVout->oVidParam.uiVRate_Hz);
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LBuffer_CreateVideo(
                        hDevice,
                        ePixFmt,
                        poVout->oVidParam.uiHActive,
                        poVout->oVidParam.uiVActive,
                        &(poVout->hFrameBuffer));

        // Fill the initial frame buffer with gray color.
        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            LColor oColor;

            oColor.ePixelFormat = ePixFmt;
            oColor.uiAlpha      = 0;
            oColor.uiRed        = 128;
            oColor.uiGreen      = 128;
            oColor.uiBlue       = 128;

            ClearBuffer(hDevice, poVout->hFrameBuffer, &oColor);
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = VoutMod_SetInputFrameRate(poVout, uiInFrameRateNum, uiInFrameRateDen);
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = ModSync_Init(&poVout->oModSync);
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        poVout->hDevice                     = hDevice;
        // Need the submitted buffers to be synchronized.
        poVout->oInLink.bSyncBefore         = MTRUE;
        poVout->oInLink.uiSyncFraction      = sg_uiLowLatencySliceCount;
        poVout->bControlQueueOverflow       = bControlQueueOverflow;
        poVout->bIsMasterSync               = bIsMasterSync;
        poVout->bLowLatency                 = bLowLatency;
    }
    else
    {
        VoutMod_Cleanup(poVout);
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       VoutMod_Cleanup

Description:    .

\************************************************************************************************************/
void VoutMod_Cleanup(VoutModule* poVout)
{
    if (poVout != MNULL)
    {
        if (poVout->hVout != MNULL)
        {
            LVideoOut_ReleaseHandle(poVout->hVout);
            poVout->hVout = MNULL;
        }

        if (poVout->hFrameBuffer != MNULL)
        {
            LBuffer_Destroy(poVout->hFrameBuffer);
            poVout->hFrameBuffer = MNULL;
        }

        poVout->oInLink.poModLnk = MNULL;

        ModSync_Cleanup(&(poVout->oModSync));
    }
}

/************************************************************************************************************\

Function:       VoutMod_SetInputFrameRate

Description:    .

\************************************************************************************************************/
LStatus VoutMod_SetInputFrameRate(
    VoutModule* poVout,
    MUINT       uiInFrameRateNum,
    MUINT       uiInFrameRateDen)
{
    if ((poVout == MNULL)
        || (uiInFrameRateNum == 0)
        || (uiInFrameRateDen == 0))
    {
        return LStatus_INVALID_PARAM;
    }

    poVout->uiInFrameRateNum = uiInFrameRateNum;
    poVout->uiInFrameRateDen = uiInFrameRateDen;

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       VoutMod_SetOutputFormat

Description:    .

\************************************************************************************************************/
LStatus VoutMod_SetOutputFormat(
    VoutModule*  poVout,
    LPixelFormat ePixFmt,
    MBOOL        bRgbFullYuv709)
{
    /*
    MUINT   bIsYuv                  : 1;    // MTRUE if signal is YUV, MFALSE if is RGB.
    MUINT   fColorSpace             : 4;    // Color space :
                                            //  - When 0 : RGB full.
                                            //  - When 1 : RGB limited.
                                            //  - When 2 : YUV 601.
                                            //  - When 3 : YUV 709.

    MUINT   fChromaSubSampling      : 2;    // Chroma Sub-Sampling :
                                            //  - When 0 : LCSSAMPLING_444.
                                            //  - When 1 : LCSSAMPLING_420 (YUV only).
                                            //  - When 2 : LCSSAMPLING_422 (YUV only).

    MUINT   fColorDepth             : 2;    // Color depth :
                                            //  - When 0 :  8-bit per channel
                                            //  - When 1 : 10-bit per channel
                                            //  - When 2 : 12-bit per channel
                                            //  - When 3 : 16-bit per channel
    */

    if (poVout == MNULL)
    {
        return LStatus_INVALID_PARAM;
    }

    switch (ePixFmt)
    {
        case LPixelFormat_R8G8B8A8:
            poVout->oVidParam.bIsYuv               = 0;
            poVout->oVidParam.fColorSpace          = (bRgbFullYuv709 == MTRUE) ? 0 : 1;
            poVout->oVidParam.fChromaSubSampling   = 0;
            poVout->oVidParam.fColorDepth          = 0;
            break;

        case LPixelFormat_R10G10B10A2:
            poVout->oVidParam.bIsYuv               = 0;
            poVout->oVidParam.fColorSpace          = (bRgbFullYuv709 == MTRUE) ? 0 : 1;
            poVout->oVidParam.fChromaSubSampling   = 0;
            poVout->oVidParam.fColorDepth          = 1;
            break;

        case LPixelFormat_MP_Y8_U8V8_420:
            poVout->oVidParam.bIsYuv               = 1;
            poVout->oVidParam.fColorSpace          = (bRgbFullYuv709 == MTRUE) ? 2 : 3;
            poVout->oVidParam.fChromaSubSampling   = 1;
            poVout->oVidParam.fColorDepth          = 0;
            break;

        case LPixelFormat_MP_Y8_U8V8_422:
            poVout->oVidParam.bIsYuv               = 1;
            poVout->oVidParam.fColorSpace          = (bRgbFullYuv709 == MTRUE) ? 2 : 3;
            poVout->oVidParam.fChromaSubSampling   = 2;
            poVout->oVidParam.fColorDepth          = 0;
            break;

        case LPixelFormat_MP_Y8_U8V8_444:
            poVout->oVidParam.bIsYuv               = 1;
            poVout->oVidParam.fColorSpace          = (bRgbFullYuv709 == MTRUE) ? 2 : 3;
            poVout->oVidParam.fChromaSubSampling   = 0;
            poVout->oVidParam.fColorDepth          = 0;
            break;

        case LPixelFormat_MP_Y82_U82V82_X2_420:
            poVout->oVidParam.bIsYuv               = 1;
            poVout->oVidParam.fColorSpace          = (bRgbFullYuv709 == MTRUE) ? 2 : 3;
            poVout->oVidParam.fChromaSubSampling   = 1;
            poVout->oVidParam.fColorDepth          = 1;
            break;

        case LPixelFormat_MP_Y82_U82V82_X2_422:
            poVout->oVidParam.bIsYuv               = 1;
            poVout->oVidParam.fColorSpace          = (bRgbFullYuv709 == MTRUE) ? 2 : 3;
            poVout->oVidParam.fChromaSubSampling   = 2;
            poVout->oVidParam.fColorDepth          = 1;
            break;

        case LPixelFormat_MP_Y82_U82_V82_X2_444:
            poVout->oVidParam.bIsYuv               = 1;
            poVout->oVidParam.fColorSpace          = (bRgbFullYuv709 == MTRUE) ? 2 : 3;
            poVout->oVidParam.fChromaSubSampling   = 0;
            poVout->oVidParam.fColorDepth          = 1;
            break;

        default:
            return LStatus_INVALID_PARAM;
    }

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       VoutMod_DropOldestBuffer

Description:    .

\************************************************************************************************************/
void VoutMod_DropOldestBuffer(
    VoutModule* poVout)
{
    BufferInfo* poOldestBuffer = MNULL;

    LStatus eStatus = ModLnkIn_GetSubmittedBuffer(
                        &(poVout->oInLink),
                        100,
                        0,
                        MNULL,
                        &poOldestBuffer,
                        MNULL,
                        MNULL);

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        ModLnkIn_ReturnBuffer(&(poVout->oInLink), poOldestBuffer, MNULL, NO_TAG);
    }

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        // We dropped the oldest buffer, so update base time to the timestamp of the new oldest buffer.
        eStatus = ModLnkIn_InspectSubmittedBuffer(
                                    &(poVout->oInLink),
                                    &poOldestBuffer,
                                    MNULL);

        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            ModSyncMst_SetBaseTime(&(poVout->oModSync.oModSyncMst), poOldestBuffer->uiSyncPtsUsec, MFALSE);
        }
    }
}

/************************************************************************************************************\

Function:       VoutMod_CpuThread

Description:    .

\************************************************************************************************************/
LStatus VoutMod_CpuThread(void* pvData)
{
    if (pvData == MNULL)
    {
        MsgLogErr("ERROR! NULL data.");
        return LStatus_INVALID_PARAM;
    }

    VoutModule* poVout                  = (VoutModule*) pvData;
    ModuleSync* poSync                  = &(poVout->oModSync);
    ModuleSyncMaster* poSyncMst         = &(poSync->oModSyncMst);
    BufferInfo* poDisplayingBuffer      = MNULL;
    MBOOL32     bRendering              = poVout->bIsMasterSync ? MTRUE : MFALSE;
    MUINT64     uiNbSec                 = 0;
    MUINT64     uiStartTimeUsec         = GetMonoTimeUsec();
    MUINT64     uiInFrameDurationNsec   = ((MUINT64)poVout->uiInFrameRateDen * 1000 * 1000 * 1000)
                                          / poVout->uiInFrameRateNum;
    MUINT64     uiOutFrameDurationNsec  = ((MUINT64)((poVout->oVidParam.uiHFPorch
                                                      + poVout->oVidParam.uiHActive
                                                      + poVout->oVidParam.uiHBPorch
                                                      + poVout->oVidParam.uiHSync)
                                                     * (poVout->oVidParam.uiVFPorch
                                                        + poVout->oVidParam.uiVActive
                                                        + poVout->oVidParam.uiVBPorch
                                                        + poVout->oVidParam.uiVSync)) * 1000 * 1000)
                                          / poVout->oVidParam.uiPixelClock_kHz;

    // Debugging only: Set the Nth frame to show first 256 pixel values. Set to zero to disable.
    MUINT   uiShowPixels = 0;

    // Used for latency profiling only.
    MUINT32 uiTickFreq   = 0;
    MUINT   uiFrameCount = 0;

    if (sg_bEnableLatencyProfile)
    {
        LBoard_GetTickReferenceFrequency(poVout->hDevice, &uiTickFreq);
    }

    ModThread_SetName(poVout->szModuleName);
    MsgLog(2, "Start thread %p.", pthread_self());

    ModSync_SetParams(
            poSync,
            uiInFrameDurationNsec,
            uiOutFrameDurationNsec,
            poVout->oInLink.poModLnk->uiBufferCount,
            poVout->bControlQueueOverflow,
            poVout->bIsMasterSync ? MFALSE : MTRUE,
            poVout->bLowLatency);

    if(!poVout->bIsMasterSync)
    {
        ModSyncMst_SetParams(
                &(poSync->oModSyncMst),
                uiInFrameDurationNsec / 1000,
                poVout->oInLink.poModLnk->uiBufferCount,
                min(8, (poVout->oInLink.poModLnk->uiBufferCount / 2)),
                MFALSE);
    }

    while (!poVout->oCpuThread.bKillThread)
    {
        LStatus     eStatus         = LStatus_FAIL;
        BufferInfo* poSrcBuffer     = MNULL;
        MUINT       uiQueueSize     = 0;
        MINT64      iCurDriftNsec   = 0;

        if(bRendering)
        {
            // Both slave and master are ready, take a buffer to display.
            eStatus = ModLnkIn_GetSubmittedBuffer(
                                &(poVout->oInLink),
                                100,
                                0,
                                MNULL,
                                &poSrcBuffer,
                                MNULL,
                                &uiQueueSize);
        }
        else
        {
            // Not ready to start rendering yet, pre-sync with master stream.

            eStatus = ModLnkIn_InspectSubmittedBuffer(
                                        &(poVout->oInLink),
                                        &poSrcBuffer,
                                        &uiQueueSize);

            if(LSTATUS_IS_SUCCESS(eStatus))
            {
                // Lock must be kept until new base time is updated when oldest buffer is dropped.
                ModSyncMst_Lock(poSyncMst);

                MBOOL bDropOldest = MFALSE;

                ModSyncMst_PreSync(
                            poSyncMst,
                            uiQueueSize,
                            poSrcBuffer->uiSyncPtsUsec,
                            MFALSE,
                            &bRendering,
                            &bDropOldest);

                if(bDropOldest)
                {
                    VoutMod_DropOldestBuffer(poVout);
                }

                ModSyncMst_Unlock(poSyncMst);
            }

            eStatus = LStatus_FAIL;
            poSrcBuffer = MNULL;
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            if (poSrcBuffer->bEndOfStream)
            {
                MsgLog(4, "END-OF-STREAM");

                poVout->oCpuThread.bKillThread = MTRUE;
                eStatus = LStatus_END_OF_STREAM;
            }
            else
            {
                // Debugging only
                if (uiShowPixels > 0)
                {
                    if (--uiShowPixels == 0)
                    {
                        MUINT8* puiBuffer = MNULL;

                        if (LSTATUS_IS_SUCCESS(LBuffer_BeginAccess(poSrcBuffer->hBuffer, 0, 1, &puiBuffer)))
                        {
                            MsgLog(0, "Buffer data:");
                            MUINT i;
                            for (i = 0; i < 256; i++)
                            {
                                MsgLog(0, "  [%u] RGB(%u, %u, %u)", i, puiBuffer[i*4], puiBuffer[i*4 + 1],
                                          puiBuffer[i*4 + 2]);
                            }

                            LBuffer_EndAccess(poSrcBuffer->hBuffer);
                        }
                    }
                }

                MUINT64 uiTimestampNsec = ((poVout->bIsMasterSync)
                                            ? (poSrcBuffer->uiTimestampUsec)
                                            : (poSrcBuffer->uiSyncPtsUsec)) * 1000;

                if (ModSync_PrePresent(&(poVout->oModSync), uiTimestampNsec, uiQueueSize, &iCurDriftNsec))
                {
                    MsgLog(4, "LVideoOut_SetBuffer(Buffer[%u])...", poSrcBuffer->uiId);

                    eStatus = LVideoOut_SetBuffer(
                                poVout->hVout,
                                poSrcBuffer->hBuffer,
                                MTRUE,
                                LVideoOut_FlipType_NEXT_FIELD_WAIT
                                | (poVout->bLowLatency ? 0x1000 : 0));

                    MsgLog(4, "LVideoOut_SetBuffer done. (status= %d - %s)", eStatus, GetStatusStr(eStatus));

                    if (LSTATUS_IS_SUCCESS(eStatus))
                    {
                        if (sg_bEnableLatencyProfile)
                        {
                            uiFrameCount++;

                            if (((uiFrameCount % 60) == 0) && (uiTickFreq > 0))
                            {
                                MUINT64 uiCurTick = 0;

                                if (LSTATUS_IS_SUCCESS(
                                        LBoard_GetTickReferenceCounter(poVout->hDevice, &uiCurTick)))
                                {
                                    MUINT64 uiCurTimeUsec = (uiCurTick * 1000 * 1000) / uiTickFreq;

                                    MsgLog(0, "Latency = %ld usec",
                                              (MINT64)uiCurTimeUsec - poSrcBuffer->uiTimestampUsec);
                                }
                            }
                        }

                        ModSync_PostPresent(&(poVout->oModSync), uiTimestampNsec);

                        if (poDisplayingBuffer != MNULL)
                        {
                            ModLnkIn_ReturnBuffer(&(poVout->oInLink), poDisplayingBuffer, MNULL, NO_TAG);
                        }

                        poDisplayingBuffer = poSrcBuffer;
                    }
                    else
                    {
                        MsgLogErr("ERROR! LVideoOut_SetBuffer return %d (%s).", eStatus, GetStatusStr(eStatus));
                    }
                }
                else
                {
                    eStatus = LStatus_TIMEOUT;
                }
            }

            if (LSTATUS_IS_FAIL(eStatus))
            {
                ModLnkIn_ReturnBuffer(&(poVout->oInLink), poSrcBuffer, MNULL, NO_TAG);
            }
        }

        // Print statistics.
        if(GetMonoTimeUsec() >= (uiStartTimeUsec + (uiNbSec * 1000lu * 1000lu)))
        {
            uiNbSec++;

            MUINT64 uiHour = (uiNbSec / 3600);
            MUINT64 uiMin  = (uiNbSec / 60  )                   - (uiHour * 60);
            MUINT64 uiSec  = (uiNbSec       )   - (uiMin * 60)  - (uiHour * 3600);

            if(poVout->bIsMasterSync)
            {
                MsgLog(3, "[%03lu:%02lu:%02lu]: Queue[%u].",
                          uiHour, uiMin, uiSec, uiQueueSize);
            }
            else
            {
                MsgLog(3, "[%03lu:%02lu:%02lu]: Queue[%u], SyncDrift[%ld usec].",
                          uiHour, uiMin, uiSec, uiQueueSize, iCurDriftNsec / 1000);
            }
        }

        if (LSTATUS_IS_FAIL(eStatus)
            && (eStatus != LStatus_TIMEOUT))
        {
            usleep(1000);
        }
    }

    MsgLog(2, "Disable Vout and return displaying buffer...");

    if (poDisplayingBuffer != MNULL)
    {
        LVideoOut_Disable(poVout->hVout);
        ModLnkIn_ReturnBuffer(&(poVout->oInLink), poDisplayingBuffer, MNULL, NO_TAG);
    }

    MsgLog(2, "Kill thread.");

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       VoutMod_Start

Description:    .

\************************************************************************************************************/
LStatus VoutMod_Start(VoutModule* poVout)
{
    LStatus eStatus = LStatus_INVALID_PARAM;

    MsgLog(2, "{...");

    if ((poVout != MNULL)
        && (poVout->hVout != MNULL)
        && (poVout->hFrameBuffer != MNULL))
    {
        if (poVout->oCpuThread.hThread == MNULL)
        {
            eStatus = LVideoOut_Enable(
                        poVout->hVout,
                        LVideo_SignalType_DISPLAYPORT,
                        &(poVout->oVidParam),
                        poVout->hFrameBuffer,
                        LVideoOut_PowerState_ON);

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                eStatus = ModThread_Start(&(poVout->oCpuThread), poVout, VoutMod_CpuThread);
            }
            else
            {
                MsgLogErr("ERROR! LVideoOut_Enable returned %d (%s)", eStatus, GetStatusStr(eStatus));
            }
        }
        else
        {
            eStatus = LStatus_OK;
        }
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       VoutMod_Stop

Description:    .

\************************************************************************************************************/
void VoutMod_Stop(VoutModule* poVout)
{
    MsgLog(2, "{...");

    if (poVout != MNULL)
    {
        ModThread_Stop(&(poVout->oCpuThread));
        LVideoOut_Disable(poVout->hVout);
    }

    MsgLog(2, "...}");
}
