/************************************************************************************************************\

Module Name:    VoutGlxModule.c

Description:    .

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "VoutGlxModule.h"

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

static const    MCHAR8      g_szModuleNameBase[]    = "GlxVout";
static          MUINT32     g_uiVoutGlxModCount     = 0;

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Function:       VoutGlxMod_CreateGlxContext

Description:    .

\************************************************************************************************************/
LStatus VoutGlxMod_CreateGlxContext(VoutGlxModule* poVoutGlx, LRECT* poWindowRect)
{
    // Open the display.
    poVoutGlx->hDisplay = XOpenDisplay(":0");

    MINT    iScreen = 0;
    LStatus eStatus = (poVoutGlx->hDisplay != MNULL) ? LStatus_OK : LStatus_FAIL;

    // Create and map the window.
    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        XSetWindowAttributes oWinAttrib;
        LPOS                 oWindowPos;

        iScreen = DefaultScreen(poVoutGlx->hDisplay);

        if (poWindowRect == MNULL)
        {
            oWindowPos.iX                   = 0;
            oWindowPos.iY                   = 0;
            poVoutGlx->oWindowSize.iWidth   = XDisplayWidth(poVoutGlx->hDisplay,  iScreen);
            poVoutGlx->oWindowSize.iHeight  = XDisplayHeight(poVoutGlx->hDisplay, iScreen);
        }
        else
        {
            oWindowPos.iX                   = poWindowRect->iLeft;
            oWindowPos.iY                   = poWindowRect->iTop;
            poVoutGlx->oWindowSize.iWidth   = poWindowRect->iRight  - poWindowRect->iLeft;
            poVoutGlx->oWindowSize.iHeight  = poWindowRect->iBottom - poWindowRect->iTop;
        }

        poVoutGlx->oWindowSize.iWidth   = max(min(poVoutGlx->oWindowSize.iWidth,  4096), 64);
        poVoutGlx->oWindowSize.iHeight  = max(min(poVoutGlx->oWindowSize.iHeight, 4096), 64);
        poVoutGlx->uiWinWidthBytes      = (poVoutGlx->oWindowSize.iWidth
                                           * XBitmapUnit(poVoutGlx->hDisplay)) / 8;
        poVoutGlx->uiWinSizeBytes       = poVoutGlx->uiWinWidthBytes * poVoutGlx->oWindowSize.iHeight;

        // Create the window.
        poVoutGlx->uiWindowId = XCreateSimpleWindow(
                                    poVoutGlx->hDisplay,
                                    RootWindow(poVoutGlx->hDisplay, iScreen),
                                    oWindowPos.iX,
                                    oWindowPos.iY,
                                    poVoutGlx->oWindowSize.iWidth,
                                    poVoutGlx->oWindowSize.iHeight,
                                    0,
                                    0,
                                    BlackPixel(poVoutGlx->hDisplay, iScreen));

        // Set non managed window.
        oWinAttrib.override_redirect = True;
        XChangeWindowAttributes(poVoutGlx->hDisplay, poVoutGlx->uiWindowId, CWOverrideRedirect, &oWinAttrib);

        // Set the events we want to receive.
        oWinAttrib.event_mask = StructureNotifyMask
                                | ExposureMask
                                | KeyPressMask
                                | KeyReleaseMask
                                | ButtonPressMask
                                | ButtonReleaseMask;
        XChangeWindowAttributes(poVoutGlx->hDisplay, poVoutGlx->uiWindowId, CWEventMask, &oWinAttrib);

        XMapWindow(poVoutGlx->hDisplay, poVoutGlx->uiWindowId);

        // Hide the cursor
        const char auiData[] = {0, 0, 0, 0, 0, 0, 0, 0};

        Pixmap hPixmap = XCreateBitmapFromData(poVoutGlx->hDisplay, poVoutGlx->uiWindowId, auiData, 8, 8);

        if (hPixmap != None)
        {
            XColor oColor;
            Cursor hCursor;

            oColor.red   = 0;
            oColor.green = 0;
            oColor.blue  = 0;

            hCursor = XCreatePixmapCursor(poVoutGlx->hDisplay, hPixmap, hPixmap, &oColor, &oColor, 0, 0);

            if (hCursor != None)
            {
                XDefineCursor(poVoutGlx->hDisplay, poVoutGlx->uiWindowId, hCursor);
                XFreeCursor(poVoutGlx->hDisplay, hCursor);
            }

            XFreePixmap(poVoutGlx->hDisplay, hPixmap);
        }
    }
    else
    {
        MsgLogErr("ERROR! Cannot open X display.");
    }

    // Create the GLX context.
    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        int aoGlxAttribs[] =
        {
            GLX_RGBA,
            GLX_RED_SIZE, 1,
            GLX_GREEN_SIZE, 1,
            GLX_BLUE_SIZE, 1,
            GLX_DEPTH_SIZE, 1,
            GLX_DOUBLEBUFFER,
            None
        };

        XVisualInfo* poVisualInfo = glXChooseVisual(poVoutGlx->hDisplay, iScreen, aoGlxAttribs);

        if (poVisualInfo != MNULL)
        {
            poVoutGlx->hGlxCtx = glXCreateContext(poVoutGlx->hDisplay, poVisualInfo, MNULL, True);
        }

        eStatus = (poVoutGlx->hGlxCtx != MNULL) ? LStatus_OK : LStatus_FAIL;

        if (LSTATUS_IS_FAIL(eStatus))
        {
            MsgLogErr("ERROR! Cannot create GLX context.");
        }
    }

    // Init GL context, buffers and textures.
    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        glXMakeCurrent(poVoutGlx->hDisplay, poVoutGlx->uiWindowId, poVoutGlx->hGlxCtx);

        glClearColor(1.0, 1.0, 1.0, 1.0);
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
        glViewport(0, 0, poVoutGlx->oWindowSize.iWidth, poVoutGlx->oWindowSize.iHeight);
        glFlush();

        // Init GL parameters.
        glShadeModel(GL_FLAT);
        glDisable(GL_LIGHTING);
        glDisable(GL_DEPTH_TEST);
        glEnable(GL_TEXTURE_2D);
        glTexParameteri(GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_FALSE);

        // Create and init a GL texture.
        poVoutGlx->uiGlTextureId = 0;
        glGenTextures(1, &(poVoutGlx->uiGlTextureId));

        if (poVoutGlx->uiGlTextureId != 0)
        {
            glBindTexture(GL_TEXTURE_2D, poVoutGlx->uiGlTextureId);
            glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
            glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
            glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
            glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
            glTexImage2D(GL_TEXTURE_2D,
                         0,
                         GL_RGBA,
                         poVoutGlx->oWindowSize.iWidth,
                         poVoutGlx->oWindowSize.iHeight,
                         0,
                         GL_RGBA,
                         GL_UNSIGNED_BYTE,
                         0);
            glBindTexture(GL_TEXTURE_2D, 0);

            // Create and init GL buffer.
            poVoutGlx->uiGlBufferId = 0;
            glGenBuffers(1, &(poVoutGlx->uiGlBufferId));

            if (poVoutGlx->uiGlBufferId != 0)
            {
                glBindBuffer(GL_PIXEL_UNPACK_BUFFER, poVoutGlx->uiGlBufferId);
                glBufferData(GL_PIXEL_UNPACK_BUFFER, poVoutGlx->uiWinSizeBytes, 0, GL_STREAM_DRAW);
                glBindBuffer(GL_PIXEL_UNPACK_BUFFER, 0);
            }
            else
            {
                MsgLogErr("ERROR! Cannot generate GL buffer.");
                eStatus = LStatus_FAIL;
            }
        }
        else
        {
            MsgLogErr("ERROR! Cannot generate GL texture.");
            eStatus = LStatus_FAIL;
        }

        glXMakeCurrent(poVoutGlx->hDisplay, 0, NULL);
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       VoutGlxMod_DestroyGlxContext

Description:    .

\************************************************************************************************************/
void VoutGlxMod_DestroyGlxContext(VoutGlxModule* poVoutGlx)
{
    if (poVoutGlx->hDisplay != MNULL)
    {
        if (poVoutGlx->hGlxCtx != MNULL)
        {
            glXDestroyContext(poVoutGlx->hDisplay, poVoutGlx->hGlxCtx);
        }

        if (poVoutGlx->uiWindowId != 0)
        {
            XUnmapWindow(poVoutGlx->hDisplay, poVoutGlx->uiWindowId);
            XDestroyWindow(poVoutGlx->hDisplay, poVoutGlx->uiWindowId);
        }

        XCloseDisplay(poVoutGlx->hDisplay);

        poVoutGlx->hGlxCtx      = MNULL;
        poVoutGlx->uiWindowId   = 0;
        poVoutGlx->hDisplay     = MNULL;
    }
}

/************************************************************************************************************\

Function:       VoutGlxMod_XferBuffer

Description:    .

\************************************************************************************************************/
LStatus VoutGlxMod_XferBuffer(
    VoutGlxModule* poVoutGlx,
    LBuffer_Handle hSrcBuffer,
    MUINT64        uiTimestampUsec,
    MUINT64        uiInFrameDurationUsec,
    MINT64*        piBaseTimeUsec)
{
    LRECT32 oDstRect = {0, 0, poVoutGlx->oWindowSize.iWidth, poVoutGlx->oWindowSize.iHeight};
    LPOS32  oPos     = {0, 0};

    MsgLog(4, "{...");

    XForceScreenSaver(poVoutGlx->hDisplay, ScreenSaverReset);

    LStatus eStatus = LBlit_Blit(
                        poVoutGlx->hBlit,
                        hSrcBuffer,
                        poVoutGlx->hTmpBuffer,
                        &oPos,
                        &oDstRect,
                        MNULL,
                        LBlit_Bop_S,
                        MNULL,
                        MNULL);

    if (LSTATUS_IS_SUCCESS(eStatus) && !ENABLE_AUTO_SYNC)
    {
        MUINT64 uiStartTimeUsec = GetMonoTimeUsec();

        while (MTRUE)
        {
            eStatus = LDeviceThread_WaitCompletionTag(
                            poVoutGlx->hDevThread,
                            LDEVICETHREAD_DEFAULT_LUID,
                            LDEVICETHREAD_DEFAULT_TAG,
                            1000);

            if (eStatus == LStatus_TIMEOUT)
            {
                MsgLog(0, "Waited LBlit_Blit for %u msecs. Continue...",
                          (GetMonoTimeUsec() - uiStartTimeUsec) / 1000);
            }
            else
            {
                break;
            }
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        // Set the openGL context
        glXMakeCurrent(poVoutGlx->hDisplay, poVoutGlx->uiWindowId, poVoutGlx->hGlxCtx);
        glBindTexture(GL_TEXTURE_2D, poVoutGlx->uiGlTextureId);
        glBindBuffer(GL_PIXEL_UNPACK_BUFFER, poVoutGlx->uiGlBufferId);

        // Map the buffer and copy the data
        MUINT8* puiTmpBuffer = MNULL;
        MUINT8* puiGlBuffer  = (MUINT8*)glMapBuffer(GL_PIXEL_UNPACK_BUFFER, GL_WRITE_ONLY);

        if (puiGlBuffer != MNULL)
        {
            eStatus = LBuffer_BeginAccess(poVoutGlx->hTmpBuffer, 0, 1, &puiTmpBuffer);

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                if (poVoutGlx->uiWinWidthBytes == poVoutGlx->uiTmpBufferPitchBytes)
                {
                    memcpy(puiGlBuffer, puiTmpBuffer, poVoutGlx->uiWinSizeBytes);
                }
                else
                {
                    MINT y;
                    for (y = 0; y < poVoutGlx->oWindowSize.iHeight; y++)
                    {
                        memcpy(puiGlBuffer, puiTmpBuffer, poVoutGlx->uiWinWidthBytes);

                        puiGlBuffer  += poVoutGlx->uiWinWidthBytes;
                        puiTmpBuffer += poVoutGlx->uiTmpBufferPitchBytes;
                    }
                }

                LBuffer_EndAccess(poVoutGlx->hTmpBuffer);
            }

            glUnmapBuffer(GL_PIXEL_UNPACK_BUFFER);

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                glTexSubImage2D(
                    GL_TEXTURE_2D,
                    0,
                    0,
                    0,
                    poVoutGlx->oWindowSize.iWidth,
                    poVoutGlx->oWindowSize.iHeight,
                    GL_RGBA,
                    GL_UNSIGNED_BYTE,
                    0);

                // Render buffer
                glBegin(GL_QUADS);
                glTexCoord2f(0, 0);     glVertex3f(-1,  1, 0);
                glTexCoord2f(0, 1);     glVertex3f(-1, -1, 0);
                glTexCoord2f(1, 1);     glVertex3f( 1, -1, 0);
                glTexCoord2f(1, 0);     glVertex3f( 1,  1, 0);
                glEnd();

                // Do synchronization. It seems that glXSwapBuffers make some synchronization itself and set
                // the buffer timestamp to the current time. In such a case, all we can do is to call it at the
                // time the frame should be presented.
                if (uiInFrameDurationUsec > 0)
                {
                    MUINT64 uiCurrentTimeUsec       = GetMonoTimeUsec();
                    MINT64  iPresentTimeOffsetUsec  = *piBaseTimeUsec + uiTimestampUsec - uiCurrentTimeUsec;

                    if ((iPresentTimeOffsetUsec > (MINT64)uiInFrameDurationUsec)
                        || (iPresentTimeOffsetUsec < 0))
                    {
                        MsgLog(2, "Reset present time: Offset= %ld usec.", iPresentTimeOffsetUsec);
                        *piBaseTimeUsec = (MINT64)uiCurrentTimeUsec - uiTimestampUsec;
                    }
                    else
                    {
                        MUINT64 uiPresentTimeUsec = uiCurrentTimeUsec + iPresentTimeOffsetUsec;

                        // For more precision, sleep only 1 usec at a time.
                        while (uiPresentTimeUsec > uiCurrentTimeUsec)
                        {
                            usleep(1);
                            uiCurrentTimeUsec = GetMonoTimeUsec();
                        }
                    }
                }

                glXSwapBuffers(poVoutGlx->hDisplay, poVoutGlx->uiWindowId);
            }
        }
        else
        {
            eStatus = LStatus_FAIL;
        }

        // Free the openGL context, texture and buffer to avoid clash with another thread using openGL.
        glBindBuffer(GL_PIXEL_UNPACK_BUFFER, 0);
        glBindTexture(GL_TEXTURE_2D, 0);
        glXMakeCurrent(poVoutGlx->hDisplay, 0, NULL);
    }

    MsgLog(4, "...} Status= %d", eStatus);

    return eStatus;
}

/************************************************************************************************************\

Function:       VoutGlxMod_Init

Description:    .

\************************************************************************************************************/
LStatus VoutGlxMod_Init(
    VoutGlxModule*  poVoutGlx,
    LDevice_Handle  hDevice,
    LRECT*          poWindowRect,
    MUINT           uiInFrameRateNum,
    MUINT           uiInFrameRateDen,
    MBOOL           bControlQueueOverflow)
{
    MsgLog(2, "{...");

    VoutGlxMod_Cleanup(poVoutGlx);

    LStatus eStatus = ((poVoutGlx != MNULL)
                       && (hDevice != MNULL))
                       ? LStatus_OK : LStatus_INVALID_PARAM;

    // Create the window to display the stream on the BayTrail.
    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        snprintf(
                    poVoutGlx->szModuleName,
                    sizeof(poVoutGlx->szModuleName),
                    "%s%d",
                    g_szModuleNameBase,
                    g_uiVoutGlxModCount);

        eStatus = VoutGlxMod_CreateGlxContext(poVoutGlx, poWindowRect);
    }

    LBuffer_PlaneInfo oPlaneInfo;

    // Create the transfert buffer.
    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LBuffer_SystemVideoAttributes oSysVideoAttrib;

        oSysVideoAttrib.eAttributeType = LBuffer_Type_SYSTEM_VIDEO;
        oSysVideoAttrib.ePixelFormat   = LPixelFormat_R8G8B8A8;
        oSysVideoAttrib.uiWidth        = poVoutGlx->oWindowSize.iWidth;
        oSysVideoAttrib.uiHeight       = poVoutGlx->oWindowSize.iHeight;
        oSysVideoAttrib.uiSize         = 0;
        oSysVideoAttrib.pvSystemMemory = MNULL;

        eStatus = LBuffer_Create(hDevice, &(oSysVideoAttrib.eAttributeType), &(poVoutGlx->hTmpBuffer));

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LBuffer_GetPlaneInfo(poVoutGlx->hTmpBuffer, &oPlaneInfo);

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                poVoutGlx->uiTmpBufferPitchBytes = oPlaneInfo.auiPitchInBytes[0];
            }
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LDeviceThread_Create(hDevice, MNULL, &(poVoutGlx->hDevThread));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LBlit_Create(hDevice, poVoutGlx->hDevThread, MNULL, &(poVoutGlx->hBlit));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = VoutGlxMod_SetInputFrameRate(poVoutGlx, uiInFrameRateNum, uiInFrameRateDen);
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        // Need the submitted buffers to be synchronized.
        poVoutGlx->oInLink.bSyncBefore      = MTRUE;
        poVoutGlx->bControlQueueOverflow    = bControlQueueOverflow;
    }
    else
    {
        VoutGlxMod_Cleanup(poVoutGlx);
    }

    ++g_uiVoutGlxModCount;

    MsgLog(2, "...} (status= %d)", eStatus);

    return eStatus;
}

/************************************************************************************************************\

Function:       VoutGlxMod_Cleanup

Description:    .

\************************************************************************************************************/
void VoutGlxMod_Cleanup(VoutGlxModule* poVoutGlx)
{
    if (poVoutGlx != MNULL)
    {
        VoutGlxMod_DestroyGlxContext(poVoutGlx);

        if (poVoutGlx->hBlit != MNULL)
        {
            while (LSTATUS_IS_FAIL(LBlit_Destroy(poVoutGlx->hBlit)))
            {
                MsgLog(0, "LBlit_Destroy failed! Retry...");
                usleep(1000*1000);
            }
        }

        if (poVoutGlx->hDevThread != MNULL)
        {
            LDeviceThread_Destroy(poVoutGlx->hDevThread);
        }

        if (poVoutGlx->hTmpBuffer != MNULL)
        {
            LBuffer_Destroy(poVoutGlx->hTmpBuffer);
        }

        poVoutGlx->hTmpBuffer       = MNULL;
        poVoutGlx->hBlit            = MNULL;
        poVoutGlx->hDevThread       = MNULL;
        poVoutGlx->oInLink.poModLnk = MNULL;
    }
}

/************************************************************************************************************\

Function:       VoutGlxMod_SetInputFrameRate

Description:    .

\************************************************************************************************************/
LStatus VoutGlxMod_SetInputFrameRate(
    VoutGlxModule*  poVoutGlx,
    MUINT           uiInFrameRateNum,
    MUINT           uiInFrameRateDen)
{
    if ((poVoutGlx == MNULL)
        || (uiInFrameRateNum == 0)
        || (uiInFrameRateDen == 0))
    {
        return LStatus_INVALID_PARAM;
    }

    poVoutGlx->uiInFrameRateNum = uiInFrameRateNum;
    poVoutGlx->uiInFrameRateDen = uiInFrameRateDen;

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       VoutGlxMod_CpuThread

Description:    .

\************************************************************************************************************/
LStatus VoutGlxMod_CpuThread(void* pvData)
{
    if (pvData == MNULL)
    {
        MsgLogErr("ERROR! NULL data.");
        return LStatus_INVALID_PARAM;
    }

    VoutGlxModule* poVoutGlx = (VoutGlxModule*)pvData;

    ModThread_SetName(poVoutGlx->szModuleName);
    MsgLog(2, "Start thread %p.", pthread_self());

    MUINT64 uiInFrameDurationUsec   = ((MUINT64)poVoutGlx->uiInFrameRateDen * 1000 * 1000)
                                      / poVoutGlx->uiInFrameRateNum;
    MUINT   uiMaxQueueLengthAllowed = poVoutGlx->bControlQueueOverflow
                                      ? (poVoutGlx->oInLink.poModLnk->uiBufferCount / 2)
                                      : (poVoutGlx->oInLink.poModLnk->uiBufferCount * 2);
    MINT64  iBaseTimeUsec = 0;

    while (!poVoutGlx->oCpuThread.bKillThread)
    {
        BufferInfo* poSrcBuffer;
        MUINT       uiQueueSize;

        LStatus eStatus = ModLnkIn_GetSubmittedBuffer(
                                &(poVoutGlx->oInLink),
                                100,
                                0,
                                MNULL,
                                &poSrcBuffer,
                                MNULL,
                                &uiQueueSize);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            if (poSrcBuffer->bEndOfStream)
            {
                MsgLog(4, "END-OF-STREAM");

                poVoutGlx->oCpuThread.bKillThread = MTRUE;
                eStatus = LStatus_END_OF_STREAM;
            }
            else
            {
                if (uiQueueSize <= uiMaxQueueLengthAllowed)
                {
                    eStatus = VoutGlxMod_XferBuffer(
                                poVoutGlx,
                                poSrcBuffer->hBuffer,
                                poSrcBuffer->uiTimestampUsec,
                                uiInFrameDurationUsec,
                                &iBaseTimeUsec);
                }
                else
                {
                    MsgLog(2, "Skip Buffer: Q-Size= %u", uiQueueSize);
                }
            }

            ModLnkIn_ReturnBuffer(&(poVoutGlx->oInLink), poSrcBuffer, MNULL, NO_TAG);
        }

        if (LSTATUS_IS_FAIL(eStatus)
            && (eStatus != LStatus_TIMEOUT))
        {
            usleep(1000);
        }
    }

    MsgLog(2, "Kill thread.");

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       VoutGlxMod_Start

Description:    .

\************************************************************************************************************/
LStatus VoutGlxMod_Start(VoutGlxModule* poVoutGlx)
{
    LStatus eStatus = LStatus_INVALID_PARAM;

    MsgLog(2, "{...");

    if ((poVoutGlx != MNULL)
        && (poVoutGlx->hTmpBuffer != MNULL)
        && (poVoutGlx->hDevThread != MNULL)
        && (poVoutGlx->hBlit != MNULL)
        && (poVoutGlx->hGlxCtx != MNULL))
    {
        if (!ModThread_IsRunning(&(poVoutGlx->oCpuThread)))
        {
            eStatus = ModThread_Start(&(poVoutGlx->oCpuThread), poVoutGlx, VoutGlxMod_CpuThread);
        }
        else
        {
            eStatus = LStatus_OK;
        }
    }

    MsgLog(2, "...} (status= %d)", eStatus);

    return eStatus;
}

/************************************************************************************************************\

Function:       VoutGlxMod_Stop

Description:    .

\************************************************************************************************************/
void VoutGlxMod_Stop(VoutGlxModule* poVoutGlx)
{
    MsgLog(2, "{...");

    if (poVoutGlx != MNULL)
    {
        ModThread_Stop(&(poVoutGlx->oCpuThread));
    }

    MsgLog(2, "...}");
}
