/************************************************************************************************************\

Module Name:    VinModule.c

Description:    .

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "VinModule.h"
#include "LBoard.h"

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

static const    MCHAR8      g_szModuleNameBase[]    = "Vin";

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------

static void VinMod_DestroyBuffers(VinModule* poVinMod);
static MBOOL VinMod_IsSourceModeValid(VinModule* poVinMod);

/************************************************************************************************************\

Function:       VinMod_Init

Description:    .

\************************************************************************************************************/
LStatus VinMod_Init(
            VinModule*      poVinMod,
            LDevice_Handle  hDevice,
            MUINT           uiVinIndex,
            MUINT           uiBufferCount,
            MUINT           uiRateDivisor,
            MUINT           uiSkipPeriod,
            LPixelFormat    ePreferredPixFmt,
            MBOOL           bLowLatency)
{
    MsgLog(2, "{...");

    VinMod_Cleanup(poVinMod);

    LStatus eStatus = ((poVinMod != MNULL)
                       && (hDevice != MNULL)
                       && (uiRateDivisor > 0))
                      ? LStatus_OK : LStatus_INVALID_PARAM;

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        snprintf(
                    poVinMod->szModuleName,
                    sizeof(poVinMod->szModuleName),
                    "%s%d",
                    g_szModuleNameBase,
                    uiVinIndex);
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LBoard_GetTickReferenceFrequency(hDevice, &(poVinMod->uiTickFrequency));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LVideoIn_GetHandle(hDevice, uiVinIndex, LAccessMode_READWRITE_EXCLUSIVE, &(poVinMod->hVin));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = VinMod_GetSourceMode(poVinMod, ePreferredPixFmt);
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LBuffer_Attributes oBufAttrib;

        if (poVinMod->bFailSafe)
        {
            oBufAttrib.eAttributeType                 = LBuffer_Type_VIDEO;
            oBufAttrib.oVideoAttributes.uiWidth       = poVinMod->oFrameSize.iWidth;
            oBufAttrib.oVideoAttributes.uiHeight      = poVinMod->oFrameSize.iHeight;
            oBufAttrib.oVideoAttributes.ePixelFormat  = poVinMod->ePixelFormat;
        }

        eStatus = ModLnk_Init(
                    &(poVinMod->oOutLink),
                    hDevice,
                    uiBufferCount,
                    poVinMod->bFailSafe ? &oBufAttrib : MNULL,
                    !poVinMod->bFailSafe,
                    0,
                    poVinMod->szModuleName);

        if (poVinMod->bFailSafe && LSTATUS_IS_SUCCESS(eStatus))
        {
            MINT i;
            for (i = 0; i < poVinMod->oOutLink.uiBufferCount; i++)
            {
                LColor oColor;

                oColor.ePixelFormat = LPixelFormat_ALIAS_ABGR32;
                oColor.uiAlpha      = 0;
                oColor.uiRed        = 128;
                oColor.uiGreen      = 128;
                oColor.uiBlue       = 128;

                ClearBuffer(hDevice, poVinMod->oOutLink.aoBufferInfo[i].hBuffer, &oColor);
            }
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        poVinMod->hDevice               = hDevice;
        poVinMod->uiRateDivisor         = uiRateDivisor;
        poVinMod->uiSkipPeriod          = uiSkipPeriod;
        poVinMod->uiElapsedTimeUsec     = 0;
        poVinMod->uiTotalFrameCount     = 0;
        poVinMod->uiCapturedFrameCount  = 0;
        poVinMod->bLowLatency           = bLowLatency;
    }
    else
    {
        VinMod_Cleanup(poVinMod);
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       VinMod_Cleanup

Description:    .

\************************************************************************************************************/
void VinMod_Cleanup(VinModule* poVinMod)
{
    MsgLog(2, "{...");

    if (poVinMod != MNULL)
    {
        VinMod_DestroyBuffers(poVinMod);
        ModLnk_Cleanup(&(poVinMod->oOutLink));

        if (poVinMod->hVin != MNULL)
        {
            LVideoIn_ReleaseHandle(poVinMod->hVin);
            poVinMod->hVin = MNULL;
        }

        poVinMod->bFailSafe         = MFALSE;
        poVinMod->bSourceModeValid  = MFALSE;
    }

    MsgLog(2, "...}");
}

/************************************************************************************************************\

Function:       VinMod_GetNextBufferNoWait

Description:    .

\************************************************************************************************************/
LStatus VinMod_GetNextBufferNoWait(
    VinModule*           poVinMod,
    LDeviceThread_Handle hDt,
    LRECT32*             aoSliceRect,
    MUINT*               auiSliceDelayTick,
    LBuffer_Handle*      phBuffer,
    MUINT64*             puiFieldCount,
    MUINT64*             puiTickRef,
    MBOOL32*             pbIsTopField)
{
    MUINT64 uiReadyFieldCount = 0;
    MUINT64 uiReadyTickRef    = 0;
    MBOOL   bGotReady         = MFALSE;
    LStatus eStatus           = LStatus_OK;

    while (MTRUE)
    {
        eStatus = LVideoIn_WaitForFieldStart(
                                poVinMod->hVin,
                                bGotReady ? 0 : 100,
                                puiFieldCount,
                                puiTickRef,
                                pbIsTopField);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            uiReadyFieldCount = *puiFieldCount;
            uiReadyTickRef    = *puiTickRef;
            bGotReady         = MTRUE;
        }
        else
        {
            if (bGotReady)
            {
                eStatus = LStatus_OK;
            }

            break;
        }
    }

    while (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LVideoIn_GetNextBuffer(
                                poVinMod->hVin,
                                MFALSE,
                                0,
                                phBuffer,
                                puiFieldCount,
                                puiTickRef,
                                pbIsTopField);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            if (*puiFieldCount > uiReadyFieldCount)
            {
                *puiTickRef = uiReadyTickRef
                              + (((*puiFieldCount - uiReadyFieldCount)
                                  / LVIDEO_FIELD_COUNTER_INCREMENT)
                                 * poVinMod->uiFrameDurationTick);

                // Use this only to check if the buffers are already captured (> 0%).
                if (0)
                {
                    MUINT64 uiCurTick = 0;

                    if (LSTATUS_IS_SUCCESS(LBoard_GetTickReferenceCounter(poVinMod->hDevice, &uiCurTick)))
                    {
                        if (uiCurTick > *puiTickRef)
                        {
                            MsgLog(0, "Buffer %u%% captured.",
                                      ((uiCurTick - *puiTickRef) * 100) / poVinMod->uiFrameDurationTick);
                        }
                    }
                }

                MsgLog(4, "Got not ready buffer: Field count: %lu | %lu, TickRef: %lu | %lu",
                           uiReadyFieldCount, *puiFieldCount, uiReadyTickRef, *puiTickRef);

                break;
            }

            MsgLog(2, "Skip buffer: Field count: %lu | %lu, TickRef: %lu | %lu",
                       uiReadyFieldCount, *puiFieldCount, uiReadyTickRef, *puiTickRef);

            LVideoIn_ReleaseBuffer(poVinMod->hVin, *phBuffer);
            *phBuffer = MNULL;
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        MUINT64 uiCurTick    = 0;
        MINT    iTickOffset  = 0;

        // Make sure the device thread is idle in order to avoid wait time accumulation.
        LDeviceThread_WaitCompletionTag(hDt, LDEVICETHREAD_DEFAULT_LUID, LDEVICETHREAD_DEFAULT_TAG, 1000);

        if (LSTATUS_IS_SUCCESS(LBoard_GetTickReferenceCounter(poVinMod->hDevice, &uiCurTick)))
        {
            iTickOffset = (MINT)((MINT64)uiCurTick - *puiTickRef);

            if ((iTickOffset < 0)
                && labs(iTickOffset) >= poVinMod->uiFrameDurationTick)
            {
                MsgLog(1, "WARNING! Got buffer sooner than expected (TickOffset= %d).",
                          iTickOffset);
            }
        }

        eStatus = LDeviceThread_SyncWithTimeMark(hDt, 0, MTRUE, MFALSE, LINFINITE_TIMEOUT);

        MUINT i;
        for (i = 0; (i < sg_uiLowLatencySliceCount) && LSTATUS_IS_SUCCESS(eStatus); i++)
        {
            if ((MINT)auiSliceDelayTick[i] > iTickOffset)
            {
                MUINT uiDelayDtTick = ((MUINT64)(auiSliceDelayTick[i] - iTickOffset)
                                       * LDEVICETHREAD_TICK_COUNTER_FREQUENCY)
                                      / poVinMod->uiTickFrequency;

                if (uiDelayDtTick < (LDEVICETHREAD_TICK_COUNTER_FREQUENCY / 1000))
                {
                    uiDelayDtTick = LDEVICETHREAD_TICK_COUNTER_FREQUENCY / 1000;
                }

                eStatus = LDeviceThread_SyncWithTimeMark(
                                            hDt,
                                            uiDelayDtTick,
                                            MFALSE,
                                            MFALSE,
                                            LINFINITE_TIMEOUT);

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    eStatus = LBuffer_AddAutoSyncRect(*phBuffer, hDt, MFALSE, &aoSliceRect[i]);
                }
            }
        }

        if (!LSTATUS_IS_SUCCESS(eStatus))
        {
            MsgLogErr("ERROR! Cannot add auto-sync with time mark.");
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       VinMod_CpuThread

Description:    .

\************************************************************************************************************/
LStatus VinMod_CpuThread(void* pvData)
{
    if (pvData == MNULL)
    {
        MsgLogErr("ERROR! NULL data.");
        return LStatus_INVALID_PARAM;
    }

    VinModule* poVinMod = (VinModule*)pvData;

    ModThread_SetName(poVinMod->szModuleName);
    MsgLog(2, "Start thread %p.", pthread_self());

    LDeviceThread_CreateAttributes oAttributes = {LDeviceThread_AttributeType_CREATE, MTRUE, MTRUE, MFALSE, "Vin"};
    LDeviceThread_Handle           hDt         = MNULL;

    LStatus eStatus = LDeviceThread_Create(poVinMod->hDevice, &oAttributes.eType, &hDt);

    if (LSTATUS_IS_FAIL(eStatus))
    {
        MsgLogErr("ERROR! Cannot create device thread (status= %d)", eStatus);
        return eStatus;
    }

    eStatus = poVinMod->bFailSafe ? LStatus_OK : LVideoIn_StartCapture(poVinMod->hVin);

    MUINT64    uiPrevTickRef         = ~0;
    MUINT      uiGotBufferCount      = 0;
    MUINT      uiReleasedBufferCount = 0;

    const MUINT uiLineSizePixels = poVinMod->oVidParam.uiHFPorch
                                   + poVinMod->oVidParam.uiHSync
                                   + poVinMod->oVidParam.uiHBPorch
                                   + poVinMod->oVidParam.uiHActive;
    MUINT uiLinePerSlice = (poVinMod->oVidParam.uiVActive + sg_uiLowLatencySliceCount - 1)
                           / sg_uiLowLatencySliceCount;

    uiLinePerSlice = ((uiLinePerSlice + sg_uiLowLatencySliceAlign - 1) / sg_uiLowLatencySliceAlign)
                     * sg_uiLowLatencySliceAlign;

    LRECT32 aoSliceRect[sg_uiLowLatencySliceCount];
    MUINT   auiSliceDelayTick[sg_uiLowLatencySliceCount];
    MUINT   uiLinePos = 0;

    MUINT i;
    for (i = 0; i < sg_uiLowLatencySliceCount; i++)
    {
        aoSliceRect[i].iLeft    = 0;
        aoSliceRect[i].iTop     = uiLinePos;
        aoSliceRect[i].iRight   = poVinMod->oVidParam.uiHActive;

        uiLinePos += uiLinePerSlice;

        if (uiLinePos > poVinMod->oVidParam.uiVActive)
        {
            uiLinePos = poVinMod->oVidParam.uiVActive;
        }

        aoSliceRect[i].iBottom  = uiLinePos;

        MUINT64 uiPixelCount = uiLinePos * uiLineSizePixels;

        auiSliceDelayTick[i] = (MUINT)((uiPixelCount * poVinMod->uiTickFrequency)
                                       / (poVinMod->oVidParam.uiPixelClock_kHz * 1000));

        MsgLog(2, "Slice[%u]: Rect= %d,%d,%d,%d, Delay= %u ticks",
                  i, aoSliceRect[i].iLeft, aoSliceRect[i].iTop, aoSliceRect[i].iRight,
                  aoSliceRect[i].iBottom, auiSliceDelayTick[i]);
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        const MUINT64 uiStartTimeUsec = GetMonoTimeUsec();

        poVinMod->uiElapsedTimeUsec     = 0;
        poVinMod->uiTotalFrameCount     = 0;
        poVinMod->uiCapturedFrameCount  = 0;

        while (!poVinMod->oCpuThread.bKillThread)
        {
            if (!VinMod_IsSourceModeValid(poVinMod))
            {
                MsgLog(2, "Mode changed. Stop capturing.");
                poVinMod->oCpuThread.bKillThread = MTRUE;
                break;
            }

            BufferInfo* poBuffer;

            eStatus = ModLnk_GetReturnedBuffer(&(poVinMod->oOutLink), 100, MNULL, &poBuffer);

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                MUINT64 uiFieldCount    = poVinMod->uiTotalFrameCount + 1;
                MUINT64 uiTickRef       = 0;
                MBOOL32 bIsTopField     = MTRUE;

                if (!poVinMod->bFailSafe)
                {
                    if (poBuffer->hBuffer != MNULL)
                    {
                        MsgLog(4, "LVideoIn_ReleaseBuffer(Buffer[%u])...", poBuffer->uiId);

                        eStatus = LVideoIn_ReleaseBuffer(poVinMod->hVin, poBuffer->hBuffer);

                        if (LSTATUS_IS_SUCCESS(eStatus))
                        {
                            poBuffer->hBuffer = MNULL;
                            uiReleasedBufferCount++;
                        }
                        else
                        {
                            MsgLogErr(
                                "ERROR! LVideoIn_ReleaseBuffer returned %d (%s).",
                                eStatus,
                                GetStatusStr(eStatus));
                        }
                    }

                    if (LSTATUS_IS_SUCCESS(eStatus))
                    {
                        MsgLog(4, "LVideoIn_GetNextBuffer(Buffer[%u])...", poBuffer->uiId);

                        if (poVinMod->bLowLatency)
                        {
                            eStatus = VinMod_GetNextBufferNoWait(
                                            poVinMod,
                                            hDt,
                                            aoSliceRect,
                                            auiSliceDelayTick,
                                            &(poBuffer->hBuffer),
                                            &uiFieldCount,
                                            &uiTickRef,
                                            &bIsTopField);
                        }
                        else
                        {
                            eStatus = LVideoIn_GetNextBuffer(
                                            poVinMod->hVin,
                                            MTRUE,
                                            1000,
                                            &(poBuffer->hBuffer),
                                            &uiFieldCount,
                                            &uiTickRef,
                                            &bIsTopField);
                        }

                        if ( g_uiMsgLogLevel >= 2)
                        {
                            MBOOL32 bBufferAlreadyReturned  = MFALSE;

                            for (i = 0; (i < poVinMod->uiNbOfVinBufferHandle) && !bBufferAlreadyReturned; ++i)
                            {
                                bBufferAlreadyReturned = (poVinMod->pahVinBufferHandle[i] == poBuffer->hBuffer);
                            }

                            if (!bBufferAlreadyReturned)
                            {
                                poVinMod->pahVinBufferHandle[poVinMod->uiVinBufferCurrentIndex] = poBuffer->hBuffer;

                                MUINT64     uiLUID = 0;

                                eStatus = LBuffer_GetLUID(poBuffer->hBuffer, &uiLUID);

                                if (LSTATUS_IS_SUCCESS(eStatus))
                                {
                                    MsgLog(
                                                2,
                                                "Module %s created Buffer %d LUID: 0x%lx",
                                                poVinMod->szModuleName,
                                                poVinMod->uiVinBufferCurrentIndex,
                                                uiLUID)
                                }
                                else
                                {
                                    MsgLog(
                                                2,
                                                "Module %s create Buffer %d : Fail to get Buffer LUID with status %d (%s)",
                                                poVinMod->szModuleName,
                                                poVinMod->uiVinBufferCurrentIndex,
                                                eStatus,
                                                GetStatusStr(eStatus));
                                }

                                ++poVinMod->uiVinBufferCurrentIndex;
                            }
                        }

                        if (LSTATUS_IS_SUCCESS(eStatus))
                        {
                            MsgLog(4, "LVideoIn_GetNextBuffer done. (FieldCount= %lu, TickRef= %lu, "
                                      "status= %d) Got count = %u", uiFieldCount, uiTickRef, eStatus,
                                      poVinMod->uiCapturedFrameCount);

                            uiFieldCount /= LVIDEO_FIELD_COUNTER_INCREMENT;
                            uiGotBufferCount++;
                        }
                        else
                        {
                            MsgLogErr(
                                "WARNING! LVideoIn_GetNextBuffer returned %d (%s). Buffer counts: "
                                "Created= %u, Got= %u, Released= %u",
                                eStatus, GetStatusStr(eStatus),
                                poVinMod->oOutLink.uiBufferCount, uiGotBufferCount, uiReleasedBufferCount);
                        }
                    }
                }

                poVinMod->uiElapsedTimeUsec = GetMonoTimeUsec() - uiStartTimeUsec;

                MBOOL bKeepFrame = MFALSE;

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    if (poVinMod->bFailSafe)
                    {
                        MUINT64 uiTimestampUsec = (uiFieldCount * 1000 * 1000 * 1000)
                                                  / poVinMod->uiFrameRate1000x;

                        if (uiTimestampUsec > poVinMod->uiElapsedTimeUsec)
                        {
                            usleep(uiTimestampUsec - poVinMod->uiElapsedTimeUsec);
                        }

                        poBuffer->uiTimestampUsec = GetMonoTimeUsec();
                    }
                    else
                    {
                        poBuffer->uiTimestampUsec = (uiTickRef * 1000 * 1000) / poVinMod->uiTickFrequency;
                    }

                    poBuffer->uiSyncPtsUsec = poBuffer->uiTimestampUsec;

                    poVinMod->uiTotalFrameCount = uiFieldCount;

                    if (uiPrevTickRef != ~0)
                    {
                        // Check tick reference
                        if (uiTickRef < uiPrevTickRef)
                        {
                            MsgLog(0, "WARNING! New tick reference (%lu) smaller than the new one (%lu).",
                                      uiTickRef, uiPrevTickRef);
                        }
                        else if ((uiTickRef - uiPrevTickRef) > 2*1000*1000)
                        {
                            MsgLog(0, "WARNING! Big difference between two tick refs: (%lu - %lu)/%u sec.",
                                      uiTickRef, uiPrevTickRef, poVinMod->uiTickFrequency);
                        }
                    }

                    uiPrevTickRef = uiTickRef;

                    if ((uiFieldCount % poVinMod->uiRateDivisor) == 0)
                    {
                        bKeepFrame = MTRUE;

                        if (poVinMod->uiSkipPeriod != 0)
                        {
                            if ((uiFieldCount % (poVinMod->uiRateDivisor * poVinMod->uiSkipPeriod)) == 0)
                            {
                                bKeepFrame = MFALSE;
                            }
                        }
                    }
                }

                if (bKeepFrame)
                {
                    poVinMod->uiCapturedFrameCount++;

                    ModLnk_SubmitBuffer(&(poVinMod->oOutLink), poBuffer, MNULL, NO_TAG);
                }
                else
                {
                    ModLnk_ReleaseBuffer(&(poVinMod->oOutLink), poBuffer);
                }
            }

            if (LSTATUS_IS_FAIL(eStatus)
                && (eStatus != LStatus_TIMEOUT))
            {
                usleep(1000);
            }
        }

        if (!poVinMod->bFailSafe)
        {
            LVideoIn_StopCapture(poVinMod->hVin);
        }
    }
    else
    {
        MsgLogErr("ERROR! LVideoIn_StartCapture returned %d (%s)", eStatus, GetStatusStr(eStatus));
    }

    if (hDt != MNULL)
    {
        LDeviceThread_Destroy(hDt);
    }

    MsgLog(2, "Kill thread.");

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       VinMod_IsSourceChanged

Description:    .

\************************************************************************************************************/
MBOOL VinMod_IsSourceModeValid(VinModule* poVinMod)
{
    if (poVinMod->hVin == MNULL)
    {
        return MFALSE;
    }

    LVideoIn_Event eEvents;

    while (LSTATUS_IS_SUCCESS(LVideoIn_WaitForEvent(poVinMod->hVin, 0, &eEvents)))
    {
        if ((eEvents & LVideoIn_Event_SOURCE_CHANGED) != 0)
        {
            poVinMod->bSourceModeValid = MFALSE;
        }
    }

    return poVinMod->bSourceModeValid;
}

/************************************************************************************************************\

Function:       VinMod_GetSourceMode

Description:    .

\************************************************************************************************************/
LStatus VinMod_GetSourceMode(VinModule* poVinMod, LPixelFormat ePreferredPixFmt)
{
    LStatus eStatus = (poVinMod->hVin != MNULL) ? LStatus_OK : LStatus_FAIL;

    while (LSTATUS_IS_SUCCESS(eStatus)
           && !VinMod_IsSourceModeValid(poVinMod))
    {
        MBOOL   bIsCapturable = MFALSE;
        MBOOL   bIsDetected   = MFALSE;
        MUINT32 uiWidth;
        MUINT32 uiHeight;

        eStatus = LVideoIn_DetectSource(poVinMod->hVin, &bIsCapturable, &bIsDetected, &uiWidth, &uiHeight);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            if (bIsCapturable && bIsDetected)
            {
                LVideo_VidParam oVidParam;
                LPixelFormat    ePixFmt;
                MINT            i = 0;

                poVinMod->oFrameSize.iWidth  = uiWidth;
                poVinMod->oFrameSize.iHeight = uiHeight;

                while (LSTATUS_IS_SUCCESS(LVideoIn_EnumSupportedPixelFormat(poVinMod->hVin, i, &ePixFmt)))
                {
                    // Keep the first supported format or the preferred one.
                    if ((i == 0) || (ePixFmt == ePreferredPixFmt))
                    {
                        poVinMod->ePixelFormat = ePixFmt;
                    }

                    MsgLog(2, "Suported pixel format [%u] = 0x%08X", i, ePixFmt);
                    i++;
                }

                eStatus = LVideoIn_GetDetectedVideoParameters(poVinMod->hVin, &oVidParam);

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    MUINT uiHorzSize = oVidParam.uiHFPorch
                                       + oVidParam.uiHActive
                                       + oVidParam.uiHBPorch
                                       + oVidParam.uiHSync;
                    MUINT uiVertSize = oVidParam.uiVFPorch
                                       + oVidParam.uiVActive
                                       + oVidParam.uiVBPorch
                                       + oVidParam.uiVSync;


                    poVinMod->uiFrameRate1000x = (MUINT)(((MUINT64)oVidParam.uiPixelClock_kHz * 1000 * 1000)
                                                         / (uiHorzSize * uiVertSize));
                    poVinMod->uiFrameDurationTick = (MUINT)(((MUINT64)uiHorzSize
                                                              * uiVertSize
                                                              * poVinMod->uiTickFrequency)
                                                    / (oVidParam.uiPixelClock_kHz * 1000));

                    if (oVidParam.bInterlaced)
                    {
                        poVinMod->oFrameSize.iHeight *= 2;
                        poVinMod->bInterlaced = MTRUE;
                    }
                    else
                    {
                        poVinMod->bInterlaced = MFALSE;
                    }

                    MsgLog(2, "PixelClock= %u kHz, HFPorch= %u, HActive= %u, HBPorch= %u, HSync= %u",
                                oVidParam.uiPixelClock_kHz, oVidParam.uiHFPorch, oVidParam.uiHActive,
                                oVidParam.uiHBPorch, oVidParam.uiHSync);
                    MsgLog(2, "VFPorch= %u, VActive= %u, VBPorch= %u, VSync= %u",
                                oVidParam.uiVFPorch, oVidParam.uiVActive, oVidParam.uiVBPorch,
                                oVidParam.uiVSync);

                    poVinMod->oVidParam        = oVidParam;
                    poVinMod->bSourceModeValid = MTRUE;

                    MsgLog(2, "Detected source: %ux%u%s@%u.%03uHz, PixelFormat= 0x%08X",
                              poVinMod->oFrameSize.iWidth, poVinMod->oFrameSize.iHeight,
                              poVinMod->bInterlaced ? "i" : "",
                              poVinMod->uiFrameRate1000x / 1000, poVinMod->uiFrameRate1000x % 1000,
                              poVinMod->ePixelFormat);
                }
                else
                {
                    MsgLogErr(
                        "ERROR! LVideoIn_GetDetectedVideoParameters returned %d (%s)",
                        eStatus,
                        GetStatusStr(eStatus));
                }
            }
            else
            {
                static const LVideo_VidParam oDefaultVidParam
                    = {1920, 1080, 60, 68, 148500, 88, 44, 148, 4, 5, 36, MFALSE, MFALSE, MTRUE, MTRUE, 0};

                MsgLog(2, "WARNING! Video signal not detected(%d) nor capturable(%d). Go in failsafe mode.",
                          bIsDetected, bIsCapturable);

                poVinMod->bFailSafe             = MTRUE;
                poVinMod->oFrameSize.iWidth     = 1920;
                poVinMod->oFrameSize.iHeight    = 1080;
                poVinMod->uiFrameRate1000x      = 60000;
                poVinMod->ePixelFormat          = ePreferredPixFmt;
                poVinMod->bInterlaced           = MFALSE;
                poVinMod->bSourceModeValid      = MTRUE;
                poVinMod->oVidParam             = oDefaultVidParam;
            }
        }
        else
        {
            MsgLogErr(
                "ERROR! LVideoIn_DetectSource returned %d (%s).",
                eStatus,
                GetStatusStr(eStatus));
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       VinMod_Start

Description:    .

\************************************************************************************************************/
LStatus VinMod_Start(VinModule* poVinMod)
{
    MsgLog(2, "{...");

    LStatus eStatus = ((poVinMod != MNULL) && (poVinMod->hVin != MNULL))
                      ? LStatus_OK
                      : LStatus_INVALID_PARAM;

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        if (poVinMod->oOutLink.uiSubmitCount == 0)
        {
            MsgLogErr("ERROR! Bad connection.");
            eStatus = LStatus_FAIL;
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        VinMod_DestroyBuffers(poVinMod);

        if (!poVinMod->bFailSafe)
        {
            LBuffer_Options oVinBuffOpt;

            oVinBuffOpt.uiOptionTypes       = LBuffer_OptionType_AutoSync;
            oVinBuffOpt.eAutoSyncOption     = LBuffer_AutoSyncOption_PerRegion;
            oVinBuffOpt.uiAutoSyncTimeout   = LINFINITE_TIMEOUT;

            MsgLog(2, "LVideoIn_SetCreateBuffersOptions(OptTypes= %p, SyncOpt= 0x%p, SyncTimeOut= %p)...",
                oVinBuffOpt.uiOptionTypes, oVinBuffOpt.eAutoSyncOption, oVinBuffOpt.uiAutoSyncTimeout);

            // Buffer options should be set before the creation call.
            eStatus = LVideoIn_SetCreateBuffersOptions(poVinMod->hVin, &oVinBuffOpt);

            MsgLog(2, "LVideoIn_CreateBuffers(%ux%u, PixFmt= 0x%08X, Count= %u)...",
                      poVinMod->oFrameSize.iWidth, poVinMod->oFrameSize.iHeight, poVinMod->ePixelFormat,
                      poVinMod->oOutLink.uiBufferCount);

            eStatus = LVideoIn_CreateBuffers(
                        poVinMod->hVin,
                        poVinMod->oFrameSize.iWidth,
                        poVinMod->oFrameSize.iHeight,
                        poVinMod->ePixelFormat,
                        poVinMod->oOutLink.uiBufferCount + 2);

            // +2 buffers count gives to LVideoIn_CreateBuffers(), 2 are allocated more by LVideoIn Module
            poVinMod->uiVinBufferCurrentIndex   = 0;
            poVinMod->uiNbOfVinBufferHandle     = poVinMod->oOutLink.uiBufferCount + 4;
            poVinMod->pahVinBufferHandle        = malloc(poVinMod->uiNbOfVinBufferHandle* sizeof(LBuffer_Handle));


            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                MsgLog(2, "LVideoIn_CreateBuffers done.");
                poVinMod->bInternalBufferCreated = MTRUE;
            }
            else
            {
                MsgLogErr(
                    "ERROR! LVideoIn_CreateBuffers returned %d (%s).",
                    eStatus,
                    GetStatusStr(eStatus));
            }
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = ModThread_Start(&(poVinMod->oCpuThread), poVinMod, VinMod_CpuThread);
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       VinMod_Stop

Description:    .

\************************************************************************************************************/
void VinMod_Stop(VinModule* poVinMod)
{
    MsgLog(2, "{...");

    if (poVinMod != MNULL)
    {
        ModThread_Stop(&(poVinMod->oCpuThread));

        if (poVinMod->pahVinBufferHandle != MNULL)
        {
            free(poVinMod->pahVinBufferHandle);
            poVinMod->pahVinBufferHandle = MNULL;
        }

        poVinMod->uiVinBufferCurrentIndex   = 0;
        poVinMod->uiNbOfVinBufferHandle     = 0;
    }

    MsgLog(2, "...}");
}

/************************************************************************************************************\

Function:       VinMod_DestroyBuffers

Description:    .

\************************************************************************************************************/
void VinMod_DestroyBuffers(VinModule* poVinMod)
{
    if (poVinMod != MNULL)
    {
        if (poVinMod->bInternalBufferCreated)
        {
            MUINT i;

            for (i = 0; i < poVinMod->oOutLink.uiBufferCount; i++)
            {
                BufferInfo* poBufferInfo = &(poVinMod->oOutLink.aoBufferInfo[i]);

                if (poBufferInfo->hBuffer != MNULL)
                {
                    LVideoIn_ReleaseBuffer(poVinMod->hVin,  poBufferInfo->hBuffer);
                    poBufferInfo->hBuffer = MNULL;
                }
            }

            LVideoIn_DestroyBuffers(poVinMod->hVin);
            poVinMod->bInternalBufferCreated = MFALSE;
        }
    }
}
