/************************************************************************************************************\

Module Name:    RotateModule.c

Description:    .

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "RotateModule.h"

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

static const    MCHAR8      g_szModuleNameBase[]    = "Rotate";
static          MUINT32     g_uiRotateModCount      = 0;

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Function:       RotMod_Init

Description:    .

\************************************************************************************************************/
LStatus RotMod_Init(
            RotateModule*       poRotMod,
            LDevice_Handle      hDevice,
            MUINT               uiOutBufferCount,
            LBuffer_Attributes* poOutBufferAttributes,
            LSIZE*              poOutFrameSize,
            LRECT32*            poSrcRect,
            LPOS*               poDstPos,
            LVPC_Rotation       eRotation)
{
    MsgLog(2, "{...");

    MUINT i;

    RotMod_Cleanup(poRotMod);

    LStatus eStatus = ((poRotMod != MNULL)
                       && (hDevice != MNULL)
                       && (poOutBufferAttributes != MNULL)
                       && (poOutBufferAttributes->eAttributeType == LBuffer_Type_VIDEO)
                       && (poOutFrameSize != MNULL)
                       && (poSrcRect != MNULL)
                       && (poDstPos != MNULL))
                      ? LStatus_OK : LStatus_INVALID_PARAM;

    for (i = 0; (i < ROTMOD_VPC_COUNT) && LSTATUS_IS_SUCCESS(eStatus); i++)
    {
        eStatus = LDeviceThread_Create(hDevice, MNULL, &(poRotMod->ahDt[i]));

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LVPC_Create(hDevice, poRotMod->ahDt[i], MNULL, &(poRotMod->ahVpc[i]));
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            LVP_ColorSpace oColorSpace;

            oColorSpace.eFunction    = LVP_TransferFunction_Other;
            oColorSpace.eMatrix      = LVP_TransferMatrix_BT709;
            oColorSpace.poUserMatrix = MNULL;

            eStatus = LVPC_SetSourceParam(
                        poRotMod->ahVpc[i],
                        0,
                        LVPC_Source_COLOR_SPACE,
                        sizeof(LVP_ColorSpace),
                        &oColorSpace);

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                eStatus = LVPC_SetTargetParam(
                            poRotMod->ahVpc[i],
                            LVPC_Target_COLOR_SPACE,
                            sizeof(LVP_ColorSpace),
                            &oColorSpace);
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                LVPC_CompositorColorSpace oCompColSpace;

                oCompColSpace.bYCbCrA
                    = LPixelFormat_IsRgb(poOutBufferAttributes->oVideoAttributes.ePixelFormat)
                      ? MFALSE : MTRUE;
                oCompColSpace.oColorSpace = oColorSpace;

                eStatus = LVPC_SetCompositorParam(
                            poRotMod->ahVpc[i],
                            LVPC_Compositor_COLOR_SPACE,
                            sizeof(LVPC_CompositorColorSpace),
                            &oCompColSpace);
            }
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            LVPC_TargetRotation oRotation;

            oRotation.eRotation = eRotation;

            eStatus = LVPC_SetTargetParam(
                        poRotMod->ahVpc[i],
                        LVPC_Target_ROTATION,
                        sizeof(oRotation),
                        &oRotation);
        }
    }

    // Set rectangles

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LVP_Rect oSrcRect;
        LVP_Rect oDstRect;
        LRECT32  oClipRect;
        LVP_Rect oSubClipRect;
        LSIZE    oBufferSize;
        LPOS     oPadOffset;

        // Where is the padding before rotation ('.' = Frame, 'x' = Column Padding, 'y' = Line padding):
        //
        //  None        Flip-None   180         Flip-180    90          Flip-90     270         Flip-270
        //
        //  .....x      x.....      yyyyyy      yyyyyy      y....       yxxxx       xxxxy       ....y
        //  .....x      x.....      x.....      .....x      y....       y....       ....y       ....y
        //  yyyyyy      yyyyyy      x.....      .....x      y....       y....       ....y       ....y
        //                                                  yxxxx       y....       ....y       xxxxy

        if ((eRotation == LVPC_Rotation_NONE)
            || (eRotation == LVPC_Rotation_180)
            || (eRotation == LVPC_Rotation_FLIP_NONE)
            || (eRotation == LVPC_Rotation_FLIP_180))
        {
            oBufferSize.iWidth  = poOutBufferAttributes->oVideoAttributes.uiWidth;
            oBufferSize.iHeight = poOutBufferAttributes->oVideoAttributes.uiHeight;
            oPadOffset.iX       = oBufferSize.iWidth - poOutFrameSize->iWidth;
            oPadOffset.iY       = oBufferSize.iHeight - poOutFrameSize->iHeight;
        }
        else
        {
            oBufferSize.iWidth  = poOutBufferAttributes->oVideoAttributes.uiHeight;
            oBufferSize.iHeight = poOutBufferAttributes->oVideoAttributes.uiWidth;
            oPadOffset.iX       = oBufferSize.iWidth - poOutFrameSize->iHeight;
            oPadOffset.iY       = oBufferSize.iHeight - poOutFrameSize->iWidth;
        }

        if ((eRotation == LVPC_Rotation_NONE)
            || (eRotation == LVPC_Rotation_FLIP_180)
            || (eRotation == LVPC_Rotation_90)
            || (eRotation == LVPC_Rotation_FLIP_270))
        {
            oPadOffset.iX = 0;
        }

        if ((eRotation == LVPC_Rotation_NONE)
            || (eRotation == LVPC_Rotation_FLIP_NONE)
            || (eRotation == LVPC_Rotation_270)
            || (eRotation == LVPC_Rotation_FLIP_270))
        {
            oPadOffset.iY = 0;
        }

        oSrcRect.bEnable        = MTRUE;
        oSrcRect.oRect          = *poSrcRect;

        oDstRect.bEnable        = MTRUE;
        oDstRect.oRect.iLeft    = poDstPos->iX + oPadOffset.iX;
        oDstRect.oRect.iTop     = poDstPos->iY + oPadOffset.iY;
        oDstRect.oRect.iRight   = oDstRect.oRect.iLeft + poSrcRect->iRight - poSrcRect->iLeft;
        oDstRect.oRect.iBottom  = oDstRect.oRect.iTop + poSrcRect->iBottom - poSrcRect->iTop;

        oClipRect.iLeft         = max(oDstRect.oRect.iLeft,   0);
        oClipRect.iTop          = max(oDstRect.oRect.iTop,    0);
        oClipRect.iRight        = min(oDstRect.oRect.iRight,  oBufferSize.iWidth);
        oClipRect.iBottom       = min(oDstRect.oRect.iBottom, oBufferSize.iHeight);

        MUINT uiClipHeight      = ((oClipRect.iBottom > oClipRect.iTop)
                                   && (oClipRect.iRight > oClipRect.iLeft))
                                  ? (oClipRect.iBottom - oClipRect.iTop) : 0;
        MUINT uiSubClipHeight   = (uiClipHeight + ROTMOD_VPC_COUNT - 1) / ROTMOD_VPC_COUNT;

        oSubClipRect.bEnable        = MTRUE;
        oSubClipRect.oRect.iLeft    = oClipRect.iLeft;
        oSubClipRect.oRect.iTop     = oClipRect.iTop;
        oSubClipRect.oRect.iRight   = oClipRect.iRight;

        for (i = 0; (i < ROTMOD_VPC_COUNT) && LSTATUS_IS_SUCCESS(eStatus); i++)
        {
            if (uiSubClipHeight > uiClipHeight)
            {
                uiSubClipHeight = uiClipHeight;
            }

            poRotMod->abUseVpc[i] = (uiSubClipHeight > 0) ? MTRUE : MFALSE;

            if (poRotMod->abUseVpc[i])
            {
                eStatus = LVPC_SetSrcRect(poRotMod->ahVpc[i], 0, &oSrcRect);

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    eStatus = LVPC_SetDstRect(poRotMod->ahVpc[i], 0, &oDstRect);
                }

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    oSubClipRect.oRect.iBottom = oSubClipRect.oRect.iTop + uiSubClipHeight;

                    eStatus = LVPC_SetClipRect(poRotMod->ahVpc[i], &oSubClipRect);

                    oSubClipRect.oRect.iTop = oSubClipRect.oRect.iBottom;
                    uiClipHeight -= uiSubClipHeight;
                }
            }
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {

        snprintf(
                    poRotMod->szModuleName,
                    sizeof(poRotMod->szModuleName),
                    "%s%d",
                    g_szModuleNameBase,
                    g_uiMsgLogLevel);

        eStatus = ModLnk_Init(
                        &(poRotMod->oOutLink),
                        hDevice,
                        uiOutBufferCount,
                        poOutBufferAttributes,
                        MFALSE,
                        0,
                        poRotMod->szModuleName);
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LColor oColor;

        oColor.ePixelFormat = LPixelFormat_B8G8R8A8;
        oColor.uiAlpha      = 0;
        oColor.uiRed        = 128;
        oColor.uiGreen      = 128;
        oColor.uiBlue       = 128;

        for (i = 0; i < poRotMod->oOutLink.uiBufferCount; i++)
        {
            // Don't care if the clear has failed (it will with packed pixel formats).
            ClearBuffer(hDevice, poRotMod->oOutLink.aoBufferInfo[i].hBuffer, &oColor);
        }
    }

    if (LSTATUS_IS_FAIL(eStatus))
    {
        RotMod_Cleanup(poRotMod);
    }

    ++g_uiRotateModCount;

    MsgLog(2, "...} (status= %d -%s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       RotMod_Cleanup

Description:    .

\************************************************************************************************************/
void RotMod_Cleanup(RotateModule* poRotMod)
{
    MsgLog(2, "{...");

    if (poRotMod != MNULL)
    {
        ModLnk_Cleanup(&(poRotMod->oOutLink));

        MUINT i;

        for (i = 0; i < ROTMOD_VPC_COUNT; i++)
        {
            if (poRotMod->ahVpc[i] != MNULL)
            {
                while (LSTATUS_IS_FAIL(LVPC_Destroy(poRotMod->ahVpc[i])))
                {
                    MsgLog(0, "LVPC_Destroy failed! Retry...");
                    usleep(1000*1000);
                }
            }

            if (poRotMod->ahDt[i] != MNULL)
            {
                LDeviceThread_Destroy(poRotMod->ahDt[i]);
            }

            poRotMod->ahVpc[i]       = MNULL;
            poRotMod->ahDt[i] = MNULL;
        }

        poRotMod->oInLink.poModLnk  = MNULL;
    }

    MsgLog(2, "...}");
}

/************************************************************************************************************\

Function:       RotMod_CpuThread

Description:    .

\************************************************************************************************************/
LStatus RotMod_CpuThread(void* pvData)
{
    if (pvData == MNULL)
    {
        MsgLogErr("ERROR! NULL data.");
        return LStatus_INVALID_PARAM;
    }

    RotateModule* poRotMod    = (RotateModule*)pvData;
    BufferInfo*  poDstBuffer = MNULL;
    BufferInfo*  poSrcBuffer = MNULL;

    ModThread_SetName(poRotMod->szModuleName);
    MsgLog(2, "Start thread %p.", pthread_self());

    while (!poRotMod->oCpuThread.bKillThread)
    {
        LStatus eStatus = LStatus_OK;

        if (poDstBuffer == MNULL)
        {
            eStatus = ModLnk_GetReturnedBuffer(&(poRotMod->oOutLink), 100, poRotMod->ahDt[0], &poDstBuffer);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = ModLnkIn_GetSubmittedBuffer(
                            &(poRotMod->oInLink),
                            100,
                            0,
                            poRotMod->ahDt[0],
                            &poSrcBuffer,
                            MNULL,
                            MNULL);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            if (poSrcBuffer->bEndOfStream)
            {
                MsgLog(4, "END-OF-STREAM");

                poDstBuffer->bEndOfStream = MTRUE;
                poRotMod->oCpuThread.bKillThread = MTRUE;
            }
            else
            {
                MsgLog(4, "LVPC_ExecuteFrame(DstBuffer[%u], SrcBuffer[%u])...",
                          poDstBuffer->uiId, poSrcBuffer->uiId);

                MUINT i;

                for (i = 1; i < ROTMOD_VPC_COUNT; i++)
                {
                    if (poRotMod->abUseVpc[i])
                    {
                        LDeviceThread_SyncDt(
                            poRotMod->ahDt[i],
                            poRotMod->ahDt[0],
                            &poRotMod->oCpuThread.bKillThread);
                    }
                }

                for (i = 0; i < ROTMOD_VPC_COUNT; i++)
                {
                    if (poRotMod->abUseVpc[i])
                    {
                        eStatus = LVPC_Execute(
                                        poRotMod->ahVpc[i],
                                        poSrcBuffer->hBuffer,
                                        poDstBuffer->hBuffer,
                                        MFALSE);

                        if (i > 0)
                        {
                            LDeviceThread_SyncDt(
                                poRotMod->ahDt[0],
                                poRotMod->ahDt[i],
                                &poRotMod->oCpuThread.bKillThread);
                        }
                    }
                }


                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    MsgLog(4, "LVPC_ExecuteFrame done. (status= %d -%s)", eStatus, GetStatusStr(eStatus));
                }
                else
                {
                    MsgLogErr(
                        "ERROR! LVPC_ExecuteFrame returned status= %d (%s).",
                        eStatus,
                        GetStatusStr(eStatus));
                }
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                poDstBuffer->uiSyncPtsUsec = poSrcBuffer->uiSyncPtsUsec;
                poDstBuffer->uiTimestampUsec = poSrcBuffer->uiTimestampUsec;
                ModLnk_SubmitBuffer(&(poRotMod->oOutLink), poDstBuffer, poRotMod->ahDt[0], NO_TAG);
                ModLnkIn_ReturnBuffer(&(poRotMod->oInLink), poSrcBuffer, poRotMod->ahDt[0], NO_TAG);
            }
            else
            {
                ModLnk_ReleaseBuffer(&(poRotMod->oOutLink), poDstBuffer);
                ModLnkIn_ReturnBuffer(&(poRotMod->oInLink), poSrcBuffer, MNULL, NO_TAG);
            }

            poDstBuffer = MNULL;
            poSrcBuffer = MNULL;
        }

        if (LSTATUS_IS_FAIL(eStatus)
            && (eStatus != LStatus_TIMEOUT))
        {
            usleep(1000);
        }
    }

    if (poDstBuffer != MNULL)
    {
        ModLnk_ReleaseBuffer(&(poRotMod->oOutLink), poDstBuffer);
    }

    MsgLog(2, "Kill thread.");

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       RotMod_Start

Description:    .

\************************************************************************************************************/
LStatus RotMod_Start(RotateModule* poRotMod)
{
    LStatus eStatus = LStatus_INVALID_PARAM;

    MsgLog(2, "{...");

    if ((poRotMod != MNULL)
        && (poRotMod->ahVpc[0] != MNULL))
    {
        if ((poRotMod->oInLink.poModLnk != MNULL)
            && (poRotMod->oOutLink.uiSubmitCount > 0))
        {
            eStatus = ModThread_Start(&(poRotMod->oCpuThread), poRotMod, RotMod_CpuThread);
        }
        else
        {
            MsgLogErr("ERROR! Bad connection.");
            eStatus = LStatus_FAIL;
        }
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       RdFileMod_Stop

Description:    .

\************************************************************************************************************/
void RotMod_Stop(RotateModule* poRotMod)
{
    MsgLog(2, "{...");

    if (poRotMod != MNULL)
    {
        ModThread_Stop(&(poRotMod->oCpuThread));
    }

    MsgLog(2, "...}");
}

