/************************************************************************************************************\

Module Name:    ReadFileModule.c

Description:    .

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "ReadFileModule.h"

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

static const    MCHAR8      g_szModuleNameBase[]    = "RdFile";
static          MUINT32     g_uiReadFileModCount    = 0;

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Function:       RdFileMod_Init

Description:    .

\************************************************************************************************************/
LStatus RdFileMod_Init(
            ReadFileModule*     poRdFileMod,
            LDevice_Handle      hDevice,
            LSIZE*              poFrameSize,
            MUINT               uiBufferCount,
            LBuffer_Attributes* poBufferAttributes,
            const char*         szFilePath,
            MBOOL               bLoop,
            MUINT               uiMaxFrameCount,
            MUINT               uiMinFramePeriodUsec)
{
    MsgLog(2, "{...");

    RdFileMod_Cleanup(poRdFileMod);

    LStatus eStatus = ((poRdFileMod != MNULL)
                       && (hDevice != MNULL)
                       && (poBufferAttributes != MNULL)
                       && (szFilePath != MNULL))
                      ? LStatus_OK : LStatus_INVALID_PARAM;

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LBuffer_PlaneInfo* poPlaneInfo = &(poRdFileMod->oFilePlaneInfo);

        if (poBufferAttributes->eAttributeType == LBuffer_Type_VIDEO)
        {
            const LBuffer_VideoAttributes* poVidAttrib = &(poBufferAttributes->oVideoAttributes);

            switch (poVidAttrib->ePixelFormat)
            {
                case LPixelFormat_MP_Y8_U8V8_420:
                    poPlaneInfo->uiPlaneCount       = 2;
                    poPlaneInfo->aePlaneFormat[0]   = LPixelFormat_Y8;
                    poPlaneInfo->aePlaneFormat[1]   = LPixelFormat_U8V8;
                    poPlaneInfo->auiWidth[0]        = poFrameSize->iWidth;
                    poPlaneInfo->auiWidth[1]        = poFrameSize->iWidth / 2;
                    poPlaneInfo->auiWidthInBytes[0] = poFrameSize->iWidth;
                    poPlaneInfo->auiWidthInBytes[1] = poFrameSize->iWidth;
                    poPlaneInfo->auiHeight[0]       = poFrameSize->iHeight;
                    poPlaneInfo->auiHeight[1]       = poFrameSize->iHeight / 2;
                    break;

                case LPixelFormat_MP_Y8_U8_V8_420:
                    poPlaneInfo->uiPlaneCount       = 3;
                    poPlaneInfo->aePlaneFormat[0]   = LPixelFormat_Y8;
                    poPlaneInfo->aePlaneFormat[1]   = LPixelFormat_U8;
                    poPlaneInfo->aePlaneFormat[2]   = LPixelFormat_V8;
                    poPlaneInfo->auiWidth[0]        = poFrameSize->iWidth;
                    poPlaneInfo->auiWidth[1]        = poFrameSize->iWidth / 2;
                    poPlaneInfo->auiWidth[2]        = poFrameSize->iWidth / 2;
                    poPlaneInfo->auiWidthInBytes[0] = poFrameSize->iWidth;
                    poPlaneInfo->auiWidthInBytes[1] = poFrameSize->iWidth / 2;
                    poPlaneInfo->auiWidthInBytes[2] = poFrameSize->iWidth / 2;
                    poPlaneInfo->auiHeight[0]       = poFrameSize->iHeight;
                    poPlaneInfo->auiHeight[1]       = poFrameSize->iHeight / 2;
                    poPlaneInfo->auiHeight[2]       = poFrameSize->iHeight / 2;
                    break;

                case LPixelFormat_R8G8B8A8:
                case LPixelFormat_B8G8R8A8:
                    poPlaneInfo->uiPlaneCount       = 1;
                    poPlaneInfo->aePlaneFormat[0]   = LPixelFormat_R8G8B8A8;
                    poPlaneInfo->auiWidth[0]        = poFrameSize->iWidth;
                    poPlaneInfo->auiWidthInBytes[0] = poFrameSize->iWidth * 4;
                    poPlaneInfo->auiHeight[0]       = poFrameSize->iHeight;
                    break;

                default:
                    MsgLogErr("ERROR! Pixel format not supported.");
                    eStatus = LStatus_INVALID_PARAM;
                    break;
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                poRdFileMod->uiMaxFrameCount        = uiMaxFrameCount;
                poRdFileMod->uiFileFrameSizeBytes   = 0;

                MUINT i;
                for (i = 0; i < poPlaneInfo->uiPlaneCount; i++)
                {
                    poPlaneInfo->auiPitch[i]            = poPlaneInfo->auiWidth[i];
                    poPlaneInfo->auiPitchInBytes[i]     = poPlaneInfo->auiWidthInBytes[i];
                    poRdFileMod->uiFileFrameSizeBytes  += (poPlaneInfo->auiPitchInBytes[i]
                                                           * poPlaneInfo->auiHeight[i]);
                }
            }
        }
        else if ((poBufferAttributes->eAttributeType == LBuffer_Type_LINEAR)
                 || (poBufferAttributes->eAttributeType == LBuffer_Type_SYSTEM_LINEAR))
        {
            MUINT uiSize = (poBufferAttributes->eAttributeType == LBuffer_Type_LINEAR)
                           ? poBufferAttributes->oLinearAttributes.uiSize
                           : poBufferAttributes->oSystemLinearAttributes.uiSize;

            poPlaneInfo->uiPlaneCount       = 1;
            poPlaneInfo->aePlaneFormat[0]   = LPixelFormat_L8;
            poPlaneInfo->auiWidth[0]        = uiSize;
            poPlaneInfo->auiWidthInBytes[0] = uiSize;
            poPlaneInfo->auiPitch[0]        = uiSize;
            poPlaneInfo->auiPitchInBytes[0] = uiSize;
            poPlaneInfo->auiHeight[0]       = 1;

            // No frame in linear
            poRdFileMod->uiMaxFrameCount        = MUINT_MAX;
            poRdFileMod->uiFileFrameSizeBytes   = 1;
        }
        else if (poBufferAttributes->eAttributeType == LBuffer_Type_SYSTEM_VIDEO)
        {
            const LBuffer_SystemVideoAttributes* poSysVidAttrib = &(poBufferAttributes->oSystemVideoAttributes);

            poPlaneInfo->auiWidth[0]  = poFrameSize->iWidth;
            poPlaneInfo->auiHeight[0] = poFrameSize->iHeight;
            poPlaneInfo->auiPitch[0]  = poPlaneInfo->auiWidth[0];

            switch (poSysVidAttrib->ePixelFormat)
            {
                case LPixelFormat_EXCH_8_YUYV_422:
                    // 8 bytes for 4 pixels ==> 2 bytes / pixel, granularity of 4 pixels or 8 bytes.
                    poPlaneInfo->auiWidthInBytes[0] = poFrameSize->iWidth * 2;

                    // All VPG lines must be alined on 128 bytes.
                    poPlaneInfo->auiWidthInBytes[0] = (poPlaneInfo->auiWidthInBytes[0] + 127) & ~127;
                    poPlaneInfo->uiPlaneCount       = 1;
                    poPlaneInfo->aePlaneFormat[0]   = LPixelFormat_L8;
                    break;

                case LPixelFormat_EXCH_10_YUYV_422:
                    // 40 bytes for 16 pixels ==> 2.5 bytes / pixel, granularity of 16 pixels or 40 bytes.
                    poPlaneInfo->auiWidthInBytes[0] = (poFrameSize->iWidth * 40) / 16;

                    // All VPG lines must be alined on 128 bytes.
                    poPlaneInfo->auiWidthInBytes[0] = (poPlaneInfo->auiWidthInBytes[0] + 127) & ~127;
                    poPlaneInfo->uiPlaneCount       = 1;
                    poPlaneInfo->aePlaneFormat[0]   = LPixelFormat_L8;
                    break;

                case LPixelFormat_EXCH_V210_422:
                    // 128 bytes for 48 pixels ==> 2.67 bytes / pixel, granularity of 48 pixels or 128 bytes.
                    poPlaneInfo->auiWidthInBytes[0] = (poFrameSize->iWidth * 128) / 48;

                    // All VPG lines must be alined on 128 bytes.
                    poPlaneInfo->auiWidthInBytes[0] = (poPlaneInfo->auiWidthInBytes[0] + 127) & ~127;
                    poPlaneInfo->uiPlaneCount       = 1;
                    poPlaneInfo->aePlaneFormat[0]   = LPixelFormat_L8;
                    break;

                case LPixelFormat_EXCH_8_YUAYVA_4224:
                    // 24 bytes for 8 pixels ==> 3 bytes / pixel, granularity of 8 pixels or 24 bytes.
                    poPlaneInfo->auiWidthInBytes[0] = poFrameSize->iWidth * 3;

                    // All VPG lines must be alined on 128 bytes.
                    poPlaneInfo->auiWidthInBytes[0] = (poPlaneInfo->auiWidthInBytes[0] + 127) & ~127;
                    poPlaneInfo->uiPlaneCount       = 1;
                    poPlaneInfo->aePlaneFormat[0]   = LPixelFormat_L8;
                    break;

                case LPixelFormat_EXCH_10_YUAYVA_4224:
                    // 8 bytes for 2 pixels ==> 4 bytes / pixel, granularity of 2 pixels or 8 bytes.
                    poPlaneInfo->auiWidthInBytes[0] = poFrameSize->iWidth * 4;

                    // All VPG lines must be alined on 128 bytes.
                    poPlaneInfo->auiWidthInBytes[0] = (poPlaneInfo->auiWidthInBytes[0] + 127) & ~127;
                    poPlaneInfo->uiPlaneCount       = 1;
                    poPlaneInfo->aePlaneFormat[0]   = LPixelFormat_L8;
                    break;

                case LPixelFormat_MP_Y8_U8V8_420:
                    poPlaneInfo->uiPlaneCount       = 2;
                    poPlaneInfo->aePlaneFormat[0]   = LPixelFormat_Y8;
                    poPlaneInfo->aePlaneFormat[1]   = LPixelFormat_U8V8;
                    poPlaneInfo->auiWidth[0]        = poFrameSize->iWidth;
                    poPlaneInfo->auiWidth[1]        = poFrameSize->iWidth / 2;
                    poPlaneInfo->auiWidthInBytes[0] = poFrameSize->iWidth;
                    poPlaneInfo->auiWidthInBytes[1] = poFrameSize->iWidth;
                    poPlaneInfo->auiHeight[0]       = poFrameSize->iHeight;
                    poPlaneInfo->auiHeight[1]       = poFrameSize->iHeight / 2;
                    break;

                case LPixelFormat_MP_Y8_U8_V8_420:
                    poPlaneInfo->uiPlaneCount       = 3;
                    poPlaneInfo->aePlaneFormat[0]   = LPixelFormat_Y8;
                    poPlaneInfo->aePlaneFormat[1]   = LPixelFormat_U8;
                    poPlaneInfo->aePlaneFormat[2]   = LPixelFormat_V8;
                    poPlaneInfo->auiWidth[0]        = poFrameSize->iWidth;
                    poPlaneInfo->auiWidth[1]        = poFrameSize->iWidth / 2;
                    poPlaneInfo->auiWidth[2]        = poFrameSize->iWidth / 2;
                    poPlaneInfo->auiWidthInBytes[0] = poFrameSize->iWidth;
                    poPlaneInfo->auiWidthInBytes[1] = poFrameSize->iWidth / 2;
                    poPlaneInfo->auiWidthInBytes[2] = poFrameSize->iWidth / 2;
                    poPlaneInfo->auiHeight[0]       = poFrameSize->iHeight;
                    poPlaneInfo->auiHeight[1]       = poFrameSize->iHeight / 2;
                    poPlaneInfo->auiHeight[2]       = poFrameSize->iHeight / 2;
                    break;

                case LPixelFormat_R8G8B8A8:
                case LPixelFormat_B8G8R8A8:
                    poPlaneInfo->uiPlaneCount       = 1;
                    poPlaneInfo->aePlaneFormat[0]   = LPixelFormat_R8G8B8A8;
                    poPlaneInfo->auiWidth[0]        = poFrameSize->iWidth;
                    poPlaneInfo->auiWidthInBytes[0] = poFrameSize->iWidth * 4;
                    poPlaneInfo->auiHeight[0]       = poFrameSize->iHeight;
                    break;

                default:
                    MsgLogErr("ERROR! Pixel format not supported.");
                    eStatus = LStatus_INVALID_PARAM;
                    break;
            }

            poPlaneInfo->auiPitchInBytes[0]   = poPlaneInfo->auiWidthInBytes[0];
            poRdFileMod->uiFileFrameSizeBytes = poPlaneInfo->auiPitchInBytes[0] * poPlaneInfo->auiHeight[0];
            poRdFileMod->uiMaxFrameCount      = uiMaxFrameCount;
        }       
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        snprintf(
                    poRdFileMod->szModuleName,
                    sizeof(poRdFileMod->szModuleName),
                    "%s%d",
                    g_szModuleNameBase,
                    g_uiReadFileModCount);

        eStatus = ModLnk_Init(
                    &(poRdFileMod->oOutLink),
                    hDevice,
                    uiBufferCount,
                    poBufferAttributes,
                    MFALSE,
                    0,
                    poRdFileMod->szModuleName);
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        poRdFileMod->bInit                  = MTRUE;
        poRdFileMod->bLoop                  = bLoop;
        poRdFileMod->uiMinFramePeriodUsec   = uiMinFramePeriodUsec;
        poRdFileMod->uiReadFrameCount       = 0;
        poRdFileMod->uiElapsedTimeUsec      = 0;

        strncpy_wz(poRdFileMod->szFilePath, szFilePath, sizeof(poRdFileMod->szFilePath));
    }
    else
    {
        RdFileMod_Cleanup(poRdFileMod);
    }

    ++g_uiReadFileModCount;

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       RdFileMod_Cleanup

Description:    .

\************************************************************************************************************/
void RdFileMod_Cleanup(ReadFileModule* poRdFileMod)
{
    MsgLog(2, "{...");

    if (poRdFileMod != MNULL)
    {
        ModLnk_Cleanup(&(poRdFileMod->oOutLink));

        poRdFileMod->bInit = MFALSE;
    }

    MsgLog(2, "...}");
}

/************************************************************************************************************\

Function:       RdFileMod_CpuThread

Description:    .

\************************************************************************************************************/
LStatus RdFileMod_CpuThread(void* pvData)
{
    if (pvData == MNULL)
    {
        MsgLogErr("ERROR! NULL data.");
        return LStatus_INVALID_PARAM;
    }

    ReadFileModule*     poRdFileMod     = (ReadFileModule*)pvData;
    LBuffer_PlaneInfo*  poFilePlaneInfo = &(poRdFileMod->oFilePlaneInfo);
    long int            iFileSize       = 0;
    MUINT               uiFrameCount    = 0;
    MUINT64             uiStartTime     = GetMonoTimeUsec();
    MUINT64             uiNextTime      = uiStartTime + poRdFileMod->uiMinFramePeriodUsec;
    MUINT               uiTmpBufferSize = 0;
    MUINT               i;

    ModThread_SetName(poRdFileMod->szModuleName);
    MsgLog(2, "Start thread %p.", pthread_self());

    for (i = 0; i < poFilePlaneInfo->uiPlaneCount; i++)
    {
        uiTmpBufferSize = max(poFilePlaneInfo->auiWidthInBytes[i], uiTmpBufferSize);
    }

    // Many times faster using an intermediate buffer in system memory.
    MUINT8* puiTmpBuffer = malloc(uiTmpBufferSize);

    poRdFileMod->uiReadFrameCount   = 0;
    poRdFileMod->uiElapsedTimeUsec  = 0;

    FILE* hFile = fopen(poRdFileMod->szFilePath, "rb");

    if ((hFile != MNULL) && (puiTmpBuffer != MNULL))
    {
        fseek(hFile, 0, SEEK_END);
        iFileSize = ftell(hFile);
        fseek(hFile, 0, SEEK_SET);
    }
    else
    {
        MsgLogErr("ERROR! Cannot open file: %s", poRdFileMod->szFilePath);
        poRdFileMod->oCpuThread.bKillThread = MTRUE;
    }

    while (!poRdFileMod->oCpuThread.bKillThread)
    {
        BufferInfo* poDstBuffer;

        LStatus eStatus = ModLnk_GetReturnedBuffer(&(poRdFileMod->oOutLink), 100, MNULL, &poDstBuffer);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            // Check if we are at the end of the file
            if (((iFileSize > 0) && ((iFileSize - ftell(hFile)) < (long int)poRdFileMod->uiFileFrameSizeBytes))
                || (uiFrameCount >= poRdFileMod->uiMaxFrameCount))
            {
                if (poRdFileMod->bLoop && (0 <= fseek(hFile, 0, SEEK_SET)))
                {
                    uiFrameCount = 0;
                }
                else
                {
                    MsgLog(4, "END-OF-STREAM");

                    poDstBuffer->bEndOfStream = MTRUE;
                    poRdFileMod->oCpuThread.bKillThread = MTRUE;
                    eStatus = LStatus_END_OF_STREAM;
                }
            }
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            LBuffer_Type eBufferType = LBuffer_GetType(poDstBuffer->hBuffer);

            if ((eBufferType == LBuffer_Type_VIDEO) || (eBufferType == LBuffer_Type_SYSTEM_VIDEO))
            {
                LBuffer_PlaneInfo oPlaneInfo;

                eStatus = LBuffer_GetPlaneInfo(poDstBuffer->hBuffer, &oPlaneInfo);

                if (LSTATUS_IS_SUCCESS(eStatus)
                    && (oPlaneInfo.uiPlaneCount == poFilePlaneInfo->uiPlaneCount))
                {
                    MUINT8* apuiBuffer[LBUFFER_MAXPLANE];

                    MsgLog(4, "Read frame[%u] at offset %ld... (Buffer[%u])",
                              uiFrameCount, ftell(hFile), poDstBuffer->uiId);

                    eStatus = LBuffer_BeginAccess(
                                    poDstBuffer->hBuffer,
                                    0,
                                    oPlaneInfo.uiPlaneCount,
                                    apuiBuffer);

                    if (LSTATUS_IS_SUCCESS(eStatus))
                    {
                        for (i = 0; i < oPlaneInfo.uiPlaneCount; i++)
                        {
                            MUINT8* puiBuffer      = apuiBuffer[i];
                            MUINT   uiBytesPerLine = min(oPlaneInfo.auiWidthInBytes[i],
                                                         poFilePlaneInfo->auiWidthInBytes[i]);
                            MUINT   uiLineCount    = min(oPlaneInfo.auiHeight[i],
                                                         poFilePlaneInfo->auiHeight[i]);
                            long int uiSeekPerLine = (poFilePlaneInfo->auiPitchInBytes[i] > uiBytesPerLine)
                                                     ? (poFilePlaneInfo->auiPitchInBytes[i] - uiBytesPerLine)
                                                     : 0;
                            MUINT y;
                            for (y = 0; y < uiLineCount; y++)
                            {
                                (void) fread(puiTmpBuffer, 1, uiBytesPerLine, hFile);
                                memcpy(puiBuffer, puiTmpBuffer, uiBytesPerLine);

                                if (uiSeekPerLine > 0)
                                {
                                    fseek(hFile, uiSeekPerLine, SEEK_CUR);
                                }

                                puiBuffer += oPlaneInfo.auiPitchInBytes[i];
                            }

                            if (uiLineCount < poFilePlaneInfo->auiHeight[i])
                            {
                                long int iSeekOffset = (poFilePlaneInfo->auiHeight[i] - uiLineCount)
                                                       * poFilePlaneInfo->auiPitchInBytes[i];
                                fseek(hFile, iSeekOffset, SEEK_CUR);
                            }
                        }

                        LBuffer_EndAccess(poDstBuffer->hBuffer);
                        uiFrameCount++;

                        MsgLog(4, "Read next frame done.");
                    }
                    else
                    {
                        MsgLogErr("ERROR! Cannot begin access");
                    }
                }
                else
                {
                    MsgLogErr("ERROR! Cannot get or invalid plane info.");
                }
            }
            else if ((eBufferType == LBuffer_Type_LINEAR)
                     || (eBufferType == LBuffer_Type_SYSTEM_LINEAR))
            {
                MUINT8* puiBuffer;

                MsgLog(4, "Read linear blop at offset %ld (Buffer[%u])...", ftell(hFile), poDstBuffer->uiId);

                eStatus = LBuffer_BeginAccess(poDstBuffer->hBuffer, 0, 1, &puiBuffer);

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    size_t uiMaxSize  = ((iFileSize > 0) ? poRdFileMod->oFilePlaneInfo.auiWidth[0] : 4096);
                    size_t uiSizeRead = fread(puiTmpBuffer, 1, uiMaxSize, hFile);

                    memcpy(puiBuffer, puiTmpBuffer, uiSizeRead);

                    LBuffer_EndAccess(poDstBuffer->hBuffer);

                    MsgLog(4, "Read next frame done (size= %u/%u).", (MUINT)uiSizeRead, (MUINT)uiMaxSize);

                    if (uiSizeRead > 0)
                    {
                        poDstBuffer->uiSizeBytes    = uiSizeRead;
                        poDstBuffer->uiStartOffset  = 0;
                    }
                    else
                    {
                        if (poRdFileMod->bLoop && (0 <= fseek(hFile, 0, SEEK_SET)))
                        {
                            uiFrameCount = 0;
                            eStatus = LStatus_FAIL;
                        }
                        else
                        {
                            MsgLog(4, "END-OF-STREAM");

                            poDstBuffer->bEndOfStream = MTRUE;
                            poRdFileMod->oCpuThread.bKillThread = MTRUE;
                            eStatus = LStatus_END_OF_STREAM;
                        }
                    }
                }
            }
        }

        if (LSTATUS_IS_SUCCESS(eStatus)
            || (eStatus == LStatus_END_OF_STREAM))
        {
            MUINT64 uiCurTime = GetMonoTimeUsec();

            if (uiCurTime < uiNextTime)
            {
                usleep(uiNextTime - uiCurTime);
            }

            poDstBuffer->uiTimestampUsec = GetMonoTimeUsec();
            ModLnk_SubmitBuffer(&(poRdFileMod->oOutLink), poDstBuffer, MNULL, NO_TAG);

            poRdFileMod->uiReadFrameCount++;
            poRdFileMod->uiElapsedTimeUsec = (MUINT)(GetMonoTimeUsec() - uiStartTime);

            uiNextTime += poRdFileMod->uiMinFramePeriodUsec;
        }
        else
        {
            ModLnk_ReleaseBuffer(&(poRdFileMod->oOutLink), poDstBuffer);
        }

        if (LSTATUS_IS_FAIL(eStatus)
            && (eStatus != LStatus_TIMEOUT))
        {
            usleep(1000);
        }
    }

    if (hFile != MNULL)
    {
        fclose(hFile);
    }

    if (puiTmpBuffer != MNULL)
    {
        free(puiTmpBuffer);
    }

    MsgLog(2, "Kill thread.");

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       RdFileMod_Start

Description:    .

\************************************************************************************************************/
LStatus RdFileMod_Start(ReadFileModule* poRdFileMod)
{
    MsgLog(2, "{...");

    LStatus eStatus = LStatus_INVALID_PARAM;

    if ((poRdFileMod != MNULL) && poRdFileMod->bInit)
    {
        if (poRdFileMod->oOutLink.uiSubmitCount > 0)
        {
            eStatus = ModThread_Start(&(poRdFileMod->oCpuThread), poRdFileMod, RdFileMod_CpuThread);
        }
        else
        {
            MsgLogErr("ERROR! Bad connection.");
            eStatus = LStatus_FAIL;
        }
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       RdFileMod_Stop

Description:    .

\************************************************************************************************************/
void RdFileMod_Stop(ReadFileModule* poRdFileMod)
{
    MsgLog(2, "{...");

    if (poRdFileMod != MNULL)
    {
        ModThread_Stop(&(poRdFileMod->oCpuThread));
    }

    MsgLog(2, "...}");
}

