/************************************************************************************************************\

Module Name:    ExchConvertModule.c

Description:    .

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "ExchConvertModule.h"

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------
//static const    MCHAR8      g_szModuleNameBase[]    = "ExchConv";
static          MUINT32     g_uiExchConvertModCount = 0;

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Function:       ExchConvertMod_Init

Description:    .

\************************************************************************************************************/
LStatus ExchConvertMod_Init(
            ExchConvertModule*  poExchConvertMod,
            LDevice_Handle      hDevice,
            MUINT               uiOutBufferCount,
            LBuffer_Attributes* poOutBufferAttributes)
{
    MINT         i = 0;
    LPixelFormat eDerivedPixelFormat = LPixelFormat_INVALID;

    MsgLog(2, "{...");

    ExchConvertMod_Cleanup(poExchConvertMod);

    LStatus eStatus = ((poExchConvertMod != MNULL)
                       && (hDevice != MNULL)
                       && (poOutBufferAttributes != MNULL)
                       && (poOutBufferAttributes->eAttributeType == LBuffer_Type_VIDEO))
                      ? LStatus_OK : LStatus_INVALID_PARAM;

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        snprintf(
                    poExchConvertMod->szModuleName,
                    sizeof(poExchConvertMod->szModuleName),
                    "ExchConvertMod%d",
                    g_uiExchConvertModCount);

        LDeviceThread_CreateAttributes oAttributes = {LDeviceThread_AttributeType_CREATE, MTRUE, MTRUE, MFALSE, ""};

        snprintf(oAttributes.szName, sizeof(oAttributes.szName), "cvt%d", i);

        eStatus = LDeviceThread_Create(hDevice, &oAttributes.eType, &(poExchConvertMod->hDevThread));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LBlit_Create(hDevice, poExchConvertMod->hDevThread, MNULL, &(poExchConvertMod->hBlit));
    }

    if (poOutBufferAttributes->oVideoAttributes.ePixelFormat == LPixelFormat_MP_Y8_U8V8_A8_422)
    {
        eDerivedPixelFormat = LPixelFormat_MP_Y8_U8V8_422;
    }
    else if (poOutBufferAttributes->oVideoAttributes.ePixelFormat == LPixelFormat_MP_Y82_U82V82_A10_X2_422)
    {
        eDerivedPixelFormat = LPixelFormat_MP_Y82_U82V82_X2_422;
    }

    if (eDerivedPixelFormat != LPixelFormat_INVALID)
    {
        poExchConvertMod->bIsDerivedBuffer = MTRUE;

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = ModLnk_Init(
                            &(poExchConvertMod->oOutLink),
                            hDevice,
                            uiOutBufferCount,
                            MNULL,
                            MTRUE,
                            sizeof(LBuffer_Handle),
                            poExchConvertMod->szModuleName);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            LBuffer_Options oOptions;

            oOptions.uiOptionTypes      = LBuffer_OptionType_AutoSync;
            oOptions.eAutoSyncOption    = ENABLE_AUTO_SYNC ? LBuffer_AutoSyncOption_PerRegion
                                                           : LBuffer_AutoSyncOption_Disabled;
            oOptions.uiAutoSyncTimeout  = LDEVICETHREAD_NO_TIMEOUT;

            for (i = 0; i <poExchConvertMod->oOutLink.uiBufferCount; i++)
            {
                LBuffer_Handle* phPrivBuffer = (LBuffer_Handle*)poExchConvertMod->oOutLink.aoBufferInfo[i].pvPrivateData;

                poExchConvertMod->oOutLink.aoBufferInfo[i].hBuffer = MNULL;
                *phPrivBuffer = MNULL;

                eStatus = LBuffer_Create(hDevice,
                                         (LBuffer_Type*)poOutBufferAttributes,
                                         phPrivBuffer);

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    LBuffer_Handle  hDerivedBuffer;

                    eStatus = LBuffer_SetOptions(*phPrivBuffer, &oOptions);

                    if (LSTATUS_IS_SUCCESS(eStatus))
                    {
                        eStatus = LBuffer_Derive(hDevice, *phPrivBuffer, LBuffer_PlaneID_ALL, eDerivedPixelFormat, &hDerivedBuffer);

                        if (LSTATUS_IS_SUCCESS(eStatus))
                        {
                            poExchConvertMod->oOutLink.aoBufferInfo[i].hBuffer = hDerivedBuffer;
                        }
                    }
                }

                if (LSTATUS_IS_FAIL(eStatus))
                {
                    break;
                }
            }
        }
    }
    else
    {        
        poExchConvertMod->bIsDerivedBuffer = MFALSE;

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = ModLnk_Init(
                        &(poExchConvertMod->oOutLink),
                        hDevice,
                        uiOutBufferCount,
                        poOutBufferAttributes,
                        MFALSE,
                        0,
                        poExchConvertMod->szModuleName);
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        poExchConvertMod->oSrcPos.iX       = 0;
        poExchConvertMod->oSrcPos.iY       = 0;
        poExchConvertMod->oDstRect.iLeft   = 0;
        poExchConvertMod->oDstRect.iTop    = 0;
        poExchConvertMod->oDstRect.iRight  = poOutBufferAttributes->oVideoAttributes.uiWidth;
        poExchConvertMod->oDstRect.iBottom = poOutBufferAttributes->oVideoAttributes.uiHeight;
    }
    else
    {
        ExchConvertMod_Cleanup(poExchConvertMod);
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       ExchConvertMod_Cleanup

Description:    .

\************************************************************************************************************/
void ExchConvertMod_Cleanup(ExchConvertModule* poExchConvertMod)
{
    MsgLog(2, "{...");

    if (poExchConvertMod != MNULL)
    {
        if (poExchConvertMod->bIsDerivedBuffer)
        {
            LBuffer_Handle* phBuffer;
            MINT i;

            for (i = 0; i <poExchConvertMod->oOutLink.uiBufferCount; i++)
            {
                // Destroy the derived buffer first.
                if (poExchConvertMod->oOutLink.aoBufferInfo[i].hBuffer != MNULL)
                {
                    LBuffer_Destroy(poExchConvertMod->oOutLink.aoBufferInfo[i].hBuffer);
                    poExchConvertMod->oOutLink.aoBufferInfo[i].hBuffer = MNULL;
                }

                // Then destroy the "main" buffer.
                phBuffer = (LBuffer_Handle*)poExchConvertMod->oOutLink.aoBufferInfo[i].pvPrivateData;
                if (*phBuffer != MNULL)
                {
                    LBuffer_Destroy(*phBuffer);
                    *phBuffer = MNULL;
                }
            }

            poExchConvertMod->bIsDerivedBuffer = MFALSE;
        }

        ModLnk_Cleanup(&(poExchConvertMod->oOutLink));

        if (poExchConvertMod->hBlit != MNULL)
        {
            while (LSTATUS_IS_FAIL(LBlit_Destroy(poExchConvertMod->hBlit)))
            {
                MsgLog(0, "LBlit_Destroy failed! Retry...");
                usleep(1000*1000);
            }
        }

        if (poExchConvertMod->hDevThread != MNULL)
        {
            LDeviceThread_Destroy(poExchConvertMod->hDevThread);
        }

        poExchConvertMod->hBlit            = MNULL;
        poExchConvertMod->hDevThread       = MNULL;
        poExchConvertMod->oInLink.poModLnk = MNULL;
    }

    MsgLog(2, "...}");
}

/************************************************************************************************************\

Function:       ExchConvertMod_CpuThread

Description:    .

\************************************************************************************************************/
LStatus ExchConvertMod_CpuThread(void* pvData)
{
    if (pvData == MNULL)
    {
        MsgLogErr("ERROR! NULL data.");
        return LStatus_INVALID_PARAM;
    }

    ExchConvertModule* poExchConvertMod   = (ExchConvertModule*)pvData;
    BufferInfo* poDstBuffer = MNULL;
    BufferInfo* poSrcBuffer = MNULL;

    ModThread_SetName(poExchConvertMod->szModuleName);
    MsgLog(2, "Start thread.");

    while (!poExchConvertMod->oCpuThread.bKillThread)
    {
        LStatus eStatus = LStatus_OK;

        if (poDstBuffer == MNULL)
        {
            eStatus = ModLnk_GetReturnedBuffer(
                        &(poExchConvertMod->oOutLink),
                        100,
                        poExchConvertMod->hDevThread,
                        &poDstBuffer);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = ModLnkIn_GetSubmittedBuffer(
                        &(poExchConvertMod->oInLink),
                        100,
                        0,
                        poExchConvertMod->hDevThread,
                        &poSrcBuffer,
                        MNULL,
                        MNULL);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            if (poSrcBuffer->bEndOfStream)
            {
                MsgLog(4, "END-OF-STREAM");

                poDstBuffer->bEndOfStream = MTRUE;
                poExchConvertMod->oCpuThread.bKillThread = MTRUE;
            }
            else
            {
                MsgLog(4, "LBlit_ConvertEXCH(DstBuffer[%u], SrcBuffer[%u])...",
                            poDstBuffer->uiId, poSrcBuffer->uiId);

                LBuffer_Handle hDstBuffer = poExchConvertMod->bIsDerivedBuffer ? *((LBuffer_Handle*)poDstBuffer->pvPrivateData) : poDstBuffer->hBuffer;

                eStatus = LBlit_ConvertEXCH(poExchConvertMod->hBlit,
                                                poSrcBuffer->hBuffer,
                                                hDstBuffer,
                                                0,
                                                0,
                                                MFALSE);

                MsgLog(4, "LBlit_ConvertEXCH done. (status= %d - %s)", eStatus, GetStatusStr(eStatus));
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                poDstBuffer->uiTimestampUsec = poSrcBuffer->uiTimestampUsec;
                ModLnk_SubmitBuffer(&(poExchConvertMod->oOutLink), poDstBuffer, poExchConvertMod->hDevThread, NO_TAG);
                ModLnkIn_ReturnBuffer(&(poExchConvertMod->oInLink), poSrcBuffer, poExchConvertMod->hDevThread, NO_TAG);
            }
            else
            {
                ModLnk_ReleaseBuffer(&(poExchConvertMod->oOutLink), poDstBuffer);
                ModLnkIn_ReturnBuffer(&(poExchConvertMod->oInLink), poSrcBuffer, MNULL, NO_TAG);
            }

            poDstBuffer = MNULL;
            poSrcBuffer = MNULL;
        }

        if (LSTATUS_IS_FAIL(eStatus)
            && (eStatus != LStatus_TIMEOUT))
        {
            usleep(1000);
        }
    }

    if (poDstBuffer != MNULL)
    {
        ModLnk_ReleaseBuffer(&(poExchConvertMod->oOutLink), poDstBuffer);
    }

    MsgLog(2, "Kill thread.");

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       ExchConvertMod_Start

Description:    .

\************************************************************************************************************/
LStatus ExchConvertMod_Start(ExchConvertModule* poExchConvertMod)
{
    LStatus eStatus = LStatus_INVALID_PARAM;

    MsgLog(2, "{...");

    if ((poExchConvertMod != MNULL)
        && (poExchConvertMod->hBlit != MNULL))
    {
        if ((poExchConvertMod->oInLink.poModLnk != MNULL)
            && (poExchConvertMod->oOutLink.uiSubmitCount > 0))
        {
            eStatus = ModThread_Start(&(poExchConvertMod->oCpuThread), poExchConvertMod, ExchConvertMod_CpuThread);
        }
        else
        {
            MsgLogErr("ERROR! Bad connection.");
            eStatus = LStatus_FAIL;
        }
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       ExchConvertMod_Stop

Description:    .

\************************************************************************************************************/
void ExchConvertMod_Stop(ExchConvertModule* poExchConvertMod)
{
    MsgLog(2, "{...");

    if (poExchConvertMod != MNULL)
    {
        ModThread_Stop(&(poExchConvertMod->oCpuThread));
    }

    MsgLog(2, "...}");
}

