/************************************************************************************************************\

Module Name:    EncoderModule.c

Description:    .

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "EncoderModule.h"
#include "VpeModule.h"

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------
#define NB_SEI_ENTRY 32

static const MUINT      g_uiPadOffset           = 0;       // Set to a non-zero value to test padding.
static const MBOOL      g_bDoPadding            = MTRUE;   // Set to MFALSE to disable padding.
static const MBOOL      g_bWildStop             = MFALSE;
static const MBOOL      g_bSetScalinglist       = MTRUE;
static const MUINT      g_uiSeiMsgPayloadType   = 5;
static const char       g_szSeiMsgUuid[8]       = {'S', 'E', 'I', 'T', 'S', 'U', 'I', 'D'};

static const MUINT8 g_uiScListBase4x4[4*4] =
{
    16, 21, 26, 31,
    21, 26, 31, 36,
    26, 31, 36, 41,
    31, 36, 41, 48,
};

static const MUINT8 g_uiScListBase8x8[8*8] =
{
    16, 18, 20, 22, 24, 26, 28, 30,
    18, 18, 22, 24, 26, 28, 30, 32,
    20, 22, 24, 26, 28, 30, 32, 34,
    22, 24, 26, 28, 30, 32, 34, 36,
    24, 26, 28, 30, 32, 34, 36, 39,
    26, 28, 30, 32, 34, 36, 39, 42,
    28, 30, 32, 34, 36, 39, 42, 45,
    30, 32, 34, 36, 39, 42, 45, 48,
};

#pragma pack(push, 1)
typedef struct
{
    // Here, the timestamp is interleaved with pseudo UUID in order to avoid false start code sequence.
    MUINT16 m_uiTsUsec0;
    MCHAR8  m_auiUuidA[2];
    MUINT16 m_uiTsUsec1;
    MCHAR8  m_auiUuidB[2];
    MUINT16 m_uiTsUsec2;
    MCHAR8  m_auiUuidC[2];
    MUINT16 m_uiTsUsec3;
    MCHAR8  m_auiUuidD[2];

} SeiTsUserMsgPayload;

typedef struct
{
    MUINT8  uiPayloadType;
    MUINT8  uiPayloadSize;

    SeiTsUserMsgPayload oPayload;

} SeiTsUserMsg;
#pragma pack(pop)

typedef struct
{
    SeiTsUserMsg*   aoSeiMsgEntry;
    MBOOL           abSeiMsgEntryFree[NB_SEI_ENTRY];
    LBuffer_Handle  hSeiBuffer;
    MUINT           uiNextSeiMsgEntry;
} EncModPrivData;

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

static const    MCHAR8      g_szModuleNameBase[]    = "Enc";
static          MUINT32     g_uiEncModCount         = 0;

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Function:       EncMod_Init

Description:    .

\************************************************************************************************************/
LStatus EncMod_Init(
            EncoderModule*      poEncMod,
            LDevice_Handle      hDevice,
            MUINT               uiOutBufferCount,
            LBuffer_Attributes* poOutBufferAttributes,
            LH264E_EncoderMode  eEncodeMode,
            LH264_Profile       eProfile,
            LH264_Level         eLevel,
            MUINT               uiSliceCount,
            MUINT               uiFaq,
            MBOOL               bDfEnabled)
{
    MsgLog(2, "{...");

    EncMod_Cleanup(poEncMod);

    LStatus eStatus = ((poEncMod != MNULL)
                       && (hDevice != MNULL)
                       && (poOutBufferAttributes != MNULL)
                       && (uiSliceCount > 0)
                       && (uiSliceCount <= ENC_MAX_NB_SLICES)
                       && (uiFaq >= 0)
                       && (uiFaq <= 100))
                      ? LStatus_OK : LStatus_INVALID_PARAM;

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LDeviceThread_CreateAttributes oAttributes = {LDeviceThread_AttributeType_CREATE, MTRUE, MTRUE, MFALSE, "encin"};

        eStatus = LDeviceThread_Create(hDevice, &oAttributes.eType, &(poEncMod->hDevThreadIn));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LDeviceThread_CreateAttributes oAttributes = {LDeviceThread_AttributeType_CREATE, MTRUE, MTRUE, MFALSE, "encout"};

        eStatus = LDeviceThread_Create(hDevice, &oAttributes.eType, &(poEncMod->hDevThreadOut));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LDeviceThread_CreateAttributes oAttributes = {LDeviceThread_AttributeType_CREATE, MTRUE, MTRUE, MFALSE, "encpad"};

        eStatus = LDeviceThread_Create(hDevice, &oAttributes.eType, &(poEncMod->hDevThreadPad));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LDeviceThread_CreateAttributes oAttributes = {LDeviceThread_AttributeType_CREATE, MTRUE, MTRUE, MFALSE, "enccpy"};

        eStatus = LDeviceThread_Create(hDevice, &oAttributes.eType, &(poEncMod->hDevThreadCopy));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LH264E_CreateOpt oCreateOpt;

        memset(&oCreateOpt, 0, sizeof(oCreateOpt));

        MUINT uiMaxNaluSize = 0;

        if (poOutBufferAttributes->eAttributeType == LBuffer_Type_LINEAR)
        {
            uiMaxNaluSize = poOutBufferAttributes->oLinearAttributes.uiSize;
        }
        else if (poOutBufferAttributes->eAttributeType == LBuffer_Type_SYSTEM_LINEAR)
        {
            uiMaxNaluSize = poOutBufferAttributes->oSystemLinearAttributes.uiSize;
        }

        oCreateOpt.eType            = LH264E_CreateOptTypeHeader_STANDARD;
        oCreateOpt.hDTInput         = (MUINT64)poEncMod->hDevThreadIn;
        oCreateOpt.hDTOutput        = (MUINT64)poEncMod->hDevThreadOut;
        oCreateOpt.flCreateOptFlags = 0;
        if ((uiMaxNaluSize < (3840*2160*2)) &&
            (LH264_Profile_BASELINE != eProfile) &&
            (LH264_Profile_EXTENDED != eProfile) &&
            (LH264_Profile_CAVLC444INTRA != eProfile))
        {
            oCreateOpt.flCreateOptFlags |= LH264E_CreateOptFlag_CABAC;
        }
        if (LH264_Profile_CAVLC444INTRA == eProfile)
        {
            oCreateOpt.flCreateOptFlags |=  LH264E_CreateOptFlag_IDRONLY;
        }
        oCreateOpt.eEncoderMode     = eEncodeMode;
        oCreateOpt.ePriority        = LH264_Priority_REALTIME;
        oCreateOpt.eProfile         = eProfile;
        oCreateOpt.eLevel           = eLevel;
        oCreateOpt.flConstraints    = 0;
        oCreateOpt.uiGOPOffset      = 0;
        oCreateOpt.eStartCodeMode   = LH264E_StartCodeMode_NORMAL;

        if (ENABLE_AUTO_SYNC)
        {
           oCreateOpt.flCreateOptFlags  |= LH264E_CreateOptFlag_BUFFERAUTOSYNC;
           oCreateOpt.flCreateOptFlags  |= LH264E_CreateOptFlag_PICTUREAUTOSYNCONRELEASE;
           oCreateOpt.uiAutoSyncTimeOut  = LDEVICETHREAD_NO_TIMEOUT;
        }

        eStatus = LH264E_Create(hDevice, (LH264E_CreateOptTypeHeader*)&oCreateOpt, &(poEncMod->hEncoder));

        if (LSTATUS_IS_SUCCESS(eStatus) && (g_uiMsgLogLevel > 0))
        {
            printf("LH264E_CreateOpt\n"
                   "{\n"
                   "    .eType              = %d\n"
                   "    .hDTInput           = %p\n"
                   "    .hDTOutput          = %p\n"
                   "    .flCreateOptFlags   = 0x%08X\n"
                   "    .eEncoderMode       = %d\n"
                   "    .ePriority          = %d\n"
                   "    .eProfile           = %d\n"
                   "    .eLevel             = %d\n"
                   "    .flConstraints      = 0x%08X\n"
                   "    .uiGOPOffset        = %u\n"
                   "    .eStartCodeMode     = %d\n"
                   "    .uiAutoSyncTimeOut  = %u\n"
                   "}\n",
                   oCreateOpt.eType,
                   (void*)oCreateOpt.hDTInput,
                   (void*)oCreateOpt.hDTOutput,
                   oCreateOpt.flCreateOptFlags,
                   oCreateOpt.eEncoderMode,
                   oCreateOpt.ePriority,
                   oCreateOpt.eProfile,
                   oCreateOpt.eLevel,
                   oCreateOpt.flConstraints,
                   oCreateOpt.uiGOPOffset,
                   oCreateOpt.eStartCodeMode,
                   oCreateOpt.uiAutoSyncTimeOut);
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LVPE_Create(hDevice, poEncMod->hDevThreadPad, MNULL, &poEncMod->hVpe);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            LVP_Rect oClipRect;

            oClipRect.bEnable = MFALSE;

            eStatus = LVPE_SetClipRect(poEncMod->hVpe, &oClipRect);
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LBlit_Create(hDevice, poEncMod->hDevThreadCopy, MNULL, &poEncMod->hBlit);
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {

        snprintf(
                    poEncMod->szModuleName,
                    sizeof(poEncMod->szModuleName),
                    "%s%d",
                    g_szModuleNameBase,
                    g_uiEncModCount);

        eStatus = ModLnk_Init(
                    &(poEncMod->oOutLink),
                    hDevice,
                    uiOutBufferCount,
                    poOutBufferAttributes,
                    MFALSE,
                    0,
                    poEncMod->szModuleName);
    }

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        poEncMod->pvPrivData = malloc(sizeof(EncModPrivData));

        if(poEncMod->pvPrivData != MNULL)
        {
            EncModPrivData* poPrivData = (EncModPrivData*) poEncMod->pvPrivData;

            eStatus = LBuffer_CreateLinear(
                                hDevice,
                                NB_SEI_ENTRY * sizeof(SeiTsUserMsg),
                                &poPrivData->hSeiBuffer);

            if(LSTATUS_IS_SUCCESS(eStatus))
            {
                eStatus = LBuffer_BeginAccess(poPrivData->hSeiBuffer, 0, 1, (MUINT8**)&(poPrivData->aoSeiMsgEntry));
            }

            if(LSTATUS_IS_SUCCESS(eStatus))
            {
                MUINT i;
                for(i = 0; i < NB_SEI_ENTRY; i++)
                {
                    poPrivData->aoSeiMsgEntry[i].uiPayloadType  = g_uiSeiMsgPayloadType;
                    poPrivData->aoSeiMsgEntry[i].uiPayloadSize  = sizeof(poPrivData->aoSeiMsgEntry[i].oPayload);

                    poPrivData->aoSeiMsgEntry[i].oPayload.m_auiUuidA[0] = g_szSeiMsgUuid[0];
                    poPrivData->aoSeiMsgEntry[i].oPayload.m_auiUuidA[1] = g_szSeiMsgUuid[1];
                    poPrivData->aoSeiMsgEntry[i].oPayload.m_auiUuidB[0] = g_szSeiMsgUuid[2];
                    poPrivData->aoSeiMsgEntry[i].oPayload.m_auiUuidB[1] = g_szSeiMsgUuid[3];
                    poPrivData->aoSeiMsgEntry[i].oPayload.m_auiUuidC[0] = g_szSeiMsgUuid[4];
                    poPrivData->aoSeiMsgEntry[i].oPayload.m_auiUuidC[1] = g_szSeiMsgUuid[5];
                    poPrivData->aoSeiMsgEntry[i].oPayload.m_auiUuidD[0] = g_szSeiMsgUuid[6];
                    poPrivData->aoSeiMsgEntry[i].oPayload.m_auiUuidD[1] = g_szSeiMsgUuid[7];

                    poPrivData->abSeiMsgEntryFree[i] = MTRUE;
                }

                poPrivData->uiNextSeiMsgEntry = 0;
            }
        }
        else
        {
            eStatus = LStatus_OUT_OF_MEMORY;
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        poEncMod->bEncoding  = MFALSE;
        poEncMod->uiNbSlices = uiSliceCount;
        poEncMod->eProfile   = eProfile;
        poEncMod->eLevel     = eLevel;

        //----------------------------------------------------------------------------------------------------

        memset(&(poEncMod->oEncodeOpt), 0, sizeof(poEncMod->oEncodeOpt));

        poEncMod->oEncodeOpt.eType                      = LH264E_EncodeOptTypeHeader_STANDARD;
        poEncMod->oEncodeOpt.hPicture                   = 0;
        poEncMod->oEncodeOpt.ePictureType               = LH264_PictureType_UNSPECIFIED;
        poEncMod->oEncodeOpt.fQP                        = 26.0;
        poEncMod->oEncodeOpt.flPictureFlags             = 0;
        poEncMod->oEncodeOpt.uiSliceIndex               = 0;
        poEncMod->oEncodeOpt.ePictureStruct             = LH264_PictureStruct_PROGRESSIVEFRAME;
        poEncMod->oEncodeOpt.auiPictureTimestamp[0]     = 0;
        poEncMod->oEncodeOpt.auiPictureTimestamp[1]     = 0;
        poEncMod->oEncodeOpt.auiPictureTimestamp[2]     = 0;
        poEncMod->oEncodeOpt.uiRecoveryFrameCnt         = 0;
        poEncMod->oEncodeOpt.bExactMatchFlag            = MFALSE;
        poEncMod->oEncodeOpt.bBrokenLinkFlag            = MFALSE;
        poEncMod->oEncodeOpt.hSEIRbspBuffer             = 0;
        poEncMod->oEncodeOpt.uiSEIRbspStartOffset       = 0;
        poEncMod->oEncodeOpt.uiSEIRbspSize              = 0;

        //----------------------------------------------------------------------------------------------------

        memset(&(poEncMod->oSeqData), 0, sizeof(poEncMod->oSeqData));

        poEncMod->oSeqData.eType                        = LH264_StreamInfoTypeHeader_ENCODERSEQUENCEDATA;
        poEncMod->oSeqData.flSequenceDataFlags          = 0
            | LH264_StreamInfoEncoderSequenceDataFlag_PICTUREINFOPRESENT
            | LH264_StreamInfoEncoderSequenceDataFlag_RATECONTROLPRESENT;

        // LH264_StreamInfoEncoderSequenceDataFlag_PICTUREINFOPRESENT
        poEncMod->oSeqData.uiPictureWidth               = 640;
        poEncMod->oSeqData.uiPictureHeight              = 480;
        poEncMod->oSeqData.ePixelFormat                 = LPixelFormat_MP_Y8_U8V8_420;
        poEncMod->oSeqData.eScanMode                    = LH264_ScanMode_PROGRESSIVE;
        poEncMod->oSeqData.eFirstField                  = LH264_TOPFIELD;
        poEncMod->oSeqData.oFrameRate.uiNumerator       = 60;
        poEncMod->oSeqData.oFrameRate.uiDenominator     = 1;
        poEncMod->oSeqData.eDFDisableIdc                = bDfEnabled
                                                          ? LH264E_DFDisableIdc_SLICEENABLED
                                                          : LH264E_DFDisableIdc_DISABLED;
        poEncMod->oSeqData.iDFAlphaC0Param              = 0;
        poEncMod->oSeqData.iDFBetaParam                 = 0;
        poEncMod->oSeqData.fFAQStrength                 = (MFLOAT32)uiFaq / 100;

        // LH264_StreamInfoEncoderSequenceDataFlag_RATECONTROLPRESENT;
        poEncMod->oSeqData.flRateControlFlags           = 0;
        poEncMod->oSeqData.uiTargetBitRate              = 1 * 1000 * 1000;
        poEncMod->oSeqData.fCorrectionTensor            = 1.0;
        poEncMod->oSeqData.fQPMin                       = 10;
        poEncMod->oSeqData.fQPMax                       = 48;
        poEncMod->oSeqData.eHRDMode                     = LH264_HRDMode_CBR;
        switch (poEncMod->eLevel)
        {
            case LH264_Level_1B:
                poEncMod->oSeqData.uiCPBSizeInBits      = 360000;
                break;

            case LH264_Level_1:
                poEncMod->oSeqData.uiCPBSizeInBits      = 210000;
                break;

            case LH264_Level_1_1:
                poEncMod->oSeqData.uiCPBSizeInBits      = 600000;
                break;

            case LH264_Level_1_2:
                poEncMod->oSeqData.uiCPBSizeInBits      = 1200000;
                break;

            case LH264_Level_1_3:
            case LH264_Level_2:
                poEncMod->oSeqData.uiCPBSizeInBits      = 2400000;
                break;

            case LH264_Level_2_1:
            case LH264_Level_2_2:
                poEncMod->oSeqData.uiCPBSizeInBits      = 4800000;
                break;

            case LH264_Level_3:
                poEncMod->oSeqData.uiCPBSizeInBits      = 12000000;
                break;

            case LH264_Level_3_1:
                poEncMod->oSeqData.uiCPBSizeInBits      = 16800000;
                break;

            case LH264_Level_3_2:
                poEncMod->oSeqData.uiCPBSizeInBits      = 24000000;
                break;

            case LH264_Level_4:
                poEncMod->oSeqData.uiCPBSizeInBits      = 30000000;
                break;

            case LH264_Level_4_1:
            case LH264_Level_4_2:
            case LH264_Level_5:
            case LH264_Level_5_1:
            case LH264_Level_5_2:
            default:
                poEncMod->oSeqData.uiCPBSizeInBits      = 75000000;
                break;
        }
        poEncMod->oSeqData.uiCPBMaxBitRate              = 0;
        poEncMod->oSeqData.uiBigPictureMaxSize          = 0;

        // LH264_StreamInfoEncoderSequenceDataFlag_PICTURECROPPEDRECTANGLEPRESENT
        poEncMod->oSeqData.oPictureCroppedRect.iLeft    = 0;
        poEncMod->oSeqData.oPictureCroppedRect.iTop     = 0;
        poEncMod->oSeqData.oPictureCroppedRect.iRight   = 0;
        poEncMod->oSeqData.oPictureCroppedRect.iBottom  = 0;

        //----------------------------------------------------------------------------------------------------

        memset(&(poEncMod->oSubSeqData), 0, sizeof(poEncMod->oSubSeqData));

        poEncMod->oSubSeqData.eType                     = LH264_StreamInfoTypeHeader_ENCODERSUBSEQUENCEDATA;
        poEncMod->oSubSeqData.flSubSequenceDataFlags    = 0
            | LH264_StreamInfoEncoderSubSequenceDataFlag_REFERENCEPRESENT;

        // LH264_StreamInfoEncoderSubSequenceDataFlag_QPPARAMETERPRESENT
        poEncMod->oSubSeqData.iQPChroma1Offset          = 0;
        poEncMod->oSubSeqData.iQPChroma2Offset          = 0;

        // LH264_StreamInfoEncoderSubSequenceDataFlag_REFERENCEPRESENT
        poEncMod->oSubSeqData.uiNbRefForP               = 2;
        poEncMod->oSubSeqData.eRefMode                  = LH264_ReferenceMode_AUTOMATIC;
        poEncMod->oSubSeqData.eFieldMode                = LH264_FieldReferenceMode_SAMEPARITY;
        poEncMod->oSubSeqData.uiIdrPeriod               = 1;
        poEncMod->oSubSeqData.uiIPeriod                 = (LH264_Profile_CAVLC444INTRA == poEncMod->eProfile) ? 1 : 90;
        poEncMod->oSubSeqData.uiPPeriod                 = 1;
    }
    else
    {
        EncMod_Cleanup(poEncMod);
    }

    ++g_uiEncModCount;

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       EncMod_Cleanup

Description:    .

\************************************************************************************************************/
void EncMod_Cleanup(EncoderModule* poEncMod)
{
    MsgLog(2, "{...");

    if (poEncMod != MNULL)
    {
        // Release all internal buffers.
        if (poEncMod->hEncoder != MNULL)
        {
            MUINT i;

            for (i = 0; i < poEncMod->oOutLink.uiBufferCount; i++)
            {
                BufferInfo* poBufferInfo = &(poEncMod->oOutLink.aoBufferInfo[i]);

                if ((poBufferInfo->hBuffer != MNULL)
                    && (poBufferInfo->pvPrivateData != MNULL))
                {
                    LStatus eStatus;
                    MBOOL bWaiting = MTRUE;
                    while (bWaiting)
                    {
                        eStatus = LH264E_ReleaseNalu(
                                poEncMod->hEncoder,
                                (LH264E_NaluInfoTypeHeader*)(poBufferInfo->pvPrivateData));
                        if (eStatus != LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL)
                        {
                            bWaiting = MFALSE;
                        }
                        else
                        {
                            usleep(100);
                        }
                    }
                    poBufferInfo->hBuffer = MNULL;
                }
            }
        }

        if (poEncMod->hBlit != MNULL)
        {
            while (LSTATUS_IS_FAIL(LBlit_Destroy(poEncMod->hBlit)))
            {
                MsgLog(0, "LBlit_Destroy failed! Retry...");
                usleep(1000*1000);
            }
        }

        if (poEncMod->hVpe != MNULL)
        {
            while (LSTATUS_IS_FAIL(LVPE_Destroy(poEncMod->hVpe)))
            {
                MsgLog(0, "LVPE_Destroy failed! Retry...");
                usleep(1000*1000);
            }
        }

        if (poEncMod->hEncoder != MNULL)
        {
            MsgLog(2, "LH264E_Destroy...");

            while (LSTATUS_IS_FAIL(LH264E_Destroy(poEncMod->hEncoder)))
            {
                MsgLog(2, "LH264E_Destroy returns error. Wait and retry...");
                usleep(10*1000);
            }

            MsgLog(2, "LH264E_Destroy done.");
        }

        ModLnk_Cleanup(&(poEncMod->oOutLink));

        if (poEncMod->hDevThreadCopy != MNULL)
        {
            MsgLog(2, "LDeviceThread_Destroy(copy)...");
            LDeviceThread_Destroy(poEncMod->hDevThreadCopy);
        }

        if (poEncMod->hDevThreadPad != MNULL)
        {
            MsgLog(2, "LDeviceThread_Destroy(pad)...");
            LDeviceThread_Destroy(poEncMod->hDevThreadPad);
        }

        if (poEncMod->hDevThreadOut != MNULL)
        {
            MsgLog(2, "LDeviceThread_Destroy(out)...");
            LDeviceThread_Destroy(poEncMod->hDevThreadOut);
        }

        if (poEncMod->hDevThreadIn != MNULL)
        {
            MsgLog(2, "LDeviceThread_Destroy(in)...");
            LDeviceThread_Destroy(poEncMod->hDevThreadIn);
        }

        if(poEncMod->pvPrivData != MNULL)
        {
            EncModPrivData* poPrivData = (EncModPrivData*) poEncMod->pvPrivData;

            if(poPrivData->hSeiBuffer != MNULL)
            {
                if(poPrivData->aoSeiMsgEntry != MNULL)
                {
                    LBuffer_EndAccess(poPrivData->hSeiBuffer);
                }

                LBuffer_Destroy(poPrivData->hSeiBuffer);
            }

            free(poEncMod->pvPrivData);
            poEncMod->pvPrivData = MNULL;
        }

        poEncMod->hEncoder          = MNULL;
        poEncMod->hVpe              = MNULL;
        poEncMod->hBlit             = MNULL;
        poEncMod->hDevThreadIn      = MNULL;
        poEncMod->hDevThreadOut     = MNULL;
        poEncMod->hDevThreadPad     = MNULL;
        poEncMod->hDevThreadCopy    = MNULL;
        poEncMod->oInLink.poModLnk  = MNULL;
    }

    MsgLog(2, "...}");
}

/************************************************************************************************************\

Function:       EncMod_SetParam

Description:    .

\************************************************************************************************************/
LStatus EncMod_SetParam(
        EncoderModule* poEncMod,
        LSIZE*  poFrameSize,
        MUINT   uiFrameRateNum,
        MUINT   uiFrameRateDen,
        MUINT   uiTargetBitRateKbps,
        MUINT   uiIPeriod,
        MUINT   uiPPeriod,
        MUINT   uiQpMin,
        MUINT   uiQpMax)
{
    if ((poFrameSize != MNULL)
        && (uiFrameRateNum > 0)
        && (uiFrameRateDen > 0)
        && (uiTargetBitRateKbps >= 50)
        && (uiTargetBitRateKbps <= ENC_MAX_BITRATE_KBPS)
        && (uiIPeriod > 0)
        && (uiPPeriod > 0)
        && (uiPPeriod <= 8)
        && (uiQpMin >= 0)
        && (uiQpMin <= 51)
        && (uiQpMax >= 0)
        && (uiQpMax <= 51)
        && (uiQpMax >= uiQpMin))
    {
        poEncMod->oFrameSize                        = *poFrameSize;
        poEncMod->oSeqData.oFrameRate.uiNumerator   = uiFrameRateNum;
        poEncMod->oSeqData.oFrameRate.uiDenominator = uiFrameRateDen;
        poEncMod->oSeqData.uiTargetBitRate          = uiTargetBitRateKbps * 1000;
        poEncMod->oSeqData.fCorrectionTensor        = 1.0;
        poEncMod->oSeqData.fQPMin                   = uiQpMin;
        poEncMod->oSeqData.fQPMax                   = uiQpMax;
        poEncMod->oEncodeOpt.fQP                    = (uiQpMin + uiQpMax) / 2;
        poEncMod->oSubSeqData.uiIPeriod             = (LH264_Profile_CAVLC444INTRA == poEncMod->eProfile) ? 1 : uiIPeriod;
        poEncMod->oSubSeqData.uiPPeriod             = (LH264_Profile_CAVLC444INTRA == poEncMod->eProfile) ? 1 : uiPPeriod;

        return LStatus_OK;
    }
    else
    {
        return LStatus_INVALID_PARAM;
    }
}

/************************************************************************************************************\

Function:       EncMod_DrawPadding

Description:    Duplicate the last column and the last line into the buffer padding.

\************************************************************************************************************/
LStatus EncMod_DrawPadding(
    EncoderModule* poEncMod,
    LBuffer_Handle hBuffer)
{
    const MUINT uiMinSrcWidth  = 2;
    const MUINT uiMinSrcHeight = 1;

    LStatus  eStatus = LStatus_OK;
    LVP_Rect oSrcRect;
    LVP_Rect oDstRect;

    oSrcRect.bEnable = MTRUE;
    oDstRect.bEnable = MTRUE;

    // Do column padding

    if ((poEncMod->oFramePadSize.iWidth > 0) || (g_uiPadOffset > 0))
    {
        oSrcRect.oRect.iLeft    = poEncMod->oFrameSize.iWidth - uiMinSrcWidth - g_uiPadOffset;
        oSrcRect.oRect.iTop     = 0;
        oSrcRect.oRect.iRight   = poEncMod->oFrameSize.iWidth - g_uiPadOffset;
        oSrcRect.oRect.iBottom  = poEncMod->oFrameSize.iHeight;

        oDstRect.oRect.iLeft    = poEncMod->oFrameSize.iWidth - g_uiPadOffset;
        oDstRect.oRect.iTop     = oSrcRect.oRect.iTop;
        oDstRect.oRect.iRight   = poEncMod->oFrameSize.iWidth + poEncMod->oFramePadSize.iWidth;
        oDstRect.oRect.iBottom  = oSrcRect.oRect.iBottom;

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LVPE_SetSrcRect(poEncMod->hVpe, &oSrcRect);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LVPE_SetDstRect(poEncMod->hVpe, &oDstRect);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LVPE_Execute(poEncMod->hVpe, hBuffer, hBuffer);
        }
    }

    // Do line padding

    if ((poEncMod->oFramePadSize.iHeight > 0) || (g_uiPadOffset > 0))
    {
        oSrcRect.oRect.iLeft    = 0;
        oSrcRect.oRect.iTop     = poEncMod->oFrameSize.iHeight - uiMinSrcHeight - g_uiPadOffset;
        oSrcRect.oRect.iRight   = poEncMod->oFrameSize.iWidth + poEncMod->oFramePadSize.iWidth;
        oSrcRect.oRect.iBottom  = poEncMod->oFrameSize.iHeight - g_uiPadOffset;

        oDstRect.oRect.iLeft    = oSrcRect.oRect.iLeft;
        oDstRect.oRect.iTop     = poEncMod->oFrameSize.iHeight - g_uiPadOffset;
        oDstRect.oRect.iRight   = oSrcRect.oRect.iRight;
        oDstRect.oRect.iBottom  = poEncMod->oFrameSize.iHeight + poEncMod->oFramePadSize.iHeight;

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LVPE_SetSrcRect(poEncMod->hVpe, &oSrcRect);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LVPE_SetDstRect(poEncMod->hVpe, &oDstRect);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LVPE_Execute(poEncMod->hVpe, hBuffer, hBuffer);
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        if (!ENC_ENABLE_SYNC_CPU_PATCH)
        {
            eStatus = LDeviceThread_SyncDt(
                        poEncMod->hDevThreadIn,
                        poEncMod->hDevThreadPad,
                        &poEncMod->oCpuThreadIn.bKillThread);
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       EncMod_CpuThreadIn

Description:    .

\************************************************************************************************************/
LStatus EncMod_CpuThreadIn(void* pvData)
{
    if (pvData == MNULL)
    {
        MsgLogErr("ERROR! NULL data.");
        return LStatus_INVALID_PARAM;
    }

    EncoderModule*  poEncMod    = (EncoderModule*)pvData;
    LStatus         eStatus     = LStatus_OK;
    EncModPrivData* poPrivData  = (EncModPrivData*) poEncMod->pvPrivData;

    MCHAR8  szThreadName[16] = "";
    snprintf(szThreadName, sizeof(szThreadName), "%.12s-%s", poEncMod->szModuleName, "In");

    ModThread_SetName(szThreadName);
    MsgLog(2, "Start thread %p.", pthread_self());

    if (g_uiMsgLogLevel > 0)
    {
        printf("LH264E_EncodeOpt\n"
               "{\n"
               "    .eType                  = %d\n"
               "    .hPicture               = %p\n"
               "    .uiPictureId            = %lu\n"
               "    .ePictureType           = %d\n"
               "    .fQP                    = %f\n"
               "    .uiSliceIndex           = %u\n"
               "    .flPictureFlags         = 0x%08X\n"
               "    .ePictureStruct         = %d\n"
               "    .auiPictureTimestamp[0] = %lu\n"
               "    .auiPictureTimestamp[1] = %lu\n"
               "    .auiPictureTimestamp[2] = %lu\n"
               "    .uiRecoveryFrameCnt     = %u\n"
               "    .bExactMatchFlag        = %u\n"
               "    .bBrokenLinkFlag        = %u\n"
               "    .hSEIRbspBuffer         = %p\n"
               "    .uiSEIRbspStartOffset   = %u\n"
               "    .uiSEIRbspSize          = %u\n"
               "}\n",
               poEncMod->oEncodeOpt.eType,
               (void*)poEncMod->oEncodeOpt.hPicture,
               poEncMod->oEncodeOpt.uiPictureId,
               poEncMod->oEncodeOpt.ePictureType,
               poEncMod->oEncodeOpt.fQP,
               poEncMod->oEncodeOpt.uiSliceIndex,
               poEncMod->oEncodeOpt.flPictureFlags,
               poEncMod->oEncodeOpt.ePictureStruct,
               poEncMod->oEncodeOpt.auiPictureTimestamp[0],
               poEncMod->oEncodeOpt.auiPictureTimestamp[1],
               poEncMod->oEncodeOpt.auiPictureTimestamp[2],
               poEncMod->oEncodeOpt.uiRecoveryFrameCnt,
               poEncMod->oEncodeOpt.bExactMatchFlag,
               poEncMod->oEncodeOpt.bBrokenLinkFlag,
               (void*)poEncMod->oEncodeOpt.hSEIRbspBuffer,
               poEncMod->oEncodeOpt.uiSEIRbspStartOffset,
               poEncMod->oEncodeOpt.uiSEIRbspSize);
    }

    const MBOOL bDrawPadding = (g_bDoPadding && ((poEncMod->oFramePadSize.iWidth > 0)
                                                 || (poEncMod->oFramePadSize.iHeight > 0)
                                                 || (g_uiPadOffset > 0))) ? MTRUE : MFALSE;

    while (!poEncMod->oCpuThreadIn.bKillThread)
    {
        BufferInfo* poSrcBuffer;

        eStatus = ModLnkIn_GetSubmittedBuffer(
                    &(poEncMod->oInLink),
                    100,
                    0,
                    (bDrawPadding || (ENC_ENABLE_SYNC_CPU_PATCH && !ENABLE_AUTO_SYNC))
                    ? poEncMod->hDevThreadPad
                    : poEncMod->hDevThreadIn,
                    &poSrcBuffer,
                    MNULL,
                    MNULL);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            if (poSrcBuffer->bEndOfStream)
            {
                MsgLog(4, "END-OF-STREAM");

                poEncMod->oCpuThreadIn.bKillThread = MTRUE;
                eStatus = LStatus_END_OF_STREAM;
            }
            else
            {
                eStatus = LBuffer_SetPrivateData(poSrcBuffer->hBuffer, 0, (MUINT64)poSrcBuffer);

                if (bDrawPadding && LSTATUS_IS_SUCCESS(eStatus))
                {
                    eStatus = EncMod_DrawPadding(poEncMod, poSrcBuffer->hBuffer);
                }

                if (ENC_ENABLE_SYNC_CPU_PATCH && LSTATUS_IS_SUCCESS(eStatus))
                {
                    if (ENABLE_AUTO_SYNC)
                    {
                        LBuffer_ExecuteAutoSync(poSrcBuffer->hBuffer, MNULL, MTRUE);
                    }
                    else
                    {
                        MUINT64 uiTag;

                        eStatus = LDeviceThread_GetSubmissionTag(poEncMod->hDevThreadPad, &uiTag);

                        if (LSTATUS_IS_SUCCESS(eStatus))
                        {
                            while (!poEncMod->oCpuThreadIn.bKillThread)
                            {
                                eStatus = LDeviceThread_WaitCompletionTag(
                                                poEncMod->hDevThreadPad,
                                                LDEVICETHREAD_DEFAULT_LUID,
                                                uiTag,
                                                2000);

                                if (eStatus != LStatus_TIMEOUT)
                                {
                                    break;
                                }

                                MsgLog(0, "LDeviceThread_WaitCompletionTag timeout. Continue...");
                            }
                        }
                    }
                }

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    poEncMod->oEncodeOpt.hPicture = (MUINT64)(poSrcBuffer->hBuffer);

                    // TODO: Enable picture timing when it works.
//                    poEncMod->oEncodeOpt.flPictureFlags |= LH264E_PictureFlag_PICTURETIMINGPRESENT;
//                    poEncMod->oEncodeOpt.uiPictureTimestamp[0] = poSrcBuffer->uiTimestampUsec;

                    poEncMod->oEncodeOpt.uiPictureId = poSrcBuffer->uiSyncPtsUsec;

                    poEncMod->oEncodeOpt.flPictureFlags &= ~(LH264E_PictureFlag_SEIRBSPPRESENT);

                    // Find free SEI entry.
                    MUINT i;
                    for (i = 0; i < NB_SEI_ENTRY; i++)
                    {
                        if(poPrivData->abSeiMsgEntryFree[poPrivData->uiNextSeiMsgEntry])
                        {
                            break;
                        }

                        poPrivData->uiNextSeiMsgEntry++;
                        poPrivData->uiNextSeiMsgEntry %= NB_SEI_ENTRY;
                    }

                    // Store synchronization timestamp inside SEI.
                    if(poPrivData->abSeiMsgEntryFree[poPrivData->uiNextSeiMsgEntry])
                    {
                        eStatus = LBuffer_SetPrivateData(poSrcBuffer->hBuffer, 1, (MUINT64)poPrivData->uiNextSeiMsgEntry);

                        if (LSTATUS_IS_SUCCESS(eStatus))
                        {
                            SeiTsUserMsg* poSeiMsg = &poPrivData->aoSeiMsgEntry[poPrivData->uiNextSeiMsgEntry];

                            MUINT64 uiTsUsec = poSrcBuffer->uiTimestampUsec;
                            poSeiMsg->oPayload.m_uiTsUsec0 = (MUINT16)((uiTsUsec >>  0) & 0xFFFF);
                            poSeiMsg->oPayload.m_uiTsUsec1 = (MUINT16)((uiTsUsec >> 16) & 0xFFFF);
                            poSeiMsg->oPayload.m_uiTsUsec2 = (MUINT16)((uiTsUsec >> 32) & 0xFFFF);
                            poSeiMsg->oPayload.m_uiTsUsec3 = (MUINT16)((uiTsUsec >> 48) & 0xFFFF);

                            poPrivData->abSeiMsgEntryFree[poPrivData->uiNextSeiMsgEntry] = MFALSE;

                            poEncMod->oEncodeOpt.flPictureFlags      |= LH264E_PictureFlag_SEIRBSPPRESENT;
                            poEncMod->oEncodeOpt.hSEIRbspBuffer       = (MUINT64)poPrivData->hSeiBuffer;
                            poEncMod->oEncodeOpt.uiSEIRbspStartOffset = poPrivData->uiNextSeiMsgEntry * sizeof(SeiTsUserMsg);
                            poEncMod->oEncodeOpt.uiSEIRbspSize        = sizeof(SeiTsUserMsg);
                        }
                    }
                    else
                    {
                        eStatus = LStatus_FAIL;
                        MsgLogErr("ERROR! Cannot find free SEI entry.");
                    }

                    for (poEncMod->oEncodeOpt.uiSliceIndex = 0;
                         poEncMod->oEncodeOpt.uiSliceIndex < poEncMod->uiNbSlices;
                         poEncMod->oEncodeOpt.uiSliceIndex++)
                    {
                        while (MTRUE)
                        {
                            MsgLog(4, "LH264E_Encode(SrcBuffer[%u], SliceIndex= %u)...",
                                      poSrcBuffer->uiId, poEncMod->oEncodeOpt.uiSliceIndex);

                            eStatus = LH264E_Encode(
                                        poEncMod->hEncoder,
                                        (LH264E_EncodeOptTypeHeader*)&(poEncMod->oEncodeOpt));

                            MsgLog(
                                4,
                                "LH264E_Encode done. (status= %d - %s)",
                                eStatus,
                                GetStatusStr(eStatus));

                            if (LSTATUS_IS_SUCCESS(eStatus))
                            {
                                // Only send SEI on the first slice.
                                poEncMod->oEncodeOpt.flPictureFlags &= ~LH264E_PictureFlag_SEIRBSPPRESENT;
                                poEncMod->bEncoding = MTRUE;
                                break;
                            }
                            else if (eStatus == LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL)
                            {
                                MsgLog(3, "LH264E_Encode(Buffer[%u]) returned DEVICETHREAD_COMMAND_QUEUE_FULL. "
                                          "Take a nap and retry...", poSrcBuffer->uiId);
                                usleep(1000);
                            }
                            else
                            {
                                MsgLogErr("ERROR! LH264E_Encode returned %d (%s). Stop encoding...",
                                          eStatus, GetStatusStr(eStatus));
                                poEncMod->oCpuThreadIn.bKillThread = MTRUE;
                                break;
                            }
                        }

                        // Encode each slice separately in low latency only
                        if (1)
                        {
                            break;
                        }
                    }
                }
            }

            if (LSTATUS_IS_FAIL(eStatus))
            {
                ModLnkIn_ReturnBuffer(&(poEncMod->oInLink), poSrcBuffer, MNULL, NO_TAG);
            }
        }

        if (LSTATUS_IS_FAIL(eStatus)
            && (eStatus != LStatus_TIMEOUT))
        {
            usleep(1000);
        }
    }

    if (!g_bWildStop)
    {
        if (poEncMod->bEncoding)
        {
            MsgLog(4, "LH264E_EndStream...");

            while (MTRUE)
            {
                eStatus = LH264E_EndStream(poEncMod->hEncoder);

                if (eStatus != LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL)
                {
                    break;
                }

                usleep(1000);
            }

            MsgLog(4, "LH264E_EndStream done. (status= %d - %s)", eStatus, GetStatusStr(eStatus));
        }
    }

    MsgLog(2, "Kill thread.");

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       EncMod_CpuThreadOut

Description:    .

\************************************************************************************************************/
LStatus EncMod_CpuThreadOut(void* pvData)
{
    if (pvData == MNULL)
    {
        MsgLogErr("ERROR! NULL data.");
        return LStatus_INVALID_PARAM;
    }

    EncoderModule* poEncMod = (EncoderModule*)pvData;
    EncModPrivData* poPrivData = (EncModPrivData*)poEncMod->pvPrivData;

    MUINT64 uiFirstTime     = 0;
    MUINT64 uiLastNaluTime  = 0;

    poEncMod->uiNaluCount       = 0;
    poEncMod->uiPictureCount    = 0;
    poEncMod->uiElapsedTimeUsec = 0;
    poEncMod->uiEncodedBytes    = 0;

    MCHAR8  szThreadName[16] = "";
    snprintf(szThreadName, sizeof(szThreadName), "%.11s-%s", poEncMod->szModuleName, "Out");

    ModThread_SetName(szThreadName);
    MsgLog(2, "Start thread %p.", pthread_self());

    while (!poEncMod->oCpuThreadOut.bKillThread)
    {
        BufferInfo* poDstBuffer;

        if (!poEncMod->bEncoding)
        {
            usleep(10*1000);
            continue;
        }

        LStatus eStatus = ModLnk_GetReturnedBuffer(
                            &(poEncMod->oOutLink),
                            100,
                            poEncMod->hDevThreadOut,
                            &poDstBuffer);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            LH264E_NaluInfo             oNaluInfo;
            LH264E_NaluInfoTypeHeader*  poNaluInfoHdr = (LH264E_NaluInfoTypeHeader*)&oNaluInfo;

            *poNaluInfoHdr = LH264E_NaluInfoTypeHeader_STANDARD;

            MsgLog(4, "LH264E_GetNextNalu...");

            eStatus = LH264E_GetNextNalu(poEncMod->hEncoder, poNaluInfoHdr, 100);

            MUINT64 uiCurTime = GetMonoTimeUsec();

            if (uiFirstTime == 0)
            {
                uiFirstTime = uiCurTime;
            }

            uiCurTime -= uiFirstTime;
            poEncMod->uiElapsedTimeUsec = uiCurTime;

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                MsgLog(4, "LH264E_GetNextNalu succeed: PicRel = %p, Size = %u, Offset = %u",
                          oNaluInfo.hPictureReleased, oNaluInfo.uiSize, oNaluInfo.uiStartOffset);

                if (oNaluInfo.hPictureReleased != 0)
                {
                    BufferInfo* poSrcBuffer;

                    if (poPrivData->hSeiBuffer != MNULL)
                    {
                        MUINT64 uiSeiMsgEntry = 0;

                        if (LSTATUS_IS_SUCCESS(LBuffer_GetPrivateData(
                                                (LBuffer_Handle)(oNaluInfo.hPictureReleased),
                                                1,
                                                &uiSeiMsgEntry)))
                        {
                            if (uiSeiMsgEntry < NB_SEI_ENTRY)
                            {
                                poPrivData->abSeiMsgEntryFree[uiSeiMsgEntry] = MTRUE;
                            }
                            else
                            {
                                MsgLogErr("ERROR! Invalid SEI entry index. (%lu)", uiSeiMsgEntry);
                            }
                        }
                        else
                        {
                            MsgLogErr("ERROR! The released buffer has no private data[1].");
                        }
                    }

                    if (LSTATUS_IS_SUCCESS(LBuffer_GetPrivateData(
                                                (LBuffer_Handle)(oNaluInfo.hPictureReleased),
                                                0,
                                                (MUINT64*)(&poSrcBuffer))))
                    {
                        MsgLog(4, "Released picture: Buffer[%u])", poSrcBuffer->uiId);

                        ModLnkIn_ReturnBuffer(&(poEncMod->oInLink), poSrcBuffer, MNULL, NO_TAG);
                    }
                    else
                    {
                        MsgLogErr("ERROR! The released buffer has no private data.");
                    }
                }

                if (oNaluInfo.uiSize > 0)
                {
                    MsgLog(4, "LBlit_Copy(DstBuffer[%u]: Offset= %u, Size= %u)...",
                              poDstBuffer->uiId, oNaluInfo.uiStartOffset, oNaluInfo.uiSize);

                    // Align on dword for better performance.
                    MUINT uiLeftPad = oNaluInfo.uiStartOffset & 3;

                    eStatus = LBlit_Copy(
                                poEncMod->hBlit,
                                (LBuffer_Handle)oNaluInfo.hBuffer,
                                poDstBuffer->hBuffer,
                                oNaluInfo.uiStartOffset - uiLeftPad,
                                0,
                                (oNaluInfo.uiSize + uiLeftPad + 3) & ~3);

                    if (LSTATUS_IS_SUCCESS(eStatus))
                    {
                        poDstBuffer->uiStartOffset  = uiLeftPad;
                        poDstBuffer->uiSizeBytes    = oNaluInfo.uiSize;

                        MsgLog(4, "LBlit_Copy done. Dst: Offset= %u, Size= %u (status= %d)",
                                  poDstBuffer->uiStartOffset, poDstBuffer->uiSizeBytes, eStatus);

                        poEncMod->uiNaluCount++;
                        poEncMod->uiPictureCount  = oNaluInfo.uiAUCnt;
                        poEncMod->uiEncodedBytes += oNaluInfo.uiSize;

                        // Get timestamp from source picture (stored in picture id).
                        poDstBuffer->uiSyncPtsUsec   = oNaluInfo.uiPictureId;
                        poDstBuffer->uiTimestampUsec = oNaluInfo.uiPictureId;
                        poDstBuffer->eNaluType = (LH264_NaluType) oNaluInfo.eNaluType;

                        MsgLog(4, "***** Got NALU #%u in %u usec (avg= %u usec) *****",
                                  poEncMod->uiNaluCount,
                                  (MUINT)(poEncMod->uiElapsedTimeUsec - uiLastNaluTime),
                                  (MUINT)(poEncMod->uiElapsedTimeUsec / poEncMod->uiNaluCount));

                        uiLastNaluTime = poEncMod->uiElapsedTimeUsec;

                        LDeviceThread_SyncDt(
                            poEncMod->hDevThreadOut,
                            poEncMod->hDevThreadCopy,
                            &poEncMod->oCpuThreadOut.bKillThread);
                    }
                    else
                    {
                        MsgLogErr(
                            "ERROR! LBlit_Copy returned status= %d (%s).",
                            eStatus,
                            GetStatusStr(eStatus));
                    }

                    MsgLog(4, "LH264E_ReleaseNalu(Buffer[%u])...", poDstBuffer->uiId);

                    while ((eStatus = LH264E_ReleaseNalu(poEncMod->hEncoder, poNaluInfoHdr))
                           == LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL)
                    {
                        usleep(10);
                    }

                    MsgLog(4, "LH264E_ReleaseNalu(Buffer[%u]) done. (status= %d)",
                              poDstBuffer->uiId, eStatus);
                }
                else
                {
                    eStatus = LStatus_TIMEOUT;
                }
            }
            else if (eStatus == LStatus_END_OF_STREAM)
            {
                MsgLog(2, "***** END-OF-STREAM (%u NALUs in %u msec) *****",
                          poEncMod->uiNaluCount, (MUINT)(poEncMod->uiElapsedTimeUsec / 1000));

                poDstBuffer->bEndOfStream   = MTRUE;
                poEncMod->bEncoding         = MFALSE;
                poEncMod->oCpuThreadOut.bKillThread = MTRUE;

                eStatus = LStatus_OK;
            }
            else
            {
                MsgLog(4, "LH264E_GetNextNalu return status= %d (%s).", eStatus, GetStatusStr(eStatus));
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                ModLnk_SubmitBuffer(&(poEncMod->oOutLink), poDstBuffer, poEncMod->hDevThreadCopy, NO_TAG);
            }
            else
            {
                ModLnk_ReleaseBuffer(&(poEncMod->oOutLink), poDstBuffer);
            }
        }

        if (LSTATUS_IS_FAIL(eStatus)
            && (eStatus != LStatus_TIMEOUT))
        {
            usleep(1000);
        }
    }

    MsgLog(2, "Kill thread.");

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       EncMod_Print4x4

Description:    .

\************************************************************************************************************/
void EncMod_Print4x4(
    const char*   szTab,
    const char*   szFieldName,
    const MUINT8* auiData)
{
    MUINT i;

    printf("%s%s\n%s{\n", szTab, szFieldName, szTab);

    for (i = 0; i < 4; i++)
    {
        printf("%s  %3u, %3u, %3u, %3u,\n", szTab, auiData[0], auiData[1], auiData[2], auiData[3]);
        auiData += 4;
    }

    printf("%s}\n", szTab);
}

/************************************************************************************************************\

Function:       EncMod_Print8x8

Description:    .

\************************************************************************************************************/
void EncMod_Print8x8(
    const char*   szTab,
    const char*   szFieldName,
    const MUINT8* auiData)
{
    MUINT i;

    printf("%s%s\n%s{\n", szTab, szFieldName, szTab);

    for (i = 0; i < 8; i++)
    {
        printf("%s  %3u, %3u, %3u, %3u, %3u, %3u, %3u, %3u,\n", szTab, auiData[0], auiData[1],
               auiData[2], auiData[3], auiData[4], auiData[5], auiData[6], auiData[7]);
        auiData += 8;
    }

    printf("%s}\n", szTab);
}

/************************************************************************************************************\

Function:       EncMod_Start

Description:    .

\************************************************************************************************************/
LStatus EncMod_Start(EncoderModule* poEncMod)
{
    LStatus eStatus = LStatus_INVALID_PARAM;

    MsgLog(2, "{...");

    if ((poEncMod != MNULL)
        && (poEncMod->hEncoder != MNULL))
    {
        if ((poEncMod->oInLink.poModLnk != MNULL)
            && (poEncMod->oOutLink.uiSubmitCount > 0))
        {
            eStatus = LStatus_OK;
        }
        else
        {
            MsgLogErr("ERROR! Bad connection.");
            eStatus = LStatus_FAIL;
        }

        LBuffer_VideoAttributes oInVideoAttrib;
        LSIZE oCropUnit = {1, 1};

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LStatus_FAIL;

            if (poEncMod->oInLink.poModLnk->uiBufferCount > 0)
            {
                LBuffer_Handle hBuffer = poEncMod->oInLink.poModLnk->aoBufferInfo[0].hBuffer;

                oInVideoAttrib.eAttributeType = LBuffer_GetType(hBuffer);

                if (oInVideoAttrib.eAttributeType == LBuffer_Type_VIDEO)
                {
                    eStatus = LBuffer_GetAttributes(hBuffer, &(oInVideoAttrib.eAttributeType));

                    if (LSTATUS_IS_SUCCESS(eStatus))
                    {
                        switch (oInVideoAttrib.ePixelFormat)
                        {
                            case LPixelFormat_Y8:
                            case LPixelFormat_Y82Y82Y82X2:
                                oCropUnit.iWidth    = 1;
                                oCropUnit.iHeight   = 1;
                                break;

                            case LPixelFormat_MP_Y8_U8V8_420:
                            case LPixelFormat_MP_Y82_U82V82_X2_420:
                                oCropUnit.iWidth    = 2;
                                oCropUnit.iHeight   = 2;
                                break;

                            case LPixelFormat_MP_Y8_U8V8_422:
                            case LPixelFormat_MP_Y82_U82V82_X2_422:
                                oCropUnit.iWidth    = 2;
                                oCropUnit.iHeight   = 1;
                                break;

                            case LPixelFormat_MP_Y8_U8_V8_444:
                            case LPixelFormat_MP_Y82_U82_V82_X2_444:
                                oCropUnit.iWidth    = 1;
                                oCropUnit.iHeight   = 1;
                                break;

                            default:
                                eStatus = LStatus_INVALID_PARAM;
                                break;
                        }
                    }
                }
            }

            if (LSTATUS_IS_FAIL(eStatus))
            {
                MsgLogErr(
                    "ERROR! Something wrong with input buffers (status= %d, %s)",
                    eStatus,
                    GetStatusStr(eStatus));
            }
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            poEncMod->oSeqData.uiPictureWidth   = AlignPow2(poEncMod->oFrameSize.iWidth, ENC_WIDTH_ALIGN);
            poEncMod->oSeqData.uiPictureHeight  = AlignPow2(poEncMod->oFrameSize.iHeight, ENC_HEIGHT_ALIGN);
            poEncMod->oSeqData.ePixelFormat     = oInVideoAttrib.ePixelFormat;
            poEncMod->oFramePadSize.iWidth      = poEncMod->oSeqData.uiPictureWidth
                                                  - poEncMod->oFrameSize.iWidth;
            poEncMod->oFramePadSize.iHeight     = poEncMod->oSeqData.uiPictureHeight
                                                  - poEncMod->oFrameSize.iHeight;

            if ((poEncMod->oSeqData.uiPictureWidth > oInVideoAttrib.uiWidth)
                || (poEncMod->oSeqData.uiPictureHeight > oInVideoAttrib.uiHeight))
            {
                MsgLogErr("ERROR! Buffer dimensions (%ux%u) smaller than picture dimensions (%ux%u).",
                          oInVideoAttrib.uiWidth, oInVideoAttrib.uiHeight,
                          poEncMod->oSeqData.uiPictureWidth, poEncMod->oSeqData.uiPictureHeight);
                eStatus = LStatus_FAIL;
            }
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            // Set cropping rectangle if necessary.
            poEncMod->oSeqData.oPictureCroppedRect.iLeft    = 0;
            poEncMod->oSeqData.oPictureCroppedRect.iTop     = 0;
            poEncMod->oSeqData.oPictureCroppedRect.iRight
                = (poEncMod->oSeqData.uiPictureWidth - poEncMod->oFrameSize.iWidth) / oCropUnit.iWidth;
            poEncMod->oSeqData.oPictureCroppedRect.iBottom
                = (poEncMod->oSeqData.uiPictureHeight - poEncMod->oFrameSize.iHeight) / oCropUnit.iHeight;

            if ((poEncMod->oSeqData.oPictureCroppedRect.iRight != 0)
                || (poEncMod->oSeqData.oPictureCroppedRect.iBottom != 0))
            {
                poEncMod->oSeqData.flSequenceDataFlags
                    |= LH264_StreamInfoEncoderSequenceDataFlag_PICTURECROPPEDRECTANGLEPRESENT;
            }
            else
            {
                poEncMod->oSeqData.flSequenceDataFlags
                    &= ~LH264_StreamInfoEncoderSequenceDataFlag_PICTURECROPPEDRECTANGLEPRESENT;
            }

            MsgLog(2, "LH264E_UpdateStreamInfo(%ux%u, PixFmt= %08X)...", poEncMod->oSeqData.uiPictureWidth,
                      poEncMod->oSeqData.uiPictureHeight, poEncMod->oSeqData.ePixelFormat);

            eStatus = LH264E_UpdateStreamInfo(
                        poEncMod->hEncoder,
                        (LH264_StreamInfoTypeHeader*)&(poEncMod->oSeqData));

            if (LSTATUS_IS_SUCCESS(eStatus) && (g_uiMsgLogLevel > 0))
            {
                printf("LH264_StreamInfoEncoderSequenceData\n"
                       "{\n"
                       "    .eType                          = %d\n"
                       "    .flSequenceDataFlags            = 0x%08X\n"
                       "    .uiPictureWidth                 = %u\n"
                       "    .uiPictureHeight                = %u\n"
                       "    .ePixelFormat                   = 0x%08X\n"
                       "    .eScanMode                      = %d\n"
                       "    .eFirstField                    = %d\n"
                       "    .oFrameRate.uiNumerator         = %u\n"
                       "    .oFrameRate.uiDenominator       = %u\n"
                       "    .eDFDisableIdc                  = %u\n"
                       "    .iDFAlphaC0Param                = %d\n"
                       "    .iDFBetaParam                   = %d\n"
                       "    .flRateControlFlags             = 0x%08X\n"
                       "    .uiTargetBitRate                = %u\n"
                       "    .fCorrectionTensor              = %f\n"
                       "    .fQPMin                         = %f\n"
                       "    .fQPMax                         = %f\n"
                       "    .eHRDMode                       = %d\n"
                       "    .uiCPBSizeInBits                = %u\n"
                       "    .uiCPBMaxBitRate                = %u\n"
                       "    .uiBigPictureMaxSize            = %u\n"
                       "    .oPictureCroppedRect.iLeft      = %d\n"
                       "    .oPictureCroppedRect.iTop       = %d\n"
                       "    .oPictureCroppedRect.iRight     = %d\n"
                       "    .oPictureCroppedRect.iBottom    = %d\n"
                       "}\n",
                       poEncMod->oSeqData.eType,
                       poEncMod->oSeqData.flSequenceDataFlags,
                       poEncMod->oSeqData.uiPictureWidth,
                       poEncMod->oSeqData.uiPictureHeight,
                       poEncMod->oSeqData.ePixelFormat,
                       poEncMod->oSeqData.eScanMode,
                       poEncMod->oSeqData.eFirstField,
                       poEncMod->oSeqData.oFrameRate.uiNumerator,
                       poEncMod->oSeqData.oFrameRate.uiDenominator,
                       poEncMod->oSeqData.eDFDisableIdc,
                       poEncMod->oSeqData.iDFAlphaC0Param,
                       poEncMod->oSeqData.iDFBetaParam,
                       poEncMod->oSeqData.flRateControlFlags,
                       poEncMod->oSeqData.uiTargetBitRate,
                       poEncMod->oSeqData.fCorrectionTensor,
                       poEncMod->oSeqData.fQPMin,
                       poEncMod->oSeqData.fQPMax,
                       poEncMod->oSeqData.eHRDMode,
                       poEncMod->oSeqData.uiCPBSizeInBits,
                       poEncMod->oSeqData.uiCPBMaxBitRate,
                       poEncMod->oSeqData.uiBigPictureMaxSize,
                       poEncMod->oSeqData.oPictureCroppedRect.iLeft,
                       poEncMod->oSeqData.oPictureCroppedRect.iTop,
                       poEncMod->oSeqData.oPictureCroppedRect.iRight,
                       poEncMod->oSeqData.oPictureCroppedRect.iBottom);
            }
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LH264E_UpdateStreamInfo(
                        poEncMod->hEncoder,
                        (LH264_StreamInfoTypeHeader*)&(poEncMod->oSubSeqData));

            if (LSTATUS_IS_SUCCESS(eStatus) && (g_uiMsgLogLevel > 0))
            {
                printf("LH264_StreamInfoEncoderSubSequenceData\n"
                       "{\n"
                       "    .eType                  = %d\n"
                       "    .flSubSequenceDataFlags = 0x%08X\n"
                       "    .iQPChroma1Offset       = %d\n"
                       "    .iQPChroma2Offset       = %d\n"
                       "    .uiNbRefForP            = %u\n"
                       "    .eRefMode               = %d\n"
                       "    .eFieldMode             = %d\n"
                       "    .uiIdrPeriod            = %u\n"
                       "    .uiIPeriod              = %u\n"
                       "    .uiPPeriod              = %u\n"
                       "}\n",
                       poEncMod->oSubSeqData.eType,
                       poEncMod->oSubSeqData.flSubSequenceDataFlags,
                       poEncMod->oSubSeqData.iQPChroma1Offset,
                       poEncMod->oSubSeqData.iQPChroma2Offset,
                       poEncMod->oSubSeqData.uiNbRefForP,
                       poEncMod->oSubSeqData.eRefMode,
                       poEncMod->oSubSeqData.eFieldMode,
                       poEncMod->oSubSeqData.uiIdrPeriod,
                       poEncMod->oSubSeqData.uiIPeriod,
                       poEncMod->oSubSeqData.uiPPeriod);
            }
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            MUINT32 uiMinNbSlices = 0;
            MUINT32 uiMaxNbSlices = 0;

            eStatus = LH264E_GetNbSliceRange(poEncMod->hEncoder, &uiMinNbSlices, &uiMaxNbSlices);

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                MsgLog(2, "H265E: Nb slices= %u, Min= %u, Max= %u",
                          poEncMod->uiNbSlices, uiMinNbSlices, uiMaxNbSlices);

                LH264_StreamInfoEncoderNbSlices oNbSlices;

                memset(&oNbSlices, 0, sizeof(oNbSlices));

                oNbSlices.eType      = LH264_StreamInfoTypeHeader_ENCODERNBSLICES;
                oNbSlices.uiNbSlices = min(max(poEncMod->uiNbSlices, uiMinNbSlices), uiMaxNbSlices);

                eStatus = LH264E_UpdateStreamInfo(
                            poEncMod->hEncoder,
                            (LH264_StreamInfoTypeHeader*)&oNbSlices);

                if (LSTATUS_IS_SUCCESS(eStatus) && (g_uiMsgLogLevel > 0))
                {
                    printf("LH264_StreamInfoEncoderNbSlices\n"
                           "{\n"
                           "    .eType      = %d\n"
                           "    .uiNbSlices = %u\n"
                           "}\n",
                           oNbSlices.eType,
                           oNbSlices.uiNbSlices);
                }
            }
        }

        if (g_bSetScalinglist && LSTATUS_IS_SUCCESS(eStatus) &&
            (LH264_Profile_BASELINE != poEncMod->eProfile) &&
            (LH264_Profile_MAIN != poEncMod->eProfile) &&
            (LH264_Profile_EXTENDED != poEncMod->eProfile))
        {
            LH264_StreamInfoScalingList oScalingList;

            memset(&oScalingList, 0, sizeof(oScalingList));

            oScalingList.eType                  = LH264_StreamInfoTypeHeader_SPSSCALINGLIST;
            oScalingList.bUseScalingList        = MTRUE;
            oScalingList.flScalingListPresent   = 0
                | LH264_StreamInfoScalingListPresentFlag_YINTRA4X4
                | LH264_StreamInfoScalingListPresentFlag_YINTER4X4
                | LH264_StreamInfoScalingListPresentFlag_YINTRA8X8
                | LH264_StreamInfoScalingListPresentFlag_YINTER8X8;
            memcpy(oScalingList.auiYIntra4x4,   g_uiScListBase4x4, sizeof(oScalingList.auiYIntra4x4));
            memcpy(oScalingList.auiYInter4x4,   g_uiScListBase4x4, sizeof(oScalingList.auiYInter4x4));
            memcpy(oScalingList.auiCbIntra4x4,  g_uiScListBase4x4, sizeof(oScalingList.auiCbIntra4x4));
            memcpy(oScalingList.auiCbInter4x4,  g_uiScListBase4x4, sizeof(oScalingList.auiCbInter4x4));
            memcpy(oScalingList.auiCrIntra4x4,  g_uiScListBase4x4, sizeof(oScalingList.auiCrIntra4x4));
            memcpy(oScalingList.auiCrInter4x4,  g_uiScListBase4x4, sizeof(oScalingList.auiCrInter4x4));
            memcpy(oScalingList.auiYIntra8x8,   g_uiScListBase8x8, sizeof(oScalingList.auiYIntra8x8));
            memcpy(oScalingList.auiYInter8x8,   g_uiScListBase8x8, sizeof(oScalingList.auiYInter8x8));
            memcpy(oScalingList.auiCbIntra8x8,  g_uiScListBase8x8, sizeof(oScalingList.auiCbIntra8x8));
            memcpy(oScalingList.auiCbInter8x8,  g_uiScListBase8x8, sizeof(oScalingList.auiCbInter8x8));
            memcpy(oScalingList.auiCrIntra8x8,  g_uiScListBase8x8, sizeof(oScalingList.auiCrIntra8x8));
            memcpy(oScalingList.auiCrInter8x8,  g_uiScListBase8x8, sizeof(oScalingList.auiCrInter8x8));

            eStatus = LH264E_UpdateStreamInfo(
                        poEncMod->hEncoder,
                        (LH264_StreamInfoTypeHeader*)&oScalingList);

            if (LSTATUS_IS_SUCCESS(eStatus) && (g_uiMsgLogLevel > 0))
            {
                printf("LH264_StreamInfoScalingList\n"
                       "{\n"
                       "    .eType                  = %d\n"
                       "    .bUseScalingList        = %u\n"
                       "    .flScalingListPresent   = 0x%08X\n",
                       oScalingList.eType,
                       oScalingList.bUseScalingList,
                       oScalingList.flScalingListPresent);

                EncMod_Print4x4("    ", ".auiYIntra4x4",    oScalingList.auiYIntra4x4);
                EncMod_Print4x4("    ", ".auiYInter4x4",    oScalingList.auiYInter4x4);
                EncMod_Print4x4("    ", ".auiCbIntra4x4",   oScalingList.auiCbIntra4x4);
                EncMod_Print4x4("    ", ".auiCbInter4x4",   oScalingList.auiCbInter4x4);
                EncMod_Print4x4("    ", ".auiCrIntra4x4",   oScalingList.auiCrIntra4x4);
                EncMod_Print4x4("    ", ".auiCrInter4x4",   oScalingList.auiCrInter4x4);
                EncMod_Print8x8("    ", ".auiYIntra8x8",    oScalingList.auiYIntra8x8);
                EncMod_Print8x8("    ", ".auiYInter8x8",    oScalingList.auiYInter8x8);
                EncMod_Print8x8("    ", ".auiCbIntra8x8",   oScalingList.auiCbIntra8x8);
                EncMod_Print8x8("    ", ".auiCbInter8x8",   oScalingList.auiCbInter8x8);
                EncMod_Print8x8("    ", ".auiCrIntra8x8",   oScalingList.auiCrIntra8x8);
                EncMod_Print8x8("    ", ".auiCrInter8x8",   oScalingList.auiCrInter8x8);

                printf("}\n");
            }
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            LH264_StreamInfoVUI oVui;

            memset(&oVui, 0, sizeof(oVui));

            oVui.eType                      = LH264_StreamInfoTypeHeader_VUI;
            oVui.flVUIFlags                 = 0
                | LH264_StreamInfoVUIFlag_BITSTREAMRESTRICTIONPRESENT
                | LH264_StreamInfoVUIFlag_COLOURDESCRIPTIONPRESENT
                | LH264_StreamInfoVUIFlag_VIDEOSIGNALPRESENT;
            oVui.bMotionVectorsOverPicBoundariesFlag = 1;
            oVui.uiMaxDecFrameBuffering     = (LH264_Profile_CAVLC444INTRA == poEncMod->eProfile) ? 0 : 4;
            oVui.uiNumReorderFrames         = (LH264_Profile_CAVLC444INTRA == poEncMod->eProfile) ? 0 : 4;
            oVui.uiMaxBytesPerPicDenom = 2;
            oVui.uiMaxBitsPerMBDenom = 1;
            oVui.uiLog2MaxMVLengthHorizontal = 16;
            oVui.uiLog2MaxMVLengthVertical = 16;

            // Hardcoded to BT709 for now.
            oVui.eColourPrimaries           = LH264_ColourPrimaries_MODE1;
            oVui.eTransferCharacteristics   = LH264_TransferCharacteristics_MODE1;
            oVui.eMatrixCoefficients        = LH264_MatrixCoefficients_MODE1;
            oVui.eVideoFormat               = LH264_VideoFormat_UNSPECIFIED;
            oVui.bVideoFullRange            = MFALSE;

            eStatus = LH264E_UpdateStreamInfo(
                        poEncMod->hEncoder,
                        (LH264_StreamInfoTypeHeader*)&oVui);

            if (LSTATUS_IS_SUCCESS(eStatus) && (g_uiMsgLogLevel > 0))
            {
                printf("LH264_StreamInfoVUI\n"
                       "{\n"
                       "    .eType                       = %d\n"
                       "    .flVUIFlags                  = 0x%08X\n"
                       "    .eAspectRatio                = %d\n"
                       "    .uiSARWidth                  = %u\n"
                       "    .uiSARHeight                 = %u\n"
                       "    .eVideoFormat                = %d\n"
                       "    .bVideoFullRange             = %u\n"
                       "    .eColourPrimaries            = %d\n"
                       "    .eTransferCharacteristics    = %d\n"
                       "    .eMatrixCoefficients         = %d\n"
                       "    .eChromaLocationTopField     = %d\n"
                       "    .eChromaLocationBottomField  = %d\n"
                       "    .uiNumUnitsInTick            = %u\n"
                       "    .uiTimeScale                 = %u\n"
                       "    .uiNumReorderFrames          = %u\n"
                       "    .uiMaxDecFrameBuffering      = %u\n"
                       "    .uiMaxBytesPerPicDenom       = %u\n"
                       "    .uiMaxBitsPerMBDenom         = %u\n"
                       "    .uiLog2MaxMVLengthHorizontal = %u\n"
                       "    .uiLog2MaxMVLengthVertical   = %u\n"
                       "}\n",
                       oVui.eType,
                       oVui.flVUIFlags,
                       oVui.eAspectRatio,
                       oVui.uiSARWidth,
                       oVui.uiSARHeight,
                       oVui.eVideoFormat,
                       oVui.bVideoFullRange,
                       oVui.eColourPrimaries,
                       oVui.eTransferCharacteristics,
                       oVui.eMatrixCoefficients,
                       oVui.eChromaLocationTopField,
                       oVui.eChromaLocationBottomField,
                       oVui.uiNumUnitsInTick,
                       oVui.uiTimeScale,
                       oVui.uiNumReorderFrames,
                       oVui.uiMaxDecFrameBuffering,
                       oVui.uiMaxBytesPerPicDenom,
                       oVui.uiMaxBitsPerMBDenom,
                       oVui.uiLog2MaxMVLengthHorizontal,
                       oVui.uiLog2MaxMVLengthVertical);
            }
        }

        MsgLog(2, "LH264E_UpdateStreamInfo done. (status= %d)", eStatus);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = ModThread_Start(&(poEncMod->oCpuThreadOut), poEncMod, EncMod_CpuThreadOut);

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                eStatus = ModThread_Start(&(poEncMod->oCpuThreadIn), poEncMod, EncMod_CpuThreadIn);

                if (LSTATUS_IS_FAIL(eStatus))
                {
                    ModThread_Stop(&(poEncMod->oCpuThreadOut));
                }
            }
        }
    }

    MsgLog(2, "...} (status= %d -%s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       EncMod_Stop

Description:    .

\************************************************************************************************************/
void EncMod_Stop(EncoderModule* poEncMod)
{
    MsgLog(2, "{...");

    if (poEncMod != MNULL)
    {
        ModThread_Stop(&(poEncMod->oCpuThreadIn));

        if (!g_bWildStop)
        {
            MsgLog(2, "Wait EOS...");
            while (poEncMod->bEncoding)
            {
                usleep(10*1000);
            }
            MsgLog(2, "Wait EOS done.");
        }
        else
        {
            poEncMod->bEncoding = MFALSE;
        }

        ModThread_Stop(&(poEncMod->oCpuThreadOut));

        if(poEncMod->pvPrivData != MNULL)
        {
            EncModPrivData* poPrivData = (EncModPrivData*) poEncMod->pvPrivData;

            MUINT i;
            for(i = 0; i < NB_SEI_ENTRY; i++)
            {
                poPrivData->abSeiMsgEntryFree[i] = MTRUE;
            }

            poPrivData->uiNextSeiMsgEntry = 0;
        }
    }

    MsgLog(2, "...}");
}

/************************************************************************************************************\

Function:       EncMod_GetSeiPayload

Description:    .

\************************************************************************************************************/
MBOOL EncMod_GetSeiPayload(
    MUINT8** ppuiBuffer,
    MUINT*   puiSeiRemSize,
    MUINT*   puiSeiPayloadType,
    MUINT*   puiSeiPayloadSize)
{
    MBOOL bRet = MFALSE;

    *puiSeiPayloadType = 0;
    *puiSeiPayloadSize = 0;

    while (*puiSeiRemSize > 0)
    {
        (*puiSeiRemSize)--;

        MUINT8 uiByte = *(*ppuiBuffer)++;

        *puiSeiPayloadType += uiByte;

        if (uiByte != 255)
        {
            break;
        }
    }

    while (*puiSeiRemSize > 0)
    {
        (*puiSeiRemSize)--;

        MUINT8 uiByte = *(*ppuiBuffer)++;

        *puiSeiPayloadSize += uiByte;

        if (uiByte != 255)
        {
            bRet = (*puiSeiPayloadSize <= *puiSeiRemSize);
            break;
        }
    }

    return bRet;
}

/************************************************************************************************************\

Function:       EncMod_GetSeiTsUserMsgPriv

Description:    .

\************************************************************************************************************/
MBOOL EncMod_GetSeiTsUserMsgPriv(
    MUINT8*         puiBuffer,
    MUINT           uiSeiMsgSize,
    MUINT64*        puiTsUsec)
{
    MBOOL   bRet             = MFALSE;
    MUINT   uiSeiRemSize     = uiSeiMsgSize;
    MUINT   uiSeiPayloadType = 0;
    MUINT   uiSeiPayloadSize = 0;

    while (EncMod_GetSeiPayload(&puiBuffer, &uiSeiRemSize, &uiSeiPayloadType, &uiSeiPayloadSize))
    {
        if ((uiSeiPayloadType == g_uiSeiMsgPayloadType)
            && (uiSeiPayloadSize == sizeof(SeiTsUserMsgPayload)))
        {
            SeiTsUserMsgPayload* poPayload = (SeiTsUserMsgPayload*) puiBuffer;

            if ((poPayload->m_auiUuidA[0] == g_szSeiMsgUuid[0])
                && (poPayload->m_auiUuidA[1] == g_szSeiMsgUuid[1])
                && (poPayload->m_auiUuidB[0] == g_szSeiMsgUuid[2])
                && (poPayload->m_auiUuidB[1] == g_szSeiMsgUuid[3])
                && (poPayload->m_auiUuidC[0] == g_szSeiMsgUuid[4])
                && (poPayload->m_auiUuidC[1] == g_szSeiMsgUuid[5])
                && (poPayload->m_auiUuidD[0] == g_szSeiMsgUuid[6])
                && (poPayload->m_auiUuidD[1] == g_szSeiMsgUuid[7]))
            {
                *puiTsUsec = ((MUINT64)poPayload->m_uiTsUsec0 << 0)
                            | ((MUINT64)poPayload->m_uiTsUsec1 << 16)
                            | ((MUINT64)poPayload->m_uiTsUsec2 << 32)
                            | ((MUINT64)poPayload->m_uiTsUsec3 << 48);
                bRet = MTRUE;
            }
        }
        else
        {
            // Add emulation bytes (0 0 3) in payload size.
            MUINT uiZeroCount = 0;
            MUINT i;

            for (i = 0; i < uiSeiPayloadSize; i++)
            {
                if (puiBuffer[i] == 0)
                {
                    uiZeroCount++;
                }
                else
                {
                    if ((puiBuffer[i] == 3) && (uiZeroCount >= 2))
                    {
                        if (uiSeiPayloadSize == uiSeiRemSize)
                        {
                            break;
                        }

                        uiSeiPayloadSize++;
                    }

                    uiZeroCount = 0;
                }
            }
        }

        uiSeiRemSize -= uiSeiPayloadSize;
        puiBuffer    += uiSeiPayloadSize;
    }

    return bRet;
}


/************************************************************************************************************\

Function:       EncMod_GetSeiTsUserMsg

Description:    .

\************************************************************************************************************/
MBOOL EncMod_GetSeiTsUserMsg(
    LBuffer_Handle  hSeiBuffer,
    MUINT           uiSeiBufferOffset,
    MUINT           uiSeiMsgSize,
    MUINT64*        puiTsUsec)
{
    MBOOL bRet = MFALSE;

    if (uiSeiMsgSize >= sizeof(SeiTsUserMsg))
    {
        MUINT8* puiSeiBuffer = MNULL;

        if (LSTATUS_IS_SUCCESS(LBuffer_BeginAccess(hSeiBuffer, 0, 1, &puiSeiBuffer)))
        {
            bRet = EncMod_GetSeiTsUserMsgPriv(puiSeiBuffer + uiSeiBufferOffset, uiSeiMsgSize, puiTsUsec);

            LBuffer_EndAccess(hSeiBuffer);
        }
    }

    return bRet;
}
