/************************************************************************************************************\

Module Name:    DecoderModule.c

Description:    .

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "DecoderModule.h"

extern MBOOL EncMod_GetSeiTsUserMsg(
                LBuffer_Handle hSeiBuffer,
                MUINT uiSeiBufferOffset,
                MUINT uiSeiMsgSize,
                MUINT64* puiTsUsec);

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

static const    MCHAR8      g_szModuleNameBase[]    = "Dec";
static          MUINT32     g_uiDecModCount         = 0;

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Function:       DecMod_Init

Description:    .

\************************************************************************************************************/
LStatus DecMod_Init(
            DecoderModule*  poDecMod,
            LDevice_Handle  hDevice,
            MUINT           uiOutBufferCount,
            MUINT           uiFrameRateNum,
            MUINT           uiFrameRateDen,
            MBOOL           bDecodeNalBlobs,
            MBOOL           bReduceLocalMemUsage)
{
    MsgLog(2, "{...");

    DecMod_Cleanup(poDecMod);

    LStatus eStatus = ((poDecMod != MNULL)
                       && (hDevice != MNULL)
                       && (uiFrameRateDen != 0)
                       && (uiFrameRateNum != 0))
                      ? LStatus_OK : LStatus_INVALID_PARAM;

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LDeviceThread_CreateAttributes oAttributes = {LDeviceThread_AttributeType_CREATE, MTRUE, MTRUE, MFALSE, "decin"};

        eStatus = LDeviceThread_Create(hDevice, &oAttributes.eType, &(poDecMod->hDevThreadIn));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LDeviceThread_CreateAttributes oAttributes = {LDeviceThread_AttributeType_CREATE, MTRUE, MTRUE, MFALSE, "decout"};

        eStatus = LDeviceThread_Create(hDevice, &oAttributes.eType, &(poDecMod->hDevThreadOut));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LH264D_CreateOpt oCreateOpt;

        memset(&oCreateOpt, 0, sizeof(oCreateOpt));

        oCreateOpt.eType                            = LH264D_CreateOptTypeHeader_STANDARD;
        oCreateOpt.hDTInput                         = (MUINT64)(poDecMod->hDevThreadIn);
        oCreateOpt.hDTOutput                        = (MUINT64)(poDecMod->hDevThreadOut);
        oCreateOpt.flCreateOptFlags                 = 0
            | (bReduceLocalMemUsage ? LH264D_CreateOptFlag_OPTIMIZEDSLICEDISTRIBUTION : 0);
        oCreateOpt.eDecoderMode                     = LH264D_DecoderMode_NORMAL;
        oCreateOpt.ePriority                        = LH264_Priority_REALTIME;

        // LH264D_CreateOptFlag_DYNAMICPICTURERESOLUTION
        oCreateOpt.eMaxLevel                        = LH264_Level_5_2;
        oCreateOpt.eMaxProfile                      = LH264_Profile_HIGH444;

        oCreateOpt.oDefaultFrameRate.uiNumerator    = uiFrameRateNum;
        oCreateOpt.oDefaultFrameRate.uiDenominator  = uiFrameRateDen;
        oCreateOpt.oMaxFrameRate.uiNumerator        = uiFrameRateNum;
        oCreateOpt.oMaxFrameRate.uiDenominator      = uiFrameRateDen;
        oCreateOpt.uiMaxBitRate                     = 0;

        oCreateOpt.uiOutputBufferSizeInFrames       = min(uiOutBufferCount, LH264D_MAX_OUTPUT_BUFFER);
        oCreateOpt.uiSEIRbspBufferingSize           = 4 * 1024;

        if (ENABLE_AUTO_SYNC)
        {
           oCreateOpt.flCreateOptFlags  |= LH264D_CreateOptFlag_BUFFERAUTOSYNC;
           oCreateOpt.uiAutoSyncTimeOut  = LDEVICETHREAD_NO_TIMEOUT;
        }

        eStatus = LH264D_Create(hDevice, (LH264D_CreateOptTypeHeader*)&oCreateOpt, &(poDecMod->hDecoder));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        snprintf(
                    poDecMod->szModuleName,
                    sizeof(poDecMod->szModuleName),
                    "%s%d",
                    g_szModuleNameBase,
                    g_uiDecModCount);

        eStatus = ModLnk_Init(
                        &(poDecMod->oOutLink),
                        hDevice,
                        uiOutBufferCount,
                        MNULL,
                        MTRUE,
                        sizeof(LH264D_PictureInfo),
                        poDecMod->szModuleName);
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        poDecMod->bDecoding         = MFALSE;
        poDecMod->uiFrameRateNum    = uiFrameRateNum;
        poDecMod->uiFrameRateDen    = uiFrameRateDen;
        poDecMod->bDecodeNalBlobs   = bDecodeNalBlobs;
    }
    else
    {
        DecMod_Cleanup(poDecMod);
    }

    ++g_uiDecModCount;

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       DecMod_Cleanup

Description:    .

\************************************************************************************************************/
void DecMod_Cleanup(DecoderModule* poDecMod)
{
    MsgLog(2, "{...");

    if (poDecMod != MNULL)
    {
        // Release all internal buffers.
        if (poDecMod->hDecoder != MNULL)
        {
            BufferInfo* poOldestBuffer;

            do
            {
                MUINT i;
                poOldestBuffer = MNULL;

                for (i = 0; i < poDecMod->oOutLink.uiBufferCount; i++)
                {
                    BufferInfo* poBuffer = &(poDecMod->oOutLink.aoBufferInfo[i]);

                    if ((poBuffer->hBuffer != MNULL)
                        && (poBuffer->pvPrivateData != MNULL))
                    {
                        if (poOldestBuffer != MNULL)
                        {
                            LH264D_PictureInfo* poInfo       = (LH264D_PictureInfo*) poBuffer->pvPrivateData;
                            LH264D_PictureInfo* poOldestInfo = (LH264D_PictureInfo*) poOldestBuffer->pvPrivateData;

                            if (poInfo->uiStreamCnt < poOldestInfo->uiStreamCnt)
                            {
                                poOldestBuffer = poBuffer;
                            }
                        }
                        else
                        {
                            poOldestBuffer = poBuffer;
                        }
                    }
                }

                if (poOldestBuffer != MNULL)
                {
                    while (LH264D_ReleasePicture(
                                poDecMod->hDecoder,
                                (LH264D_PictureInfoTypeHeader*)(poOldestBuffer->pvPrivateData))
                           == LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL)
                    {
                        usleep(1000);
                    }

                    poOldestBuffer->hBuffer = MNULL;
                }
            }
            while (poOldestBuffer != MNULL);
        }

        ModLnk_Cleanup(&(poDecMod->oOutLink));

        if (poDecMod->hDecoder != MNULL)
        {
            MsgLog(2, "LH264D_Destroy...");

            while (LSTATUS_IS_FAIL(LH264D_Destroy(poDecMod->hDecoder)))
            {
                MsgLog(2, "LH264D_Destroy returns error. Waiting...");
                usleep(10*1000);
            }

            MsgLog(2, "LH264D_Destroy done.");
        }

        if (poDecMod->hDevThreadOut != MNULL)
        {
            MsgLog(2, "LDeviceThread_Destroy(out)...");
            LDeviceThread_Destroy(poDecMod->hDevThreadOut);
        }

        if (poDecMod->hDevThreadIn != MNULL)
        {
            MsgLog(2, "LDeviceThread_Destroy(in)...");
            LDeviceThread_Destroy(poDecMod->hDevThreadIn);
        }

        poDecMod->hDecoder          = MNULL;
        poDecMod->hDevThreadIn      = MNULL;
        poDecMod->hDevThreadOut     = MNULL;
        poDecMod->oInLink.poModLnk  = MNULL;
    }

    MsgLog(2, "...}");
}

/************************************************************************************************************\

Function:       DecMod_CpuThreadIn

Description:    .

\************************************************************************************************************/
LStatus DecMod_CpuThreadIn(void* pvData)
{
    if (pvData == MNULL)
    {
        MsgLogErr("ERROR! NULL data.");
        return LStatus_INVALID_PARAM;
    }

    MsgLog(2, "Start thread %p.", pthread_self());

    DecoderModule*  poDecMod    = (DecoderModule*)pvData;
    LStatus         eStatus     = LStatus_OK;
    MBOOL           bDecodeFailure    = MFALSE;

    poDecMod->uiAccessUnitId   = 0;
    poDecMod->uiTotalSizeBytes = 0;

    MCHAR8  szThreadName[16] = "";
    snprintf(szThreadName, sizeof(szThreadName), "%.12s-%s", poDecMod->szModuleName, "In");

    ModThread_SetName(szThreadName);

    while (!poDecMod->oCpuThreadIn.bKillThread)
    {
        BufferInfo* poSrcBuffer;

        eStatus = ModLnkIn_GetSubmittedBuffer(
                    &(poDecMod->oInLink),
                    100,
                    0,
                    poDecMod->hDevThreadIn,
                    &poSrcBuffer,
                    MNULL,
                    MNULL);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            LDeviceThread_Handle hLastVCDT   = MNULL;
            MUINT64              uiLastVCTag = NO_TAG;

            if (poSrcBuffer->bEndOfStream)
            {
                MsgLog(4, "END-OF-STREAM");

                poDecMod->oCpuThreadIn.bKillThread = MTRUE;
                eStatus = LStatus_END_OF_STREAM;
            }
            else if ((poSrcBuffer->uiSizeBytes > 0) && !bDecodeFailure)
            {
                LH264D_DecodeOpt oDecodeOpt;

                oDecodeOpt.eType            = LH264D_DecodeOptTypeHeader_STANDARD;
                oDecodeOpt.hBuffer          = (MUINT64)(poSrcBuffer->hBuffer);
                oDecodeOpt.uiAccessUnitId   = (MUINT64) poSrcBuffer->uiSyncPtsUsec;
                oDecodeOpt.uiStartOffset    = poSrcBuffer->uiStartOffset;
                oDecodeOpt.uiSize           = poSrcBuffer->uiSizeBytes;
                oDecodeOpt.flDecodeFlags    = 0;

                while (!poDecMod->oCpuThreadIn.bKillThread)
                {
                    MUINT32 uiBytesConsumed = 0;

                    MsgLog(4, "LH264D_DecodeNALBlob(Buffer[%u], Offset= %u, Size= %u)...",
                              poSrcBuffer->uiId, oDecodeOpt.uiStartOffset, oDecodeOpt.uiSize);

                    eStatus = LH264D_DecodeNALBlob(
                                    poDecMod->hDecoder,
                                    (LH264D_DecodeOptTypeHeader*)&oDecodeOpt,
                                    &uiBytesConsumed);

                    MsgLog(
                        4,
                        "LH264D_DecodeNALBlob done. (status= %d - %s, Consumed= %u)",
                        eStatus,
                        GetStatusStr(eStatus),
                        uiBytesConsumed);

                    if (LSTATUS_IS_SUCCESS(eStatus))
                    {
                        break;
                    }

                    if (((eStatus == LStatus_INCOMPLETE_COMMAND)
                         || (eStatus == LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL))
                        && (uiBytesConsumed < oDecodeOpt.uiSize))
                    {
                        MsgLog(3, "LH264D_DecodeNALBlob(Buffer[%u]) incompleted (bytes consumed= %u). "
                                  "Take a nap and retry...", poSrcBuffer->uiId, uiBytesConsumed);
                        oDecodeOpt.uiStartOffset += uiBytesConsumed;
                        oDecodeOpt.uiSize        -= uiBytesConsumed;
                        usleep(1000);
                    }
                    else
                    {
                        MsgLogErr("ERROR! LH264D_DecodeNALBlob returned %d (%s). Stop decoding...",
                                  eStatus, GetStatusStr(eStatus));
                        bDecodeFailure = MTRUE;
                        break;
                    }
                }

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    if (eStatus != LStatus_OK_MISSING_FIRST_SPS)
                    {
                        poDecMod->bDecoding = MTRUE;
                        poDecMod->uiAccessUnitId++;
                        poDecMod->uiTotalSizeBytes += poSrcBuffer->uiSizeBytes;
                    }
                    eStatus = LH264D_GetDTInputLastSubmissionTag(
                                                    poDecMod->hDecoder,
                                                    &uiLastVCTag,
                                                    &hLastVCDT);
                    if (LSTATUS_IS_FAIL(eStatus))
                    {
                        MsgLogErr("ERROR! LH264D_GetDTInputLastSubmissionTag returned %d (%s). Stop decoding...",
                                  eStatus, GetStatusStr(eStatus));
                        bDecodeFailure = MTRUE;
                    }
                }
            }
            else
            {
                if (poSrcBuffer->uiSizeBytes == 0)
                {
                    MsgLogErr("ERROR! Source linear buffer[%u] size not initialized.", poSrcBuffer->uiId);
                }

                eStatus = LStatus_FAIL;
            }

            ModLnkIn_ReturnBuffer(
                &(poDecMod->oInLink),
                poSrcBuffer,
                LSTATUS_IS_SUCCESS(eStatus) ? hLastVCDT : MNULL,
                uiLastVCTag);
        }

        if (LSTATUS_IS_FAIL(eStatus)
            && (eStatus != LStatus_TIMEOUT))
        {
            usleep(1000);
        }
    }

    if (poDecMod->bDecoding)
    {
        if (poDecMod->bDecodeNalBlobs && (eStatus != LStatus_END_OF_STREAM))
        {
            // Flush codec input DeviceThreads
            eStatus = LH264D_FlushDTInput(poDecMod->hDecoder);
            if(LSTATUS_IS_SUCCESS(eStatus))
            {
                poDecMod->bNalBlobFlushed = MTRUE;
            }
            else
            {
                MsgLogErr("ERROR! LH264D_FlushDTInput returned %d (%s).", eStatus, GetStatusStr(eStatus));
            }
        }

        MsgLog(2, "LH264D_EndStream...");

        while (MTRUE)
        {
            eStatus = LH264D_EndStream(poDecMod->hDecoder);

            if (eStatus != LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL)
            {
                break;
            }

            usleep(1000);
        }

        MsgLog(2, "LH264D_EndStream done. (status= %d - %s)", eStatus, GetStatusStr(eStatus));
    }

    poDecMod->oOutLink.bSkipAll = MTRUE;

    MsgLog(2, "Kill thread.");

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       DecMod_CpuThreadOut

Description:    .

\************************************************************************************************************/
LStatus DecMod_CpuThreadOut(void* pvData)
{
    const MFLAG32 flBadPictureStatusFlags = 0
        | LH264D_PictureStatusFlag_FATALFIRMWAREERROR
        | LH264D_PictureStatusFlag_FATALSTREAMERROR
        | LH264D_PictureStatusFlag_MISSINGDATA
        | LH264D_PictureStatusFlag_CORRUPTEDDATA
        | LH264D_PictureStatusFlag_UNSUPPORTEDINPUT;

    if (pvData == MNULL)
    {
        MsgLogErr("ERROR! NULL data.");
        return LStatus_INVALID_PARAM;
    }

    DecoderModule* poDecMod         = (DecoderModule*)pvData;
    MUINT64        uiFirstTime      = 0;
    MUINT64        uiLastPictTime   = 0;
    MUINT64        uiLastStreamCnt  = 0;
    MUINT64        uiRelStreamCnt   = 0;

    poDecMod->uiPictureCount    = 0;
    poDecMod->uiElapsedTimeUsec = 0;

    MCHAR8  szThreadName[16] = "";
    snprintf(szThreadName, sizeof(szThreadName), "%.11s-%s", poDecMod->szModuleName, "Out");

    ModThread_SetName(szThreadName);
    MsgLog(2, "Start thread %p.", pthread_self());

    while (!poDecMod->oCpuThreadOut.bKillThread)
    {
        BufferInfo* poDstBuffer;

        if (!poDecMod->bDecoding)
        {
            usleep(10*1000);
            continue;
        }

        LStatus eStatus = ModLnk_GetReturnedBuffer(
                            &(poDecMod->oOutLink),
                            100,
                            poDecMod->hDevThreadOut,
                            &poDstBuffer);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = ((poDstBuffer->pvPrivateData != MNULL) && poDstBuffer->bInternal)
                      ? LStatus_OK : LStatus_FAIL;

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                LH264D_PictureInfo* poPictureInfo = (LH264D_PictureInfo*)(poDstBuffer->pvPrivateData);

                MUINT uiTimeoutMsec = 1;

                if (poDstBuffer->hBuffer != MNULL)
                {
                    MsgLog(4, "LH264D_ReleasePicture: Buffer[%u] ReleaseCnt[%X].", poDstBuffer->uiId,
                              poPictureInfo->uiStreamCnt);

                    while ((eStatus = LH264D_ReleasePicture(poDecMod->hDecoder,
                                                            (LH264D_PictureInfoTypeHeader*)poPictureInfo))
                           == LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL)
                    {
                        usleep(1000);
                    }

                    if (LSTATUS_IS_SUCCESS(eStatus))
                    {
                        uiRelStreamCnt = poPictureInfo->uiStreamCnt;
                    }
                    else
                    {
                        MsgLogErr("ERROR! LH264D_ReleasePicture returned status= %d (%s).",
                                  eStatus, GetStatusStr(eStatus));
                        MsgLogErr("Stream count: %lu, Previous: %lu. Continue...",
                                  poPictureInfo->uiStreamCnt, uiRelStreamCnt);
                    }

                    poDstBuffer->hBuffer = MNULL;
                    uiTimeoutMsec = 0;
                }

                poPictureInfo->eType = LH264D_PictureInfoTypeHeader_STANDARD;

                MsgLog(6, "LH264D_GetNextPicture(timeout= %u msec)...", uiTimeoutMsec);

                eStatus = LH264D_GetNextPicture(
                            poDecMod->hDecoder,
                            (LH264D_PictureInfoTypeHeader*)poPictureInfo,
                            uiTimeoutMsec);

                MUINT64 uiCurTime = GetMonoTimeUsec();

                if (uiFirstTime == 0)
                {
                    uiFirstTime = uiCurTime;
                }

                uiCurTime -= uiFirstTime;
                poDecMod->uiElapsedTimeUsec = uiCurTime;

                MsgLog(6, "LH264D_GetNextPicture done. (Buffer[%u], status= %d - %s)",
                          poDstBuffer->uiId, eStatus, GetStatusStr(eStatus));

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    MBOOL bTsDone = MFALSE;

                    poDstBuffer->uiTimestampUsec = 0;

                    if (poPictureInfo->flPictureInfoFlags & LH264D_PictureInfoFlag_SEIRBSPPRESENT)
                    {
                        MUINT64 uiTimestampUsec = 0;

                        if (EncMod_GetSeiTsUserMsg(
                                            (LBuffer_Handle)poPictureInfo->hSEIRbspBuffer,
                                            poPictureInfo->uiSEIRbspStartOffset,
                                            poPictureInfo->uiSEIRbspSize,
                                            &uiTimestampUsec))
                        {
                            poDstBuffer->uiTimestampUsec = uiTimestampUsec;
                            bTsDone = MTRUE;
                        }
                        else if (poPictureInfo->uiSEIRbspSize != 0)
                        {
                            MsgLog(1, "WARNING! SEI seems invalid (Offset= %u|%u, Size= %u|%u, hSei= %p)",
                                      poPictureInfo->uiSEIRbspStartOffset, poPictureInfo->uiSEIRbspContOffset,
                                      poPictureInfo->uiSEIRbspSize, poPictureInfo->uiSEIRbspContSize,
                                      (void*)poPictureInfo->hSEIRbspBuffer);
                        }
                    }

                    if (!bTsDone)
                    {
                        MUINT64 uiPictCnt = poPictureInfo->uiStreamCnt / 16;

                        poDstBuffer->uiTimestampUsec = (uiPictCnt * poDecMod->uiFrameRateDen * 1000 * 1000)
                                                       / poDecMod->uiFrameRateNum;
                    }

                    if (poPictureInfo->flStreamInfoFlags & LH264D_StreamInfoUpdateFlag_VUI)
                    {
                        LH264_StreamInfoVUI oVui;

                        memset(&oVui, 0, sizeof(oVui));

                        oVui.eType = LH264_StreamInfoTypeHeader_VUI;

                        if (LSTATUS_IS_SUCCESS(LH264D_GetStreamInfo(
                                                    poDecMod->hDecoder,
                                                    (LH264_StreamInfoTypeHeader*)&oVui)))
                        {
                            if (oVui.flVUIFlags & LH264_StreamInfoVUIFlag_COLOURDESCRIPTIONPRESENT)
                            {
                                MsgLog(2, "ColourPrimaries= %d, TransFn= %d, MatrixCoeff= %d",
                                          oVui.eColourPrimaries, oVui.eTransferCharacteristics,
                                          oVui.eMatrixCoefficients);
                            }
                        }
                    }

                    if (poPictureInfo->flPictureInfoFlags & LH264D_PictureInfoFlag_PICTURETIMESTAMPRESENT)
                    {
                        poDstBuffer->uiSyncPtsUsec = poPictureInfo->auiPictureTimestamp[0];
                    }
                    else
                    {
                        poDstBuffer->uiSyncPtsUsec = poPictureInfo->uiAccessUnitId;
                    }

                    if (poDstBuffer->uiSyncPtsUsec == 0)
                    {
                        poDstBuffer->uiSyncPtsUsec = poDstBuffer->uiTimestampUsec;
                    }

                    if ((poPictureInfo->flPictureInfoFlags & LH264D_PictureInfoFlag_PICTURENOTREADY) == 0)
                    {
                        poDecMod->uiPictureCount++;

                        MsgLog(4, "***** Got picture #%u in %u usec (avg= %u usec, timestamp= %lu, "
                                  "Buffer[%x], ReleaseCnt[%X]) *****",
                                  poDecMod->uiPictureCount,
                                  (MUINT)(poDecMod->uiElapsedTimeUsec - uiLastPictTime),
                                  (MUINT)(poDecMod->uiElapsedTimeUsec / poDecMod->uiPictureCount),
                                  poDstBuffer->uiTimestampUsec,
                                  poDstBuffer->uiId,
                                  poPictureInfo->uiStreamCnt);

                        uiLastPictTime = poDecMod->uiElapsedTimeUsec;
                    }
                    else
                    {
                        // TODO: Handle low-latency mode
                        MsgLog(2, "Picture not ready. StreamId= %u, StreamCnt= %lu",
                                  poPictureInfo->uiStreamId, poPictureInfo->uiStreamCnt);
                        // We need to keep stream count ordering: So, play the not-ready pictures. Anyway,
                        // it's not yet planned to support low-latency here.
                        // eStatus = LStatus_TIMEOUT;
                    }

                    poDstBuffer->hBuffer    = (LBuffer_Handle)(poPictureInfo->hBuffer);
                    poDstBuffer->eFrameType =
                        (poPictureInfo->eField == LH264_FRAME)       ? FrameType_Progressive :
                        (poPictureInfo->eField == LH264_TOPFIELD)    ? FrameType_TopField    :
                        (poPictureInfo->eField == LH264_BOTTOMFIELD) ? FrameType_BottomField :
                                                                       FrameType_Unknown;

                    if ((poPictureInfo->flPictureStatusFlags & flBadPictureStatusFlags) != 0)
                    {
                        MsgLog(1, "Bad picture! (status= 0x%08X)", poPictureInfo->flPictureStatusFlags);
                        // We need to keep stream count ordering: So, play the bad pictures....
                        // eStatus = LStatus_FAIL;
                    }

                    if ((poPictureInfo->uiStreamCnt <= uiLastStreamCnt)
                        || ((poPictureInfo->uiStreamCnt & 0xF) != 0))
                    {
                        MsgLogErr("ERROR! Bad stream count (Current= %lu, Last= %lu). Continue...",
                                  poPictureInfo->uiStreamCnt, uiLastStreamCnt);
                    }

                    uiLastStreamCnt = poPictureInfo->uiStreamCnt;
                }
                else if (eStatus == LStatus_END_OF_STREAM)
                {
                    MsgLog(2, "***** END-OF-STREAM (%u pictures in %u msec) *****",
                              poDecMod->uiPictureCount, (MUINT)(poDecMod->uiElapsedTimeUsec/1000));

                    poDstBuffer->bEndOfStream = MTRUE;
                    poDecMod->bDecoding       = MFALSE;
                    poDecMod->oCpuThreadOut.bKillThread = MTRUE;

                    eStatus = LStatus_OK;
                }
                else if ((eStatus != LStatus_TIMEOUT) && (eStatus != LStatus_STREAM_NOT_INITIALIZED))
                {
                    MsgLogErr("ERROR! LH264D_GetNextPicture returned %d (%s).",
                              eStatus, GetStatusStr(eStatus));
                }
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                ModLnk_SubmitBuffer(&(poDecMod->oOutLink), poDstBuffer, MNULL, NO_TAG);
            }
            else
            {
                ModLnk_ReleaseBuffer(&(poDecMod->oOutLink), poDstBuffer);
            }
        }

        if (LSTATUS_IS_FAIL(eStatus)
            && (eStatus != LStatus_TIMEOUT))
        {
            usleep(1000);
        }
    }

    MsgLog(2, "Kill thread.");

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       DecMod_Start

Description:    .

\************************************************************************************************************/
LStatus DecMod_Start(DecoderModule* poDecMod)
{
    LStatus eStatus = LStatus_INVALID_PARAM;

    MsgLog(2, "{...");

    if ((poDecMod != MNULL)
        && (poDecMod->hDecoder != MNULL))
    {
        if ((poDecMod->oInLink.poModLnk != MNULL)
            && (poDecMod->oOutLink.uiSubmitCount > 0))
        {
            eStatus = LStatus_OK;
        }
        else
        {
            MsgLogErr("ERROR! Bad connection.");
            eStatus = LStatus_FAIL;
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            poDecMod->bNalBlobFlushed = MFALSE;

            eStatus = ModThread_Start(&(poDecMod->oCpuThreadOut), poDecMod, DecMod_CpuThreadOut);

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                eStatus = ModThread_Start(&(poDecMod->oCpuThreadIn), poDecMod, DecMod_CpuThreadIn);

                if (LSTATUS_IS_FAIL(eStatus))
                {
                    ModThread_Stop(&(poDecMod->oCpuThreadOut));
                }
            }
        }
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       DecMod_Stop

Description:    .

\************************************************************************************************************/
void DecMod_Stop(DecoderModule* poDecMod)
{
    MsgLog(2, "{...");

    if (poDecMod != MNULL)
    {
        ModThread_Stop(&(poDecMod->oCpuThreadIn));

        if (poDecMod->bDecoding && poDecMod->bDecodeNalBlobs && !poDecMod->bNalBlobFlushed)
        {
            // If the last NAL blob has not been flushed, we have to flush it since it could be very long
            // to wait for.  Flush codec input DeviceThreads
            LStatus eStatus = LH264D_FlushDTInput(poDecMod->hDecoder);
            if(LSTATUS_IS_SUCCESS(eStatus))
            {
                poDecMod->bNalBlobFlushed = MTRUE;
            }
            else
            {
                MsgLogErr("ERROR! LH264D_FlushDTInput returned %d (%s).", eStatus, GetStatusStr(eStatus));
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                while (MTRUE)
                {
                    eStatus = LH264D_EndStream(poDecMod->hDecoder);

                    if (eStatus != LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL)
                    {
                        break;
                    }

                    usleep(1000);
                }
            }

            if (LSTATUS_IS_FAIL(eStatus))
            {
                MsgLogErr("Error trying to flush and post EOS! (eStatus = %d, %s). Continue...",
                          eStatus, GetStatusStr(eStatus));
            }
        }

        MsgLog(2, "Wait EOS...");
        while (poDecMod->bDecoding)
        {
            usleep(10*1000);
        }
        MsgLog(2, "Wait EOS done.");

        ModThread_Stop(&(poDecMod->oCpuThreadOut));

        poDecMod->oOutLink.bSkipAll = MFALSE;
    }

    MsgLog(2, "...}");
}
