/************************************************************************************************************\

Module Name:    AudioEncoderModule.c

Description:    Module for Liberatus audio encoder.

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "AudioEncoderModule.h"

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

typedef enum
{
    ChannelCfg_MONO             = 1,
    ChannelCfg_STEREO           = 2,

} AudioSpecificConfig_ChannelCfg;

typedef enum
{
    SamplingRate_96000          = 0,
    SamplingRate_88200          = 1,
    SamplingRate_64000          = 2,
    SamplingRate_48000          = 3,
    SamplingRate_44100          = 4,
    SamplingRate_32000          = 5,
    SamplingRate_24000          = 6,
    SamplingRate_22050          = 7,
    SamplingRate_16000          = 8,
    SamplingRate_12000          = 9,
    SamplingRate_11025          = 10,
    SamplingRate_8000           = 11,
    SamplingRate_7350           = 12,

} AudioSpecificConfig_SamplingRate;

typedef enum
{
    ObjectType_NULL             = 0,
    ObjectType_AAC_MAIN         = 1,
    ObjectType_AAC_LC           = 2,
    ObjectType_SBR              = 5,
    ObjectType_PS               = 29,

} AudioSpecificConfig_ObjectType;

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

static const    MCHAR8      g_szModuleNameBase[]    = "AEnc";
static          MUINT32     g_uiAudioEncModCount    = 0;

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Function:       EncodeAudioSpecificCfg

Description:    .

\************************************************************************************************************/
LStatus EncodeAudioSpecificCfg(
    MUINT16*                    puiAudioSpecificCfg,
    LAudioCodec_SamplingRate    eSamplingRate,
    LAudioCodec_ChannelCfg      eChannelCfg,
    LAudioCodec_AACProfile      eProfile)
{
    LStatus eStatus = LStatus_OK;

    // AudioSpecificConfig (big-endian): OOOOORRRRCCCCPPP
    //   O: Audio object type.
    //   R: Sampling rate.
    //   C: Channel configuration.
    //   P: Padding.

    MUINT16 uiConfig = 0;

    // Encode audio object type.
    switch(eProfile)
    {
        case LAudioCodec_AACProfile_LC:
        {
            uiConfig |= ObjectType_AAC_LC;
            break;
        }
        case LAudioCodec_AACProfile_HEV1:
        {
            uiConfig |= ObjectType_SBR;
            break;
        }
        case LAudioCodec_AACProfile_HEV2:
        {
            uiConfig |= ObjectType_PS;
            break;
        }
        default:
        {
            eStatus = LStatus_INVALID_PARAM;
            break;
        }
    }

    uiConfig <<= 4;

    // Encode sampling rate.
    switch(eSamplingRate)
    {
        case LAudioCodec_SamplingRate_8000Hz:
        {
            uiConfig |= SamplingRate_8000;
            break;
        }
        case LAudioCodec_SamplingRate_11025Hz:
        {
            uiConfig |= SamplingRate_11025;
            break;
        }
        case LAudioCodec_SamplingRate_12000Hz:
        {
            uiConfig |= SamplingRate_12000;
            break;
        }
        case LAudioCodec_SamplingRate_16000Hz:
        {
            uiConfig |= SamplingRate_16000;
            break;
        }
        case LAudioCodec_SamplingRate_22050Hz:
        {
            uiConfig |= SamplingRate_22050;
            break;
        }
        case LAudioCodec_SamplingRate_24000Hz:
        {
            uiConfig |= SamplingRate_24000;
            break;
        }
        case LAudioCodec_SamplingRate_32000Hz:
        {
            uiConfig |= SamplingRate_32000;
            break;
        }
        case LAudioCodec_SamplingRate_44100Hz:
        {
            uiConfig |= SamplingRate_44100;
            break;
        }
        case LAudioCodec_SamplingRate_48000Hz:
        {
            uiConfig |= SamplingRate_48000;
            break;
        }
        case LAudioCodec_SamplingRate_88200Hz:
        {
            uiConfig |= SamplingRate_88200;
            break;
        }
        case LAudioCodec_SamplingRate_96000Hz:
        {
            uiConfig |= SamplingRate_96000;
            break;
        }
        default:
        {
            eStatus = LStatus_INVALID_PARAM;
            break;
        }
    }

    uiConfig <<= 4;

    // Encode channel configuration.
    switch(eChannelCfg)
    {
        case LAudioCodec_ChannelCfg_MONO:
        {
            uiConfig |= ChannelCfg_MONO;
            break;
        }
        case LAudioCodec_ChannelCfg_STEREO:
        {
            uiConfig |= ChannelCfg_STEREO;
            break;
        }
        default:
        {
            eStatus = LStatus_INVALID_PARAM;
            break;
        }
    }

    uiConfig <<= 3;

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        // Reverse endianness.
        *puiAudioSpecificCfg = ((uiConfig & 0x00ff) << 8) + ((uiConfig & 0xff00) >> 8);
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       AudioEncMod_Init

Description:    .

\************************************************************************************************************/
LStatus AudioEncMod_Init(
            AudioEncModule*             poAEncMod,
            LDevice_Handle              hDevice,
            MUINT                       uiOutBufferCount,
            LBuffer_Type                eOutBufferType,
            LAudioCodec_SamplingRate    eSamplingRate,
            LAudioCodec_ChannelCfg      eChannelCfg,
            LAudioCodec_AACStreamFormat eStreamFormat,
            LAudioCodec_AACProfile      eProfile,
            LAudioCodec_AACQualityLevel eQualityLevel,
            MBOOL32                     bUseTns,
            MUINT32                     uiBitRate)
{
    MsgLog(2, "{...");

    AudioEncMod_Cleanup(poAEncMod);

    LStatus eStatus = ((poAEncMod != MNULL)
                       && (hDevice != MNULL)
                       && (uiOutBufferCount > 0))
                      ? LStatus_OK : LStatus_INVALID_PARAM;

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LAudioEncoder_CreateOpt oCreateOpt  = { LAudioEncoder_CreateOptTypeHeader_STANDARD };
        oCreateOpt.hDevice                              = hDevice;
        oCreateOpt.uiNbOutputBuffers                    = uiOutBufferCount;
        oCreateOpt.eOutputBufferType                    = eOutBufferType;
        oCreateOpt.oPCMAudioAttributes.eBitDepth        = LAudioCodec_BitDepth_16;
        oCreateOpt.oPCMAudioAttributes.ePCMMode         = LAudioCodec_PCMMode_LINEAR;
        oCreateOpt.oPCMAudioAttributes.eSamplingRate    = eSamplingRate;
        oCreateOpt.oPCMAudioAttributes.eChannelCfg      = eChannelCfg;
        oCreateOpt.oPCMAudioAttributes.flFlags          = 0;

        LAudioEncoder_AACOptions oAacOpt    = { LAudioEncoder_CodecOptTypeHeader_AAC };
        oCreateOpt.peEncoderCodecOptions    = &(oAacOpt.eType);
        oAacOpt.eProfile                    = eProfile;
        oAacOpt.eQualityLevel               = eQualityLevel;
        oAacOpt.eStreamFormat               = eStreamFormat;
        oAacOpt.uiBitRate                   = uiBitRate;
        oAacOpt.flEncodingFlags             = bUseTns ? LAudioEncoder_AACFlags_USE_TNS : 0;

        eStatus = LAudioEncoder_Create(
                      &(oCreateOpt.eType),
                      &(poAEncMod->hEncoder));
    }

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        LAudioEncoder_BufferInfo oBufInfo = { LAudioEncoder_InfoTypeHeader_BUFFER_INFO };

        eStatus = LAudioEncoder_GetInfo(poAEncMod->hEncoder, &(oBufInfo.eType));

        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            poAEncMod->uiMaxEncodedBufSize = oBufInfo.uiMaximumOutputBufferSize;
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        snprintf(
                    poAEncMod->szModuleName,
                    sizeof(poAEncMod->szModuleName),
                    "%s%d",
                    g_szModuleNameBase,
                    g_uiAudioEncModCount);

        eStatus = ModLnk_Init(
                    &(poAEncMod->oOutLink),
                    hDevice,
                    uiOutBufferCount,
                    MNULL,
                    MTRUE,
                    0,
                    poAEncMod->szModuleName);
    }

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = EncodeAudioSpecificCfg(
                        &poAEncMod->uiAudioSpecificConfig,
                        eSamplingRate,
                        eChannelCfg,
                        eProfile);
    }

    if (LSTATUS_IS_FAIL(eStatus))
    {
        AudioEncMod_Cleanup(poAEncMod);
    }

    ++g_uiAudioEncModCount;

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       AudioEncMod_Cleanup

Description:    .

\************************************************************************************************************/
void AudioEncMod_Cleanup(AudioEncModule* poAEncMod)
{
    MsgLog(2, "{...");

    if (poAEncMod != MNULL)
    {
        ModLnk_Cleanup(&(poAEncMod->oOutLink));

        if (poAEncMod->hEncoder != MNULL)
        {
            MsgLog(2, "LAudioEncoder_Destroy...");

            LStatus eStatus = LAudioEncoder_Destroy(poAEncMod->hEncoder);
            if(LSTATUS_IS_FAIL(eStatus))
            {
                MsgLog(0, "LAudioEncoder_Destroy returns error: %d", eStatus);
            }

            MsgLog(2, "LAudioEncoder_Destroy done.");
        }

        poAEncMod->hEncoder                 = MNULL;
        poAEncMod->oInLink.poModLnk         = MNULL;
        poAEncMod->uiAudioSpecificConfig    = 0;
        poAEncMod->uiCopiedBytes            = 0;
        poAEncMod->uiElapsedTimeUsec        = 0;
        poAEncMod->uiEncodedBuffers         = 0;
    }

    MsgLog(2, "...}");
}

/************************************************************************************************************\

Function:       AudioEncMod_EncodeBuffer

Description:    .

\************************************************************************************************************/
LStatus AudioEncMod_EncodeBuffer(AudioEncModule* poAEncMod, MBOOL32 bReplaceOldestIfFull)
{
    LStatus eStatus = LStatus_OK;

    MsgLog(4, "LAudioEncoder_EncodeBuffer...");

    LAudioEncoder_EncodeOpt oEncodeOpt
            = { LAudioEncoder_EncodeOptTypeHeader_STANDARD };

    oEncodeOpt.bReplaceOldestBuffer = bReplaceOldestIfFull;

    eStatus = LAudioEncoder_EncodeBuffer(
                  poAEncMod->hEncoder,
                  &oEncodeOpt.eType);

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        poAEncMod->uiEncodedBuffers++;
        MsgLog(4, "LAudioEncoder_EncodeBuffer done");
    }
    else
    {
        if(eStatus == LStatus_NO_MORE_DATA)
        {
            // OK: encoded all the data copied in the codec internal buffer.
        }
        else if(eStatus == LStatus_OUT_OF_RESOURCES)
        {
            // OK: no available output buffer to store encoded buffer.
        }
        else if(eStatus != LStatus_END_OF_STREAM)
        {
            MsgLog(0, "Error, can't encode buffer: %d", eStatus);
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       AudioEncMod_CpuThreadIn

Description:    .

\************************************************************************************************************/
LStatus AudioEncMod_CpuThreadIn(void* pvData)
{
    if (pvData == MNULL)
    {
        MsgLogErr("ERROR! NULL data.");
        return LStatus_INVALID_PARAM;
    }

    AudioEncModule* poAEncMod   = (AudioEncModule*)pvData;
    LStatus         eStatus     = LStatus_OK;
    BufferInfo*     poSrcBuffer = MNULL;

    poAEncMod->uiCopiedBytes = 0;
    poAEncMod->uiEncodedBuffers = 0;

    MCHAR8  szThreadName[16] = "";
    snprintf(szThreadName, sizeof(szThreadName), "%.12s-%s", poAEncMod->szModuleName, "In");

    ModThread_SetName(szThreadName);
    MsgLog(2, "Start thread %p.", pthread_self());

    while (!poAEncMod->oCpuThreadIn.bKillThread)
    {
        eStatus = LStatus_OK;

        // If all data has been copied to the codec, take new buffer.
        if(poSrcBuffer == MNULL)
        {
            eStatus = ModLnkIn_GetSubmittedBuffer(
                        &(poAEncMod->oInLink),
                        100,
                        0,
                        MNULL,
                        &poSrcBuffer,
                        MNULL,
                        MNULL);
        }

        // Try to copy input data in the codec.
        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            if (poSrcBuffer->bEndOfStream)
            {
                MsgLog(4, "END-OF-STREAM");

                poAEncMod->oCpuThreadIn.bKillThread = MTRUE;
                eStatus = LStatus_END_OF_STREAM;
            }
            else if (poSrcBuffer->uiSizeBytes > 0)
            {
                LAudioCodec_BufferInfo oBufInfo = { LAudioCodec_BufferInfoTypeHeader_STANDARD };
                oBufInfo.hBuffer                = poSrcBuffer->hBuffer;
                oBufInfo.uiSize                 = poSrcBuffer->uiSizeBytes;
                oBufInfo.uiStartOffset          = poSrcBuffer->uiStartOffset;

                MsgLog(4, "LAudioEncoder_PutData(Buffer[%u], Size=%u)...",
                       poSrcBuffer->uiId, poSrcBuffer->uiSizeBytes);

                eStatus = LAudioEncoder_PutData(
                                            poAEncMod->hEncoder,
                                            &(oBufInfo.eType),
                                            poSrcBuffer->uiSyncPtsUsec);

                if(LSTATUS_IS_SUCCESS(eStatus))
                {
                    poAEncMod->uiCopiedBytes += poSrcBuffer->uiSizeBytes;

                    // Buffer copied, return it and clear it to get a new one on next iteration.
                    ModLnkIn_ReturnBuffer(
                        &(poAEncMod->oInLink),
                        poSrcBuffer,
                        MNULL,
                        NO_TAG);

                    poSrcBuffer = MNULL;
                }
                else
                {
                    if(eStatus == LStatus_OUT_OF_RESOURCES)
                    {
                        MsgLog(4, "Warning: no more space in audio codec internal buffer! "
                                "(copied:%u bytes, trying to copy:%u bytes)",
                                poAEncMod->uiCopiedBytes,
                                poSrcBuffer->uiSizeBytes);
                        eStatus = LStatus_OK;
                    }
                }
            }
            else
            {
                if (poSrcBuffer->uiSizeBytes == 0)
                {
                    MsgLogErr("ERROR! Source linear buffer[%u] size not initialized.", poSrcBuffer->uiId);
                }

                eStatus = LStatus_FAIL;
            }
        }

        eStatus = LStatus_OK;

        // Encode as much data as possible.
        while(LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = AudioEncMod_EncodeBuffer(poAEncMod, MFALSE);
        }

        if(eStatus == LStatus_OUT_OF_RESOURCES)
        {
            usleep(1000);
        }
    }

    MsgLog(2, "LAudioEncoder_EndStream...");

    MBOOL32 bFlushPendingData = (poAEncMod->uiEncodedBuffers > 0) ? MFALSE : MTRUE;

    eStatus = LAudioEncoder_EndStream(
                  poAEncMod->hEncoder,
                  bFlushPendingData);

    MsgLog(2, "LAudioEncoder_EndStream done. (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    if(!bFlushPendingData)
    {
        // Encode remaining data.
        MsgLog(2, "Stream ended, encoding remaining data...");

        while(AudioEncMod_EncodeBuffer(poAEncMod, MFALSE) != LStatus_END_OF_STREAM)
        {
            usleep(1000);
        }

        MsgLog(2, "Encode remaining data done.");
    }

    MsgLog(2, "Total copied bytes: %lu, Total encoded buffers: %u",
           poAEncMod->uiCopiedBytes,
           poAEncMod->uiEncodedBuffers);

    MsgLog(2, "Kill thread.");

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       AudioEncMod_CpuThreadOut

Description:    .

\************************************************************************************************************/
LStatus AudioEncMod_CpuThreadOut(void* pvData)
{
    if (pvData == MNULL)
    {
        MsgLogErr("ERROR! NULL data.");
        return LStatus_INVALID_PARAM;
    }

    AudioEncModule* poAEncMod   = (AudioEncModule*)pvData;
    MBOOL32 bIsEos              = MFALSE;

    MCHAR8  szThreadName[16] = "";
    snprintf(szThreadName, sizeof(szThreadName), "%.11s-%s", poAEncMod->szModuleName, "Out");

    ModThread_SetName(szThreadName);
    MsgLog(2, "Start thread %p.", pthread_self());

    while(!bIsEos)
    {
        BufferInfo* poDstBuffer = MNULL;

        LStatus eStatus = ModLnk_GetReturnedBuffer(
                            &(poAEncMod->oOutLink),
                            100,
                            MNULL,
                            &poDstBuffer);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = poDstBuffer->bInternal ? LStatus_OK : LStatus_FAIL;

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                if(poDstBuffer->hBuffer != MNULL)
                {
                    MsgLog(4, "LAudioEncoder_ReleaseBuffer: Buffer[%u]", poDstBuffer->uiId);

                    eStatus = LAudioEncoder_ReleaseBuffer(poAEncMod->hEncoder, poDstBuffer->hBuffer);

                    if(LSTATUS_IS_SUCCESS(eStatus))
                    {
                        MsgLog(4, "LAudioEncoder_ReleaseBuffer: Buffer[%u] done", poDstBuffer->uiId);
                    }
                    else
                    {
                        MsgLog(0, "ERROR, LAudioEncoder_ReleaseBuffer returned error %d", eStatus);
                    }

                    poDstBuffer->hBuffer = MNULL;
                }

                if(LSTATUS_IS_SUCCESS(eStatus))
                {
                    MUINT32 uiNbLostBuffers = 0;
                    MUINT64 uiTimestamp = 0;

                    LAudioCodec_BufferInfo oBufInfo
                            = { LAudioCodec_BufferInfoTypeHeader_STANDARD };

                    MsgLog(4, "LAudioEncoder_GetNextBuffer...");
                    eStatus = LAudioEncoder_GetNextBuffer(
                                  poAEncMod->hEncoder,
                                  &(oBufInfo.eType),
                                  100,
                                  &uiTimestamp,
                                  &uiNbLostBuffers);

                    if(LSTATUS_IS_SUCCESS(eStatus))
                    {
                        MsgLog(4, "LAudioEncoder_GetNextBuffer done. (timestamp=%lu, nb lost buffers=%u)",
                               uiTimestamp, uiNbLostBuffers);
                        poDstBuffer->hBuffer = oBufInfo.hBuffer;
                        poDstBuffer->uiSizeBytes = oBufInfo.uiSize;
                        poDstBuffer->uiStartOffset = oBufInfo.uiStartOffset;
                        poDstBuffer->uiTimestampUsec = uiTimestamp;
                        poDstBuffer->uiSyncPtsUsec = uiTimestamp;
                    }
                    else
                    {
                        if(eStatus == LStatus_TIMEOUT)
                        {
                            MsgLog(1, "LAudioEncoder_GetNextBuffer timeout.");
                        }
                        else if(eStatus == LStatus_END_OF_STREAM)
                        {
                            MsgLog(2, "End of stream reached, stopping.");
                            poDstBuffer->bEndOfStream = MTRUE;
                            bIsEos = MTRUE;
                            poAEncMod->oCpuThreadOut.bKillThread = MTRUE;
                        }
                        else
                        {
                            MsgLog(0, "ERROR, LAudioEncoder_GetNextBuffer returned status %d", eStatus);
                        }
                    }
                }
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                ModLnk_SubmitBuffer(&(poAEncMod->oOutLink), poDstBuffer, MNULL, NO_TAG);
            }
            else
            {
                ModLnk_ReleaseBuffer(&(poAEncMod->oOutLink), poDstBuffer);
            }
        }

        if (LSTATUS_IS_FAIL(eStatus)
            && (eStatus != LStatus_TIMEOUT)
            && (eStatus != LStatus_END_OF_STREAM))
        {
            MsgLog(2, "ERROR: %d", eStatus);
            usleep(1000);
        }
    }

    MsgLog(2, "Kill thread.");

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       AudioEncMod_Start

Description:    .

\************************************************************************************************************/
LStatus AudioEncMod_Start(AudioEncModule* poAEncMod)
{
    LStatus eStatus = LStatus_INVALID_PARAM;

    MsgLog(2, "{...");

    if ((poAEncMod != MNULL)
        && (poAEncMod->hEncoder != MNULL))
    {
        if ((poAEncMod->oInLink.poModLnk != MNULL)
            && (poAEncMod->oOutLink.uiSubmitCount > 0))
        {
            eStatus = LStatus_OK;
        }
        else
        {
            MsgLogErr("ERROR! Bad connection.");
            eStatus = LStatus_FAIL;
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = ModThread_Start(&(poAEncMod->oCpuThreadOut), poAEncMod, AudioEncMod_CpuThreadOut);

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                eStatus = ModThread_Start(&(poAEncMod->oCpuThreadIn), poAEncMod, AudioEncMod_CpuThreadIn);

                if (LSTATUS_IS_FAIL(eStatus))
                {
                    ModThread_Stop(&(poAEncMod->oCpuThreadOut));
                }
            }
        }
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       AudioEncMod_Stop

Description:    .

\************************************************************************************************************/
void AudioEncMod_Stop(AudioEncModule* poAEncMod)
{
    MsgLog(2, "{...");

    if (poAEncMod != MNULL)
    {
        MsgLog(2, "Stop input thread...");
        ModThread_Stop(&(poAEncMod->oCpuThreadIn));
        MsgLog(2, "Stop input thread done.");

        MsgLog(2, "Stop output thread...");
        ModThread_Stop(&(poAEncMod->oCpuThreadOut));
        MsgLog(2, "Stop output thread done.");
    }

    MsgLog(2, "...}");
}
