/************************************************************************************************************\

Module Name:    AudioDecoderModule.c

Description:    .

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "AudioDecoderModule.h"

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

typedef enum
{
    ChannelCfg_MONO             = 1,
    ChannelCfg_STEREO           = 2,

} AudioSpecificConfig_ChannelCfg;

typedef enum
{
    SamplingRate_96000          = 0,
    SamplingRate_88200          = 1,
    SamplingRate_64000          = 2,
    SamplingRate_48000          = 3,
    SamplingRate_44100          = 4,
    SamplingRate_32000          = 5,
    SamplingRate_24000          = 6,
    SamplingRate_22050          = 7,
    SamplingRate_16000          = 8,
    SamplingRate_12000          = 9,
    SamplingRate_11025          = 10,
    SamplingRate_8000           = 11,
    SamplingRate_7350           = 12,

} AudioSpecificConfig_SamplingRate;

typedef enum
{
    ObjectType_NULL             = 0,
    ObjectType_AAC_MAIN         = 1,
    ObjectType_AAC_LC           = 2,
    ObjectType_SBR              = 5,
    ObjectType_PS               = 29,

} AudioSpecificConfig_ObjectType;

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

static const    MCHAR8      g_szModuleNameBase[]    = "ADec";
static          MUINT32     g_uiAudioDecModCount    = 0;

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------



/************************************************************************************************************\

Function:       DecodeAudioSpecificCfg

Description:    .

\************************************************************************************************************/
LStatus DecodeAudioSpecificCfg(
    MUINT16                     uiAudioSpecificCfg,
    LAudioCodec_SamplingRate*   peSamplingRate,
    MUINT32*                    puiSamplingRate,
    LAudioCodec_ChannelCfg*     peChannelCfg,
    MUINT32*                    puiNbChannels,
    LAudioCodec_AACProfile*     peProfile)
{
    LStatus eStatus = LStatus_OK;

    LAudioCodec_SamplingRate   eSamplingRate    = LAudioCodec_SamplingRate_UNKNOWN;
    MUINT32                    uiSamplingRate   = 0;
    LAudioCodec_ChannelCfg     eChannelCfg      = LAudioCodec_ChannelCfg_UNKNOWN;
    MUINT32                    uiNbChannels     = 0;
    LAudioCodec_AACProfile     eProfile         = LAudioCodec_AACProfile_UNKNOWN;

    // AudioSpecificConfig (big-endian): OOOOORRRRCCCCPPP
    //   O: Audio object type.
    //   R: Sampling rate.
    //   C: Channel configuration.
    //   P: Padding.

    // Reverse endianness so the config can be parsed as represented in IEC-14496-3.
    MUINT16 uiConfig = ((uiAudioSpecificCfg & 0x00ff) << 8) + ((uiAudioSpecificCfg & 0xff00) >> 8);

    uiConfig >>= 3;

    switch(uiConfig & 0xf)
    {
        case ChannelCfg_MONO:
        {
            eChannelCfg = LAudioCodec_ChannelCfg_MONO;
            uiNbChannels = 1;
            break;
        }
        case ChannelCfg_STEREO:
        {
            eChannelCfg = LAudioCodec_ChannelCfg_STEREO;
            uiNbChannels = 2;
            break;
        }
        default:
        {
            eStatus = LStatus_INVALID_PARAM;
            break;
        }
    }

    uiConfig >>= 4;

    // Decode sampling rate.
    switch(uiConfig & 0xf)
    {
        case SamplingRate_8000:
        {
            eSamplingRate = LAudioCodec_SamplingRate_8000Hz;
            uiSamplingRate = 8000;
            break;
        }
        case SamplingRate_11025:
        {
            eSamplingRate = LAudioCodec_SamplingRate_11025Hz;
            uiSamplingRate = 11025;
            break;
        }
        case SamplingRate_12000:
        {
            eSamplingRate = LAudioCodec_SamplingRate_12000Hz;
            uiSamplingRate = 12000;
            break;
        }
        case SamplingRate_16000:
        {
            eSamplingRate = LAudioCodec_SamplingRate_16000Hz;
            uiSamplingRate = 16000;
            break;
        }
        case SamplingRate_22050:
        {
            eSamplingRate = LAudioCodec_SamplingRate_22050Hz;
            uiSamplingRate = 22050;
            break;
        }
        case SamplingRate_24000:
        {
            eSamplingRate = LAudioCodec_SamplingRate_24000Hz;
            uiSamplingRate = 24000;
            break;
        }
        case SamplingRate_32000:
        {
            eSamplingRate = LAudioCodec_SamplingRate_32000Hz;
            uiSamplingRate = 32000;
            break;
        }
        case SamplingRate_44100:
        {
            eSamplingRate = LAudioCodec_SamplingRate_44100Hz;
            uiSamplingRate = 44100;
            break;
        }
        case SamplingRate_48000:
        {
            eSamplingRate = LAudioCodec_SamplingRate_48000Hz;
            uiSamplingRate = 48000;
            break;
        }
        case SamplingRate_88200:
        {
            eSamplingRate = LAudioCodec_SamplingRate_88200Hz;
            uiSamplingRate = 88200;
            break;
        }
        case SamplingRate_96000:
        {
            eSamplingRate = LAudioCodec_SamplingRate_96000Hz;
            uiSamplingRate = 96000;
            break;
        }
        default:
        {
            eStatus = LStatus_INVALID_PARAM;
            break;
        }
    }

    uiConfig >>= 4;

    // Decode audio object type.
    switch(uiConfig & 0x1f)
    {
        case ObjectType_AAC_MAIN:
        case ObjectType_AAC_LC:
        {
            eProfile = LAudioCodec_AACProfile_LC;
            break;
        }
        case ObjectType_SBR:
        {
            eProfile = LAudioCodec_AACProfile_HEV1;
            break;
        }
        case ObjectType_PS:
        {
            eProfile = LAudioCodec_AACProfile_HEV2;
            break;
        }
        default:
        {
            eStatus = LStatus_INVALID_PARAM;
            break;
        }
    }

    if (peSamplingRate != MNULL)
    {
        *peSamplingRate = eSamplingRate;
    }
    if (puiSamplingRate != MNULL)
    {
        *puiSamplingRate = uiSamplingRate;
    }
    if (peChannelCfg != MNULL)
    {
        *peChannelCfg = eChannelCfg;
    }
    if (puiNbChannels != MNULL)
    {
        *puiNbChannels = uiNbChannels;
    }
    if (peProfile != MNULL)
    {
        *peProfile = eProfile;
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       GetSamplingRate

Description:    .

\************************************************************************************************************/
MUINT32 GetSamplingRate(LAudioCodec_SamplingRate eRate)
{
    switch(eRate)
    {
        case LAudioCodec_SamplingRate_8000Hz    : return 8000;
        case LAudioCodec_SamplingRate_11025Hz   : return 11025;
        case LAudioCodec_SamplingRate_12000Hz   : return 12000;
        case LAudioCodec_SamplingRate_16000Hz   : return 16000;
        case LAudioCodec_SamplingRate_22050Hz   : return 22050;
        case LAudioCodec_SamplingRate_24000Hz   : return 24000;
        case LAudioCodec_SamplingRate_32000Hz   : return 32000;
        case LAudioCodec_SamplingRate_44100Hz   : return 44100;
        case LAudioCodec_SamplingRate_48000Hz   : return 48000;
        case LAudioCodec_SamplingRate_88200Hz   : return 88200;
        case LAudioCodec_SamplingRate_96000Hz   : return 96000;
        default                                 : return 0;
    }
}

/************************************************************************************************************\

Function:       GetNbChannels

Description:    .

\************************************************************************************************************/
MUINT32 GetNbChannels(LAudioCodec_ChannelCfg eChannelCfg)
{
    switch(eChannelCfg)
    {
        case LAudioCodec_ChannelCfg_MONO:   return 1;
        case LAudioCodec_ChannelCfg_STEREO: return 2;
        default:                            return 0;
    }
}

/************************************************************************************************************\

Function:       GetAacProfile

Description:    .

Comments:       None.

\************************************************************************************************************/
const char* GetAacProfile(LAudioCodec_AACProfile eProfile)
{
    switch(eProfile)
    {
        case LAudioCodec_AACProfile_LC:     return "LC-AAC";
        case LAudioCodec_AACProfile_HEV1:   return "HE-AACv1(SBR)";
        case LAudioCodec_AACProfile_HEV2:   return "HE-AACv2(SBR+PS)";
        default:                            return "Unknown";
    }
}

/************************************************************************************************************\

Function:       GetStreamFormat

Description:    .

\************************************************************************************************************/
const char* GetStreamFormat(LAudioCodec_AACStreamFormat eFormat)
{
    switch(eFormat)
    {
        case LAudioCodec_AACStreamFormat_RAW:   return "Raw";
        case LAudioCodec_AACStreamFormat_ADTS:  return "ADTS";
        case LAudioCodec_AACStreamFormat_ADIF:  return "ADIF";
        default:                                return "Unknown";
    }
}

/************************************************************************************************************\

Function:       AudioDecMod_InitWithConfig

Description:    Initialize audio decoder with AudioSpecificConfig.

\************************************************************************************************************/
LStatus AudioDecMod_InitWithConfig(
    AudioDecModule*             poADecMod,
    LDevice_Handle              hDevice,
    MUINT32                     uiOutBufferCount,
    LBuffer_Type                eOutBufferType,
    MUINT16                     uiAudioSpecificCfg,
    MBOOL32                     bFramedInput)
{
    LAudioCodec_ChannelCfg      eChannelCfg     = LAudioCodec_ChannelCfg_UNKNOWN;
    LAudioCodec_SamplingRate    eSamplingRate   = LAudioCodec_SamplingRate_UNKNOWN;
    LAudioCodec_AACProfile      eAacProfile     = LAudioCodec_AACProfile_UNKNOWN;

    MsgLog(2, "{...");

    LStatus eStatus = DecodeAudioSpecificCfg(
                            uiAudioSpecificCfg,
                            &eSamplingRate,
                            MNULL,
                            &eChannelCfg,
                            MNULL,
                            &eAacProfile);

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = AudioDecMod_Init(
                            poADecMod,
                            hDevice,
                            uiOutBufferCount,
                            eOutBufferType,
                            eSamplingRate,
                            eChannelCfg,
                            eAacProfile,
                            bFramedInput);
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       AudioDecMod_Init

Description:    Initialize audio decoder without AudioSpecificConfig.

\************************************************************************************************************/
LStatus AudioDecMod_Init(
    AudioDecModule*             poADecMod,
    LDevice_Handle              hDevice,
    MUINT32                     uiOutBufferCount,
    LBuffer_Type                eOutBufferType,
    LAudioCodec_SamplingRate    eSamplingRate,
    LAudioCodec_ChannelCfg      eChannelCfg,
    LAudioCodec_AACProfile      eAacProfile,
    MBOOL32                     bFramedInput)
{
    MsgLog(2, "{...");

    AudioDecMod_Cleanup(poADecMod);

    LStatus eStatus = ((poADecMod != MNULL)
                       && (hDevice != MNULL))
                      ? LStatus_OK : LStatus_INVALID_PARAM;

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LAudioDecoder_CreateOpt  oCreateOpt = { LAudioDecoder_CreateOptTypeHeader_STANDARD };
        LAudioDecoder_AACOptions oAacOpt    = { LAudioDecoder_CodecOptTypeHeader_AAC };

        // Create audio decoder
        oCreateOpt.hDevice                                  = hDevice;
        oCreateOpt.oEncodedAudioAttributes.eBitDepth        = LAudioCodec_BitDepth_16;
        oCreateOpt.oEncodedAudioAttributes.ePCMMode         = LAudioCodec_PCMMode_LINEAR;
        oCreateOpt.oEncodedAudioAttributes.eChannelCfg      = eChannelCfg;
        oCreateOpt.oEncodedAudioAttributes.flFlags          = 0;
        oCreateOpt.oEncodedAudioAttributes.eSamplingRate    = eSamplingRate;

        oAacOpt.eProfile        = eAacProfile;
        oAacOpt.eStreamFormat   = LAudioCodec_AACStreamFormat_UNKNOWN;
        oAacOpt.bIsFramed       = bFramedInput;
        oAacOpt.flFlags         = 0;

        oCreateOpt.peDecoderCodecOptions    = &(oAacOpt.eType);
        oCreateOpt.uiNbOutputBuffers        = uiOutBufferCount;
        oCreateOpt.eOutputBufferType        = eOutBufferType;

        eStatus = LAudioDecoder_Create(&oCreateOpt.eType, &poADecMod->hDecoder);
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        snprintf(
                    poADecMod->szModuleName,
                    sizeof(poADecMod->szModuleName),
                    "%s%d",
                    g_szModuleNameBase,
                    g_uiAudioDecModCount);

        eStatus = ModLnk_Init(
                        &(poADecMod->oOutLink),
                        hDevice,
                        uiOutBufferCount,
                        MNULL,
                        MTRUE,
                        0,
                        poADecMod->szModuleName);
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        // Init members.
        poADecMod->bIsEos           = MFALSE;
        poADecMod->uiSampleSize     = 16;
        poADecMod->uiSamplingRate   = GetSamplingRate(eSamplingRate);
        poADecMod->uiNbChannels     = GetNbChannels(eChannelCfg);
    }
    else
    {
        AudioDecMod_Cleanup(poADecMod);
    }

    ++g_uiAudioDecModCount;

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       AudioDecMod_Cleanup

Description:    .

\************************************************************************************************************/
void AudioDecMod_Cleanup(AudioDecModule* poADecMod)
{
    MsgLog(2, "{...");

    if (poADecMod != MNULL)
    {
        ModLnk_Cleanup(&(poADecMod->oOutLink));

        if (poADecMod->hDecoder != MNULL)
        {
            MsgLog(2, "LAudioDecoder_Destroy...");

            LStatus eStatus = LAudioDecoder_Destroy(poADecMod->hDecoder);
            if(LSTATUS_IS_FAIL(eStatus))
            {
                MsgLog(0, "LAudioDecoder_Destroy returns error: %d", eStatus);
            }

            MsgLog(2, "LAudioDecoder_Destroy done.");
        }

        poADecMod->hDecoder          = MNULL;
        poADecMod->oInLink.poModLnk  = MNULL;
        poADecMod->bIsInitDone       = MFALSE;
        poADecMod->uiSamplingRate    = 0;
        poADecMod->uiNbChannels      = 0;
        poADecMod->uiSampleSize      = 0;
    }

    MsgLog(2, "...}");
}

/************************************************************************************************************\

Function:       AudioDecMod_DecodeBuffer

Description:    .

\************************************************************************************************************/
LStatus AudioDecMod_DecodeBuffer(AudioDecModule* poADecMod, MBOOL32 bReplaceOldestIfFull)
{
    LStatus eStatus = LStatus_OK;
    MUINT32 uiDecodedBytes = 0;

    if(poADecMod->bIsInitDone)
    {
        // Decode buffer.
        MsgLog(4, "LAudioDecoder_DecodeBuffer...");

        LAudioDecoder_DecodeOpt oDecodeOpt
                = { LAudioDecoder_DecodeOptTypeHeader_STANDARD };

        oDecodeOpt.bReplaceOldestBuffer = bReplaceOldestIfFull;

        eStatus = LAudioDecoder_DecodeBuffer(
                      poADecMod->hDecoder,
                      &oDecodeOpt.eType,
                      &uiDecodedBytes);

        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            poADecMod->uiDecodedBuffers++;
            poADecMod->uiDecodedBytes += uiDecodedBytes;
            MsgLog(4, "LAudioDecoder_DecodeBuffer done (decoded:%u bytes, total=%u bytes)",
                   uiDecodedBytes,
                   poADecMod->uiDecodedBytes);
        }
        else
        {
            if(eStatus == LStatus_NO_MORE_DATA)
            {
                // OK: decoded all the data copied in the codec internal buffer.
            }
            else if(eStatus == LStatus_OUT_OF_RESOURCES)
            {
                // OK: no available output buffer to store decoded buffer.
            }
            else if(eStatus != LStatus_END_OF_STREAM)
            {
                MsgLog(0, "Error, can't decode buffer: %d", eStatus);
            }
        }
    }
    else
    {
        // Try to Initialize decoder with data copied to the codec.
        if(poADecMod->uiCopiedBytes > 0)
        {
            MsgLog(2, "Initialize decoder. Total copied bytes=%u\n", poADecMod->uiCopiedBytes);
            LAudioDecoder_InitDecoderOpt oInitDecoderOpt
                    = { LAudioDecoder_DecodeOptTypeHeader_INIT_DECODER };

            eStatus = LAudioDecoder_DecodeBuffer(
                          poADecMod->hDecoder,
                          &oInitDecoderOpt.eType,
                          &uiDecodedBytes);

            if(LSTATUS_IS_SUCCESS(eStatus))
            {
                MsgLog(2, "Initialize decoder done.");
                poADecMod->bIsInitDone = MTRUE;
                poADecMod->uiDecodedBytes += uiDecodedBytes;

                LAudioDecoder_AACStreamInfo     oAacInfo = {LAudioDecoder_InfoTypeHeader_AAC_STREAM_INFO};
                LAudioDecoder_GenericStreamInfo oGenInfo = {LAudioDecoder_InfoTypeHeader_GENERIC_STREAM_INFO};

                if( LSTATUS_IS_SUCCESS(LAudioDecoder_GetInfo(poADecMod->hDecoder, &(oAacInfo.eType)))
                    && LSTATUS_IS_SUCCESS(LAudioDecoder_GetInfo(poADecMod->hDecoder, &(oGenInfo.eType))))
                {
                    MsgLog(2, "AAC stream information:");
                    MsgLog(2, "  Sampling rate      : %u", GetSamplingRate(oGenInfo.eSamplingRate));
                    MsgLog(2, "  Number of channels : %u", GetNbChannels(oGenInfo.eChannelCfg));
                    MsgLog(2, "  Profile            : %s", GetAacProfile(oAacInfo.eProfile));
                    MsgLog(2, "  Stream format      : %s", GetStreamFormat(oAacInfo.eStreamFormat));
                }
                else
                {
                    MsgLog(0, "Error, can't query AAC decoder stream information.");
                }
            }
            else
            {
                MsgLog(1, "Warning: can't initialize decoder: %d, flushing...", eStatus);
                LAudioDecoder_EndStream(poADecMod->hDecoder, MTRUE);
            }
        }
        else
        {
            eStatus = LStatus_FAIL;
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       AudioDecMod_CpuThreadIn

Description:    .

\************************************************************************************************************/
LStatus AudioDecMod_CpuThreadIn(void* pvData)
{
    if (pvData == MNULL)
    {
        MsgLogErr("ERROR! NULL data.");
        return LStatus_INVALID_PARAM;
    }

    AudioDecModule* poADecMod   = (AudioDecModule*)pvData;
    LStatus         eStatus     = LStatus_OK;
    BufferInfo*     poSrcBuffer = MNULL;

    MCHAR8  szThreadName[16] = "";
    snprintf(szThreadName, sizeof(szThreadName), "%.12s-%s", poADecMod->szModuleName, "In");

    ModThread_SetName(szThreadName);
    MsgLog(2, "Start thread %p.", pthread_self());

    poADecMod->uiCopiedBytes = 0;
    poADecMod->uiDecodedBytes = 0;
    poADecMod->uiDecodedBuffers = 0;

    while (!poADecMod->oCpuThreadIn.bKillThread)
    {
        eStatus = LStatus_OK;

        // If all data has been copied to the codec, take new buffer.
        if(poSrcBuffer == MNULL)
        {
            eStatus = ModLnkIn_GetSubmittedBuffer(
                        &(poADecMod->oInLink),
                        100,
                        0,
                        MNULL,
                        &poSrcBuffer,
                        MNULL,
                        MNULL);
        }

        // Try to copy input data in the codec.
        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            if (poSrcBuffer->bEndOfStream)
            {
                MsgLog(4, "END-OF-STREAM");

                poADecMod->oCpuThreadIn.bKillThread = MTRUE;
                eStatus = LStatus_END_OF_STREAM;
            }
            else if (poSrcBuffer->uiSizeBytes > 0)
            {
                LAudioCodec_BufferInfo oBufInfo = { LAudioCodec_BufferInfoTypeHeader_STANDARD };
                oBufInfo.hBuffer                = poSrcBuffer->hBuffer;
                oBufInfo.uiSize                 = poSrcBuffer->uiSizeBytes;
                oBufInfo.uiStartOffset          = poSrcBuffer->uiStartOffset;

                MsgLog(4, "LAudioDecoder_PutData(Buffer[%u], Size=%u)...",
                       poSrcBuffer->uiId, poSrcBuffer->uiSizeBytes);

                eStatus = LAudioDecoder_PutData(
                                        poADecMod->hDecoder,
                                        &(oBufInfo.eType),
                                        poSrcBuffer->uiSyncPtsUsec);

                if(LSTATUS_IS_SUCCESS(eStatus))
                {
                    poADecMod->uiCopiedBytes += poSrcBuffer->uiSizeBytes;
                    MsgLog(4, "PutData done (%u copied bytes).", poADecMod->uiCopiedBytes);

                    // Buffer copied, return it and clear it to get a new one on next iteration.
                    ModLnkIn_ReturnBuffer(
                        &(poADecMod->oInLink),
                        poSrcBuffer,
                        MNULL,
                        NO_TAG);

                    poSrcBuffer = MNULL;
                }
                else
                {
                    if(eStatus == LStatus_OUT_OF_RESOURCES)
                    {
                        MsgLog(4, "Warning: no more space in audio codec internal buffer! "
                                "(copied:%u bytes, decoded:%u bytes, trying to copy:%u bytes)",
                                poADecMod->uiCopiedBytes,
                                poADecMod->uiDecodedBytes,
                                poSrcBuffer->uiSizeBytes);
                        eStatus = LStatus_OK;
                    }
                }
            }
            else
            {
                ModLnkIn_ReturnBuffer(&(poADecMod->oInLink), poSrcBuffer, MNULL, NO_TAG);
            }
        }

        eStatus = LStatus_OK;

        // Decode as much data as possible.
        while(LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = AudioDecMod_DecodeBuffer(poADecMod, MFALSE);
        }

        if((eStatus == LStatus_OUT_OF_RESOURCES))
        {
            usleep(1000);
        }
    }

    MsgLog(2, "LAudioDecoder_EndStream...");

    MBOOL32 bFlushPendingData = (poADecMod->uiDecodedBuffers > 0) ? MFALSE : MTRUE;

    poADecMod->bIsEos = MTRUE;

    eStatus = LAudioDecoder_EndStream(
                  poADecMod->hDecoder,
                  bFlushPendingData);

    MsgLog(2, "LAudioDecoder_EndStream done. (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    if(!bFlushPendingData)
    {
        // Decode remaining data.
        MsgLog(2, "Stream ended, decoding remaining data...");

        while(AudioDecMod_DecodeBuffer(poADecMod, MFALSE) != LStatus_END_OF_STREAM)
        {
            usleep(1000);
        }

        MsgLog(2, "Decode remaining data done.");
    }

    MsgLog(2, "Total copied bytes: %lu, Total decoded bytes: %lu, Total decoded buffers: %u",
           poADecMod->uiCopiedBytes,
           poADecMod->uiDecodedBytes,
           poADecMod->uiDecodedBuffers);

    MsgLog(2, "Kill thread.");

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       AudioDecMod_CpuThreadOut

Description:    .

\************************************************************************************************************/
LStatus AudioDecMod_CpuThreadOut(void* pvData)
{
    if (pvData == MNULL)
    {
        MsgLogErr("ERROR! NULL data.");
        return LStatus_INVALID_PARAM;
    }

    AudioDecModule* poADecMod   = (AudioDecModule*)pvData;
    MBOOL32 bIsEos              = MFALSE;

    MCHAR8  szThreadName[16] = "";
    snprintf(szThreadName, sizeof(szThreadName), "%.11s-%s", poADecMod->szModuleName, "Out");

    ModThread_SetName(szThreadName);
    MsgLog(2, "Start thread %p.", pthread_self());

    while(!bIsEos)
    {
        BufferInfo* poDstBuffer = MNULL;

        LStatus eStatus = ModLnk_GetReturnedBuffer(
                            &(poADecMod->oOutLink),
                            100,
                            MNULL,
                            &poDstBuffer);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = poDstBuffer->bInternal ? LStatus_OK : LStatus_FAIL;

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                if(poDstBuffer->hBuffer != MNULL)
                {
                    MsgLog(4, "LAudioDecoder_ReleaseBuffer: Buffer[%u]", poDstBuffer->uiId);
                    eStatus = LAudioDecoder_ReleaseBuffer(poADecMod->hDecoder, poDstBuffer->hBuffer);

                    if(LSTATUS_IS_SUCCESS(eStatus))
                    {
                        MsgLog(4, "LAudioDecoder_ReleaseBuffer: Buffer[%u] done", poDstBuffer->uiId);
                    }
                    else
                    {
                        MsgLog(0, "ERROR, LAudioDecoder_ReleaseBuffer returned error %d", eStatus);
                    }

                    poDstBuffer->hBuffer = MNULL;
                }

                if(LSTATUS_IS_SUCCESS(eStatus))
                {
                    MUINT32 uiNbLostBuffers = 0;
                    MUINT64 uiTimestamp = 0;

                    LAudioCodec_BufferInfo oBufInfo
                            = { LAudioCodec_BufferInfoTypeHeader_STANDARD };

                    MsgLog(4, "LAudioDecoder_GetNextBuffer...");
                    eStatus = LAudioDecoder_GetNextBuffer(
                                  poADecMod->hDecoder,
                                  &(oBufInfo.eType),
                                  100,
                                  &uiTimestamp,
                                  &uiNbLostBuffers);

                    if(LSTATUS_IS_SUCCESS(eStatus))
                    {
                        MsgLog(4, "LAudioDecoder_GetNextBuffer done. (timestamp=%lu, nb lost buffers=%u)",
                               uiTimestamp, uiNbLostBuffers);

                        poDstBuffer->hBuffer = oBufInfo.hBuffer;
                        poDstBuffer->uiSizeBytes = oBufInfo.uiSize;
                        poDstBuffer->uiStartOffset = oBufInfo.uiStartOffset;
                        poDstBuffer->uiTimestampUsec = uiTimestamp;
                        poDstBuffer->uiSyncPtsUsec = uiTimestamp;
                    }
                    else
                    {
                        if(eStatus == LStatus_TIMEOUT)
                        {
                            MsgLog(1, "LAudioDecoder_GetNextBuffer timeout.");
                        }
                        else if(eStatus == LStatus_END_OF_STREAM)
                        {
                            MsgLog(2, "End of stream reached, stopping.");
                            poDstBuffer->bEndOfStream = MTRUE;
                            if(poADecMod->bIsEos)
                            {
                                bIsEos = MTRUE;
                            }
                            poADecMod->oCpuThreadOut.bKillThread = MTRUE;
                        }
                        else
                        {
                            MsgLog(0, "ERROR, LAudioDecoder_GetNextBuffer returned status %d", eStatus);
                        }
                    }
                }
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                ModLnk_SubmitBuffer(&(poADecMod->oOutLink), poDstBuffer, MNULL, NO_TAG);
            }
            else
            {
                ModLnk_ReleaseBuffer(&(poADecMod->oOutLink), poDstBuffer);
            }
        }

        if (LSTATUS_IS_FAIL(eStatus)
            && (eStatus != LStatus_TIMEOUT))
        {
            usleep(1000);
        }
    }

    MsgLog(2, "Kill thread.");

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       AudioDecMod_Start

Description:    .

\************************************************************************************************************/
LStatus AudioDecMod_Start(AudioDecModule* poADecMod)
{
    LStatus eStatus = LStatus_INVALID_PARAM;

    MsgLog(2, "{...");

    if ((poADecMod != MNULL)
        && (poADecMod->hDecoder != MNULL))
    {
        if ((poADecMod->oInLink.poModLnk != MNULL)
            && (poADecMod->oOutLink.uiSubmitCount > 0))
        {
            eStatus = LStatus_OK;
        }
        else
        {
            MsgLogErr("ERROR! Bad connection.");
            eStatus = LStatus_FAIL;
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = ModThread_Start(&(poADecMod->oCpuThreadOut), poADecMod, AudioDecMod_CpuThreadOut);

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                eStatus = ModThread_Start(&(poADecMod->oCpuThreadIn), poADecMod, AudioDecMod_CpuThreadIn);

                if (LSTATUS_IS_FAIL(eStatus))
                {
                    ModThread_Stop(&(poADecMod->oCpuThreadOut));
                }
            }
        }
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       AudioDecMod_Stop

Description:    .

\************************************************************************************************************/
void AudioDecMod_Stop(AudioDecModule* poADecMod)
{
    MsgLog(2, "{...");

    if (poADecMod != MNULL)
    {
        MsgLog(2, "Stop input thread...");
        ModThread_Stop(&(poADecMod->oCpuThreadIn));
        MsgLog(2, "Stop input thread done.");

        MsgLog(2, "Stop output thread...");
        ModThread_Stop(&(poADecMod->oCpuThreadOut));
        MsgLog(2, "Stop output thread done.");
    }

    MsgLog(2, "...}");
}
