/************************************************************************************************************\

Module Name:    AoutModule.c

Description:    .

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "AoutModule.h"
#include "CommonUtils.h"
#include "LBoard.h"
#include "math.h"

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

static const MCHAR8     g_szModuleNameBase[] = "Aout";

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------

LStatus AoutMod_InitProcessor(
    AoutModule*     poAout,
    LDevice_Handle  hDevice,
    LAudioFormat    eFormat,
    MUINT           uiFramesPerBuf,
    MBOOL           bDoClkComp,
    MBOOL           bQueueLvlClkComp);

/************************************************************************************************************\

Function:       AoutMod_Init

Description:    .

\************************************************************************************************************/
LStatus AoutMod_Init(
    AoutModule*         poAout,
    LDevice_Handle      hDevice,
    MUINT32             uiAoutIndex,
    LAudioFormat        eFormat,
    MUINT32             uiFramesPerBuffer,
    ModuleSyncMaster*   poModSyncMst,
    MBOOL               bDoClkComp,
    MBOOL               bQueueLvlClkComp)
{
    MsgLog(2, "{...");

    AoutMod_Cleanup(poAout);

    LStatus eStatus = ((poAout != MNULL)
                       && (hDevice != MNULL)
                       && (eFormat != LAudioFormat_INVALID)
                       && (LAudioFormat_GetSampleSize(eFormat) == 16)
                       && (uiFramesPerBuffer != 0))
                      ? LStatus_OK : LStatus_INVALID_PARAM;

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        snprintf(
                    poAout->szModuleName,
                    sizeof(poAout->szModuleName),
                    "%s%d",
                    g_szModuleNameBase,
                    uiAoutIndex);

        eStatus = LAudioOut_GetHandle(
                        hDevice,
                        uiAoutIndex,
                        LAccessMode_READWRITE_EXCLUSIVE,
                        &(poAout->hAout));

        if(eStatus != LStatus_OK)
        {
            MsgLog(2, "LAudioOut_GetHandle Failed\n");
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LAudioFormat    eSupportedFmt = LAudioFormat_INVALID;
        MBOOL           bSupported    = MFALSE;
        MUINT           i;

        for (i = 0; LSTATUS_IS_SUCCESS(eStatus); i++)
        {
            eStatus = LAudioOut_EnumSupportedAudioFormat(
                                    poAout->hAout,
                                    i,
                                    &eSupportedFmt);

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                if (eSupportedFmt == eFormat)
                {
                    bSupported = MTRUE;
                }

                MsgLog(2, "Supported audio format: %uBits - %uCh @ %uHz%s",
                       LAudioFormat_GetSampleSize(eSupportedFmt),
                       LAudioFormat_GetNumberOfChannel(eSupportedFmt),
                       LAudioFormat_GetSampleRate(eSupportedFmt),
                       (eSupportedFmt == eFormat) ? " --> Selected" : "");
            }
        }

        if(bSupported)
        {
            eStatus = LStatus_OK;
        }
        else
        {
            MsgLog(0, "ERROR: Audio format unsupported by audio output: %uHz - %u bits - %u ch\n",
                       LAudioFormat_GetSampleRate(eFormat),
                       LAudioFormat_GetSampleSize(eFormat),
                       LAudioFormat_GetNumberOfChannel(eFormat));

            eStatus = LStatus_FAIL;
        }
    }

    MUINT32 uiBufferSize = 0;

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LAudioFormat_ComputeBufferSizeBasedOnFrame(
                        eFormat,
                        uiFramesPerBuffer,
                        &uiBufferSize);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            MsgLog(2, "Audio format: 0x%08x, bufferSize: %u", eFormat, uiBufferSize);

            eStatus = LAudioOut_SetBufferAttributes(
                            poAout->hAout,
                            eFormat,
                            uiBufferSize);
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LBuffer_CreateLinear(hDevice, uiBufferSize, &poAout->hSilenceBuffer);

        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            MUINT8* pbyBuf = MNULL;

            eStatus = LBuffer_BeginAccess(poAout->hSilenceBuffer, 0, 1, &pbyBuf);

            if(LSTATUS_IS_SUCCESS(eStatus))
            {
                memset(pbyBuf, 0, uiBufferSize);

                LBuffer_EndAccess(poAout->hSilenceBuffer);
            }
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LBoard_GetTickReferenceFrequency(hDevice, &(poAout->uiTickFrequency));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        // Create an audio processor for audio output clock compensation and/or buffer size conversion.
        eStatus = AoutMod_InitProcessor(poAout, hDevice, eFormat, uiFramesPerBuffer, bDoClkComp, bQueueLvlClkComp);
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        poAout->hDevice                 = hDevice;
        poAout->eFormat                 = eFormat;
        poAout->uiFramesPerInputBuffer  = 0;
        poAout->uiFramesPerOutputBuffer = uiFramesPerBuffer;
        poAout->uiBufferSize            = uiBufferSize;
        poAout->poModSyncMst            = poModSyncMst;
        poAout->bDoClkComp              = bDoClkComp;
        poAout->bQueueLvlClkComp        = bQueueLvlClkComp;

        poAout->uiBytesPerSample        = LAudioFormat_GetNumberOfChannel(eFormat)
                                          * (((LAudioFormat_GetSampleSize(eFormat) + (7)) & (~7)) / 8);
    }
    else
    {
        AoutMod_Cleanup(poAout);
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       AoutMod_Cleanup

Description:    .

\************************************************************************************************************/
void AoutMod_Cleanup(AoutModule* poAout)
{
    if (poAout != MNULL)
    {
        if (poAout->hAout != MNULL)
        {
            LAudioOut_ReleaseHandle(poAout->hAout);
            poAout->hAout = MNULL;
        }

        if (poAout->hSilenceBuffer != MNULL)
        {
            LBuffer_Destroy(poAout->hSilenceBuffer);
            poAout->hSilenceBuffer = MNULL;
        }

        if(poAout->hProc != MNULL)
        {
            LAudioProc_Destroy(poAout->hProc);
            poAout->hProc = MNULL;
        }

        poAout->oInLink.poModLnk = MNULL;
        poAout->poModSyncMst = MNULL;
        poAout->uiBufferSize = 0;
    }
}

/************************************************************************************************************\

Function:       AoutMod_ReturnOutputBuffer

Description:    .

\************************************************************************************************************/
void AoutMod_ReturnOutputBuffer(AoutModule* poAout, BufferInfo* poBuffer)
{
    if(poAout->bUseProcessor)
    {
        LAudioProc_ReleaseBuffer(poAout->hProc, poBuffer->hBuffer);
    }
    else
    {
        ModLnkIn_ReturnBuffer(&(poAout->oInLink), poBuffer, MNULL, NO_TAG);
    }
}

/************************************************************************************************************\

Function:       AoutMod_DropOldestBuffer

Description:    .

\************************************************************************************************************/
MBOOL AoutMod_DropOldestBuffer(
    AoutModule* poAout)
{
    BufferInfo* poOldestBuffer  = MNULL;
    MBOOL       bDropped        = MFALSE;

    LStatus eStatus = ModLnkIn_GetSubmittedBuffer(
                        &(poAout->oInLink),
                        0,
                        0,
                        MNULL,
                        &poOldestBuffer,
                        MNULL,
                        MNULL);

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        ModLnkIn_ReturnBuffer(&(poAout->oInLink), poOldestBuffer, MNULL, NO_TAG);
        bDropped = MTRUE;
    }

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        // We dropped the oldest buffer, so update base time to the timestamp of the new oldest buffer.
        eStatus = ModLnkIn_InspectSubmittedBuffer(
                                    &(poAout->oInLink),
                                    &poOldestBuffer,
                                    MNULL);

        if(LSTATUS_IS_SUCCESS(eStatus)
            && (poAout->poModSyncMst != MNULL))
        {
            ModSyncMst_SetBaseTime(poAout->poModSyncMst, poOldestBuffer->uiSyncPtsUsec, MTRUE);
        }
    }

    return bDropped;
}

/************************************************************************************************************\

Function:       AoutMod_StartPlayback

Description:    Start playback with 2 silence buffers.

\************************************************************************************************************/
LStatus AoutMod_StartPlayback(
    AoutModule* poAout,
    MBOOL* pbQuit)
{
    LStatus eStatus = LAudioOut_SetBuffer(
                                    poAout->hAout,
                                    poAout->hSilenceBuffer,
                                    poAout->uiBufferSize,
                                    MFALSE);

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        MsgLog(2, "Starting playback.");
        eStatus = LStatus_FAIL;
        MUINT uiCount = 0;

        while(LSTATUS_IS_FAIL(eStatus) && !(*pbQuit))
        {
            eStatus = LAudioOut_StartPlayback(poAout->hAout);

            if(LSTATUS_IS_FAIL(eStatus))
            {
                usleep(1000);
                if((++uiCount % 50) == 0)
                {
                    MsgLog(1, "Cannot start audio playback (%d - %s) Retrying...", eStatus, GetStatusStr(eStatus));
                }

                // Keep the queue empty to avoid stalling upstream buffers.
                while(AoutMod_DropOldestBuffer(poAout));
            }
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       AoutMod_ProcessInput

Description:    .

\************************************************************************************************************/
LStatus AoutMod_ProcessInput(
    AoutModule* poAout,
    BufferInfo** ppoSrcBuffer,
    LAudioProc_BufferDesc* poBufDesc)
{
    LStatus eStatus = LStatus_OK;

    BufferInfo* poSrcBuffer = *ppoSrcBuffer;

    if(poBufDesc->hBuffer == MNULL)
    {
        poBufDesc->hBuffer        = poSrcBuffer->hBuffer;
        poBufDesc->uiSize         = poSrcBuffer->uiSizeBytes;
        poBufDesc->uiStartOffset  = poSrcBuffer->uiStartOffset;
        poAout->uiLastInputTs     = poSrcBuffer->uiSyncPtsUsec;
    }

    MsgLog(4, "LAudioProc_ProcessInputStream(Buffer[%u])...", poSrcBuffer->uiId);

    MUINT32 uiConsumedSize = 0;
    eStatus = LAudioProc_ProcessInputStream(
                    poAout->hProc,
                    0,
                    &(poBufDesc->eType),
                    poAout->uiLastInputTs,
                    &uiConsumedSize);

    MsgLog(4, "LAudioProc_ProcessInputStream done. (status= %d - %s).", eStatus, GetStatusStr(eStatus));

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        poBufDesc->uiSize -= uiConsumedSize;
        poBufDesc->uiStartOffset += uiConsumedSize;

        if(poBufDesc->uiSize == 0)
        {
            // Input buffer is fully copied inside audio processor, return buffer.
            ModLnkIn_ReturnBuffer(
                    &(poAout->oInLink),
                    poSrcBuffer,
                    MNULL,
                    NO_TAG);

            *ppoSrcBuffer = MNULL;
            poBufDesc->hBuffer = MNULL;
        }

        // Adjust next timestamp if we didn't consume all the data.
        if(poBufDesc->uiSize > 0)
        {
            MUINT64 uiConsumedSamples = uiConsumedSize / poAout->uiBytesPerSample;
            poAout->uiLastInputTs += (uiConsumedSamples * 1000lu * 1000lu)
                                       / LAudioFormat_GetSampleRate(poAout->eFormat);
        }
    }
    else if(eStatus == LStatus_OUT_OF_RESOURCES)
    {
        MsgLog(1, "Warning: LAudioProcessor internal buffer full.");
        eStatus = LStatus_OK;
    }
    else
    {
        MsgLogErr("Error in LAudioProc_ProcessInputStream: %d (%s)", eStatus, GetStatusStr(eStatus));
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       AoutMod_GetProcessedBuffer

Description:    .

\************************************************************************************************************/
LStatus AoutMod_GetProcessedBuffer(
    AoutModule* poAout,
    BufferInfo* poProcessedBuffer)
{
    // Process an output buffer.
    LStatus eStatus = LAudioProc_ProcessOutputStream(poAout->hProc, poAout->uiFramesPerOutputBuffer);

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        // Get processed output buffer.
        LAudioProc_BufferDesc oBufferDesc = {LAudioProc_BufferDescTypeHeader_STANDARD};

        MUINT64 uiTimestampUsec = 0;
        eStatus = LAudioProc_GetNextBuffer(poAout->hProc, 0, &(oBufferDesc.eType), &uiTimestampUsec);

        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            poProcessedBuffer->hBuffer        = oBufferDesc.hBuffer;
            poProcessedBuffer->uiSizeBytes    = oBufferDesc.uiSize;
            poProcessedBuffer->uiSyncPtsUsec  = uiTimestampUsec;
        }
        else
        {
            MsgLog(2, "Warning: can't get processed buffer (%d)", eStatus);
        }
    }
    else if(eStatus == LStatus_NO_MORE_DATA)
    {
        MsgLog(4, "Warning: LAudioProcessor - no more data");
    }
    else
    {
        MsgLogErr("ERROR! LAudioProc_ProcessOutputStream returned %d (%s).", eStatus, GetStatusStr(eStatus));
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       AoutMod_GetClkCompLevel

Description:    .

\************************************************************************************************************/
MFLOAT64 AoutMod_GetClkCompLevel(MFLOAT64 fError)
{
    MFLOAT64 fLevel     = 0.0;
    MFLOAT64 fAbsError  = fabs(fError);

    if(fAbsError > 1.0)
    {
        if(fError > 0.0)
        {
            fLevel = -1.0;
        }
        else
        {
            fLevel = 1.0;
        }
    }
    else if(fAbsError > 0.5)
    {
        if(fError > 0.0)
        {
            fLevel = -0.5;
        }
        else
        {
            fLevel = 0.5;
        }
    }
    else if(fAbsError > 0.25)
    {
        if(fError > 0.0)
        {
            fLevel = -0.25;
        }
        else
        {
            fLevel = 0.25;
        }
    }
    else if(fAbsError > 0.125)
    {
        if(fError > 0.0)
        {
            fLevel = -0.125;
        }
        else
        {
            fLevel = 0.125;
        }
    }

    return fLevel;
}

/************************************************************************************************************\

Function:       AoutMod_UpdateProcessor

Description:    .

\************************************************************************************************************/
void AoutMod_UpdateProcessor(
    AoutModule* poAout,
    MUINT64     uiTickCount,
    MUINT32     uiInitialBufferQueued,
    MUINT32     uiQueueLevel,
    MUINT32     uiUnprocessedRingBufSize,
    MUINT32     uiProcessedRingBufSize)
{
    // Update timestamp for clock compensation calculation.
    if(poAout->bDoClkComp)
    {
        if(poAout->bQueueLvlClkComp)
        {
            MUINT uiBufferSize = poAout->uiFramesPerInputBuffer * poAout->uiBytesPerSample;

            // Total level of buffering: (input queue) + (unprocessed data ring buffer) + (processed data ring buffer)
            MUINT uiTotalBuffering = (uiQueueLevel * uiBufferSize)
                                         + (uiUnprocessedRingBufSize + uiProcessedRingBufSize);

            MFLOAT64 fInitialBufferQueued = uiInitialBufferQueued;
            MFLOAT64 fCurrentBufferQueued = (MFLOAT64) uiTotalBuffering / uiBufferSize;

            poAout->fTotalBufferingCummul += fCurrentBufferQueued;
            poAout->uiBufferingAvgCnt++;

            if(poAout->uiBufferingAvgCnt == 10)
            {
                MFLOAT64 fAvgBufferQueued   = poAout->fTotalBufferingCummul / poAout->uiBufferingAvgCnt;
                MFLOAT64 fError             = fAvgBufferQueued - fInitialBufferQueued;
                MFLOAT64 fCtrlLevel         = AoutMod_GetClkCompLevel(fError);

                if(fCtrlLevel != poAout->fDirectCtrlLvl)
                {
                    // Update direct clock compensation control level.
                    LAudioProc_ClockCompensationOpt oOpt;
                    oOpt.eType          = LAudioProc_ProcessTypeHeader_CLK_COMPENSATION;
                    oOpt.fDirectCtrlLvl = fCtrlLevel;
                    oOpt.eControlMode   = LAudioProc_ClkCompCtrlMode_DIRECT_CTRL;

                    LStatus eStatus = LAudioProc_SetDynamicProcessConfig(poAout->hProc, 1, &(oOpt.eType));

                    if(LSTATUS_IS_SUCCESS(eStatus))
                    {
                        poAout->fDirectCtrlLvl = fCtrlLevel;
                    }
                    else
                    {
                        MsgLog(0, "Error, failed to set clock compensation level to %f (%d - %s)",
                                  fCtrlLevel, eStatus, GetStatusStr(eStatus));
                    }
                }

                poAout->fTotalBufferingCummul = 0.0;
                poAout->uiBufferingAvgCnt = 0;
            }
        }

        LAudioProc_UpdateTimeStamp(poAout->hProc, 0, uiTickCount);
    }
}

/************************************************************************************************************\

Function:       GetEventString

Description:    .

\************************************************************************************************************/
char* GetEventString(MFLAG32 flEvents, LSTR256* pszEvents)
{
    char*  szEvents = (char*) pszEvents;
    size_t uiSize   = sizeof(*pszEvents);

    struct
    {
        LAudioOut_Event eEnum;
        const char*     szName;

    } aoEvents[5] =
    {
        {LAudioOut_Event_START_PENDING, "start-pending"},
        {LAudioOut_Event_STARTED,       "started"},
        {LAudioOut_Event_STATE_CHANGED, "state-changed"},
        {LAudioOut_Event_OUT_OF_BUFFER, "out-of-buffers"},
        {LAudioOut_Event_STOPPED,       "stopped"}
    };

    if (flEvents > 0)
    {
        memset(szEvents, 0, uiSize);

        MUINT i;
        MBOOL bAddPrefix = MFALSE;

        for (i = 0; i < sizeof(aoEvents) / sizeof(aoEvents[0]); i++)
        {
            if (flEvents & aoEvents[i].eEnum)
            {
                if (bAddPrefix)
                {
                    strncat_wz(szEvents, "/", uiSize);
                }

                strncat_wz(szEvents, aoEvents[i].szName, uiSize);

                bAddPrefix = MTRUE;
            }
        }
    }
    else
    {
        strncpy_wz(szEvents, "none", uiSize);
    }

    return szEvents;
}

/************************************************************************************************************\

Function:       AoutMod_EndInit

Description:    .

\************************************************************************************************************/
void AoutMod_EndInit(AoutModule* poAout, BufferInfo* poSrcBuffer)
{
    poAout->uiFramesPerInputBuffer = poSrcBuffer->uiSizeBytes / poAout->uiBytesPerSample;

    if((poAout->uiFramesPerInputBuffer != poAout->uiFramesPerOutputBuffer)
        || (poAout->bDoClkComp))
    {
        // Use processor when clock compensation is enabled or when buffer size conversion is needed.
        poAout->bUseProcessor = MTRUE;
    }

    if(poAout->poModSyncMst != MNULL)
    {
        ModSyncMst_SetParams(
                poAout->poModSyncMst,
                ((MUINT64)(poAout->uiFramesPerInputBuffer) * 1000lu * 1000lu)
                    / LAudioFormat_GetSampleRate(poAout->eFormat),
                poAout->oInLink.poModLnk->uiBufferCount,
                0,
                MTRUE);
    }

    if(poAout->bUseProcessor)
    {
        MsgLog(2, "Initialization done: using audio processor (Buffer %u->%u, ClkComp=%s)",
                  poAout->uiFramesPerInputBuffer,
                  poAout->uiFramesPerOutputBuffer,
                  poAout->bDoClkComp ? "yes" : "no");
    }
    else
    {
        MsgLog(2, "Initialization done: not using audio processor.");
    }
}

/************************************************************************************************************\

Function:       AoutMod_CpuThread

Description:    .

\************************************************************************************************************/
LStatus AoutMod_CpuThread(void* pvData)
{
    if (pvData == MNULL)
    {
        MsgLogErr("ERROR! NULL data.");
        return LStatus_INVALID_PARAM;
    }

    AoutModule*         poAout              = (AoutModule*) pvData;
    ModuleSyncMaster*   poSyncMst           = poAout->poModSyncMst;
    MBOOL               bInitDone           = MFALSE;
    BufferInfo*         poSrcBuffer         = MNULL;

    LAudioProc_BufferDesc oBufDesc          = {LAudioProc_BufferDescTypeHeader_STANDARD};
    oBufDesc.hBuffer                        = MNULL;

    MUINT64         uiNbSec                 = 0;
    MUINT64         uiStartTimeUsec         = GetMonoTimeUsec();
    MUINT           uiInitialQueueSize      = 0;
    MFLAG32         flEvents                = 0;
    LSTR256         szEvents                = {0};
    MUINT           uiOutOfBufCnt           = 0;

    BufferInfo      oPlaybackBuffer         = {0};
    BufferInfo*     poPlaybackBuffer        = MNULL;
    BufferInfo      oNewPlaybackBuffer      = {0};
    BufferInfo*     poNewPlaybackBuffer     = MNULL;

    enum
    {
        State_PreSync   = 0,
        State_Ready,
        State_Playing,
        State_Failsafe,

    } eState;

    eState = (poSyncMst != MNULL) ? State_PreSync : State_Ready;

    ModThread_SetName(poAout->szModuleName);
    MsgLog(2, "Start thread %p.", pthread_self());

    LStatus eStatus = AoutMod_StartPlayback(
                                poAout,
                                &poAout->oCpuThread.bKillThread);

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        MsgLog(2, "Audio playback started.");
    }
    else
    {
        MsgLog(0, "ERROR: Failed to start audio playback (%d - %s)", eStatus, GetStatusStr(eStatus));
    }

    while (!poAout->oCpuThread.bKillThread)
    {
        MUINT uiQueueSize = 0;
        memset(&oNewPlaybackBuffer, 0, sizeof(oNewPlaybackBuffer));

        if(eState > State_PreSync)
        {
            eStatus = LStatus_OK;

            // 1 - Get input buffer when needed.
            if(poSrcBuffer == MNULL)
            {
                eStatus = ModLnkIn_GetSubmittedBuffer(
                                &(poAout->oInLink),
                                0,
                                0,
                                MNULL,
                                &poSrcBuffer,
                                MNULL,
                                &uiQueueSize);
            }
        }
        else
        {
            // Not ready to start playing yet, pre-sync with slave stream.

            eStatus = ModLnkIn_InspectSubmittedBuffer(
                                        &(poAout->oInLink),
                                        &poSrcBuffer,
                                        &uiQueueSize);

            if(LSTATUS_IS_SUCCESS(eStatus))
            {
                // End initialization once we received the first buffer.
                if(!bInitDone)
                {
                    AoutMod_EndInit(poAout, poSrcBuffer);

                    if(poAout->bUseProcessor)
                    {
                        poPlaybackBuffer    = &oPlaybackBuffer;
                        poNewPlaybackBuffer = &oNewPlaybackBuffer;
                    }

                    bInitDone = MTRUE;
                }

                // Lock must be kept until new base time is updated when oldest buffer is dropped.
                ModSyncMst_Lock(poSyncMst);

                MBOOL bDropOldest = MFALSE;
                MBOOL bReadyToPlay = MFALSE;

                ModSyncMst_PreSync(
                            poSyncMst,
                            uiQueueSize,
                            poSrcBuffer->uiSyncPtsUsec,
                            MTRUE,
                            &bReadyToPlay,
                            &bDropOldest);

                if(bDropOldest)
                {
                    AoutMod_DropOldestBuffer(poAout);
                }

                ModSyncMst_Unlock(poSyncMst);

                if(bReadyToPlay)
                {
                    uiInitialQueueSize = uiQueueSize;
                    eState = State_Ready;
                }
            }

            eStatus = LStatus_FAIL;
            poSrcBuffer = MNULL;
        }

        // 2 - Process input buffer.
        if(LSTATUS_IS_SUCCESS(eStatus) && (poSrcBuffer != MNULL))
        {
            // End initialization once we received the first buffer.
            if(!bInitDone)
            {
                AoutMod_EndInit(poAout, poSrcBuffer);

                if(poAout->bUseProcessor)
                {
                    poPlaybackBuffer    = &oPlaybackBuffer;
                    poNewPlaybackBuffer = &oNewPlaybackBuffer;
                }

                bInitDone = MTRUE;
            }

            if(poSrcBuffer->uiSizeBytes != (poAout->uiFramesPerInputBuffer * poAout->uiBytesPerSample))
            {
                MsgLog(0, "Warning: got buffer with size %u instead of %u",
                          poSrcBuffer->uiSizeBytes, poAout->uiFramesPerInputBuffer * poAout->uiBytesPerSample);
            }

            if (poSrcBuffer->bEndOfStream)
            {
                MsgLog(4, "END-OF-STREAM");

                poAout->oCpuThread.bKillThread = MTRUE;
                eStatus = LStatus_END_OF_STREAM;
            }
            else if (poAout->bUseProcessor)
            {
                eStatus = AoutMod_ProcessInput(poAout, &poSrcBuffer, &oBufDesc);
            }
            else
            {
                // Processor bypassed, directly playback input buffer.
                poNewPlaybackBuffer = poSrcBuffer;
                poSrcBuffer         = MNULL;
            }
        }

        // 3 - Process new playback buffer if processor is enabled.
        //     Note: don't check status here, the LAudioProcessor might hold enough internal
        //     data to create a new buffer without any new input data.
        if(poAout->bUseProcessor && (eState > State_PreSync))
        {
            eStatus = AoutMod_GetProcessedBuffer(poAout, poNewPlaybackBuffer);
        }

        // 4 - Gather events.
        LAudioOut_Event eEvent = LAudioOut_Event_INVALID;
        while (LSTATUS_IS_SUCCESS(LAudioOut_WaitForEvent(poAout->hAout, 0, &eEvent)))
        {
            flEvents |= eEvent;

            if (eEvent == LAudioOut_Event_OUT_OF_BUFFER)
            {
                uiOutOfBufCnt++;
            }
        }

        // 5 - Playback new buffer.
        MUINT64 uiTickCount = 0;
        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            MsgLog(4, "LAudioOut_SetBuffer(Buffer[%p])...", poNewPlaybackBuffer->hBuffer);

            eStatus = LAudioOut_SetBuffer(
                            poAout->hAout,
                            poNewPlaybackBuffer->hBuffer,
                            poNewPlaybackBuffer->uiSizeBytes,
                            MFALSE);

            MsgLog(4, "LAudioOut_SetBuffer done. (status= %d - %s)", eStatus, GetStatusStr(eStatus));

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                eStatus = LAudioOut_WaitForBufferChange(
                                            poAout->hAout,
                                            1000,
                                            MNULL,
                                            &uiTickCount);

                if(LSTATUS_IS_SUCCESS(eStatus) && (poAout->poModSyncMst != MNULL))
                {
                    // Update timestamp for synchronization.
                    ModSyncMst_UpdateTimestamp(poAout->poModSyncMst, poNewPlaybackBuffer->uiSyncPtsUsec, MTRUE);
                }

                if (eState == State_Failsafe)
                {
                    MsgLog(2, "Exit failsafe mode.");
                }

                eState = State_Playing;
                uiOutOfBufCnt = 0;
            }
            else
            {
                MsgLogErr("ERROR! LAudioOut_SetBuffer returned %d (%s).", eStatus, GetStatusStr(eStatus));
                if(poNewPlaybackBuffer != MNULL)
                {
                    AoutMod_ReturnOutputBuffer(poAout, poNewPlaybackBuffer);
                }
            }
        }
        else if (((eState == State_Ready) || (eState == State_Playing))
                  && (uiOutOfBufCnt >= 2))
        {
            // When we have no source buffer and we detected at least two out-of-buffer
            // events, enter a failsafe mode where silence buffers are played until we have
            // a new source buffer. This is to avoid looping on the last two buffers in case
            // input stops receiving data.

            MsgLog(2, "Enter failsafe mode.");

            MUINT i;
            for (i = 0; i < 2; i++)
            {
                LAudioOut_SetBuffer(
                            poAout->hAout,
                            poAout->hSilenceBuffer,
                            poAout->uiBufferSize,
                            MTRUE);
            }

            eState = State_Failsafe;
        }

        MUINT32 uiUnprocessedBufSize = 0;
        MUINT32 uiProcessedBufSize = 0;
        MUINT32 uiUnprocessedFilledSize = 0;
        MUINT32 uiProcessedFilledSize = 0;

        // 6 - Gather internal buffer fullness information
        if(poAout->bDoClkComp)
        {
            LAudioProc_InternalBufferInfo oBufInfo = {LAudioProc_InfoTypeHeader_INTERNAL_BUFFER_INFO};
            oBufInfo.uiInputStreamIdx = 0;
            LAudioProc_GetInfo(poAout->hProc, &(oBufInfo.eType));

            uiUnprocessedBufSize    = oBufInfo.uiUnprocessedDataBufTotSize;
            uiProcessedBufSize      = oBufInfo.uiProcessedDataBufTotSize;
            uiUnprocessedFilledSize = uiUnprocessedBufSize  - oBufInfo.uiUnprocessedDataBufFreeSize;
            uiProcessedFilledSize   = uiProcessedBufSize    - oBufInfo.uiProcessedDataBufFreeSize;
        }

        // 7 - Return previously playing buffer and update timestamp.
        if(LSTATUS_IS_SUCCESS(eStatus)
            && (poPlaybackBuffer != MNULL)
            && (poPlaybackBuffer->hBuffer != MNULL))
        {
            AoutMod_ReturnOutputBuffer(poAout, poPlaybackBuffer);

            AoutMod_UpdateProcessor(
                        poAout,
                        uiTickCount,
                        uiInitialQueueSize,
                        uiQueueSize,
                        uiUnprocessedFilledSize,
                        uiProcessedFilledSize);
        }

        // 8 - Store currently playing buffer.
        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            if(poAout->bUseProcessor)
            {
                oPlaybackBuffer = oNewPlaybackBuffer;
            }
            else
            {
                poPlaybackBuffer = poNewPlaybackBuffer;
                poNewPlaybackBuffer = MNULL;
            }
        }

        // 9 - Print statistics.
        if(GetMonoTimeUsec() >= (uiStartTimeUsec + (uiNbSec * 1000lu * 1000lu)))
        {
            uiNbSec++;

            MUINT64 uiHour = (uiNbSec / 3600);
            MUINT64 uiMin  = (uiNbSec / 60  )                   - (uiHour * 60);
            MUINT64 uiSec  = (uiNbSec       )   - (uiMin * 60)  - (uiHour * 3600);

            if(poAout->bDoClkComp)
            {
                LAudioProc_StreamState oStreamState = {LAudioProc_InfoTypeHeader_STREAM_STATE};
                // Clock compensation on output stream
                oStreamState.uiStreamIdx = 1;
                LAudioProc_GetInfo(poAout->hProc, &(oStreamState.eType));

                if(poAout->bQueueLvlClkComp)
                {
                    MsgLog(3, "[%03lu:%02lu:%02lu]: Queue[%u], Samples[%lu -> %lu], DriftPPM[%s%0.1f/%s%0.1f], "
                              "Buf[%0.1f%%, %0.1f%%], ClkComp[%0.3f]%s%s.",
                              uiHour, uiMin, uiSec,
                              uiQueueSize,
                              oStreamState.uiTotConsumedSamples,
                              oStreamState.uiTotProcessedSamples,
                              (oStreamState.fClockDrift > 1.0) ? "+" : "",
                              (oStreamState.fClockDrift - 1.0) * 1000000,
                              ((oStreamState.fClockDrift / oStreamState.fClockDriftFromTarget) > 1.0) ? "+" : "",
                              ((oStreamState.fClockDrift / oStreamState.fClockDriftFromTarget) - 1.0) * 1000000,
                              ((MFLOAT64)uiUnprocessedFilledSize / uiUnprocessedBufSize) * 100.0,
                              ((MFLOAT64)uiProcessedFilledSize / uiProcessedBufSize) * 100.0,
                              poAout->fDirectCtrlLvl,
                              (flEvents != 0) ? ", Event:" : "",
                              (flEvents != 0) ? GetEventString(flEvents, &szEvents) : "");
                }
                else
                {
                    MsgLog(3, "[%03lu:%02lu:%02lu]: Queue[%u], Samples[%lu -> %lu], DriftPPM[%s%0.3f/%s%0.3f -> %s%0.3f], "
                              "Buf[%0.1f%%, %0.1f%%]%s%s.",
                              uiHour, uiMin, uiSec,
                              uiQueueSize,
                              oStreamState.uiTotConsumedSamples,
                              oStreamState.uiTotProcessedSamples,
                              (oStreamState.fClockDrift > 1.0) ? "+" : "",
                              (oStreamState.fClockDrift - 1.0) * 1000000,
                              ((oStreamState.fClockDrift / oStreamState.fClockDriftFromTarget) > 1.0) ? "+" : "",
                              ((oStreamState.fClockDrift / oStreamState.fClockDriftFromTarget) - 1.0) * 1000000,
                              (oStreamState.fClockDriftFromTarget > 1.0) ? "+" : "",
                              (oStreamState.fClockDriftFromTarget - 1.0) * 1000000,
                              ((MFLOAT64)uiUnprocessedFilledSize / uiUnprocessedBufSize) * 100.0,
                              ((MFLOAT64)uiProcessedFilledSize / uiProcessedBufSize) * 100.0,
                              (flEvents != 0) ? ", Event:" : "",
                              (flEvents != 0) ? GetEventString(flEvents, &szEvents) : "");
                }
            }
            else
            {
                MsgLog(3, "[%03lu:%02lu:%02lu]: Queue[%u]%s%s",
                          uiHour, uiMin, uiSec, uiQueueSize,
                          (flEvents != 0) ? ", Event:" : "",
                          (flEvents != 0) ? GetEventString(flEvents, &szEvents) : "");
            }

            flEvents = 0;
        }

        if (LSTATUS_IS_FAIL(eStatus)
            && (eStatus != LStatus_TIMEOUT))
        {
            usleep(1000);
        }
    }

    LAudioOut_StopPlayback(poAout->hAout);

    if((poPlaybackBuffer != MNULL) && (poPlaybackBuffer->hBuffer != MNULL))
    {
        AoutMod_ReturnOutputBuffer(poAout, poPlaybackBuffer);
    }

    if(!poAout->bUseProcessor && (poSrcBuffer == MNULL))
    {
        poSrcBuffer = poNewPlaybackBuffer;
    }

    if(poSrcBuffer != MNULL)
    {
        ModLnkIn_ReturnBuffer(&(poAout->oInLink), poSrcBuffer, MNULL, NO_TAG);
        poSrcBuffer = MNULL;
    }

    MsgLog(2, "Kill thread.");

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       AoutMod_Start

Description:    .

\************************************************************************************************************/
LStatus AoutMod_Start(AoutModule* poAout)
{
    LStatus eStatus = LStatus_INVALID_PARAM;

    MsgLog(2, "{...");

    if ((poAout != MNULL)
        && (poAout->hAout != MNULL))
    {
        eStatus = ModThread_Start(&(poAout->oCpuThread), poAout, AoutMod_CpuThread);
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       AoutMod_Stop

Description:    .

\************************************************************************************************************/
void AoutMod_Stop(AoutModule* poAout)
{
    MsgLog(2, "{...");

    if (poAout != MNULL)
    {
        ModThread_Stop(&(poAout->oCpuThread));
    }

    ModLnkIn_CancelSubmittedBuffers(&(poAout->oInLink));

    MsgLog(2, "...}");
}

/************************************************************************************************************\

Function:       AoutMod_InitProcessor

Description:    .

\************************************************************************************************************/
LStatus AoutMod_InitProcessor(
    AoutModule*     poAout,
    LDevice_Handle  hDevice,
    LAudioFormat    eFormat,
    MUINT           uiFramesPerBuf,
    MBOOL           bDoClkComp,
    MBOOL           bQueueLvlClkComp)
{
    LStatus eStatus = LStatus_INVALID_PARAM;

    if(poAout != MNULL)
    {
        // Input stream descriptor.
        LAudioProc_Stream oInputStream          = {LAudioProc_StreamTypeHeader_STANDARD};
        oInputStream.apeStreamProcesses         = MNULL;
        oInputStream.uiNbProcesses              = 0;
        oInputStream.eFormat                    = eFormat;
        oInputStream.uiTickReferenceFreq        = 1000000;  // Timestamps in usec.

        LAudioProc_StreamTypeHeader* apeInputStreams[1] = {&(oInputStream.eType)};

        // Output stream clock compensation process options.
        LAudioProc_ExtTimeRefDesc oExtTimeRef   = {LAudioProc_TimeRefTypeHeader_EXTERNAL};
        oExtTimeRef.uiId                        = 0;
        oExtTimeRef.uiTickReferenceFreq         = poAout->uiTickFrequency;  // Timestamps in ticks.
        oExtTimeRef.fNominalUpdatePeriod        = ((MFLOAT64) uiFramesPerBuf / LAudioFormat_GetSampleRate(eFormat))
                                                    * oExtTimeRef.uiTickReferenceFreq;

        LAudioProc_ClockCompensationOpt oClockCompOpt = {LAudioProc_ProcessTypeHeader_CLK_COMPENSATION};
        oClockCompOpt.uiMinCompensationInterval = 256;
        oClockCompOpt.peTimeRefDesc             = &(oExtTimeRef.eType);
        oClockCompOpt.fDirectCtrlLvl            = 0.0;
        oClockCompOpt.eControlMode              = bQueueLvlClkComp
                                                    ? LAudioProc_ClkCompCtrlMode_DIRECT_CTRL
                                                    : LAudioProc_ClkCompCtrlMode_TS_DRIFT;

        LAudioProc_ProcessTypeHeader* apeProcesses[1] = {&(oClockCompOpt.eType)};

        // Output stream descriptor.
        LAudioProc_Stream oOutputStream         = {LAudioProc_StreamTypeHeader_STANDARD};
        oOutputStream.apeStreamProcesses        = bDoClkComp ? apeProcesses : MNULL;
        oOutputStream.uiNbProcesses             = bDoClkComp ? 1            : 0;
        oOutputStream.eFormat                   = eFormat;
        oOutputStream.uiTickReferenceFreq       = 1000000;  // Timestamps in usec.

        // Create options.
        LAudioProc_CreateOpt oOpt               = {LAudioProc_CreateOptTypeHeader_STANDARD};
        oOpt.hDevice                            = hDevice;
        oOpt.apeInputStreams                    = apeInputStreams;
        oOpt.uiNbInputStreams                   = 1;
        oOpt.peOutputStream                     = &(oOutputStream.eType);
        oOpt.uiNbOutputBuffers                  = 2;
        oOpt.eOutputBufferType                  = LBuffer_Type_LINEAR;
        oOpt.peOutputTsTimeRef                  = MNULL;    // Use input stream to generate output timestamps.
        oOpt.uiOutputBufferSize                 = uiFramesPerBuf
                                                    * LAudioFormat_GetNumberOfChannel(eFormat)
                                                    * (LAudioFormat_GetSampleSize(eFormat) / 8);

        eStatus = LAudioProc_Create(&(oOpt.eType), &(poAout->hProc));
    }

    return eStatus;
}
