/************************************************************************************************************\

Module Name:    AinAlsaModule.c

Description:    .

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "AinAlsaModule.h"

#ifdef ALSA_SUPPORT
#include "alsa/asoundlib.h"
#endif

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------
#ifdef ALSA_SUPPORT

#define MAX_ALSA_DEVICES 32
static LSTR64  g_aszDevNames[MAX_ALSA_DEVICES] = {0};
static MUINT32 g_uiNbDev = 0;

/************************************************************************************************************\

Function:       AinAlsaMod_EnumDevices

Description:    .

\************************************************************************************************************/
LStatus AinAlsaMod_EnumDevices(MUINT32* puiNbDevices)
{
    LStatus eStatus = EnumAlsaDeviceNames(
                                g_aszDevNames,
                                MAX_ALSA_DEVICES,
                                &g_uiNbDev,
                                MTRUE);

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        *puiNbDevices = g_uiNbDev;
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       AinAlsaMod_GetDeviceName

Description:    .

\************************************************************************************************************/
LStatus AinAlsaMod_GetDeviceName(MUINT32 uiDeviceIdx, LSTR64* pszName)
{
    LStatus eStatus = LStatus_INVALID_PARAM;

    if ((uiDeviceIdx < g_uiNbDev)
        && (pszName != MNULL))
    {
        strcpy(*pszName, g_aszDevNames[uiDeviceIdx]);
        eStatus = LStatus_OK;
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       AinAlsaMod_Init

Description:    .

\************************************************************************************************************/
LStatus AinAlsaMod_Init(
    AinAlsaModule*  poAin,
    LDevice_Handle  hDevice,
    MUINT32         uiBufferCount,
    MUINT32         uiDevIdx,
    MUINT32         uiSamplingRate,
    MUINT32         uiNbChannels,
    MUINT32         uiSampleSize,
    MBOOL32         bBigEndian,
    MBOOL32         bUnsigned,
    MUINT32         uiFramesPerBuffer)
{
    MsgLog(2, "{...");

    AinAlsaMod_Cleanup(poAin);

    LStatus eStatus = LStatus_INVALID_PARAM;

    if((poAin != MNULL)
        && (uiDevIdx < g_uiNbDev)
        && (uiFramesPerBuffer > 0))
    {
        poAin->uiSamplingRate      = uiSamplingRate;
        poAin->uiNbChannels        = uiNbChannels;
        poAin->uiFramesPerBuffer   = uiFramesPerBuffer;
        poAin->uiSampleSize        = uiSampleSize;
        poAin->bBigEndian          = bBigEndian;
        poAin->bUnsigned           = bUnsigned;
        poAin->uiDevIdx            = uiDevIdx;
        eStatus = LStatus_OK;
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        int iErr = 0;
        if((iErr = snd_pcm_open(
                        (snd_pcm_t**)&poAin->hAlsaDev,
                        g_aszDevNames[uiDevIdx],
                        SND_PCM_STREAM_CAPTURE,
                        0)) < 0)
        {
            MsgLogErr("Error, snd_pcm_open(%s) failed: %s",
                      g_aszDevNames[uiDevIdx], snd_strerror(iErr));
            eStatus = LStatus_FAIL;
        }
    }

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = ConfigureAlsaDevice(
                    poAin->hAlsaDev,
                    poAin->uiSamplingRate,
                    poAin->uiNbChannels,
                    poAin->uiSampleSize,
                    poAin->bBigEndian,
                    poAin->bUnsigned,
                    poAin->uiFramesPerBuffer);

        if(LSTATUS_IS_FAIL(eStatus))
        {
            MsgLogErr("Error, can't configure ALSA device");
        }
    }

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        LBuffer_Attributes oBufferAttribute = {LBuffer_Type_SYSTEM_LINEAR};
        oBufferAttribute.oSystemLinearAttributes.pvSystemMemory = MNULL;
        oBufferAttribute.oSystemLinearAttributes.uiSize = snd_pcm_frames_to_bytes(
                                                              poAin->hAlsaDev,
                                                              poAin->uiFramesPerBuffer);

        eStatus = ModLnk_Init(
                    &(poAin->oOutLink),
                    hDevice,
                    uiBufferCount,
                    &oBufferAttribute,
                    MFALSE,
                    0,
                    g_aszDevNames[uiDevIdx]);
    }

    if (LSTATUS_IS_FAIL(eStatus))
    {
        AinAlsaMod_Cleanup(poAin);
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       AinAlsaMod_Cleanup

Description:    .

\************************************************************************************************************/
void AinAlsaMod_Cleanup(AinAlsaModule* poAin)
{
    if (poAin != MNULL)
    {
        if (poAin->hAlsaDev != MNULL)
        {
            snd_pcm_close(poAin->hAlsaDev);
            poAin->hAlsaDev = MNULL;
        }

        ModLnk_Cleanup(&poAin->oOutLink);
    }
}

/************************************************************************************************************\

Function:       AinAlsaMod_CpuThread

Description:    .

\************************************************************************************************************/
LStatus AinAlsaMod_CpuThread(void* pvData)
{
    if(pvData == MNULL)
    {
        MsgLogErr("ERROR! NULL data.");
        return LStatus_INVALID_PARAM;
    }

    AinAlsaModule* poAin = (AinAlsaModule*)pvData;

    ModThread_SetName(g_aszDevNames[poAin->uiDevIdx]);
    MsgLog(2, "Start thread %p.", pthread_self());

    MUINT32 uiErrorCount = 0;
    const MUINT64 uiStartTimeUsec = GetMonoTimeUsec();
    poAin->uiElapsedTimeUsec     = 0;
    poAin->uiTotalBufferCount    = 0;
    poAin->uiCapturedBufferCount = 0;

    while(!poAin->oCpuThread.bKillThread)
    {
        BufferInfo* poDstBuffer;

        poAin->uiElapsedTimeUsec = GetMonoTimeUsec() - uiStartTimeUsec;

        LStatus eStatus = ModLnk_GetReturnedBuffer(&(poAin->oOutLink), 100, MNULL, &poDstBuffer);

        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            MUINT8* puiBuffer = MNULL;

            eStatus = LBuffer_BeginAccess(poDstBuffer->hBuffer, 0, 1, &puiBuffer);

            if(LSTATUS_IS_SUCCESS(eStatus))
            {
                snd_pcm_state_t eState = snd_pcm_state(poAin->hAlsaDev);

                poAin->uiTotalBufferCount++;

                if((eState != SND_PCM_STATE_PREPARED)
                    && (eState != SND_PCM_STATE_RUNNING))
                {
                    uiErrorCount++;
                    MsgLog(1, "Warning, state=%u, preparing PCM device. (error count=%u)", eState, uiErrorCount);
                    snd_pcm_prepare(poAin->hAlsaDev);
                }

                snd_pcm_sframes_t iFramesToRead  = snd_pcm_bytes_to_frames(
                                                       poAin->hAlsaDev,
                                                       poDstBuffer->uiSizeBytes);

                snd_pcm_sframes_t iReadFrames = snd_pcm_readi(
                                                    poAin->hAlsaDev,
                                                    puiBuffer,
                                                    iFramesToRead);

                if(iReadFrames >= 0)
                {
                    if(iReadFrames != iFramesToRead)
                    {
                        uiErrorCount++;
                        MsgLog(1, "Warning, read %u frames out of %u (error count=%u)",
                               iReadFrames, iFramesToRead, uiErrorCount);
                    }

                    poDstBuffer->uiSizeBytes    = snd_pcm_frames_to_bytes(poAin->hAlsaDev, iReadFrames);
                    poDstBuffer->uiStartOffset  = 0;
                }
                else
                {
                    uiErrorCount++;
                    MsgLog(0, "Error in snd_pcm_readi: %s (error count=%u)",
                           snd_strerror(iReadFrames), uiErrorCount);
                    eStatus = LStatus_FAIL;
                }

                LBuffer_EndAccess(poDstBuffer->hBuffer);
            }
        }

        if(LSTATUS_IS_SUCCESS(eStatus)
            || (eStatus == LStatus_END_OF_STREAM))
        {
            poAin->uiCapturedBufferCount++;
            poDstBuffer->uiTimestampUsec = GetMonoTimeUsec();
            ModLnk_SubmitBuffer(&(poAin->oOutLink), poDstBuffer, MNULL, NO_TAG);
        }
        else
        {
            ModLnk_ReleaseBuffer(&(poAin->oOutLink), poDstBuffer);
        }

        if(LSTATUS_IS_FAIL(eStatus)
            && (eStatus != LStatus_TIMEOUT))
        {
            usleep(1000);
        }
    }

    MsgLog(2, "Kill thread.");

    return LStatus_OK;
}


/************************************************************************************************************\

Function:       AinAlsaMod_Start

Description:    .

\************************************************************************************************************/
LStatus AinAlsaMod_Start(AinAlsaModule* poAin)
{
    LStatus eStatus = LStatus_INVALID_PARAM;

    MsgLog(2, "{...");

    if(poAin != MNULL)
    {
        if(poAin->oOutLink.uiSubmitCount > 0)
        {
            eStatus = ModThread_Start(&(poAin->oCpuThread), poAin, AinAlsaMod_CpuThread);
        }
        else
        {
            MsgLogErr("ERROR! Bad connection.");
            eStatus = LStatus_FAIL;
        }
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       AinAlsaMod_Stop

Description:    .

\************************************************************************************************************/
void AinAlsaMod_Stop(AinAlsaModule* poAin)
{
    MsgLog(2, "{...");

    LStatus eStatus = LStatus_INVALID_PARAM;

    if (poAin != MNULL)
    {
        eStatus = ModThread_Stop(&poAin->oCpuThread);
        if(eStatus != LStatus_OK)
        {
            MsgLog(0, "ModThread_Stop failed (%d)\n", eStatus);
        }
    }

    MsgLog(2, "...}");
}

#else

/************************************************************************************************************\

Function:       AinAlsaMod_EnumDevices

Description:    Stub.

\************************************************************************************************************/
LStatus AinAlsaMod_EnumDevices(MUINT32* puiNbDevices)
{
    (void)puiNbDevices;
    MsgLog(1, "No ALSA support.");
    return LStatus_UNSUPPORTED;
}

/************************************************************************************************************\

Function:       AinAlsaMod_GetDeviceName

Description:    Stub.

\************************************************************************************************************/

LStatus AinAlsaMod_GetDeviceName(MUINT32 uiDeviceIdx, LSTR64* pszName)
{
    (void)uiDeviceIdx;
    (void)pszName;
    return LStatus_UNSUPPORTED;
}

/************************************************************************************************************\

Function:       AinAlsaMod_Init

Description:    Stub.

\************************************************************************************************************/
LStatus AinAlsaMod_Init(
    AinAlsaModule*  poAin,
    LDevice_Handle  hDevice,
    MUINT32         uiBufferCount,
    MUINT32         uiDevIdx,
    MUINT32         uiSamplingRate,
    MUINT32         uiNbChannels,
    MUINT32         uiSampleSize,
    MBOOL32         bBigEndian,
    MBOOL32         bUnsigned,
    MUINT32         uiFramesPerBuffer)
{
    (void)poAin;
    (void)hDevice;
    (void)uiBufferCount;
    (void)uiDevIdx;
    (void)uiSamplingRate;
    (void)uiNbChannels;
    (void)uiSampleSize;
    (void)bBigEndian;
    (void)bUnsigned;
    (void)uiFramesPerBuffer;
    MsgLog(1, "No ALSA support.");
    return LStatus_UNSUPPORTED;
}

/************************************************************************************************************\

Function:       AinAlsaMod_Cleanup

Description:    Stub.

\************************************************************************************************************/
void AinAlsaMod_Cleanup(AinAlsaModule* poAin)
{
    (void)poAin;
}

/************************************************************************************************************\

Function:       AinAlsaMod_Start

Description:    Stub.

\************************************************************************************************************/
LStatus AinAlsaMod_Start(AinAlsaModule* poAin)
{
    (void)poAin;
    return LStatus_UNSUPPORTED;
}

/************************************************************************************************************\

Function:       AinAlsaMod_Stop

Description:    Stub.

\************************************************************************************************************/
void AinAlsaMod_Stop(AinAlsaModule* poAin)
{
    (void)poAin;
}

#endif
