/************************************************************************************************************\

Module Name:    ModuleThread.c

Description:    .

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------
#include "ModuleThread.h"
#include "CommonUtils.h"


// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Function:       ModThread_Start

Description:    Start the cpu thread.

\************************************************************************************************************/
LStatus ModThread_Start(
            ModuleThread*       poModThread,
            void*               pvData,
            LCpuThread_Function pfnCpuThread)
{
    LStatus eStatus = LStatus_OK;

    if (poModThread->hThread == MNULL)
    {
        poModThread->bKillThread = MFALSE;

        eStatus = LCpuThread_Create(pfnCpuThread, pvData, &(poModThread->hThread));

        if (LSTATUS_IS_FAIL(eStatus))
        {
            MsgLogErr("ERROR! Cannot create thread.");
            poModThread->bKillThread = MTRUE;
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       ModThread_Stop

Description:    Stop the cpu thread.

\************************************************************************************************************/
LStatus ModThread_Stop(ModuleThread* poModThread)
{
    LStatus eRetStatus = LStatus_OK;

    if (poModThread->hThread != MNULL)
    {
        poModThread->bKillThread = MTRUE;

        if (LSTATUS_IS_SUCCESS(LCpuThread_Destroy(poModThread->hThread, &eRetStatus)))
        {
            MsgLog(
                2,
                "Cpu thread returned status= %d (%s)",
                eRetStatus,
                GetStatusStr(eRetStatus));
        }
        else
        {
            MsgLogErr("ERROR! Cannot join running thread. Continue...");
            eRetStatus = LStatus_FAIL;
        }

        poModThread->hThread = MNULL;
    }

    return eRetStatus;
}

/************************************************************************************************************\

Function:       ModThread_SetName

Description:    Set current cpu thread name. This help to identify the running cpu thread when debugging.

\************************************************************************************************************/
void ModThread_SetName(const char* szThreadName)
{
   #ifdef LINUX
    prctl(PR_SET_NAME, szThreadName, 0, 0, 0);
   #else
    (void)(szThreadName);
    // Not supported on windows
   #endif
}

/************************************************************************************************************\

Function:       ModThread_IsRunning

Description:    Return MTRUE if the cpu thread is running.

\************************************************************************************************************/
MBOOL ModThread_IsRunning(const ModuleThread* poModThread)
{
    return (poModThread->bKillThread == MFALSE);
}
