/************************************************************************************************************\

Module Name:    ModuleSync.c

Description:    .

    Copyright (c) 2016, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "ModuleSync.h"
#include "CommonUtils.h"

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------
// Don't try to synchronize if drift exceeds 10 seconds.
static const MUINT64 g_uiMaxCorrectableDriftUsec = 10000000lu;

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------

MBOOL ModSyncMst_GetMasterTimeUsec(
            ModuleSyncMaster* poModSyncMst,
            MUINT64* puiStreamTime);

/************************************************************************************************************\

Function:       ModSync_Init

Description:    .

\************************************************************************************************************/
LStatus ModSync_Init(
    ModuleSync* poModSync)
{
    return LMutex_Create(&(poModSync->oModSyncMst.hMutex));
}

/************************************************************************************************************\

Function:       ModSync_Cleanup

Description:    .

\************************************************************************************************************/
void ModSync_Cleanup(ModuleSync* poModSync)
{
    if(poModSync != MNULL)
    {
        if(poModSync->oModSyncMst.hMutex != MNULL)
        {
            LMutex_Destroy(poModSync->oModSyncMst.hMutex);
        }

        memset(poModSync, 0, sizeof(*poModSync));
    }
}

/************************************************************************************************************\

Function:       ModSync_SetParams

Description:    .

\************************************************************************************************************/
void ModSync_SetParams(
    ModuleSync* poModSync,
    MUINT64     uiInFrameDurationNsec,
    MUINT64     uiOutFrameDurationNsec,
    MUINT       uiMaxQueueLength,
    MBOOL       bControlQueueOverflow,
    MBOOL       bExternalMasterSync,
    MBOOL       bLowLatency)
{
    MsgLog(2, "Input Frame duration= %lu nsec, Output frame duration= %lu nsec",
              uiInFrameDurationNsec, uiOutFrameDurationNsec);

    ModSyncMst_Lock(&(poModSync->oModSyncMst));

    poModSync->iTimestampOffset         = 0;
    poModSync->uiInFrameDuration        = uiInFrameDurationNsec;
    poModSync->uiOutFrameDuration       = uiOutFrameDurationNsec;
    poModSync->uiTimestamp              = 0;
    poModSync->uiMaxTimestampDrift      = uiInFrameDurationNsec / 4;
    poModSync->uiPresentTime            = 0;
    poModSync->uiMaxPresentTimeDrift    = uiOutFrameDurationNsec / 4;
    poModSync->uiMaxQueueLengthAllowed  = bLowLatency
                                          ? 0
                                          : bControlQueueOverflow
                                            ? (uiMaxQueueLength - 4)
                                            : (uiMaxQueueLength * 2);
    poModSync->bLowLatency              = bLowLatency;
    poModSync->uiLowlOverQueueCount     = 0;
    poModSync->bResync                  = MTRUE;
    poModSync->bExternalMasterSync      = bExternalMasterSync;
    // Drift tolerance from master stream: 1/2 frame + 1/10 frame --> 6/10 frames
    // We must add the arbitrary 1/10 frame to the hysteresis or else the drift might bounce from
    // -1/2 frame to +1/2 frame, skipping and repeating in loop.
    poModSync->uiMaxMstStrmDriftAllowed = (uiOutFrameDurationNsec * 6) / 10;

    memset(poModSync->szQueueLevel, 'o', sizeof(poModSync->szQueueLevel));

    poModSync->uiPrevTimestampNsec = 0;
    poModSync->bIsConfigured = MTRUE;

    ModSyncMst_Unlock(&(poModSync->oModSyncMst));
}

/************************************************************************************************************\

Function:       ModSync_PrePresent

Description:    .

\************************************************************************************************************/
MBOOL ModSync_PrePresent(
    ModuleSync*     poModSync,
    MUINT64         uiTimestampNsec,
    MUINT           uiQueueLength,
    MINT64*         piCurDriftFromMstNsec)
{
    // For debugging only.
    if (0)
    {
        MUINT uiQueueLevel = min(uiQueueLength, sizeof(poModSync->szQueueLevel) - 1);

        poModSync->szQueueLevel[uiQueueLevel] = 0;
        MsgLog(2, "Q-Size: %u\t\t%s", uiQueueLength, poModSync->szQueueLevel);
        poModSync->szQueueLevel[uiQueueLevel] = 'o';
    }

    // Low latency mode
    if (poModSync->bLowLatency)
    {
        if ((uiQueueLength <= poModSync->uiMaxQueueLengthAllowed)
            && (uiQueueLength > 0))
        {
            if (++poModSync->uiLowlOverQueueCount > 240)
            {
                poModSync->uiMaxQueueLengthAllowed--;

                MsgLog(2, "Low latency: Decrease max Q-len to %u.", poModSync->uiMaxQueueLengthAllowed);
            }
        }
        else
        {
            poModSync->uiLowlOverQueueCount = 0;

            if (uiQueueLength > poModSync->uiMaxQueueLengthAllowed)
            {
                poModSync->uiMaxQueueLengthAllowed++;

                MsgLog(2, "Low latency: Increase max Q-len to %u.", poModSync->uiMaxQueueLengthAllowed);
            }
        }

        if (uiQueueLength <= poModSync->uiMaxQueueLengthAllowed)
        {
            return MTRUE;
        }
        else
        {
            MsgLog(2, "Low latency: Queue overflow (Q-len = %u). Skip frame.", uiQueueLength);
            return MFALSE;
        }
    }

    if (poModSync->bResync)
    {
        return MTRUE;
    }

    // Check queue overflow
    if (uiQueueLength > poModSync->uiMaxQueueLengthAllowed)
    {
        MsgLog(2, "Queue overflow (Queue length = %u). Skip frame and resync.", uiQueueLength);
        poModSync->bResync = MTRUE;
        return MFALSE;
    }

    if (poModSync->bExternalMasterSync)
    {
        MINT64 iDriftNsec = (MINT64) uiTimestampNsec - poModSync->uiTimestamp;

        if(piCurDriftFromMstNsec != MNULL)
        {
            *piCurDriftFromMstNsec = iDriftNsec;
        }

        if(uiTimestampNsec > poModSync->uiPrevTimestampNsec)
        {
            MUINT64 uiDriftNsec = labs(iDriftNsec);

            if(uiDriftNsec > poModSync->uiMaxMstStrmDriftAllowed)
            {
                // Don't try to synchronize if the drift is too large. This is to avoid skipping/repeating
                // when the timestamp of one of the two streams wraps-around before the other.
                if (uiDriftNsec < (g_uiMaxCorrectableDriftUsec * 1000))
                {
                    if (iDriftNsec < 0)
                    {
                        MBOOL bSkipFrame = (uiQueueLength > 0);
                        MsgLog(2, "Timestamp drift too much (%ld nsec). %sesync.",
                                  iDriftNsec, bSkipFrame ? "Skip frame and r" : "R");
                        poModSync->bResync = MTRUE;
                        return !bSkipFrame;
                    }
                    else
                    {
                        while(uiDriftNsec > poModSync->uiMaxMstStrmDriftAllowed)
                        {
                            MsgLog(2, "Timestamp drift too much (%lu nsec). Repeat.", uiDriftNsec);
                            MUINT64 uiTimeNsec = GetMonoTimeNsec();
                            usleep(poModSync->uiOutFrameDuration / 1000);
                            MUINT64 uiTimeDiff = (GetMonoTimeNsec() - uiTimeNsec);
                            uiDriftNsec -= min(uiDriftNsec, uiTimeDiff);
                        }
                    }
                }
                else
                {
                    MsgLog(1, "Warning: drift is too large to be corrected! (%ld nsec)", iDriftNsec);
                }
            }
        }

        poModSync->uiPrevTimestampNsec = uiTimestampNsec;

        return MTRUE;
    }
    else
    {
        uiTimestampNsec += poModSync->iTimestampOffset;

        // Check timestamp drifting
        if ((uiTimestampNsec < (poModSync->uiTimestamp - poModSync->uiMaxTimestampDrift))
            || (uiTimestampNsec > (poModSync->uiTimestamp + poModSync->uiMaxTimestampDrift)))
        {
            MINT64  iDriftTime = (MINT64)uiTimestampNsec - poModSync->uiTimestamp;
            MBOOL   bSkipFrame = (uiQueueLength > 0) && (iDriftTime < 0);

            MsgLog(2, "Timestamp drift too much (%ld nsec). %sesync.",
                      iDriftTime, bSkipFrame ? "Skip frame and r" : "R");
            poModSync->bResync = MTRUE;
            return !bSkipFrame;
        }

        MBOOL bPresentLate = MFALSE;

        // Go to next present time if we are too late.
        while (((MINT64)poModSync->uiPresentTime) < GetMonoTimeNsec())
        {
            poModSync->uiPresentTime += poModSync->uiOutFrameDuration;
            bPresentLate = MTRUE;
        }

        // Wait until the present time is ahead of the timestamp.
        while (poModSync->uiPresentTime < ((MINT64)poModSync->uiTimestamp))
        {
            usleep(poModSync->uiOutFrameDuration / 1000);
            poModSync->uiPresentTime += poModSync->uiOutFrameDuration;
        }

        poModSync->uiTimestamp += (MINT64)poModSync->uiInFrameDuration;

        // Present the frame if its time range intersect the present time.
        if (poModSync->uiPresentTime <= ((MINT64)poModSync->uiTimestamp))
        {
            MUINT64 uiCurTimeNsec = GetMonoTimeNsec();

            if (poModSync->uiPresentTime > uiCurTimeNsec)
            {
                MUINT64 uiSleepTimeUsec = (poModSync->uiPresentTime - uiCurTimeNsec) / 1000;

                usleep(uiSleepTimeUsec);
            }

            return MTRUE;
        }

        // Check queue underflow
        if (bPresentLate && (uiQueueLength == 0))
        {
            MUINT64 uiSleepTimeUsec = 3 * poModSync->uiInFrameDuration / 2000;

            MsgLog(2, "Queue underflow. Wait %lu usec and resync.", uiSleepTimeUsec);
            usleep(uiSleepTimeUsec);
            poModSync->bResync = MTRUE;
        }

        MsgLog(3, "Skip frame.");

        return MFALSE;
    }
}

/************************************************************************************************************\

Function:       ModSync_PostPresent

Description:    .

\************************************************************************************************************/
void ModSync_PostPresent(
    ModuleSync* poModSync,
    MUINT64     uiTimestampNsec)
{
    // Low latency mode
    if (poModSync->bLowLatency)
    {
        const MUINT64 uiSleepDurationUsec = (7 * poModSync->uiOutFrameDuration) / 8000;

        MINT64 iTimeUsec   = (MINT64)GetMonoTimeUsec();
        MINT64 iTimeOffset = iTimeUsec - poModSync->uiTimestamp;

        poModSync->uiTimestamp = iTimeUsec + (poModSync->uiOutFrameDuration / 1000);

        if ((iTimeOffset < 0)
            || (iTimeOffset > (MINT64)(uiSleepDurationUsec / 2)))
        {
            iTimeOffset = 0;
        }

        usleep(uiSleepDurationUsec - iTimeOffset);

        poModSync->bResync = MFALSE;
        return;
    }

    if(poModSync->bExternalMasterSync)
    {
        MUINT64 uiMasterTimeUsec = 0;

        if(ModSyncMst_GetMasterTimeUsec(&(poModSync->oModSyncMst), &uiMasterTimeUsec))
        {
            poModSync->uiTimestamp = (uiMasterTimeUsec*1000lu) + poModSync->uiOutFrameDuration;
            poModSync->bResync = MFALSE;
        }
    }
    else
    {
        MUINT64 uiPresentTime = GetMonoTimeNsec() - (poModSync->uiOutFrameDuration / 2);

        // Check present time drifting.
        if (!poModSync->bResync
            && ((uiPresentTime < (poModSync->uiPresentTime - poModSync->uiMaxPresentTimeDrift))
                || (uiPresentTime > (poModSync->uiPresentTime + poModSync->uiMaxPresentTimeDrift))))
        {
            MsgLog(2, "Present time drift too much (%ld nsec). Resync.",
                      (MINT64)uiPresentTime - poModSync->uiPresentTime);
            poModSync->bResync = MTRUE;
        }

        if (poModSync->bResync)
        {
            poModSync->iTimestampOffset    = (MINT64)uiPresentTime - uiTimestampNsec;
            poModSync->uiTimestamp         = uiPresentTime + poModSync->uiInFrameDuration;
            poModSync->uiPresentTime       = uiPresentTime;
            poModSync->bResync             = MFALSE;

            MsgLog(3, "Resync: TimestampOffset= %ld, Timestamp (next)= %lu, PresentTime= %lu",
                      poModSync->iTimestampOffset, poModSync->uiTimestamp, poModSync->uiPresentTime);
        }

        poModSync->uiPresentTime += poModSync->uiOutFrameDuration;
    }
}

/************************************************************************************************************\

Function:       ModSyncMst_SetParams

Description:    .

\************************************************************************************************************/
void ModSyncMst_SetParams(
    ModuleSyncMaster*   poModSyncMst,
    MUINT64             uiBufferTimeUsec,
    MUINT               uiInputQueueLength,
    MUINT               uiTargetQueueSize,
    MBOOL               bMasterStream)
{
    if(poModSyncMst != MNULL)
    {
        ModSyncMst_Lock(poModSyncMst);

        ModuleSyncStreamInfo* poStrmInfo = bMasterStream
                                             ? &(poModSyncMst->oMstInfo)
                                             : &(poModSyncMst->oSlvInfo);

        poStrmInfo->uiBufferTimeUsec    = uiBufferTimeUsec;
        poStrmInfo->uiInputQueueLength  = uiInputQueueLength;
        poStrmInfo->eState              = StreamState_NoBuffer;
        poStrmInfo->bIsConfigured       = MTRUE;
        poStrmInfo->szName              = bMasterStream ? "Master" : "Slave";

        MUINT uiMaxTargetQueueSize = max(((MINT)uiInputQueueLength)-2, 1);

        if(bMasterStream)
        {
            // The master stream has no target queue buffering. Its initial level of buffering
            // will depend on the slave stream target queue buffering and on the latency gap
            // between the two streams.
            poStrmInfo->uiTargetQueueSize = uiMaxTargetQueueSize;
        }
        else
        {
            poStrmInfo->uiTargetQueueSize = min(uiTargetQueueSize, uiMaxTargetQueueSize);
        }

        ModSyncMst_Unlock(poModSyncMst);

        MsgLog(2, "%s stream: InputBufferLength = %u, BufferTime = %u us",
                  poStrmInfo->szName, uiInputQueueLength, uiBufferTimeUsec);

        if(bMasterStream)
        {
            MsgLog(2, "Master stream maximum queue size = %u", poStrmInfo->uiTargetQueueSize);
        }
        else
        {
            MsgLog(2, "Slave stream target queue size = %u", poStrmInfo->uiTargetQueueSize);
        }
    }
}

/************************************************************************************************************\

Function:       ModSyncMst_GetMasterTimeUsec

Description:    .

\************************************************************************************************************/
MBOOL ModSyncMst_GetMasterTimeUsec(
    ModuleSyncMaster*   poModSyncMst,
    MUINT64*            puiStreamTime)
{
    MBOOL bSuccess = MFALSE;

    if(poModSyncMst != MNULL)
    {
        ModSyncMst_Lock(poModSyncMst);

        ModuleSyncStreamInfo* poStrmInfo = &(poModSyncMst->oMstInfo);

        if(poStrmInfo->bFirstTsUpdated)
        {
            *puiStreamTime = poStrmInfo->uiLastValidBufTsUsec
                                + (GetMonoTimeUsec() - poStrmInfo->uiLastBufSysTimeUsec);

            bSuccess = MTRUE;
        }

        ModSyncMst_Unlock(poModSyncMst);
    }

    return bSuccess;
}

/************************************************************************************************************\

Function:       ModSyncMst_SetBaseTime

Description:    .

Comments:       Must be called with lock acquired.

\************************************************************************************************************/
void ModSyncMst_SetBaseTime(
    ModuleSyncMaster*   poModSyncMst,
    MUINT64             uiBaseTimeUsec,
    MBOOL               bMasterStream)
{
    if(poModSyncMst != MNULL)
    {
        ModuleSyncStreamInfo* poStrmInfo = bMasterStream
                                             ? &(poModSyncMst->oMstInfo)
                                             : &(poModSyncMst->oSlvInfo);

        poStrmInfo->uiBaseTime = uiBaseTimeUsec;
        poStrmInfo->bBaseTimeSet = MTRUE;
    }
}

/************************************************************************************************************\

Function:       ModSyncMst_UpdateTimestamp

Description:    .

\************************************************************************************************************/
void ModSyncMst_UpdateTimestamp(
    ModuleSyncMaster*   poModSyncMst,
    MUINT64             uiTimestampUsec,
    MBOOL               bMasterStream)
{
    if(poModSyncMst != MNULL)
    {
        ModSyncMst_Lock(poModSyncMst);

        ModuleSyncStreamInfo* poStrmInfo = bMasterStream
                                             ? &(poModSyncMst->oMstInfo)
                                             : &(poModSyncMst->oSlvInfo);

        if((!poStrmInfo->bFirstTsUpdated) || (uiTimestampUsec > poStrmInfo->uiLastBufTsUsec))
        {
            poStrmInfo->uiLastValidBufTsUsec    = uiTimestampUsec;
            poStrmInfo->uiLastBufSysTimeUsec    = GetMonoTimeUsec();
            poStrmInfo->bFirstTsUpdated         = MTRUE;
        }

        if(!poStrmInfo->bBaseTimeSet)
        {
            poStrmInfo->uiBaseTime = uiTimestampUsec;
            poStrmInfo->bBaseTimeSet = MTRUE;
        }

        poStrmInfo->uiLastBufTsUsec = uiTimestampUsec;

        ModSyncMst_Unlock(poModSyncMst);
    }
}

/************************************************************************************************************\

Function:       ModSyncMst_LogState

Description:    .

\************************************************************************************************************/
void ModSyncMst_LogState(
    ModuleSyncStreamInfo*   poStrmInfo,
    MUINT                   uiQueueSize,
    MUINT64                 uiTimestampUsec)
{
    switch(poStrmInfo->eState)
    {
        case StreamState_BufferReceived:
        {
            MsgLog(3, "%s received buffer, dropping. (queue=%u/%u, Base=%lu, Ts=%lu)",
                      poStrmInfo->szName, uiQueueSize, poStrmInfo->uiTargetQueueSize,
                      poStrmInfo->uiBaseTime, uiTimestampUsec);
            break;
        }
        case StreamState_BufferingUp:
        {
            if(uiQueueSize != poStrmInfo->uiCurQueueSize)
            {
                MsgLog(3, "%s buffering up. (queue=%u/%u, Base=%lu, Ts=%lu)",
                          poStrmInfo->szName, uiQueueSize, poStrmInfo->uiTargetQueueSize,
                          poStrmInfo->uiBaseTime, uiTimestampUsec);
            }
            break;
        }
        case StreamState_BufferingDown:
        {
            if(uiQueueSize != poStrmInfo->uiCurQueueSize)
            {
                MsgLog(3, "%s buffering down. (queue=%u/%u, Base=%lu, Ts=%lu)",
                          poStrmInfo->szName, uiQueueSize, poStrmInfo->uiTargetQueueSize,
                          poStrmInfo->uiBaseTime, uiTimestampUsec);
            }
            break;
        }
        case StreamState_Buffered:
        {
            if(uiQueueSize > poStrmInfo->uiTargetQueueSize)
            {
                MsgLog(3, "%s buffered, dropping. (queue=%u/%u, Base=%lu, Ts=%lu)",
                          poStrmInfo->szName, uiQueueSize, poStrmInfo->uiTargetQueueSize,
                          poStrmInfo->uiBaseTime, uiTimestampUsec);
            }
            break;
        }
        case StreamState_Aligning:
        {
            MsgLog(3, "%s aligning. (queue=%u/%u, Base=%lu, Ts=%lu, Target base=%lu)",
                      poStrmInfo->szName, uiQueueSize, poStrmInfo->uiTargetQueueSize,
                      poStrmInfo->uiBaseTime, uiTimestampUsec, poStrmInfo->uiTargetBaseTime);
            break;
        }
        case StreamState_ReadyToRender:
        {
            MsgLog(3, "%s pre-synchronization done. (queue=%u/%u, Base=%lu, Ts=%lu usec)",
                      poStrmInfo->szName, uiQueueSize, poStrmInfo->uiTargetQueueSize,
                      poStrmInfo->uiBaseTime, uiTimestampUsec);
            break;
        }
        default:
        {
            break;
        }
    }
}

/************************************************************************************************************\

Function:       ModSyncMst_PreSync

Description:    .

Comments:       Must be called with lock acquired.

\************************************************************************************************************/
void ModSyncMst_PreSync(
    ModuleSyncMaster*   poModSyncMst,
    MUINT               uiQueueSize,
    MUINT64             uiTimestampUsec,
    MBOOL               bMasterStream,
    MBOOL*              pbCanRender,
    MBOOL*              pbDropOldest)
{
    if((poModSyncMst != MNULL)
        && (pbCanRender != MNULL)
        && (pbDropOldest != MNULL)
        && (uiQueueSize > 0))
    {
        *pbDropOldest = MFALSE;
        *pbCanRender = MFALSE;

        if(poModSyncMst->oMstInfo.bIsConfigured
            && poModSyncMst->oSlvInfo.bIsConfigured)
        {
            ModuleSyncStreamInfo* poMstStrm         = &(poModSyncMst->oMstInfo);
            ModuleSyncStreamInfo* poSlvStrm         = &(poModSyncMst->oSlvInfo);
            ModuleSyncStreamInfo* poStrmInfo        = MNULL;
            ModuleSyncStreamInfo* poOtherStrmInfo   = MNULL;

            if(bMasterStream)
            {
                poStrmInfo      = poMstStrm;
                poOtherStrmInfo = poSlvStrm;
            }
            else
            {
                poStrmInfo      = poSlvStrm;
                poOtherStrmInfo = poMstStrm;
            }

            // Base time should be the timestamp of the oldest buffer of the input queue.
            if(!poStrmInfo->bBaseTimeSet)
            {
                poStrmInfo->uiBaseTime = uiTimestampUsec;
                poStrmInfo->bBaseTimeSet = MTRUE;
            }

            // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
            // Pre-synchronization state machine.
            // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

            // 1 - When both streams have received the first buffer, estimate the difference of latency
            //     between both streams and start buffering queues.
            if(poStrmInfo->eState == StreamState_NoBuffer)
            {
                poStrmInfo->eState = StreamState_BufferReceived;

                if(poOtherStrmInfo->eState == StreamState_BufferReceived)
                {
                    MsgLog(3, "%s received first buffer: Base=%lu, %s Base=%lu, diff=%ld",
                              poStrmInfo->szName, poStrmInfo->uiBaseTime, poOtherStrmInfo->szName, poOtherStrmInfo->uiBaseTime,
                              (MINT64)(poStrmInfo->uiBaseTime - poOtherStrmInfo->uiBaseTime));

                    MINT64 iDrift = (MINT64)(poSlvStrm->uiBaseTime - poMstStrm->uiBaseTime);

                    if (labs(iDrift) < (MINT64)g_uiMaxCorrectableDriftUsec)
                    {
                        if(poMstStrm->uiBaseTime > poSlvStrm->uiBaseTime)
                        {
                            poSlvStrm->uiTargetBaseTime = max((MINT64)0,
                                                              (MINT64)(poMstStrm->uiBaseTime - (poSlvStrm->uiBufferTimeUsec / 2)));

                            poSlvStrm->eState = StreamState_Aligning;
                            poMstStrm->eState = StreamState_BufferingUp;
                        }
                        else
                        {
                            poMstStrm->uiTargetBaseTime = max((MINT64)0,
                                                              (MINT64)(poSlvStrm->uiBaseTime - (poMstStrm->uiBufferTimeUsec / 2)));

                            poMstStrm->eState = StreamState_Aligning;
                            poSlvStrm->eState = StreamState_BufferingUp;
                        }
                    }
                    else
                    {
                        // Timestamp drift too large, abort synchronization.
                        MsgLog(0, "ERROR: Timestamp drift too large to synchronize streams (%ld usec).\n", iDrift);

                        poMstStrm->eState = StreamState_ReadyToRender;
                        poSlvStrm->eState = StreamState_ReadyToRender;
                    }
                }
            }

            // 2 - Drop buffers until the other stream has received its first buffer.
            if(poStrmInfo->eState == StreamState_BufferReceived)
            {
                // Don't drop when we are going to empty the queue as the base time would become invalid.
                if(uiQueueSize > 1)
                {
                    *pbDropOldest = MTRUE;

                    ModSyncMst_LogState(poStrmInfo, uiQueueSize, uiTimestampUsec);
                }
            }

            // 3 - Buffer up until the level reaches the target.
            if(poStrmInfo->eState == StreamState_BufferingUp)
            {
                ModSyncMst_LogState(poStrmInfo, uiQueueSize, uiTimestampUsec);

                // Building initial target buffering.
                if(uiQueueSize >= poStrmInfo->uiTargetQueueSize)
                {
                    poStrmInfo->eState = StreamState_BufferingDown;
                }
            }

            // 4 - Drop buffers if level exceeded the target.
            if(poStrmInfo->eState == StreamState_BufferingDown)
            {
                ModSyncMst_LogState(poStrmInfo, uiQueueSize, uiTimestampUsec);

                // Initial target buffering built, drop if level exceeded.
                if(uiQueueSize > poStrmInfo->uiTargetQueueSize)
                {
                    *pbDropOldest = MTRUE;
                }
                else
                {
                    poStrmInfo->eState = StreamState_Buffered;
                }
            }

            // 5 - Ready to start once the slave stream is buffered.
            if((poStrmInfo->eState == StreamState_Buffered)
                && (!bMasterStream))
            {
                // Slave stream is buffered and base times are approximately aligned, we can start rendering.
                poMstStrm->eState = StreamState_ReadyToRender;
                poSlvStrm->eState = StreamState_ReadyToRender;
            }

            // 6 - If the stream has reached its target level of buffering, drop buffers to maintain this level.
            if((poStrmInfo->eState == StreamState_Buffered)
                && (uiQueueSize > poStrmInfo->uiTargetQueueSize))
            {
                // Waiting other stream to build target buffering, drop if target level exceeded.
                *pbDropOldest = MTRUE;

                ModSyncMst_LogState(poStrmInfo, uiQueueSize, uiTimestampUsec);
            }

            // 7 - Flush oldest buffer until base times are approximately aligned, then start rendering.
            if(poStrmInfo->eState == StreamState_Aligning)
            {
                if(poStrmInfo->uiBaseTime < poStrmInfo->uiTargetBaseTime)
                {
                    *pbDropOldest = MTRUE;

                    ModSyncMst_LogState(poStrmInfo, uiQueueSize, uiTimestampUsec);
                }
                else
                {
                    MUINT64 uiTargetBaseTimeError = poOtherStrmInfo->uiBaseTime - poStrmInfo->uiTargetBaseTime;
                    if(uiTargetBaseTimeError > (poStrmInfo->uiBufferTimeUsec / 2))
                    {
                        MsgLog(1, "Warning:\tCouldn't align %s base time with %s, %s queue might be too"
                                  " short to compensate for latency diff.",
                                  poStrmInfo->szName, poOtherStrmInfo->szName, poOtherStrmInfo->szName);
                        MsgLog(1, "\t\tSynchronization could go bad...");
                        MsgLog(1, "\t\t%s target base time: %lu us, %s current base time: %lu us, Error=%lu us",
                                  poStrmInfo->szName, poStrmInfo->uiTargetBaseTime, poOtherStrmInfo->szName,
                                  poOtherStrmInfo->uiBaseTime, uiTargetBaseTimeError);
                    }

                    // Base times are approximately aligned, slave stream can start building its initial buffering.
                    poStrmInfo->eState = StreamState_BufferingUp;
                }
            }

            // 8 - Pre-sync done, start rendering.
            if(poStrmInfo->eState == StreamState_ReadyToRender)
            {
                *pbCanRender = MTRUE;

                ModSyncMst_LogState(poStrmInfo, uiQueueSize, uiTimestampUsec);
            }

            if(*pbDropOldest)
            {
                poStrmInfo->bBaseTimeSet = MFALSE;
            }

            poStrmInfo->uiCurQueueSize = uiQueueSize;
        }
    }
}

/************************************************************************************************************\

Function:       ModSyncMst_Lock

Description:    .

\************************************************************************************************************/
void ModSyncMst_Lock(ModuleSyncMaster *poModSyncMst)
{
    LMutex_Lock(poModSyncMst->hMutex);
}

/************************************************************************************************************\

Function:       ModSyncMst_Unlock

Description:    .

\************************************************************************************************************/
void ModSyncMst_Unlock(ModuleSyncMaster *poModSyncMst)
{
    LMutex_Unlock(poModSyncMst->hMutex);
}
