/************************************************************************************************************\

Module Name:    ModuleLink.h

Description:    .

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

#ifndef INC_ModuleLink_H
#define INC_ModuleLink_H

// -----------------------------------------------------------------------------------------------------------
//                                   I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "Liberatus.h"
#include "CommonUtils.h"
#include "BufferQueue.h"

// -----------------------------------------------------------------------------------------------------------
//                                   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

#define MAX_SUBPICTURE_COUNT    16
#define MAX_SUBMIT_COUNT        16
#define MAX_BUFFER_COUNT        32

#define ENABLE_AUTO_SYNC        MTRUE

/************************************************************************************************************\

Structure:      ModuleLink

Description:    Link that manages the buffers circulation and synchronization between Liberatus modules.

Comments:       Illustration of 2 links ([1] and [2]) between 3 modules ([A], [B] and [C]):


                               +---- Return-Queue <----+            +---- Return-Queue <----+
                               |                       |            |                       |
                  .............v.......           .....|............v........          .....|............
                  .         Wait(Tag) .           . Get(Tag)     Wait(Tag)  .          . Get(Tag)       .
                  .            v      .           .    ^            v       .          .    ^           .
                  . Module[A]-OUT     .           .   IN-Module[B]-OUT      .          .   IN-Module[C] .
                  .            v      .           .    ^            v       .          .    ^           .
                  .         Get(Tag)  .           . Wait(Tag)    Get(Tag)   .          . Wait(Tag)      .
                  .............|.......           .....^............|........          .....^............
                               |                       |            |                       |
                               +----> Submit-Queue ----+            +----> Submit-Queue ----+

                              \.........Link[1]........./          \.........Link[2]........./


                    - Module[A] is the owner of Link[1];
                    - Link[1] output is attached to Module[B];
                    - Module[B] is the owner of Link[2];
                    - Link[2] output is attached to Module[C];
                    - Module[C] is not owner of a link;

\************************************************************************************************************/
typedef struct tagModuleLink
{
    MUINT       uiSubmitCount;                      // Number of modules attached to the output.
    BufferQueue aoSubmitQueue[MAX_SUBMIT_COUNT];    // One buffer queue per attached module.
    MBOOL       bSyncBefore;                        // Synchronize buffers before to push into the queue.
    MUINT       uiSyncFraction;                     // Synchronize only a fraction of the buffer.
    MUINT64     uiTotalTimeUsec;                    // Total time waiting for the sync (perf check only).
    MUINT       uiTimedBufferCount;                 // Number of synched buffer (perf check only).
    MBOOL       bReturnAllBuffer;                   // Return all submitted buffer immediately.
    MBOOL       abStubOutput[MAX_SUBMIT_COUNT];     // Skip buffer submit on output links independently.
    MBOOL       bSkipAll;                           // Used to tell the connected ModLinkInput to return all 
                                                    // submitted buffers.
    BufferQueue oReturnQueue;                       // Queue where the attached modules return the submitted
                                                    // buffers that are not used anymore.
    MUINT       uiBufferCount;                      // Number of buffers allocated for this link.
    BufferInfo  aoBufferInfo[MAX_BUFFER_COUNT];     // Buffers allocated.
} ModuleLink;

#define ModuleLink_Construct { /*.uiSubmitCount       =*/ 0, \
                               /*.aoSubmitQueue[0]    =*/{ BufferQueue_Construct, \
                               /*.aoSubmitQueue[1]    =*/  BufferQueue_Construct, \
                               /*.aoSubmitQueue[2]    =*/  BufferQueue_Construct, \
                               /*.aoSubmitQueue[3]    =*/  BufferQueue_Construct, \
                               /*.aoSubmitQueue[4]    =*/  BufferQueue_Construct, \
                               /*.aoSubmitQueue[5]    =*/  BufferQueue_Construct, \
                               /*.aoSubmitQueue[6]    =*/  BufferQueue_Construct, \
                               /*.aoSubmitQueue[7]    =*/  BufferQueue_Construct, \
                               /*.aoSubmitQueue[8]    =*/  BufferQueue_Construct, \
                               /*.aoSubmitQueue[9]    =*/  BufferQueue_Construct, \
                               /*.aoSubmitQueue[10]   =*/  BufferQueue_Construct, \
                               /*.aoSubmitQueue[11]   =*/  BufferQueue_Construct, \
                               /*.aoSubmitQueue[12]   =*/  BufferQueue_Construct, \
                               /*.aoSubmitQueue[13]   =*/  BufferQueue_Construct, \
                               /*.aoSubmitQueue[14]   =*/  BufferQueue_Construct, \
                               /*.aoSubmitQueue[15]   =*/  BufferQueue_Construct }, \
                               /*.bSyncBefore         =*/ MFALSE, \
                               /*.uiSyncFraction      =*/ 1, \
                               /*.uiTotalTimeUsec     =*/ 0, \
                               /*.uiTimedBufferCount  =*/ 0, \
                               /*.bReturnAllBuffer    =*/ MFALSE, \
                               /*.abStubbedOutput     =*/ {MFALSE}, \
                               /*.bSkipAll            =*/ MFALSE, \
                               /*.oReturnQueue        =*/ BufferQueue_Construct, \
                               /*.uiBufferCount       =*/ 0 }

/************************************************************************************************************\

Structure:      ModuleLinkInput

Description:    Input where is attached a module link.

\************************************************************************************************************/
typedef struct tagModuleLinkInput
{
    MBOOL       bSyncBefore;        // Synchronize buffers before to push into the queue.
    MUINT       uiSyncFraction;     // Synchronize only a fraction of the buffer.
    ModuleLink* poModLnk;           // Module link where this module is attached to.
    MUINT       uiSubmitIdx;        // Assigned queue index.
    MUINT       uiRateDivisor;      // Get only 1 frame on 'uiRateDivisor'.
    MUINT       uiSkipFrameCount;   // Used with frame rate divisor.
    MBOOL       bSkipAll;           // Return all submitted buffers.
    MUINT64     uiTotalTimeUsec;    // Total time waiting for the sync (perf check only).
    MUINT       uiTimedBufferCount; // Number of synched buffer (perf check only).

} ModuleLinkInput;

#define ModuleLinkInput_Construct { /*.bSyncBefore          =*/ MFALSE, \
                                    /*.uiSyncFraction       =*/ 1, \
                                    /*.poModLnk             =*/ MNULL, \
                                    /*.uiSubmitIdx          =*/ 0, \
                                    /*.uiRateDivisor        =*/ 1, \
                                    /*.uiSkipFrameCount     =*/ 0, \
                                    /*.bSkipAll             =*/ MFALSE, \
                                    /*.uiTotalTimeUsec      =*/ 0, \
                                    /*.uiTimedBufferCount   =*/ 0 }

// -----------------------------------------------------------------------------------------------------------
//                  G L O B A L   V A R I A B L E / F U N C T I O N   R E F E R E N C E S
// -----------------------------------------------------------------------------------------------------------

LStatus ModLnk_Init(ModuleLink*         poModuleLink,
                    LDevice_Handle      hDevice,
                    MUINT               uiBufferCount,
                    LBuffer_Attributes* poBufferAttributes,
                    MBOOL               bInternalBuffer,
                    MUINT               uiPrivateDataSize,
            const   MCHAR8*             pszModuleName);
void ModLnk_Cleanup(ModuleLink* poModuleLink);
LStatus ModLnk_Connect(ModuleLink* poModuleLink, ModuleLinkInput* poModLnkInput);
LStatus ModLnk_SubmitBuffer(ModuleLink* poModuleLink,
                            BufferInfo* poBufferInfo,
                            LDeviceThread_Handle hDevThread,
                            MUINT64 uiTag);
LStatus ModLnkIn_GetSubmittedBuffer(ModuleLinkInput* poModLnkInput,
                                    MUINT uiQueueTimeoutMsec,
                                    MUINT uiSubPicture,
                                    LDeviceThread_Handle hDevThreadDst,
                                    BufferInfo** ppoBufferInfo,
                                    MBOOL *pbDtSynched,
                                    MUINT* puiQueueSize);
LStatus ModLnkIn_InspectSubmittedBuffer(ModuleLinkInput* poModLnkInput,
                                        BufferInfo** ppoBufferInfo,
                                        MUINT* puiQueueSize);
LStatus ModLnkIn_ReturnBuffer(ModuleLinkInput* poModLnkInput,
                              BufferInfo* poBufferInfo,
                              LDeviceThread_Handle hDevThread,
                              MUINT64 uiTag);
LStatus ModLnk_GetReturnedBuffer(ModuleLink* poModuleLink,
                                 MUINT uiQueueTimeoutMsec,
                                 LDeviceThread_Handle hDevThreadDst,
                                 BufferInfo** ppoBufferInfo);
MBOOL   ModLnk_ReleaseBuffer(ModuleLink* poModuleLink,
                             BufferInfo* poBufferInfo);
void ModLnk_ShowState(ModuleLink* poModuleLink);
void ModLnkIn_CancelSubmittedBuffers(ModuleLinkInput* poModLnkInput);

#endif // INC_ModuleLink_H
