/************************************************************************************************************\

Module Name:    ModuleLink.c

Description:    .

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "ModuleLink.h"

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

static const MBOOL g_bCheckPerformance  = MFALSE;
static const MBOOL g_bForceWaitBefore   = MFALSE;

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Function:       ModLnk_Init

Description:    Create and initialized resources need (buffers and queues).

Parameters:     poModuleLink        Module link object.
                hDevice             Liberatus device handle.
                uiBufferCount       Number of buffers to create.
                poBufferAttributes  Liberatus buffer attributes to be created. Ignored with internal buffers.
                bInternalBuffer     MFALSE: Liberatus buffers will be created and owned by this module link.
                                    MTRUE:  Liberatus buffers will be created and owned by the module owner
                                            of this module link.
                uiPrivateDataSize   When not zero, allocate data attached to the buffers that should be used
                                    by the module owner only.
                pszModuleName       Specified the module name that initialized the Module link object.
                                    For debugging purpose. MNULL if not specified.

\************************************************************************************************************/
LStatus ModLnk_Init(
            ModuleLink*         poModuleLink,
            LDevice_Handle      hDevice,
            MUINT               uiBufferCount,
            LBuffer_Attributes* poBufferAttributes,
            MBOOL               bInternalBuffer,
            MUINT               uiPrivateDataSize,
    const   MCHAR8*             pszModuleName)
{
    // Buffer uniq id that should be used for debugging only.
    static MUINT s_uiBufferId = 0;
    MINT i;

    MsgLog(2, "{...");

    ModLnk_Cleanup(poModuleLink);

    LStatus eStatus = ((poModuleLink != MNULL)
                       && (hDevice != MNULL)
                       && (uiBufferCount > 0)
                       && (uiBufferCount <= MAX_BUFFER_COUNT)
                       && ((poBufferAttributes != MNULL) || bInternalBuffer))
                      ? LStatus_OK : LStatus_INVALID_PARAM;

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = BufQ_Init(&(poModuleLink->oReturnQueue));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        for (i = 0; (i < MAX_SUBMIT_COUNT); i++)
        {
            eStatus = BufQ_Init(&(poModuleLink->aoSubmitQueue[i]));

            if (LSTATUS_IS_FAIL(eStatus))
            {
                break;
            }
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        poModuleLink->uiTotalTimeUsec       = 0;
        poModuleLink->uiTimedBufferCount    = 0;
        poModuleLink->uiBufferCount         = uiBufferCount;

        for (i = 0; i < MAX_BUFFER_COUNT; i++)
        {
            BufferInfo* poBufferInfo = &(poModuleLink->aoBufferInfo[i]);

            poBufferInfo->hBuffer           = MNULL;
            poBufferInfo->bInternal         = bInternalBuffer;
            poBufferInfo->eFrameType        = FrameType_Unknown;
            poBufferInfo->eNaluType         = LH264_NaluType_UNSPECIFIED;
            poBufferInfo->uiSubmitCount     = 0;
            poBufferInfo->bEndOfStream      = MFALSE;
            poBufferInfo->pvPrivateData     = MNULL;
            poBufferInfo->uiSubPictureCount = MAX_SUBPICTURE_COUNT;
            poBufferInfo->uiCounterLUID     = NO_COUNTER;
            poBufferInfo->uiStartOffset     = 0;
            poBufferInfo->uiSizeBytes       = 0;
        }

        // Create buffers (if not internal) and allocate private data.

        LBuffer_Options oOptions;

        oOptions.uiOptionTypes      = LBuffer_OptionType_AutoSync;
        oOptions.eAutoSyncOption    = ENABLE_AUTO_SYNC ? LBuffer_AutoSyncOption_PerRegion
                                                       : LBuffer_AutoSyncOption_Disabled;
        oOptions.uiAutoSyncTimeout  = LDEVICETHREAD_NO_TIMEOUT;

        for (i = 0; i < uiBufferCount; i++)
        {
            BufferInfo* poBufferInfo = &(poModuleLink->aoBufferInfo[i]);

            if (!bInternalBuffer)
            {
                eStatus = LBuffer_Create(
                            hDevice,
                            (LBuffer_Type*)poBufferAttributes,
                            &(poBufferInfo->hBuffer));

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    eStatus = LBuffer_SetOptions(poBufferInfo->hBuffer, &oOptions);

                    if ( g_uiMsgLogLevel >= 2)
                    {
                        const MCHAR8*   szUnknownName           = "Unknown";
                        const MCHAR8*   pszInternalModuleName   = szUnknownName;
                        if (pszModuleName != MNULL)
                        {
                            pszInternalModuleName = pszModuleName;
                        }

                        MUINT64         uiLUID = 0;
                        eStatus = LBuffer_GetLUID(poBufferInfo->hBuffer, &uiLUID);

                        if (LSTATUS_IS_SUCCESS(poBufferInfo->hBuffer))
                        {
                            MsgLog(
                                        2,
                                        "Module %s created Buffer %d LUID: 0x%lx",
                                        pszInternalModuleName,
                                        i,
                                        uiLUID)
                        }
                        else
                        {
                            MsgLog(
                                        2,
                                        "Module %s create Buffer %d : Fail to get Buffer LUID with status %d",
                                        pszInternalModuleName,
                                        i,
                                        eStatus);
                        }
                    }

                }

                if (LSTATUS_IS_FAIL(eStatus))
                {
                    break;
                }
            }

            if (uiPrivateDataSize > 0)
            {
                poBufferInfo->pvPrivateData = malloc(uiPrivateDataSize);

                if (poBufferInfo->pvPrivateData == MNULL)
                {
                    eStatus = LStatus_OUT_OF_MEMORY;
                    break;
                }
            }

            poBufferInfo->uiId = s_uiBufferId++;
            MsgLog(4, "Created Buffer[%u].", poBufferInfo->uiId);
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        // Put all the buffers into the returned queue.
        for (i = 0; i < uiBufferCount; i++)
        {
            ModLnk_ReleaseBuffer(poModuleLink, &(poModuleLink->aoBufferInfo[i]));
        }
    }
    else
    {
        ModLnk_Cleanup(poModuleLink);
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       ModLnk_Cleanup

Description:    Free all resources attached to this module link.

\************************************************************************************************************/
void ModLnk_Cleanup(ModuleLink* poModuleLink)
{
    MsgLog(2, "{...");

    if (poModuleLink != MNULL)
    {
        BufferInfo* poBufferInfo;
        MINT i;

        // Cancel all submitted buffers.

        ModuleLinkInput oModLnkInput = ModuleLinkInput_Construct;

        oModLnkInput.poModLnk = poModuleLink;

        for (i = 0; i < poModuleLink->uiSubmitCount; i++)
        {
            oModLnkInput.uiSubmitIdx = i;
            ModLnkIn_CancelSubmittedBuffers(&oModLnkInput);
        }

        // Remove all returned buffers.

        MUINT uiRetBufferCount = 0;

        while (LSTATUS_IS_SUCCESS(ModLnk_GetReturnedBuffer(poModuleLink, 0, MNULL, &poBufferInfo)))
        {
            MsgLog(6, "Remove returned buffer[%u]", poBufferInfo->uiId);
            uiRetBufferCount++;
        }

        if (uiRetBufferCount < poModuleLink->uiBufferCount)
        {
            MsgLogErr("WARNING! Only %u buffers on %u has been returned. Continue...",
                      uiRetBufferCount, poModuleLink->uiBufferCount);
        }

        // Destroy buffers and private data

        for (i = 0; i < poModuleLink->uiBufferCount; i++)
        {
            poBufferInfo = &(poModuleLink->aoBufferInfo[i]);

            if (poBufferInfo->pvPrivateData != MNULL)
            {
                free(poBufferInfo->pvPrivateData);
                poBufferInfo->pvPrivateData = MNULL;
            }

            if (!poBufferInfo->bInternal
                && (poBufferInfo->hBuffer != MNULL))
            {
                LBuffer_Destroy(poBufferInfo->hBuffer);
                poBufferInfo->hBuffer = MNULL;
            }
        }

        poModuleLink->uiBufferCount = 0;
        poModuleLink->uiSubmitCount = 0;

        BufQ_Cleanup(&(poModuleLink->oReturnQueue));

        for (i = 0; i < MAX_SUBMIT_COUNT; i++)
        {
            BufQ_Cleanup(&(poModuleLink->aoSubmitQueue[i]));
        }
    }

    MsgLog(2, "...}");
}

/************************************************************************************************************\

Function:       ModLnk_Connect

Description:    Do connection between two modules.

Parameters:     poModuleLink    Module link object owned by the source module
                poModLnkInput   Module link input of the target module.

\************************************************************************************************************/
LStatus ModLnk_Connect(ModuleLink* poModuleLink, ModuleLinkInput* poModLnkInput)
{
    if ((poModuleLink == MNULL)
        || (poModLnkInput == MNULL))
    {
        return LStatus_INVALID_PARAM;
    }

    if ((poModLnkInput->poModLnk == MNULL)
        && (poModuleLink->uiSubmitCount < MAX_SUBMIT_COUNT))
    {
        poModLnkInput->poModLnk    = poModuleLink;
        poModLnkInput->uiSubmitIdx = poModuleLink->uiSubmitCount;
        poModuleLink->uiSubmitCount++;

        if (poModLnkInput->bSyncBefore)
        {
            poModuleLink->bSyncBefore    = MTRUE;
            poModuleLink->uiSyncFraction = poModLnkInput->uiSyncFraction;
        }

        poModLnkInput->uiTotalTimeUsec      = 0;
        poModLnkInput->uiTimedBufferCount   = 0;

        MsgLog(2, "ModLnk[%p]--->ModLnkInput[%u]-[%p]",
                  poModuleLink, poModLnkInput->uiSubmitIdx, poModLnkInput);
    }
    else if (poModLnkInput->poModLnk != poModuleLink)
    {
        return LStatus_FAIL;
    }

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       ModLnk_SubmitBuffer

Description:    Submit a buffer with valid content that is ready to be read by the modules attached to the
                output.

                This function should be called from the module owner of this link.

Paramaters:     poModuleLink    Module link object.
                poBufferInfo    Buffer informations.
                hDevThread      Handle of the Liberatus device thread from where the last writing operation
                                was done and where the synchronization will be done.
                                No synchronization if set to MNULL.
                uiTag           If hDevThread != MNULL and uiTag != NO_TAG, synchronization is done based
                                on this tag.

\************************************************************************************************************/
LStatus ModLnk_SubmitBuffer(
        ModuleLink*          poModuleLink,
        BufferInfo*          poBufferInfo,
        LDeviceThread_Handle hDevThread,
        MUINT64              uiTag)
{
    if ((poModuleLink == MNULL)
        || (poBufferInfo == MNULL))
    {
        return LStatus_INVALID_PARAM;
    }

    if (g_bCheckPerformance)
    {
        poBufferInfo->uiSubmitTimeUsec = GetMonoTimeUsec();
        poBufferInfo->uiReturnTimeUsec = 0;
    }

    MsgLog(6, "Buffer[%u].", poBufferInfo->uiId);

    MBOOL bSyncDone = MFALSE;

    if (ENABLE_AUTO_SYNC && (poModuleLink->bSyncBefore || g_bForceWaitBefore))
    {
        if (poModuleLink->uiSyncFraction > 1)
        {
            LBuffer_PlaneInfo oPlaneInfo;
            LRECT32 oSubRect;

            if (LSTATUS_IS_SUCCESS(LBuffer_GetPlaneInfo(poBufferInfo->hBuffer, &oPlaneInfo)))
            {
                oSubRect.iLeft   = 0;
                oSubRect.iTop    = 0;
                oSubRect.iRight  = oPlaneInfo.auiWidth[0];
                oSubRect.iBottom = (oPlaneInfo.auiHeight[0] + poModuleLink->uiSyncFraction - 1)
                                   / poModuleLink->uiSyncFraction;

                if (LSTATUS_IS_SUCCESS(
                        LBuffer_ExecuteAutoSyncRect(poBufferInfo->hBuffer, MNULL, MTRUE, &oSubRect)))
                {
                    bSyncDone = MTRUE;
                }
            }
        }

        if (!bSyncDone)
        {
            if (LSTATUS_IS_SUCCESS(LBuffer_ExecuteAutoSync(poBufferInfo->hBuffer, MNULL, MTRUE)))
            {
                bSyncDone = MTRUE;
            }
        }
    }

    if (hDevThread != MNULL)
    {
        if (uiTag == NO_TAG)
        {
            LDeviceThread_GetSubmissionTag(hDevThread, &uiTag);
        }

        if (!bSyncDone && (poModuleLink->bSyncBefore || g_bForceWaitBefore))
        {
            const MUINT uiTimeout = 2*1000;

            MUINT64 uiTime = g_bCheckPerformance ? GetMonoTimeUsec() : 0;

            while (MTRUE)
            {
                LStatus eStatus = LDeviceThread_WaitCompletionTag(
                                        hDevThread,
                                        LDEVICETHREAD_DEFAULT_LUID,
                                        uiTag,
                                        uiTimeout);

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    hDevThread  = MNULL;
                    uiTag       = NO_TAG;
                    break;
                }
                else if (eStatus == LStatus_TIMEOUT)
                {
                    MsgLogErr("Wait(Buffer[%u], DevThread= %p, Tag= 0x%lx)...",
                              poBufferInfo->uiId, hDevThread, uiTag);
                }
                else
                {
                    return eStatus;
                }
            }

            if (g_bCheckPerformance)
            {
                poModuleLink->uiTotalTimeUsec += (GetMonoTimeUsec() - uiTime);
                poModuleLink->uiTimedBufferCount++;

                if (poModuleLink->uiTotalTimeUsec >= 1*1000*1000)
                {
                    MsgLog(0, "Took %lu usec to complete %u buffers.",
                              poModuleLink->uiTotalTimeUsec, poModuleLink->uiTimedBufferCount);

                    poModuleLink->uiTimedBufferCount = 0;
                    poModuleLink->uiTotalTimeUsec    = 0;
                }
            }
        }
    }

    if (poModuleLink->bReturnAllBuffer)
    {
        ModLnk_ReleaseBuffer(poModuleLink, poBufferInfo);
        return LStatus_OK;
    }

    poBufferInfo->uiSubmitCount += poModuleLink->uiSubmitCount;

    MUINT i;
    for (i = 0; i < poModuleLink->uiSubmitCount; i++)
    {
        if (hDevThread != MNULL)
        {
            LDeviceThread_AddRef(hDevThread);
        }

        if (!poModuleLink->abStubOutput[i] || poBufferInfo->bEndOfStream)
        {
            BufQ_Push(&(poModuleLink->aoSubmitQueue[i]), poBufferInfo, uiTag, hDevThread);
        }
        else
        {
            BufQ_Push(&(poModuleLink->oReturnQueue), poBufferInfo, uiTag, hDevThread);
        }
    }

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       ModLnk_SyncDevThreads

Description:    Synchronize two device threads.

Parameters:     hDevThreadDst   Synchronize this device thread. If MNULL, synchronize the cpu.
                hDevThreadSrc   Synchronize from this device thread...
                uiCounterLuid   .. or this counter if not equal to NO_COUNTER...
                uiTag           ...at this submission tag.

\************************************************************************************************************/
static LStatus ModLnk_SyncDevThreads(
            LDeviceThread_Handle hDevThreadDst,
            LDeviceThread_Handle hDevThreadSrc,
            MUINT64              uiCounterLuid,
            MUINT64              uiTag,
            MUINT                uiBufferId)
{
    LStatus eStatus = LStatus_OK;

    if (!ENABLE_AUTO_SYNC && (uiTag != NO_TAG))
    {
        const MUINT uiTimeout = 2*1000;

        do
        {
            if (eStatus == LStatus_TIMEOUT)
            {
                MsgLogErr("Wait(Buffer[%u], DtDst= %p, DtSrc= %p, CLuid= %lx, Tag= 0x%lx)...",
                          uiBufferId, hDevThreadDst, hDevThreadSrc, uiCounterLuid, uiTag);
            }

            if (hDevThreadDst == MNULL)
            {
                if (hDevThreadSrc != MNULL)
                {
                    eStatus = LDeviceThread_WaitCompletionTag(
                                    hDevThreadSrc,
                                    LDEVICETHREAD_DEFAULT_LUID,
                                    uiTag,
                                    uiTimeout);
                }
            }
            else if (uiCounterLuid == NO_COUNTER)
            {
                if ((hDevThreadSrc != MNULL)
                    && (hDevThreadSrc != hDevThreadDst))
                {
                    MUINT64 uiDevThreadSrcLuid;

                    eStatus = LDeviceThread_GetLUID(hDevThreadSrc, &uiDevThreadSrcLuid);

                    if (LSTATUS_IS_SUCCESS(eStatus))
                    {
                        eStatus = LDeviceThread_SyncWithSubmission(
                                        hDevThreadDst,
                                        uiDevThreadSrcLuid,
                                        uiTag,
                                        MFALSE,
                                        MFALSE,
                                        uiTimeout);
                    }
                }
            }
            else
            {
                eStatus = LDeviceThread_SyncWithCounter(
                                hDevThreadDst,
                                uiCounterLuid,
                                uiTag,
                                MFALSE,
                                MFALSE,
                                uiTimeout);
            }

        } while (eStatus == LStatus_TIMEOUT);
    }

    if (LSTATUS_IS_FAIL(eStatus))
    {
        MsgLogErr(
            "ERROR! Waiting tag: DtDst= %p, DtSrc= %p, CLuid= %lx, Tag= 0x%lx, status= %d (%s). Continue...",
            hDevThreadDst,
            hDevThreadSrc,
            uiCounterLuid,
            uiTag,
            eStatus,
            GetStatusStr(eStatus));
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       ModLnkIn_GetSubmittedBuffer

Description:    Get a buffer previously submitted.

                This function should be called from the module(s) attached to the output of this link.

Paramaters:     poModLnkInput       Module link input.
                uiQueueTimeoutMsec  If the queue is empty, wait until a new buffer is submitted until this
                                    timeout expired. No waiting when set to 0.
                uiSubPicture        Sub-picture index (0 = No sub-picture).
                ppoBufferInfo       Returns a pointer on the buffer information.
                pbDtSynched         If not NULL, returns MTRUE if a synchronization has been submitted.
                puiQueueSize        If not NULL, returns the number of buffers in queue.

\************************************************************************************************************/
LStatus ModLnkIn_GetSubmittedBuffer(
            ModuleLinkInput*     poModLnkInput,
            MUINT                uiQueueTimeoutMsec,
            MUINT                uiSubPicture,
            LDeviceThread_Handle hDevThreadDst,
            BufferInfo**         ppoBufferInfo,
            MBOOL*               pbDtSynched,
            MUINT*               puiQueueSize)
{
    if ((poModLnkInput == MNULL)
        || (poModLnkInput->poModLnk == MNULL)
        || (poModLnkInput->uiSubmitIdx >= poModLnkInput->poModLnk->uiSubmitCount)
        || (ppoBufferInfo == MNULL))
    {
        return LStatus_INVALID_PARAM;
    }

    const MUINT64 uiStartTimeUsec = (!poModLnkInput->bSkipAll && (poModLnkInput->uiSkipFrameCount == 0))
                                    ? 0 : GetMonoTimeUsec();

    if (pbDtSynched != MNULL)
    {
        *pbDtSynched = MFALSE;
    }

    while (MTRUE)
    {
        MUINT64              uiQTime        = g_bCheckPerformance ? GetMonoTimeUsec() : 0;
        MUINT64              uiTag          = NO_TAG;
        LDeviceThread_Handle hDevThreadSrc  = MNULL;
        PopData              oPopData;
        BufferQueue* poQueue = &(poModLnkInput->poModLnk->aoSubmitQueue[poModLnkInput->uiSubmitIdx]);

        *ppoBufferInfo = MNULL;

        if (BufQ_Pop(poQueue, uiQueueTimeoutMsec, &oPopData))
        {
            uiTag           = oPopData.oElement.uiTag;
            hDevThreadSrc   = oPopData.oElement.hDeviceThread;
            *ppoBufferInfo  = oPopData.oElement.poBufferInfo;

            if (puiQueueSize != MNULL)
            {
                *puiQueueSize = oPopData.uiQueueSize;
            }
        }

        if (*ppoBufferInfo != MNULL)
        {
            MUINT64 uiDtTime = g_bCheckPerformance ? GetMonoTimeUsec() : 0;

            if (hDevThreadSrc != MNULL)
            {
                if (uiTag != NO_TAG)
                {
                    MUINT64 uiCounterLuid = (*ppoBufferInfo)->uiCounterLUID;

                    if ((uiCounterLuid != NO_COUNTER)
                        && (hDevThreadDst != MNULL))
                    {
                        if ((uiSubPicture > 0)
                            && (uiSubPicture < (*ppoBufferInfo)->uiSubPictureCount))
                        {
                            uiTag -= MAX_SUBPICTURE_COUNT
                                     - ((uiSubPicture * MAX_SUBPICTURE_COUNT)
                                        / (*ppoBufferInfo)->uiSubPictureCount);
                        }
                    }

                    ModLnk_SyncDevThreads(hDevThreadDst, hDevThreadSrc, uiCounterLuid, uiTag,
                                          (*ppoBufferInfo)->uiId);

                    if (pbDtSynched != MNULL)
                    {
                        *pbDtSynched = MTRUE;
                    }
                }

                LDeviceThread_UnRef(hDevThreadSrc);
            }

            if (g_bCheckPerformance)
            {
                MUINT64 uiCurTime = GetMonoTimeUsec();

                if (uiQTime > (*ppoBufferInfo)->uiSubmitTimeUsec)
                {
                    MsgLog(2, "Got Buffer[%u]: In Q for %lu usec, Wait DT for %lu usec.",
                              (*ppoBufferInfo)->uiId, uiDtTime - (*ppoBufferInfo)->uiSubmitTimeUsec,
                              uiCurTime - uiDtTime);
                }
                else
                {
                    MsgLog(2, "Got Buffer[%u]: Wait Q for %lu usec, Wait DT for %lu usec.",
                              (*ppoBufferInfo)->uiId, uiDtTime - uiQTime, uiCurTime - uiDtTime);
                }
            }
            else
            {
                MsgLog(6, "Got Buffer[%u] (status= 0).", (*ppoBufferInfo)->uiId);
            }

            if (!poModLnkInput->bSkipAll && (poModLnkInput->uiSkipFrameCount == 0))
            {
                poModLnkInput->uiSkipFrameCount = poModLnkInput->uiRateDivisor - 1;
                return LStatus_OK;
            }
            else
            {
                MsgLog(4, "Skip buffer[%u].", (*ppoBufferInfo)->uiId);

                ModLnkIn_ReturnBuffer(poModLnkInput, *ppoBufferInfo, MNULL, NO_TAG);
                if (poModLnkInput->uiSkipFrameCount > 0)
                {
                    poModLnkInput->uiSkipFrameCount--;
                }
                *ppoBufferInfo = MNULL;

                MUINT uiElapseTimeMsec = (GetMonoTimeUsec() - uiStartTimeUsec) / 1000;
                uiQueueTimeoutMsec -= min(uiQueueTimeoutMsec, uiElapseTimeMsec);

                if ((uiQueueTimeoutMsec == 0) && (oPopData.uiQueueSize == 0))
                {
                    return LStatus_TIMEOUT;
                }
            }
        }
        else
        {
            if (g_bCheckPerformance && (uiQueueTimeoutMsec > 0))
            {
                MsgLog(2, "TIMEOUT! (%lu usec)", GetMonoTimeUsec() - uiQTime);
            }

            return LStatus_TIMEOUT;
        }
    }
}

/************************************************************************************************************\

Function:       ModLnkIn_InspectSubmittedBuffer

Description:    Get oldest submitted buffer without removing it from que queue.

                This function should be called from the module(s) attached to the output of this link.

Paramaters:     poModLnkInput       Module link input.
                ppoBufferInfo       If not NULL, returns a pointer on the oldest submitted buffer information.
                puiQueueSize        If not NULL, returns the number of buffers in queue.
                puiQueueLatencyUsec If not NULL, returns the queue latency in microseconds.

\************************************************************************************************************/
LStatus ModLnkIn_InspectSubmittedBuffer(
    ModuleLinkInput*    poModLnkInput,
    BufferInfo**        ppoBufferInfo,
    MUINT*              puiQueueSize)
{
    if((poModLnkInput == MNULL)
        || (poModLnkInput->poModLnk == MNULL))
    {
        return LStatus_INVALID_PARAM;
    }

    LStatus         eStatus     = LStatus_FAIL;
    BufferQueue*    poQueue     = &(poModLnkInput->poModLnk->aoSubmitQueue[poModLnkInput->uiSubmitIdx]);
    PopData         oPopData;

    if (BufQ_Peek(poQueue, &oPopData))
    {
        if(ppoBufferInfo != MNULL)
        {
            *ppoBufferInfo = oPopData.oElement.poBufferInfo;
        }

        if(puiQueueSize != MNULL)
        {
            *puiQueueSize = oPopData.uiQueueSize;
        }

        eStatus = LStatus_OK;
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       ModLnkIn_ReturnBuffer

Description:    Return the buffer to the owner when not needed anymore.

                This function should be called from the module(s) attached to the output of this link.

Paramaters:     poModLnkInput   Module link input.
                poBufferInfo    Buffer informations.
                hDevThread      Handle of the Liberatus device thread from where the last reading operation
                                was done and where the synchronization will be done.
                                No synchronization if set to MNULL.
                uiTag           If hDevThread != MNULL and uiTag != NO_TAG, synchronization is done based
                                on this tag.

\************************************************************************************************************/
LStatus ModLnkIn_ReturnBuffer(
            ModuleLinkInput*     poModLnkInput,
            BufferInfo*          poBufferInfo,
            LDeviceThread_Handle hDevThread,
            MUINT64              uiTag)
{
    if ((poModLnkInput == MNULL)
        || (poModLnkInput->poModLnk == MNULL)
        || (poBufferInfo == MNULL))
    {
        return LStatus_INVALID_PARAM;
    }

    if (g_bCheckPerformance && (poBufferInfo->uiReturnTimeUsec == 0))
    {
        poBufferInfo->uiReturnTimeUsec = GetMonoTimeUsec();
    }

    MsgLog(6, "Buffer[%u].", poBufferInfo->uiId);

    if (ENABLE_AUTO_SYNC && g_bForceWaitBefore)
    {
        LBuffer_ExecuteAutoSync(poBufferInfo->hBuffer, hDevThread, MFALSE);
    }

    if (hDevThread != MNULL)
    {
        LDeviceThread_AddRef(hDevThread);
        if (uiTag == NO_TAG)
        {
            LDeviceThread_GetSubmissionTag(hDevThread, &uiTag);
        }

        if (g_bForceWaitBefore)
        {
            const MUINT uiTimeout = 2*1000;

            MUINT64 uiTime = g_bCheckPerformance ? GetMonoTimeUsec() : 0;

            while (MTRUE)
            {
                LStatus eStatus = LDeviceThread_WaitCompletionTag(
                                        hDevThread,
                                        LDEVICETHREAD_DEFAULT_LUID,
                                        uiTag,
                                        uiTimeout);

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    hDevThread  = MNULL;
                    uiTag       = NO_TAG;
                    break;
                }
                else if (eStatus == LStatus_TIMEOUT)
                {
                    MsgLogErr("Wait(Buffer[%u], DevThread= %p, Tag= 0x%lx)...",
                              poBufferInfo->uiId, hDevThread, uiTag);
                }
                else
                {
                    return eStatus;
                }
            }

            if (g_bCheckPerformance)
            {
                poModLnkInput->uiTotalTimeUsec += (GetMonoTimeUsec() - uiTime);
                poModLnkInput->uiTimedBufferCount++;

                if (poModLnkInput->uiTotalTimeUsec >= 1*1000*1000)
                {
                    MsgLog(0, "Took %lu usec to complete %u buffers.",
                              poModLnkInput->uiTotalTimeUsec, poModLnkInput->uiTimedBufferCount);

                    poModLnkInput->uiTimedBufferCount = 0;
                    poModLnkInput->uiTotalTimeUsec    = 0;
                }
            }
        }
    }

    BufQ_Push(&(poModLnkInput->poModLnk->oReturnQueue), poBufferInfo, uiTag, hDevThread);

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       ModLnk_GetReturnedBuffer

Description:    Get a buffer previously returned. The buffer is free and can be used for a new operation.

                This function should be called from the module owner of this link.

Paramaters:     poModuleLink        Module link object.
                uiQueueTimeoutMsec  If the queue is empty, wait until a new buffer is returned until this
                                    timeout expired. No waiting when set to 0.
                hDevThreadDst       Device thread on wich the returned buffer must be synchronized.
                ppoBufferInfo       Return a pointer on the buffer informations.

\************************************************************************************************************/
LStatus ModLnk_GetReturnedBuffer(
            ModuleLink*          poModuleLink,
            MUINT                uiQueueTimeoutMsec,
            LDeviceThread_Handle hDevThreadDst,
            BufferInfo**         ppoBufferInfo)
{
    if ((poModuleLink == MNULL)
        || (ppoBufferInfo == MNULL))
    {
        return LStatus_INVALID_PARAM;
    }

    PopData oPopData;

    *ppoBufferInfo = MNULL;

    do
    {
        MUINT64 uiQTime = g_bCheckPerformance ? GetMonoTimeUsec() : 0;

        if (!BufQ_Pop(
                    &(poModuleLink->oReturnQueue),
                    uiQueueTimeoutMsec,
                    &oPopData))
        {
            if (g_bCheckPerformance && (uiQueueTimeoutMsec > 0))
            {
                MsgLog(2, "TIMEOUT! (%lu usec)", GetMonoTimeUsec() - uiQTime);
            }
            else
            {
                MsgLog(8, "(status= TIMEOUT)");
            }

            return LStatus_TIMEOUT;
        }

        MUINT64 uiDtTime = g_bCheckPerformance ? GetMonoTimeUsec() : 0;

        if (oPopData.oElement.hDeviceThread != MNULL)
        {
            if (oPopData.oElement.uiTag != NO_TAG)
            {
                ModLnk_SyncDevThreads(
                    hDevThreadDst,
                    oPopData.oElement.hDeviceThread,
                    NO_COUNTER,
                    oPopData.oElement.uiTag,
                    oPopData.oElement.poBufferInfo->uiId);
            }

            LDeviceThread_UnRef(oPopData.oElement.hDeviceThread);
        }

        if (oPopData.oElement.poBufferInfo->uiSubmitCount > 0)
        {
            oPopData.oElement.poBufferInfo->uiSubmitCount--;
        }

        if (g_bCheckPerformance)
        {
            MUINT64 uiCurTime = GetMonoTimeUsec();

            if (uiQTime > oPopData.oElement.poBufferInfo->uiReturnTimeUsec)
            {
                MsgLog(2, "Got Buffer[%u]: SubCnt= %u, In Q for %lu usec, Wait DT for %lu usec.",
                          oPopData.oElement.poBufferInfo->uiId, oPopData.oElement.poBufferInfo->uiSubmitCount,
                          uiDtTime - oPopData.oElement.poBufferInfo->uiReturnTimeUsec, uiCurTime - uiDtTime);
            }
            else
            {
                MsgLog(1, "Got Buffer[%u]: SubCnt= %u, Wait Q for %lu usec, Wait DT for %lu usec.",
                          oPopData.oElement.poBufferInfo->uiId, oPopData.oElement.poBufferInfo->uiSubmitCount,
                          uiDtTime - uiQTime, uiCurTime - uiDtTime);
            }

        }

    } while (oPopData.oElement.poBufferInfo->uiSubmitCount > 0);

    *ppoBufferInfo = oPopData.oElement.poBufferInfo;

    MsgLog(6, "Got Buffer[%u] (status = 0).", (*ppoBufferInfo)->uiId);

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       ModLnk_ReleaseBuffer

Description:    Return the buffer without doing any processing on it.

                This function should be called from the module owner of this link.

\************************************************************************************************************/
MBOOL ModLnk_ReleaseBuffer(
        ModuleLink* poModuleLink,
        BufferInfo* poBufferInfo)
{
    if ((poModuleLink != MNULL) && (poBufferInfo != MNULL))
    {
        if (g_bCheckPerformance && (poBufferInfo->uiReturnTimeUsec == 0))
        {
            poBufferInfo->uiReturnTimeUsec = GetMonoTimeUsec();
        }

        MsgLog(6, "Buffer[%u]", poBufferInfo->uiId);

        poBufferInfo->uiSubmitCount = 1;
        return BufQ_Push(&(poModuleLink->oReturnQueue), poBufferInfo, NO_TAG, MNULL);
    }
    else
    {
        return MFALSE;
    }
}

/************************************************************************************************************\

Function:       ModLnk_ShowState

\************************************************************************************************************/
void ModLnk_ShowState(ModuleLink* poModuleLink)
{
    MUINT i;

    printf("  Buffer count:   %u\n", poModuleLink->uiBufferCount);

    for (i = 0; i < poModuleLink->uiSubmitCount; i++)
    {
        printf("  SubmitQueue[%u]: %u buffers\n", i, BufQ_GetSize(&poModuleLink->aoSubmitQueue[i]));
    }

    printf("  ReturnQueue:    %u buffers\n", BufQ_GetSize(&poModuleLink->oReturnQueue));
}

/************************************************************************************************************\

Function:       ModLnkIn_CancelSubmittedBuffers

\************************************************************************************************************/
void ModLnkIn_CancelSubmittedBuffers(ModuleLinkInput* poModLnkInput)
{
    if((poModLnkInput != MNULL)
        && (poModLnkInput->poModLnk != MNULL))
    {
        BufferInfo* poBufferInfo    = MNULL;
        LStatus     eStatus         = LStatus_OK;

        while (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = ModLnkIn_GetSubmittedBuffer(
                                    poModLnkInput,
                                    0,
                                    0,
                                    MNULL,
                                    &poBufferInfo,
                                    MNULL,
                                    MNULL);

            if(LSTATUS_IS_SUCCESS(eStatus))
            {
                ModLnkIn_ReturnBuffer(poModLnkInput, poBufferInfo, MNULL, NO_TAG);
            }
        }
    }
}
