/************************************************************************************************************\

Module Name:    CommonUtils.h

Description:    Utility functions common to all samples.

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

#ifndef INC_CommonUtils_H
#define INC_CommonUtils_H

// -----------------------------------------------------------------------------------------------------------
//                                   I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "Liberatus.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#ifdef LINUX
#include <termios.h>
#include <unistd.h>
#include <pthread.h>
#include <errno.h>
#include <sys/prctl.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <linux/fb.h>
#else
#include <pthread.h>
#endif
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <time.h>

#ifdef _WIN32
#include <assert.h>
#endif

#ifdef ALSA_SUPPORT
#include "alsa/asoundlib.h"
#endif

// -----------------------------------------------------------------------------------------------------------
//                                   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

#define NO_TAG      LDEVICETHREAD_DEFAULT_TAG
#define NO_COUNTER  LDEVICETHREAD_DEFAULT_LUID

#ifndef min
    #define min(a, b)   ((a) < (b) ? (a) : (b))
#endif
#ifndef max
    #define max(a, b)   ((a) > (b) ? (a) : (b))
#endif

#ifdef _WIN32
 #ifndef __func__
  #define __func__ __FUNCTION__
 #endif
#endif

// Message log macros
#if 1
    #define MsgLog(level, str, ...) \
        if (level <= g_uiMsgLogLevel) MsgLogExt(__FILE__, __func__, __LINE__, str, ##__VA_ARGS__);
#else
    #define MsgLog(...)
#endif

// This message log will be active even in release mode.
#define MsgLogErr(str, ...) MsgLogExt(__FILE__, __func__, __LINE__, str, ##__VA_ARGS__);

// Align the given value on the given power of 2 value.
#define AlignPow2(n, a) ((n + a - 1) & ~(a - 1))

typedef struct LMUTEXT_OBJECT*    LMutext_Handle;
typedef struct LCONDITION_OBJECT* LCondition_Handle;
typedef struct LCPUTHREAD_OBJECT* LCpuThread_Handle;

typedef LStatus (*LCpuThread_Function)(void* poRef);

static const MBOOL sg_bEnableLatencyProfile  = MFALSE;
#ifdef _WIN32
  #define sg_uiLowLatencySliceCount 4
#else
  static const MUINT sg_uiLowLatencySliceCount = 4;
#endif
static const MUINT sg_uiLowLatencySliceAlign = 24;

#ifndef LDEVICETHREAD_TICK_COUNTER_FREQUENCY
  #define LDEVICETHREAD_TICK_COUNTER_FREQUENCY (27*1000*1000)
#endif

// -----------------------------------------------------------------------------------------------------------
//                  G L O B A L   V A R I A B L E / F U N C T I O N   R E F E R E N C E S
// -----------------------------------------------------------------------------------------------------------

extern MUINT g_uiMsgLogLevel;

int GetChar();
MUINT64 GetMonoTimeUsec();
MUINT64 GetMonoTimeNsec();
void MsgLogExt(const char *szFile, const char* szFnName, MUINT uiLine, const char* szFormat, ...);
void SetMsgLogLevel(MUINT uiLevel);
MUINT GetNextH264StartCode(const MUINT8* puiData, MUINT uiSizeBytes, MUINT* puiPreZeroCount);
LStatus SetVideoCodecFwTraceLevel(LDevice_Handle hDev, MUINT uiLevel);
LStatus ClearBuffer(LDevice_Handle hDev, LBuffer_Handle hBuffer, LColor *poColor);
MUINT GetCpuUsage(MUINT64* puiPrevBusyTime, MUINT64* puiPrevTotalTime);
MUINT GetMemUsage();
MBOOL LPixelFormat_IsRgb(LPixelFormat ePixFmt);
MUINT LPixelFormat_Get3xBppAllPlanes(LPixelFormat ePixelFormat);

LStatus LCpuThread_Create(LCpuThread_Function pfnThread, void* poRef, LCpuThread_Handle* phThread);
LStatus LCpuThread_Destroy(LCpuThread_Handle hCpuThread, LStatus* peRetStatus);
LStatus LMutex_Create(LMutext_Handle* phMutex);
LStatus LMutex_Destroy(LMutext_Handle hMutex);
LStatus LMutex_Lock(LMutext_Handle hMutex);
LStatus LMutex_TryLock(LMutext_Handle hMutex);
LStatus LMutex_Unlock(LMutext_Handle hMutex);
LStatus LCondition_Create(LCondition_Handle *phCond);
LStatus LCondition_Destroy(LCondition_Handle hCond);
LStatus LCondition_Signal(LCondition_Handle hCond);
LStatus LCondition_Wait(LCondition_Handle hCond, LMutext_Handle hMutex, MUINT32 uiTimeoutMilliSec);

void GetRectOnEllipsoidalPath(
        const MUINT     uiAnglePos1000x,
        const LSIZE*    poBackSize,
        const LSIZE*    poBorderSize,
        const MUINT     uiCornerFactor,
              LRECT32*  poDstRect);
LStatus LDeviceThread_SyncDt(
            LDeviceThread_Handle hDstDt,
            LDeviceThread_Handle hSrcDt,
            volatile MBOOL*      pbStopWaiting);

#ifdef _WIN32
MBOOL32 usleep(long usec);
int c99_snprintf(char* str, size_t size, const char* format, ...);
#define snprintf c99_snprintf
#endif //_WIN32

#ifdef ALSA_SUPPORT
LStatus EnumAlsaDeviceNames(
            LSTR64*                 aszDevNames,
            MUINT32                 uiMaxNbDev,
            MUINT32*                puiNbDev,
            MBOOL32                 bCapture);

LStatus CheckAlsaDeviceCapabilities(
            snd_pcm_t*              hAlsaDev,
            snd_pcm_hw_params_t*    poHwParams,
            MUINT32                 uiSamplingRate,
            MUINT32                 uiNbChannels,
            MUINT32                 uiSampleSize,
            MBOOL32                 bBigEndian,
            MBOOL32                 bUnsigned,
            snd_pcm_format_t*       peFormat);

LStatus ConfigureAlsaDevice(
            snd_pcm_t*      hAlsaDev,
            MUINT32         uiSamplingRate,
            MUINT32         uiNbChannels,
            MUINT32         uiSampleSize,
            MBOOL32         bBigEndian,
            MBOOL32         bUnsigned,
            MUINT32         uiFramesPerBuffer);
#endif

const MCHAR8*   GetStatusStr(LStatus eStatus);
char* strncat_wz(char* szDst, const char* szSrc, size_t uiDstSize);
char* strncpy_wz(char* szDst, const char* szSrc, size_t uiDstSize);

#endif // INC_CommonUtils_H
