/************************************************************************************************************\

Module Name:    CommonUtils.c

Description:    Utility functions common to all samples.

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include <pthread.h>
#include <stdio.h>
#include <errno.h>

#ifdef LINUX
 #include <sys/time.h>
 #include <unistd.h>
 #include <termios.h>
#endif
#ifdef _WIN32
  #include  <conio.h>
 #include <windows.h>
 #define _USE_MATH_DEFINES //for M_PI
 #include "psapi.h"        //GetProcessMemoryInfo()
#endif
#include <math.h>
#include "CommonUtils.h"
#include "LH264VideoCodec.h"
#include "LBlit.h"
#include "ModuleLink.h"

#ifdef ALSA_SUPPORT
 #include "alsa/asoundlib.h"
#endif

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

typedef void* (*PosixThread_Function)(void *pRef);

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

MUINT g_uiMsgLogLevel   = 0;
#ifdef LINUX
 MBOOL g_bShowThreadName = MTRUE;
#else
 MBOOL g_bShowThreadName = MFALSE;
#endif
MBOOL g_bShowTimestamp  = MTRUE;
MBOOL g_bShowFileName   = MFALSE;
MBOOL g_bShowFunction   = MTRUE;
MBOOL g_bShowLineNumber = MTRUE;

#ifdef _WIN32
int clock_gettime(int X, struct timespec* tv);
int  gettimeofday(struct timeval * tp, struct timezone * tzp);
#define vsnprintf c99_vsnprintf
int c99_vsnprintf(char* str, size_t size, const char* format, va_list ap);

#ifndef CLOCK_MONOTONIC
#define CLOCK_MONOTONIC 0
#endif

#endif //_WIN32



/************************************************************************************************************\

Function: GetMonoTimeUsec

Description: Gets the current time and returns it as an MUINT64


Parameters: None

Return Value:  the current time

\************************************************************************************************************/
MUINT64 GetMonoTimeUsec()
{
    struct timespec oTime;

    clock_gettime(CLOCK_MONOTONIC, &oTime);

    return (MUINT64)oTime.tv_sec * 1000000 + (oTime.tv_nsec / 1000);
}


/************************************************************************************************************\

Function:       GetMonoTimeNsec

Description:    Same as GetMonoTimeUsec but in nanoseconds.

\************************************************************************************************************/
MUINT64 GetMonoTimeNsec()
{
    struct timespec oTime;

    clock_gettime(CLOCK_MONOTONIC, &oTime);

    return (MUINT64)oTime.tv_sec * 1000 * 1000 * 1000 + oTime.tv_nsec;
}

/************************************************************************************************************\

Function:       SetMsgLogLevel

Description:    Set message log level. Every message with the level below or equal to the given level will
                be printed into the consol (see MsgLog macro).

\************************************************************************************************************/
void SetMsgLogLevel(MUINT uiLevel)
{
    g_uiMsgLogLevel = uiLevel;
}

/************************************************************************************************************\

Function:       MsgLogExt

Description:    .

\************************************************************************************************************/
void MsgLogExt(const char* szFile, const char* szFnName, MUINT uiLine, const char* szFormat, ...)
{
    va_list arglist;
    char    szTmp[128];
    char    szMessage[256];

    strncpy_wz(szMessage, "LSMPL", sizeof(szMessage));

    if (g_bShowTimestamp)
    {
        snprintf(szTmp, sizeof(szTmp), ":%010lu", GetMonoTimeUsec());
        strncat_wz(szMessage, szTmp, sizeof(szMessage));
    }

    if (g_bShowThreadName)
    {
        char szThreadName[16];
      #ifdef LINUX
        prctl(PR_GET_NAME, szThreadName, 0, 0, 0);
      #else
        OutputDebugString("prctl(PR_GET_NAME, szName, 0, 0, 0); not supported on windows");
      #endif
        snprintf(szTmp, sizeof(szTmp), ":%-15s", szThreadName);
        strncat_wz(szMessage, szTmp, sizeof(szMessage));
    }

    if (g_bShowFileName)
    {
        const char* szFileExt = strrchr(szFile, '/');

        if (szFileExt == MNULL)
        {
            szFileExt = szFile;
        }
        else
        {
            szFileExt++;
        }

        snprintf(szTmp, sizeof(szTmp), ":%-22s", szFileExt);
        strncat_wz(szMessage, szTmp, sizeof(szMessage));
    }

    if (g_bShowFunction)
    {
        snprintf(szTmp, sizeof(szTmp), ":%-28s", szFnName);
        strncat_wz(szMessage, szTmp, sizeof(szMessage));
    }

    if (g_bShowLineNumber)
    {
        snprintf(szTmp, sizeof(szTmp), "@%04u", uiLine);
        strncat_wz(szMessage, szTmp, sizeof(szMessage));
    }

    strncat_wz(szMessage, ": ", sizeof(szMessage));

    MUINT uiOffset = strlen(szMessage);

    va_start(arglist, szFormat);
    vsnprintf(szMessage + uiOffset, sizeof(szMessage) - uiOffset, szFormat, arglist);
    va_end(arglist);

    strncat_wz(szMessage, "\n", sizeof(szMessage));

    fprintf(stdout, "%s", szMessage);
    fflush(stdout);
}

/************************************************************************************************************\

Function:       GetNextH264StartCode

Description:    Get the next start code (0:0:1) in the given H264 stream.

\************************************************************************************************************/
MUINT GetNextH264StartCode(const MUINT8* puiData, MUINT uiSizeBytes, MUINT* puiPreZeroCount)
{
    // Sometimes, we have already some zeros taken at the end of a previous buffer.
    if ((puiPreZeroCount != MNULL)
        && (*puiPreZeroCount > 0)
        && (uiSizeBytes > 0))
    {
        if (puiData[0] == 0)
        {
            if ((uiSizeBytes > 1)
                && (puiData[1] == 1))
            {
                *puiPreZeroCount = 1;
                return 0;
            }
        }
        else if ((puiData[0] == 1)
                 && (*puiPreZeroCount > 1))
        {
            *puiPreZeroCount = 2;
            return 0;
        }

        *puiPreZeroCount = 0;
    }

    MUINT uiOffset = 2;

    while (uiOffset < uiSizeBytes)
    {
        if (puiData[uiOffset] == 0)
        {
            // Maybe next byte is 1?
            uiOffset++;
        }
        else
        {
            if ((puiData[uiOffset] == 1)
                && (puiData[uiOffset - 1] == 0)
                && (puiData[uiOffset - 2] == 0))
            {
                return uiOffset - 2;
            }

            // Can jump 3 bytes forward.
            uiOffset += 3;
        }
    }

    return uiSizeBytes;
}

/************************************************************************************************************\

Function:       SetVideoCodecFwTraceLevel

Description:    Set the trace level (0 = disabled) of the Video codec firmware.

\************************************************************************************************************/
LStatus SetVideoCodecFwTraceLevel(LDevice_Handle hDev, MUINT uiLevel)
{
    LH264_TraceOptions hTraceOptions = MNULL;

    LStatus eStatus = LH264_CreateFirmwareTraceOptions(hDev, &hTraceOptions);

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LH264_SetFirmwareGlobalTraceLevel(hTraceOptions, uiLevel);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LH264_ApplyFirmwareTraceOptions(hTraceOptions);
        }

        LH264_DestroyFirmwareTraceOptions(hTraceOptions);
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       ClearBuffer

Description:    Fill the whole buffer with the given color.

\************************************************************************************************************/
LStatus ClearBuffer(LDevice_Handle hDev, LBuffer_Handle hBuffer, LColor *poColor)
{
    LDeviceThread_Handle hDevThread;

    LDeviceThread_CreateAttributes oAttributes = {LDeviceThread_AttributeType_CREATE, MTRUE, MTRUE, MFALSE, "clr"};

    LStatus eStatus = LDeviceThread_Create(hDev, &oAttributes.eType, &hDevThread);

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        LBlit_Handle hBlit;

        eStatus = LBlit_Create(hDev, hDevThread, MNULL, &hBlit);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LBlit_SolidFill(hBlit, hBuffer, poColor, MNULL, MNULL, LBlit_Bop_S, MNULL);

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                LDeviceThread_WaitCompletionTag(
                    hDevThread,
                    LDEVICETHREAD_DEFAULT_LUID,
                    LDEVICETHREAD_DEFAULT_TAG,
                    1000);
            }

            while (LBlit_Destroy(hBlit) == LStatus_COMMAND_PENDING)
            {
                usleep(1000);
            }
        }

        LDeviceThread_Destroy(hDevThread);
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       GetCpuUsage

Description:    Return the % of cpu usage x10.

\************************************************************************************************************/
#ifdef LINUX
MUINT GetCpuUsage(MUINT64* puiPrevBusyTime, MUINT64* puiPrevTotalTime)
{
    MUINT uiCpuUsage10x = 0;
    FILE* hFile         = fopen("/proc/stat", "r");

    if (hFile != NULL)
    {
        MUINT64 uiUser, uiNice, uiSys, uiIdle, uiIoWait, uiIrq, uiSoftIrq, uiSteal;
        char    szTextBuf[6];

        while (fscanf(hFile, "%4s %lu %lu %lu %lu %lu %lu %lu %lu",
                             szTextBuf, &uiUser, &uiNice, &uiSys, &uiIdle,
                             &uiIoWait, &uiIrq, &uiSoftIrq, &uiSteal) != EOF)
        {
            if (strcmp(szTextBuf, "cpu") == 0)
            {
                MUINT64 uiBusyTime  = uiUser + uiNice + uiSys + uiIoWait + uiIrq + uiSoftIrq + uiSteal;
                MUINT64 uiTotalTime = uiBusyTime + uiIdle;

                if (uiTotalTime > *puiPrevTotalTime)
                {
                    uiCpuUsage10x = (1000 * (uiBusyTime - *puiPrevBusyTime))
                                    / (uiTotalTime - *puiPrevTotalTime);
                }

                *puiPrevBusyTime  = uiBusyTime;
                *puiPrevTotalTime = uiTotalTime;

                break;
            }
        }

        fclose(hFile);
    }

    return uiCpuUsage10x;
}
#endif

#ifdef _WIN32
MUINT GetCpuUsage(MUINT64* puiPrevBusyTime, MUINT64* puiPrevTotalTime)
{
    MUINT uiCpuUsage10x = 0;

    static MBOOL bIsInitialized = MFALSE;

    static ULARGE_INTEGER s_ulLastCPU, s_ulLastSysCPU, s_ulLastUserCPU;
    static int s_uiNumProcessors;
    FILETIME ftime, fsys, fuser;
    ULARGE_INTEGER now, sys, user;
    double percent;

    if (!bIsInitialized)
    {
        SYSTEM_INFO oSysInfo;
        FILETIME ftime, fsys, fuser;

        GetSystemInfo(&oSysInfo);
        s_uiNumProcessors = oSysInfo.dwNumberOfProcessors;

        GetSystemTimeAsFileTime(&ftime);
        memcpy(&s_ulLastCPU, &ftime, sizeof(FILETIME));

        GetProcessTimes(GetCurrentProcess(), &ftime, &ftime, &fsys, &fuser);
        memcpy(&s_ulLastSysCPU, &fsys, sizeof(FILETIME));
        memcpy(&s_ulLastUserCPU, &fuser, sizeof(FILETIME));

        bIsInitialized = MTRUE;
    }

    GetSystemTimeAsFileTime(&ftime);
    memcpy(&now, &ftime, sizeof(FILETIME));

    if (GetProcessTimes(GetCurrentProcess(), &ftime, &ftime, &fsys, &fuser)
       && (now.QuadPart != s_ulLastCPU.QuadPart))
    {

        memcpy(&sys, &fsys, sizeof(FILETIME));
        memcpy(&user, &fuser, sizeof(FILETIME));

        percent = (sys.QuadPart - s_ulLastSysCPU.QuadPart) +
            (user.QuadPart - s_ulLastUserCPU.QuadPart);

        percent /= (now.QuadPart - s_ulLastCPU.QuadPart);

        percent /= s_uiNumProcessors;

        s_ulLastCPU = now;
        s_ulLastUserCPU = user;
        s_ulLastSysCPU = sys;

        uiCpuUsage10x = percent * 100 * 10;
    }

    return uiCpuUsage10x;

}
#endif
/************************************************************************************************************\

Function:       GetMemUsage

Description:    Return the % of memory usage x10.

\************************************************************************************************************/
#ifdef LINUX
MUINT GetMemUsage()
{
    MUINT uiMemUsage10x = 0;
    FILE* hFile         = fopen("/proc/meminfo", "r");

    if (hFile != NULL)
    {
        MUINT uiMemTotal = 0;
        MUINT uiMemFree  = 0;

        if (fscanf(hFile, "MemTotal: %u kB\n", &uiMemTotal) != EOF)
        {
            if (fscanf(hFile, "MemFree: %u kB\n", &uiMemFree) != EOF)
            {
                if ((uiMemTotal > 0) && (uiMemTotal >= uiMemFree))
                {
                    uiMemUsage10x = ((uiMemTotal - uiMemFree) * 1000) / uiMemTotal;
                }
            }
        }

        fclose(hFile);
    }

    return uiMemUsage10x;
}
#endif

#ifdef _WIN32
MUINT GetMemUsage()
{
    MUINT uiMemUsage10x = 0;
    MEMORYSTATUSEX oProcessMemStatus;

    oProcessMemStatus.dwLength = sizeof(oProcessMemStatus);
    if (GlobalMemoryStatusEx(&oProcessMemStatus))
    {

        PROCESS_MEMORY_COUNTERS   pmc;

        if (GetProcessMemoryInfo(GetCurrentProcess(), &pmc, sizeof(pmc)))
        {
            uiMemUsage10x = 10 * pmc.WorkingSetSize / (oProcessMemStatus.ullTotalPhys / 1024);
        }
    }
    return uiMemUsage10x;
}
#endif



/************************************************************************************************************\

Function:       LPixelFormat_IsRgb

Description:    Return MTRUE if the given pixel format is in RGB color space.

\************************************************************************************************************/
MBOOL LPixelFormat_IsRgb(LPixelFormat ePixFmt)
{
    switch (ePixFmt)
    {
        case LPixelFormat_R10R10R10X2:
        case LPixelFormat_G10G10G10X2:
        case LPixelFormat_B10B10B10X2:
        case LPixelFormat_B5G6R5:
        case LPixelFormat_B5G5R5A1:
        case LPixelFormat_B4G4R4A4:
        case LPixelFormat_R8G8B8A8:
        case LPixelFormat_B8G8R8A8:
        case LPixelFormat_R10G10B10A2:
        case LPixelFormat_R82G82B82A2:
        case LPixelFormat_B10G10R10A2:
        case LPixelFormat_B82G82R82A2:
        case LPixelFormat_R11G11B10_FLOAT:
        case LPixelFormat_R9G9B9E5_SHAREDEXP_FLOAT:
        case LPixelFormat_R16G16B16A16_FLOAT:
        case LPixelFormat_R16G16B16A16:
        case LPixelFormat_MP_G8_R8_B8:
        case LPixelFormat_MP_G10_R10_B10:
        case LPixelFormat_MP_G16_R16_B16:
            return MTRUE;

        default:
            return MFALSE;
    }
}

/************************************************************************************************************\

Function:       LPixelFormat_Get3xBppAllPlanes

Description:    Return the total number of bits per pixel of all planes of the given pixel format multiplied
                by 3. The purpose of the 3 factor is for some packed pixel formats like 422-10 where we have
                3 pixels per 64 bits that gives a not very convenient 21.33333 bpp.

\************************************************************************************************************/
MUINT LPixelFormat_Get3xBppAllPlanes(LPixelFormat ePixelFormat)
{
    MUINT               uiPlaneCount = Get_LPixelFormat_PlaneCount(ePixelFormat) + 1;
    LChromaSubSampling  eChromaSub   = Get_LPixelFormat_SubSampling(ePixelFormat);
    MUINT               uiChromaDiv  = (eChromaSub == LCSSAMPLING_411) ? 4 :
                                       (eChromaSub == LCSSAMPLING_420) ? 4 :
                                       (eChromaSub == LCSSAMPLING_422) ? 2 : 1;
    MUINT uiTotalBpp  = 0;
    MUINT uiPackedDiv = 1;
    MUINT i;

    for (i = 0; i < uiPlaneCount; i++)
    {
        MUINT uiPixPerUnit  = LPixelFormat_GetNbPixelsPerColorUnit(ePixelFormat, i);
        MUINT uiBpp         = LPixelFormat_GetBpp(ePixelFormat, i) * uiPackedDiv;

        uiTotalBpp  *= uiPixPerUnit;
        uiPackedDiv *= uiPixPerUnit;

        if (i > 0)
        {
            uiBpp /= uiChromaDiv;
        }

        uiTotalBpp += uiBpp;
    }

    if (uiPackedDiv != 0)
    {
        uiTotalBpp = (3 * uiTotalBpp) / uiPackedDiv;

        return uiTotalBpp;
    }
    else
    {
        return 0;
    }
}


/************************************************************************************************************\

Function:       LCpuThread_Create

Description:

Parameters:     fnThread
                *pRef

Return Value:   Not Specified.

Comments:       None.

\************************************************************************************************************/
LStatus LCpuThread_Create(LCpuThread_Function pfnThread, void *poRef, LCpuThread_Handle* phCpuThread)
{
    if ((pfnThread != MNULL)
        && (poRef != MNULL)
        && (phCpuThread != MNULL))
    {
        pthread_t* poCpuThread = malloc(sizeof(pthread_t));

        if (poCpuThread != MNULL)
        {
            if (pthread_create(poCpuThread, MNULL, (PosixThread_Function)pfnThread, poRef) != 0)
            {
                free(poCpuThread);
                poCpuThread = MNULL;
            }
        }

        *phCpuThread = (LCpuThread_Handle)poCpuThread;

        return (*phCpuThread != MNULL) ? LStatus_OK : LStatus_FAIL;
    }
    else
    {
        return LStatus_INVALID_PARAM;
    }
}

/************************************************************************************************************\

Function:       LCpuThread_Destroy

Description:

Parameters:     hThread

Return Value:   Not Specified.

Comments:       None.

\************************************************************************************************************/
LStatus LCpuThread_Destroy(LCpuThread_Handle hCpuThread, LStatus* peRetStatus)
{
    if (hCpuThread != MNULL)
    {
        pthread_t*  poCpuThread = (pthread_t*)hCpuThread;
        void*       pvRet       = MNULL;

        if (pthread_join(*poCpuThread, &pvRet) != 0)
        {
            return LStatus_FAIL;
        }

        if (peRetStatus != MNULL)
        {
            *peRetStatus = (LStatus)pvRet;
        }

        free(poCpuThread);

        return LStatus_OK;
    }
    else
    {
        return LStatus_FAIL;
    }
}

/************************************************************************************************************\

Function:       LCpuThread_CreateMutex

Description:    Write description here.

Parameters:     None.

Return Value:   Not Specified.

Comments:       None.

\************************************************************************************************************/
LStatus LMutex_Create(LMutext_Handle *phMutex)
{
    LStatus eStatus = LStatus_FAIL;

    if (phMutex != MNULL)
    {
        pthread_mutex_t* poMutex = malloc(sizeof(pthread_mutex_t));

        if (poMutex != MNULL)
        {
            if (pthread_mutex_init(poMutex, NULL) == 0)
            {
                eStatus = LStatus_OK;
            }
            else
            {
                free(poMutex);
                poMutex = MNULL;
            }

            *phMutex = (LMutext_Handle)poMutex;
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       LMutex_Destroy

Description:    Write description here.

Parameters:     hMutex

Return Value:   Not Specified.

Comments:       None.

\************************************************************************************************************/
LStatus LMutex_Destroy(LMutext_Handle hMutex)
{
    LStatus eStatus = LStatus_FAIL;

    if (hMutex != MNULL)
    {
        pthread_mutex_t* poMutex = (pthread_mutex_t*)hMutex;

        eStatus = (pthread_mutex_destroy(poMutex) == 0) ? LStatus_OK : LStatus_FAIL;

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            free(poMutex);
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       LMutex_Lock

Description:    Write description here.

Parameters:     hMutex

Return Value:   Not Specified.

Comments:       None.

\************************************************************************************************************/
LStatus LMutex_Lock(LMutext_Handle hMutex)
{
    if (hMutex != MNULL)
    {
        return pthread_mutex_lock((pthread_mutex_t*)hMutex) ? LStatus_FAIL : LStatus_OK;
    }
    else
    {
        return LStatus_INVALID_PARAM;
    }
}

/************************************************************************************************************\

Function:       LMutex_TryLock

Description:    Write description here.

Parameters:     hMutex

Return Value:   Not Specified.

Comments:       None.

\************************************************************************************************************/
LStatus LMutex_TryLock(LMutext_Handle hMutex)
{
    if (hMutex != MNULL)
    {
        return pthread_mutex_trylock((pthread_mutex_t*)hMutex) ? LStatus_FAIL : LStatus_OK;
    }
    else
    {
        return LStatus_INVALID_PARAM;
    }
}


/************************************************************************************************************\

Function:       LMutex_Unlock

Description:    Write description here.

Parameters:     hMutex

Return Value:   Not Specified.

Comments:       None.

\************************************************************************************************************/
LStatus LMutex_Unlock(LMutext_Handle hMutex)
{
    if (hMutex != MNULL)
    {
        return  pthread_mutex_unlock((pthread_mutex_t *)hMutex) ? LStatus_FAIL : LStatus_OK;
    }
    else
    {
        return LStatus_INVALID_PARAM;
    }
}

/************************************************************************************************************\

Function:       LCondition_Create

Description:    Write description here.

Parameters:     None.

Return Value:   Not Specified.

Comments:       None.

\************************************************************************************************************/
LStatus LCondition_Create(LCondition_Handle *phCond)
{
    LStatus eStatus = LStatus_FAIL;

    if (phCond != MNULL)
    {
        pthread_cond_t* poCond = malloc(sizeof(pthread_cond_t));

        if (poCond != MNULL)
        {
            if (pthread_cond_init(poCond, NULL) == 0)
            {
                eStatus = LStatus_OK;
            }
            else
            {
                free(poCond);
                poCond = MNULL;
            }

            *phCond = (LCondition_Handle)poCond;
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       LCondition_Destroy

Description:    Write description here.

Parameters:     hCond

Return Value:   Not Specified.

Comments:       None.

\************************************************************************************************************/
LStatus LCondition_Destroy(LCondition_Handle hCond)
{
    LStatus eStatus = LStatus_FAIL;

    if (hCond != MNULL)
    {
        pthread_cond_t* poCond = (pthread_cond_t*)hCond;

        eStatus = (pthread_cond_destroy(poCond) == 0) ? LStatus_OK : LStatus_FAIL;

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            free(poCond);
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       LCondition_Signal

Description:    Write description here.

Parameters:     hCond

Return Value:   Not Specified.

Comments:       None.

\************************************************************************************************************/
LStatus LCondition_Signal(LCondition_Handle hCond)
{
    if (hCond != MNULL)
    {
        return (pthread_cond_signal((pthread_cond_t*)hCond) == 0) ? LStatus_OK : LStatus_FAIL;
    }
    else
    {
        return LStatus_FAIL;
    }
}

/************************************************************************************************************\

Function:       LCondition_Wait

Description:    Write description here.

Parameters:     hCond
                hMutext
                uiTimeoutMilliSec

Return Value:   Not Specified.

Comments:       None.

\************************************************************************************************************/
LStatus LCondition_Wait(LCondition_Handle hCond, LMutext_Handle hMutex, MUINT32 uiTimeoutMilliSec)
{
    if (hCond != MNULL)
    {
        if (uiTimeoutMilliSec == LINFINITE_TIMEOUT)
        {
            return (pthread_cond_wait((pthread_cond_t*)hCond, (pthread_mutex_t*)hMutex) == 0)
                   ? LStatus_OK : LStatus_FAIL;
        }
        else
        {
            struct timeval  oTime;
            struct timespec oTimeout;

            gettimeofday(&oTime, NULL);

            oTime.tv_usec += (uiTimeoutMilliSec * 1000);

            while (oTime.tv_usec >= (1000*1000))
            {
                oTime.tv_usec -= (1000*1000);
                oTime.tv_sec  += 1;
            }

            oTimeout.tv_nsec = oTime.tv_usec * 1000;
            oTimeout.tv_sec  = oTime.tv_sec;

            switch (pthread_cond_timedwait((pthread_cond_t*)hCond, (pthread_mutex_t*)hMutex, &oTimeout))
            {
                case 0:         return LStatus_OK;
                case ETIMEDOUT: return LStatus_TIMEOUT;
                case EINVAL:    return LStatus_INVALID_PARAM;
                default:        return LStatus_FAIL;
            }
        }
    }
    else
    {
        return LStatus_FAIL;
    }
}

/************************************************************************************************************\

Function:       GetRectOnEllipsoidalPath

\************************************************************************************************************/
void GetRectOnEllipsoidalPath(
        const MUINT     uiAnglePos1000x,
        const LSIZE*    poBackSize,
        const LSIZE*    poBorderSize,
        const MUINT     uiCornerFactor,
              LRECT32*  poDstRect)
{
    LSIZE oFrameSize;
    LSIZE oHalfSizeDiff;

    oFrameSize.iWidth     = poDstRect->iRight - poDstRect->iLeft;
    oFrameSize.iHeight    = poDstRect->iBottom - poDstRect->iTop;
    oHalfSizeDiff.iWidth  = (poBackSize->iWidth  - oFrameSize.iWidth)  / 2;
    oHalfSizeDiff.iHeight = (poBackSize->iHeight - oFrameSize.iHeight) / 2;

    MUINT uiSubAngle = uiAnglePos1000x % (90*1000);

    if (uiSubAngle > 45*1000)
    {
        uiSubAngle = 90*1000 - uiSubAngle;
    }

    MFLOAT32 fFactor    = 1.0f + pow((MFLOAT32)(uiSubAngle * uiCornerFactor) / (1000*1000), 2);
    MFLOAT32 fAngle     = (M_PI * 2 * uiAnglePos1000x) / (360*1000);
    MINT     x          = (MINT)(cos(fAngle) * fFactor * (oHalfSizeDiff.iWidth  - poBorderSize->iWidth));
    MINT     y          = (MINT)(sin(fAngle) * fFactor * (oHalfSizeDiff.iHeight - poBorderSize->iHeight));

    poDstRect->iLeft    = max(x + oHalfSizeDiff.iWidth,  0);
    poDstRect->iTop     = max(y + oHalfSizeDiff.iHeight, 0);
    poDstRect->iRight   = poDstRect->iLeft + oFrameSize.iWidth;
    poDstRect->iBottom  = poDstRect->iTop  + oFrameSize.iHeight;
}

/************************************************************************************************************\

Function:       LDeviceThread_SyncDt

Description:    Synchronize two different device threads.

Parameters:     hDstDt          Device thread handle to synchronize.
                hSrcDt          Device thread handle to synchronize from.
                pbStopWaiting   Flag indicationg that we must stop waiting after submission timeout.

\************************************************************************************************************/
LStatus LDeviceThread_SyncDt(
    LDeviceThread_Handle hDstDt,
    LDeviceThread_Handle hSrcDt,
    volatile MBOOL*      pbStopWaiting)
{
    if (ENABLE_AUTO_SYNC)
    {
        return LStatus_OK;
    }

    MUINT64 uiSrcDtLuid;
    MUINT64 uiSrcTag;

    LStatus eStatus = LDeviceThread_GetLUID(hSrcDt, &uiSrcDtLuid);

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LDeviceThread_GetSubmissionTag(hSrcDt, &uiSrcTag);
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        while (MTRUE)
        {
            eStatus = LDeviceThread_SyncWithSubmission(
                            hDstDt,
                            uiSrcDtLuid,
                            uiSrcTag,
                            MFALSE,
                            MFALSE,
                            2000);

            if ((eStatus == LStatus_TIMEOUT) && (*pbStopWaiting == MFALSE))
            {
                MsgLog(0, "LDeviceThread_SyncWithSubmission timeout! Continue...");
                usleep(10);
            }
            else
            {
                break;
            }
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function: GetChar

Description: Returns the next character from the standard input (stdin).

Parameters: None

Return Value:  The character read as an int value

\************************************************************************************************************/
int GetChar()
{
#ifdef _WIN32
    int ch = EOF;
    if(_kbhit())
    {
      ch = _getch();
    }

    /*In linux, a keypress enter is a LFeed with a value of 10. In Windows,
    The enter key calls a carriage return which has an ASCII value of 13 instead
    To avoid having to modify it at each call of getchar we go it once here so that
    it's consistant in what it expects between Windows and Linux*/
    if (ch == 13)
        ch = 10;
    return ch;
#else
    struct termios oOldAttrib;
    struct termios oNewAttrib;

    tcgetattr(STDIN_FILENO, &oOldAttrib);
    oNewAttrib = oOldAttrib;
    oNewAttrib.c_lflag &= ~(ICANON | ECHO);
    tcsetattr(STDIN_FILENO, TCSANOW, &oNewAttrib);

    int iOldFlags = fcntl(STDIN_FILENO, F_GETFL, 0);
    fcntl(STDIN_FILENO, F_SETFL, iOldFlags | O_NONBLOCK);

    int ch = getchar();

    tcsetattr(STDIN_FILENO, TCSANOW, &oOldAttrib);
    fcntl(STDIN_FILENO, F_SETFL, iOldFlags);

    return ch;
 #endif

}


/************************************************************************************************************\

Function: usleep

Description:    Sleeps a CPU thread execution for the specified number of microseconds.

Parameters:     Number of microsecond the CPU thread should be
                stalled.

Return Value:   MTRUE if CPU thread was successfully stalled for the specified amount of
                milliseconds,MALSE otherwise.

Comments: This is not a active wait, it release the CPU to execute another thread.

\************************************************************************************************************/
#ifdef _WIN32
MBOOL32 usleep(long usec)
{
    LARGE_INTEGER lFrequency;
    LARGE_INTEGER lEndTime;
    LARGE_INTEGER lCurTime;

    if (usec >= 1000)
    {
        
        //set timer resolution to 1 millisecond
        timeBeginPeriod(1);
        Sleep(usec / 1000);
        //restore default timer resolution
        timeEndPeriod(1);
        return MTRUE;
    }
     
    QueryPerformanceFrequency(&lFrequency);

    if (lFrequency.QuadPart)
    {

        QueryPerformanceCounter(&lEndTime);

        lEndTime.QuadPart += (LONGLONG)usec * lFrequency.QuadPart / 1000000;

        do
        {
            QueryPerformanceCounter(&lCurTime);
            Sleep(0);

        } while (lCurTime.QuadPart < lEndTime.QuadPart);

    }

    return MTRUE;
}
#endif

/************************************************************************************************************\

Function: clock_gettime

Description: Gets the current time and returns it as a timespec structure.

Parameters: int X - not used
            out - struct* timespec - used to return the current time



Return Value:  0 if successful, -1 otherwise

\************************************************************************************************************/
#ifdef _WIN32
int clock_gettime(int X, struct timespec* tv)
{
    LARGE_INTEGER  oCounterVal;
    double         Nanoseconds;
    double         frequencyToNanoseconds;
    LARGE_INTEGER  lPerformanceFrequency;

    if (QueryPerformanceFrequency(&lPerformanceFrequency))
    {
        frequencyToNanoseconds = (double)lPerformanceFrequency.QuadPart / 1000000000.;

        QueryPerformanceCounter(&oCounterVal);

        Nanoseconds = (double)oCounterVal.QuadPart / frequencyToNanoseconds;

        oCounterVal.QuadPart = Nanoseconds;
        tv->tv_sec = oCounterVal.QuadPart / 1000000000.;

        assert(oCounterVal.QuadPart >= (tv->tv_sec * 1000000000.));

        tv->tv_nsec = oCounterVal.QuadPart - (tv->tv_sec * 1000000000.);
        return 0;
    }

    return -1;
}
#endif

/************************************************************************************************************\

Function:    gettimeofday

Description:

Parameters: None

Return Value:

\************************************************************************************************************/
#ifdef _WIN32
int  gettimeofday(struct timeval * tp, struct timezone * tzp)
{
    FILETIME    file_time;
    SYSTEMTIME  system_time;
    ULARGE_INTEGER ularge;

    /* FILETIME of Jan 1 1970 00:00:00. */
    static const MUINT64 EPOCH = ((MUINT64)116444736000000000ULL);


    GetSystemTime(&system_time);
    SystemTimeToFileTime(&system_time, &file_time);
    ularge.LowPart = file_time.dwLowDateTime;
    ularge.HighPart = file_time.dwHighDateTime;

    tp->tv_sec = (long)((ularge.QuadPart - EPOCH) / 10000000L);
    tp->tv_usec = (long)(system_time.wMilliseconds * 1000);
    return 0;
}
#endif

#if defined(_MSC_VER) || defined(__MINGW32__)
/************************************************************************************************************\

Function:

Description:

Parameters: None

Return Value:

\************************************************************************************************************/
int c99_vsnprintf(char* str, size_t size, const char* format, va_list ap)
{
    int count = -1;

    if (size != 0)
        count = _vsnprintf_s(str, size, _TRUNCATE, format, ap);
    if (count == -1)
        count = _vscprintf(format, ap);

    return count;
}

/************************************************************************************************************\

Function:

Description:

Parameters: None

Return Value:

\************************************************************************************************************/
int c99_snprintf(char* str, size_t size, const char* format, ...)
{
    int count;
    va_list ap;
    va_start(ap, format);
    count = c99_vsnprintf(str, size, format, ap);
    va_end(ap);

    return count;
}

#endif

/************************************************************************************************************\

Function:       EnumAlsaDeviceNames

Description:    .

\************************************************************************************************************/
#ifdef ALSA_SUPPORT
LStatus EnumAlsaDeviceNames(
    LSTR64*     aszDevNames,
    MUINT32     uiMaxNbDev,
    MUINT32*    puiNbDev,
    MBOOL32     bCapture)
{
    if ((aszDevNames == MNULL) || (uiMaxNbDev == 0))
    {
        return LStatus_INVALID_PARAM;
    }

    char **aszHints;
    MINT iErr = snd_device_name_hint(-1, "pcm", (void***) &aszHints);

    if (iErr != 0)
    {
        return LStatus_FAIL;
    }

    char**      szHint  = aszHints;
    MUINT32     uiNbDev = 0;
    const char* szIoId  = bCapture ? "Input" : "Output";

    while ((*szHint != MNULL) && (uiNbDev < uiMaxNbDev))
    {
        char* szName    = snd_device_name_get_hint(*szHint, "NAME");
        char* szDevIoId = snd_device_name_get_hint(*szHint, "IOID");

        if ((szName != MNULL) && (strcmp("null", szName) != 0))
        {
            if ((szDevIoId == MNULL)  /* NULL means both Input/Output */
                 || (strcmp(szDevIoId, szIoId) == 0))
            {
                if ((strlen(szName) + 1) <= sizeof(aszDevNames[uiNbDev]))
                {
                    strcpy(aszDevNames[uiNbDev], szName);
                    uiNbDev++;
                }
            }
        }

        if (szName != MNULL)
        {
            free(szName);
        }

        if (szDevIoId != MNULL)
        {
            free(szDevIoId);
        }

        szHint++;
    }

    snd_device_name_free_hint((void**)aszHints);

    if (puiNbDev != MNULL)
    {
        *puiNbDev = uiNbDev;
    }

    return LStatus_OK;
}
#endif

/************************************************************************************************************\

Function:       CheckAlsaDeviceCapabilities

Description:    .

\************************************************************************************************************/
#ifdef ALSA_SUPPORT
LStatus CheckAlsaDeviceCapabilities(
    snd_pcm_t*              hAlsaDev,
    snd_pcm_hw_params_t*    poHwParams,
    MUINT32                 uiSamplingRate,
    MUINT32                 uiNbChannels,
    MUINT32                 uiSampleSize,
    MBOOL32                 bBigEndian,
    MBOOL32                 bUnsigned,
    snd_pcm_format_t*       peFormat)
{
    LStatus eStatus = LStatus_FAIL;
    MINT iErr = 0;
    MBOOL32 bRateMatch = MFALSE;
    MBOOL32 bChannelsMatch = MFALSE;
    MBOOL32 bFormatMatch = MFALSE;

    snd_pcm_uframes_t   uiBufferSizeMin = 0;
    snd_pcm_uframes_t   uiBufferSizeMax = 0;
    snd_pcm_uframes_t   uiPeriodSizeMin = 0;
    snd_pcm_uframes_t   uiPeriodSizeMax = 0;
    MINT                iDir            = 0;

    if((iErr = snd_pcm_hw_params_get_buffer_size_min(poHwParams, &uiBufferSizeMin)) == 0)
    {
        if((iErr = snd_pcm_hw_params_get_buffer_size_max(poHwParams, &uiBufferSizeMax)) == 0)
        {
            if((iErr = snd_pcm_hw_params_get_period_size_min(poHwParams, &uiPeriodSizeMin, &iDir)) == 0)
            {
                if((iErr = snd_pcm_hw_params_get_period_size_max(poHwParams, &uiPeriodSizeMax, &iDir)) == 0)
                {
                    MsgLog(2, "Supported buffer size (frames) = [min=%u,max=%u]", uiBufferSizeMin, uiBufferSizeMax);
                    MsgLog(2, "Supported period size (frames) = [min=%u,max=%u]", uiPeriodSizeMin, uiPeriodSizeMax);
                }
            }
        }
    }

    if(iErr == 0)
    {
        MUINT32 uiMaxChannels = 0;
        if((iErr = snd_pcm_hw_params_get_channels_max(poHwParams, &uiMaxChannels)) == 0)
        {
            MUINT32 uiMinChannels = 0;
            if((iErr = snd_pcm_hw_params_get_channels_min(poHwParams, &uiMinChannels)) == 0)
            {
                MsgLog(2, "Supported channels = [min=%u,max=%u]", uiMinChannels, uiMaxChannels);
            }
        }
    }

    if(iErr == 0)
    {
        eStatus = LStatus_OK;

        const MUINT32 auiSamplingRates[] =
        {
            5512, 8000, 11025, 12000, 16000, 22050, 24000, 32000,
            44100, 48000, 64000, 88200, 96000, 176400, 192000,
        };

        MUINT i = 0;

        for(i=0; i<sizeof(auiSamplingRates)/sizeof(auiSamplingRates[0]); i++)
        {
            if(snd_pcm_hw_params_test_rate(
                    hAlsaDev,
                    poHwParams,
                    auiSamplingRates[i],
                    0) == 0)
            {
                if(auiSamplingRates[i] == uiSamplingRate)
                {
                    bRateMatch = MTRUE;
                }

                MsgLog(2, "Supported rate: %u", auiSamplingRates[i]);
            }
        }

        bChannelsMatch = (snd_pcm_hw_params_test_channels(
                                     hAlsaDev,
                                     poHwParams,
                                     uiNbChannels) == 0) ? (MTRUE) : (MFALSE);

        const snd_pcm_format_t aeFormats[] =
        {
            SND_PCM_FORMAT_S8,
            SND_PCM_FORMAT_U8,
            SND_PCM_FORMAT_S16_LE,
            SND_PCM_FORMAT_S16_BE,
            SND_PCM_FORMAT_U16_LE,
            SND_PCM_FORMAT_U16_BE,
            SND_PCM_FORMAT_S24_LE,
            SND_PCM_FORMAT_S24_BE,
            SND_PCM_FORMAT_U24_LE,
            SND_PCM_FORMAT_U24_BE,
            SND_PCM_FORMAT_S32_LE,
            SND_PCM_FORMAT_S32_BE,
            SND_PCM_FORMAT_U32_LE,
            SND_PCM_FORMAT_U32_BE,
            SND_PCM_FORMAT_FLOAT_LE,
            SND_PCM_FORMAT_FLOAT_BE,
            SND_PCM_FORMAT_FLOAT64_LE,
            SND_PCM_FORMAT_FLOAT64_BE,
            SND_PCM_FORMAT_IEC958_SUBFRAME_LE,
            SND_PCM_FORMAT_IEC958_SUBFRAME_BE,
            SND_PCM_FORMAT_MU_LAW,
            SND_PCM_FORMAT_A_LAW,
            SND_PCM_FORMAT_IMA_ADPCM,
            SND_PCM_FORMAT_MPEG,
            SND_PCM_FORMAT_GSM,
            SND_PCM_FORMAT_SPECIAL,
            SND_PCM_FORMAT_S24_3LE,
            SND_PCM_FORMAT_S24_3BE,
            SND_PCM_FORMAT_U24_3LE,
            SND_PCM_FORMAT_U24_3BE,
            SND_PCM_FORMAT_S20_3LE,
            SND_PCM_FORMAT_S20_3BE,
            SND_PCM_FORMAT_U20_3LE,
            SND_PCM_FORMAT_U20_3BE,
            SND_PCM_FORMAT_S18_3LE,
            SND_PCM_FORMAT_S18_3BE,
            SND_PCM_FORMAT_U18_3LE,
            SND_PCM_FORMAT_U18_3BE,
            SND_PCM_FORMAT_S16,
            SND_PCM_FORMAT_U16,
            SND_PCM_FORMAT_S24,
            SND_PCM_FORMAT_U24,
            SND_PCM_FORMAT_S32,
            SND_PCM_FORMAT_U32,
            SND_PCM_FORMAT_FLOAT,
            SND_PCM_FORMAT_FLOAT64,
            SND_PCM_FORMAT_IEC958_SUBFRAME,
        };

        for (i = 0; i < sizeof(aeFormats)/sizeof(aeFormats[0]); i++)
        {
            if (snd_pcm_hw_params_test_format(hAlsaDev, poHwParams, aeFormats[i]) == 0)
            {
                if((!bFormatMatch)
                    && (snd_pcm_format_physical_width(aeFormats[i]) == uiSampleSize)
                    && (snd_pcm_format_big_endian(aeFormats[i]) == bBigEndian)
                    && (snd_pcm_format_unsigned(aeFormats[i]) == bUnsigned))
                {
                    bFormatMatch = MTRUE;
                    *peFormat = aeFormats[i];
                }
                MsgLog(2, "Supported format: %s", snd_pcm_format_name(aeFormats[i]));
            }
        }

        const snd_pcm_access_t aeAccessType[] =
        {
            SND_PCM_ACCESS_MMAP_INTERLEAVED,
            SND_PCM_ACCESS_MMAP_NONINTERLEAVED,
            SND_PCM_ACCESS_MMAP_COMPLEX,
            SND_PCM_ACCESS_RW_INTERLEAVED,
            SND_PCM_ACCESS_RW_NONINTERLEAVED,
        };

        for(i=0; i < sizeof(aeAccessType)/sizeof(aeAccessType[0]); i++)
        {
            if(snd_pcm_hw_params_test_access(hAlsaDev, poHwParams, aeAccessType[i]) == 0)
            {
                MsgLog(2, "Supported access type: %s", snd_pcm_access_name(aeAccessType[i]));
            }
        }

        if(!bRateMatch)
        {
            MsgLogErr("Unsupported sampling rate: %u Hz.", uiSamplingRate);
            eStatus = LStatus_INVALID_PARAM;
        }

        if(!bChannelsMatch)
        {
            MsgLogErr("Unsupported number of channels: %u", uiNbChannels);
            eStatus = LStatus_INVALID_PARAM;
        }

        if(!bFormatMatch)
        {
            MsgLogErr("Unsupported format: %ubits/%s/%s",
                      uiSampleSize,
                      bUnsigned ? "UNSIGNED" : "SIGNED",
                      bBigEndian ? "BE" : "LE");

            eStatus = LStatus_INVALID_PARAM;
        }
    }

    return eStatus;
}
#endif

/************************************************************************************************************\

Function:       ConfigureAlsaDevice

Description:    .

\************************************************************************************************************/
#ifdef ALSA_SUPPORT
LStatus ConfigureAlsaDevice(
    snd_pcm_t*      hAlsaDev,
    MUINT32         uiSamplingRate,
    MUINT32         uiNbChannels,
    MUINT32         uiSampleSize,
    MBOOL32         bBigEndian,
    MBOOL32         bUnsigned,
    MUINT32         uiFramesPerBuffer)
{
    LStatus                 eStatus        = LStatus_OK;
    snd_pcm_hw_params_t*    poHwParams     = MNULL;
    MINT32                  iErr           = 0;

    /*
     * Hardware parameters
     */

    if (LSTATUS_IS_SUCCESS(eStatus)
         && (iErr = snd_pcm_hw_params_malloc(&poHwParams)) < 0)
    {
        eStatus = LStatus_OUT_OF_MEMORY;
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        /* Init hwparams with full configuration space */
        if((iErr = snd_pcm_hw_params_any(
                        hAlsaDev,
                        poHwParams)) < 0)
        {
            eStatus = LStatus_FAIL;
        }
    }

    snd_pcm_format_t eFormat = SND_PCM_FORMAT_UNKNOWN;

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        /* Check that programmed settings match device capabilities. */
        eStatus = CheckAlsaDeviceCapabilities(
                      hAlsaDev,
                      poHwParams,
                      uiSamplingRate,
                      uiNbChannels,
                      uiSampleSize,
                      bBigEndian,
                      bUnsigned,
                      &eFormat);
    }

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LStatus_FAIL;
        /* Set access type. */
        if((iErr = snd_pcm_hw_params_set_access(
                        hAlsaDev,
                        poHwParams,
                        SND_PCM_ACCESS_RW_INTERLEAVED)) == 0)
        {
            /* Set sample format. */
            if ((iErr = snd_pcm_hw_params_set_format(
                            hAlsaDev,
                            poHwParams,
                            eFormat)) == 0)
            {
                eStatus = LStatus_OK;
            }
        }
    }

    MINT iDir = 0;

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LStatus_FAIL;

        /* Set sample rate. */
        if((iErr = snd_pcm_hw_params_set_rate(
                        hAlsaDev,
                        poHwParams,
                        uiSamplingRate,
                        0)) == 0)
        {
            /* Set number of channels. */
            if ((iErr = snd_pcm_hw_params_set_channels(
                            hAlsaDev,
                            poHwParams,
                            uiNbChannels)) == 0)
            {
                eStatus = LStatus_OK;
            }
        }
    }

    snd_pcm_uframes_t uiBufferSize = 0;
    snd_pcm_uframes_t uiPeriodSize = 0;

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LStatus_FAIL;

        /* Period size: PCM buffer size, buffer size: two periods. */
        uiBufferSize = uiFramesPerBuffer * 2;
        uiPeriodSize = uiFramesPerBuffer;

        /* Set buffer size in frames. */
        if((iErr = snd_pcm_hw_params_set_buffer_size_near(
                        hAlsaDev,
                        poHwParams,
                        &uiBufferSize)) == 0)
        {
            /* Set period size in frames. */
            if((iErr = snd_pcm_hw_params_set_period_size_near(
                            hAlsaDev,
                            poHwParams,
                            &uiPeriodSize,
                            &iDir)) == 0)
            {
                eStatus = LStatus_OK;
            }
        }
    }

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        MUINT32 uiPeriodTime = 0;
        if((iErr = snd_pcm_hw_params_get_period_time(
                        poHwParams,
                        &uiPeriodTime,
                        &iDir)) == 0)
        {
            MUINT32 uiBufferTime = 0;
            if((iErr = snd_pcm_hw_params_get_buffer_time(
                            poHwParams,
                            &uiBufferTime,
                            &iDir)) == 0)
            {
                MsgLog(2, "Configured settings: buffer size=%u, period size=%u, buffer time=%u, period time=%u",
                       uiBufferSize, uiPeriodSize, uiBufferTime, uiPeriodTime);
            }
        }
    }

    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        /* Apply HW parameter to PCM device. */
        if ((iErr = snd_pcm_hw_params(hAlsaDev, poHwParams)) < 0)
        {
            eStatus = LStatus_FAIL;
        }
    }

    if(poHwParams != MNULL)
    {
        snd_pcm_hw_params_free(poHwParams);
    }

    /*
     * Software parameters
     */
    if(LSTATUS_IS_SUCCESS(eStatus))
    {
        snd_pcm_sw_params_t* poSwParams = MNULL;

        if((iErr = snd_pcm_sw_params_malloc(
                        &poSwParams)) < 0)
        {
            eStatus = LStatus_OUT_OF_MEMORY;
        }

        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            if((iErr = snd_pcm_sw_params_current(
                            hAlsaDev,
                            poSwParams)) == 0)
            {
                /* Don't start playback until one period has been buffered. */
                if((iErr = snd_pcm_sw_params_set_start_threshold(
                                hAlsaDev,
                                poSwParams,
                                uiBufferSize)) < 0)
                {
                    eStatus = LStatus_FAIL;
                }
            }
        }

        if(LSTATUS_IS_SUCCESS(eStatus))
        {
            /* Apply SW parameters to PCM device. */
            if((iErr = snd_pcm_sw_params(hAlsaDev, poSwParams)) < 0)
            {
                eStatus = LStatus_FAIL;
            }
        }

        if(poSwParams != MNULL)
        {
            snd_pcm_sw_params_free(poSwParams);
        }
    }

    return eStatus;
}
#endif

/******************************************************************************\

Function:       GetStatusStr

Description:    Utility function to get the LStatus string.

Parameters:     eStatus     LStatus to convert to string.

Return Value:   None

Comments:       
 
\******************************************************************************/
const MCHAR8* GetStatusStr(
                        LStatus eStatus)
{
    switch (eStatus)
    {
        case LStatus_SEQUENCE_RESET:                     return "SEQUENCE_RESET";
        case LStatus_DISORDERED_RELEASE_COUNT:           return "DISORDERED_RELEASE_COUNT";
        case LStatus_CRITICAL_END_OF_STREAM:             return "CRITICAL_END_OF_STREAM";
        case LStatus_STREAM_NOT_INITIALIZED:             return "STREAM_NOT_INITIALIZED";
        case LStatus_INCOMPLETE_COMMAND:                 return "INCOMPLETE_COMMAND";
        case LStatus_REORDERED_SEQUENCE_PENDING:         return "REORDERED_SEQUENCE_PENDING";
        case LStatus_END_OF_STREAM:                      return "END_OF_STREAM";
        case LStatus_INFO_NOT_AVAILABLE:                 return "INFO_NOT_AVAILABLE";
        case LStatus_TEMPORARY_LACK_OF_RESOURCES:        return "TEMPORARY_LACK_OF_RESOURCES";
        case LStatus_HARDWARE_MALFUNCTION:               return "HARDWARE_MALFUNCTION";
        case LStatus_CONNECTION_LOST:                    return "CONNECTION LOST";
        case LStatus_COMMAND_PENDING:                    return "COMMAND PENDING";
        case LStatus_CANCELED:                           return "CANCELED";
        case LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL:    return "DEVICETHREAD_COMMAND_QUEUE_FULL";
        case LStatus_UNSUPPORTED:                        return "UNSUPPORTED";
        case LStatus_ACCESS_DENIED:                      return "ACCESS_DENIED";
        case LStatus_RESOURCES_BUSY:                     return "RESOURCES_BUSY";
        case LStatus_OUT_OF_RESOURCES:                   return "OUT_OF_RESOURCES";
        case LStatus_OUT_OF_MEMORY:                      return "OUT_OF_MEMORY";
        case LStatus_NO_MORE_DATA:                       return "NO_MORE_DATA";
        case LStatus_TIMEOUT:                            return "TIMEOUT";
        case LStatus_INVALID_PARAM:                      return "INVALID_PARAM";
        case LStatus_FAIL:                               return "FAIL";
        case LStatus_OK:                                 return "OK";
        case LStatus_NOT_OPTIMAL:                        return "NOT_OPTIMAL";
        case LStatus_OK_INCOMPLETE_REORDERED_SEQUENCE:   return "INCOMPLETE_REORDERED_SEQUENCE";
        case LStatus_OK_MISSING_FIRST_SPS:               return "LStatus_OK_MISSING_FIRST_SPS";
        case LStatus_OK_STREAM_NOT_LOWLATENCY:           return "LStatus_OK_STREAM_NOT_LOWLATENCY";
    }
    return " UNKNOWN";
}

/************************************************************************************************************\

Function:       strncat_wz and strncpy_wz

Description:    strncat and strncpy do not put a null char at the end if the source has been trunked.
                '_wz' means 'with zero'.

\************************************************************************************************************/
char* strncat_wz(char* szDst, const char* szSrc, size_t uiDstSize)
{
    szDst[uiDstSize - 1] = 0;
    return strncat(szDst, szSrc, uiDstSize - strlen(szDst) - 1);
}

char* strncpy_wz(char* szDst, const char* szSrc, size_t uiDstSize)
{
    szDst[uiDstSize - 1] = 0;
    return strncpy(szDst, szSrc, uiDstSize - 1);
}
