/************************************************************************************************************\

Module Name:    BufferQueue.h

Description:    Queue for Liberatus buffers.

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

#ifndef INC_BufferQueue_H
#define INC_BufferQueue_H

// -----------------------------------------------------------------------------------------------------------
//                                   I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "Liberatus.h"
#include "CommonUtils.h"
#include "LH264VideoCodec.h"

// -----------------------------------------------------------------------------------------------------------
//                                   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Enumerate:      FrameType

\************************************************************************************************************/
typedef enum
{
    FrameType_Unknown = 0,      // Unknown or doesn't apply to the buffer. Assume progressive.
    FrameType_Progressive,      // Progressive frame.
    FrameType_TopField,         // Interlaced, top field.
    FrameType_BottomField       // Interlaced, bottom field.

} FrameType;

/************************************************************************************************************\

Structure:      BufferInfo

Description:    LBuffer informations that need to be shared between the modules via the buffer queues.

\************************************************************************************************************/
typedef struct tagBufferInfo
{
    LBuffer_Handle  hBuffer;            // Liberatus buffer handle.
    MUINT           uiId;               // Uniq buffer identifier. Should be used for debugging only.
    MBOOL           bInternal;          // MTRUE:  LBuffer created inside the Liberatus module.
                                        // MFALSE: LBuffer created by the application.
    FrameType       eFrameType;         // Identify the frame type.
    LH264_NaluType  eNaluType;          // Nalu type of the buffer. Invalid for non-H264 buffers.
    MUINT           uiSubmitCount;      // Number of modules where this LBuffer has been submitted.
    MBOOL           bEndOfStream;       // A/V stream ended (this buffer is not valid).
    void*           pvPrivateData;      // Data owned by the module which created the buffer.
    MUINT           uiSubPictureCount;  // Number of sub-pictures to be completed (1 to MAX_SUBPICTURE_COUNT).
    MUINT64         uiCounterLUID;      // Counter used to synchronize sub-pictures.
    MUINT           uiStartOffset;      // Starting byte offset into the buffer.
    MUINT           uiSizeBytes;        // Number of bytes from the starting offset. Don't care if set to 0.
    MUINT64         uiTimestampUsec;    // Presentation timestamp in microseconds.
    MUINT64         uiSyncPtsUsec;      // Presentation timestamp used for synchronization on another stream
                                        // in microseconds. Ignore if no synchronization is done.
    MUINT64         uiSubmitTimeUsec;   // Use for performance checking only.
    MUINT64         uiReturnTimeUsec;   // Use for performance checking only.

} BufferInfo;

/************************************************************************************************************\

Structure:      BufferQueueData

Description:    Element pushed to and popped from the queue.

\************************************************************************************************************/
typedef struct tagBufferQueueElement
{
    BufferInfo*          poBufferInfo;
    MUINT64              uiTag;
    LDeviceThread_Handle hDeviceThread;

} BufferQueueElement;

/************************************************************************************************************\

Structure:      PopData

Description:    Data returned on pop queue.

\************************************************************************************************************/
typedef struct tagPopData
{
    BufferQueueElement  oElement;
    MUINT64             uiQueueLatencyUsec;
    MUINT               uiQueueSize;
} PopData;

/************************************************************************************************************\

Structure:      BufferQueue

Description:    Queue used to transmit Liberatus buffers between modules.

\************************************************************************************************************/
typedef struct tagBufferQueue
{
    LMutext_Handle      hMutex;
    LCondition_Handle   hNotEmptyCond;
    BufferQueueElement* aoElement;
    MUINT               uiElementCount;
    MUINT               uiPushIdx;
    MUINT               uiPopIdx;
    MUINT64             uiLastTimestampUsec;

} BufferQueue;

#define BufferQueue_Construct { /*.hMutex           =*/ MNULL, \
                                /*.hNotEmptyCond    =*/ MNULL, \
                                /*.aoElement        =*/ MNULL, \
                                /*.uiElementCount   =*/ 0, \
                                /*.uiPushIdx        =*/ 0, \
                                /*.uiPopIdx         =*/ 0, \
                                /*.uiLastTimestamp  =*/ 0 }

// -----------------------------------------------------------------------------------------------------------
//                  G L O B A L   V A R I A B L E / F U N C T I O N   R E F E R E N C E S
// -----------------------------------------------------------------------------------------------------------

LStatus BufQ_Init(BufferQueue *poBufferQ);
void  BufQ_Cleanup(BufferQueue* poBufferQ);
MBOOL BufQ_Push(BufferQueue* poBufferQ,
                BufferInfo* poBufferInfo,
                MUINT64 uiTag,
                LDeviceThread_Handle hDevThread);
MBOOL BufQ_Pop(BufferQueue* poBufferQ,
               MUINT uiTimeoutMsec,
               PopData* poData);
MBOOL BufQ_Peek(BufferQueue* poBufferQ,
                PopData* poData);
MUINT BufQ_GetSize(BufferQueue* poBufferQ);

#endif // INC_BufferQueue_H
