#!/bin/bash

# -----------------------------------------------------------------
# Copyright  2020  Matrox Graphic Inc.
#   All rights reserved
#   Written by: SZM
#
# Description:
# ============
#   This script searches for files, contained  in  the  "/rw/config" 
#   directory.  The  number at  the  end of  the file  name is  the 
#   connector index where the override is to be applied. When a file 
#   is found it will apply the EDID override for the given connector.
#   Conversely, the  EDID  override  will  be  remove  when a  given 
#   connector that has  an EDID  override  set  but  does not have a 
#   matching file found.    
#
#   Valid  file names are:
#               dbedidoverride<0-3>.txt
#               edidoverride<0-3>.txt
#   NOTE: 
#      Priority is given to EDID files that are prefixed with "db".
# -----------------------------------------------------------------

#*****************************************************************# 
#*****************************************************************# 
#           S C R I P T   G L O B A L  V A R I A B L E S          #
#*****************************************************************# 
#*****************************************************************#
FILEFOUND=0  
PATH_TO_TOOLS="/usr/local/bin" 
PATH_TO_OVERRIDE_FILES="/rw/config" 


#*****************************************************************# 
#*****************************************************************# 
#           S C R I P T   M A I N   R O U T I N E S               #
#*****************************************************************# 
#*****************************************************************#
#echo LEdidOverride.sh was launched | sudo tee /dev/kmsg    # For testing purposes.


#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Validate driver is loaded, exit immediately if not loaded.
#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
SV2DRIVER=$(lsmod | grep sv2 >& /dev/null; echo $?)
if [ ${SV2DRIVER} != "0" ]; then exit; fi


#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   
# Main routine.
#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

# Get connector and monitor info.
CONN_INFO=$($PATH_TO_TOOLS/LConnectorTest -gci -gmi) >& /dev/null; 

# Extract specific field data within the connector information.
ARR_OUT=$(echo -e "$CONN_INFO" | grep "VideoOutput:" | awk '{print $2}') >& /dev/null;
ARR_DDC=$(echo -e "$CONN_INFO" | grep "DDC:" | awk '{print $2}') >& /dev/null;
ARR_HPD=$(echo -e "$CONN_INFO" | grep "HPD:" | awk '{print $2}') >& /dev/null;
ARR_OVR=$(echo -e "$CONN_INFO" | grep "IsEdidOverridePresent:" | awk '{print $2}') >& /dev/null;

# Determine how many connectors we are dealing with.
CONN_CNT=$(echo -e "$ARR_OUT" | wc -w)  >& /dev/null;

#Loop through the connectors found.
for (( CONN_IDX=0; CONN_IDX<CONN_CNT; CONN_IDX++ ))
do   
    # Reset flag to no file found.
    FILEFOUND=0
    
    # Get current connector field values from array.
    OUT=$(echo $ARR_OUT | awk '{print $(CONN_IDX+1)}')
    DDC=$(echo $ARR_DDC | awk '{print $(CONN_IDX+1)}')
    HPD=$(echo $ARR_HPD | awk '{print $(CONN_IDX+1)}')
    OVR=$(echo $ARR_OVR | awk '{print $(CONN_IDX+1)}')
    
    # Validate connector is an Output and supports DDC and HPD.
    if [[ $OUT == 1 ]] && [[ $DDC == 1 ]] && [[ $HPD == 1 ]]; then
    
        #Loop through EDID override files for matching connector index.
        for FILE in $PATH_TO_OVERRIDE_FILES/* 
        do
            if [[ $FILE == *"dbedidoverride$CONN_IDX"* ]] || \
               [[ $FILE == *"edidoverride$CONN_IDX"* ]] ; then
               
                # Apply override and set found flag when file is found.
                $PATH_TO_TOOLS/MtxEdidFlasher -c $CONN_IDX -f $FILE -o 
                FILEFOUND=1
                break
            fi
        done
        
        # Remove EDID override when no matching file and override exists.
        if [[ $OVR == 1 ]]  && [[ $FILEFOUND == 0 ]]; then
            $PATH_TO_TOOLS/MtxEdidFlasher -c $CONN_IDX -r
        fi
    fi
done

exit

