/***************************************************************************************\

Module Name:    MtxBaseTypes.h

Description:    Matrox Base Types.

References:     

    Copyright (c) 2006, Matrox Graphics Inc.
    All Rights Reserved.

\***************************************************************************************/

#ifndef INC_MTXBASETYPES_H
#define INC_MTXBASETYPES_H

/***************************************************************************************\

Matrox Base Types Overview
--------------------------

    This file defines the following types:

        MVOID       Standard void type for consistency
        MINT        Variable sized signed integer type
        MINT8       8-bit signed integer type
        MINT16      16-bit signed integer type
        MINT32      32-bit signed integer type
        MINT64      64-bit signed integer type
        MINTPTR     Signed integer type same size as pointer
        MUINT       Variable sized unsigned integer type
        MUINT8      8-bit unsigned integer type
        MUINT16     16-bit unsigned integer type
        MUINT32     32-bit unsigned integer type
        MUINT64     64-bit unsigned integer type
        MUINTPTR    Unsigned integer type same size as pointer
        MCHAR       Basic character type
        MWCHAR      Wide character type
        MCHAR8      Ansi or MBCS character type
        MCHAR16     Unicode UTF-16 character type
        MFLOAT32    Standard single-precision floating point type
        MFLOAT64    Standard double-precision floating point type
        MBOOL       Variable sized boolean type
        MBOOL8      8-bit boolean type
        MBOOL16     16-bit boolean type
        MBOOL32     32-bit boolean type
        MBOOL64     64-bit boolean type
        MFLAG8      8-bit flag type
        MFLAG16     16-bit flag type
        MFLAG32     32-bit flag type
        MFLAG64     64-bit flag type

    This file defines the following macros:

        MFALSE      Boolean value for MBOOLx types
        MTRUE       Boolean value for MBOOLx types
        MINTx_MIN   Minimum values for signed integers
        MINTx_MAX   Maximum values for signed integers
        MUINTx_MIN  Minimum values for unsigned integers
        MUINTx_MAX  Maximum values for unsigned integers
        MFLOATx_MIN Minimum values for floating point types
        MFLOATx_MAX Maximum values for floating point types
        MNULL       Null pointer constant for pointer types

    This file sets up the following private configuration macros:

        MPRIV_COMPILER_VENDOR_MICROSOFT
            Detection of microsoft compiler

        MPRIV_COMPILER_VENDOR_GNU
            Detection of GNU compiler (gcc or g++)

        MPRIV_COMPILER_VENDOR_INTEL
            Detection of Intel compiler

        MPRIV_TARGET_PROCESSOR_X86
            Detection of X86 family of processors

        MPRIV_TARGET_PROCESSOR_AMD64
            Detection of AMD64 processor

        MPRIV_TARGET_PROCESSOR_IA64
            Detection of IA64 processor

        MPRIV_TARGET_PROCESSOR_ARM
            Detection of ARM processor

        MPRIV_TARGET_PROCESSOR_EBC
            Detection of EBC processor

        MPRIV_TARGET_PROCESSOR_MICROBLAZE
            Detection of Microblaze processor

        MPRIV_BYTES_PER_Xx
            Size of compiler-specific types

\***************************************************************************************/

/***************************************************************************************\

Macro:          MPRIV_COMPILER_VENDOR_Xx

Description:    Vendor type for compiler.  Vendor may be GNU or MICROSOFT.  Other
                vendors may be added such as BORLAND or INTEL.

Comments:     - If you add a new compiler vendor, please adjust all base types and
                macros for the new compiler vendor.

              - The microsoft compiler should be detected last since the _MSC_VER macro
                is sometimes defined by header files to fake a microsoft compiler.

              - To differentiate between different versions of the microsoft compiler,
                first detect the microsoft vendor, then compare _MSC_VER with the
                following version numbers:

                     800    Visual C++ 1.52 (1993)
                    1200    Visual C++ 6.0  (1998)
                    1300    Visual C++ 7.0  (2002)
                    1310    Visual C++ 7.1  (2003)
                    1400    Visual C++ 8.0  (2005)

\***************************************************************************************/
#if defined(__GNUC__)
    /* Detected GNU compiler */
#   define MPRIV_COMPILER_VENDOR_GNU
/* Insert detection of other vendors here using #elif... */
#elif defined(__INTEL_EBC_COMPILER)
#   define MPRIV_COMPILER_VENDOR_INTEL
#elif defined(_MSC_VER)
    /* Detected Microsoft compiler */
#   define MPRIV_COMPILER_VENDOR_MICROSOFT
#else
#   error Unknown compiler vendor!
#endif

/***************************************************************************************\

Macro:          MPRIV_TARGET_PROCESSOR_Xx

Description:    Type of processor running on target machine.

Comments:     - The ARM processor currently supported is only 32-bit.  If a 64-bit ARM
                processor exists, then please add a ARM64 processor type.

\***************************************************************************************/
#if defined(MPRIV_COMPILER_VENDOR_MICROSOFT)
    /* Detecting processor for Microsoft compiler... */
#   if defined(_M_IA64)
        /* Detected IA64 processor */
#       define MPRIV_TARGET_PROCESSOR_IA64
#   elif defined(_M_AMD64)
        /* Detected AMD64 processor */
#       define MPRIV_TARGET_PROCESSOR_AMD64
#   elif (defined(_M_IX86) || \
          defined(_M_I86))
        /* Detected X86 processor */
#       define MPRIV_TARGET_PROCESSOR_X86
#   elif defined(_M_ARM)
        /* Detected ARM processor */
#       define MPRIV_TARGET_PROCESSOR_ARM
    /* Insert detection of other processors here using #elif... */
#   else
#       error Unable to detect processor!
#   endif
#elif defined(MPRIV_COMPILER_VENDOR_GNU)
    /* Detecting processor for GNU compiler... */
#   if (defined(_x86_64) || defined(__x86_64) || defined(__x86_64__) || defined(__amd64) || defined(__amd64__))
        /* Detected AMD64 processor */
#       define MPRIV_TARGET_PROCESSOR_AMD64
#   elif (defined(__ia64) || defined(__ia64__))
        /* Detected IA64 processor */
#       define MPRIV_TARGET_PROCESSOR_IA64
#   elif (defined(__i386) || defined(__i386__))
        /* Detected X86 processor */
#       define MPRIV_TARGET_PROCESSOR_X86
#   elif (defined(__arm) || defined(__arm__))
        /* Detected ARM processor */
#       define MPRIV_TARGET_PROCESSOR_ARM
#   elif (defined(__microblaze__) || defined (__MICROBLAZE__))
        /* Detected Microblaze processor */
#       define MPRIV_TARGET_PROCESSOR_MICROBLAZE
    /* Insert detection of other processors here using #elif... */
#   else
#       error Unable to detect processor!
#   endif
/* Handle other compiler vendors here using #elif... */
#elif defined(MPRIV_COMPILER_VENDOR_INTEL)
    /* Detecting processor for Intel compiler */
#   if defined(__INTEL_EBC_COMPILER)
#       if (defined(_INTEGRAL_MAX_BITS) && \
            (_INTEGRAL_MAX_BITS == 64))
#           define MPRIV_TARGET_PROCESSOR_EBC
#       else
#           error Intel EBC Compiler should support 64bits!
#       endif
#   else
#       error Unable to detect processor!
#   endif
#else /* #if defined(MPRIV_COMPILER_VENDOR_Xx) */
#   error Unable to detect processor!
#endif /* #if defined(MPRIV_COMPILER_VENDOR_Xx) */

/***************************************************************************************\

Macro:          MPRIV_BYTES_PER_Xx

Description:    Detect presence and size of compiler-specific types.  These definitions
                are used to select the right type for each sized integer types.  The
                macro must be defined to zero for unsupported types. 

Comments:     - Please do not use sizeof since the preprocessor does not know how
                to preprocess that keyword and we need the size before compile-time.

              - You may add an ASSERT to check that these sizes match the result of
                the sizeof(Xx) compile-time expression.

              - The __intx types are only available for microsoft compilers.

              - The __int64 type is only available for certain versions of microsoft
                compilers.

              - The wchar_t is not always a native type.  The __wchar_t type is supported
                by microsoft compilers since version 1300 Visual C++ 7.0 (2002).  The
                wchar_t may be a native type if the /Zc:wchar_t compiler switch is
                added, we detect that in the code using the _NATIVE_WCHAR_T_DEFINED
                preprocessor macro.

              - The long long type is supported by microsoft starting at build 1310
                Visual C++ 7.1 (2003).  It is identical to __int64 so no casting
                problems when using one or the other.

\***************************************************************************************/

/* sizeof(char) */
#define MPRIV_BYTES_PER_CHAR                    1

/* sizeof(wchar_t) */
#if defined(MPRIV_COMPILER_VENDOR_GNU)
#  if defined(__SIZEOF_WCHAR_T__)
#   define MPRIV_BYTES_PER_WCHAR                __SIZEOF_WCHAR_T__
#  elif (defined(__MINGW32__)  || \
         defined(__MINGW64__)  || \
         defined(__CYGWIN__)   || \
         defined(__CYGWIN32__) || \
         defined(__WIN32__)    || \
         defined(_WIN32))
#   define MPRIV_BYTES_PER_WCHAR                2
#  else
#   define MPRIV_BYTES_PER_WCHAR                4
#  endif
#elif defined(MPRIV_COMPILER_VENDOR_MICROSOFT)
#   define MPRIV_BYTES_PER_WCHAR                2
#else
#   define MPRIV_BYTES_PER_WCHAR                0
#endif

/* Is wchar_t a native type ?*/
#if (defined(MPRIV_COMPILER_VENDOR_GNU) && \
      defined(__cplusplus))
#   define MPRIV_WCHAR_T_DEFINED
#elif (defined(MPRIV_COMPILER_VENDOR_MICROSOFT) && \
        defined(_WCHAR_T_DEFINED) && \
        defined(_NATIVE_WCHAR_T_DEFINED))
#   define MPRIV_WCHAR_T_DEFINED
#endif

/* sizeof(short int) */
#define MPRIV_BYTES_PER_SHORT_INT               2

/* sizeof(int) */
#if (defined(MPRIV_COMPILER_VENDOR_MICROSOFT) && \
     (_MSC_VER == 800))
#   define MPRIV_BYTES_PER_INT                  2
#else
#   define MPRIV_BYTES_PER_INT                  4
#endif

/* sizeof(long int) */
#if defined(MPRIV_COMPILER_VENDOR_MICROSOFT)
#   define MPRIV_BYTES_PER_LONG_INT             4
#elif defined(MPRIV_COMPILER_VENDOR_GNU) && defined(__SIZEOF_LONG__)
#   if (__SIZEOF_LONG__ == 4)
#       define MPRIV_BYTES_PER_LONG_INT         4
#   elif (__SIZEOF_LONG__ == 8)
#       define MPRIV_BYTES_PER_LONG_INT         8
#   else
#       error Unable to detect sizeof long
#   endif
#elif (defined(MPRIV_COMPILER_VENDOR_INTEL) && \
    defined(MPRIV_TARGET_PROCESSOR_EBC))
#   define MPRIV_BYTES_PER_LONG_INT             0
#else /* #if defined(MPRIV_COMPILER_VENDOR_Xx) */
    /* size of long varies for GNU compiler */
#   if (defined(MPRIV_TARGET_PROCESSOR_AMD64) || \
        defined(MPRIV_TARGET_PROCESSOR_IA64))
        /* long is 64-bit for 64-bit processor */
#       define MPRIV_BYTES_PER_LONG_INT         8
#   else /* #if defined(MPRIV_TARGET_PROCESSOR_Xx) */
        /* long is 32-bit for 32-bit processor */
#       define MPRIV_BYTES_PER_LONG_INT         4
#   endif /* #if defined(MPRIV_TARGET_PROCESSOR_Xx) */
#endif /* #if defined(MPRIV_COMPILER_VENDOR_Xx) */

/* sizeof(long long) */
#if defined(MPRIV_COMPILER_VENDOR_MICROSOFT)
#   define MPRIV_BYTES_PER_LONG_LONG            0
#elif defined(MPRIV_COMPILER_VENDOR_GNU) && defined(__SIZEOF_LONG_LONG__)
#   if (__SIZEOF_LONG_LONG__ == 8)
#       define MPRIV_BYTES_PER_LONG_LONG        8
#   elif (__SIZEOF_LONG_LONG__ == 16)
#       define MPRIV_BYTES_PER_LONG_LONG        16
#   else
#       error Unable to detect sizeof long long
#   endif
#elif (defined(MPRIV_COMPILER_VENDOR_INTEL) && \
    defined(MPRIV_TARGET_PROCESSOR_EBC))
#   define MPRIV_BYTES_PER_LONG_LONG            0
#else
#   if (defined(MPRIV_TARGET_PROCESSOR_AMD64) || \
        defined(MPRIV_TARGET_PROCESSOR_IA64))
#       define MPRIV_BYTES_PER_LONG_LONG        16
#   else
#       define MPRIV_BYTES_PER_LONG_LONG        8
#   endif
#endif

/* sizeof(float) */
#if (defined(MPRIV_COMPILER_VENDOR_INTEL) && \
    defined(MPRIV_TARGET_PROCESSOR_EBC))
#   define MPRIV_BYTES_PER_FLOAT                0
#else
    #define MPRIV_BYTES_PER_FLOAT               4
#endif

/* sizeof(double) */
#if (defined(MPRIV_COMPILER_VENDOR_INTEL) && \
    defined(MPRIV_TARGET_PROCESSOR_EBC))
#   define MPRIV_BYTES_PER_DOUBLE               0
#else
    #define MPRIV_BYTES_PER_DOUBLE              8
#endif

/* @todo: support for long double */

/* @todo: support for bool */

/* sizeof(__int8) */
#if (defined(MPRIV_COMPILER_VENDOR_MICROSOFT) && \
     (_MSC_VER > 800))
#   define MPRIV_BYTES_PER_INT8                 1
#else
#   define MPRIV_BYTES_PER_INT8                 0
#endif

/* sizeof(__int16) */
#if (defined(MPRIV_COMPILER_VENDOR_MICROSOFT) && \
     (_MSC_VER > 800))
#   define MPRIV_BYTES_PER_INT16                2
#else
#   define MPRIV_BYTES_PER_INT16                0
#endif

/* sizeof(__int32) */
#if (defined(MPRIV_COMPILER_VENDOR_MICROSOFT) && \
     (_MSC_VER > 800))
#   define MPRIV_BYTES_PER_INT32                4
#else
#   define MPRIV_BYTES_PER_INT32                0
#endif

/* sizeof(__int64) */
#if (defined(MPRIV_COMPILER_VENDOR_MICROSOFT) && \
     (_MSC_VER > 800))
#   define MPRIV_BYTES_PER_INT64                8
#elif (defined(MPRIV_COMPILER_VENDOR_INTEL) && \
    defined(MPRIV_TARGET_PROCESSOR_EBC))
#   define MPRIV_BYTES_PER_INT64                8
#else
#   define MPRIV_BYTES_PER_INT64                0
#endif

/* @todo: support for __int128 */

/* sizeof(void*) */
#if (defined(MPRIV_TARGET_PROCESSOR_AMD64)  || \
     defined(MPRIV_TARGET_PROCESSOR_IA64)   || \
     defined(MPRIV_TARGET_PROCESSOR_EBC)    )
#   define MPRIV_BYTES_PER_POINTER              8
#else
#   define MPRIV_BYTES_PER_POINTER              4
#endif

/*

    For 16-bit compiler, pointers may be 16-bit or 32-bit, depending on
    memory model.

        TINY    near code, near data
        SMALL   near code, near data
        MEDIUM   far code, near data
        COMPACT near code,  far data
        LARGE    far code,  far data
        HUGE     far code,  far data (> 64Kb)

*/
#if 0
#   if (defined(MPRIV_COMPILER_VENDOR_MICROSOFT) && \
        (_MSC_VER == 800))
#       if (defined(_M_I86CM) || defined(_M_I86LM) || defined(_M_I86HM))
#           define MPRIV_BYTES_PER_POINTER          4
#       else
#           define MPRIV_BYTES_PER_POINTER          2
#       endif
#   endif
#endif

/***************************************************************************************\

Macro:          MPRIV_SUPPORT_Mx

Description:    Indicate support for each matrox base type.  The matrox base type is
                supported when the macro is defined, else the type is unsupported.

Comments:     - The only exceptions so far are the 64-bit types which are not supported
                by the microsoft 16-bit compiler.

\***************************************************************************************/

/* Support for matrox sized types */
#define MPRIV_SUPPORT_MINT
#define MPRIV_SUPPORT_MINT8
#define MPRIV_SUPPORT_MINT16
#define MPRIV_SUPPORT_MINT32
#define MPRIV_SUPPORT_MINT64
#define MPRIV_SUPPORT_MINTPTR
#define MPRIV_SUPPORT_MCHAR
#define MPRIV_SUPPORT_MWCHAR
#define MPRIV_SUPPORT_MCHAR8
#define MPRIV_SUPPORT_MCHAR16
#if !(defined(MPRIV_COMPILER_VENDOR_INTEL)  && \
    defined(MPRIV_TARGET_PROCESSOR_EBC)     )
#   define MPRIV_SUPPORT_MFLOAT32
#   define MPRIV_SUPPORT_MFLOAT64
#endif
#define MPRIV_SUPPORT_MBOOL
#define MPRIV_SUPPORT_MBOOL8
#define MPRIV_SUPPORT_MBOOL16
#define MPRIV_SUPPORT_MBOOL32
#define MPRIV_SUPPORT_MBOOL64
#define MPRIV_SUPPORT_MFLAG8
#define MPRIV_SUPPORT_MFLAG16
#define MPRIV_SUPPORT_MFLAG32
#define MPRIV_SUPPORT_MFLAG64

/* Special case for 16-bit compiler: no support for 64-bit types! */
#if (defined(MPRIV_COMPILER_VENDOR_MICROSOFT) && \
     (_MSC_VER == 800))
#   undef MPRIV_SUPPORT_MINT64
#   undef MPRIV_SUPPORT_MBOOL64
#   undef MPRIV_SUPPORT_MFLAG64
#endif

/***************************************************************************************\

Type:           MVOID

Description:    Standard void type. This is included for type naming consistency.

Comments:       None.

\***************************************************************************************/
typedef void MVOID;

/***************************************************************************************\

Type:           MINT, MUINT

Description:    Compiler-specific unsized integer types.  Size is guaranteed to be at least 16 bits.

Comments:     - Be very careful when using this type in structures that are shared
                between modules.  The size of the type may be different for each module.

              - You may use this type for local variables.  For example: a loop counter
                or storage for the intermediate result of a calculation.

              - To simulate 16 bit environment, test with the type forced to a short
                int.

\***************************************************************************************/
#if defined(MPRIV_SUPPORT_MINT)
typedef   signed int MINT;
typedef unsigned int MUINT;
#endif /* #if defined(MPRIV_SUPPORT_MINT) */

/***************************************************************************************\

Type:           MINT8, MUINT8

Description:    8-bit signed and unsigned integer types.

Comments:       None.

\***************************************************************************************/
#if defined(MPRIV_SUPPORT_MINT8)
#   if (MPRIV_BYTES_PER_CHAR == 1)
typedef   signed char MINT8;
typedef unsigned char MUINT8;
#   elif (MPRIV_BYTES_PER_INT8 == 1)
typedef   signed __int8 MINT8;
typedef unsigned __int8 MUINT8;
#   else
#       error Unable to define MINT8 and MUINT8 types!
#   endif
#endif /* #if defined(MPRIV_SUPPORT_MINT8) */

/***************************************************************************************\

Type:           MINT16, MUINT16

Description:    16-bit signed and unsigned integer types.

Comments:       None.

\***************************************************************************************/
#if defined(MPRIV_SUPPORT_MINT16)
#   if (MPRIV_BYTES_PER_SHORT_INT == 2)
typedef   signed short int MINT16;
typedef unsigned short int MUINT16;
#   elif (MPRIV_BYTES_PER_INT16 == 2)
typedef   signed __int16 MINT16;
typedef unsigned __int16 MUINT16;
#   elif (MPRIV_BYTES_PER_INT == 2)
typedef   signed int MINT16;
typedef unsigned int MUINT16;
#   else
#       error Unable to define MINT16 and MUINT16 types!
#   endif
#endif /* #if defined(MPRIV_SUPPORT_MINT16) */

/***************************************************************************************\

Type:           MINT32, MUINT32

Description:    32-bit signed and unsigned integer types.

Comments:       None.

\***************************************************************************************/
#if defined(MPRIV_SUPPORT_MINT32)
#   if (MPRIV_BYTES_PER_INT == 4)
typedef   signed int MINT32;
typedef unsigned int MUINT32;
#   elif (MPRIV_BYTES_PER_INT32 == 4)
typedef   signed __int32 MINT32;
typedef unsigned __int32 MUINT32;
#   elif (MPRIV_BYTES_PER_LONG_INT == 4)
typedef   signed long int MINT32;
typedef unsigned long int MUINT32;
#   else
#       error Unable to define MINT32 and MUINT32 types!
#   endif
#endif /* #if defined(MPRIV_SUPPORT_MINT32) */

/***************************************************************************************\

Type:           MINT64, MUINT64

Description:    64-bit signed and unsigned integer types.

Comments:     - Declare MINT64 literals using cast operation.  This only works if the
                literal does not exceed 32-bit limits.  For example:

                    MINT64 iTest = ((MINT64)0x12345678);
                    MUINT64 uiTest = ((MUINT64)0x12345678);

              - For older microsoft compilers (_MSC_VER < 1310), literals may be declared
                using I64 or UI64 suffix.  For compilers that support the "long long"
                integer types, such as newer microsoft compilers (_MSC_VER >= 1310) and
                for GNU compilers, literals may be declared using LL or ULL suffix.

              - Update: added MINT64_CONST and MUINT64_CONST macros to help create
                literals for MINT64 and MUINT64 types.  Please use these macros to
                generate literals that exceed 32-bit limits.  For example:

                    MINT64 iTest = MINT64_CONST(0x123456789ABCDEF0);
                    MUINT64 uiTest = MUINT64_CONST(0x123456789ABCDEF0);

\***************************************************************************************/
#if defined(MPRIV_SUPPORT_MINT64)
#   if (MPRIV_BYTES_PER_INT64 == 8)
typedef   signed __int64 MINT64;
typedef unsigned __int64 MUINT64;
#define MINT64_CONST(i64Value) ((MINT64)(i64Value ## I64))
#define MUINT64_CONST(ui64Value) ((MUINT64)(ui64Value ## UI64))
#   elif (MPRIV_BYTES_PER_INT == 8)
typedef   signed int MINT64;
typedef unsigned int MUINT64;
#define MINT64_CONST(i64Value) ((MINT64)(i64Value))
#define MUINT64_CONST(ui64Value) ((MUINT64)(ui64Value))
#   elif (MPRIV_BYTES_PER_LONG_INT == 8)
typedef   signed long int MINT64;
typedef unsigned long int MUINT64;
#define MINT64_CONST(i64Value) ((MINT64)(i64Value ## L))
#define MUINT64_CONST(ui64Value) ((MUINT64)(ui64Value ## UL))
#   elif (MPRIV_BYTES_PER_LONG_LONG == 8)
typedef   signed long long MINT64;
typedef unsigned long long MUINT64;
#define MINT64_CONST(i64Value) ((MINT64)(i64Value ## LL))
#define MUINT64_CONST(ui64Value) ((MUINT64)(ui64Value ## ULL))
#   else
#       error Unable to define MINT64 type!
#   endif
#endif /* #if defined(MPRIV_SUPPORT_MINT64) */

/***************************************************************************************\

Type:           MINTPTR, MUINTPTR

Description:    signed and unsigned integer types of same size as pointer.

Comments:     - Use these types to do pointer arithmetic.

\***************************************************************************************/
#if defined(MPRIV_SUPPORT_MINTPTR)
#   if (MPRIV_BYTES_PER_POINTER == 8)
typedef MINT64  MINTPTR;
typedef MUINT64 MUINTPTR;
#   elif (MPRIV_BYTES_PER_POINTER == 4)
#       if (defined(MPRIV_COMPILER_VENDOR_MICROSOFT) && \
            defined(MPRIV_TARGET_PROCESSOR_X86) && \
            (_MSC_VER >= 1300))
typedef __w64 MINT32  MINTPTR;
typedef __w64 MUINT32 MUINTPTR;
#       else
typedef MINT32  MINTPTR;
typedef MUINT32 MUINTPTR;
#       endif
#   elif (MPRIV_BYTES_PER_POINTER == 2)
typedef MINT16  MINTPTR;
typedef MUINT16 MUINTPTR;
#   else
#       error Unable to define MINTPTR type!
#   endif
#endif /* #if defined(MPRIV_SUPPORT_MINTPTR) */

/***************************************************************************************\

Type:           MxINTx_MIN, MxINTx_MAX, MFLOATx_MIN, MFLOATx_MAX

Description:    Integer and floating-point limits.

Comments:     - The minimum value for an n-bit unsigned integer is binary 0 for all n
                bits.  Ex: for the MUINT32 type, the minimum is equal to 0x00000000.

              - The maximum value for an n-bit unsigned integer is binary 1 for all n
                bits.  Ex: for the MUINT32 type, the maximum is equal to 0xFFFFFFFF.

              - The minimum value for an n-bit signed integer is binary 0 for all n
                bits except for the sign bit which must be set to 1.  Ex: for the MINT32
                type, the minimum is equal to 0x80000000.

              - The maximum value for an n-bit signed integer is binary 1 for all n
                bits except for the sign bit which must be set to 0.  Ex: for the MINT32
                type, the maximum is equal to 0x7FFFFFFF.

              - This method assumes that signed integers use two's complement.

\***************************************************************************************/

#define MUINT_MIN         ((MUINT)0)
#define MUINT_MAX         ((MUINT)~MUINT_MIN)
#define MINT_MAX          ((MINT)(MUINT_MAX >> 1))
#define MINT_MIN          ((MINT)(~(MINT_MAX)))

#define MUINT8_MIN        ((MUINT8)0)
#define MUINT8_MAX        ((MUINT8)~MUINT8_MIN)
#define MINT8_MAX         ((MINT8)(MUINT8_MAX >> 1))
#define MINT8_MIN         ((MINT8)(~(MINT8_MAX)))

#define MUINT16_MIN       ((MUINT16)0)
#define MUINT16_MAX       ((MUINT16)~MUINT16_MIN)
#define MINT16_MAX        ((MINT16)(MUINT16_MAX >> 1))
#define MINT16_MIN        ((MINT16)(~(MINT16_MAX)))

#define MUINT32_MIN       ((MUINT32)0)
#define MUINT32_MAX       ((MUINT32)~MUINT32_MIN)
#define MINT32_MAX        ((MINT32)(MUINT32_MAX >> 1))
#define MINT32_MIN        ((MINT32)(~(MINT32_MAX)))

#define MUINT64_MIN       ((MUINT64)0)
#define MUINT64_MAX       ((MUINT64)~MUINT64_MIN)
#define MINT64_MAX        ((MINT64)(MUINT64_MAX >> 1))
#define MINT64_MIN        ((MINT64)(~(MINT64_MAX)))

#define MFLOAT32_MIN      1.175494351e-38F
#define MFLOAT32_MAX      3.402823466e+38F
#define MFLOAT64_MIN      2.2250738585072014e-308
#define MFLOAT64_MAX      1.7976931348623158e+308

/***************************************************************************************\

Type:           MCHAR

Description:    Compiler-specific basic character type.  

Comments:     - This type is used to hold character literals (ex 'a').

              - The size of this type is guaranteed to be 1.

\***************************************************************************************/

#if defined(MPRIV_SUPPORT_MCHAR)
#   if (MPRIV_BYTES_PER_CHAR == 1)
typedef char MCHAR;
#   elif (MPRIV_BYTES_PER_INT8 == 1)
typedef __int8 MCHAR;
#   else
#       error Unable to define MCHAR type!
#   endif
#endif /* #if defined(MPRIV_SUPPORT_MCHAR) */

/***************************************************************************************\

Type:           MWCHAR

Description:    Compiler-specific wide character type.  

Comments:     - This type is used to hold wide character literals (ex L'a').

              - The size of this type is compiler dependent.

\***************************************************************************************/

#if defined(MPRIV_SUPPORT_MWCHAR)
#   if defined(MPRIV_WCHAR_T_DEFINED)
typedef wchar_t MWCHAR;
#   elif (MPRIV_BYTES_PER_WCHAR == 2)
#       if (MPRIV_BYTES_PER_SHORT_INT == 2)
typedef unsigned short int MWCHAR;
#       elif (MPRIV_BYTES_PER_INT == 2)
typedef unsigned int MWCHAR;
#       elif (MPRIV_BYTES_PER_INT16 == 2)
typedef unsigned __int16 MWCHAR;
#       else
#           error Unable to define MWCHAR type!
#       endif
#   elif (MPRIV_BYTES_PER_WCHAR == 4)
#      if (MPRIV_BYTES_PER_INT == 4)
typedef int MWCHAR;
#      elif (MPRIV_BYTES_PER_INT32 == 4)
typedef __int32 MWCHAR;
#      elif (MPRIV_BYTES_PER_LONG_INT == 4)
typedef long int MWCHAR;
#      else
#          error Unable to define MWCHAR type!
#      endif
#   else
#       error Unable to define MWCHAR type!
#   endif
#endif /* #if defined(MPRIV_SUPPORT_MWCHAR) */


/***************************************************************************************\

Type:           MCHARXx

Description:    8-bit and 16-bit character types.

Comments:     - Use MCHAR8 for ANSI and MBCS characters.  Use MCHAR16 for UNICODE UTF-16
                characters.

              - Declare MCHAR16 literals with L prefix (L'Z' or L"Hello World").  No
                prefix is necessary for MCHAR8 literals.

              - Please use a compiler command line switch to select between a signed
                or an unsigned char.  By default a char is usually signed.

\***************************************************************************************/
#if defined(MPRIV_SUPPORT_MCHAR8)
#   if (MPRIV_BYTES_PER_CHAR == 1)
typedef char MCHAR8;
#   elif (MPRIV_BYTES_PER_INT8 == 1)
typedef __int8 MCHAR8;
#   else
#       error Unable to define MCHAR8 type!
#   endif
#endif /* #if defined(MPRIV_SUPPORT_MCHAR8) */

#if defined(MPRIV_SUPPORT_MCHAR16)
#   if ((MPRIV_BYTES_PER_WCHAR == 2) && \
        defined(MPRIV_WCHAR_T_DEFINED))
typedef wchar_t MCHAR16;
#   elif (MPRIV_BYTES_PER_SHORT_INT == 2)
typedef unsigned short int MCHAR16;
#   elif (MPRIV_BYTES_PER_INT == 2)
typedef unsigned int MCHAR16;
#   elif (MPRIV_BYTES_PER_INT16 == 2)
typedef unsigned __int16 MCHAR16;
#   else
#       error Unable to define MCHAR16 type!
#   endif
#endif /* #if defined(MPRIV_SUPPORT_MCHAR16) */

/***************************************************************************************\

Type:           MFLOATXx

Description:    Sized floating-point types.

Comments:       None.

\***************************************************************************************/
#if defined(MPRIV_SUPPORT_MFLOAT32)
#   if (MPRIV_BYTES_PER_FLOAT == 4)
typedef float MFLOAT32;
#   elif (MPRIV_BYTES_PER_DOUBLE == 4)
typedef double MFLOAT32;
#   else
#       error Unable to define MFLOAT32 type!
#   endif
#endif /* #if defined(MPRIV_SUPPORT_MFLOAT32) */

#if defined(MPRIV_SUPPORT_MFLOAT64)
#   if (MPRIV_BYTES_PER_DOUBLE == 8)
typedef double MFLOAT64;
#   else
#       error Unable to define MFLOAT64 type!
#   endif
#endif /* #if defined(MPRIV_SUPPORT_MFLOAT64) */

/***************************************************************************************\

Macro:          MFALSE, MTRUE

Description:    Values to use with MBOOLXx types

Comments:       

\***************************************************************************************/
#define MFALSE  0
#define MTRUE   1

/***************************************************************************************\

Macro:          MNULL

Description:    Values to use with pointers types.

Comments:     - Do not use MNULL with MINTPTR and MUINTPTR types when the source
                language is C.  Use the literal constant zero (0) for the MINTPTR
                and MUINTPTR types.

\***************************************************************************************/
#if defined(__cplusplus)
#   define  MNULL   0
#else
#   define  MNULL   ((void *)0)
#endif

/***************************************************************************************\

Macro:          MNULL64

Description:    Null pointer forced to 64bits

Comments:       When holding handles from another processor we should
                use 64bits handle, which in turns should be initialized / compared with
                this constant.

\***************************************************************************************/
#define MNULL64         ((MUINT64)0)

/***************************************************************************************\

Type:           MBOOLXx

Description:    Sized boolean types.

Comments:     - Please use these types with MFALSE and MTRUE constants.

              - For C++ only code that is not shared between modules, you may use the
                C++ bool compiler type.  Beware! the sizeof(bool) varies according to
                compiler and processor.  It varies between 1 and 8 bytes.

\***************************************************************************************/
#if defined(MPRIV_SUPPORT_MBOOL)
typedef MINT    MBOOL;
#endif
#if defined(MPRIV_SUPPORT_MBOOL8)
typedef MINT8   MBOOL8;
#endif
#if defined(MPRIV_SUPPORT_MBOOL16)
typedef MINT16  MBOOL16;
#endif
#if defined(MPRIV_SUPPORT_MBOOL32)
typedef MINT32  MBOOL32;
#endif
#if defined(MPRIV_SUPPORT_MBOOL64)
typedef MINT64  MBOOL64;
#endif

/***************************************************************************************\

Type:           MFLAGXx

Description:    Sized flag types.

Comments:       None.

\***************************************************************************************/
#if defined(MPRIV_SUPPORT_MFLAG8)
typedef MUINT8  MFLAG8;
#endif
#if defined(MPRIV_SUPPORT_MFLAG16)
typedef MUINT16 MFLAG16;
#endif
#if defined(MPRIV_SUPPORT_MFLAG32)
typedef MUINT32 MFLAG32;
#endif
#if defined(MPRIV_SUPPORT_MFLAG64)
typedef MUINT64 MFLAG64;
#endif

/***************************************************************************************\

Type:           MHANDLE

Description:    A generic matrox handle type

Comments:       None.

\***************************************************************************************/
typedef void*     MHANDLE;

#endif  /* #ifndef INC_MTXBASETYPES_H */

