/************************************************************************************************************\

Module Name:    LVideoProcessor.h

Description:    Liberatus for Video Processor module interface (This include the VPC, VPE and commune types VP).

References:     LVideoProcessor.doc.

    Copyright (c) 2013, Matrox Graphics Inc.
    All Rights Reserved.

\************************************************************************************************************/

#ifndef INC_LVIDEOPROCESSOR_H
#define INC_LVIDEOPROCESSOR_H

// -----------------------------------------------------------------------------------------------------------
//                                   I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------
typedef struct LVPC_OBJECT*          LVPC_Handle;
typedef struct LVPE_OBJECT*          LVPE_Handle;

// -----------------------------------------------------------------------------------------------------------
//                                   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

// Constant and types are defined in the following order:
// Generic VP (VPC and VPE) types
// VPC types
// VPE types

// ---------------------------------------------- V P   T Y P E S --------------------------------------------

/************************************************************************************************************\

Structure:      LVP_Matrix3x3f

Description:    Defines a 3x3 matrix with single-precision values.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVP_Matrix3x3f
{
    MFLOAT32 fMatrix[3][3];
} LVP_Matrix3x3f;

/************************************************************************************************************\

Structure:      LVP_Vector3f

Description:    Defines a 3-element vector with single-precision values.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVP_Vector3f
{
    MFLOAT32 x;
    MFLOAT32 y;
    MFLOAT32 z;
} LVP_Vector3f;

/************************************************************************************************************\

Enumeration:    LVP_AlphaFillMode

Description:    Specifies how the output alpha values are calculated for execute frame operations.

Comments:       For the VPC, LVP_AlphaFillMode_SOURCE_STREAM is the output alpha of the compositor.

\************************************************************************************************************/
typedef enum tagLVP_AlphaFillMode
{
    LVP_AlphaFillMode_SOURCE_STREAM = 0,
    LVP_AlphaFillMode_ALPHA_VALUE   = 1,
    LVP_AlphaFillMode_SOURCE_MASKED = 2
} LVP_AlphaFillMode;

/************************************************************************************************************\

Structure:      LVP_TargetAlphaFill

Description:    Specifies how the output alpha values are calculated for the execution of frame operations.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVP_TargetAlphaFill
{
    LVP_AlphaFillMode   eMode;
    MFLOAT32            fAlphaValue;
} LVP_TargetAlphaFill;

/************************************************************************************************************\

Structure:      LVP_ChromaFilterMode

Description:    Specify the Chroma filtering mode to be applied on the target.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVP_ChromaFilterMode
{
    LVP_ChromaFilterMode_DISABLE = 0,
    LVP_ChromaFilterMode_ENABLE  = 1
} LVP_ChromaFilterMode;

/************************************************************************************************************\

Structure:      LVP_ChromaFilter

Description:    Specify if the Chroma filtering needs to be applied on the target.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVP_ChromaFilter
{
    LVP_ChromaFilterMode eMode;
} LVP_ChromaFilter;

/************************************************************************************************************\

Enumeration:    LVP_TransferFunction

Description:    Defines the transfer function of an RGB surface format.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVP_TransferFunction
{
    LVP_TransferFunction_LINEAR = 0,
    LVP_TransferFunction_BT709  = 1,
    LVP_TransferFunction_sRGB   = 2,
    LVP_TransferFunction_Other  = 3
} LVP_TransferFunction;

/************************************************************************************************************\

Enumeration:    LVP_TransferMatrix

Description:    Defines the transfer matrices between Y'CbCr and R'G'B'.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVP_TransferMatrix
{
    LVP_TransferMatrix_BT601        = 0,
    LVP_TransferMatrix_BT709        = 1,
    LVP_TransferMatrix_USER_DEFINED = 2
} LVP_TransferMatrix;

/************************************************************************************************************\

Structure:      LVP_Color

Description:    Defines a color value.

Comments:       Values have a nominal range of [0...1].

\************************************************************************************************************/
typedef struct tagLVP_Color
{
    MFLOAT32 fYR;
    MFLOAT32 fCbG;
    MFLOAT32 fCrB;
    MFLOAT32 fAlpha;
} LVP_Color;

/************************************************************************************************************\

Structure:      LVP_Color32

Description:    Defines an 8 bits per component color value.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVP_Color32
{
    MUINT8 uiYR;
    MUINT8 uiCbG;
    MUINT8 uiCrB;
    MUINT8 uiAlpha;
} LVP_Color32;

/************************************************************************************************************\

Structure:      LVP_NominalRange

Description:    Specifies the nominal range of a user-defined matrix.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVP_NominalRange
{
    LVP_Color oMin;
    LVP_Color oMax;
} LVP_NominalRange;

/************************************************************************************************************\

Structure:      LVP_UserDefinedMatrix

Description:    Specifies the user-defined matrix.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVP_UserDefinedMatrix
{
    LVP_Matrix3x3f      oYCbCrToRGBMatrix;
    LVP_Vector3f        oYCbCrToRGBVector;
    LVP_NominalRange    oYCbCrToRGBRange;
    LVP_Matrix3x3f      oRGBToYCbCrMatrix;
    LVP_Vector3f        oRGBToYCbCrVector;
    LVP_NominalRange    oRGBToYCbCrRange;
} LVP_UserDefinedMatrix;

/************************************************************************************************************\

Structure:      LVP_ColorSpace

Description:    Specifies the color space for a source, compositor, or target stream.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVP_ColorSpace
{
    LVP_TransferFunction    eFunction;
    LVP_TransferMatrix      eMatrix;
    LVP_UserDefinedMatrix*  poUserMatrix;
} LVP_ColorSpace;

/************************************************************************************************************\

Enumeration:    LVP_FrameFormat

Description:    Describes the content of a frame.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVP_FrameFormat
{
    LVP_FrameFormat_PROGRESSIVE                   = 0,
    LVP_FrameFormat_INTERLACED_TOP_FIELD_FIRST    = 1,
    LVP_FrameFormat_INTERLACED_BOTTOM_FIELD_FIRST = 2,
    LVP_FrameFormat_SINGLE_TOP_FIELD              = 3,
    LVP_FrameFormat_SINGLE_BOTTOM_FIELD           = 4
} LVP_FrameFormat;

/************************************************************************************************************\

Structure:      LVP_SourceFrameFormat

Description:    Specifies how a source is interlaced.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVP_SourceFrameFormat
{
    LVP_FrameFormat eFrameFormat;
} LVP_SourceFrameFormat;

/************************************************************************************************************\

Structure:      LVP_Rect

Description:    Specifies a rectangle for a stream.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVP_Rect
{
    MBOOL   bEnable;
    LRECT32 oRect;
} LVP_Rect;

/************************************************************************************************************\

Structure:      LVP_ProcAmpFilters

Description:    Specifies the ProcAmp control adjustment values on a stream.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVP_ProcAmpFilters
{
    MBOOL       bEnable;
    MFLOAT32    fBrightness;
    MFLOAT32    fContrast;
    MFLOAT32    fHue;
    MFLOAT32    fSaturation;
} LVP_ProcAmpFilters;

// --------------------------------------------- V P C   T Y P E S -------------------------------------------


/************************************************************************************************************\

Enum:           LVPC_AttributeType

Description:    Enumeration of attribute types.

Comments:       None.

\************************************************************************************************************/
typedef enum
{
    LVPC_AttributeType_CREATE  = 0            // Attributes structure for the Create method.
} LVPC_AttributeType;

/************************************************************************************************************\

Structure:      LVPC_CreateAttributes

Description:    LVPC creation attributes.

Comments:       LVPC_AttributeType_CREATE.

\************************************************************************************************************/
typedef struct tagLVPC_CreateAttributes
{
    LVPC_AttributeType  eType;                  // Shall be LVPC_AttributeType_CREATE
    MBOOL32             bLoadBalancing;         // Load balacing
    MBOOL32             bWaitforDMA;            // Wait for DMA
} LVPC_CreateAttributes;

/************************************************************************************************************\

Enumeration:    LVPC_LUT

Description:    The LUT contains the color palette entries for a source stream and the alpha channel used for 
                the LUT alpha blending.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPC_LUT
{
    MBOOL       bYCbCrA;
    MUINT32     uiCount;
    LVP_Color32 *paoEntries;
} LVPC_LUT;

/************************************************************************************************************\

Enumeration:    LVPC_Target

Description:    List of target parameters for the execute frame operation.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVPC_Target
{
    LVPC_Target_RECT            = 0,
    LVPC_Target_COLOR_SPACE     = 1,
    LVPC_Target_ALPHA_FILL      = 2,
    LVPC_Target_ROTATION        = 3,
    LVPC_Target_PROCAMP_FILTERS = 4,
    LVPC_Target_CHROMA_FILTER   = 5
} LVPC_Target;

/************************************************************************************************************\

Enumeration:    LVPC_Source

Description:    List of source parameters for the execute frame operation.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVPC_Source
{
    LVPC_Source_FRAME_FORMAT            = 0,
    LVPC_Source_COLOR_SPACE             = 1,
    LVPC_Source_SOURCE_RECT             = 2,
    LVPC_Source_DESTINATION_RECT        = 3,
    LVPC_Source_ALPHA_COLOR_CONSTANT    = 4,
    LVPC_Source_PROCAMP_FILTERS         = 5,
    LVPC_Source_PEDESTAL                = 6
} LVPC_Source;

/************************************************************************************************************\

Enumeration:    LVPC_Compositor

Description:    List of compositor parameters for the execute frame operation.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVPC_Compositor
{
    LVPC_Compositor_COLOR_SPACE = 0,
    LVPC_Compositor_PEDESTAL    = 1,
    LVPC_Compositor_DIVIDER     = 2,
    LVPC_Compositor_ALPHA_OUT   = 3
} LVPC_Compositor;

/************************************************************************************************************\

Enumeration:    LVPC_KeyerBlender

Description:    List of KeyerBlender parameters for the execute frame operation.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVPC_KeyerBlender
{
    LVPC_KeyerBlender_KEYER_KEY               = 0,
    LVPC_KeyerBlender_KEYER_SOURCE            = 1,
    LVPC_KeyerBlender_KEYER_RANGE             = 2,
    LVPC_KeyerBlender_BLENDER_PHI0            = 3,
    LVPC_KeyerBlender_BLENDER_PHI1            = 4,
    LVPC_KeyerBlender_BLENDER_ALPHAOUT        = 5,
    LVPC_KeyerBlender_BLENDER_ALPHACONSTANT   = 6,
    LVPC_KeyerBlender_BLENDER_ALPHALUT        = 7,
    LVPC_KeyerBlender_BLENDER_ALPHAVALUE      = 8,
    LVPC_KeyerBlender_BLENDER_FACTOR0         = 9,
    LVPC_KeyerBlender_BLENDER_FACTOR1         = 10,
    LVPC_KeyerBlender_BLENDER_ALPHA           = 11
} LVPC_KeyerBlender;

/**********************************************************************************************************\

Structure:      LVPC_SourceData

Description:    Contain per-source data for the LVPC_ExecuteFrame method

Comments:       None.

\**********************************************************************************************************/
typedef struct tagLVPC_SourceData
{
    MBOOL           bEnable;
    LBuffer_Handle  hInputSurface;
} LVPC_SourceData;

/************************************************************************************************************\

Enumeration:    LVPC_Rotation

Description:    Specifies output orientation for the execution of frame operations.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVPC_Rotation
{
    LVPC_Rotation_NONE      = 0,
    LVPC_Rotation_90        = 1,
    LVPC_Rotation_180       = 2,
    LVPC_Rotation_270       = 3,
    LVPC_Rotation_FLIP_NONE = 4,
    LVPC_Rotation_FLIP_90   = 5,
    LVPC_Rotation_FLIP_180  = 6,
    LVPC_Rotation_FLIP_270  = 7
} LVPC_Rotation;

/************************************************************************************************************\

Structure:      LVPC_TargetRotation

Description:    Specifies output orientation for execute frame operations.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPC_TargetRotation
{
    LVPC_Rotation    eRotation;
} LVPC_TargetRotation;

/************************************************************************************************************\

Structure:      LVPC_CompositorColorSpace

Description:    Specifies the compositor color space.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPC_CompositorColorSpace
{
    MBOOL           bYCbCrA;
    LVP_ColorSpace  oColorSpace;
} LVPC_CompositorColorSpace;

/************************************************************************************************************\

Structure:      LVPC_CompositorPedestal

Description:    Specifies the pedestal values to be added to the compositor output.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPC_CompositorPedestal
{
    MBOOL       bEnable;
    LVP_Color   oPedestal;
} LVPC_CompositorPedestal;

/************************************************************************************************************\

Structure:      LVPC_CompositorDivider

Description:    Specifies if the compositor output needs to be divided by the resulting alpha (Z.aO) of 
                Keyer/Blender unit 0.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPC_CompositorDivider
{
    MBOOL           bEnable;
} LVPC_CompositorDivider;

/************************************************************************************************************\

Enumeration:    LVPC_AlphaOut

Description:    Enumerated type use to specify the alpha of the compositor output surface.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVPC_AlphaOut
{
    LVPC_AlphaOut_OUT       = 0,
    LVPC_AlphaOut_CONSTANT  = 1,
    LVPC_AlphaOut_LUT       = 2
} LVPC_AlphaOut;

/************************************************************************************************************\

Structure:      LVPC_CompositorAlphaOut

Description:    Specifies the alpha of the compositor output.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPC_CompositorAlphaOut
{
    LVPC_AlphaOut    eAlphaOut;
} LVPC_CompositorAlphaOut;

/************************************************************************************************************\

Structure:      LVPC_SourceAlphaColorConstant

Description:    Specifies the alpha and color constant value for a source stream.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPC_SourceAlphaColorConstant
{
    MBOOL       bYCbCrA;
    LVP_Color   oColor;
} LVPC_SourceAlphaColorConstant;

/************************************************************************************************************\

Structure:      LVPC_SourcePedestal

Description:    Specifies the pedestal values to be subtracted from the source.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPC_SourcePedestal
{
    MBOOL           bEnable;
    LVP_Color       oPedestal;
} LVPC_SourcePedestal;

/************************************************************************************************************\

Structure:      LVPC_KeyerBlenderKeyerKey

Description:    Specifies the color key range for a Keyer/Blender unit.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPC_KeyerBlenderKeyerKey
{
    LVP_Color        oLower;
    LVP_Color        oUpper;
} LVPC_KeyerBlenderKeyerKey;

/************************************************************************************************************\

Enumeration:    LVPC_KeyerSource

Description:    Defines inputs (P or Q) to use as source for keying range comparison.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVPC_KeyerSource
{
    LVPC_KeyerSource_P      = 0,
    LVPC_KeyerSource_Q      = 1
} LVPC_KeyerSource;

/************************************************************************************************************\

Structure:      LVPC_KeyerBlenderKeyerSource

Description:    Specifies the inputs (P or Q) to use as source for keying range comparison.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPC_KeyerBlenderKeyerSource
{
    LVPC_KeyerSource eKeyerSource;
} LVPC_KeyerBlenderKeyerSource;

/************************************************************************************************************\

Enumeration:    LVPC_KeyerRange

Description:    Defines inputs (P, Q or BLENDED) that can be used when the pixel is in the range or out of the range.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVPC_KeyerRange
{
    LVPC_KeyerRange_P       = 0,
    LVPC_KeyerRange_Q       = 1,
    LVPC_KeyerRange_BLENDED = 2
} LVPC_KeyerRange;

/************************************************************************************************************\

Structure:      LVPC_KeyerBlenderKeyerRange

Description:    Specifies the action to take when the pixel is in the range or out of the range.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPC_KeyerBlenderKeyerRange
{
    LVPC_KeyerRange  eKeyerInRange;
    LVPC_KeyerRange  eKeyerOutRange;
} LVPC_KeyerBlenderKeyerRange;

/************************************************************************************************************\

Enumeration:    LVPC_BlenderPhi

Description:    Defines the inputs (0, 1, Factor or 1-Factor) that can be used to generate Phi.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVPC_BlenderPhi
{
    LVPC_BlenderPhi_0               = 0,
    LVPC_BlenderPhi_1               = 1,
    LVPC_BlenderPhi_FACTOR          = 2,
    LVPC_BlenderPhi_1_MINUS_FACTOR  = 3
} LVPC_BlenderPhi;

/************************************************************************************************************\

Structure:      LVPC_KeyerBlenderBlenderPhi0

Description:    Specifies the inputs (0, 1, Factor0 or 1- Factor0) to use to generate Phi0.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPC_KeyerBlenderBlenderPhi0
{
    LVPC_BlenderPhi  eBlenderPhi0;
} LVPC_KeyerBlenderBlenderPhi0;

/************************************************************************************************************\

Structure:      LVPC_KeyerBlenderBlenderPhi1

Description:    Specifies the inputs (0, 1, Factor1 or 1- Factor1) to use to generate Phi1.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPC_KeyerBlenderBlenderPhi1
{
    LVPC_BlenderPhi  eBlenderPhi1;
} LVPC_KeyerBlenderBlenderPhi1;

/************************************************************************************************************\

Enumeration:    LVPC_BlenderAlphaOut

Description:    Defines the inputs (Alpha value, 1, Alpha or 1- Alpha) that can be used to generate alpha out (Z.aO).

Comments:       None.

\************************************************************************************************************/
// Todo: Need to update LVideoProcessor.doc
typedef enum tagLVPC_BlenderAlphaOut
{
    LVPC_BlenderAlphaOut_ALPHAVALUE     = 0,
    LVPC_BlenderAlphaOut_1              = 1,
    LVPC_BlenderAlphaOut_ALPHA          = 2,
    LVPC_BlenderAlphaOut_1_MINUS_ALPHA  = 3
} LVPC_BlenderAlphaOut;

/************************************************************************************************************\

Structure:      LVPC_KeyerBlenderBlenderAlphaOut

Description:    Specifies the inputs (Alpha value, 1, Alpha or 1- Alpha) to use to generate alpha out (Z.aO).

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPC_KeyerBlenderBlenderAlphaOut
{
    LVPC_BlenderAlphaOut eBlenderAlphaOut;
} LVPC_KeyerBlenderBlenderAlphaOut;

/************************************************************************************************************\

Enumeration:    LVPC_BlenderAlphaConstant

Description:    Defines the inputs (Always 1, 1, P.aC, Q.aC, Always Alpha value, Alpha value, Q.aS or Z.aO + Q.aS) that can be used to 
                generate alpha constant output (Z.aC).

Comments:       None.

\************************************************************************************************************/
// Todo: Need to update LVideoProcessor.doc
typedef enum tagLVPC_BlenderAlphaConstant
{
    LVPC_BlenderAlphaConstant_ALWAYS_1                  = 0,
    LVPC_BlenderAlphaConstant_1                         = 1,
    LVPC_BlenderAlphaConstant_P_ALPHA_CST               = 2,
    LVPC_BlenderAlphaConstant_Q_ALPHA_CST               = 3,
    LVPC_BlenderAlphaConstant_ALWAYS_ALPHAVALUE         = 4,
    LVPC_BlenderAlphaConstant_ALPHAVALUE                = 5,
    LVPC_BlenderAlphaConstant_Q_ALPHA_SRC               = 6,
    LVPC_BlenderAlphaConstant_ALPHAOUT_PLUS_Q_ALPHA_SRC = 7
} LVPC_BlenderAlphaConstant;

/************************************************************************************************************\

Structure:      LVPC_KeyerBlenderBlenderAlphaConstant

Description:    Specifies the inputs (Always 1, 1, P.aC, Q.aC, Always Alpha value, Alpha value, Q.aS or Z.aO + Q.aS) to use to generate 
                alpha constant output (Z.aC). 

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPC_KeyerBlenderBlenderAlphaConstant
{
    LVPC_BlenderAlphaConstant    eBlenderAlphaConstant;
} LVPC_KeyerBlenderBlenderAlphaConstant;

/************************************************************************************************************\

Enumeration:    LVPC_BlenderAlphaLUT

Description:    Defines the inputs (Always 1, 1, P.aL or Q.aL) that can be used to generate alpha LUT output (Z.aL).

Comments:       None.

\************************************************************************************************************/
// Todo: Need to update LVideoProcessor.doc
typedef enum tagLVPC_BlenderAlphaLUT
{
    LVPC_BlenderAlphaLUT_ALWAYS_1       = 0,
    LVPC_BlenderAlphaLUT_1              = 1,
    LVPC_BlenderAlphaLUT_P_ALPHA_LUT    = 2,
    LVPC_BlenderAlphaLUT_Q_ALPHA_LUT    = 3
} LVPC_BlenderAlphaLUT;

/************************************************************************************************************\

Structure:      LVPC_KeyerBlenderBlenderAlphaLUT

Description:    Specifies the inputs (Always 1, 1, P.aL or Q.aL) to use to generate alpha LUT output (Z.aL). 

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPC_KeyerBlenderBlenderAlphaLUT
{
    LVPC_BlenderAlphaLUT eBlenderAlphaLUT;
} LVPC_KeyerBlenderBlenderAlphaLUT;

/************************************************************************************************************\

Structure:      LVPC_KeyerBlenderBlenderAlphaValue

Description:    Specifies an alpha constant value for a Keyer/Blender unit that can be selected as Z.aO and 
                can also be selected as Z.aC.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPC_KeyerBlenderBlenderAlphaValue
{
    MFLOAT32    fAlphaValue;
} LVPC_KeyerBlenderBlenderAlphaValue;

/************************************************************************************************************\

Enumeration:    LVPC_BlenderOperand0

Description:    Defines the inputs that can be used for multiplication as operand 0.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVPC_BlenderOperand0
{
    LVPC_BlenderOperand0_0              = 0,
    LVPC_BlenderOperand0_1              = 1,
    LVPC_BlenderOperand0_P_ALPHA_LUT    = 2,
    LVPC_BlenderOperand0_Q_ALPHA_LUT    = 3
} LVPC_BlenderOperand0;

/************************************************************************************************************\

Enumeration:    LVPC_BlenderOperand1

Description:    Defines the inputs that can be used for multiplication as operand 1.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVPC_BlenderOperand1
{
    LVPC_BlenderOperand1_0                                  = 0,
    LVPC_BlenderOperand1_1                                  = 1,
    LVPC_BlenderOperand1_P_ALPHA_SRC                        = 2,
    LVPC_BlenderOperand1_1_MINUS_P_ALPHA_SRC                = 3,
    LVPC_BlenderOperand1_P_ALPHA_CST_T_P_ALPHA_SRC          = 4,
    LVPC_BlenderOperand1_1_MINUS_P_ALPHA_CST_T_P_ALPHA_SRC  = 5,
    LVPC_BlenderOperand1_P_ALPHA_CST                        = 6,
    LVPC_BlenderOperand1_1_MINUS_P_ALPHA_CST                = 7
} LVPC_BlenderOperand1;

/************************************************************************************************************\

Enumeration:    LVPC_BlenderOperand2

Description:    Defines the inputs that can be used for multiplication as operand 2.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVPC_BlenderOperand2
{
    LVPC_BlenderOperand2_0                                  = 0,
    LVPC_BlenderOperand2_1                                  = 1,
    LVPC_BlenderOperand2_Q_ALPHA_SRC                        = 2,
    LVPC_BlenderOperand2_1_MINUS_Q_ALPHA_SRC                = 3,
    LVPC_BlenderOperand2_Q_ALPHA_CST_T_Q_ALPHA_SRC          = 4,
    LVPC_BlenderOperand2_1_MINUS_Q_ALPHA_CST_T_Q_ALPHA_SRC  = 5,
    LVPC_BlenderOperand2_Q_ALPHA_CST                        = 6,
    LVPC_BlenderOperand2_1_MINUS_Q_ALPHA_CST                = 7
} LVPC_BlenderOperand2;

/************************************************************************************************************\

Structure:      LVPC_KeyerBlenderBlenderFactor0

Description:    Specifies the three operands required to generate the blending factor 0.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPC_KeyerBlenderBlenderFactor0
{
    LVPC_BlenderOperand0 eOperand0;
    LVPC_BlenderOperand1 eOperand1;
    LVPC_BlenderOperand2 eOperand2;
} LVPC_KeyerBlenderBlenderFactor0;

/************************************************************************************************************\

Structure:      LVPC_KeyerBlenderBlenderFactor1

Description:    

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPC_KeyerBlenderBlenderFactor1
{
    LVPC_BlenderOperand0 eOperand0;
    LVPC_BlenderOperand1 eOperand1;
    LVPC_BlenderOperand2 eOperand2;
} LVPC_KeyerBlenderBlenderFactor1;

/************************************************************************************************************\

Structure:      LVPC_KeyerBlenderBlenderAlpha

Description:    Specifies the three operands required to generate the blending factor 1.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPC_KeyerBlenderBlenderAlpha
{
    LVPC_BlenderOperand0 eOperand0;
    LVPC_BlenderOperand1 eOperand1;
    LVPC_BlenderOperand2 eOperand2;
} LVPC_KeyerBlenderBlenderAlpha;

// --------------------------------------------- V P E   T Y P E S -------------------------------------------

/************************************************************************************************************\

Enum:           LVPE_AttributeType

Description:    Enumeration of attribute types.

Comments:       None.

\************************************************************************************************************/
typedef enum
{
    LVPE_AttributeType_CREATE  = 0            // Attributes structure for the Create method.
} LVPE_AttributeType;

/************************************************************************************************************\

Structure:      LVPE_CreateAttributes

Description:    LVPE creation attributes.

Comments:       LVPE_AttributeType_CREATE.

\************************************************************************************************************/
typedef struct tagLVPE_CreateAttributes
{
    LVPE_AttributeType  eType;                  // Shall be LVPE_AttributeType_CREATE
    MBOOL32             bLoadBalancing;         // Load balacing
    MBOOL32             bWaitforDMA;            // Wait for DMA
} LVPE_CreateAttributes;

/************************************************************************************************************\

Enumeration:    LVPE_Target

Description:    List of target parameters for the execute frame operation.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVPE_Target
{
    LVPE_Target_RECT               = 0,
    LVPE_Target_COLOR_SPACE        = 1,
    LVPE_Target_ALPHA_FILL         = 2,
    LVPE_Target_PROCAMP_FILTERS    = 3,
    LVPE_Target_CHROMA_FILTER      = 4
} LVPE_Target;

/************************************************************************************************************\

Enumeration:    LVPE_Source

Description:    List of source parameters for the execute frame operation.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVPE_Source
{
    LVPE_Source_FRAME_FORMAT              = 0,
    LVPE_Source_COLOR_SPACE               = 1,
    LVPE_Source_SOURCE_RECT               = 2,
    LVPE_Source_DESTINATION_RECT          = 3,
    LVPE_Source_OPERATION                 = 4,
    LVPE_Source_ENHANCEMENT_FILTERS       = 5,
    LVPE_Source_BAYER_FILTER              = 6,
    LVPE_Source_SCALER_FILTERS            = 7,
    LVPE_Source_MADI_FEATURES             = 8
} LVPE_Source;

/************************************************************************************************************\

Enumeration:    LVPE_CurrentFrameOrField

Description:    Specify which part of the current source buffer will be used for processing.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVPE_CurrentFrameOrField
{
    LVPE_CurrentFrameOrField_FRAME        = 0,
    LVPE_CurrentFrameOrField_TOP_FIELD    = 1,
    LVPE_CurrentFrameOrField_BOTTOM_FIELD = 2
} LVPE_CurrentFrameOrField;

/************************************************************************************************************\

Enumeration:    LVPE_OperationType

Description:    Lists the video processing operation that can be performed.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVPE_OperationType
{
    LVPE_OperationType_PROGRESSIVE  = 0,
    LVPE_OperationType_WEAVE        = 1,
    LVPE_OperationType_BOB          = 2,
    LVPE_OperationType_MADI         = 3
} LVPE_OperationType;

/************************************************************************************************************\

Structure:      LVPE_SourceOperation

Description:    Specifies the video processing operation to performs.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPE_SourceOperation
{
    LVPE_OperationType eOperationType;
} LVPE_SourceOperation;

/************************************************************************************************************\

Enumeration:    LVPE_SharpeningWidth

Description:    Specifies the sharpening width.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVPE_SharpeningWidth
{
    LVPE_SharpeningWidth_WIDE       = 3,    // 2560
    LVPE_SharpeningWidth_MID_WIDE   = 2,    // 1280
    LVPE_SharpeningWidth_MID_NARROW = 1,    // 640
    LVPE_SharpeningWidth_NARROW     = 0     // 320
} LVPE_SharpeningWidth;

/************************************************************************************************************\

Enumeration:    LVPE_SharpeningStrength

Description:    Specifies the sharpening strength of the enhancement filters module.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVPE_SharpeningStrength
{
    LVPE_SharpeningStrength_STRONG     = 3, // 1/8
    LVPE_SharpeningStrength_MID_STRONG = 2, // 1/5
    LVPE_SharpeningStrength_MID_WEAK   = 1, // 1/4
    LVPE_SharpeningStrength_WEAK       = 0  // 1/2
} LVPE_SharpeningStrength;

/************************************************************************************************************\

Structure:      LVPE_SourceEnhancementFilters

Description:    Specifies the advanced enhancement filters.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPE_SourceEnhancementFilters
{
    MBOOL                   bSharpeningEnable;      // Default: FALSE
    LVPE_SharpeningWidth    eSharpeningWidth;
    LVPE_SharpeningStrength eSharpeningStrength;
    MBOOL                   bNoiseFilterEnable;     // Default: FALSE
    MFLOAT32                fNoiseDifferenceLow;    // [0.0-1.0]
    MFLOAT32                fNoiseDifferenceHigh;   // [0.0-1.0]
    MBOOL                   bAdaptiveEnable;        // Default: FALSE
} LVPE_SourceEnhancementFilters;

/************************************************************************************************************\

Enumeration:    LVPE_BayerPattern

Description:    Specifies the Bayer pattern.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVPE_BayerPattern
{
    LVPE_BayerPattern_GRBG = 0,
    LVPE_BayerPattern_GBRG = 1,
    LVPE_BayerPattern_RGGB = 2,
    LVPE_BayerPattern_BGGR = 3
} LVPE_BayerPattern;

/************************************************************************************************************\

Enumeration:    LVPE_BayerFilter

Description:    Specifies the type of Bayer filtering.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVPE_BayerFilter
{
    LVPE_BayerFilter_BILINEAR = 0,
    LVPE_BayerFilter_AVERAGE  = 1
} LVPE_BayerFilter;

/************************************************************************************************************\

Structure:      LVPE_SourceBayerFilter

Description:    Specifies the Bayer conversion filter.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPE_SourceBayerFilter
{
    MBOOL               bEnable;
    LVPE_BayerPattern   ePattern;
    LVPE_BayerFilter    eFilter;
} LVPE_SourceBayerFilter;

/************************************************************************************************************\

Enumeration:    LVPE_ScalerFilterType

Description:    Defines the high quality scaler filters types.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVPE_ScalerFilterType
{
    LVPE_ScalerFilterType_AUTO          = 0,
    LVPE_ScalerFilterType_GAUSS         = 1,
    LVPE_ScalerFilterType_KAISER        = 2,
    LVPE_ScalerFilterType_USER_DEFINED  = 3,
    LVPE_ScalerFilterType_DISABLED      = 4,
    LVPE_ScalerFilterType_AUTO_1        = 5
} LVPE_ScalerFilterType;

/************************************************************************************************************\

Enumeration:    LVPE_ScalerFilterConstants

Description:    

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVPE_ScalerFilterConstants
{
    LVPE_ScalerFilterConstants_COEFF_ARRAY_SIZE = 102,
    LVPE_ScalerFilterConstants_SUBPIXELS = 17,
    LVPE_ScalerFilterConstants_TAPS = 6
} LVPE_ScalerFilterConstants;

/************************************************************************************************************\

Structure:      LVPE_ScalerFilter

Description:    Specifies the high quality scaler filter parameters.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPE_ScalerFilter
{
    LVPE_ScalerFilterType   eFilter;
    MFLOAT32                fParam;
    MFLOAT32                (*pafCoefficients)[LVPE_ScalerFilterConstants_SUBPIXELS] [LVPE_ScalerFilterConstants_TAPS];
} LVPE_ScalerFilter;

/************************************************************************************************************\

Enumeration:    LVPE_SourceScalerFiltersConstants

Description:    

Comments:       None.

\************************************************************************************************************/
typedef enum tagLVPE_SourceScalerFiltersConstants
{
    LVPE_SourceScalerFiltersConstants_HORIZONTAL_LUMA   = 0,
    LVPE_SourceScalerFiltersConstants_HORIZONTAL_CHROMA = 1,
    LVPE_SourceScalerFiltersConstants_VERTICAL_LUMA     = 2,
    LVPE_SourceScalerFiltersConstants_VERTICAL_CHROMA   = 3,
    LVPE_SourceScalerFiltersConstants_NUMBER            = 4
} LVPE_SourceScalerFiltersConstants;

/************************************************************************************************************\

Structure:      LVPE_SourceScalerFilters

Description:    Specifies the high quality scaler filters.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPE_SourceScalerFilters
{
    LVPE_ScalerFilter aoFilters[LVPE_SourceScalerFiltersConstants_NUMBER];
} LVPE_SourceScalerFilters;

/************************************************************************************************************\

Structure:      LVPE_SourceMadiFeatures

Description:    This is used to control the MADI features like CINETEL and AAP.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLVPE_SourceMadiFeatures
{
    MBOOL   bCINETELEnable;
    MBOOL   bAAPEnable;
} LVPE_SourceMadiFeatures;

// -----------------------------------------------------------------------------------------------------------
//                           G L O B A L   V A R I A B L E   R E F E R E N C E S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                         G L O B A L   F U N C T I O N   D E C L A R A T I O N S
// -----------------------------------------------------------------------------------------------------------

#if defined (__cplusplus)
extern "C" {
#endif

// -------------------------------------- V P C   F U N C T I O N S ------------------------------------------

/************************************************************************************************************\

Function:       LVPC_Create

Description:    Create a VPC module interface.

Parameters:     IN  hDevice         Handle to a device object.
                IN  hDeviceThread   Handle to a device thread.
                IN  poAttributes    Pointer to the attribute type member of a CreateAttributes
                                    structure describing the attributes of the server object to
                                    create. NULL indicates the use of the default attributes.
                OUT phVPC           Pointer where the handle of the VPC module object is returned.

Return Value:   LStatus_OK              Function  completed successfully.
                LStatus_FAIL            Function  failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_OUT_OF_MEMORY   Function failed due to Insufficient memory.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVPC_Create
(
    LDevice_Handle              hDevice,
    LDeviceThread_Handle        hDeviceThread,
    LVPC_AttributeType*         poAttributes,
    LVPC_Handle*                phVPC
);

/************************************************************************************************************\

Function:       LVPC_Destroy

Description:    Destroys a VPC object.

Parameters:     IN  hVPC    Handle to VPC object.

Return Value:   LStatus_OK              Function  completed successfully.
                LStatus_FAIL            Function  failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_COMMAND_PENDING Function failed due to uncompleted command.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVPC_Destroy
(
    LVPC_Handle                 hVPC
);

/************************************************************************************************************\

Function:       LVPC_GetDefaultAttributes

Description:    Gets the default values of a given type of attributes.

Parameters:     IN  hVPC                    Handle of a LVPC object.
                OUT peAttributes            Pointer to the attribute type member of a structure of attributes.
                                            On input, the type member describes the type of the structure.
                                            On output, the remaining members of the structure are filled with
                                            the default value of the attributes.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This function provides a mechanism for retrieving an attribute structure filled with default
                  values. The application is concerned only with the subset of the members it's interested in.
                - Only one type of attribute at a time can be retrieved. Only AttributeType_CREATE is
                  currently supported.

\************************************************************************************************************/
LAPI LStatus LVPC_GetDefaultAttributes
(
    LVPC_Handle            hVPC,
    LVPC_AttributeType*    peAttributes
);

/************************************************************************************************************\

Function:       LVPC_GetAttributes

Description:    Gets the current values of a given type of attributes.

Parameters:     IN  hVPC                    Handle of a LVPC object.
                OUT peAttributes            Pointer to the attribute type member of a structure of attributes.
                                            On input, the type member describes the type of the structure.
                                            On output, the remaining members of the structure are filled with
                                            the default value of the attributes.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This function provides a mechanism for retrieving an attribute structure filled with default
                  values. The application is concerned only with the subset of the members it's interested in.
                - Only one type of attribute at a time can be retrieved. Only AttributeType_CREATE is
                  currently supported.

\************************************************************************************************************/
LAPI LStatus LVPC_GetAttributes
(
    LVPC_Handle            hVPC,
    LVPC_AttributeType*    peAttributes
);

/************************************************************************************************************\

Function:       LVPC_ExecuteFrame

Description:    Performs the execution of the frame.

Parameters:     IN  hVPC            Handle to a VPC object.
                IN  hTarget         Handle to the LBuffer_Handle interface. 
                IN  uiSourceCount   Number of sources to process.
                IN  paoSource       Pointer to an array of LVPC_SourceData structures that contain information 
                                    about the sources.

Return Value:   LStatus_OK              Function  completed successfully.
                LStatus_FAIL            Function  failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_OUT_OF_MEMORY   Insufficient memory.
                LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL  Function failed. Device thread is full. 
                LStatus_UNSUPPORTED     Current function isn't supported.

Comments:       Perform the execution of the frame by processing the source(s) and the destination relative to 
                the parameters set, and writes the result target to the LBuffer_Handle set.

\************************************************************************************************************/
LAPI LStatus LVPC_ExecuteFrame
(
    LVPC_Handle                 hVPC,
    LBuffer_Handle              hTarget,
    MUINT32                     uiSourceCount,
    LVPC_SourceData*            paoSource
);

/************************************************************************************************************\

Function:       LVPC_GetSourceParam

Description:    Gets a source parameter.

Parameters:     IN  hVPC            Handle to a VPC object.
                IN  uiSourceNumber  Zero-based index of the source stream.
                IN  eParam          Enumerated type of the source parameters to Get.
                IN  uiDataSize      Size, in bytes, of the buffer pointed to by pData.
                IN  pData           Pointer to a buffer that contains the source parameters data.

Return Value:   LStatus_OK              Function  completed successfully.
                LStatus_FAIL            Function  failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.

Comments:       The compositor can work with up to four (4) sources. The sources need to be arranged in Z-order, 
                from bottom to top.

\************************************************************************************************************/
LAPI LStatus LVPC_GetSourceParam
(
    LVPC_Handle                 hVPC,
    MUINT32                     uiSourceNumber,
    LVPC_Source                 eParam,
    MUINT32                     uiDataSize,
    void*                       pData
);

/************************************************************************************************************\

Function:       LVPC_GetTargetParam

Description:    Gets a target parameter for the execution of a frame operation.

Parameters:     IN  hVPC        Handle to a VPC object.
                IN  eParam      Enumerated type of the target parameters to Get.
                IN  uiDataSize  Size, in bytes, of the buffer pointed to by pData.
                IN  pData       Pointer to a buffer that contains the target parameters data.

Return Value:   LStatus_OK              Function  completed successfully.
                LStatus_FAIL            Function  failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVPC_GetTargetParam
(
    LVPC_Handle                 hVPC,
    LVPC_Target                 eParam,
    MUINT32                     uiDataSize,
    void*                       pData
);

/************************************************************************************************************\

Function:       LVPC_GetCompositorParam

Description:    Get a compositor parameter.

Parameters:     IN  hVPC            Handle to a VPC object.
                IN  eParam          Enumerated type of the compositor parameters to Get.
                IN  uiDataSize      Size, in bytes, of the buffer pointed to by pData.
                IN  pData           Pointer to a buffer that contains the compositor parameters data. 

Return Value:   LStatus_OK              Function  completed successfully.
                LStatus_FAIL            Function  failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.

Comments:       The compositor can work with up to four (4) sources and is composed of three (3) cascaded 
                keyer/ blender units. Each unit has two inputs (P and Q) and one output (Z). Unit 2 uses the 
                bottom source as input Q, which is at index 0 in the array of LVPC_SourceData structures of 
                LVPC_ExecuteFrame structure, and the source at index 1 as input P. Output Z of unit 2 is used 
                as input Q of unit 1. Unit 1 uses the source at index 2 as input P. The output Z of unit 1 is 
                used as input Q of unit 0. Unit 0 uses the source at index 3 as input P. Output Z of unit 0 
                then goes to the post-blending processing. 

\************************************************************************************************************/
LAPI LStatus LVPC_GetCompositorParam
(
    LVPC_Handle                 hVPC,
    LVPC_Compositor             eParam,
    MUINT32                     uiDataSize,
    void*                       pData
);

/************************************************************************************************************\

Function:       LVPC_GetKeyerBlenderParam

Description:    Get a Keyer/Blender parameter.

Parameters:     IN  hVPC            Handle to a VPC object.
                IN  uiUnitNumber    Zero-based index of the Keyer/Blender unit.
                IN  eParam          Enumerated type of the Keyer/Blender parameters to Get.
                IN  uiDataSize      Size, in bytes, of the buffer pointed to by pData.
                IN  pData           Pointer to a buffer that contains the Keyer/Blender parameters data. 

Return Value:   LStatus_OK              Function  completed successfully.
                LStatus_FAIL            Function  failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVPC_GetKeyerBlenderParam
(
    LVPC_Handle                 hVPC,
    MUINT32                     uiUnitNumber,
    LVPC_KeyerBlender           eParam,
    MUINT32                     uiDataSize,
    void*                       pData
);

/************************************************************************************************************\

Function:       LVPC_GetLUT

Description:    Get the LUT for palettized surface.

Parameters:     IN  hVPC        Handle to a VPC object.
                IN  poLUT       The LUT contains the color palette entries for a source stream and the alpha 
                                channel used for the LUT alpha blending.

Return Value:   LStatus_OK              Function  completed successfully.
                LStatus_FAIL            Function  failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVPC_GetLUT
(
    LVPC_Handle                 hVPC,
    LVPC_LUT*                   poLUT
);

/************************************************************************************************************\

Function:       LVPC_SetSourceParam

Description:    Sets a source parameter.

Parameters:     IN  hVPC            Handle to a VPC object.
                IN  uiSourceNumber  Zero-based index of the source stream.
                IN  eParam          Enumerated type of the source parameters to set.
                IN  uiDataSize      Size, in bytes, of the buffer pointed to by pData.
                IN  pData           Pointer to a buffer that contains the source parameters data.

Return Value:   LStatus_OK              Function  completed successfully.
                LStatus_FAIL            Function  failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.

Comments:       The compositor can work with up to four (4) sources. The sources need to be arranged in Z-order, 
                from bottom to top.

\************************************************************************************************************/
LAPI LStatus LVPC_SetSourceParam
(
    LVPC_Handle                 hVPC,
    MUINT32                     uiSourceNumber,
    LVPC_Source                 eParam,
    MUINT32                     uiDataSize,
    void*                       pData
);

/************************************************************************************************************\

Function:       LVPC_SetTargetParam

Description:    Sets a target parameter for the execution of a frame operation.

Parameters:     IN  hVPC        Handle to a VPC object.
                IN  eParam      Enumerated type of the target parameters to set.
                IN  uiDataSize  Size, in bytes, of the buffer pointed to by pData.
                IN  pData       Pointer to a buffer that contains the target parameters data.

Return Value:   LStatus_OK              Function  completed successfully.
                LStatus_FAIL            Function  failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVPC_SetTargetParam
(
    LVPC_Handle                 hVPC,
    LVPC_Target                 eParam,
    MUINT32                     uiDataSize,
    void*                       pData
);

/************************************************************************************************************\

Function:       LVPC_SetCompositorParam

Description:    Set a compositor parameter.

Parameters:     IN  hVPC            Handle to a VPC object.
                IN  eParam          Enumerated type of the compositor parameters to set.
                IN  uiDataSize      Size, in bytes, of the buffer pointed to by pData.
                IN  pData           Pointer to a buffer that contains the compositor parameters data. 

Return Value:   LStatus_OK              Function  completed successfully.
                LStatus_FAIL            Function  failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.

Comments:       The compositor can work with up to four (4) sources and is composed of three (3) cascaded 
                keyer/ blender units. Each unit has two inputs (P and Q) and one output (Z). Unit 2 uses the 
                bottom source as input Q, which is at index 0 in the array of LVPC_SourceData structures of 
                LVPC_ExecuteFrame structure, and the source at index 1 as input P. Output Z of unit 2 is used 
                as input Q of unit 1. Unit 1 uses the source at index 2 as input P. The output Z of unit 1 is 
                used as input Q of unit 0. Unit 0 uses the source at index 3 as input P. Output Z of unit 0 
                then goes to the post-blending processing. 

\************************************************************************************************************/
LAPI LStatus LVPC_SetCompositorParam
(
    LVPC_Handle                 hVPC,
    LVPC_Compositor             eParam,
    MUINT32                     uiDataSize,
    void*                       pData
);

/************************************************************************************************************\

Function:       LVPC_SetKeyerBlenderParam

Description:    Set a Keyer/Blender parameter.

Parameters:     IN  hVPC            Handle to a VPC object.
                IN  uiUnitNumber    Zero-based index of the Keyer/Blender unit.
                IN  eParam          Enumerated type of the Keyer/Blender parameters to set.
                IN  uiDataSize      Size, in bytes, of the buffer pointed to by pData.
                IN  pData           Pointer to a buffer that contains the Keyer/Blender parameters data. 

Return Value:   LStatus_OK              Function  completed successfully.
                LStatus_FAIL            Function  failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVPC_SetKeyerBlenderParam
(
    LVPC_Handle                 hVPC,
    MUINT32                     uiUnitNumber,
    LVPC_KeyerBlender           eParam,
    MUINT32                     uiDataSize,
    void*                       pData
);

/************************************************************************************************************\

Function:       LVPC_SetLUT

Description:    Set the LUT for palettized surface.

Parameters:     IN  hVPC        Handle to a VPC object.
                IN  poLUT       The LUT contains the color palette entries for a source stream and the alpha 
                                channel used for the LUT alpha blending.

Return Value:   LStatus_OK              Function  completed successfully.
                LStatus_FAIL            Function  failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVPC_SetLUT
(
    LVPC_Handle                 hVPC,
    LVPC_LUT*                   poLUT
);

/************************************************************************************************************\

Function:       LVPC_ConvertRGB

Description:    Converts a normalized RGB linear color to a specified color space.

Parameters:     IN  hVPC                Handle to a VPC object.
                IN  poRGBColor          Pointer to the normalized RGB linear color to convert.
                IN  bYCbCrA             Specifies the color terms to use for the conversion which can be either a Y'CbCr or RGB.
                IN  poColorSpace        Pointer to the color space to use for the conversion.
                OUT poConvertedColor    Pointer to the normalized converted color.

Return Value:   LStatus_OK              Function  completed successfully.
                LStatus_FAIL            Function  failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.

Comments:       This helper function can be used to convert a normalized RGB linear color to a specified color 
                space and color term. The poConvertedColor parameter can be used to set any normalized color in 
                a specific color space used by the compositor that's defined by LVP_Color.

\************************************************************************************************************/
LAPI LStatus LVPC_ConvertRGB
(
    LVPC_Handle                 hVPC,
    LVP_Color*                  poRGBColor,
    MBOOL                       bYCbCrA,
    LVP_ColorSpace*             poColorSpace,
    LVP_Color*                  poConvertedColor
);

// -------------------------------------- V P E   F U N C T I O N S ------------------------------------------

/************************************************************************************************************\

Function:       LVPE_Create

Description:    Creates a VPE module interface.

Parameters:     IN  hDevice         Handle to a device object.
                IN  hDeviceThread   Handle to a device thread.
                IN  poAttributes    Pointer to the attribute type member of a CreateAttributes
                                    structure describing the attributes of the server object to
                                    create. NULL indicates the use of the default attributes.
                OUT phVPE           Pointer on the handle to the LVPE context.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_FAIL            Function failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_OUT_OF_MEMORY   Function failed due to Insufficient memory.

Comments:       When the handle is not used anymore, it must be destroyed using LVPE_Destroy.

\************************************************************************************************************/
LAPI LStatus LVPE_Create
(
    LDevice_Handle              hDevice,
    LDeviceThread_Handle        hDeviceThread,
    LVPE_AttributeType*         poAttributes,
    LVPE_Handle*                phVPE
);

/************************************************************************************************************\

Function:       LVPE_Destroy

Description:    Destroys a VPE object.

Parameters:     IN  hVPE            Handle to a VPE object.

Return Value:   LStatus_OK              Function  completed successfully.
                LStatus_FAIL            Function  failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_COMMAND_PENDING Function failed due to uncompleted command.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVPE_Destroy
(
    LVPE_Handle                 hVPE
);

/************************************************************************************************************\

Function:       LVPE_GetDefaultAttributes

Description:    Gets the default values of a given type of attributes.

Parameters:     IN  hVPE                    Handle of a LVPE object.
                OUT peAttributes            Pointer to the attribute type member of a structure of attributes.
                                            On input, the type member describes the type of the structure.
                                            On output, the remaining members of the structure are filled with
                                            the default value of the attributes.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This function provides a mechanism for retrieving an attribute structure filled with default
                  values. The application is concerned only with the subset of the members it's interested in.
                - Only one type of attribute at a time can be retrieved. Only AttributeType_CREATE is
                  currently supported.

\************************************************************************************************************/
LAPI LStatus LVPE_GetDefaultAttributes
(
    LVPE_Handle            hVPE,
    LVPE_AttributeType*    peAttributes
);

/************************************************************************************************************\

Function:       LVPE_GetAttributes

Description:    Gets the current values of a given type of attributes.

Parameters:     IN  hVPE                    Handle of a LVPE object.
                OUT peAttributes            Pointer to the attribute type member of a structure of attributes.
                                            On input, the type member describes the type of the structure.
                                            On output, the remaining members of the structure are filled with
                                            the default value of the attributes.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This function provides a mechanism for retrieving an attribute structure filled with default
                  values. The application is concerned only with the subset of the members it's interested in.
                - Only one type of attribute at a time can be retrieved. Only AttributeType_CREATE is
                  currently supported.

\************************************************************************************************************/
LAPI LStatus LVPE_GetAttributes
(
    LVPE_Handle            hVPE,
    LVPE_AttributeType*    peAttributes
);

/************************************************************************************************************\

Function:       LVPE_ExecuteFrame

Description:    Performs the execution of the frame.

Parameters:     IN  hVPE                    Handle to a VPE object.
                IN  hTarget                 Handle to the LBuffer_Handle interface.
                IN  eCurrentFrameOrField    Enumerated type that specifies the part of the current source buffer to process. 
                IN  hPreviousSource         Handle to the LBuffer_Handle interface that contains the previous source frame. 
                IN  hCurrentSource          Handle to the LBuffer_Handle interface that contains the current source frame.
                IN  hNextSource             Handle to the LBuffer_Handle interface that contains the next source frame.

Return Value:   LStatus_OK              Function  completed successfully.
                LStatus_FAIL            Function  failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_OUT_OF_MEMORY   Insufficient memory.
                LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL  Function failed. Device thread is full. 
                LStatus_UNSUPPORTED     Current function isn't supported.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVPE_ExecuteFrame
(
    LVPE_Handle                 hVPE,
    LBuffer_Handle              hTarget,
    LVPE_CurrentFrameOrField    eCurrentFrameOrField,
    LBuffer_Handle              hPreviousSource,
    LBuffer_Handle              hCurrentSource,
    LBuffer_Handle              hNextSource
);

/************************************************************************************************************\

Function:       LVPE_GetSourceParam

Description:    Gets a source parameter.

Parameters:     IN  hVPE        Handle to a VPE object.
                IN  eParam      Enumerated type of the source parameters to Get.
                IN  uiDataSize  Size, in bytes, of the buffer pointed to by pData.
                IN  pData       Pointer to a buffer that contains the source parameters data.

Return Value:   LStatus_OK              Function  completed successfully.
                LStatus_FAIL            Function  failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVPE_GetSourceParam
(
    LVPE_Handle                 hVPE,
    LVPE_Source                 eParam,
    MUINT32                     uiDataSize,
    void*                       pData
);

/************************************************************************************************************\

Function:       LVPE_GetTargetParam

Description:    Gets a target parameter for the execution of a frame operation.

Parameters:     IN  hVPE        Handle to a VPE object.
                IN  eParam      Enumerated type of the target parameters to Get.
                IN  uiDataSize  Size, in bytes, of the buffer pointed to by pData.
                IN  pData       Pointer to a buffer that contains the target parameters data.

Return Value:   LStatus_OK              Function  completed successfully.
                LStatus_FAIL            Function  failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVPE_GetTargetParam
(
    LVPE_Handle                 hVPE,
    LVPE_Target                 eParam,
    MUINT32                     uiDataSize,
    void*                       pData
);

/************************************************************************************************************\

Function:       LVPE_SetSourceParam

Description:    Sets a source parameter.

Parameters:     IN  hVPE        Handle to a VPE object.
                IN  eParam      Enumerated type of the source parameters to set.
                IN  uiDataSize  Size, in bytes, of the buffer pointed to by pData.
                IN  pData       Pointer to a buffer that contains the source parameters data.

Return Value:   LStatus_OK              Function  completed successfully.
                LStatus_FAIL            Function  failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVPE_SetSourceParam
(
    LVPE_Handle                 hVPE,
    LVPE_Source                 eParam,
    MUINT32                     uiDataSize,
    void*                       pData
);

/************************************************************************************************************\

Function:       LVPE_SetTargetParam

Description:    Sets a target parameter for the execution of a frame operation.

Parameters:     IN  hVPE        Handle to a VPE object.
                IN  eParam      Enumerated type of the target parameters to set.
                IN  uiDataSize  Size, in bytes, of the buffer pointed to by pData.
                IN  pData       Pointer to a buffer that contains the target parameters data.

Return Value:   LStatus_OK              Function  completed successfully.
                LStatus_FAIL            Function  failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LVPE_SetTargetParam
(
    LVPE_Handle                 hVPE,
    LVPE_Target                 eParam,
    MUINT32                     uiDataSize,
    void*                       pData
);

// -----------------------------------------------------------------------------------------------------------
//                      I N L I N E S   A N D   T E M P L A T E   D E F I N I T I O N S
// -----------------------------------------------------------------------------------------------------------

#if defined (__cplusplus)
}
#endif

#endif  // #ifndef INC_LVIDEOPROCESSOR_H
