/************************************************************************************************************\

Module Name:    LVideo.h

Description:    Liberatus video header file.

References:     LVideoOut.doc Revision 1.7, LVideoIn.doc Revision 0.19.

    Copyright (c) 2015, Matrox Graphics Inc.
    All Rights Reserved.

\************************************************************************************************************/

#ifndef INC_LVIDEO_H
#define INC_LVIDEO_H

// -----------------------------------------------------------------------------------------------------------
//                                   I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "LAPI.h"
#include "Liberatus.h"
#include "LStatus.h"

// -----------------------------------------------------------------------------------------------------------
//                                   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------  

/************************************************************************************************************\

Overview:       Field counter increment

Description:    Defining the video the field counter.

Comments:       The Subfield part is use to schedule an operation on a partial field (Picture), or wait until
                a partial fields is ready when the LVideoIn_ControlId_SUB_FIELD_MODE mode is activate or
                H264.SubPicture mode feature is enable.

                The subfield has a granularity of ( 1 / LVIDEO_FIELD_COUNTER_INCREMENT) field.

                Count = 1
                 ___________________
                |                   |
                |                   |
                |                   |
                |                   |
                |                   |
                |                   |
                |  Half field       |
                |   SubField = 8    |
                |-------------------|
                |                   |
                |                   |
                |                   |
                |  7/8 field        |
                |   Subfield = 14   |
                |-------------------|
                |                   |
                |___________________|

                Count = 2

                SubField can be use for the low latency scenario.
                Some operation can be schedule in advance on a partial captured buffer.


\************************************************************************************************************/
#define LVIDEO_FIELD_COUNTER_INCREMENT                  16
#define LVIDEO_HALF_FIELD_COUNTER_INCREMENT             (LVIDEO_FIELD_COUNTER_INCREMENT >> 1)
#define LVIDEO_QUARTER_FIELD_COUNTER_INCREMENT          (LVIDEO_FIELD_COUNTER_INCREMENT >> 2)
#define LVIDEO_EIGHTH_FIELD_COUNTER_INCREMENT           (LVIDEO_FIELD_COUNTER_INCREMENT >> 3)
#define LVIDEO_SIXTEENTH_FIELD_COUNTER_INCREMENT        (LVIDEO_FIELD_COUNTER_INCREMENT >> 4)

#define LVIDEO_SUBFIELD_COUNTER_MASK                    0x000000000000000F
#define LVIDEO_SUBFIELD_COUNTER_SHIFT                   4
#define LVIDEO_GET_FIELD_COUNTER(SUBFIELD_CNT)          ((SUBFIELD_CNT) >> LVIDEO_SUBFIELD_COUNTER_SHIFT)

/************************************************************************************************************\

Enum:           LVideo_SignalType

Description:    Enumerates the possible capture and output types of video signals.

Comments:       None.

\************************************************************************************************************/
typedef enum
{
    LVideo_SignalType_USEDEFAULT        = 0x00000000,   // Use default signal type (wild card).
    LVideo_SignalType_NONE              = 0xFFFFFFFF,   // No output to program.
    LVideo_SignalType_ANALOG            = 0x00000001,   // Analog RGB, HSync, and VSync.
    LVideo_SignalType_DVO               = 0x00000002,   // Digital video output (internal signal without
                                                        // encoder used to grab video capture).
    LVideo_SignalType_COMPOSITE         = 0x00000003,   // Analog Composite.
    LVideo_SignalType_HDMI              = 0x00000010,   // Digital 3/6 lines HDMI RGB or YUV format.
    LVideo_SignalType_DISPLAYPORT       = 0x00000020,   // DisplayPort (2 or 4 lines).
    LVideo_SignalType_DVI               = 0x00000030,   // Digital 3/6 lines DVI RGB format.
    LVideo_SignalType_SDI               = 0x00000040,   // SDI or SDI 3G.
    LVideo_SignalType_USER_DEFINE       = 0x00008000,   // 0x8000-0xFFFF Reserved for a third party to define.
    LVideo_SignalType_FORCE32           = 0x7FFFFFFF,   // Dummy value to force enumeration to use 32 bits.
} LVideo_SignalType;

/************************************************************************************************************\

Enum:           LVideo_VidParamFamily

Description:    Enumerates all possible video parameters families.

Comments:       - None.

\************************************************************************************************************/
typedef enum
{
    LVideo_VidParamFamily_NOTVALID  = 0xFFFFFFFF,   // Not valid.
    LVideo_VidParamFamily_CUSTOM    = 0x00000000,   // Not standard timing or unknown standard.
    LVideo_VidParamFamily_DMT       = 0x00000001,   // Video parameters define into VESA Computer Display 
                                                    // Monitor Timing Version 1.0, Revision 13,  except 
                                                    // reduced blanking flavors.
    LVideo_VidParamFamily_DMT_R     = 0x00000002,   // Video parameters define into VESA Computer Display  
                                                    // Monitor Timing Version 1.0, Revision 13,  Reduced 
                                                    // Blanking flavor only.
    LVideo_VidParamFamily_CVT       = 0x00000003,   // Video parameters calculate with Coordinated Video  
                                                    // Timings Standard, Version 1.1, Using full blanking 
                                                    // option.
    LVideo_VidParamFamily_CVT_R     = 0x00000004,   // Video parameters calculate with Coordinated Video 
                                                    // Timings Standard, Version 1.1, using Reduced Blanking 
                                                    // option.
    LVideo_VidParamFamily_GTF       = 0x00000005,   // Video parameters calculate with Generalized Timing 
                                                    // Formula Standard, Version: 1.1
    LVideo_VidParamFamily_CEA       = 0x00000006,   // Video parameters define into EIA/CEA-861F
    LVideo_VidParamFamily_SMPTE     = 0x00000010,   // Video parameters define into SMPTE specification.
    LVideo_VidParamFamily_FORCE32   = 0x7FFFFFFF,   // Dummy value to force enumeration to use 32 bits.
} LVideo_VidParamFamily;

/************************************************************************************************************\

Enum:           LVideo_ColorSpace

Description:    Enumerates all possible color space.

Comments:       - None.

\************************************************************************************************************/
typedef enum
{
    LVideo_ColorSpace_RGB_FULL      = 0,            // RGB full
    LVideo_ColorSpace_RGB_LIMITED   = 1,            // RGB limited
    LVideo_ColorSpace_YUV_709       = 2,            // YUV 709
    LVideo_ColorSpace_YUV_601       = 3,            // YUV 601
    LVideo_ColorSpace_YUV_601_FULL  = 4,            // Full swing for BT.601
    LVideo_ColorSpace_YUV_2020      = 5,            // YUV 2020
} LVideo_ColorSpace;

/************************************************************************************************************\

Enum:           LVideo_ChromaSubSampling

Description:    Enumerates all possible color sub sampling.

Comments:       - None.

\************************************************************************************************************/
typedef enum
{
    LVideo_ChromaSubSampling_444    = 0,            // LCSSAMPLING_444.
    LVideo_ChromaSubSampling_420    = 1,            // LCSSAMPLING_420. (YUV only).
    LVideo_ChromaSubSampling_422    = 2,            // LCSSAMPLING_422. (YUV only).
} LVideo_ChromaSubSampling;

/************************************************************************************************************\

Enum:           LVideo_TransferCharacteristics

Description:    Enumerates all possible transfer characteristics.

Comments:       - None.

\************************************************************************************************************/
typedef enum
{
    LVideo_TransferCharacteristic_SDR      = 0,        // SDR-TV in accordance with Recommendation ITU-R BT.709 as referenced by
                                                        // SMPTE ST 274 and ST 296.
    LVideo_TransferCharacteristic_HLG      = 1,        // HLG HDR-TV in accordance with Recommendation ITU-R BT.2100.
    LVideo_TransferCharacteristic_PQ       = 2,        // PQ HDR-TV in accordance with Recommendation ITU-R BT.2100.
} LVideo_TransferCharacteristic;

/************************************************************************************************************\

Enum:           LVideo_ColorDepth

Description:    Enumerates all possible color depth.

Comments:       - None.

\************************************************************************************************************/
typedef enum
{
    LVideo_ColorDepth_8BIT          = 0,            //  8-bit.
    LVideo_ColorDepth_10BIT         = 1,            // 10-bit.
    LVideo_ColorDepth_12BIT         = 2,            // 12-bit.
    LVideo_ColorDepth_16BIT         = 3,            // 16-bit.

} LVideo_ColorDepth;

/************************************************************************************************************\

Enum:           LVideo_SubFieldMode

Description:    Enumerates the possible sub field modes.

Comments:       None.

\************************************************************************************************************/
// This enum is common to both videoIn and videoOut and also used by HMC
typedef enum
{
    LVideo_SubfieldMode_1_1     = 0,    // Disable sub-field mode.
    LVideo_SubfieldMode_1_2     = 1,    // Enable half field mode.
    LVideo_SubfieldMode_1_4     = 2,    // Enable quarter field mode. 
    LVideo_SubfieldMode_1_8     = 3,    // Enable eighth field mode.
    LVideo_SubfieldMode_1_16    = 4,    // Enable sixteenth field mode.
} LVideo_SubfieldMode;

/************************************************************************************************************\

Struct:         LVideo_VidParam

Description:    Structure defining Video output Video Parameters.

Comments:       - Video timing parameters definitions: (Applies to both vertical and horizontal parameters 
                  unless specified otherwise)


                           Field Start Event (Vertical only)      Vertical Blank Event (Vertical only)
                                   |                                          | 
                                   v                                          v
                                   |<--------------------------- Line or Field --------------------------->|
                                   |<------------- Active video ------------->|<-------- Blanking -------->|

                                    __________________________________________                              __
                   Video __________|__________|XXXXXXXXXXXXXXXXXXXX|__________|____________________________|__
                         |<-Back ->|<-Border->|<-Adressable Video->|<-Border->|<-Front->|<-Sync->|<-Back ->|<-
                         |  Porch  |          |                    |          |  Porch  |        |  Porch  |
                                                                     
                                              |<-------- Blank Strat -------->|<------- Blank Time ------->|
                                              |<------------- Sync Start -------------->|<-Sync->|
                                                                                           Time   
                                                                                         ________
                   Sync (+) ____________________________________________________________|        |____________
                                                                     
                   Sync (-) ____________________________________________________________          ____________
                                                                                        |________|


                - Timings calculation formulas:
                                           
                    Total Horizontal Pixels = uiHActive + uiHFPorch + uiHSync + uiHBPorch

                    if (Progressive timing)
                    {
                        Total Vertical lines = uiVActive + uiVFPorch + uiVSync + uiVBPorch
                    }
                    else
                    {
                        Total Vertical lines = uiVActive + (uiVFPorch*2) + (uiVSync*2) + (uiVBPorch*2) + 1 
                    }

                    Total Pixels            = Total Horizontal Pixels * Total Vertical lines
                    Horizontal Rate (kHz)   = uiPixClock_kHz / Total Horizontal Pixels
                    Vertical Rate (Hz)      = (uiPixClock_kHz * 1000) / Total Pixels

                - The 'uiVRate_Hz' and 'uiHRate_kHz' fields are for information only when video parameters are
                  queried, while when set, they are ignored.  Their values are computed from other video 
                  parameters values using the timings calculation formulas.
                
                - The 'uiHActive' and 'uiVActive' parameters include left/top and right/bottom border values
                  which are always 0 unless they were modified via the LVideoOut_SetActiveRegion service.
                  
\************************************************************************************************************/
typedef struct tagLVideo_VidParam
{
    MUINT32 uiHActive;                      // Horizontal active video in pixels.
    MUINT32 uiVActive;                      // Vertical active video in lines.
    MUINT32 uiVRate_Hz;                     // Vertical Rate in Hz (Frame Rate in interlaced). 
    MUINT32 uiHRate_kHz;                    // Horizontal Rate in kHz.
    MUINT32 uiPixelClock_kHz;               // Pixel Clock in kHz.
    MUINT32 uiHFPorch;                      // Horizontal Front Porch in pixels.
    MUINT32 uiHSync;                        // Horizontal Sync in pixels.
    MUINT32 uiHBPorch;                      // Horizontal Back Porch in pixels.
    MUINT32 uiVFPorch;                      // Vertical Front Porch in lines.
    MUINT32 uiVSync;                        // Vertical Sync in lines.
    MUINT32 uiVBPorch;                      // Vertical Back Porch in lines.
    MUINT   bInterlaced             : 1;    // 
    MUINT   bBottomFieldFirst       : 1;    // 
    MUINT   bHorizontalSyncPositive : 1;    // MTRUE if horizontal sync is positive, MFALSE if it is negative.
    MUINT   bVerticalSyncPositive   : 1;    // MTRUE if vertical sync is positive, MFALSE if it is negative.
    MUINT   bIsYuv                  : 1;    // MTRUE if signal is YUV, MFALSE if is RGB.
    MUINT   fColorSpace             : 4;    // Color space :
                                            //  - When 0 : RGB full.
                                            //  - When 1 : RGB limited.
                                            //  - When 2 : YUV 709.
                                            //  - When 3 : YUV 601.
                                            //  - When 4 : YUV 601 full swing.
                                            //  - When 5 : YUV 2020.

    MUINT   fChromaSubSampling      : 2;    // Chroma Sub-Sampling :
                                            //  - When 0 : LCSSAMPLING_444.
                                            //  - When 1 : LCSSAMPLING_420 (YUV only).
                                            //  - When 2 : LCSSAMPLING_422 (YUV only).

    MUINT   fColorDepth             : 2;    // Color depth :
                                            //  - When 0 :  8-bit per channel
                                            //  - When 1 : 10-bit per channel
                                            //  - When 2 : 12-bit per channel
                                            //  - When 3 : 16-bit per channel
    MUINT   fIs3GLevelB             : 1;    // MTRUE if SDI signal is 3G Level B, MFALSE if it is negative.

    MUINT   fTransferCharacteristic : 2;    // Transfer characteristic :
                                            //  - When 0 : SDR
                                            //  - When 1 : HLG
                                            //  - When 2 : PQ

    MUINT   uiReserved1             : 16;   // reserved for future usage

    MUINT64 uiReserved2;                    // Reserved for internal use.  For video parameters 
                                            //  gotten via:
                                            //      - LVideoOut_GetOptimalVideoParameters
                                            //      - LVideoOut_EnumVideoParameters
                                            //      - LVideoOut_GetVideoParameters
                                            //  services the reserved bits must be preserved while for
                                            //  custom generated video parameters, the reserved bits 
                                            //  must be set to zero.
} LVideo_VidParam;

// -----------------------------------------------------------------------------------------------------------
//                           G L O B A L   V A R I A B L E   R E F E R E N C E S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                         G L O B A L   F U N C T I O N   D E C L A R A T I O N S
// -----------------------------------------------------------------------------------------------------------
#if defined (__cplusplus)
extern "C" {
#endif

/************************************************************************************************************\

Function:       LVideo_GetFamilyVidParam

Description:    Gets video parameters of the specified family.
                
Parameters:     IN  hDev                Device handle.
                IN  eVidParamFamily     Video parameters family.
                IN  uiHActive           Horizontal active video, in pixels, for which family video parameters 
                                        should be returned.  Can be set to 0 to enumerate all video parameters 
                                        no matter their horizontal active video.
                IN  uiVActive           Vertical active video, in lines, for which family video parameters 
                                        should be returned.  Can be set to 0 to enumerate all video parameters 
                                        no matter their vertical active video.
                IN  uiVRate_Hz          Vertical rate (Frame Rate in interlaced), in Hz, for which family 
                                        video parameters should be returned.  Can be set to 0 to enumerate all
                                        video parameters no matter their vertical rate.
                IN  uiIndex             0-based index of the video parameters to get.
                OUT poVidParam          Pointer to a LVideo_VidParam object to be filled with video parameters
                                        matching specified criteria.


Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LSTATUS_NO_MORE_DATA    No video parameters available for the specified index.

Comments:       - The video parameters can be invalid for a particular video output.
                - For GTF family, 'uiHActive', 'uiVActive' and 'uiVRate_Hz' parameters must be specified.
                - For CVT and CVT-R family, 'uiHActive' and 'uiVActive' must be specified.
                - CVT or CVT-R family will return video parameters even if 'uiHActive', 'uiVActive' and 
                  'uiVRate_Hz' values are not fully CVT compliant.

\************************************************************************************************************/
LAPI LStatus LVideo_GetFamilyVidParam(
                LDevice_Handle          hDev,
                LVideo_VidParamFamily   eVidParamFamily,
                MUINT32                 uiHActive,
                MUINT32                 uiVActive,
                MUINT32                 uiVRate_Hz,
                MUINT32                 uiIndex,
                LVideo_VidParam*        poVidParam);

/************************************************************************************************************\

Function:       LVideo_GetVidParamFamily

Description:    Gets video parameters associated family.

Parameters:     IN  hDev                Device handle.
                IN  poVidParam          Pointer to a LVideo_VidParam object containing the video parameters
                                        information for which associated family should be returned.
                OUT peVidParamFamily    Pointer to a LVideo_VidParamFamily object to be filled with video
                                        parameters family.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - LVideo_VidParamFamily_CUSTOM is returned as family if passed in video parameters do not match
                  any known family video parameters.

\************************************************************************************************************/
LAPI LStatus LVideo_GetVidParamFamily(
                LDevice_Handle          hDev,
                LVideo_VidParam*        poVidParam,
                LVideo_VidParamFamily*  peVidParamFamily);

// -----------------------------------------------------------------------------------------------------------
//                      I N L I N E S   A N D   T E M P L A T E   D E F I N I T I O N S
// -----------------------------------------------------------------------------------------------------------

#if defined (__cplusplus)
}
#endif

#endif // #ifndef INC_LVIDEO_H




