/************************************************************************************************************\

Module Name:    LPixelFormat.h

Description:    .

References:     None.

    Copyright (c) 2010, Matrox Graphics Inc.
    All Rights Reserved.

\************************************************************************************************************/

#ifndef INC_LPIXELFORMAT_H
#define INC_LPIXELFORMAT_H

// -----------------------------------------------------------------------------------------------------------
//                                   I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "LAPI.h"
#include "MtxBaseTypes.h"

// -----------------------------------------------------------------------------------------------------------
//                                   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

#define LPIXEL_PLANE_FORMAT_BITS    8
#define LPIXEL_PLANECOUNT_BITS      4
#define LPIXEL_SUBSAMPLING_BITS     4
// Planes format starts at bit position 0
// So next field is at bit position 24 (0 + 3*8)
#define LPIXEL_SUBSAMPLING_SHIFT    24
#define LPIXEL_PLANECOUNT_SHIFT     28
#define LPIXEL_PLANECOUNT_MASK      0xF0000000
#define LPIXEL_SUBSAMPLING_MASK     0x0F000000
#define LPIXEL_PLANES_MASK          0X00FFFFFF
#define LPIXEL_FIRST_PLANE_MASK     ((1 << LPIXEL_PLANE_FORMAT_BITS) - 1)


// Tool macros for the users of the pixel format:
#define Get_LPixelFormat_PlaneCount(format) \
    (((format) & LPIXEL_PLANECOUNT_MASK) >> LPIXEL_PLANECOUNT_SHIFT)
#define Get_LPixelFormat_SubSampling(format) \
    (((format) & LPIXEL_SUBSAMPLING_MASK) >> LPIXEL_SUBSAMPLING_SHIFT)
#define Set_LPixelFormat_SubSampling(subsampling) \
    (((subsampling) << LPIXEL_SUBSAMPLING_SHIFT) & LPIXEL_SUBSAMPLING_MASK)
// Notes:
// - The first mask is to ensure if we exceeds the last plane idx we will get the invalid format
// - The second mask is to mask out the other planes format.
#define Get_LPixelFormat_PlaneFormat(format, plane) \
    ((Get_LPixelFormat_PlaneCount(format) == LFPlaneCount_1) ? (format) : \
    (((format & LPIXEL_PLANES_MASK) >> (LPIXEL_PLANE_FORMAT_BITS * plane)) & \
     ((1 << LPIXEL_PLANE_FORMAT_BITS) - 1)))
#define Set_LPixelFormat_PlaneFormat(value, plane) \
    (((value) & ((1 << LPIXEL_PLANE_FORMAT_BITS) - 1))  << (LPIXEL_PLANE_FORMAT_BITS * plane))

// Macros to simplify multiplane format constant computations
#define Make_LPixelFormat_2planes(chroma_subsampling, plane0, plane1) (\
    (LFPlaneCount_2 << LPIXEL_PLANECOUNT_SHIFT) |\
    (chroma_subsampling << LPIXEL_SUBSAMPLING_SHIFT) |\
    Set_LPixelFormat_PlaneFormat(plane0, 0) |\
    Set_LPixelFormat_PlaneFormat(plane1, 1))

#define Make_LPixelFormat_3planes(chroma_subsampling, plane0, plane1, plane2) (\
    (LFPlaneCount_3 << LPIXEL_PLANECOUNT_SHIFT) |\
    (chroma_subsampling << LPIXEL_SUBSAMPLING_SHIFT) |\
    Set_LPixelFormat_PlaneFormat(plane0, 0) |\
    Set_LPixelFormat_PlaneFormat(plane1, 1) |\
    Set_LPixelFormat_PlaneFormat(plane2, 2))

// Tools constants for format declarations

// These are already shifted sub sampling constant, they can be use along with
// the sub sampling mask to test the sampling of a pixel format
#define LPIXEL_SUBSAMPLING_422      Set_LPixelFormat_SubSampling(LCSSAMPLING_422)
#define LPIXEL_SUBSAMPLING_444      Set_LPixelFormat_SubSampling(LCSSAMPLING_444)

/************************************************************************************************************\
 
Enumeration:    LFormatPlaneCount

Description:    
 
Comments:       To save bits we use a -1 bias so a plane count of 1 is encoded as 0, 2 is encoded as 1, etc.
 
\************************************************************************************************************/
typedef enum tagLFormatPlaneCount
{
    LFPlaneCount_1 = 0,
    LFPlaneCount_2 = 1,
    LFPlaneCount_3 = 2,
} LFormatPlaneCount;
     
/************************************************************************************************************\

  Enum:         Chroma sub sampling

  Description:  Chroma subsampling for multiplane

  Comments:

\************************************************************************************************************/
typedef enum tagLChromaSubSampling
{
    // No sampling, or undefined subsampling (by default we use a 1:1 sampling)
    LCSSAMPLING_444 = 0,
    LCSSAMPLING_420,
    LCSSAMPLING_422,
    LCSSAMPLING_411,
} LChromaSubSampling;     

/************************************************************************************************************\
 
Enum:         LPixelFormat

Description:  Supported format for pixel.

Comments:       
 
\************************************************************************************************************/
typedef enum tagLPixelFormat
{

    //////////////////////////
    // 1 plane formats
    LPixelFormat_INVALID                  = 0,

    // BPP1
    LPixelFormat_MONO                     = 2,
    LPixelFormat_MONOWIN                  = 3,

    // BPP8
    LPixelFormat_L8                       = 4,
    LPixelFormat_TYPELESS8                = 5,
    LPixelFormat_P8                       = 6,
    LPixelFormat_P4A4                     = 7,
    LPixelFormat_A8                       = 8,
    LPixelFormat_Y8                       = 9,
    LPixelFormat_U8                       = 10,
    LPixelFormat_V8                       = 11,
    LPixelFormat_G8                       = 12,
    LPixelFormat_R8                       = 13,
    LPixelFormat_B8                       = 14,

    // BPP10
    LPixelFormat_L10L10L10X2              = 15,
    LPixelFormat_A10A10A10X2              = 16,
    LPixelFormat_L82L82L82X2              = 17,
    LPixelFormat_Y82Y82Y82X2              = 18,
    LPixelFormat_U82U82U82X2              = 19,
    LPixelFormat_V82V82V82X2              = 20,
    LPixelFormat_Y10Y10Y10X2              = 21,
    LPixelFormat_U10U10U10X2              = 22,
    LPixelFormat_V10V10V10X2              = 23,
    LPixelFormat_R10R10R10X2              = 24,
    LPixelFormat_G10G10G10X2              = 25,
    LPixelFormat_B10B10B10X2              = 26,

    // BPP16
    LPixelFormat_P8A8                     = 27,
    LPixelFormat_L8A8                     = 28,
    LPixelFormat_B5G6R5                   = 29,
    LPixelFormat_B5G5R5A1                 = 30,
    LPixelFormat_B4G4R4A4                 = 31,
    LPixelFormat_A16                      = 32,
    LPixelFormat_L16                      = 33,
    LPixelFormat_TYPELESS16               = 34,
    LPixelFormat_Z16                      = 35,
    LPixelFormat_U8Y8V8Y8                 = 36 | LPIXEL_SUBSAMPLING_422,
    LPixelFormat_Y8U8Y8V8                 = 37 | LPIXEL_SUBSAMPLING_422,
    LPixelFormat_L88                      = 38,
    LPixelFormat_Y88                      = 39,
    LPixelFormat_U88                      = 40,
    LPixelFormat_V88                      = 41,
    LPixelFormat_G16                      = 42,
    LPixelFormat_R16                      = 43,
    LPixelFormat_B16                      = 44,
    LPixelFormat_U8V8                     = 45,

    // BPP20
    LPixelFormat_U10V10                   = 46,
    LPixelFormat_U82V82                   = 47,

    // BPP32
    LPixelFormat_TYPELESS32               = 48,
    LPixelFormat_L32                      = 49,
    LPixelFormat_Z32                      = 50,
    LPixelFormat_Z24S8                    = 51,
    LPixelFormat_R8G8B8A8                 = 52,
    LPixelFormat_U8Y8V8A8                 = 53 | LPIXEL_SUBSAMPLING_444,
    LPixelFormat_B8G8R8A8                 = 54,
    LPixelFormat_V8Y8U8A8                 = 55 | LPIXEL_SUBSAMPLING_444,
    LPixelFormat_V8U8Y8A8                 = 56 | LPIXEL_SUBSAMPLING_444,
    LPixelFormat_R10G10B10A2              = 57,
    LPixelFormat_U10Y10V10A2              = 58 | LPIXEL_SUBSAMPLING_444,
    LPixelFormat_R82G82B82A2              = 59,
    LPixelFormat_U82Y82V82A2              = 60 | LPIXEL_SUBSAMPLING_444,
    LPixelFormat_B10G10R10A2              = 61,
    LPixelFormat_V10Y10U10A2              = 62 | LPIXEL_SUBSAMPLING_444,
    LPixelFormat_B82G82R82A2              = 63,
    LPixelFormat_V82Y82U82A2              = 64 | LPIXEL_SUBSAMPLING_444,
    LPixelFormat_R11G11B10_FLOAT          = 65,
    LPixelFormat_R9G9B9E5_SHAREDEXP_FLOAT = 66,
    LPixelFormat_Y88U88Y88V88             = 67 | LPIXEL_SUBSAMPLING_422,
    LPixelFormat_CURSOR_ALPHA             = 68,
    LPixelFormat_CURSOR_MONO              = 69,
    LPixelFormat_U88V88                   = 70,

    // BPP64
    LPixelFormat_R16G16B16A16_FLOAT       = 71,
    LPixelFormat_R16G16B16A16             = 72,
    LPixelFormat_U88Y88V88A16             = 73 | LPIXEL_SUBSAMPLING_444,

    // EXCH Host format
    LPixelFormat_EXCH_8_YUYV_422          = 74 | LPIXEL_SUBSAMPLING_422,
    LPixelFormat_EXCH_10_YUYV_422         = 75 | LPIXEL_SUBSAMPLING_422,
    LPixelFormat_EXCH_V210_422            = 76 | LPIXEL_SUBSAMPLING_422,
    LPixelFormat_EXCH_8_YUAYVA_4224       = 77 | LPIXEL_SUBSAMPLING_422,
    LPixelFormat_EXCH_10_YUAYVA_4224      = 78 | LPIXEL_SUBSAMPLING_422,

    // Any new exposed SINGLE PLANE format must be added here to guarantee enum value coherency.

    //////////////////////////
    // 2 planes formats
    LPixelFormat_MP_Y8_U8V8_444       = Make_LPixelFormat_2planes(LCSSAMPLING_444, LPixelFormat_Y8, LPixelFormat_U8V8),
    LPixelFormat_MP_Y8_U8V8_422       = Make_LPixelFormat_2planes(LCSSAMPLING_422, LPixelFormat_Y8, LPixelFormat_U8V8),
    LPixelFormat_MP_Y8_U8V8_420       = Make_LPixelFormat_2planes(LCSSAMPLING_420, LPixelFormat_Y8, LPixelFormat_U8V8),
    LPixelFormat_MP_Y8_U8V8_411       = Make_LPixelFormat_2planes(LCSSAMPLING_411, LPixelFormat_Y8, LPixelFormat_U8V8),
    LPixelFormat_MP_Y10_U10V10_X2_444 = Make_LPixelFormat_2planes(LCSSAMPLING_444, LPixelFormat_Y10Y10Y10X2, LPixelFormat_U10V10),
    LPixelFormat_MP_Y10_U10V10_X2_422 = Make_LPixelFormat_2planes(LCSSAMPLING_422, LPixelFormat_Y10Y10Y10X2, LPixelFormat_U10V10),
    LPixelFormat_MP_Y10_U10V10_X2_420 = Make_LPixelFormat_2planes(LCSSAMPLING_420, LPixelFormat_Y10Y10Y10X2, LPixelFormat_U10V10),
    LPixelFormat_MP_Y10_U10V10_X2_411 = Make_LPixelFormat_2planes(LCSSAMPLING_411, LPixelFormat_Y10Y10Y10X2, LPixelFormat_U10V10),
    LPixelFormat_MP_Y82_U82V82_X2_444 = Make_LPixelFormat_2planes(LCSSAMPLING_444, LPixelFormat_Y82Y82Y82X2, LPixelFormat_U82V82),
    LPixelFormat_MP_Y82_U82V82_X2_422 = Make_LPixelFormat_2planes(LCSSAMPLING_422, LPixelFormat_Y82Y82Y82X2, LPixelFormat_U82V82),
    LPixelFormat_MP_Y82_U82V82_X2_420 = Make_LPixelFormat_2planes(LCSSAMPLING_420, LPixelFormat_Y82Y82Y82X2, LPixelFormat_U82V82),
    LPixelFormat_MP_Y82_U82V82_X2_411 = Make_LPixelFormat_2planes(LCSSAMPLING_411, LPixelFormat_Y82Y82Y82X2, LPixelFormat_U82V82),
    LPixelFormat_MP_Y88_U88V88_444    = Make_LPixelFormat_2planes(LCSSAMPLING_444, LPixelFormat_Y88, LPixelFormat_U88V88),
    LPixelFormat_MP_Y88_U88V88_422    = Make_LPixelFormat_2planes(LCSSAMPLING_422, LPixelFormat_Y88, LPixelFormat_U88V88),
    LPixelFormat_MP_Y88_U88V88_420    = Make_LPixelFormat_2planes(LCSSAMPLING_420, LPixelFormat_Y88, LPixelFormat_U88V88),
    LPixelFormat_MP_Y88_U88V88_411    = Make_LPixelFormat_2planes(LCSSAMPLING_411, LPixelFormat_Y88, LPixelFormat_U88V88),

    //////////////////////////
    // 3 planes formats
    LPixelFormat_MP_Y8_U8_V8_444          = Make_LPixelFormat_3planes(LCSSAMPLING_444, LPixelFormat_Y8, LPixelFormat_U8, LPixelFormat_V8),
    LPixelFormat_MP_Y8_U8_V8_422          = Make_LPixelFormat_3planes(LCSSAMPLING_422, LPixelFormat_Y8, LPixelFormat_U8, LPixelFormat_V8),
    LPixelFormat_MP_Y8_U8_V8_420          = Make_LPixelFormat_3planes(LCSSAMPLING_420, LPixelFormat_Y8, LPixelFormat_U8, LPixelFormat_V8),
    LPixelFormat_MP_Y8_U8_V8_411          = Make_LPixelFormat_3planes(LCSSAMPLING_411, LPixelFormat_Y8, LPixelFormat_U8, LPixelFormat_V8),
    LPixelFormat_MP_Y8_V8_U8_444          = Make_LPixelFormat_3planes(LCSSAMPLING_444, LPixelFormat_Y8, LPixelFormat_V8, LPixelFormat_U8),
    LPixelFormat_MP_Y8_V8_U8_422          = Make_LPixelFormat_3planes(LCSSAMPLING_422, LPixelFormat_Y8, LPixelFormat_V8, LPixelFormat_U8),
    LPixelFormat_MP_Y8_V8_U8_420          = Make_LPixelFormat_3planes(LCSSAMPLING_420, LPixelFormat_Y8, LPixelFormat_V8, LPixelFormat_U8),
    LPixelFormat_MP_Y8_V8_U8_411          = Make_LPixelFormat_3planes(LCSSAMPLING_411, LPixelFormat_Y8, LPixelFormat_V8, LPixelFormat_U8),
    LPixelFormat_MP_Y8_U8V8_A8_444        = Make_LPixelFormat_3planes(LCSSAMPLING_444, LPixelFormat_Y8, LPixelFormat_U8V8, LPixelFormat_A8),
    LPixelFormat_MP_Y8_U8V8_A8_422        = Make_LPixelFormat_3planes(LCSSAMPLING_422, LPixelFormat_Y8, LPixelFormat_U8V8, LPixelFormat_A8),
    LPixelFormat_MP_Y8_U8V8_A8_420        = Make_LPixelFormat_3planes(LCSSAMPLING_420, LPixelFormat_Y8, LPixelFormat_U8V8, LPixelFormat_A8),
    LPixelFormat_MP_Y8_U8V8_A8_411        = Make_LPixelFormat_3planes(LCSSAMPLING_411, LPixelFormat_Y8, LPixelFormat_U8V8, LPixelFormat_A8),
    LPixelFormat_MP_G8_R8_B8              = Make_LPixelFormat_3planes(LCSSAMPLING_444, LPixelFormat_G8, LPixelFormat_R8, LPixelFormat_B8),
    LPixelFormat_MP_Y10_U10_V10_X2_444    = Make_LPixelFormat_3planes(LCSSAMPLING_444, LPixelFormat_Y10Y10Y10X2, LPixelFormat_U10U10U10X2, LPixelFormat_V10V10V10X2),
    LPixelFormat_MP_Y10_U10_V10_X2_422    = Make_LPixelFormat_3planes(LCSSAMPLING_422, LPixelFormat_Y10Y10Y10X2, LPixelFormat_U10U10U10X2, LPixelFormat_V10V10V10X2),
    LPixelFormat_MP_Y10_U10_V10_X2_420    = Make_LPixelFormat_3planes(LCSSAMPLING_420, LPixelFormat_Y10Y10Y10X2, LPixelFormat_U10U10U10X2, LPixelFormat_V10V10V10X2),
    LPixelFormat_MP_Y10_U10_V10_X2_411    = Make_LPixelFormat_3planes(LCSSAMPLING_411, LPixelFormat_Y10Y10Y10X2, LPixelFormat_U10U10U10X2, LPixelFormat_V10V10V10X2),
    LPixelFormat_MP_Y10_V10_U10_X2_444    = Make_LPixelFormat_3planes(LCSSAMPLING_444, LPixelFormat_Y10Y10Y10X2, LPixelFormat_V10V10V10X2, LPixelFormat_U10U10U10X2),
    LPixelFormat_MP_Y10_V10_U10_X2_422    = Make_LPixelFormat_3planes(LCSSAMPLING_422, LPixelFormat_Y10Y10Y10X2, LPixelFormat_V10V10V10X2, LPixelFormat_U10U10U10X2),
    LPixelFormat_MP_Y10_V10_U10_X2_420    = Make_LPixelFormat_3planes(LCSSAMPLING_420, LPixelFormat_Y10Y10Y10X2, LPixelFormat_V10V10V10X2, LPixelFormat_U10U10U10X2),
    LPixelFormat_MP_Y10_V10_U10_X2_411    = Make_LPixelFormat_3planes(LCSSAMPLING_411, LPixelFormat_Y10Y10Y10X2, LPixelFormat_V10V10V10X2, LPixelFormat_U10U10U10X2),
    LPixelFormat_MP_Y82_U82_V82_X2_444    = Make_LPixelFormat_3planes(LCSSAMPLING_444, LPixelFormat_Y82Y82Y82X2, LPixelFormat_U82U82U82X2, LPixelFormat_V82V82V82X2),
    LPixelFormat_MP_Y82_U82_V82_X2_422    = Make_LPixelFormat_3planes(LCSSAMPLING_422, LPixelFormat_Y82Y82Y82X2, LPixelFormat_U82U82U82X2, LPixelFormat_V82V82V82X2),
    LPixelFormat_MP_Y82_U82_V82_X2_420    = Make_LPixelFormat_3planes(LCSSAMPLING_420, LPixelFormat_Y82Y82Y82X2, LPixelFormat_U82U82U82X2, LPixelFormat_V82V82V82X2),
    LPixelFormat_MP_Y82_U82_V82_X2_411    = Make_LPixelFormat_3planes(LCSSAMPLING_411, LPixelFormat_Y82Y82Y82X2, LPixelFormat_U82U82U82X2, LPixelFormat_V82V82V82X2),
    LPixelFormat_MP_Y82_V82_U82_X2_444    = Make_LPixelFormat_3planes(LCSSAMPLING_444, LPixelFormat_Y82Y82Y82X2, LPixelFormat_V82V82V82X2, LPixelFormat_U82U82U82X2),
    LPixelFormat_MP_Y82_V82_U82_X2_422    = Make_LPixelFormat_3planes(LCSSAMPLING_422, LPixelFormat_Y82Y82Y82X2, LPixelFormat_V82V82V82X2, LPixelFormat_U82U82U82X2),
    LPixelFormat_MP_Y82_V82_U82_X2_420    = Make_LPixelFormat_3planes(LCSSAMPLING_420, LPixelFormat_Y82Y82Y82X2, LPixelFormat_V82V82V82X2, LPixelFormat_U82U82U82X2),
    LPixelFormat_MP_Y82_V82_U82_X2_411    = Make_LPixelFormat_3planes(LCSSAMPLING_411, LPixelFormat_Y82Y82Y82X2, LPixelFormat_V82V82V82X2, LPixelFormat_U82U82U82X2),
    LPixelFormat_MP_Y10_U10V10_A10_X2_444 = Make_LPixelFormat_3planes(LCSSAMPLING_444, LPixelFormat_Y10Y10Y10X2, LPixelFormat_U10V10, LPixelFormat_A10A10A10X2),
    LPixelFormat_MP_Y10_U10V10_A10_X2_422 = Make_LPixelFormat_3planes(LCSSAMPLING_422, LPixelFormat_Y10Y10Y10X2, LPixelFormat_U10V10, LPixelFormat_A10A10A10X2),
    LPixelFormat_MP_Y10_U10V10_A10_X2_420 = Make_LPixelFormat_3planes(LCSSAMPLING_420, LPixelFormat_Y10Y10Y10X2, LPixelFormat_U10V10, LPixelFormat_A10A10A10X2),
    LPixelFormat_MP_Y10_U10V10_A10_X2_411 = Make_LPixelFormat_3planes(LCSSAMPLING_411, LPixelFormat_Y10Y10Y10X2, LPixelFormat_U10V10, LPixelFormat_A10A10A10X2),
    LPixelFormat_MP_Y82_U82V82_A10_X2_444 = Make_LPixelFormat_3planes(LCSSAMPLING_444, LPixelFormat_Y82Y82Y82X2, LPixelFormat_U82V82, LPixelFormat_A10A10A10X2),
    LPixelFormat_MP_Y82_U82V82_A10_X2_422 = Make_LPixelFormat_3planes(LCSSAMPLING_422, LPixelFormat_Y82Y82Y82X2, LPixelFormat_U82V82, LPixelFormat_A10A10A10X2),
    LPixelFormat_MP_Y82_U82V82_A10_X2_420 = Make_LPixelFormat_3planes(LCSSAMPLING_420, LPixelFormat_Y82Y82Y82X2, LPixelFormat_U82V82, LPixelFormat_A10A10A10X2),
    LPixelFormat_MP_Y82_U82V82_A10_X2_411 = Make_LPixelFormat_3planes(LCSSAMPLING_411, LPixelFormat_Y82Y82Y82X2, LPixelFormat_U82V82, LPixelFormat_A10A10A10X2),
    LPixelFormat_MP_G10_R10_B10           = Make_LPixelFormat_3planes(LCSSAMPLING_444, LPixelFormat_G10G10G10X2, LPixelFormat_R10R10R10X2, LPixelFormat_B10B10B10X2),
    LPixelFormat_MP_Y88_U88_V88_444       = Make_LPixelFormat_3planes(LCSSAMPLING_444, LPixelFormat_Y88, LPixelFormat_U88, LPixelFormat_V88),
    LPixelFormat_MP_Y88_U88_V88_422       = Make_LPixelFormat_3planes(LCSSAMPLING_422, LPixelFormat_Y88, LPixelFormat_U88, LPixelFormat_V88),
    LPixelFormat_MP_Y88_U88_V88_420       = Make_LPixelFormat_3planes(LCSSAMPLING_420, LPixelFormat_Y88, LPixelFormat_U88, LPixelFormat_V88),
    LPixelFormat_MP_Y88_U88_V88_411       = Make_LPixelFormat_3planes(LCSSAMPLING_411, LPixelFormat_Y88, LPixelFormat_U88, LPixelFormat_V88),
    LPixelFormat_MP_Y88_V88_U88_444       = Make_LPixelFormat_3planes(LCSSAMPLING_444, LPixelFormat_Y88, LPixelFormat_V88, LPixelFormat_U88),
    LPixelFormat_MP_Y88_V88_U88_422       = Make_LPixelFormat_3planes(LCSSAMPLING_422, LPixelFormat_Y88, LPixelFormat_V88, LPixelFormat_U88),
    LPixelFormat_MP_Y88_V88_U88_420       = Make_LPixelFormat_3planes(LCSSAMPLING_420, LPixelFormat_Y88, LPixelFormat_V88, LPixelFormat_U88),
    LPixelFormat_MP_Y88_V88_U88_411       = Make_LPixelFormat_3planes(LCSSAMPLING_411, LPixelFormat_Y88, LPixelFormat_V88, LPixelFormat_U88),
    LPixelFormat_MP_Y88_U88V88_A16_444    = Make_LPixelFormat_3planes(LCSSAMPLING_444, LPixelFormat_Y88, LPixelFormat_U88V88, LPixelFormat_A16),
    LPixelFormat_MP_Y88_U88V88_A16_422    = Make_LPixelFormat_3planes(LCSSAMPLING_422, LPixelFormat_Y88, LPixelFormat_U88V88, LPixelFormat_A16),
    LPixelFormat_MP_Y88_U88V88_A16_420    = Make_LPixelFormat_3planes(LCSSAMPLING_420, LPixelFormat_Y88, LPixelFormat_U88V88, LPixelFormat_A16),
    LPixelFormat_MP_Y88_U88V88_A16_411    = Make_LPixelFormat_3planes(LCSSAMPLING_411, LPixelFormat_Y88, LPixelFormat_U88V88, LPixelFormat_A16),
    LPixelFormat_MP_G16_R16_B16           = Make_LPixelFormat_3planes(LCSSAMPLING_444, LPixelFormat_G16, LPixelFormat_R16, LPixelFormat_B16), 

    // Alias
    LPixelFormat_ALIAS_RGB565  = LPixelFormat_B5G6R5,
    LPixelFormat_ALIAS_RGB555  = LPixelFormat_B5G5R5A1,
    LPixelFormat_ALIAS_ABGR32  = LPixelFormat_R8G8B8A8,  
    LPixelFormat_ALIAS_ARGB32  = LPixelFormat_B8G8R8A8,  
    LPixelFormat_ALIAS_AYUV    = LPixelFormat_V8U8Y8A8,  
    LPixelFormat_ALIAS_UYVY    = LPixelFormat_U8Y8V8Y8,
    LPixelFormat_ALIAS_YUY2    = LPixelFormat_Y8U8Y8V8,
    LPixelFormat_ALIAS_NV12    = LPixelFormat_MP_Y8_U8V8_420,
    LPixelFormat_ALIAS_I420    = LPixelFormat_MP_Y8_U8_V8_420,
    LPixelFormat_ALIAS_YV12    = LPixelFormat_MP_Y8_V8_U8_420,

    LPixelFormat_FORCE32        = 0xFFFFFFFF,
} LPixelFormat;

/************************************************************************************************************\

Structure:      LColor

Description:    Generic color structure.

Comments:       It can define integer colors of type ARGB or AYUV with a size up to 32 bits per component.
                The lower bits are used when a component is smaller than 32 bits.

\************************************************************************************************************/
typedef struct tagLColor
{
    LPixelFormat ePixelFormat;
    union
    {
        MUINT32 uiRed;
        MUINT32 uiY;
        MUINT32 uiLuminance;    // Used with pixel formats with luminance component (ex: LPixelFormat_L8).
        MUINT32 uiPaletteIndex; // Used with pixel formats with palette index component (ex: LPixelFormat_P8).
        MUINT32 uiZ;            // Used with pixel formats with 'Z' component (ex: LPixelFormat_Z32).
        MUINT32 uiTypeless;     // Used with typeless pixel formats (ex: LPixelFormar_TYPELESS8).
        
        // Use this structure with packed pixel formats (ex: LPixelFormat_L10L10L10X2 or LPixelFormat_Y8U8Y8V8).
        struct
        {
            MUINT32 uiC0 : 10;
            MUINT32 uiC1 : 10;
            MUINT32 uiC2 : 12;
        } PackedRed, PackedY, PackedLuminance;
    };
    union
    {
        MUINT32 uiGreen;
        MUINT32 uiU;        // U = Cb
        MUINT32 uiS;        // Used with pixel formats with 'S' component (ex: LPixelFormat_Z24S8).
        
        // Use this structure with packed pixel formats (ex: LPixelFormat_G10G10G10X2).
        struct
        {
            MUINT32 uiC0 : 10;
            MUINT32 uiC1 : 10;
            MUINT32 uiC2 : 12;
        } PackedGreen, PackedU;
    };
    union
    {
        MUINT32 uiBlue;
        MUINT32 uiV;        // V = Cr
        
        // Use this structure with packed pixel formats (ex: LPixelFormat_B10B10B10X2).
        struct
        {
            MUINT32 uiC0 : 10;
            MUINT32 uiC1 : 10;
            MUINT32 uiC2 : 12;
        } PackedBlue, PackedV;
    };
    union
    {
        MUINT32 uiAlpha;
        MUINT32 uiExp;      // Used with pixel formats with an exponent (E) component
                            // (ex: LPixelFormat_R9G9B9E5_SHAREDEXP_FLOAT).
        
        // Use this structure with packed pixel formats (ex: LPixelFormat_A10A10A10X2).
        struct
        {
            MUINT32 uiC0 : 10;
            MUINT32 uiC1 : 10;
            MUINT32 uiC2 : 12;
        } PackedAlpha;
    };
} LColor;

// -----------------------------------------------------------------------------------------------------------
//                      I N L I N E S   A N D   T E M P L A T E   D E F I N I T I O N S
// -----------------------------------------------------------------------------------------------------------

#if defined (__cplusplus)
extern "C" {
#endif

/************************************************************************************************************\

Function:       LPixelFormat_GetBpp

Description:    Get number of bits per pixel.

Parameters:     ePixelFormat    Pixel format.
                uiPlaneIdx      Plane index.

Return Value:   Number of bits per pixel or Zero (0) if the pixel format is invalid.

Comments:       None.

\************************************************************************************************************/
LAPI MUINT32 LPixelFormat_GetBpp(
    LPixelFormat ePixelFormat,
    MUINT32      uiPlaneIdx);

/************************************************************************************************************\

Function:       LPixelFormat_GetNbPixelsPerColorUnit

Description:    Get the number of pixel per color unit.

Parameters:     ePixelFormat    Pixel format.
                uiPlaneIdx      Plane index.

Return Value:   Number of pixels per color unit.

Comments:       None.

\************************************************************************************************************/
LAPI MUINT32 LPixelFormat_GetNbPixelsPerColorUnit(
    LPixelFormat ePixelFormat,
    MUINT32      uiPlaneIdx);

/************************************************************************************************************\

Function:       LPixelFormat_ConvertColor

Description:    Convert a logical color from a pixel format to another pixel format.

Parameters:     IN     poSrcColor   Color to convert in a given pixel format.
                IN OUT poDstColor   Color in the desired pixel format. The field 'ePixelFormat' must be set
                                    to the desired pixel format. The other fields will be set by this
                                    function.

Return Value:   MTRUE  on success.
                MFALSE on failure (the convertion is not supported).

Comments:       The following pixel format families are not supported:
                
                    - Palettized (ex: LPixelFormat_P8)
                    - Z (ex: LPixelFormat_Z16)
                    - Cursor (ex: LPixelFormat_CURSOR_ALPHA)
                    - Typeless (ex: LPixelFormat_TYPELESS32)
                    - Packed (ex: LPixelFormat_U8Y8V8Y8 or LPixelFormat_A10A10A10X2)

\************************************************************************************************************/
LAPI MBOOL LPixelFormat_ConvertColor(
    const LColor* poSrcColor,
          LColor* poDstColor);

/************************************************************************************************************\

Function:       LPixelFormat_LogicalToHardware

Description:    Translate a logical color (LColor structure) into a hardware color.

Parameters:     IN  poSrcColor   Logical color to convert.
                IN  uiPlaneIdx   Plane index that is used when the pixel format is a multi-planar. Set it to
                                 zero (0) with single-planar format.
                OUT puiDstColor  Resulting hardware color. The lower bits are used when a color bpp is smaller 
                                 than 64 bits.

Return Value:   MTRUE  on success.
                MFALSE on failure.

Comments:       None.

\************************************************************************************************************/
LAPI MBOOL LPixelFormat_LogicalToHardware(
    const LColor*  poSrcColor,
    const MUINT32  uiPlaneIdx,
          MUINT64* puiDstColor);

/************************************************************************************************************\

Function:       LPixelFormat_HardwareToLogical

Description:    Translate a hardware color into a logical color (LColor structure).

Parameters:     IN     uiSrcColor   Hardware color. The lower bits are used when a color bpp is smaller 
                                    than 64 bits.
                IN     uiPlaneIdx   Plane index that is used when the pixel format is a multi-planar. Set it 
                                    to zero (0) with single-planar format.
                IN OUT poDstColor   Logical color. The field 'ePixelFormat' must be set to the pixel format. 
                                    The other fields will be set by this function.

Return Value:   MTRUE  on success.
                MFALSE on failure.

Comments:       None.

\************************************************************************************************************/
LAPI MBOOL LPixelFormat_HardwareToLogical(
    const MUINT64 uiSrcColor,
    const MUINT32 uiPlaneIdx,
          LColor* poDstColor);

#if defined (__cplusplus)
}
#endif

#endif  // INC_LPIXELLPixelFormat_Type_H
