/************************************************************************************************************\

Module Name:    LMexa.h

Description:    Liberatus for Matrox EXA module.

References:     None.

    Copyright (c) 2011, Matrox Graphics Inc.
    All Rights Reserved.

\************************************************************************************************************/

#ifndef INC_LMEXA_H
#define INC_LMEXA_H

// -----------------------------------------------------------------------------------------------------------
//                                 I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------
#include "Liberatus.h"


// -----------------------------------------------------------------------------------------------------------
//                                 C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------
typedef struct LMEXA_HANDLE*         LMexa_Handle;
typedef struct LPIXMAP_HANDLE*       LPixmap_Handle;

// Flags for the LMexa_PrepareAccess function.
#define LMEXA_ACCESS_SRC    0x00000001      // Indicate that the pixmap is accessed as a source.
#define LMEXA_ACCESS_MSK    0x00000002      // Indicate that the pixmap is accessed as a mask.
#define LMEXA_ACCESS_DST    0x00000004      // Indicate that the pixmap is accessed as a destination.

/************************************************************************************************************\

Structure:      LMexa_CreatePixmapFlags

Description:    Flags for LMexa_CreatePixmap function.

Comments:       None.

\************************************************************************************************************/
typedef union tagLMexa_CreatePixmapFlags
{
    struct
    {
        MBOOL32 uiCpuAccess :  1;   // If MTRUE, the CPU access to the pixmap image shall be guarantied. 
        MUINT32 uiReserved  : 31;   // Reserved for futur expansion. Must be set to zero for now.
    };
    
    MUINT32 uiValue;                // Shortcut for setting the value of all the members at once.
    
} LMexa_CreatePixmapFlags;

/************************************************************************************************************\

Enum:           LMexa_AttributeType

Description:    Enumeration of the types of attributes supported by LMexa. It describes the type of a 
                structure containing attributes.

Comments:       The first member of an attributes structure is of type LMexa_AttributeType.

\************************************************************************************************************/
typedef enum
{
    LMexa_AttributeType_NONE = 0,    // Corresponds to an invalid attributes structure.
    LMexa_AttributeType_CONTEXT,     // Corresponds to a structure of the type LMexa_ContextAttributes.
    LMexa_AttributeType_COLORBUFFER, // Corresponds to a structure of the type LMexa_ColorBufferAttributes.
    LMexa_AttributeType_SWAPBUFFERS, // Corresponds to a structure of the type LMexa_SwapBuffersAttributes.
    LMexa_AttributeType_CONFIG,      // Corresponds to a structure of the type LMexa_configAttributes.

} LMexa_AttributeType;

/************************************************************************************************************\

Structure:      LMexa_ContextAttributes

Description:    Structure for rendering context creation attributes. It provides the immutable attributes 
                related to the creation of a rendering context.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLMexa_ContextAttributes
{
    LMexa_AttributeType  eAttributeType;    // Must be LMexa_AttributeType_CONTEXT.
    MUINT32              bColorBuffer : 1;  // Indicate that an internal color buffer shall be created. 
                                            // Default is TRUE.
    MUINT32              uiWidth;           // Width in pixels of the color and depth-stencil buffers. 
                                            // Default is 1.
    MUINT32              uiHeight;          // Height in pixels of the color and depth-stencil buffers. 
                                            // Default is 1.
    LPixelFormat         ePixelformat;      // Pixel format of the color buffers. 
                                            // Default is LPixelFormat_RGB32.
    LDeviceThread_Handle hDeviceThread;     // Handle of the device-thread to use for MEXA commands. If NULL 
                                            // is specified an internal device-thread is used.
   
} LMexa_ContextAttributes;

/************************************************************************************************************\

Structure:      LMexa_ColorBufferAttributes

Description:    Structure for color buffer attributes. It provides the handle of the current color buffer 
                being used by the rendering context.

Comments:       The current color buffer may changes as a result of calling the LMexa_SwapBuffers functions.

\************************************************************************************************************/
typedef struct tagLMexa_ColorBufferAttributes 
{
    LMexa_AttributeType eAttributeType; // Must be LMexa_AttributeType_COLORBUFFER.
    LBuffer_Handle      hBuffer;        // Handle of a buffer corresponding to the current color buffer.
   
} LMexa_ColorBufferAttributes;

/************************************************************************************************************\

Enum:           LMexa_SwapBuffersMode

Description:    Enumeration of SwapBuffers modes. It describes the modes of operation of the SwapBuffers 
                function.

Comments:       The mode of operation summarize the operation of SwapBuffers. See the SwapBuffers function 
                for a precise description of the operation being performed for a particular mode of 
                operation.

\************************************************************************************************************/
typedef enum
{
    LMexa_SwapBuffersMode_COPY      // Copy the content of the current color buffer to another one. The 
                                    // current color buffer doesn't change.

} LMexa_SwapBuffersMode;

/************************************************************************************************************\

Structure:      LMexa_SwapBuffersAttributes

Description:    Structure for swap buffers attributes.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLMexa_SwapBuffersAttributes
{
    LMexa_AttributeType   eAttributeType;   // Must be LMexa_AttributeType_SWAPBUFFERS
    LMexa_SwapBuffersMode eMode;            // Swap buffers mode of operation. 
                                            // Default is LMexa_SwapBuffersMode_COPY.
    LBuffer_Handle        hDesktop;         // Handle to a buffer serving as the desktop buffer when the 
                                            // window reside. Default is NULL.
    MUINT32               uiWindowX;        // X coordinate of the window origin within the desktop.
                                            // Default is 0.
    MUINT32               uiWindowY;        // Y coordinate of the window origin within the desktop.
                                            // Default is 0.
    MUINT32               uiWindowWidth;    // Width of the window. Default is 0.
    MUINT32               uiWindowHeight;   // Height of the window. Default is 0.
    MUINT64               uiWindowTracker;  // Handle to a window tracker object for validating the previous 
                                            // window and desktop information in kernel prior to perform the 
                                            // operation. Default is NULL.
    MFLAG32               uiFlags;          // Flags specific to the mode of operation. Default is 0.

} LMexa_SwapBuffersAttributes;

/************************************************************************************************************\

Enum:           LMexa_PixmapBufferMode

Description:    Enumeration of PixmapBuffer modes. It describes the modes of operation of the PixmapBuffer 
                function.

Comments:       

\************************************************************************************************************/
typedef enum
{
    LMexa_PixmapBufferMode_COPY     // Copy the buffer to the texture image.

} LMexa_PixmapBufferMode;

/************************************************************************************************************\

Enum:           LMexa_CopyMode

Description:    Enumeration of LMexa_Copy scan-direction modes. It describes the modes of operation of the 
                LMexa_Copy function.

Comments:       None.

\************************************************************************************************************/
typedef enum
{
    LMexa_CopyMode_RIGHT_BOTTOM,    // Scan from left to right and top to bottom.
    LMexa_CopyMode_LEFT_BOTTOM,     // Scan from right to left and top to bottom.
    LMexa_CopyMode_RIGHT_TOP,       // Scan from left to right and bottom to top.
    LMexa_CopyMode_LEFT_TOP,        // Scan from right to left and bottom to top.
    LMexa_CopyMode_AUTO             // Scan-direction automatically adjusted.

} LMexa_CopyMode;

/************************************************************************************************************\

Structure:      LMexa_ConfigAttributes

Description:    Structure for configuration attributes. It provides information about the configuration and 
                capabilities of the graphics engine.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLMexa_ConfigAttributes 
{
    LMexa_AttributeType eAttributeType;         // Must be LMexa_AttributeType_COLORBUFFER.
    MUINT32             uiMaxPixmapWidth;       // Maximum allowed width of a pixmap.
    MUINT32             uiMaxPixmapHeight;      // Maximum allowed height of a pixmap.
    MUINT32             bCopyMode : 1;          // The graphics engine supports the specification of a copy 
                                                // mode to the  LMexa_PrepareCopy function when this flag is 
                                                // TRUE. Otherwise the eCopyMode member indicate the only 
                                                // mode supported.
    MUINT32             bPowerOfTwoPicture : 1; // The graphics engine only supports power of two pictures 
                                                // when this flag is TRUE.
    LMexa_CopyMode      eCopyMode;              // The graphics engine only supports the specification of 
                                                // this copy mode to the LMexa_PrepareCopy function when 
                                                // bCopyMode is FALSE.
   
} LMexa_ConfigAttributes;

/************************************************************************************************************\

Structure:      LMexa_Matrix

Description:    Picture transformation matrix. It corresponds to a member of the LMexa_Picture structure.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLMexa_Matrix
{
    MINT32 aiValue[3][3];       // Array of 16.16 fixed point integer values of a 3x3 matrix.
 
} LMexa_Matrix;

/************************************************************************************************************\

Enum:           LMexa_Filter

Description:    Picture filtering mode. It corresponds to a member of the LMexa_Picture structure.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLMexa_Filter
{
    LMexa_Filter_NEAREST,       // Perform nearest (point-sampling) filtering.
    LMexa_Filter_BILINEAR,      // Perform bilinear filtering.
    LMexa_Filter_FAST,          // Synonym of NEAREST.
    LMexa_Filter_GOOD,          // Synonym of BILINEAR.
    LMexa_Filter_BEST           // Synonym of BILINEAR.

} LMexa_Filter;

/************************************************************************************************************\

Enum:           LMexa_Repeat

Description:    Picture repeat mode. It corresponds to a member of the LMexa_Picture structure.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLMexa_Repeat
{
    LMexa_Repeat_NONE,          // Do not repeat the source picture.
    LMexa_Repeat_NORMAL         // Repeat the source picture.

} LMexa_Repeat;

/************************************************************************************************************\

Enum:           LMexa_LogicOp

Description:    Logical operator for drawing operation.

Comments:       None.

\************************************************************************************************************/
typedef enum tagLMexa_LogicOp
{
    LMexa_LogicOp_CLEAR,            // DstBit = 0
    LMexa_LogicOp_AND,              // DstBit = SrcBit AND DstBit
    LMexa_LogicOp_AND_REVERSE,      // DstBit = SrcBit AND NOT DstBit
    LMexa_LogicOp_COPY,             // DstBit = SrcBit
    LMexa_LogicOp_AND_INVERTED,     // DstBit = NOT SrcBit AND DstBit
    LMexa_LogicOp_NOOP,             // DstBit = DstBit
    LMexa_LogicOp_XOR,              // DstBit = SrcBit XOR DstBit
    LMexa_LogicOp_OR,               // DstBit = SrcBit OR DstBit
    LMexa_LogicOp_NOR,              // DstBit = NOT SrcBit AND NOT DstBit
    LMexa_LogicOp_EQUIV,            // DstBit = NOT SrcBit XOR DstBit
    LMexa_LogicOp_INVERT,           // DstBit = NOT DstBit
    LMexa_LogicOp_OR_REVERSE,       // DstBit = SrcBit OR NOT DstBit
    LMexa_LogicOp_COPY_INVERTED,    // DstBit = NOT SrcBit
    LMexa_LogicOp_OR_INVERTED,      // DstBit = NOT SrcBit OR DstBit
    LMexa_LogicOp_NAND,             // DstBit = NOT SrcBit OR NOT DstBit
    LMexa_LogicOp_SET               // DstBit = 1   

} LMexa_LogicOp;

/************************************************************************************************************\

Enum:           LMexa_CompositeOp

Description:    Compositing operator of the LMexa_Composite command.

                For the composition of a pixel, the four channels of the R, G, B, A image are computed with 
                the following equation:
                
                    C = Cs * Fs + Cd * Fd
                    
                where C, Cs, Cd are the values of the respective channels and Fs and Fd come from the 
                comments on the following enumeration. As and Ad represent the alpha channel of the source 
                and destination colors respectively. The suffix "s" stands for source while "d" stands for 
                destination. 

Comments:       The SATURATE, DISJOINT and CONJOINT operators are not supported at this time.

\************************************************************************************************************/
typedef enum tagLMexa_CompositeOp
{
    LMexa_CompositeOp_CLEAR,                    // Fs = 0,                  Fd = 0
    LMexa_CompositeOp_SRC,                      // Fs = 1,                  Fd = 0
    LMexa_CompositeOp_DST,                      // Fs = 0,                  Fd = 1
    LMexa_CompositeOp_OVER,                     // Fs = 1,                  Fd = 1-As
    LMexa_CompositeOp_OVER_REVERSE,             // Fs = 1-Ad,               Fd = 1
    LMexa_CompositeOp_IN,                       // Fs = Ad,                 Fd = 0
    LMexa_CompositeOp_IN_REVERSE,               // Fs = 0,                  Fd = As
    LMexa_CompositeOp_OUT,                      // Fs = 1-Ad,               Fd = 0
    LMexa_CompositeOp_OUT_REVERSE,              // Fs = 0,                  Fd = 1-As
    LMexa_CompositeOp_ATOP,                     // Fs = Ad,                 Fd = 1-As
    LMexa_CompositeOp_ATOP_REVERSE,             // Fs = 1-Ad,               Fd = As
    LMexa_CompositeOp_XOR,                      // Fs = 1-Ad,               Fd = 1-As
    LMexa_CompositeOp_ADD,                      // Fs = 1,                  Fd = 1
    LMexa_CompositeOp_SATURATE,                 // Fs = min(1,(1-Ad)/As),   Fd = 1
    LMexa_CompositeOp_DISJOINT_CLEAR,           // Fs = 0,                  Fd = 0
    LMexa_CompositeOp_DISJOINT_SRC,             // Fs = 1,                  Fd = 0
    LMexa_CompositeOp_DISJOINT_DST,             // Fs = 0,                  Fd = 1
    LMexa_CompositeOp_DISJOINT_OVER,            // Fs = 1,                  Fd = min(1,(1-As)/Ad)
    LMexa_CompositeOp_DISJOINT_OVER_REVERSE,    // Fs = min(1,(1-Ad)/As),   Fd = 1
    LMexa_CompositeOp_DISJOINT_IN,              // Fs = max(1-(1-Ad)/As,0), Fd = 0
    LMexa_CompositeOp_DISJOINT_IN_REVERSE,      // Fs = 0,                  Fd = max(1-(1-As)/Ad,0)
    LMexa_CompositeOp_DISJOINT_OUT,             // Fs = min(1,(1-Ad)/As),   Fd = 0
    LMexa_CompositeOp_DISJOINT_OUT_REVERSE,     // Fs = 0,                  Fd = min(1,(1-As)/Ad)
    LMexa_CompositeOp_DISJOINT_ATOP,            // Fs = max(1-(1-Ad)/As,0)  Fd = min(1,(1-As)/Ad)
    LMexa_CompositeOp_DISJOINT_ATOP_REVERSE,    // Fs = min(1,(1-Ad)/As),   Fd = max(1-(1-As)/Ad,0)
    LMexa_CompositeOp_DISJOINT_XOR,             // Fs = min(1,(1-Ad)/As),   Fd = min(1,(1-As)/Ad)
    LMexa_CompositeOp_CONJOINT_CLEAR,           // Fs = 0,                  Fd = 0
    LMexa_CompositeOp_CONJOINT_SRC,             // Fs = 1,                  Fd = 0
    LMexa_CompositeOp_CONJOINT_DST,             // Fs = 0,                  Fd = 1
    LMexa_CompositeOp_CONJOINT_OVER,            // Fs = 1,                  Fd = max(1-As/Ad,0)
    LMexa_CompositeOp_CONJOINT_OVER_REVERSE,    // Fs = max(1-Ad/As,0),     Fd = 1
    LMexa_CompositeOp_CONJOINT_IN,              // Fs = min(1,Ad/As),       Fd = 0
    LMexa_CompositeOp_CONJOINT_IN_REVERSE,      // Fs = 0,                  Fd = min(As/Ad,1)
    LMexa_CompositeOp_CONJOINT_OUT,             // Fs = max(1-Ad/As,0),     Fd = 0
    LMexa_CompositeOp_CONJOINT_OUT_REVERSE,     // Fs = 0,                  Fd = max(1-As/Ad,0)
    LMexa_CompositeOp_CONJOINT_ATOP,            // Fs = min(1,Ad/As),       Fd = max(1-As/Ad,0)
    LMexa_CompositeOp_CONJOINT_ATOP_REVERSE,    // Fs = max(1-Ad/As,0),     Fd = min(1,As/Ad)
    LMexa_CompositeOp_CONJOINT_XOR              // Fs = max(1-Ad/As,0),     Fd = max(1-As/Ad,0)

} LMexa_CompositeOp;

/************************************************************************************************************\

Structure:      LMexa_Picture

Description:    Structure for attaching a number of attributes to a pixmap. A picture provides additional 
                states to a pixmap. These states are used for compositing operations.

Comments:       The uiFlags member of this structure matches the bitfield definition of the X-Windows 
                PictureRec structure.

\************************************************************************************************************/
typedef struct tagLMexa_Picture
{
    LPixmap_Handle  hPixmap;        // Handle of the associated pixmap. NULL means internal color buffer.
    MUINT32         uiWidth;        // Width of the associated pixmap.
    MUINT32         uiHeight;       // Height of the associated pixmap.
    LPixelFormat    ePixelFormat;   // Pixel format of the associated pixmap.
    LMexa_Matrix*   poTransform;    // Transform matrix to apply to the coordinates of the source and mask 
                                    // pixmaps. NULL means Identity.
    LMexa_Filter    eFilter;        // Filtering mode to apply to the source and mask pixmaps.

    union
    {
        struct
        {
            MUINT32
            bRepeat            : 1,     // Apply a repeat operation to the source and mask pixmaps.
            bGraphicsExposures : 1,
            bSubWindowMode     : 1,
            bPolyEdge          : 1,
            bPolyMode          : 1,
            bFreeCompClip      : 1,
            eClientClipType    : 2,
            bComponentAlpha    : 1,     // Indicates how the mask combines with the source color.
            eRepeatType        : 2,     // Repeat mode to apply to the source and mask pixmaps.
            uReserved1         : 21;
        };
 
        MUINT32 uiValue;                // Shortcut for setting the value of all the members at once.
        
    } oFlags;
 
} LMexa_Picture;

// -----------------------------------------------------------------------------------------------------------
//                           G L O B A L   V A R I A B L E   R E F E R E N C E S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                         G L O B A L   F U N C T I O N   D E C L A R A T I O N S
// -----------------------------------------------------------------------------------------------------------

#if defined (__cplusplus)
extern "C" {
#endif

/************************************************************************************************************\

Function:       LMexa_Create

Description:    Create an object of the LMexa class representing an EXA rendering context. We further use 
                the "context" terminology to refer to this object.
                
Parameters:     IN  hDevice         Handle of the device for which to create a rendering context.
                IN  peAttributes    Pointer to the attribute type member of a structure of type 
                                    LMexa_ContextAttributes describing the attributes of the context to 
                                    create. A NULL pointer indicates the use of the default attributes.
                OUT phContext       Pointer to a handle to receive the handle of the created rendering 
                                    context.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       The method receives as input a set of immutable attributes for the rendering context. The 
                structure shall be of type LMexa_AttributeType_CONTEXT. It is possible to pass a NULL pointer 
                to the attributes structure indicating that the default values shall be used.
                
                The maximum width and height of the color and pixmap buffers can be retrieved using the 
                LMexa_GetAttributes function with the LMexa_AttributeType_CONFIG type.

                The following pixel formats types are available: RGB32, ARGB32, RGB565, RGB555, ARGB555, 
                GRAY8.
                
\************************************************************************************************************/
LAPI LStatus LMexa_Create
(
    LDevice_Handle       hDevice, 
    LMexa_AttributeType* peAttributes,
    LMexa_Handle*        phContext
);

/************************************************************************************************************\

Function:       LMexa_Destroy

Description:    Destroy an LMexa rendering context.

Parameters:     IN hContext     Handle of rendering context.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       None.
                
\************************************************************************************************************/
LAPI LStatus LMexa_Destroy
(
    LMexa_Handle hContext
);

/************************************************************************************************************\

Function:       LMexa_GetAttributes

Description:    This method provide a mechanism to retrieve attributes from a rendering context. The type of 
                attributes is provided by the output structure.
                
Parameters:     IN     hContext     Handle of a rendering context.
                IN OUT peAttributes Pointer to the attribute type member of a structure of attributes. On 
                                    input the type member describes the type of the structure and on output 
                                    the remaining members of the structure are filled with the attributes 
                                    information.
                                    
Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       Only one type of attributes can be retrieved at a time.
                
\************************************************************************************************************/
LAPI LStatus LMexa_GetAttributes
(
    LMexa_Handle         hContext,
    LMexa_AttributeType* peAttributes
);

/************************************************************************************************************\

Function:       LMexa_GetDefaultAttributes

Description:    This method provide a mechanism to retrieve the default attributes from a rendering context. 
                The type of attributes is provided by the output structure.

Parameters:     IN     hContext     Handle of a rendering context.
                IN OUT peAttributes Pointer to the attribute type member of a structure of attributes. On 
                                    input the type member describes the type of the structure and on output 
                                    the remaining members of the structure are filled with the attributes 
                                    information.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       Only one type of attributes can be retrieved at a time. This function provides a mechanism 
                for retrieving an attribute structure filled with default values such that an application 
                only have to care about the subset of the members it is interested in.
                
\************************************************************************************************************/
LAPI LStatus LMexa_GetDefaultAttributes
(
    LMexa_Handle         hContext,
    LMexa_AttributeType* peAttributes
);

/************************************************************************************************************\

Function:       LMexa_SwapBuffers

Description:    This function publishes the content of the current color buffer to the desktop buffer 
                according to the specified mode of operation and attributes.
                
                The possible modes are:
                
                    LMexa_SwapBuffersMode_COPY
                    
                    The calling CPU thread is put to sleep while waiting for the previous LMexa_SwapBuffers 
                    to complete. Then the current component color buffer is copied to the desktop buffer.

Parameters:     IN hContext     Handle of a rendering context.
                IN peAttributes Pointer to the attribute type member of a structure of type 
                                LMexa_SwapBuffersAttributes describing the attributes of the swap buffers 
                                operation. A NULL pointer indicates the use of the default attributes.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       The COPY mode keeps the content of the draw buffer valid.

                The swap-buffer operation can be synchronized with the display retrace or any other event 
                issuing a synchronization command in the same device-thread prior to issuing the swap-buffer 
                operation.

                The window region tracking functionality is specific to the operating system implementation. 
                The window tracking object is totally opaque to the LMexa module that forward the handle to 
                the kernel driver along with the swap-buffer operation. In the operating system kernel the 
                tracking object provides the effective desktop buffer and region of a window for the 
                swap-buffer operation.
                
\************************************************************************************************************/
LAPI LStatus LMexa_SwapBuffers
(
    LMexa_Handle         hContext,
    LMexa_AttributeType* peAttributes
);

/************************************************************************************************************\

Function:       LMexa_PixmapBuffer

Description:    This function provides a pixmap image to a pixmap.

                The possible modes are:
                
                    LMexa_PixmapBufferMode_COPY
                    
                    Copy the content of the buffer to the pixmap image. 
                    
                The COPY mode supports a limited form of format conversion subject to the constraint that 
                buffer components can be dropped during the conversion but new components cannot be added. 
                The error LStatus_UNSUPPORTED is returned when the format conversion is not possible.

Parameters:     IN hContext     Handle of a rendering context.
                IN hPixmap      Handle of a pixmap.
                IN eMode        Mode of operation.
                IN hBuffer      Handle of a compatible buffer.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_UNSUPPORTED     The format conversion is not possible.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       This function provides format conversion capabilities that are not available with the 
                LMexa_Copy operation.
                
\************************************************************************************************************/
LAPI LStatus LMexa_PixmapBuffer
(
    LMexa_Handle           hContext,
    LPixmap_Handle         hPixmap,
    LMexa_PixmapBufferMode eMode,
    LBuffer_Handle         hBuffer
);

/************************************************************************************************************\

Function:       LMexa_GetPixmapBuffer

Description:    This function retrieve the buffer associated to a pixmap.

Parameters:     hContext    Handle of a rendering context.
                hPixmap     Handle of a pixmap.
                phBuffer    Pointer to a handle to receive the buffer handle.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       This function provides a mechanism for accessing a pixmap as a Liberatus buffer.
                
\************************************************************************************************************/
LAPI LStatus LMexa_GetPixmapBuffer
(
    LMexa_Handle    hContext,
    LPixmap_Handle  hPixmap,
    LBuffer_Handle* phBuffer
);

/************************************************************************************************************\

Function:       LMexa_MarkSync

Description:    This function can be used to synchronize with the execution of EXA commands. It retrieves a 
                synchronization marker indicating the current position in the rendering context time-line. 

Parameters:     IN  hContext    Handle of a rendering context.
                OUT puiMarker   Pointer to a 64 bit synchronization marker.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       Each rendering context has a specific time-line.
                
\************************************************************************************************************/
LAPI LStatus LMexa_MarkSync
(
    LMexa_Handle hContext,
    MUINT64*     puiMarker
);

/************************************************************************************************************\

Function:       LMexa_WaitSync

Description:    This function can be used to synchronize with the execution of EXA commands. It puts the 
                calling CPU thread to sleep until the graphics engine terminates the execution of the 
                commands anterior to the synchronization marker and that the buffers' memory is updated.

Parameters:     IN hContext                 Handle of a rendering context.
                IN uiMarker                 Synchronization marker.
                IN uiTimeOutInMilliSeconds  Maximum wait time. Set to LINFINITE_TIMEOUT to disable timeout.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.
                LStatus_TIMEOUT         The time specified has passed.

Comments:       This function performs an implicit call to LMexa_Flush to flush previous commands to the 
                rendering engine prior to wait for the synchronization marker.
                
\************************************************************************************************************/
LAPI LStatus LMexa_WaitSync
(
    LMexa_Handle hContext,
    MUINT64      uiMarker,
    MUINT32      uiTimeOutInMilliSeconds
);

/************************************************************************************************************\

Function:       LMexa_Flush

Description:    Flush accumulated drawing commands.

Parameters:     IN hContext     Handle of a rendering context

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       The API provides no guaranty on the time of execution of graphics engine commands resulting 
                from calling API functions unless explicitly stated for a specific function. When 
                LMExa_Flush is called, accumulated commands are submitted to the graphics engine command 
                queue for execution as soon as possible.
                
\************************************************************************************************************/
LAPI LStatus LMexa_Flush
(
    LMexa_Handle hContext
);

/************************************************************************************************************\

Function:       LMexa_CreatePixmap

Description:    Create a pixmap. The EXA drawing commands operate on pixmaps. A pixmap corresponds to a 2D 
                buffer of a given pixel format type. The flag oPixmapFlags.uiCpuAccess indicate 
                that CPU access to the pixmap image shall be guarantied. In the adverse this function call 
                fails with the status code LStatus_INVALID_OPERATION. Using this flag guaranty that the 
                function LMexa_PrepareAccess never fail for the created pixmap.

Parameters:     IN  hContext        Handle of a rendering context.
                IN  uiWidth         Width of the pixmap.
                IN  uiHeight        Height of the pixmap.
                IN  ePixelFormat    Pixel format of the pixmap.
                IN  oPixmapFlags    Flags associated to the creation of the pixmap. 
                OUT phPixmap        Pointer to a handle to receive the handle of the created pixmap.
                OUT puiOffset       Pointer to an unsigned integer to receive the offset in memory of the 
                                    pixmap image or NULL if this information is not required.
                OUT puiLineSize     Pointer to an unsigned integer to receive the line size in bytes of the 
                                    pixmap image or NULL if this information is not required.
                OUT puiSize         Pointer to an unsigned integer to receive the size in bytes of the pixmap
                                    image or NULL if this information is not required.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       The puiOffset, puiLineSize and puiSize values provide enough information to the caller to 
                manage the pixmap resources according to a virtual memory heap. The position and size of the 
                pixmaps in a common linear address space could be used by an application memory manager to 
                reclaim contiguous memory by the destruction of a group of adjacent pixmaps.

                The flag oPixmapFlags.uiCpuAccess should be used with parsimony as it may adversely 
                impact the performance of the drawing operations. It is useful for migrating pixmap 
                asynchronously without using the LMexa_Upload and LMexa_Download function that operate 
                synchronously with the graphics engine and CPU. A number of pixmap created with the 
                uiCpuAccess option could be filed by the CPU and transferred to regular pixmap efficiently.

\************************************************************************************************************/
LAPI LStatus LMexa_CreatePixmap
(
    LMexa_Handle    hContext,
    MUINT32         uiWidth, 
    MUINT32         uiHeight, 
    LPixelFormat    ePixelFormat,
    LMexa_CreatePixmapFlags oPixmapFlags,
    LPixmap_Handle* phPixmap,
    MUINT64*        puiOffset,
    MUINT32*        puiLineSize,
    MUINT32*        puiSize 
);

/************************************************************************************************************\

Function:       LMexa_CreatePixmapFromBuffer

Description:    Create a pixmap from a buffer. The EXA drawing commands operate on pixmaps. A pixmap 
                corresponds to a 2D buffer of a given pixel format type.

Parameters:     IN  hContext    Handle of a rendering context.
                IN  hBuffer     Handle to a compliant buffer.
                OUT phPixmap    Pointer to a handle to receive the handle of the created pixmap.
                OUT puiOffset   Pointer to an unsigned integer to receive the offset in memory of the pixmap 
                                image or NULL if this information is not required.
                OUT puiLineSize Pointer to an unsigned integer to receive the line size in bytes of the 
                                pixmap image or NULL if this information is not required.
                OUT puiSize     Pointer to an unsigned integer to receive the size in bytes of the pixmap 
                                image or NULL if this information is not required.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       The puiOffset, puiLineSize and puiSize values provide enough information to the caller to 
                manage the pixmap resources according to a virtual memory heap.
                
\************************************************************************************************************/
LAPI LStatus LMexa_CreatePixmapFromBuffer
(
    LMexa_Handle    hContext,
    LBuffer_Handle  hBuffer,
    LPixmap_Handle* phPixmap,
    MUINT64*        puiOffset,
    MUINT32*        puiLineSize,
    MUINT32*        puiSize 
);

/************************************************************************************************************\

Function:       LMexa_DestroyPixmap

Description:    Destroy a pixmap.

Parameters:     IN hContext     Handle to a rendering context.
                IN hPixmap      Handle to a pixmap.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       If a pixmap has been created from a buffer, this operation decrement the reference count of 
                the buffer and destroy the pixmap. The buffer gets destroyed with the reference count 
                reaches zero.
                
\************************************************************************************************************/
LAPI LStatus LMexa_DestroyPixmap
(
    LMexa_Handle   hContext,
    LPixmap_Handle hPixmap
);

/************************************************************************************************************\

Function:       LMexa_PrepareSolid

Description:    Prepare a series of solid fill drawing operations to the destination pixmap.

                The plane mask controls the bitwise update of the destination pixmap with the color 
                resulting of the application of the logic operator between the foreground color and the 
                destination color.

                If the operation is not supported by the graphics engine, the status code 
                LStatus_NOT_SUPPORTED is returned to indicate that the operation should be performed by the 
                CPU.

Parameters:     IN hContext             Handle of a rendering context.
                IN hPixmap              Handle of a destination pixmap. NULL indicate the use of the internal 
                                        color buffer.
                IN eLogicOp             Logic operator to apply.
                IN uiPlaneMask          Plane mask to apply.
                IN uiForegroundColor    Color of the foreground to apply.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       The plane-mask and foreground-color shall match the colors-pace format of the pixmap. The 
                lower bits up to the depth of the pixmap buffer are considered.
                
                A NULL value as the hPixmap handle indicates the use of the current internal color buffer. 
                
                The plane-mask controls the update of individual bit of the destination pixmap color 
                elements.
                
\************************************************************************************************************/
LAPI LStatus LMexa_PrepareSolid
(
    LMexa_Handle   hContext,
    LPixmap_Handle hPixmap,
    LMexa_LogicOp  eLogicOp,
    MUINT32        uiPlaneMask,
    MUINT32        uiForegroundColor 
);

/************************************************************************************************************\

Function:       LMexa_Solid

Description:    Perform a solid fill drawing operation. Fill a rectangle area of the destination pixmap with 
                the foreground color.

Parameters:     IN hContext     Handle of a rendering context.
                IN iDstX        Left X coordinate.
                IN iDstY        Top Y coordinate.
                IN iWidth       Width of rectangle.
                IN iHeight      Height of rectangle.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       None.
                
\************************************************************************************************************/
LAPI LStatus LMexa_Solid
(
    LMexa_Handle hContext,
    MINT32       iDstX,
    MINT32       iDstY,
    MINT32       iWidth,
    MINT32       iHeight
);

/************************************************************************************************************\

Function:       LMexa_DoneSolid

Description:    Complete a series of solid fill drawing operations.

Parameters:     IN hContext     Handle of a rendering context.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       Calling this function notifies the completion of a series of drawing operations. A call to 
                the LMexa_Flush function is required to force the update of the destination pixmap.
                
\************************************************************************************************************/
LAPI LStatus LMexa_DoneSolid
(
    LMexa_Handle hContext
);

/************************************************************************************************************\

Function:       LMexa_PrepareCopy

Description:    Prepare a series of solid copy drawing operations from the source pixmap to the destination 
                pixmap.

                The plane mask controls the bitwise update of the destination pixmap with the color resulting 
                of the application of the logic operator between the source pixmap color and the destination 
                pixmap color.

                If the operation is not supported by the graphics engine, the status code 
                LStatus_NOT_SUPPORTED is returned to indicate that the operation should be performed by the 
                CPU.

Parameters:     IN hContext     Handle of a rendering context.
                IN hSrcPixmap   Handle of a source pixmap. NULL indicate the use of the internal color buffer.
                IN hDstPixmap   Handle of a destination pixmap. NULL indicate the use of the internal color 
                                buffer.
                IN eMode        Copy mode to use.
                IN eLogicOp     Logic operator to apply.
                IN uiPlaneMask  Plane mask to apply.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.
                LStatus_NOT_SUPPORTED   The requested operation is not supported by the graphics engine.

Comments:       The plane-mask and foreground-color shall match the colors-pace format of the pixmap. The 
                lower bits up to the depth of the pixmap buffer are considered.

                A NULL value as the hSrcPixmap or hDstPixmap handles indicate the use of the current internal 
                color buffer.

                The copy-mode parameter provides a mechanism to deal with self-overlapping copies when 
                hSrcPixmap and hDstPixmap are equal. The attributes structure LMexa_AttributeType_CONFIG 
                provides the information about the support of a flexible copy-mode with this operation. If 
                the member bCopyMode is TRUE a flexible copy-mode is supported. Otherwise only the copy-mode 
                specified by the eCopyMode member is supported.                
                
\************************************************************************************************************/
LAPI LStatus LMexa_PrepareCopy
(
    LMexa_Handle   hContext,
    LPixmap_Handle hSrcPixmap,
    LPixmap_Handle hDstPixmap,
    LMexa_CopyMode eMode,
    LMexa_LogicOp  eLogicOp,
    MUINT32        uiPlaneMask
);

/************************************************************************************************************\

Function:       LMexa_Copy

Description:    Perform a solid copy drawing operation at the specified coordinates.

Parameters:     IN hContext     Handle of a rendering context.
                IN iSrcX        Left X coordinate of source.
                IN iSrcY        Top Y coordinate of source.
                IN iDstX        Left X coordinate of destination.
                IN iDstY        Top Y coordinate of destination.
                IN iWidth       Width of rectangle.
                IN iHeight      Height of rectangle.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       None.
                
\************************************************************************************************************/
LAPI LStatus LMexa_Copy
(
    LMexa_Handle hContext,
    MINT32       iSrcX,
    MINT32       iSrcY,
    MINT32       iDstX,
    MINT32       iDstY,
    MINT32       iWidth,
    MINT32       iHeight
);

/************************************************************************************************************\

Function:       LMexa_DoneCopy

Description:    Complete a series of solid copy drawing operations.

Parameters:     IN hContext     Handle of a rendering context.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       Calling this function notifies the completion of a series of drawing operations. A call to 
                the LMexa_Flush function is required to force the update of the destination pixmap.
                
\************************************************************************************************************/
LAPI LStatus LMexa_DoneCopy
(
    LMexa_Handle hContext
);

/************************************************************************************************************\

Function:       LMexa_CheckComposite

Description:    Check if a composition operation is supported with the given source, mask and destination 
                pictures. The difference between this function and LMexa_PrepareComposite is that when this 
                function is called the pixmap locations are not final.

                If the operation is not supported by the graphics engine, the status code 
                LStatus_NOT_SUPPORTED is returned to indicate that the operation should be performed by the 
                CPU.
                
Parameters:     IN hContext     Handle of a rendering context.
                IN eCompositeOp Composition operator.
                IN poSrcPicture Pointer to a picture structure for the source.
                IN poMskPicture Pointer to a picture structure for the mask or NULL if a mask is not used.
                IN poDstPicture Pointer to a picture structure for the destination.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.
                LStatus_UNSUPPORTED     The composition operation is not supported.

Comments:       None.
                
\************************************************************************************************************/
LAPI LStatus LMexa_CheckComposite
(
    LMexa_Handle      hContext,
    LMexa_CompositeOp eCompositeOp,
    LMexa_Picture*    poSrcPicture,
    LMexa_Picture*    poMskPicture,
    LMexa_Picture*    poDstPicture
);

/************************************************************************************************************\

Function:       LMexa_PrepareComposite

Description:    Prepare for a series of composition operations.

                If a mask is not used the compositing operation becomes "dest = source OP dest"  instead of 
                "dest = (source IN mask) OP dest".

                The source and mask picture may have a transformation matrix, a filtering mode and a repeat 
                mode. The transformation matrix, filtering mode and repeat mode have no effect for a 
                destination picture.

                If the operation is not supported by the graphics engine, the status code 
                LStatus_NOT_SUPPORTED is returned to indicate that the operation should be performed by the 
                CPU.

Parameters:     IN hContext     Handle of a rendering context.
                IN eCompositeOp Composition operator.
                IN poSrcPicture Pointer to a picture structure for the source.
                IN poMskPicture Pointer to a picture structure for the mask or NULL if a mask is not used.
                In poDstPicture Pointer to a picture structure for the destination.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.
                LStatus_UNSUPPORTED     The composition operation is not supported.

Comments:       Only the mask picture may have bComponentAlpha set to TRUE.
                
\************************************************************************************************************/
LAPI LStatus LMexa_PrepareComposite
(
    LMexa_Handle      hContext,
    LMexa_CompositeOp eCompositeOp,
    LMexa_Picture*    poSrcPicture,
    LMexa_Picture*    poMskPicture,
    LMexa_Picture*    poDstPicture
);

/************************************************************************************************************\

Function:       LMexa_Composite

Description:    Perform a composition drawing operation. The composite operation performs complex image 
                composition operation based on an extended Porter-Duff compositing model. The extension 
                resides in the use of a more flexible OP operator instead of the Porter-Duff specific OVER 
                operator. See [1] for an introduction the extended composition model.

                An important aspect of the model is that images R, G and B components are pre-multiplied by 
                the image alpha (A) component. This scheme allows for a unified composition equation for all 
                the components of a pixel (R, G, B an d A).

                    dest = (source IN mask) OP dest
    
                If a mask is not available then it is as if all the components of the mask are 1.0. Otherwise 
                the mask is represented as Cd in the following compositing operator equation. The 
                component-alpha option indicates whether each mask image component is intended as a separate 
                alpha value.

                For the composition of a pixel, the four channels of the image are computed with the 
                following equation:

                    C = Cs * Fs + Cd * Fd
    
                    where C, Cs, Cd are the values of the respective channels and Fs and Fd come from the 
                    compositing operator table.

Parameters:     IN hContext     Handle of a rendering context.
                IN iSrcX        Left X coordinate of the source.
                IN iSrcY        Top Y coordinate of the source.
                IN iMskX        Left X coordinate of the mask.
                IN iMskY        Top Y coordinate of the mask.
                IN iDstX        Left X coordinate of the destination.
                IN iDstY        Top Y coordinate of the destination.
                IN iWidth       Width of the rectangle.
                IN iHeight      Height of the rectangle.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       Note that if a transformation was set on the source or mask pictures, the source and mask 
                rectangles may not be the same size as the destination rectangle.
                
\************************************************************************************************************/
LAPI LStatus LMexa_Composite
(
    LMexa_Handle hContext,
    MINT32       iSrcX,
    MINT32       iSrcY,
    MINT32       iMskX,
    MINT32       iMskY,
    MINT32       iDstX,
    MINT32       iDstY,
    MINT32       iWidth,
    MINT32       iHeight
);

/************************************************************************************************************\

Function:       LMexa_DoneComposite

Description:    Complete a series of composite drawing operations.

Parameters:     IN hContext     Handle of a rendering context.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       Calling this function notifies the completion of a series of drawing operations. A call to 
                the LMexa_Flush function is required to force the update of the destination pixmap.
                
\************************************************************************************************************/
LAPI LStatus LMexa_DoneComposite
(
    LMexa_Handle hContext
);

/************************************************************************************************************\

Function:       LMexa_Upload

Description:    Upload a rectangle from a source image to a destination pixmap.

Parameters:     IN hContext     Handle of a rendering context.
                IN hDstPixmap   Handle of a destination pixmap. NULL means internal color buffer.
                IN iDstX        Left X coordinate of rectangle within pixmap.
                IN iDstY        Top Y coordinate of rectangle within pixmap.
                IN iWidth       Width of rectangle.
                IN iHeight      Height of rectangle.
                IN iSrcLineSize Line size in bytes of source image.
                IN pvSrcData    Pointer to the source image data.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       This function corresponds to a fallback to the functions LMexa_PrepareAccess - 
                LMexa_DoneAccess to migrate a pixmap to CPU accessible storage when the status 
                LStatus_NOT_SUPPORTED is returned.

                The pointer pvSrcData references the first pixel of the rectangle within the source image 
                (upper-left corner). 

                The pixel format of the source image must correspond to the pixel format of the pixmap.

                This function dereferences the data pointed to be pvSrcData at the time of invocation. The 
                source image can be modified by the caller immediately after this function returns.
                
\************************************************************************************************************/
LAPI LStatus LMexa_Upload
(
    LMexa_Handle   hContext,
    LPixmap_Handle hDstPixmap, 
    MINT32         iDstX,
    MINT32         iDstY,
    MINT32         iWidth,
    MINT32         iHeight,
    MINT32         iSrcLineSize,
    void*          pvSrcData
);

/************************************************************************************************************\

Function:       LMexa_Download

Description:    Download a rectangle from a source pixmap to a destination image.

Parameters:     IN hContext     Handle of a rendering context.
                IN hSrcPixmap   Handle of a source pixmap. NULL means internal color buffer.
                IN iSrcX        Left X coordinate of rectangle within pixmap.
                IN iSrcY        Top Y coordinate of rectangle within pixmap.
                IN iWidth       Width of rectangle.
                IN iHeight      Height of rectangle.
                IN iDstLineSize Line size in bytes of destination image.
                IN pvDstData    Pointer to the destination image data.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       This function corresponds to a fallback to the functions LMexa_PrepareAccess - 
                LMexa_DoneAccess to migrate a pixmap to CPU accessible storage when the status 
                LStatus_NOT_SUPPORTED is returned.

                The pointer pvDstData references the first pixel of the rectangle within the source image 
                (upper-left corner). 

                The pixel format of the destination image must correspond to the pixel format of the pixmap.

                This function dereferences the data pointed to be pvDstData at the time of invocation. The 
                destination image can be accessed by the caller immediately after this function returns.
                
\************************************************************************************************************/
LAPI LStatus LMexa_Download
(
    LMexa_Handle   hContext,
    LPixmap_Handle hSrcPixmap, 
    MINT32         iSrcX,
    MINT32         iSrcY,
    MINT32         iWidth,
    MINT32         iHeight,
    MINT32         iDstLineSize,
    void*          pvDstData
);

/************************************************************************************************************\

Function:       LMexa_PrepareAccess

Description:    Prepare for accessing a pixmap with the CPU.

                The uiFlags parameter indicates the type of access being performed on the pixmap. It 
                corresponds to an OR of the following flags LMEXA_ACCESS_SRC, LMEXA_ACCESS_MSK, 
                LMEXA_ACCESS_DST. More than one type can be specified at once. At most one call to 
                LMexa_PrepareAccess is allowed on a pixmap for each type of access. There shall be a call to 
                LMexa_DoneAccess for each type of access (either in one call using an OR of the flags or in 
                multiple calls using separate flags).

                If the operation is not supported, the status code LStatus_NOT_SUPPORTED is returned to 
                indicate that the pixmap should be migrated to application memory using the LMexa_Download 
                function.

                The operation is guarantied to be supported for pixmap created with the CpuAccess flag.

Parameters:     IN  hContext    Handle of a rendering context.
                IN  hPixmap     Handle of a pixmap. NULL means internal color buffer.
                IN  uiFlags     Flags corresponding to the type of access being performed.
                OUT ppvAddress  Pointer to a pointer to receive the address of the image data.
                OUT puiLineSize Pointer to an unsigned integer to receive the line size of the image.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.
                LStatus_NOT_SUPPORTED   The operation is not supported.

Comments:       The application is responsible to synchronize with previous drawing operations targeting the 
                pixmap before calling this function.
                
\************************************************************************************************************/
LAPI LStatus LMexa_PrepareAccess
(
    LMexa_Handle   hContext,
    LPixmap_Handle hPixmap,  
    MFLAG32        uiFlags,
    void**         ppvAddress,
    MUINT32*       puiLineSize
);

/************************************************************************************************************\

Function:       LMexa_DoneAccess

Description:    Complete the access of a pixmap by the CPU.

Parameters:     IN hContext     Handle of a rendering context.
                IN hPixmap      Handle of a pixmap. NULL means internal color buffer.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       When this function returns, the pixmap memory is ready to be used by further drawing 
                commands.
                
\************************************************************************************************************/
LAPI LStatus LMexa_DoneAccess
(
    LMexa_Handle   hContext,
    LPixmap_Handle hPixmap,
    MUINT32        uiFlags
);

// -----------------------------------------------------------------------------------------------------------
//                      I N L I N E S   A N D   T E M P L A T E   D E F I N I T I O N S
// -----------------------------------------------------------------------------------------------------------

#if defined (__cplusplus)
}
#endif

#endif  // #ifndef INC_LMEXA_H
