/************************************************************************************************************\

Module Name:    DeviceThread.h

Description:    Implements a thread synchronization module. 

References:     LDeviceThread.doc Revision 0.4.

    Copyright (c) 2013, Matrox Graphics Inc.
    All Rights Reserved.

\************************************************************************************************************/

#ifndef INC_LDEVICETHREAD_H
#define INC_LDEVICETHREAD_H

// -----------------------------------------------------------------------------------------------------------
//                                   I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "LStatus.h"
#include "Liberatus.h"

// -----------------------------------------------------------------------------------------------------------
//                                   U S A G E
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                                   O V E R V I E W
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\
Overview:
This document presents the device thread module which provides an API for submitting commands to the various 
SV2 processing engines, receiving notifications, flushing previous submissions, and synchronizing with the 
completion of commands or system events. The API also allows the CPU to synchronize with the completion of 
commands or system events.
An application creates and assigns device threads to a number of modules of the Liberatus library to achieve 
the desired execution pattern. Parallel execution is achieved using separate device threads while sequential 
execution is achieved through a single device thread.

Description:
A device thread could be associated as the prolonging of a CPU thread to the SV2 embedded scheduler. 
Its purpose is to transport submissions targeting various processing engines of the SV2 device from an 
application CPU thread to the SV2 embedded scheduler. It represents a direct communication path from the 
application to the device processing engines and is central to the Liberatus library asynchronous 
programming model. It also transports notifications from the SV2 embedded scheduler to an application 
CPU thread. The main characteristic of a device thread is that submissions execute in order, and complete 
in order.
A device thread comprises three (3) queue elements: a submission queue, a bypass queue and a notification 
queue. The device thread API provides an interface for each queue that operates asynchronously in parallel.
Each queue of a device thread is usually associated with a single CPU thread. The fundamental rule states 
that a CPU thread may issue requests through multiple device threads but multiple CPU threads may not 
issue requests to the same queue of a device thread unless those CPU threads are properly synchronized by 
the application.
A total of three (3) CPU threads can be active in a given device thread object simultaneously without blocking 
each other: a first thread may be in the WaitCompletionTag API, a second in the WaitNotification API, and a 
third in any of the remaining entry points of the API. An exception to this rule is that the Destroy API 
blocks all threads until there are no other active CPU threads in the associated device thread. 
The multi-threading blocking is a per device thread object, not a per device thread API.
A device thread may be shared by multiple Liberatus modules. Because those modules aren't using any hidden 
internal CPU thread, the application is in full control of the system CPU threads to comply with the previous 
rule.
A device thread concept has parallel execution properties similar to a CPU thread. An application obtains 
sequential execution of a sequence of instructions through a CPU thread and obtains parallel execution using 
multiple CPU threads. Similarly, an application obtains sequential execution of a sequence of submissions 
through a device thread and obtains parallel execution using multiple device threads.
An application (an operating system process) is allowed to create and use a maximum combination of 256 public 
and private device threads.
Applications can create a maximum of 256 public device threads that can be accessed by any application running 
on the system. The owner of a public device thread (the application that creates it) is the only one to have 
unrestricted access to it. The remaining applications of the system have restricted access allowing them to 
synchronize with it only.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------
#define     LDEVICETHREAD_MAX_PUBLIC_PER_DEVICE     256
#define     LDEVICETHREAD_MAX_PER_PROCESS           256
#define     LDEVICETHREAD_DEFAULT_LUID              0xFFFFFFFFFFFFFFFF
#define     LDEVICETHREAD_DEFAULT_TAG               0xFFFFFFFFFFFFFFFF
#define     LDEVICETHREAD_NO_TIMEOUT                0xFFFFFFFF
#define     LDEVICETHREAD_SUBMISSION_NO_TIMEOUT     0xFFFF
#define     LDEVICETHREAD_DEPENDENCY_NO_TIMEOUT     0xFFFFFFFF
#define     LDEVICETHREAD_TICK_COUNTER_FREQUENCY    27000000
#define     LDEVICETHREAD_TICK_COUNTER_SIZE_IN_BIT  64

typedef struct LDEVICETHREAD_OBJECT* LDeviceThread_Handle;

/************************************************************************************************************\

Enum:           LDeviceThread_AttributeType

Description:    Enumeration of attribute types.

Comments:       None.

\************************************************************************************************************/
typedef enum {
    LDeviceThread_AttributeType_CREATE  = 0,            // Attributes structure for the Create method. 
    LDeviceThread_AttributeType_FORCE32 = 0x7FFFFFFF,   // Dummy value to force to use 32-bits. 
} LDeviceThread_AttributeType;


/************************************************************************************************************\

Enum:           LDeviceThread_NotificationType

Description:    Enumeration of notification types.

Comments:       - The various notification types share a common Notification structure. The interpretation of 
                  the content is type specific.

\************************************************************************************************************/
typedef enum 
{
    LDeviceThread_NotificationType_HEADER           = 0,    // Header passed to the WaitNotification method.
    LDeviceThread_NotificationType_USER             = 1,    // User specific notification. Can be used with
                                                            //  SendNotification and received from WaitNotification.
    LDeviceThread_NotificationType_CHECKPOINT       = 2,    // Device checkpoint notification. Can be used with
                                                            //  SendNotification. It's never received.
    LDeviceThread_NotificationType_FLUSH            = 3,    // Flush operation notification. Can be received from
                                                            //  WaitNotification. It can't be sent.
    LDeviceThread_NotificationType_ERROR            = 4,    // Generic error notification. Can be received from
                                                            //  WaitNotification. It can't be sent
    LDeviceThread_NotificationType_DEPENDENCY_MISS  = 5,    // Dependency miss notification. Notification when
                                                            // a synchronization attempt for a submission is
                                                            // missed. Can be received from SyncWithSubmission,
                                                            // SyncWithCounter, SyncWithCounterTransition,
                                                            // SyncWithTimeMark, SyncWithCheckpoint. It can't be sent.

    LDeviceThread_NotificationType_FORCE32      = 0x7FFFFFFF,   // Dummy value to force to use 32-bits. 

} LDeviceThread_NotificationType;


/************************************************************************************************************\

Enum:           LDeviceThread_NotificationError

Description:    Enumeration of notification errors.

Comments:       - The various notification errors share a common Notification structure. The interpretation of
                  the content is type specific.

\************************************************************************************************************/
typedef enum
{
    LDeviceThread_NotificationError_INVALID_COMMAND_ID =  66560 + 0,        // Error reported when a submission has
                                                                            // an invalid command id.
    LDeviceThread_NotificationError_INVALID_ENGINE_ID,                      // Error reported when a submission has
                                                                            // an invalid engine id.
    LDeviceThread_NotificationError_INVALID_COUNTER_ID,                     // Error reported when a submission has
                                                                            // an invalid counter id.
    LDeviceThread_NotificationError_INVALID_EXTENSION_COMMAND_ID,           // Error reported when a submission has
                                                                            // an invalid extension command id.
    LDeviceThread_NotificationError_INVALID_SIZE,                           // Error reported when a submission has
                                                                            // an invalid size.
    LDeviceThread_NotificationError_COMMAND_FLUSHED,                        // Error reported when a submission has
                                                                            // time out.
    LDeviceThread_NotificationError_DOUBLE_FAULT                            // Error reported when a device thread
                                                                            // has a second time out.
} LDeviceThread_NotificationError;



/************************************************************************************************************\

Enum:           LDeviceThread_NotificationFlags 

Description:    Enumeration of notification types.

Comments:       - These flags are returned in the oHeader.uFlags member of the Notification structure. 
                  The flags correspond to independent bit if a bitmask and can be set simultaneously.

\************************************************************************************************************/
typedef enum 
{
    LDeviceThread_NotificationFlags_DATA_VALID   = 0x01,    // The uData member of a notification is valid.
    LDeviceThread_NotificationFlags_ENGINE_VALID = 0x02,    // The EngId member of a notification is valid.
    LDeviceThread_NotificationFlags_OVERFLOW     = 0x80,    // An overflow of the notification occurred and 
                                                            //  previous notifications were missed.
} LDeviceThread_NotificationFlags;


/************************************************************************************************************\

Structure:      LDeviceThread_CreateAttributes

Description:    Device thread creation attributes.

Comments:       LDeviceThread_AttributeType_CREATE.

\************************************************************************************************************/
typedef struct tagLDeviceThread_CreateAttributes
{
    LDeviceThread_AttributeType eType;                      // Shall be LDeviceThread_AttributeType_CREATE
    MBOOL32                     bPrivate;                   // Creation of a private or public device thread.
    MBOOL32                     bExplicitSynchronization;   // Explicit synchronization
    MBOOL32                     bAltCmdQueueProcessing;     // Alternate command queue processing.
    MCHAR8                      szName[8];                  // Device thread name
} LDeviceThread_CreateAttributes;

/************************************************************************************************************\

Structure:      LDeviceThread_Notification

Description:    Device thread notification.

Comments:       - A number of notification data bytes may follow the notification structure. 
                - The value oHeader.uSize shall be a multiple of 8 bytes.

\************************************************************************************************************/
typedef struct tagLDeviceThread_Notification
{
    LDeviceThread_NotificationType eType;   // One of the NotificationType_* enum.

    struct tagLDeviceThread_NotificationHeader
    {
        MUINT8                     uiSize;   // Total size of the notification header with extra data following
                                             //  the header structure. This total size doesn't include the
                                             //  initial eType member.
        MUINT8                     uiRsvd;   // Reserved member.
        MUINT8                     uiFlags;  // A combination of NotificationFlags_* values.
        MUINT8                     uiEngId;  // Engine identifier associated with the notification.
        MINT32                     iError;   // Error code.
        MUINT64                    uiData;   // Single data associated with the notification.
        MUINT64                    uiTag;    // Tag associated with the notification.
    } oHeader;
} LDeviceThread_Notification;


// -----------------------------------------------------------------------------------------------------------
//                           G L O B A L   V A R I A B L E   R E F E R E N C E S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                         G L O B A L   F U N C T I O N   D E C L A R A T I O N S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                      I N L I N E S   A N D   T E M P L A T E   D E F I N I T I O N S
// -----------------------------------------------------------------------------------------------------------
#if defined (__cplusplus)
extern "C" {
#endif

/************************************************************************************************************\

Function:       LDeviceThread_Create

Description:    Creates a device thread object.

Parameters:     IN  hDev                    Handle of the device for which to create a device thread.
                IN  poAttributes            Pointer to the attribute type member of a CreateAttributes 
                                             structure describing the attributes of the device thread to 
                                             create. NULL indicates the use of the default attributes.
                OUT phDT                    Pointer to receive the handle of the device thread created.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_OUT_OF_RESOURCES    No more device threads available.

Comments:       - Creates a public or private device thread object for a given device. Without creation 
                  attributes, this function creates a private device thread object.
                - A private device thread is invisible to other applications running on the system. 
                  This should be the most common type of device thread used by applications. A public device 
                  thread is visible to other applications running on the system for synchronization purposes. 
                  As there's a limited amount of public device thread resources in a system, applications 
                  should carefully plan their use. The application that creates a device thread,
                  public or private, is the only one that can issue submissions or receive notifications 
                  through it.
                - The LStatus_OUT_OF_RESOURCES error is returned if the maximum number of device thread
                  objects allowed for an application is reached.


\************************************************************************************************************/
LAPI LStatus LDeviceThread_Create(
                LDevice_Handle                   hDev,
                LDeviceThread_AttributeType*     poAttributes,
                LDeviceThread_Handle*            phDT);


/************************************************************************************************************\

Function:       LDeviceThread_AddRef

Description:    Adds a reference on a device thread object.

Parameters:     IN  hDT                     Handle of a device thread object.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - The AddRef / UnRef methods provide the necessary reference counting mechanism for sharing 
                  an object among a number of modules and making sure the object is destroyed when no 
                  reference remains.
                - This method adds a reference on a device thread object. The object exists until the number 
                  of references on it reaches zero (0). The methods UnRef and Destroy decrement the reference 
                  count.
                - Creating an object sets an initial reference on the object.

\************************************************************************************************************/
LAPI LStatus LDeviceThread_AddRef(
                LDeviceThread_Handle    hDT);


/************************************************************************************************************\

Function:       LDeviceThread_GetRefCount

Description:    Gets the number of references on a device thread object.

Parameters:     IN  hDT                     Handle of a device thread object.
                OUT MUINT32                 Pointer to an unsigned integer receiving the number
                                             of references on this object.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.

Comments:

\************************************************************************************************************/
LAPI LStatus LDeviceThread_GetRefCount(
                LDeviceThread_Handle    hDT,
                MUINT32*                puiRefCount);


/************************************************************************************************************\

Function:       LDeviceThread_UnRef

Description:    Removes a reference on a device thread object.

Parameters:     IN  hDT                     Handle of a device thread object.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - The AddRef / UnRef methods provide the necessary reference counting mechanism for sharing 
                  an object among a number of modules and making sure the object is destroyed when no 
                  reference remains.
                - If multiple references remain on the device thread object, this method removes a reference. 
                  Otherwise, it performs a Destroy operation on the device thread object.
                - An application can use either of the UnRef or Destroy methods to destroy a device thread 
                  object. We recommend matching an UnRef operation with an AddRef operation, and a Destroy 
                  operation with a Create operation 

\************************************************************************************************************/
LAPI LStatus LDeviceThread_UnRef(
                LDeviceThread_Handle    hDT);

/************************************************************************************************************\

Function:       LDeviceThread_Destroy

Description:    Destroys a device thread object.

Parameters:     IN  hDT                     Handle of a device thread object.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - If a single reference remains on the device thread object, this method destroys the object 
                  and the associated system resources. Otherwise, it performs an UnRef operation on the device 
                  thread object.
                - Remaining submissions and notifications in the device thread queues are flushed when the 
                  destroy operation takes place. An application should synchronize with the completion of the 
                  last submission to a device thread before calling this method.
                - The identifier associated with a public device thread remains the property of the 
                  application and can be reused when creating another public device thread.

\************************************************************************************************************/
LAPI LStatus LDeviceThread_Destroy(
                LDeviceThread_Handle    hDT);


/************************************************************************************************************\

Function:       LDeviceThread_GetDefaultAttributes

Description:    Gets the default values of a given type of attributes.

Parameters:     IN  hDT                     Handle of a device thread object.
                OUT peAttributes            Pointer to the attribute type member of a structure of attributes. 
                                            On input, the type member describes the type of the structure. 
                                            On output, the remaining members of the structure are filled with 
                                            the default value of the attributes.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This function provides a mechanism for retrieving an attribute structure filled with default 
                  values. The application is concerned only with the subset of the members it's interested in.
                - Only one type of attribute at a time can be retrieved. Only AttributeType_CREATE is 
                  currently supported.

\************************************************************************************************************/
LAPI LStatus LDeviceThread_GetDefaultAttributes(
                LDeviceThread_Handle            hDT,
                LDeviceThread_AttributeType*    peAttributes );


/************************************************************************************************************\

Function:       LDeviceThread_GetAttributes

Description:    Gets the current values of a given type of attributes.

Parameters:     IN  hDT                     Handle of a device thread object.
                OUT peAttributes            Pointer to the attribute type member of a structure of attributes. 
                                            On input, the type member describes the type of the structure. 
                                            On output, the remaining members of the structure are filled with 
                                            the current value of the attributes.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This function provides a mechanism for retrieving an attribute structure filled with the 
                  current values.
                - Only one type of attribute at a time can be retrieved. Only AttributeType_CREATE is 
                  currently supported.

\************************************************************************************************************/
LAPI LStatus LDeviceThread_GetAttributes(
                LDeviceThread_Handle             hDT,
                LDeviceThread_AttributeType*     peAttributes);

/************************************************************************************************************\

Function:       LDeviceThread_GetLUID

Description:    Gets a locally unique identifier of a device thread.

Parameters:     IN  hDT                     Handle of a device thread object.
                OUT puiLUID                 Pointer to an unsigned integer receiving the associated unique
                                             identifier.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This method returns the unique identifier associated with a device thread that can be used 
                  by any application running on the system for naming the associated device thread.
                - The LUID is used as a parameter to the methods WaitCompletionTag and SyncWithSubmission.

\************************************************************************************************************/
LAPI LStatus LDeviceThread_GetLUID(
                LDeviceThread_Handle    hDT,
                MUINT64*                puiLUID);


/************************************************************************************************************\

Function:       LDeviceThread_GetSubmissionTag

Description:    Gets the tag of the last submission to a device thread.

Parameters:     IN  hDT                     Handle of a device thread object.
                OUT puiLastSubmissionTag    Pointer to an unsigned integer to receive the value of the last
                                             submission tag.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This method is used to retrieve the tag of the last submission issued through the device 
                  thread. The submission may originate from the application or a Liberatus module that was 
                  assigned the device thread.
                - A device thread maintains a pseudo-timeline of the submissions flowing through its 
                  submission queue. A submission tag is associated with each submission received by the device 
                  thread. At the application level, only the SyncWith methods correspond to submissions to the 
                  device thread. The remaining submissions flowing through the device thread come from the
                  Liberatus modules. Once a submission is completed, its submission tag becomes a completion 
                  tag.

\************************************************************************************************************/
LAPI LStatus LDeviceThread_GetSubmissionTag(
                LDeviceThread_Handle    hDT,
                MUINT64*                puiLastSubmissionTag);


/************************************************************************************************************\

Function:       LDeviceThread_GetCompletionTag

Description:    Gets the tag of the last completion from a device thread.

Parameters:     IN  hDT                     Handle of a device thread object.
                OUT puiLastCompletionTag    Pointer to an unsigned integer to receive the value of the
                                             last completion tag.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This method is used to retrieve the tag of the last submission completed from a device 
                  thread. The submission may originate from the application or a Liberatus module that was 
                  assigned the device thread.

\************************************************************************************************************/
LAPI LStatus LDeviceThread_GetCompletionTag(
                LDeviceThread_Handle    hDT,
                MUINT64*                puiLastCompletionTag);

/************************************************************************************************************\

Function:       LDeviceThread_WaitCompletionTag

Description:    Waits for the completion of a submission in a given device thread.

Parameters:     IN  hDT                     Handle of a device thread object.
                IN  uiDeviceThreadLUID      Unique identifier of a device thread.
                IN  uiCompletionTag         Completion tag.
                IN  uiTimeout_ms            Timeout in milliseconds.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_TIMEOUT             Function exited after a timeout.

Comments:       - This method is used to synchronize a CPU thread with a given submission for a given device 
                  thread. The CPU thread sleeps until the submission associated with the given tag in the 
                  given device thread is complete or a timeout occurs. The waiting terminates when the given 
                  tag becomes smaller than or equal to the last completion tag of the given device thread or 
                  a timeout occurs.
                - The value ~0 (0xffffffffffffffff) for the uCompletionTag parameter indicates to wait for 
                  the last submission tag. This operation results in waiting for all the previous submissions 
                  to complete.
                - The value ~0 (0xffffffffffffffff) for the uDeviceThreadLUID parameter indicates the device 
                  thread associated with the hDT handle.
                - A timeout of 0 verifies the completion status immediately without sleeping. The error 
                  LStatus_TIMEOUT is returned if the submission hasn't completed.

\************************************************************************************************************/
LAPI LStatus LDeviceThread_WaitCompletionTag(
                LDeviceThread_Handle    hDT,
                MUINT64                 uiDeviceThreadLUID,
                MUINT64                 uiCompletionTag,
                MUINT32                 uiTimeout_ms);


/************************************************************************************************************\

Function:       LDeviceThread_SyncWithSubmission

Description:    Synchronizes the device thread with the completion of a submission in a given device thread.

Parameters:     IN  hDT                     Handle of a device thread object.
                IN  uiDeviceThreadLUID      Unique identifier of a device thread.
                IN  uiCompletionTag         Completion tag.
                IN  bMarkTime               Time mark, in milliseconds, of the completion time of the 
                                             synchronization.
                IN  bNotifyMiss             Notification of the submission completion before attempting the 
                                             synchronization.
                IN  uTimeout_ms             Timeout, in milliseconds.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_TIMEOUT             Function exited after a timeout.

Comments:       - This method issues a synchronization command through a device thread submission queue to 
                  synchronize with the completion of a submission associated with the given tag in the given 
                  device thread.
                - The timeout of the operation relates to the ability to call the CPU thread to push the 
                  synchronization command to the submission queue of the device thread. The waiting operation 
                  for the completion of the given submission has no timeout and will wait forever if the given 
                  submission never completes. The application should, in this scenario, flush the 
                  synchronization command from the submission queue to terminate the wait.
                - When bMarkTime is TRUE, the device thread time mark, in milliseconds, is set to the time 
                  when the synchronization command completed. The resulting time mark can be used in a 
                  SyncWithTimeMark operation.
                - When bNotifyMiss is TRUE, the embedded scheduler issues a notification to the device thread 
                  notification queue if the given submission from the given device thread already completed 
                  when the synchronization was first attempted.

\************************************************************************************************************/
LAPI LStatus LDeviceThread_SyncWithSubmission(
                LDeviceThread_Handle    hDT,
                MUINT64                 uiDeviceThreadLUID,
                MUINT64                 uiCompletionTag,
                MBOOL32                 bMarkTime,
                MBOOL32                 bNotifyMiss,
                MUINT32                 uiTimeout_ms );

/************************************************************************************************************\

Function:       LDeviceThread_SyncWithCounter

Description:    Synchronizes the device thread with the completion of a counter.

Parameters:     IN  hDT                     Handle of a device thread object.
                IN  uiCounterLUID           Unique identifier of a counter.
                IN  uiCompletionTag         Completion tag.
                IN  bMarkTime               Time mark, in milliseconds, of the completion time of the 
                                             synchronization
                IN  bNotifyMiss             Notification of the submission completion before attempting 
                                             the synchronization.
                IN  uiTimeout_ms            Timeout, in milliseconds.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_TIMEOUT             Function exited after a timeout.

Comments:       - This method issues a synchronization command through a device thread submission queue to 
                  synchronize with the completion of an sub-picture event associated with the given tag from 
                  a given counter.
                - The timeout of the operation relates to the ability to call the CPU thread to push the 
                  synchronization command to the submission queue of the device thread. The waiting operation 
                  for the completion of the given event has no timeout and will wait forever if the given 
                  event never completes. The application should, in this scenario, flush the synchronization 
                  command from the submission queue to terminate the waiting.
                - When bMarkTime is TRUE, the device thread time mark, in milliseconds, is set to the time 
                  when the synchronization command completed. The resulting time mark can be used in a 
                  SyncWithTimeMark operation.
                - When bNotifyMiss is TRUE, the embedded scheduler issues a notification to the device thread 
                  notification queue if the given event from the given counter already completed when the 
                  synchronization was first attempted.
                - The completion tag of the LVideoIn, LVideoOut, and the LH264VideoCodec decoder modules 
                  is in units of 1/8 of a picture. The counter LUID should be retrieved from one of these 
                  modules. No counter is currently available for synchronization for the LH264VideoCodec 
                  encoder module.

\************************************************************************************************************/
LAPI LStatus LDeviceThread_SyncWithCounter(
                LDeviceThread_Handle    hDT,
                MUINT64                 uiCounterLUID,
                MUINT64                 uiCompletionTag,
                MBOOL32                 bMarkTime,
                MBOOL32                 bNotifyMiss,
                MUINT32                 uiTimeout_ms);


/************************************************************************************************************\

Function:       LDeviceThread_SyncWithCounterTransition

Description:    Synchronizes the device thread with the completion of a number of counter transitions.

Parameters:     IN  hDT                     Handle of a device thread object.
                IN  uiCounterLUID           Unique identifier of a counter.
                IN  uiNextTransition        Next transition.
                IN  bMarkTime               Mark, in milliseconds, of the completion time of the 
                                             synchronization
                IN  bNotifyMiss             Notification of the submission completion before attempting the 
                                             synchronization.
                IN  uTimeout_ms             Timeout, in milliseconds.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_TIMEOUT             Function exited after a timeout.

Comments:       - This method issues a synchronization command through a device thread submission queue to 
                  synchronize with the next Nth completion of a full picture event from a given counter.
                - The timeout of the operation relates to the ability to call the CPU thread to push the 
                  synchronization command to the submission queue of the device thread. The waiting operation 
                  for the completion of the given event has no timeout and will wait forever if the given 
                  event never completes. The application should, in this scenario, flush the synchronization 
                  command from the submission queue to terminate the waiting.
                - When bMarkTime is TRUE, the device thread time mark, in milliseconds, is set to the time 
                  when the synchronization command completed. The resulting time mark can be used in a 
                  SyncWithTimeMark operation.
                - When bNotifyMiss is TRUE, the embedded scheduler issues a notification to the device thread 
                  notification queue if the given event from the given counter already completed when the 
                  synchronization was first attempted.
                - The counter of the LVideoIn, LVideoOut, and the LH264VideoCodec decoder modules is in units 
                  of 1/8 of a picture. The counter LUID should be retrieved from one of these modules. 
                  No counter is currently available for synchronization for the LH264VideoCodec encoder 
                  module.
                - The synchronization operation samples the given counter when the operation is first 
                  attempted and then monitors the Nth full picture transition.

\************************************************************************************************************/
LAPI LStatus LDeviceThread_SyncWithCounterTransition(
                LDeviceThread_Handle    hDT,
                MUINT64                 uiCounterLUID,
                MUINT32                 uiNextTransition,
                MBOOL32                 bMarkTime,
                MBOOL32                 bNotifyMiss,
                MUINT32                 uiTimeout_ms );


/************************************************************************************************************\

Function:       LDeviceThread_SyncWithTimeMark

Description:    Synchronizes the device thread with the completion of a time delay.

Parameters:     IN  hDT                     Handle of a device thread object.
                IN  uiTimeDelay_ms          Time delay.
                IN  bMarkTime               Time mark, in milliseconds, of the completion time of the 
                                             synchronization.
                IN  bNotifyMiss             Notification of the submission completion before attempting the 
                                             synchronization.
                IN  uiTimeout_ms            Timeout, in milliseconds.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_TIMEOUT             Function exited after a timeout.

Comments:       - This method issues a synchronization command through a device thread submission queue to 
                  synchronize with the completion of a time delay from the given device thread time-mark.
                - The timeout of the operation relates to the ability to call the CPU thread to push the 
                  synchronization command to the submission queue of the device thread. The waiting operation 
                  for the completion of the given event has no timeout and will wait almost forever if the 
                  given time-delay never completes. The application should, in this scenario, flush the 
                  synchronization command from the submission queue to terminate the waiting.
                - When bMarkTime is TRUE, the device thread time mark, in milliseconds, is set to the time 
                  when the synchronization command completed. The resulting time mark can be used in a 
                  SyncWithTimeMark operation.
                - When bNotifyMiss is TRUE, the embedded scheduler issues a notification to the device thread 
                  notification queue if the given time delay from the device thread time mark already 
                  completed when the synchronization was first attempted.

\************************************************************************************************************/
LAPI LStatus LDeviceThread_SyncWithTimeMark(
                LDeviceThread_Handle    hDT,
                MUINT64                 uiTimeDelay_ms,
                MBOOL32                 bMarkTime,
                MBOOL32                 bNotifyMiss,
                MUINT32                 uiTimeout_ms);


/************************************************************************************************************\

Function:       LStatus LDeviceThread_SetTimeMark

Description:    

Parameters:     IN  hDT                     Handle of a device thread object.
                IN  uiTimeDelay_ms          Time delay.
                IN  uiTimeout_ms            Timeout, in milliseconds.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_TIMEOUT             Function exited after a timeout.

Comments:       

\************************************************************************************************************/
LAPI LStatus LDeviceThread_SetTimeMark(
                LDeviceThread_Handle    hDT,
                MUINT64                 uiTimeMark,
                MUINT32                 uiTimeout_ms);
                
/************************************************************************************************************\

Function:       LDeviceThread_SyncWithCheckpoint

Description:    Synchronizes the device thread with the completion of a device checkpoint.

Parameters:     IN  hDT                     Handle of a device thread object.
                IN  uiDeviceThreadLUID      Unique identifier of a device thread to wait for.
                IN  uiCheckpointLUID        Unique identifier of a checkpoint to wait for.
                IN  uiCompletionTag         Completion tag to wait for.
                IN  bMarkTime               Time mark, in milliseconds, of the completion time of the 
                                             synchronization.
                IN  bNotifyMiss             Notification of the submission completion before attempting the 
                                             synchronization.
                IN  uiTimeout_ms            Timeout, in milliseconds.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_TIMEOUT             Function exited after a timeout.

Comments:       - This method issues a synchronization command through a device thread submission queue to 
                  synchronize with the completion of checkpoint event associated with the given tag from a 
                  given checkpoint.
                - The timeout of the operation relates to the ability to call the CPU thread to push the 
                  synchronization command to the submission queue of the device thread. The waiting operation 
                  for the completion of the given event has no timeout and will wait forever if the given 
                  event never completes. The application should. in this scenario. flush the synchronization 
                  command from the submission queue to terminate the waiting.
                - When bMarkTime is TRUE, the device thread time-mark, in milliseconds, is set to the time 
                  when the synchronization command completed. The resulting time-mark can be used in a 
                  SyncWithTimeMark operation.
                - When bNotifyMiss is TRUE, the embedded scheduler issues a notification to the device thread 
                  notification queue if the given event from the given counter already completed when the 
                  synchronization was first attempted.
                - The checkpoint LUID should be retrieved from the Liberatus library.

\************************************************************************************************************/
LAPI LStatus LDeviceThread_SyncWithCheckpoint(
                LDeviceThread_Handle    hDT,
                MUINT64                 uiCheckpointLUID,
                MUINT64                 uiCompletionTag,
                MBOOL32                 bMarkTime,
                MBOOL32                 bNotifyMiss,
                MUINT32                 uiTimeout_ms );


/************************************************************************************************************\

Function:       LDeviceThread_SendNotification

Description:    Sends a notification to the submission queue of a device thread.

Parameters:     IN  hDT                     Handle of a device thread object.
                IN  poNotification          Pointer to the notification type member of a notification 
                                             structure. 
                IN  uiTimeout_ms            Timeout, in milliseconds.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_TIMEOUT             Function exited after a timeout.

Comments:       - This method issues a notification of type NotificationType_USER or 
                  NotificationType_CHECKPOINT to the device thread submission queue. When processing the 
                  notification, the scheduler issues it to the device thread notification queue where the 
                  application can recover the notification properly synchronized with the device thread 
                  other submissions.
                - The timeout of the operation relates to the ability to call the CPU thread to push the 
                  notification command to the submission queue of the device thread. While processing the 
                  notification, the embedded scheduler will stall the device thread until there's enough 
                  space in the associated notification queue to issue the notification. There's no timeout to 
                  the embedded scheduler attempt to issue the notification in the notification queue. 
                  The application should regularly monitor for available notifications.

\************************************************************************************************************/
LAPI LStatus LDeviceThread_SendNotification(
                LDeviceThread_Handle                hDT,
                LDeviceThread_NotificationType*     poNotification,
                MUINT32                             uiTimeout_ms);


/************************************************************************************************************\

Function:       LDeviceThread_WaitNotification

Description:    Waits for a notification from the notification queue of a device thread.

Parameters:     IN  hDT                     Handle of a device thread object.
                OUT poNotification          Pointer to the notification type member of a notification structure. 
                                            On input, the type member describes the type of the structure and 
                                            the available size. 
                                            On output, all the members of the structure are filled with the 
                                            information of the notification and a new type is returned.
                IN  uiTimeout_ms            Timeout, in milliseconds.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_TIMEOUT             Function exited after a timeout.

Comments:       - This method is used to synchronize a CPU thread with the availability of a notification 
                  from a device thread. The CPU thread sleeps until a notification becomes available or a 
                  timeout occurs.
                - The application provides a notification structure on input where the type member shall be 
                  NotificationType_HEADER. When a notification becomes available, the corresponding 
                  notification header structure is filled and the type and size members are updated to match 
                  the current notification. Only the header of the current notification is retrieved from the 
                  notification queue when this method is called. It gives the application the opportunity to 
                  inspect the notification type and the size of the notification before it retrieves the 
                  associated data using the GetNotificationData method. 
                - An application shall monitor its notification queue regularly to prevent them from 
                  becoming full in which case it may miss notifications from the scheduler. In the event of 
                  a notification queue full, the scheduler would tag the next notification with the flag 
                  NotificationFlags_OVERFLOW to indicate that some previous notifications were missed.
                - A timeout of 0 verifies the availability of a notification immediately without sleeping. 
                  The error LStatus_TIMEOUT is returned if a notification isn't available.
                - Calling this method will flush pending data associated with a previous notification that 
                  wasn't retrieved by calling GetNotificationData.
                - The application shall synchronize the access to a notification queue such that a CPU thread 
                  retrieving a notification will get both the header and the data before another CPU thread is 
                  allowed to access the notification queue.

\************************************************************************************************************/
LAPI LStatus LDeviceThread_WaitNotification(
                LDeviceThread_Handle                hDT,
                LDeviceThread_NotificationType*     poNotification,
                MUINT32                             uiTimeout_ms);


/************************************************************************************************************\

Function:       LDeviceThread_GetNotificationData

Description:    Gets the data associated with the current notification.

Parameters:     IN  hDT                     Handle of a device thread object.
                IN  uiSize                  Size in bytes of the output data the buffer.
                IN  puiData                 Pointer to a buffer of bytes to receive the data.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This method retrieves the data associated with the current notification from the notification 
                  queue.
                - An application isn't required to retrieve all or any of the data associated with a 
                  notification. A call to WaitNotification flushes any pending data from the current 
                  notification.

\************************************************************************************************************/
LAPI LStatus LDeviceThread_GetNotificationData(
                LDeviceThread_Handle     hDT,
                MUINT32                  uiSize,
                MUINT8*                  puiData);

/************************************************************************************************************\

Function:       LDeviceThread_Flush

Description:    Flushes some previous submissions from a device thread.

Parameters:     IN  hDT                     Handle of a device thread object.
                IN  uiSubmissionTag

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This method removes submissions from the device thread as if completed, starting with the 
                  one at the front of the queue, up to, but excluding, the one associated with the given 
                  submission tag. The submissions that were scheduled to an engine that did not complete are 
                  not removed from the engine queue by this operation. The flush operation proceeds when the 
                  last of these submissions already queued to the engines completes. While flushing, 
                  synchronization commands are processed as if their dependency was resolved, while engine 
                  submissions are discarded as if they completed normally.
                - An application should be aware that flushing submissions from a device thread may cause 
                  visual artifacts, as the submissions are assumed as having completed, although no actual  
                  content was generated.
                - This API is blocking and won't return to the caller until the embedded scheduler processed 
                  the request. A NotificationType_FLUSH notification will be available from the notification 
                  queue to indicate the range of submission tags that were flushed from the device thread 
                  submission queue.

\************************************************************************************************************/
LAPI LStatus LDeviceThread_Flush(
                LDeviceThread_Handle     hDT,
                MUINT64                  uiSubmissionTag);

/************************************************************************************************************\

Function:       LDeviceThread_SetTimeOutValues

Description:    Set the current time out values.

Parameters:     IN  hDT                     Handle of a device thread object.
                IN  uiSubmissionTimeout_ms  Submission timeout in milliseconds.
                IN  uiDependencyTimeout_ms  Dependency command timeout in milliseconds.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This method provides a mechanism to specify time out values. By default, the values are
                  set to LDEVICETHREAD_SUBMISSION_NO_TIMEOUT and LDEVICETHREAD_DEPENDENCY_NO_TIMEOUT.
                - If a command execution has not completed before the submission time out value, the scheduler
                  resets the engine and issues a NotificationError_COMMAND_FLUSHED notification to the device
                  thread notification queue.
                - If a dependency has not completed before the dependency time out value, the scheduler issues
                  a NotificationError_COMMAND_FLUSHED notification to the device thread notification queue.
                  The caller must flush the dependency command before the scheduler resumes the processing of
                  the device thread.
                - If the scheduler detects two time out on the same device thread (dependency or submission),
                  the scheduler issues a NotificationError_DOUBLE_FAULT notification to the device thread
                  notification queue and suspends this device thread. A suspended device thread shall be
                  destroyed.

\************************************************************************************************************/
LAPI LStatus LDeviceThread_SetTimeOutValues(
                LDeviceThread_Handle    hDT,
                MUINT16                 uiSubmissionTimeout_ms,
                MUINT32                 uiDependencyTimeout_ms);

/************************************************************************************************************\

Function:       LDeviceThread_GetTimeOutValues

Description:    Gets the current time out values.

Parameters:     IN  hDT                     Handle of a device thread object.
                OUT puiSubmissionTimeout_ms Submission timeout in milliseconds.
                OUT puiDependencyTimeout_ms Dependency command timeout in milliseconds.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This function provides a mechanism for retrieving the current time out values.

\************************************************************************************************************/
LAPI LStatus LDeviceThread_GetTimeOutValues(
                LDeviceThread_Handle    hDT,
                MUINT16*                puiSubmissionTimeout_ms,
                MUINT32*                puiDependencyTimeout_ms);

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Private Interface
//////////////////////////////////////////////////////////////////////////////////////////////////////////////
#define     LDEVICETHREAD_REFERENCE_DATA_PACK(a, b) ((a & 0xFF) << 8 | (b & 0xFF))

/************************************************************************************************************\

Function:       LPDeviceThread_GetJHandle

Description:    .

Parameters:     

Return Value:   .

Comments:       None.

\************************************************************************************************************/
MHANDLE LPDeviceThread_GetJHandle(LDeviceThread_Handle hDeviceThread);

/************************************************************************************************************\

Function:       LDeviceThread_SetReferenceData

Description:    .

Parameters:

Return Value:   .

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LDeviceThread_SetReferenceData(LDeviceThread_Handle hDT,
                MUINT16              uiReferenceData);

#if defined (__cplusplus)
}
#endif

#endif  // #ifndef INC_LDEVICETHREAD_H
