/************************************************************************************************************\

Module Name:    LBoard.h

Description:    The LBoard module of the API provides the programming interface for the product.

References:     LBoard.doc Revision 1.13.

    Copyright (c) 2016, Matrox Graphics Inc.
    All Rights Reserved.

\************************************************************************************************************/

#ifndef INC_LBOARD_H
#define INC_LBOARD_H

// -----------------------------------------------------------------------------------------------------------
//                                   I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------
#include "Liberatus.h"

typedef struct LEVENT_QUEUE_OBJECT*         LEventQueue_Handle;
typedef struct LI2C_OBJECT*                 LI2c_Handle;
typedef struct LIO_OBJECT*                  LIo_Handle;
typedef struct LKSV_OBJECT*                 LKsv_Handle;

// -----------------------------------------------------------------------------------------------------------
//                                   U S A G E
// -----------------------------------------------------------------------------------------------------------

/*  
    // Find monitors on the primary device and apply the preferred mode on it.

    #include "Liberatus.h"
    #include "LBuffer.h"
    #include "LVideoOut.h"
    #include "LBoard.h"

    int main()
    {
        MUINT32                 uiConnectorCount    = 0;
        MUINT32                 uiIndex             = 0;
        LDevice_Handle          hDevice             = MNULL;
        LBoard_Info             oBoardInfo          = {0,};
        LBuffer_VideoAttributes oVideoAttribute     = {0,};

        Liberatus_Load();

        hDevice = Liberatus_GetDevice(0);           // Primary device handle

        oBoardInfo.uiSize = sizeof(oBoardInfo); 
        LBoard_GetBoardInformation(hDevice, &oBoardInfo);

        LBoard_DetectAllMonitors(hDevice);

        LBoard_GetConnectorCount(hDevice, &uiConnectorCount);

        for (uiIndex = 0; uiIndex < uiConnectorCount; uiIndex++ )
        {
            // Obtains the connector information.
            LBoard_ConnectorInfo    oConnectorInfo      = {0,};

            oConnectorInfo.uiSize = sizeof(oConnectorInfo);
            LBoard_GetConnectorInfo(hDevice, uiIndex, &oConnectorInfo);

            if ( oConnectorInfo.bVideoOutput )
            {
                // It's a video out connector.
                // Check if a monitor is plugged on it.
                LBoard_MonitorInfo      oMonitorInfo        = {0,};

                oMonitorInfo.uiSize = sizeof(oMonitorInfo);
                LBoard_GetMonitorInfo(hDevice, uiIndex, &oMonitorInfo);

                if ( oMonitorInfo.bIsConnected )
                {
                    // The monitor is connected set it's optimal mode.
                    MUINT32             uiVideoOutIndex = 0;
                    LVideoOut_Handle    hVideoOut       = MNULL;
                    LBuffer_Handle      hBuffer         = MNULL;
                    LVideo_VidParam     oVidParam       = {0,};

                    uiVideoOutIndex = oConnectorInfo.uiVideoOutIndex;

                    LVideoOut_GetHandle(hDevice, uiVideoOutIndex, LAccessMode_READWRITE, &hVideoOut);

                    LVideoOut_GetOptimalVideoParameters(hVideoOut,
                                                        oConnectorInfo.eVideoOutSignalType,
                                                        &oVidParam);

                    oVideoAttribute.eAttributeType  = LBuffer_Type_VIDEO;
                    oVideoAttribute.ePixelFormat    = LPixelFormat_B8G8R8A8;
                    oVideoAttribute.oDim.uiWidth    = oVidParam.uiHActive;
                    oVideoAttribute.oDim.uiHeight   = oVidParam.uiVActive;

                    LBuffer_Create(hDevice,
                                   &oVideoAttribute.eAttributeType,
                                    &hBuffer);

                    LVideoOut_Enable(hVideoOut,
                                     oConnectorInfo.eVideoOutSignalType,
                                     &oVidParam,
                                     hBuffer,
                                     LVideoOut_PowerState_ON);

                    LVideoOut_ReleaseHandle(hVideoOut);
                }        
            }
        }
        Liberatus_UnLoad();

        return 0;
    }
*/

// -----------------------------------------------------------------------------------------------------------
//                                               O V E R V I E W
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

    The Liberatus Board Module (LBoard) explains how to retrieve information about the product and 
    provides the API to communicate with internal or external devices (such as monitors). 

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------  

/************************************************************************************************************\

Enum:           LBoard_ConnectorType

Description:    Enumerates all possible connector names.

Comments:       None.

\************************************************************************************************************/
typedef enum
{
    LBoard_ConnectorType_NONE               = 0x00000000, // None
    LBoard_ConnectorType_VGA                = 0x00000001, // Analog video
    
    LBoard_ConnectorType_DVI                = 0x00000010, // Analog or digital video

    LBoard_ConnectorType_HDMI               = 0x00000020, // Digital audio and video
    LBoard_ConnectorType_HDMI_10            = 0x00000021, // Digital audio and video
    LBoard_ConnectorType_HDMI_11            = 0x00000022, // Digital audio and video
    LBoard_ConnectorType_HDMI_12            = 0x00000023, // Digital audio and video
    LBoard_ConnectorType_HDMI_12A           = 0x00000024, // Digital audio and video
    LBoard_ConnectorType_HDMI_13            = 0x00000025, // Digital audio and video
    LBoard_ConnectorType_HDMI_13A           = 0x00000026, // Digital audio and video
    LBoard_ConnectorType_HDMI_13B           = 0x00000027, // Digital audio and video
    LBoard_ConnectorType_HDMI_13B1          = 0x00000028, // Digital audio and video
    LBoard_ConnectorType_HDMI_13C           = 0x00000029, // Digital audio and video
    LBoard_ConnectorType_HDMI_14            = 0x0000002A, // Digital audio and video
    LBoard_ConnectorType_HDMI_14A           = 0x0000002B, // Digital audio and video
    LBoard_ConnectorType_HDMI_20            = 0x0000002C, // Digital audio and video

    LBoard_ConnectorType_DP                 = 0x00000030, // Digital audio and video
    LBoard_ConnectorType_DP_11              = 0x00000031, // Digital audio and video
    LBoard_ConnectorType_DP_12              = 0x00000032, // Digital audio and video
    LBoard_ConnectorType_DP_13              = 0x00000033, // Digital audio and video

    LBoard_ConnectorType_SDI                = 0x00000040, // Digital audio and video
    LBoard_ConnectorType_HD_SDI             = 0x00000041, // Digital audio and video
    LBoard_ConnectorType_DUAL_HD_SDI        = 0x00000042, // Digital audio and video
    LBoard_ConnectorType_3G_SDI             = 0x00000043, // Digital audio and video
    LBoard_ConnectorType_6G_SDI             = 0x00000044, // Digital audio and video
    LBoard_ConnectorType_12G_SDI            = 0x00000045, // Digital audio and video

    LBoard_ConnectorType_TRS                = 0x00000400, // Analog jack audio

    LBoard_ConnectorType_RJ45               = 0x00000600, // Ethernet

    LBoard_ConnectorType_USB                = 0x00004000, // USB
    LBoard_ConnectorType_USB_1              = 0x00004001, // USB 1.0
    LBoard_ConnectorType_USB_2              = 0x00004002, // USB 2.0
    LBoard_ConnectorType_USB_3              = 0x00004003, // USB 3.0
    LBoard_ConnectorType_USB_31             = 0x00004004, // USB 3.1
    
    LBoard_ConnectorType_SATA               = 0x00005000, // SATA
    LBoard_ConnectorType_SATA_I             = 0x00005001, // SATA-I
    LBoard_ConnectorType_SATA_II            = 0x00005002, // SATA-II
    LBoard_ConnectorType_SATA_III           = 0x00005003, // SATA-III

    LBoard_ConnectorType_OTHER              = 0x0000FFFF, // Connector name other than those given in the
                                                          // previous enumeration.

    LBoard_ConnectorType_FORCE32            = 0x7FFFFFFF, // Dummy value to force enumeration to use 32 bits.
} LBoard_ConnectorType;


/************************************************************************************************************\

Struct:         LBoard_ConnectorInfo

Description:    Connector information.

Comments:       Structure used by the LBoard_GetConnectorInfo service.

\************************************************************************************************************/
typedef struct tagLBoard_ConnectorInfo
{
    MUINT32                 uiSize;                     // Size of LBoard_ConnectorInfo structure.
    LBoard_ConnectorType    eType;                      // Connector name enumeration.
    LSTR64                  szName;                     // Connector name - Null terminated ANSI string.
    MUINT                   bVideoInput         : 1;    // Used to capture video stream.
    MUINT                   bVideoOutput        : 1;    // Used to display video.
    MUINT                   bAudioInput         : 1;    // Used to capture audio stream.
    MUINT                   bAudioOutput        : 1;    // Used to output audio stream.
    MUINT                   bAncInput           : 1;    // Used to capture ancillary stream.
    MUINT                   bAncOutput          : 1;    // Used to output ancillary stream.
    MUINT                   bReserved1          : 26;   // Reserved for future usage.
    MUINT                   bDdc                : 1;    // DDC available to retrieve EDID
    MUINT                   bHpd                : 1;    // Hot plug detect availability.
    MUINT                   bEdid1024Support    : 1;    // 1024 byte EDID supported.
    MUINT                   bReserved2          : 29;   // Reserved for future usage.

    MUINT32                 uiVideoInIndex;             // Index use for LVideoIn_GetHandle.
    MUINT32                 uiVideoOutIndex;            // Index use for LVideoOut_GetHandle.
    MUINT32                 uiAudioInIndex;             // Index use for LAudioIn_GetHandle.
    MUINT32                 uiAudioOutIndex;            // Index use for LAudioOut_GetHandle.
    MUINT32                 uiFpgaPortIndex;            // Index use for LFpgaPort_GetHandle.
    MUINT32                 uiAncInIndex;               // Index use for LAncIn_GetHandle.
    MUINT32                 uiAncOutIndex;              // Index used for LAncOut_GetHandle.
} LBoard_ConnectorInfo;


/************************************************************************************************************\

Struct:         LBoard_MonitorInfo

Description:    Information related to the monitor (current state, connectivity, and description).

Comments:       Structure used by the LBoard_GetMonitorInfo service.

\************************************************************************************************************/
typedef struct tagLBoard_MonitorInfo
{
    MUINT32 uiSize;                         // Size of the LBoard_MonitorInfo structure.
    MUINT   bIsConnected            : 1;    // Sense load of DDC activity to indicate the presence of a monitor.
    MUINT   bIsEdidPresent          : 1;    // Valid EDID can be returned.
    MUINT   bIsEdidPhysicalPresent  : 1;    // Valid physical EDID can be returned.
    MUINT   bIsEdidOverridePresent  : 1;    // Valid EDID override can be returned.
    MUINT   bIsHpdAsserted          : 1;    // Hot Plug Detect pins is logically high (+5V).
    MUINT   bIsDigital              : 1;    // Digital signal based on the EDID information.
    MUINT   bIsHdcp                 : 1;    // Respond to HDCP
    MUINT   bIsDvi                  : 1;    // Monitor is HDMI
    MUINT   bIsAudioCapable         : 1;    // Audio capabilities
    MUINT   uiReserved1             : 23;   // Reserved for future usage.

    MUINT32 uiMaxWidth;                     // Maximum width supported by the monitor.
    MUINT32 uiMaxHeight;                    // Maximum height supported by the monitor.
} LBoard_MonitorInfo;

/************************************************************************************************************\

Struct:         LBoard_Date

Description:    Generic date.

Comments:       Structure used by the LBoard_Info structure.

\************************************************************************************************************/
typedef union tagLBoard_Date
{
    MUINT32 uiValue;                   // Date.
    struct
    {
        MUINT32 uiDay           : 8;   // Date   <7-0>:  1 bytes: valid values: 1 -> 31
        MUINT32 uiMount         : 8;   // Month <15-8>:  1 bytes: Valid values: 1 -> 12
        MUINT32 uiYear          : 16;  // Year <31-16>:  2 bytes
    };
} LBoard_Date;

/************************************************************************************************************\

Struct:         LBoard_Info

Description:    Generic board information.

Comments:       Structure used by the LBoard_GetBoardInformation service.

\************************************************************************************************************/
typedef struct tagLBoard_Info
{
    MUINT32     uiSize;             // Size of the LBoard_Info structure.
    MUINT16     ui16DevID;          // PCI device ID.
    MUINT16     ui16VendorID;       // PCI vendor ID.
    MUINT32     uiSubSystemID;      // Chip sub system ID.
    MUINT32     uiAsicRev;          // ASIC revision.
    MUINT32     uiPcbRev;           // PCB Number, revision and sub revision.
    LSTR64      szProductName;      // Generic product name.
    LSTR64      szSerialNumber;     // Serial number.
    LSTR64      szPcbNumber;        // PCB Number.
    LSTR64      szFirmwareVersion;  // Firmware version.
    MUINT       bIsSystemProcessorOnBoard           : 1;    // Set when the a system processor installed on board.
    MUINT       bIsSystemProcessorOnPrimaryBus      : 1;    // Set when the system processor use the primary PCIe bus.
    MUINT       bIsExternalProcessorAvailable       : 1;    // Set when an external system processor is available (PC).
    MUINT       bIsMemoryInitialized                : 1;    // Set when the memory is initialized.
    MUINT       bIsAllVPEEnabled                    : 1;    // Enable the usage of all VPE.
    MUINT       bIsAllVPCEnabled                    : 1;    // Enable the usage of all VPC.
    MUINT       bIsSubFieldEnabled                  : 1;    // Enable the usage of subfield for HMC and FPGA.
    MUINT       bIsMemoryInterleaveDualMode         : 1;    // Memory interleave dual mode (half the memory).
    MUINT       bIsEncoderCapable                   : 1;    // Encoder capable.
    MUINT       bIsRgbYuvFullRangeConversion        : 1;    // Set when YUV surface is converted using full range by default
                                                            // instead of limited range in display or capture conversion.
    MUINT       bIsHdcpEnabled                      : 1;    // HDCP is enabled on this device.
    MUINT       bIsHdcpCapable                      : 1;    // HDCP can be enabled on this device.
    MUINT       bIsDecoderCapable                   : 1;    // Decoder capable.
    MUINT       bNoInterlacedModeList               : 1;    // No interlaced mode listed
    MUINT       uiReserved1                         : 18;   // Reserved for future use.
    MUINT32     uiLocalMemorySize;  // Quantity of local memory in megabytes.
    MUINT32     uiSystemMemorySize; // Quantity of system memory in megabytes.
                                    //  0 is returned when the system processor is not on board.
    MUINT32     uiMaxApertureSize;  // Quantity of aperture in megabytes.
    MUINT64     uiMemAddress;       // PCI memory physical address.
    MUINT64     uiRegAddress;       // PCI register physical address.
    MUINT32     uiBusNumber;        // PCI bus number.
    MUINT32     uiDeviceNumber;     // PCI device number.
    MUINT32     uiFunctionNumber;   // PCI function number.
    MUINT32     uiSlotNumber;       // PCI slot number.
    MUINT32     uiChipCount;        // Numbers of SV2 coprocessors on the board.
    MUINT32     uiChipIndex;        // 0-based SV2 index of multi device boards.
    LBoard_Date oPInSReleaseDate;   // PInS release date
    MUINT16     uiPCIeLinkWidth;    // PCIe Link Width in line
    MUINT16     uiPCIeLinkSpeed;    // PCI-Express link speed.
                                    //  P0-undefined , 1- 2.5 Gb/s, 2- 5.0 Gb/s

    MUINT32     uiEncoderLimit;     // Encoder limit in macro blocks per second.
                                    //   0xFFFFFFFF means not defined.
    MUINT32     uiDecoderLimit;     // Decoder limit in macro blocks per second.
                                    //   0xFFFFFFFF means not defined.
} LBoard_Info;

/************************************************************************************************************\

Struct:         LBoard_FpgaInfo

Description:    Generic FPGA information.

Comments:       Structure used by the LBoard_GetFpgaInfo service.

\************************************************************************************************************/
typedef struct tagLBoard_FpgaInfo
{
    MUINT32 uiSize;                         // Size of the LBoard_FpgaInfo structure.
    LSTR64  szName;                         // FPGA device name.
    MUINT16 ui16DevID;                      // FPGA device ID, default 0 if not used.
    MUINT16 ui16VendorID;                   // FPGA vendor ID, default 0 if not used.
    MUINT32 uiI2cBusIndex;                  // I2c bus index.
    MUINT32 uiI2cAddress;                   // FPGA I2c address.
    MUINT8  uiVersionMajor;                 // FPGA major version number.
    MUINT8  uiVersionMinor;                 // FPGA minor version number.
    MUINT16 uiVersionBuild;                 // FPGA build version number.
} LBoard_FpgaInfo;

/************************************************************************************************************\

Struct:         LBoard_HardwareMalfunctionInfo

Description:    Generic hardware malfunction information.

Comments:       Structure used by the LBoard_GetHardwareMalfunction service.

\************************************************************************************************************/
typedef struct tagLBoard_HardwareMalfunctionInfo
{
    MBOOL32 bHardwareMalfunction;           // Hardware malfunction.
    LSTR256 szHardwareMalfunction;          // Hardware malfunction source.
} LBoard_HardwareMalfunctionInfo;

// -----------------------------------------------------------------------------------------------------------
//                           G L O B A L   V A R I A B L E   R E F E R E N C E S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                         G L O B A L   F U N C T I O N   D E C L A R A T I O N S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                      I N L I N E S   A N D   T E M P L A T E   D E F I N I T I O N S
// -----------------------------------------------------------------------------------------------------------
#if defined (__cplusplus)
extern "C" {
#endif

/************************************************************************************************************\

Function:       LBoard_Initialize

Description:    Resets the hardware and puts it in a ready-to-use state.

Parameters:     IN  hDev                Device handle.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_NOT_OPTIMAL     Hardware is already initialized.

Comments:       - No longer used.
                - This function should not be used when the driver is loaded because the board is already
                  initialized when Liberatus_Load is called.

\************************************************************************************************************/
LAPI LStatus LBoard_Initialize(
                LDevice_Handle      hDev);


/************************************************************************************************************\

Function:       LBoard_GetBoardInformation

Description:    Gets the board information.

Parameters:     IN  hDev                Device handle.
                OUT poBoardInfo         Pointer to an LBoard_Info structure where the board information is 
                                        written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_FAIL            Function failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.

Comments:       - Fill poBoardInfo->uiSize with the size of LBoard_Info structure to use it.
                - The structure may increase in size in the next Liberatus revision but will remain compatible.
                - poBoardInfo->uiSize contains the size of the information returned.

\************************************************************************************************************/
LAPI LStatus LBoard_GetBoardInformation(
                LDevice_Handle      hDev,
                LBoard_Info*        poBoardInfo);


/************************************************************************************************************\

Function:       LBoard_GetConnectorCount

Description:    Gets the number of connectors for the specified board.

Parameters:     IN  hDev                Device handle.
                OUT puiCount            Pointer to a MUINT32 object where the number of available connectors
                                        is written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_FAIL            Function failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_GetConnectorCount(
                LDevice_Handle      hDev,
                MUINT32*            puiCount);


/************************************************************************************************************\

Function:       LBoard_GetConnectorInfo

Description:    Gets the connector information.

Parameters:     IN  hDev                Device handle.
                IN  uiIndex             Index of the connector.
                OUT poConnectorInfo     Pointer to an LBoard_ConnectorInfo object where the connector 
                                        information is written. 

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_FAIL            Function failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_NO_MORE_DATA    End of the enumeration.

Comments:       - Fill poConnectorInfo->uiSize with the size of LBoard_GetConnectorInfo structure to use it.
                - The structure may increase in size in the next Liberatus revision but will remain compatible.
                - poConnectorInfo->uiSize contains the size of the information returned.


\************************************************************************************************************/
LAPI LStatus LBoard_GetConnectorInfo(
                LDevice_Handle          hDev,
                MUINT32                 uiIndex,
                LBoard_ConnectorInfo*   poConnectorInfo);


/************************************************************************************************************\

Function:       LBoard_DetectAllMonitors

Description:    Detects the monitors and updates the EDID cache of a connector.

Parameters:     IN  hDev                Device handle.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_DetectAllMonitors(
                LDevice_Handle      hDev);


/************************************************************************************************************\

Function:       LBoard_DetectMonitor

Description:    Detects the monitor connected to the specified connector and updates the EDID cache of that 
                connector.

Parameters:     IN  hDev                Device handle.
                IN  uiIndex             Index of the connector.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - Some connectors may never return a valid EDID.

\************************************************************************************************************/
LAPI LStatus LBoard_DetectMonitor(
                LDevice_Handle      hDev,
                MUINT32             uiIndex);


/************************************************************************************************************\

Function:       LBoard_GetMonitorInfo

Description:    Gets the monitor information.

Parameters:     IN  hDev                Device handle.
                IN  uiIndex             Index of the connector.
                OUT poMonitorInfo       Pointer to an LBoard_MonitorInfo object where the monitor information
                                        is written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - Fill poMonitorInfo->uiSize with the size of LBoard_MonitorInfo structure to use it.
                - The structure may increase in size in the next Liberatus revision but will remain compatible.
                - poMonitorInfo->uiSize contains the size of the information returned.

\************************************************************************************************************/
LAPI LStatus LBoard_GetMonitorInfo(
                LDevice_Handle          hDev,
                MUINT32                 uiIndex,
                LBoard_MonitorInfo*     poMonitorInfo);

/************************************************************************************************************\

Function:       LBoard_GetEdidSize

Description:    Gets the EDID raw data size for the specified connector.

Parameters:     IN  hDev                Device handle.
                IN  uiIndex             Index of the connector.
                OUT puiEdidSize         Pointer to an MUINT32 object where the EDID raw data size, in bytes,
                                        is written.       

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     DDC not supported on the specified connector. No EDID available.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_GetEdidSize(
                LDevice_Handle      hDev,
                MUINT32             uiIndex,
                MUINT32*            puiEdidSize);

/************************************************************************************************************\

Function:       LBoard_GetEdid

Description:    Gets the EDID raw data for the specified connector.

Parameters:     IN  hDev                Device handle.
                IN  uiIndex             Index of the connector.
                IN  uiBufferSize        Size, in bytes, of the EDID raw data buffer.
                OUT pauiEdidData        Pointer to an EDID raw data buffer where the EDID raw data is 
                                        written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     DDC not supported on the specified connector. No EDID available.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_GetEdid(
                LDevice_Handle      hDev,
                MUINT32             uiIndex,
                MUINT32             uiBufferSize,
                MUINT8*             pauiEdidData);

/************************************************************************************************************\

Function:       LBoard_SetEdidOverride

Description:    Sets an EDID override for a specific connector.

Parameters:     IN  hDev                Device handle.
                IN  uiIndex             Index of the video output connector.
                IN  uiBufferSize        Size, in bytes, of the EDID override raw data.
                IN  pauiEdidData        Pointer to an EDID override raw data buffer where the EDID raw data
                                        is written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     DDC not supported on the specified connector. No EDID available.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_SetEdidOverride(
                LDevice_Handle      hDev,
                MUINT32             uiIndex,
                MUINT32             uiBufferSize,
                MUINT8*             pauiEdidData);

/************************************************************************************************************\

Function:       LBoard_DeleteEdidOverride

Description:    Deletes the EDID override for a specific connector.

Parameters:     IN  hDev                Device handle.
                IN  uiIndex             Index of the video output connector.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     DDC not supported on the specified connector. No EDID available.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_DeleteEdidOverride(
                LDevice_Handle      hDev,
                MUINT32             uiIndex);

/************************************************************************************************************\

Function:       LBoard_GetEdidOverrideSize

Description:    Gets the EDID override raw data size for the specified connector.

Parameters:     IN  hDev                Device handle.
                IN  uiIndex             Index of the video output connector.
                OUT puiEdidSize         Pointer to an MUINT32 object where the EDID override raw data size, in
                                        bytes, is written.       

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     EDID override not supported on the specified connector.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_GetEdidOverrideSize(
                LDevice_Handle      hDev,
                MUINT32             uiIndex,
                MUINT32*            puiEdidSize);

/************************************************************************************************************\

Function:       LBoard_GetEdidOverride

Description:    Gets the EDID override raw data for the specified connector.

Parameters:     IN  hDev                Device handle.
                IN  uiIndex             Index of the video output connector.
                IN  uiBufferSize        Size, in bytes, of the EDID override raw data buffer.
                OUT pauiEdidData        Pointer to a EDID override raw data buffer where EDID raw data is 
                                        written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     EDID override not supported on the specified connector.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_GetEdidOverride(
                LDevice_Handle      hDev,
                MUINT32             uiIndex,
                MUINT32             uiBufferSize,
                MUINT8*             pauiEdidData);

/************************************************************************************************************\

Function:       LBoard_GetHotPlugState

Description:    Probes and returns the level of Hot Plug Detect (HPD) pins.

Parameters:     IN  hDev                Device handle.
                IN  uiIndex             Index of the connector.
                OUT pbIsHpdAsserted     Pointer to an MBOOL32 object to be filled with MTRUE if the HPD pins 
                                        is asserted (+5V) and MFALSE otherwise.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     Hot plug detect not supported on the specified connector.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_GetHotPlugState(
                LDevice_Handle      hDev,
                MUINT32             uiIndex,
                MBOOL32*            pbIsHpdAsserted);

/************************************************************************************************************\

Function:       LBoard_CreateHotPlugEventQueue

Description:    Creates an event queue used to wait hot plug events on a set of connectors.

Parameters:     IN  hDev                Device handle.
                IN  uiNbOfConnector     Number of connectors in the array of connector indexes.
                IN  auiConnectorIndex   Array of connector indexes. To avoid memory overruns, this parameter
                                        must have at least the same number of entries as in uiNbOfConnector.
                OUT phEventQueue        Pointer to LEventQueue_Handle to be filled with a hot plug event queue
                                        handle on success.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     Hot plug detect not supported on the specified connector.

Comments:       - Only connectors that have a hot plug detection feature can be a part of the array of
                  connector indexes.
                - For more information on the connector capabilities, see the GetConnectorInfo function.

\************************************************************************************************************/
LAPI LStatus LBoard_CreateHotPlugEventQueue(
                LDevice_Handle       hDev,
                MUINT32              uiNbOfConnector,
                MUINT32*             auiConnectorIndex,
                LEventQueue_Handle*  phEventQueue);

/************************************************************************************************************\

Function:       LBoard_WaitForHotPlugEvent

Description:    Blocks a CPU thread until the board receives an HPD level change on the requested hot plug
                event queue on a set of connectors.

Parameters:     IN  hEventQueue         Hot plug event queue to wait for.
                IN  uiTimeoutIn_ms      Timeout, in milliseconds.
                OUT puiConnectorIndex   Pointer to an MUINT32 object to be filled with
                                        the connector index that have produced the hot plug event.
                OUT pbIsHpdAsserted     Pointer to an MBOOL32 object to be filled with MTRUE if the HPD pins
                                        is asserted (+5V) and MFALSE otherwise.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_TIMEOUT         Function exited after a timeout.

Comments:       If more than one event occur, they will be stored in HotPlugEvenQueue and dequeued
                one by one by calling this function.

\************************************************************************************************************/
LAPI LStatus LBoard_WaitForHotPlugEvent(
                LEventQueue_Handle   hEventQueue,
                MUINT32              uiTimeoutIn_ms,
                MUINT32*             puiConnectorIndex,
                MBOOL32*             pbIsHpdAsserted);

/************************************************************************************************************\

Function:       LBoard_DestroyHotPlugEventQueue

Description:    Destroys an event queue used to wait after hot plug events on a set of connectors.

Parameters:     IN  hEventQueue         Hot Plug event queue to destroy.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_DestroyHotPlugEventQueue(
                LEventQueue_Handle      hEventQueue);

/************************************************************************************************************\

Function:       LBoard_CreateIoEventQueue

Description:    Creates an event queue used to wait I/O events on a set of I/Os.

Parameters:     IN  hDev                Device handle.
                IN  uiNbOfIo            Number of I/Os in the array of I/Os indexes.
                IN  auiIoIndex          Array of I/O indexes. To avoid memory overruns, this parameter
                                        must have at least the same number of entries as in uiNbOfIo.
                OUT phEventQueue        Pointer to LEventQueue_Handle to be filled with a I/O event queue
                                        handle on success.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     Hot plug detect not supported on the specified connector.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_CreateIoEventQueue(
                LDevice_Handle       hDev,
                MUINT32              uiNbOfIo,
                MUINT32*             auiIoIndex,
                LEventQueue_Handle*  phEventQueue);

/************************************************************************************************************\

Function:       LBoard_WaitForIoEvent

Description:    Blocks a CPU thread until the board receives an level change on the requested I/O
                event queue on a set of I/Os.

Parameters:     IN  hEventQueue         I/O event queue to wait for.
                IN  uiTimeoutIn_ms      Timeout, in milliseconds.
                OUT puiIoIndex          Pointer to an MUINT32 object to be filled with
                                        the I/O index that have produced the I/O event.
                OUT pbState             Pointer to an MBOOL32 object to be filled with MTRUE if the logic
                                        level is 1 and MFALSE otherwise.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_TIMEOUT         Function exited after a timeout.

Comments:       If more than one event occur, they will be stored in the event queue and dequeued
                one by one by calling this function.

\************************************************************************************************************/
LAPI LStatus LBoard_WaitForIoEvent(
                LEventQueue_Handle   hEventQueue,
                MUINT32              uiTimeoutIn_ms,
                MUINT32*             puiIoIndex,
                MBOOL32*             pbState);

/************************************************************************************************************\

Function:       LBoard_DestroyIoEventQueue

Description:    Destroys an event queue used to wait after I/O events on a set of I/Os.

Parameters:     IN  hEventQueue         Event queue to destroy.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_DestroyIoEventQueue(
                LEventQueue_Handle      hEventQueue);

/************************************************************************************************************\

Function:       LBoard_GetTickReferenceFrequency

Description:    Gets the tick reference frequency in counts per second.

Parameters:     IN  hDev                Device handle.
                OUT puiTickRefFreq      Pointer to a MUINT32 variable that receives the current
                                        tick reference counter frequency, in counts per second.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_GetTickReferenceFrequency(
                LDevice_Handle      hDev,
                MUINT32*            puiTickRefFreq);

/************************************************************************************************************\

Function:       LBoard_GetTickReferenceCounter

Description:    Gets the value of current tick reference counter.

Parameters:     IN  hDev                Device handle.
                OUT puiTickRefCounter   Pointer to a MUINT64 variable that receives the current
                                        tick reference counter.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - ((uiTickRefCounter2 - uiTickRefCounter1) * 10^6) /  uiTickRefFreq = delay in micro seconds.

\************************************************************************************************************/
LAPI LStatus LBoard_GetTickReferenceCounter(
                LDevice_Handle      hDev,
                MUINT64*            puiTickRefCounter);


/************************************************************************************************************\

Function:       LBoard_SetTickReferenceCounter

Description:    Sets the initial value of the tick reference counter.

Parameters:     IN  hDev                Device handle.
                IN  uiTickRefCounter    MUINT64 variable containing the value of the tick reference counter
                                        to preload when the epoch is reset.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_UNSUPPORTED     Function failed because the hardware doesn't support the
                                        feeture.
                LStatus_FAIL            Function failed.

Comments:       - Use this function to synchronize with the reference counter across multiple devices.

\************************************************************************************************************/
LAPI LStatus LBoard_SetTickReferenceCounter(
                LDevice_Handle      hDev,
                MUINT64             uiTickRefCounter);

/************************************************************************************************************\

Function:       LBoard_DdcReadData

Description:    Reads the monitor data using DDC lines of the monitor on a specific connector 
                (typically for DDC/CI) or input EDID data.

Parameters:     IN  hDev                Device handle.
                IN  uiIndex             Index of the connector.
                IN  uiI2cAddress        I2C address (0xA0 for the EDID and 0x6E for DDC/CI).
                IN  uiI2cOffset         I2C offset [0, 255].
                IN  uiBufferSize        Size, in bytes, of the buffer where the read monitor data is written.
                                        Also specifies the number of consecutive reads to be performed.
                                        [1, 256]
                OUT pauiBuffer          Pointer to a MUINT8 buffer where the read monitor data is written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     DDC or DDC/CI not supported on the specified connector.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_DdcReadData(
                LDevice_Handle      hDev,
                MUINT32             uiIndex,
                MUINT32             uiI2cAddress,
                MUINT32             uiI2cOffset,
                MUINT32             uiBufferSize,
                MUINT8*             pauiBuffer);

/************************************************************************************************************\

Function:       LBoard_DdcWriteData

Description:    Writes the monitor data using DDC lines (typically for DDC/CI) or updates the input EDID data.

Parameters:     IN  hDev                Device handle.
                IN  uiIndex             Index of the connector.
                IN  uiI2cAddress        I2C address (0xA0 for the EDID and 0x6E for DDC/CI)
                IN  uiI2cOffset         I2C offset [0, 255]
                IN  uiBufferSize        Size, in bytes, of the buffer containing the monitor data to write.
                                        Also specifies the number of consecutive writes to be performed.
                                        [1, 256]
                IN  pauiBuffer          Pointer to a MUINT8 buffer where the monitor data to write is stored.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     DDC or DDC/CI not supported on the specified connector.

Comments:       - Writing data at I2C address 0xA0 may not work (EEPROM protected).
                - WARNING: On a monitor without a protected EEPROM, writing at I2C address 0xA0 may damage 
                  the monitor. 

\************************************************************************************************************/
LAPI LStatus LBoard_DdcWriteData(
                LDevice_Handle      hDev,
                MUINT32             uiIndex,
                MUINT32             uiI2cAddress,
                MUINT32             uiI2cOffset,
                MUINT32             uiBufferSize,
                MUINT8*             pauiBuffer);

/************************************************************************************************************\

Function:       LBoard_GetI2cBusCount

Description:    Gets the I2C bus count of the device.

Parameters:     IN  hDev                Device handle.
                OUT puiBusCount         Pointer to an MUINT32 object where I2C bus count is written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_GetI2cBusCount(
                LDevice_Handle      hDev,
                MUINT32*            puiBusCount);

/************************************************************************************************************\

Function:       LBoard_GetI2cBusName

Description:    Gets the name of an I2C bus.

Parameters:     IN  hDev                Device handle.
                IN  uiBusIndex          I2C bus index.
                OUT szI2cBusName        Buffer where the ANSI null terminated I2C bus name is written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_NO_MORE_DATA    No I2C bus is associated with the specified bus index.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_GetI2cBusName(
                LDevice_Handle      hDev,
                MUINT32             uiBusIndex,
                LSTR64              szI2cBusName);

/************************************************************************************************************\

Function:       LBoard_OpenI2c

Description:    Opens an I2C bus for read or write operations.

Parameters:     IN  hDev                Device handle.
                IN  uiBusIndex          I2C bus index.
                IN  uiFreqInHz          Frequency, in Hertz, to be used for bus communications.
                OUT phI2c               Pointer to an LI2c_Handle object where I2C handle is written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_NO_MORE_DATA    No I2C bus is associated with the specified bus index.    

Comments:       - An I2C handle is required to perform read/write operations on an I2C bus.
                - A single instance of an I2C bus can be open at any one time. Keeping an I2C bus open
                  prevents other process from using the I2C bus, so we recommend keeping it open for the least
                  amount of time possible.
                - Trying to open an I2C bus already opened by another process pauses the calling thread until
                  the I2C bus is closed by the other process.

\************************************************************************************************************/
LAPI LStatus LBoard_OpenI2c(
                LDevice_Handle      hDev,
                MUINT32             uiBusIndex,
                MUINT32             uiFreqInHz,
                LI2c_Handle*        phI2c);

/************************************************************************************************************\

Function:       LBoard_CloseI2c

Description:    Closes an open I2C bus.

Parameters:     IN  hI2c         Handle to the bus to close.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - An open I2C bus must be closed as soon as it's no longer needed to allow other process
                  to access the I2C bus.

\************************************************************************************************************/
LAPI LStatus LBoard_CloseI2c(
                LI2c_Handle         hI2c);

/************************************************************************************************************\

Function:       LBoard_ReadI2c

Description:    Reads data from a device on an I2C bus.

Parameters:     IN  hI2c                I2C bus handle acquired from LBoard_OpenI2c.
                IN  uiAddress           Address of the I2C device to read data from.
                IN  uiOffsetSize        Size, in bytes, of the offset. [0-4].
                IN  uiOffset            0-based offset of the first MUINT8 to be read. Ignore when 
                                        uiOffsetSize is 0 byte.
                IN  uiBufferSize        Size, in bytes, of the buffer where the I2C device data read is 
                                        stored. It also specifies the number of consecutive I2C reads to 
                                        be performed.
                OUT pauiData            Pointer to an MUINT8 buffer where the read I2C device data is stored.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - The I2C bus must be open when calling this function.

\************************************************************************************************************/
LAPI LStatus LBoard_ReadI2c(
                LI2c_Handle         hI2c,
                MUINT32             uiAddress,
                MUINT32             uiOffsetSize,
                MUINT32             uiOffset,
                MUINT32             uiBufferSize,
                MUINT8*             pauiData);

/************************************************************************************************************\

Function:       LBoard_WriteI2c

Description:    Writes data to a device on an I2C bus.

Parameters:     IN  hI2c                I2C bus handle acquired from LBoard_OpenI2c.
                IN  uiAddress           Address of the I2C device to which the data should be written.
                IN  uiOffsetSize        Size, in bytes, of the offset. [0-4].
                IN  uiOffset            0-based offset of the first MUINT8 to be written. Ignore when 
                                        uiOffsetSize is 0 byte. 
                IN  uiBufferSize        Size, in bytes, of the buffer where the I2C device data to write is 
                                        stored. Also specifies the number of consecutive I2C writes to be 
                                        performed.
                OUT pauiData            Pointer to an MUINT8 buffer where the I2C device data to write is 
                                        stored.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - The I2C bus must be open when calling this function.

\************************************************************************************************************/
LAPI LStatus LBoard_WriteI2c(
                LI2c_Handle         hI2c,
                MUINT32             uiAddress,
                MUINT32             uiOffsetSize,
                MUINT32             uiOffset,
                MUINT32             uiBufferSize,
                MUINT8*             pauiData);

/************************************************************************************************************\

Function:       LBoard_ScheduleWriteI2c

Description:    Schedules a send data to a device on an I2C bus by adding required commands to hDeviceThread.

Parameters:     IN  hDev                Device handle.
                IN  uiBusIndex          I2C bus index.
                IN  uiFreqInHz          Frequency, in Hertz, to be used for bus communications.
                IN  uiAddress           Address of the I2C device to write data to.
                IN  uiOffsetSize        Size, in bytes, of the offset. [0-4].
                IN  uiOffset            0-based offset of the first MUINT8 to be written.
                                        Ignore when uiOffsetSize is 0 byte.
                IN  uiBufferSize        Size, in bytes, of the array pointed by the pauiData parameter.
                IN  pauiData            Pointer to an array of MUINT8 object containing the data to write
                IN  hDeviceThread       Handle of the device thread used to schedule the operation.
                IN  uiTimeout_ms        Device thread is full. Queue wait timeout.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_TIMEOUT         Function exited after a timeout.

Comments:       - The I2C bus is opened and closed implicitly at execution.
                - LStatus_TIMEOUT is returned when the wait command can't be added to the device thread 
                  because it's full and the timeout period expired before space became available.

\************************************************************************************************************/
LAPI LStatus LBoard_ScheduleWriteI2c(
                LDevice_Handle          hDev,
                MUINT32                 uiBusIndex, 
                MUINT32                 uiFreqInHz, 
                MUINT32                 uiAddress, 
                MUINT32                 uiOffsetSize, 
                MUINT32                 uiOffset, 
                MUINT32                 uiBufferSize, 
                MUINT8*                 pauiData,
                LDeviceThread_Handle    hDeviceThread,
                MUINT32                 uiTimeout_ms);

/************************************************************************************************************\

Function:       LBoard_GetSclState

Description:    Gets the state of the SCL line of an I2C bus.

Parameters:     IN  hI2c                I2C bus handle acquired from LBoard_OpenI2c.
                OUT pbState             Pointer to an MBOOL32 object where the SCL line state is written.
                                         MTRUE if the SCL line is high. MFALSE if it's low.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_GetSclState(
                LI2c_Handle         hI2c, 
                MBOOL32*            pbState);

/************************************************************************************************************\

Function:       LBoard_SetSclState

Description:    Sets the state of the SCL line of an I2C bus.

Parameters:     IN  hI2c                I2C bus handle acquired from LBoard_OpenI2c.
                IN  bState              MTRUE to set the SCL line to high. MFALSE to set the SCL line to low.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_SetSclState(
                LI2c_Handle         hI2c, 
                MBOOL32             bState);

/************************************************************************************************************\

Function:       LBoard_GetSdaState

Description:    Gets the state of the SDA line of an I2C bus.

Parameters:     IN  hI2c                I2C bus handle acquired from LBoard_OpenI2c.
                OUT pbState             Pointer to an MBOOL32 object where the SDA line state is written. 
                                        Set to MTRUE if the SDA line is high. Set to MFALSE if the SDA line
                                        is low.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_GetSdaState(
                LI2c_Handle         hI2c, 
                MBOOL32*            pbState);

/************************************************************************************************************\

Function:       LBoard_SetSdaState

Description:    Sets the state of the SDA line of an I2C bus.

Parameters:     IN  hI2c                I2C bus handle acquired from LBoard_OpenI2c.
                IN  bState              MTRUE to set the SCL line to high. MFALSE to set the SCL line to low.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_SetSdaState(
                LI2c_Handle         hI2c,
                MBOOL32             bState);

/************************************************************************************************************\

Function:       LBoard_GetIoCount

Description:    Gets the IO pin count of a device.

Parameters:     IN  hDev                Device handle.
                OUT puiIoCount          Pointer to an MUINT32 object where IO pin count is written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_GetIoCount(
                LDevice_Handle      hDev,
                MUINT32*            puiIoCount);

/************************************************************************************************************\

Function:       LBoard_GetIoName

Description:    Gets the IO related name.

Parameters:     IN  hDev                Device handle.
                IN  uiIoIndex           IO index.
                OUT szIoName            LSTR64 to be filled with the null terminated IO name.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_NO_MORE_DATA    No name is available for the specified index.    

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_GetIoName(
                LDevice_Handle      hDev,
                MUINT32             uiIoIndex,
                LSTR64              szIoName);

/************************************************************************************************************\

Function:       LBoard_OpenIo

Description:    Opens an IO.

Parameters:     IN  hDev                Device handle.
                IN  uiIoIndex           IO index.
                OUT phIo                Pointer to a LIo_Handle object to be filled with an IO index. 

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_NO_MORE_DATA    No open is available for the specified index.

Comments:       - A handle is required to perform get/set operations on an IO pin.
                - Only one IO can be open at any one time. Keeping an IO open prevents other process from using
                  that IO, so we recommend keeping an IO open for the least amount of time possible.
                - Trying to open an IO opened by another user pauses the calling thread until either the IO is 
                  closed by the other user or the timeout period has expired.

\************************************************************************************************************/
LAPI LStatus LBoard_OpenIo(
                LDevice_Handle      hDev,
                MUINT32             uiIoIndex,
                LIo_Handle*         phIo);

/************************************************************************************************************\

Function:       LBoard_CloseIo

Description:    Closes an open IO.

Parameters:     IN  hIo                 Handle to the IO to close.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - An open IO must be closed as soon as it's no longer needed to allow other process to access
                  the IO.

\************************************************************************************************************/
LAPI LStatus LBoard_CloseIo(
                LIo_Handle         hIo);

/************************************************************************************************************\

Function:       LBoard_GetIoState

Description:    Gets the state of the IO line.

Parameters:     IN  hIo                 IO handle acquired from LBoard_OpenIo.
                OUT pbState             Pointer to an MBOOL32 object where the  state of the line is written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_GetIoState(
                LIo_Handle          hIo,
                MBOOL32*            pbState);

/************************************************************************************************************\

Function:       LBoard_SetIoState

Description:    Sets the state of the IO line.

Parameters:     IN  hIo                 IO handle retrieved from LBoard_OpenIo.
                IN  bState              State of the line to set.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_SetIoState(
                LIo_Handle          hIo,
                MBOOL32             bState);

/************************************************************************************************************\

Function:       LBoard_ScheduleSetIoState

Description:    Schedules a set state of the IO line by adding the required commands to hDeviceThread.

Parameters:     IN  hDev                Device handle.
                IN  uiIoIndex           IO index.
                IN  bState              State of the line to set.
                IN  hDeviceThread       Handle of the device thread used to schedule the operation.
                IN  uiTimeout_ms        Device thread is full. Queue wait timeout.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_NO_MORE_DATA    Index not valid.
                LStatus_TIMEOUT         Function exited after a timeout.

Comments:       - The IO pins are opened and closed implicitly at the execution.
                - LStatus_TIMEOUT is returned when the wait command can't be added to the device thread 
                  because it's full and the timeout period expired before space became available.


\************************************************************************************************************/
LAPI LStatus LBoard_ScheduleSetIoState(
                LDevice_Handle          hDev,
                MUINT32                 uiIoIndex,
                MBOOL32                 bState,
                LDeviceThread_Handle    hDeviceThread,
                MUINT32                 uiTimeout_ms);

/************************************************************************************************************\

Function:       LBoard_RegRead

Description:    Reads an SV2 register.

Parameters:     IN      hDev            Device handle.
                IN      uiAddress       Address of the register to read.
                OUT     puiValue        Pointer to a MUINT32 variable where the register value is stored.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_RegRead(
                LDevice_Handle          hDev,
                MUINT32                 uiAddress,
                MUINT32*                puiValue);

/************************************************************************************************************\

Function:       LBoard_RegWrite

Description:    Writes a value to an SV2 register.

Parameters:     IN      hDev            Device handle.
                IN      uiAddress       Address of the register to write.
                IN      uiValue         Value to write in the register.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_RegWrite(
                LDevice_Handle          hDev,
                MUINT32                 uiAddress,
                MUINT32                 uiValue);

/************************************************************************************************************\

Function:       LBoard_CustomCall

Description:    Calls a custom function. This can be used to extend the Liberatus APIs.

Parameters:     IN      hDev            Device handle.
                IN      uiCustomCallId  Custom call ID.
                IN      hInData         Pointer to function specific input data.
                IN      uiInDataSize    Size, in bytes, of the function specific input data.
                OUT     hOutData        Pointer to function specific output data to be filled by function.
                IN      uiOutDataSize   Size, in bytes, of the function specific output data.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - Can be used to extend the Liberatus APIs into an external device layer driver.
                - hInData and hOutData must be copied to/from kernel with size uiInDataSize and uiOutDataSize
                  respectively. As a rule, the data structures these parameters point to must not contain
                  pointers to other user data.


\************************************************************************************************************/
LAPI LStatus LBoard_CustomCall(
                LDevice_Handle          hDev,
                MUINT32                 uiCustomCallId,
                MHANDLE                 hInData,
                MUINT32                 uiInDataSize,
                MHANDLE                 hOutData,
                MUINT32                 uiOutDataSize);

/************************************************************************************************************\

Function:       LBoard_GetFpgaCount

Description:    Gets the number of FPGAs for the specified board.

Parameters:     IN  hDev                Device handle.
                OUT puiFpgaCount        Pointer to a MUINT32 object where the number of available FPGA 
                                        is written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_FAIL            Function failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_GetFpgaCount(
                LDevice_Handle          hDev,
                MUINT32*                puiFpgaCount);


/************************************************************************************************************\

Function:       LBoard_GetFpgaInfo

Description:    Gets FPGA information.

Parameters:     IN      hDev            Device handle.
                IN      uiFpgaIndex     Index of the FPGA to query for information.
                IN OUT  poFpgaInfo      Pointer to a LBoard_FpgaInfo structures where FPGA info is written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_FAIL            Function failed.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.

Comments:       - Fill poFpgaInfo->uiSize with the size of LBoard_FpgaInfo structure to use it.
                - The structure may increase in size in the next Liberatus revision but will remain compatible.
                - poFpgaInfo->uiSize contains the size of the information returned.

\************************************************************************************************************/
LAPI LStatus LBoard_GetFpgaInfo(
                LDevice_Handle          hDev,
                MUINT32                 uiFpgaIndex,
                LBoard_FpgaInfo*        poFpgaInfo);

/************************************************************************************************************\

Function:       LBoard_GetHardwareMalfunction

Description:    Obtains if LStatus_HARDWARE_MALFUNCTION has been returned.

Parameters:     IN  hDev                        Device handle.
                OUT poHardwareMalfunctionInfo   Pointer to a structure filled with hardware malfunction info.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBoard_GetHardwareMalfunction(
                LDevice_Handle                  hDev,
                LBoard_HardwareMalfunctionInfo* poHardwareMalfunctionInfo);

/************************************************************************************************************\

Struct:         LBoard_KsvList

Description:    KSV list structure

Comments:       Structure used by the following service:
                LBoard_RetrieveVideoOutKsvList
                LBoard_ApplyVideoInKsvList
                LBoard_EnumVideoInKsvList

                Absolute number of KSV is 128, which cover all cases:
                DP 1.1 supports up to 127 KSV.
                DP 1.2 supports up to 32 KSV
                HDMI 1.x supports up to 127 KSV.
                HDMI 2.x supports up to 32 KSV.

\************************************************************************************************************/
typedef struct tagLBoard_KsvList
{
    LSTR64          szName;         // KSV list name (Unique Identifier).
                                    //     <PRODUCT_NAME>,<SERIAL>,<CONNECTOR INDEX>,<CONNECTOR_NAME>
    MUINT32         uiDepth;        // Repeater cascade depth.
    MUINT32         uiCount;        // Number of devices (KSV).
    MUINT8          auiKsv[128*5];  // KSV list (each KSV is 40-bit = 5 bytes).
} LBoard_KsvList;


/************************************************************************************************************\

Function:       LBoard_GetVideoInKsvHandle

Description:    Gets a KSV handle to manage the KSV lists of the video inputs board.

Parameters:     IN  hDev                Device handle.
                OUT phKsv               Pointer to an LKsv_Handle variable where the ksv handle is
                                        written. MNULL if unsuccessful.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_UNSUPPORTED     Function failed due to lack of support.
                LStatus_FAIL            Function failed due to generic error.

Comments:       There should be only one handle requested per manager application.
                Required to call LBoard_ApplyVideoInKsvList and LBoard_RemoveVideoInKsvList services.

\************************************************************************************************************/
LAPI LStatus LBoard_GetVideoInKsvHandle(
                LDevice_Handle      hDev,
                LKsv_Handle*        phKsv);


/************************************************************************************************************\

Function:       LBoard_ReleaseVideoInKsvHandle

Description:    Releases the KSV handle.

Parameters:     IN  hKsv                KSV handle.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_UNSUPPORTED     Function failed due to lack of support.
                LStatus_FAIL            Function failed due to generic error.

Comments:       Releases all KVS lists associated to the KSV handle.

\************************************************************************************************************/
LAPI LStatus LBoard_ReleaseVideoInKsvHandle(
                LKsv_Handle         hKsv);


/************************************************************************************************************\

Function:       LBoard_ApplyVideoInKsvList

Description:    Adds or updates the KSV list to a video input connector.

Parameters:     IN  hKsv                KSV handle.
                IN  uiConnectorIndex    Connector index.
                IN  poKsvList           Pointer to KSV list structure.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_UNSUPPORTED         Function failed due to lack of support.
                LStatus_OUT_OF_RESOURCES    Function failed due to insufficient resources (too many KSV).
                LStatus_FAIL                Function failed due to generic error.

Comments:       When poKsvList->szName is already set, the KSV list is updated. Otherwise, an additional 
                KSV list is applied.
                LStatus_OUT_OF_RESOURCES is returned when the KSV lists overpass the hardware limits. 
                The video input capture may be blanked for security reason.
                More than one application can apply a KSV list on the same input using different KSV handles 
                and the union of all constrains is applied.

                *** Status_OUT_OF_RESOURCES ==> changer le nom (warning)

\************************************************************************************************************/
LAPI LStatus LBoard_ApplyVideoInKsvList(
                LKsv_Handle         hKsv,
                MUINT32             uiConnectorIndex,
                LBoard_KsvList*     poKsvList);


/************************************************************************************************************\

Function:       LBoard_RemoveVideoInKsvList

Description:    Removes the KSV list from the video input connector.

Parameters:     IN  hKsv                KSV handle.
                IN  uiConnectorIndex    Connector index.
                IN  szName              KSV list name to remove.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_UNSUPPORTED     Function failed due to lack of support.
                LStatus_FAIL            Function failed due to generic error.

Comments:       None.
                LStatus_FAIL is returned when poKsvList->szName is not found for the KSV handle.
                *** do we need LStatus_OK_NOTHING_DONE ***

\************************************************************************************************************/
LAPI LStatus LBoard_RemoveVideoInKsvList(
                LKsv_Handle         hKsv,
                MUINT32             uiConnectorIndex,
                LSTR64              szName);
                

/************************************************************************************************************\

Function:       LBoard_GetVideoKsvListId

Description:    Gets the KSV list identifier.

Parameters:     IN  hDev                Device handle.
                IN  uiConnectorIndex    Connector index.
                OUT puiKsvListId        Pointer to a variable where KSV list identifier is returned.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_UNSUPPORTED     Function failed due to lack of support.
                LStatus_FAIL            Function failed.

Comments:       The KSV list identifier changes every times the KSV list topology change.

                On video output boards, this ID is used to verify if the KSV lists have changed.
                The application is responsible to poll this value at every second.
                Refer to : LBoard_GetVideoOutKsvList to retrieve KSV list when the topology changes.

                On video input boards, this ID is used to make sure the KSV lists didn't change 
                through the enumeration. refer to : LBoard_EnumVideoInKsvList

                Value 0 is invalid and should never be returned unless there is an error.

Open Issues:    Should we auto-check polling doesn't exceed TBD ms?

\************************************************************************************************************/
LAPI LStatus LBoard_GetVideoKsvListId(
                LDevice_Handle      hDev,
                MUINT32             uiConnectorIndex,
                MUINT32*            puiKsvListId);
                

/************************************************************************************************************\

Function:       LBoard_GetVideoOutKsvList

Description:    Gets the KSV list from a video output connector.

Parameters:     IN  hDev                Device handle.
                IN  uiConnectorIndex    Connector index.
                IN  uiKsvListId         KSV list identifier retrieved by LBoard_GetVideoKsvListId services.
                OUT poKsvList           Pointer to KSV list structure.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_UNSUPPORTED     Function failed due to lack of support.
                LStatus_FAIL            Function failed due to KSV list ID mismatch.

Comments:       The KVM manager application has to retrieve KSV list and take action.

\************************************************************************************************************/
LAPI LStatus LBoard_GetVideoOutKsvList(
                LDevice_Handle      hDev,
                MUINT32             uiConnectorIndex,
                MUINT32             uiKsvListId,
                LBoard_KsvList*     poKsvList);


/************************************************************************************************************\

Function:       LBoard_EnumVideoInKsvList

Description:    Enumerates the KSV lists from the video input connector.

Parameters:     IN  hDev                Device handle.
                IN  uiConnectorIndex    Connector index.
                IN  uiKsvListId         KSV list identifier retrieved by LBoard_GetVideoKsvListId services.
                IN  uiKsvListIndex      KSV list index of the enumeration.
                OUT poKsvList           Pointer to the LBoard_KsvList object where the KSV List asked is 
                                        returned.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_NO_MORE_DATA    No data is associated to the specified index (end of enumeration).
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_UNSUPPORTED     Function failed due to lack of support.
                LStatus_FAIL            Function failed due to KSV list ID mismatch.

Comments:       The main usage of this service is for debugging purpose.  
                Not required to make a functional KSV manager system.

\************************************************************************************************************/
LAPI LStatus LBoard_EnumVideoInKsvList(
                LDevice_Handle      hDev,
                MUINT32             uiConnectorIndex,
                MUINT32             uiKsvListId,
                MUINT32             uiKsvListIndex,
                LBoard_KsvList*     poKsvList);

/************************************************************************************************************\

Function:       LBoard_GetTickReferenceId

Description:    Gets tick reference identifier.

Parameters:     IN  hDev                Device handle.
                OUT puiTickReferenceId  Pointer MUINT32 variable to be filled with the current tick reference
                                         identifier.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - The identifier changes every times the clock reference change.
                - 0 mean the initial clock domain.
                - The clock domain may never change on certain products.
                - When tick reference identifier change, the linearity of ticks increment can be affected.
                - When tick reference identifier change, tick reference frequency returned by
                   LBoard_GetTickReferenceFrequency services may also change.

\************************************************************************************************************/
LAPI LStatus LBoard_GetTickReferenceId(
                LDevice_Handle      hDev,
                MUINT32*            puiTickReferenceId);

/************************************************************************************************************\

Function:       LBoard_WaitForTickReferenceIdChange

Description:    Blocks a CPU thread until the current tick reference identifier doesn't match the tick
                reference identifier in parameters.

Parameters:     IN  hDev                Device handle.
                IN  uiTimeoutIn_ms      Timeout, in milliseconds.  0 means no wait and
                                         LINFINITE_TIMEOUT means forever.
                IN  uiTickReferenceId   MUINT32 variable sets with the tick reference identifier
                                         gets by LBoard_GetTickReferenceId services.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_TIMEOUT         Function exited after a timeout.
                LStatus_CANCELED        The tick reference is never change on this product.
                LStatus_FAIL            Function failed.

Comments:       - The clock domain may never change on certain products and LStatus_CANCELED is returned.
                - The linearity of ticks increment can be affected after clock change.
                   refer to: LBoard_GetTickReferenceCounter services
                - The tick reference frequency may also change after clock change.
                   refer to: LBoard_GetTickReferenceFrequency services.
                - LBoard_SetTickReferenceCounter following an epoch reset cause tick reference identifier
                   change.
                - Other custom call link with genlock/framelock fonctionnality cause tick reference
                   identifier change.


\************************************************************************************************************/
LAPI LStatus LBoard_WaitForTickReferenceIdChange(
                LDevice_Handle      hDev,
                MUINT32             uiTimeoutIn_ms,
                MUINT32             uiTickReferenceId);


#if defined (__cplusplus)
}
#endif

#endif // #ifndef INC_LBOARD_H
