/************************************************************************************************************\

Module Name:    LBlit.h

Description:    Liberatus for blit module interface.

References:     None.

    Copyright (c) 2011, Matrox Graphics Inc.
    All Rights Reserved.

\************************************************************************************************************/

#ifndef INC_LBLIT_H
#define INC_LBLIT_H

// -----------------------------------------------------------------------------------------------------------
//                                   I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "Liberatus.h"
#include "LPixelFormat.h"

typedef struct LBLIT_OBJECT*         LBlit_Handle;
typedef struct LCLIPRGN_OBJECT*      LClipRgn_Handle;

// -----------------------------------------------------------------------------------------------------------
//                                   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Enum:           LBlit_AttributeType

Description:    Enumeration of attribute types.

Comments:       None.

\************************************************************************************************************/
typedef enum {
    LBlit_AttributeType_CREATE  = 0            // Attributes structure for the Create method.
} LBlit_AttributeType;

/************************************************************************************************************\

Structure:      LBlit_CreateAttributes

Description:    LBlit creation attributes.

Comments:       LBlit_AttributeType_CREATE.

\************************************************************************************************************/
typedef struct tagLBlit_CreateAttributes
{
    LBlit_AttributeType eType;                  // Shall be LBlit_AttributeType_CREATE
    MBOOL32             bLoadBalancing;         // Load balacing
    MBOOL32             bWaitforDMA;            // Wait for DMA
} LBlit_CreateAttributes;

/************************************************************************************************************\

Structure:      LBlit_ColorKey

Description:    Color key parameters.

Comments:       Color key is not supported if the pixel format is over 32 bpp in one of the planes.

\************************************************************************************************************/
typedef struct tagLBlit_ColorKey
{
    LColor oColorKey;       // The color of the pixels in the source that will be transparent. Must be the 
                            // same pixel format as the source buffer.
    LColor oColorKeyMask;   // Defines the bits that is compared to the color key (1 = Compared, 0 = Ignored).
                            // Must be the same pixel format as the color key.
} LBlit_ColorKey;

/************************************************************************************************************\

Enum:           LBlit_BlendFunc

Description:    Blend functions.

Comments:       None.

\************************************************************************************************************/
typedef enum 
{
    LBlit_BlendFunc_CONSTANT,       // Dst = Dst * (1 - ConstantAlpha) + Src * ConstantAlpha
                                    // The alpha is a constant value.
    LBlit_BlendFunc_SOURCE,         // Dst = Dst * (1 - Src.Alpha) + Src * Src.Alpha
                                    // The alpha is taken from the source buffer.
    LBlit_BlendFunc_DESTINATION,    // Dst = Dst * Dst.alpha + Src * (1 - Dst.Alpha)
                                    // The alpha is taken from the destination buffer
} LBlit_BlendFunc;

/************************************************************************************************************\

Structure:      LBlit_AlphaBlend

Description:    Alpha blending parameters.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLBlit_AlphaBlend
{
    LBlit_BlendFunc eBlendFunc;     // Blend function.
    MFLOAT32        fConstantAlpha; // Constant alpha between 0 and 1. This is the taken value when 
                                    // eBlendFunc = LBlit_BlendFunc_CONSTANT. Else, this variable is ignored.
    MBOOL32         bKeepSrcAlpha;  // The alpha component of the source is...
                                    //   MFALSE: ...blended as the remaining components.
                                    //   MTRUE:  ...copied directly to the destination.
                                    // If the destination alpha has to be kept, use the plane write mask.
} LBlit_AlphaBlend;

/************************************************************************************************************\

Enum:           LBlit_Bop

Description:    Binary operations.

Comments:       None.

\************************************************************************************************************/
typedef enum
{
    LBlit_Bop_0,        // Dst = 0
    LBlit_Bop_DSon,     // Dst = NOT(Dst OR Src)
    LBlit_Bop_DSna,     // Dst = Dst AND NOT(Src)
    LBlit_Bop_Sn,       // Dst = NOT(Src)
    LBlit_Bop_SDna,     // Dst = NOT(Dst) AND Src
    LBlit_Bop_Dn,       // Dst = NOT(Dst)
    LBlit_Bop_DSx,      // Dst = Dst XOR Src
    LBlit_Bop_DSan,     // Dst = NOT(Dst AND Src)
    LBlit_Bop_DSa,      // Dst = Dst AND Src
    LBlit_Bop_DSxn,     // Dst = NOT(Dst XOR Src)
    LBlit_Bop_D,        // Dst = Dst
    LBlit_Bop_DSno,     // Dst = Dst OR NOT(Src)
    LBlit_Bop_S,        // Dst = Src
    LBlit_Bop_SDno,     // Dst = NOT(Dst) OR Src
    LBlit_Bop_DSo,      // Dst = Dst OR Src
    LBlit_Bop_1         // Dst = 1
    
} LBlit_Bop;

/************************************************************************************************************\

Enum:           LBlit_Rotation

Description:    Rotation parameters.

Comments:       None.

\************************************************************************************************************/
typedef enum 
{
    LBlit_Rotation_0,           // No rotation
    LBlit_Rotation_90,          // Rotate the source 90 degrees clockwise
    LBlit_Rotation_180,         // Rotate the source 180 degrees clockwise
    LBlit_Rotation_270,         // Rotate the source 270 degrees clockwise
    LBlit_Rotation_0_FLIP_H,    // Flip horizontally (mirror)
    LBlit_Rotation_90_FLIP_H,   // Rotate 90 and then flip horizontally
    LBlit_Rotation_180_FLIP_H,  // Rotate 180 and then flip horizontally
    LBlit_Rotation_270_FLIP_H,  // Rotate 270 and then flip horizontally
    LBlit_Rotation_0_FLIP_V     = LBlit_Rotation_180_FLIP_H,
    LBlit_Rotation_90_FLIP_V    = LBlit_Rotation_270_FLIP_H,
    LBlit_Rotation_180_FLIP_V   = LBlit_Rotation_0_FLIP_H,
    LBlit_Rotation_270_FLIP_V   = LBlit_Rotation_90_FLIP_H,
    LBlit_Rotation_0_FLIP_HV    = LBlit_Rotation_180,
    LBlit_Rotation_90_FLIP_HV   = LBlit_Rotation_270,
    LBlit_Rotation_180_FLIP_HV  = LBlit_Rotation_0,
    LBlit_Rotation_270_FLIP_HV  = LBlit_Rotation_90,
    
} LBlit_Rotation;

/************************************************************************************************************\

Enum:           LBlit_GradientDir

Description:    Gradient direction.

Comments:       None.

\************************************************************************************************************/
typedef enum
{
    LBlit_GradientDir_HORIZONTAL,
    LBlit_GradientDir_VERTICAL,
    LBlit_GradientDir_DIAGONAL_DOWN,
    LBlit_GradientDir_DIAGONAL_UP,
    
} LBlit_GradientDir;

/************************************************************************************************************\

Enum:           LClipRgn_EnumDir

Description:    Directions in which the rectangles are to be enumerated.

Comments:       None.

\************************************************************************************************************/
typedef enum
{
    LClipRgn_EnumDir_LEFT_DOWN,
    LClipRgn_EnumDir_LEFT_UP,
    LClipRgn_EnumDir_RIGHT_DOWN,
    LClipRgn_EnumDir_RIGHT_UP,
    LClipRgn_EnumDir_ANY
} LClipRgn_EnumDir;

// -----------------------------------------------------------------------------------------------------------
//                           G L O B A L   V A R I A B L E   R E F E R E N C E S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                         G L O B A L   F U N C T I O N   D E C L A R A T I O N S
// -----------------------------------------------------------------------------------------------------------

#if defined (__cplusplus)
extern "C" {
#endif

/************************************************************************************************************\

Function:       LBlit_Create

Description:    Create a LBlit context.

Parameters:     IN  hDevice         Handle to a device.
                IN  hDeviceThread   Handle to a device thread.
                OUT phContext       Pointer on the handle to the LBlit context.

Return Value:   LStatus_OK              The operation completed successfully.
                LStatus_FAIL            The operation failed.
                LStatus_OUT_OF_MEMORY   Failed because no more memory is available.
                LStatus_INVALID_PARAM   Failed because a parameter is invalid.

Comments:       When the handle is not used anymore, it must be destroyed using LBlit_Destroy.

\************************************************************************************************************/
LAPI LStatus LBlit_Create
(
    LDevice_Handle          hDevice,
    LDeviceThread_Handle    hDeviceThread,
    LBlit_AttributeType*    poAttributes,
    LBlit_Handle*           phContext
);

/************************************************************************************************************\

Function:       LBlit_Destroy

Description:    Destroy the LBlit context and all the resources attached to it.

Parameters:     IN hContext     Handle to the LBlit context.

Return Value:   LStatus_OK              The operation completed successfully.
                LStatus_FAIL            The operation failed.
                LStatus_INVALID_PARAM   Failed because a parameter is invalid.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LBlit_Destroy
(
    LBlit_Handle hContext
);

/************************************************************************************************************\

Function:       LBlit_MemoryFill

Description:    Fill a linear buffer with the given data.

Parameters:     IN hContext     Handle to the LBlit context.
                IN hBuffer      Handle to the linear buffer.
                IN uiData       4 bytes of data to fill with. The least significant byte is always copied 
                                first at the offset (see uiOffset parameter), no matter if the offset is 
                                aligned on 4 bytes or not. If the size (see uiSize parameter) is not a 
                                multiple of 4 bytes, the data will be clipped at the end of the memory fill 
                                starting with the most significant byte.
                IN uiOffset     Byte offset in the buffer where the fill start.
                IN uiSize       Number of bytes to transfer. If uiSize is equal to 0 (zero), all the memory 
                                from the offset to the end of the buffer is filled.

Return Value:   LStatus_OK              The operation completed successfully.
                LStatus_FAIL            The operation failed.
                LStatus_INVALID_PARAM   Failed because a parameter is invalid.

Comments:       The operation fails if the fill range is outside the buffer.

                The buffers type must be linear (LBuffer_Type_LINEAR).
                                
\************************************************************************************************************/
LAPI LStatus LBlit_MemoryFill
(
    LBlit_Handle    hContext,
    LBuffer_Handle  hBuffer,
    MUINT32         uiData,
    MUINT32         uiOffset,
    MUINT32         uiSize
);

/************************************************************************************************************\

Function:       LBlit_GetDefaultAttributes

Description:    Gets the default values of a given type of attributes.

Parameters:     IN  hContext                Handle of a LBlit object.
                OUT peAttributes            Pointer to the attribute type member of a structure of attributes.
                                            On input, the type member describes the type of the structure.
                                            On output, the remaining members of the structure are filled with
                                            the default value of the attributes.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This function provides a mechanism for retrieving an attribute structure filled with default
                  values. The application is concerned only with the subset of the members it's interested in.
                - Only one type of attribute at a time can be retrieved. Only AttributeType_CREATE is
                  currently supported.

\************************************************************************************************************/
LAPI LStatus LBlit_GetDefaultAttributes
(
    LBlit_Handle            hContext,
    LBlit_AttributeType*    peAttributes
);

/************************************************************************************************************\

Function:       LBlit_GetAttributes

Description:    Gets the current values of a given type of attributes.

Parameters:     IN  hContext                Handle of a LBlit object.
                OUT peAttributes            Pointer to the attribute type member of a structure of attributes.
                                            On input, the type member describes the type of the structure.
                                            On output, the remaining members of the structure are filled with
                                            the default value of the attributes.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - This function provides a mechanism for retrieving an attribute structure filled with default
                  values. The application is concerned only with the subset of the members it's interested in.
                - Only one type of attribute at a time can be retrieved. Only AttributeType_CREATE is
                  currently supported.

\************************************************************************************************************/
LAPI LStatus LBlit_GetAttributes
(
    LBlit_Handle            hContext,
    LBlit_AttributeType*    peAttributes
);

/************************************************************************************************************\

Function:       LBlit_SolidFill

Description:    Fill a list of rectangle into the buffer with a solid color.

Parameters:     IN hContext     Handle to the LBlit context.
                IN hBuffer      Handle to the buffer where the solid fill will be done.
                IN poColor      Color in the same pixel format as the buffer. It cannot be NULL.
                IN poRect       Pointer to the rectangle. The rectangle will be clipped to the boundaries of 
                                the clippping region. If this parameter is equal to NULL, the rectangle is 
                                equal to the limits of the video buffer.
                IN hClipRgn     Handle to the clipping region. This will be itself clipped to the limits of 
                                the video buffer. If this parameter is equal to NULL, the clipping region is 
                                equal to the limits of the video buffer.
                IN eBop         Binary operation. Only LBlit_Bop_S is supported if the pixel size of the 
                                buffer is over 32 bpp in one of the planes.
                IN poPlnWrtMsk  Define the pixel bits where the write operations are permit (0 = inhibit write, 
                                1 = permit write). This should have the same pixel format as the destination, 
                                The maximum pixel size supported is 32 bpp per plane. If this parameter is 
                                equal to NULL, the plane write mask is disabled (permit write on all bits).

Return Value:   LStatus_OK              The operation completed successfully.
                LStatus_FAIL            The operation failed.
                LStatus_INVALID_PARAM   Failed because a parameter is invalid.

Comments:       The buffer type must be video (LBuffer_Type_VIDEO).

                Supported destination buffer format:
                    Any (exceptions to be determined)

\************************************************************************************************************/
LAPI LStatus LBlit_SolidFill
(
    LBlit_Handle    hContext,
    LBuffer_Handle  hBuffer,
    LColor*         poColor,
    LRECT32*        poRect,
    LClipRgn_Handle hClipRgn,
    LBlit_Bop       eBop,
    LColor*         poPlnWrtMsk
);

/************************************************************************************************************\

Function:       LBlit_GradientFill

Description:    Fill a rectangle into the buffer with a gradient between two colors.

Parameters:     IN hContext     Handle to the LBlit context.
                IN hBuffer      Handle to the buffer where the gradient fill will be drawn.
                IN eGradientDir Gradient direction.
                IN poStartColor Starting color of the gradiant. This parameter cannot be NULL.
                IN poEndColor   Ending color of the gradiant. This parameter cannot be NULL.
                IN poRect       Pointer to the rectangle. The rectangle will be clipped to the boundaries of 
                                the clippping region. If this parameter is equal to NULL, the rectangle is 
                                equal to the limits of the video buffer.
                IN hClipRgn     Handle to the clipping region. This will be itself clipped to the limits of 
                                the video buffer. If this parameter is equal to NULL, the clipping region is 
                                equal to the limits of the video buffer.
                IN poAlphaBlend Alpha blending operation. If NULL, no alpha blending.
                IN poPlnWrtMsk  Define the pixel bits where the write operations are permit (0 = inhibit 
                                write, 1 = permit write). This should have the same pixel format as the 
                                destination. The maximum pixel size supported is 32 bpp per plane. If this 
                                parameter is equal to NULL, the plane write mask is disabled (permit write on 
                                all bits).

Return Value:   LStatus_OK              The operation completed successfully.
                LStatus_FAIL            The operation failed.
                LStatus_INVALID_PARAM   Failed because a parameter is invalid.

Comments:       The buffer type must be video (LBuffer_Type_VIDEO).       
                
                Multi-planes buffer is not supported.
                
                Supported destination buffer formats:
                    LPixelFormat_R8
                    LPixelFormat_B5G6R5
                    LPixelFormat_B5G5R5A1
                    LPixelFormat_B8G8R8A8

\************************************************************************************************************/
LAPI LStatus LBlit_GradientFill
(
    LBlit_Handle        hContext,
    LBuffer_Handle      hBuffer,
    LBlit_GradientDir   eGradientDir,
    LColor*             poStartColor,
    LColor*             poEndColor,
    LRECT32*            poRect,
    LClipRgn_Handle     hClipRgn,
    LBlit_AlphaBlend*   poAlphaBlend,
    LColor*             poPlnWrtMsk
);

/************************************************************************************************************\

Function:       LBlit_Copy

Description:    Copy a memory range between two linear buffers.

Parameters:     IN hContext     Handle to the LBlit context.
                IN hSrcBuffer   Handle to the source buffer.
                IN hDstBuffer   Handle to the destination buffer.
                IN uiSrcOffset  Byte offset in the source buffer where the copy start.
                IN uiDstOffset  Byte offset in the destination buffer where the copy start. 
                IN uiSize       Number of bytes to transfer. If uiSize is equal to 0 (zero), all the memory 
                                from the offset to the end of the buffer is copied (the end of the source or 
                                the end of the destination based on the first reached).

Return Value:   LStatus_OK              The operation completed successfully.
                LStatus_FAIL            The operation failed.
                LStatus_INVALID_PARAM   Failed because a parameter is invalid.

Comments:       The operation fails if the memory range is outside of one of the two buffers. It also fails 
                if the operation is on the same buffer (hSrcBuffer = hDstBuffer) and if the source memory 
                range overlaps the destination memory range.
                
                The buffers type must be linear (LBuffer_Type_LINEAR).
                
\************************************************************************************************************/
LAPI LStatus LBlit_Copy
(
    LBlit_Handle    hContext,
    LBuffer_Handle  hSrcBuffer,
    LBuffer_Handle  hDstBuffer,
    MUINT32         uiSrcOffset,
    MUINT32         uiDstOffset,
    MUINT32         uiSize
);

/************************************************************************************************************\

Function:       LBlit_Blit

Description:    Copy a rectangle between two video buffers with same pixel size. 

Parameters:     IN hContext         Handle to the LBlit context.
                IN hSrcBuffer       Handle to the source buffer.
                IN hDstBuffer       Handle to the destination buffer.
                IN poSrcPos         Position in the source buffer. The resulting source rectangles must be 
                                    inside the boundaries of the source buffer. It cannot be NULL.
                IN poDstRect        Rectangle into the destination buffer. The rectangle will be clipped to 
                                    the boundaries of the clippping region. It cannot be NULL.
                IN hClipRgn         Handle to the clipping region. This will be itself clipped to the limits 
                                    of the video buffer. If this parameter is equal to NULL, the clipping 
                                    region is equal to the limits of the video buffer.
                IN eBop             Binary operation. Only LBlit_Bop_S is supported if the pixel size of the 
                                    buffers are over 32 bpp in one of the planes or if the source and the 
                                    destination buffers are the same.
                IN poSrcColorKey    Color that defines the transparent pixels in the source buffer. If this 
                                    parameter is equal to NULL, there is no transparency. The transparency is
                                    not supported if the pixel size of the buffers is over 32 bpp in one of
                                    their planes.
                IN poPlnWrtMsk      Define the pixel bits where the write operations are permit (0 = inhibit 
                                    write, 1 = permit write). This should have the same pixel format as the 
                                    destination, The maximum pixel size supported is 32 bpp per plane. If 
                                    this parameter is equal to NULL, the plane write mask is disabled (permit
                                    write on all bits).

Return Value:   LStatus_OK              The operation completed successfully.
                LStatus_FAIL            The operation failed.
                LStatus_INVALID_PARAM   Failed because a parameter is invalid.

Comments:       The two buffers can be the same (overlap blit) but eBop must be equal to LBlit_Bop_S.

                The buffers type must be video (LBuffer_Type_VIDEO).             

                Supported source buffer format:
                    Any (exceptions to be determined)

                Supported destination buffer format:
                    Must have the same pixel size as the source. Can have different pixel format but no 
                    translation will be done.

\************************************************************************************************************/
LAPI LStatus LBlit_Blit
(
    LBlit_Handle    hContext,
    LBuffer_Handle  hSrcBuffer,
    LBuffer_Handle  hDstBuffer,
    LPOS32*         poSrcPos,
    LRECT32*        poDstRect,
    LClipRgn_Handle hClipRgn,
    LBlit_Bop       eBop,
    LBlit_ColorKey* poSrcColorKey,
    LColor*         poPlnWrtMsk
);

/************************************************************************************************************\

Function:       LBlit_Stretch

Description:    Copy a rectangle between two buffers with transformations like pixel format translation, 
                rotation, stretch or alpha blending.

Parameters:     IN hContext         Handle to the LBlit context.
                IN hSrcBuffer       Handle to the source buffer.
                IN hDstBuffer       Handle to the destination buffer.
                IN poSrcRect        Rectangle in the source buffer. It must be inside the boundaries of the 
                                    source buffer. It cannot be NULL.
                IN poDstRect        Rectangle into the destination buffer. The rectangle will be clipped to 
                                    the boundaries of the clipping region. If this parameter is equal to 
                                    NULL, the rectangle is equal to the limits of the video buffer.
                IN hClipRgn         Handle to the clipping region. This will be itself clipped to the limits 
                                    of the video buffer. If this parameter is equal to NULL, the clipping 
                                    region is equal to the limits of the video buffer.
                IN poSrcColorKey    Color that defines the transparent pixels in the source buffer. If this 
                                    parameter is equal to NULL, there is no transparency.
                IN poAlphaBlend     Alpha blending operation. If NULL, no alpha blending.
                IN eRotation        Rotation transformation.
                IN bFiltering       Apply bilinear filtering if the size ratio between the source and the 
                                    destination is not 1:1.
                IN poPlnWrtMsk      Define the pixel bits where the write operations are permit (0 = inhibit 
                                    write, 1 = permit write). This should have the same pixel format as the 
                                    destination, The maximum pixel size supported is 32 bpp per plane. If 
                                    this parameter is equal to NULL, the plane write mask is disabled (permit
                                    write on all bits).

Return Value:   LStatus_OK              The operation completed successfully.
                LStatus_FAIL            The operation failed.
                LStatus_INVALID_PARAM   Failed because a parameter is invalid.

Comments:       The buffers type must be video (LBuffer_Type_VIDEO).             

                Multi-planes buffer is not supported.
                
                The source and the destination buffers could be the same but at the same time the source 
                and the destination rectangles cannot overlap. 

                Supported source buffer formats:
                    LPixelFormat_P8
                    LPixelFormat_A8
                    LPixelFormat_L8A8
                    LPixelFormat_B5G6R5
                    LPixelFormat_B5G5R5A1
                    LPixelFormat_B4G4R4A4
                    LPixelFormat_U8Y8V8Y8
                    LPixelFormat_Y8U8Y8V8
                    LPixelFormat_B8G8R8A8

                Supported destination buffer formats:
                    LPixelFormat_R8
                    LPixelFormat_B5G6R5
                    LPixelFormat_B5G5R5A1
                    LPixelFormat_B8G8R8A8

\************************************************************************************************************/
LAPI LStatus LBlit_Stretch
(
    LBlit_Handle        hContext,
    LBuffer_Handle      hSrcBuffer,
    LBuffer_Handle      hDstBuffer,
    LRECT32*            poSrcRect,
    LRECT32*            poDstRect,
    LClipRgn_Handle     hClipRgn,
    LBlit_ColorKey*     poSrcColorKey,
    LBlit_AlphaBlend*   poAlphaBlend,
    LBlit_Rotation      eRotation,
    MBOOL32             bFiltering,
    LColor*             poPlnWrtMsk
);

typedef struct LSecureEngine_ContentKey LSecureEngine_ContentKey;

/************************************************************************************************************\

Function:       LBlit_Encrypt

Description:    Encrypt the content of a buffer.

Parameters:     IN hContext     Handle to the LBlit context.
                IN hSrcBuffer   Handle to the source buffer.
                IN hDstBuffer   Handle to the destination buffer.
                IN uiSrcOffset  Byte offset in the source buffer where the content start.
                IN uiDstOffset  Byte offset in the destination buffer where the content start.
                IN uiSize       Size in bytes of the content (multiple of 16 bytes)
                IN poContentKey Encryption key.
                IN uiCounter    Encryption counter associated to the content (64 bit wrap around).

Return Value:   LStatus_OK              The operation completed successfully.
                LStatus_FAIL            The operation failed.
                LStatus_INVALID_PARAM   Failed because a parameter is invalid.

Comments:       The buffers type can be linear and/or video.
                When using video buffer/s:
                    - the src and dst memory size buffer need to be of the same size.
                    - the offsets and size parameters needs to be set to zero (whole buffer copy only).
       
                The two buffers must have the same dimension and the same pixel format.
                
                Multi-planes buffer is not supported.

                The Media encryption counter increments automatically by one from the given value
                after the encryption of every 128 bit of plaintext data slice.
\************************************************************************************************************/
LAPI LStatus LBlit_Encrypt
(
    LBlit_Handle                hContext,
    LBuffer_Handle              hSrcBuffer,
    LBuffer_Handle              hDstBuffer,
    MUINT32                     uiSrcOffset,
    MUINT32                     uiDstOffset,
    MUINT32                     uiSize,
    LSecureEngine_ContentKey*   poContentKey,
    MUINT64                     uiCounter
);

/************************************************************************************************************\

Function:       LBlit_Decrypt

Description:    Decrypt the content of a buffer.

Parameters:     IN hContext                 Handle to the LBlit context.
                IN hSrcBuffer               Handle to the source buffer.
                IN hDstBuffer               Handle to the destination buffer.
                IN uiSrcOffset              Byte offset in the source buffer where the content start.
                IN uiDstOffset              Byte offset in the destination buffer where the content start.
                IN uiSize                   Size in bytes of the content (multiple of 16 bytes)
                IN poContentKey             Decryption key.
                IN uiCounter                Decryption counter associated to the content (64 bit wrap around).

Return Value:   LStatus_OK              The operation completed successfully.
                LStatus_FAIL            The operation failed.
                LStatus_INVALID_PARAM   Failed because a parameter is invalid.

Comments:       The buffers type can be linear and/or video.
                When using video buffer/s:
                    - the src and dst memory size buffer need to be of the same size.
                    - the offsets and size parameters needs to be set to zero (whole buffer copy only).
       
                The two buffers must have the same dimension and the same pixel format.

                Multi-planes buffer is not supported.
                
                The Media decryption counter increments automatically by one from the given value
                after the decryption of every 128 bit of encrypted data slice.
\************************************************************************************************************/
LAPI LStatus LBlit_Decrypt
(
    LBlit_Handle                hContext,
    LBuffer_Handle              hSrcBuffer,
    LBuffer_Handle              hDstBuffer,
    MUINT32                     uiSrcOffset,
    MUINT32                     uiDstOffset,
    MUINT32                     uiSize,
    LSecureEngine_ContentKey*   poContentKey,
    MUINT64                     uiCounter
);

/************************************************************************************************************\

Function:       LBlit_ConvertEXCH

Description:    Convert a host EXCH pixel format to a local pixel format.

Parameters:     IN hContext             Handle to the LBlit context.
                IN hSrcBuffer           Handle to the source buffer.
                IN hDstBuffer           Handle to the destination buffer.
                IN uiLinearOffset       Linear offset in bytes.
                IN uiNbLines            Number of lines
                IN bLinearInterleaved   Is the Linear buffer in interleave

Return Value:   LStatus_OK              The operation completed successfully.
                LStatus_FAIL            The operation failed.
                LStatus_INVALID_PARAM   Failed because a parameter is invalid.

Comments:


\************************************************************************************************************/
LAPI LStatus LBlit_ConvertEXCH
(
    LBlit_Handle    hContext,
    LBuffer_Handle  hSrcBuffer,
    LBuffer_Handle  hDstBuffer,
    MUINT32         uiLinearOffset,
    MUINT32         uiNbLines,
    MBOOL32         bLinearInterleaved
);

/************************************************************************************************************\

Function:       LClipRgn_Create

Description:    Create a clip region object and initializes it as empty.

Parameters:     OUT phClipRgn   Pointer on the handle to the clip region object. This parameter cannot be 
                                NULL.

Return Value:   LStatus_OK              The operation completed successfully.
                LStatus_FAIL            The operation failed.
                LStatus_OUT_OF_MEMORY   Failed because no more memory is available.
                LStatus_INVALID_PARAM   Failed because a parameter is invalid.

Comments:       When the handle is not used anymore, it must be destroyed using LClipRgn_Destroy.
                
\************************************************************************************************************/
LAPI LStatus LClipRgn_Create
(
    LClipRgn_Handle* phClipRgn
);

/************************************************************************************************************\

Function:       LClipRgn_Destroy

Description:    Destroy the clip object and all the resources attached to it.

Parameters:     IN hClipRgn     Handle to the clip region object to be destroyed.

Return Value:   LStatus_OK              The operation completed successfully.
                LStatus_FAIL            The operation failed.
                LStatus_INVALID_PARAM   Failed because a parameter is invalid.

Comments:       None.
                
\************************************************************************************************************/
LAPI LStatus LClipRgn_Destroy
(
    LClipRgn_Handle hClipRgn
);

/************************************************************************************************************\

Function:       LClipRgn_SetRect

Description:    Combine a rectangle with the clip region using the given binary operation.

Parameters:     IN hClipRgn     Handle to the clip region object.
                IN poRect       Pointer on the rectangle to set. If this parameter is equal to NULL, the 
                                rectangle will be considered empty.
                IN eBop         Binary operation between the rectangle and the clip region.

Return Value:   LStatus_OK              The operation completed successfully.
                LStatus_FAIL            The operation failed.
                LStatus_OUT_OF_MEMORY   Failed because no more memory is available.
                LStatus_INVALID_PARAM   Failed because a parameter is invalid.

Comments:       The binary operation acts as if the pixels inside of the rectangle and the clip region are 
                set to '1' and the pixels outside are set to '0'. Also, the rectangle is the source and the 
                clip region is the destination. 
                
                For example, to add the rectangle to the clip region, use LBlit_Bop_DSo (Source OR 
                Destination), to substract the rectangle, use the LBlit_Bop_DSna (NOT Source AND Destination) 
                or to empty a clip region use LBlit_Bop_0 (the rectangle is ignored with this binary 
                operation). 
                
                The minimum value of the coordinates of the rectangle is MINT32_MIN and the maximum value is 
                MINT32_MAX. If LClipRgn_SetRect is called with the binary operation LBlit_Bop_1, the 
                resulting clip region will be a single rectangle with the left-top coordinate set to 
                MINT32_MIN and right-bottom coordinate set to MINT32_MAX. So, when used for a drawing 
                operation, this clip region will act as if the clipping is disabled.
                
\************************************************************************************************************/
LAPI LStatus LClipRgn_SetRect
(
    LClipRgn_Handle hClipRgn,
    LRECT32*        poRect,
    LBlit_Bop       eBop
);

/************************************************************************************************************\

Function:       LClipRgn_SetClipRgn

Description:    Combine two clip regions using the given binary operation.

Parameters:     IN hClipRgnDst  Handle to the destination clip region object where the resulting clip region 
                                will be stored.
                IN hClipRgnSrc  Handle to the source clip region object. This clip region will not be 
                                modified. This parameter cannot be NULL nor be the same as hClipRgnDst even 
                                if the binary operation does not require a source (in such a case, use the 
                                function LClipRgn_SetRect).
                IN eBop         Binary operation between the two clip regions.

Return Value:   LStatus_OK              The operation completed successfully.
                LStatus_FAIL            The operation failed.
                LStatus_OUT_OF_MEMORY   Failed because no more memory is available.
                LStatus_INVALID_PARAM   Failed because a parameter is invalid.

Comments:       For more details on how to use the binary operations, see the function LClipRgn_SetRect.
                
\************************************************************************************************************/
LAPI LStatus LClipRgn_SetClipRgn
(
    LClipRgn_Handle hClipRgnDst,
    LClipRgn_Handle hClipRgnSrc,
    LBlit_Bop       eBop
);

/************************************************************************************************************\

Function:       LClipRgn_Move

Description:    Move the coordinates of the clip region by the given X and Y offsets.

Parameters:     IN hClipRgn     Handle to the clip region.
                IN iOffsetX     Offset by which the X coordinates of the clip region will be moved.
                IN iOffsetY     Offset by which the Y coordinates of the clip region will be moved.

Return Value:   LStatus_OK              The operation completed successfully.
                LStatus_FAIL            The operation failed.
                LStatus_INVALID_PARAM   Failed because a parameter is invalid.

Comments:       The resulting coordinate values will be saturated to MINT32_MIN and MINT32_MAX.
                
\************************************************************************************************************/
LAPI LStatus LClipRgn_Move
(
    LClipRgn_Handle hClipRgn,
    MINT32          iOffsetX,
    MINT32          iOffsetY
);

/************************************************************************************************************\

Function:       LClipRgn_GetBoundRect

Description:    Get the bounding rectangle of the clip region.

Parameters:     IN  hClipRgn    Handle to the clip region.
                OUT poRect      Pointer on the bounding rectangle. This parameter cannot be NULL. 

Return Value:   LStatus_OK              The operation completed successfully.
                LStatus_FAIL            The operation failed.
                LStatus_INVALID_PARAM   Failed because a parameter is invalid.

Comments:       If the clip region is empty, all the rectangle coordinates will be set to '0' (zero).
                
\************************************************************************************************************/
LAPI LStatus LClipRgn_GetBoundRect
(
    LClipRgn_Handle hClipRgn,
    LRECT32*        poRect
);

/************************************************************************************************************\

Function:       LClipRgn_GetEnumCount

Description:    Get the number of rectangles to be enumerated.

Parameters:     IN  hClipRgn    Handle to the clip region.
                OUT puiNbRect   Pointer on the number of rectangle to be enumerated. This parameter cannot be 
                                NULL.

Return Value:   LStatus_OK              The operation completed successfully.
                LStatus_FAIL            The operation failed.
                LStatus_INVALID_PARAM   Failed because a parameter is invalid.

Comments:       This function could be usefull, for example, when a memory buffer must be allocated to store 
                all the rectangles to be enumerated. If the count is equal to one, the bounding rectangle 
                returned by LClipRgn_GetBoundRect is the clip rectangle.
                
\************************************************************************************************************/
LAPI LStatus LClipRgn_GetEnumCount
(
    LClipRgn_Handle hClipRgn,
    MUINT32*        puiNbRect
);

/************************************************************************************************************\

Function:       LClipRgn_StartEnum

Description:    Start an enumeration of the list of rectangles that make up the clip region.

Parameters:     IN hClipRgn     Handle to the clip region.
                IN eDir         Determine the order in which the rectangles are to be enumerated.

Return Value:   LStatus_OK              The operation completed successfully.
                LStatus_FAIL            The operation failed.
                LStatus_INVALID_PARAM   Failed because a parameter is invalid.

Comments:       This function must be called to start an enumeration of the list of rectangles in order to 
                set the direction and to reset the internal counters.
                
\************************************************************************************************************/
LAPI LStatus LClipRgn_StartEnum
(
    LClipRgn_Handle  hClipRgn,
    LClipRgn_EnumDir eDir
);

/************************************************************************************************************\

Function:       LClipRgn_Enum

Description:    Enumerate the list of rectangles that make up the clip region.

Parameters:     IN  hClipRgn    Handle to the clip region.
                OUT poRect      Pointer on the rectangle to be filled by the enumeration. This parameter 
                                cannot be NULL

Return Value:   LStatus_OK              The operation completed successfully.
                LStatus_NO_MORE_DATA    The enumeration is terminated.
                LStatus_FAIL            The operation failed.
                LStatus_INVALID_PARAM   Failed because a parameter is invalid.

Comments:       There will never be a rectangle that intersect with another. The function LClipRgn_StartEnum 
                must be called once prior to this. When the enumeration is terminated, the function returns
                LStatus_NO_MORE_DATA and the rectangle is invalid and must not be used anymore.
                
                Example that show how to use the clip region enumeration:
                
                    void DoSomethingWithEachClipRectangle(LClipRgn_Handle hClipRgn)
                    {
                        LRECT32 oClipRect;

                        if (LClipRgn_StartEnum(hClipRgn, LClipRgn_EnumDir_ANY) == LStatus_OK)
                        {
                            while (LClipRgn_Enum(hClipRgn, &oClipRect) == LStatus_OK)
                            {
                                // Use the clip rectangle here.
                                // ...
                            }
                        }
                    }
                
\************************************************************************************************************/
LAPI LStatus LClipRgn_Enum
(
    LClipRgn_Handle hClipRgn,
    LRECT32*        poRect
);


// -----------------------------------------------------------------------------------------------------------
//                      I N L I N E S   A N D   T E M P L A T E   D E F I N I T I O N S
// -----------------------------------------------------------------------------------------------------------

#if defined (__cplusplus)
}
#endif

#endif  // #ifndef INC_LBLIT_H
