/************************************************************************************************************\

Module Name:    LAudioProcessor.h

Description:    Header declaration file of the audio processor Liberatus interface.

References:     LAudioProcessor.doc

    Copyright (c) 2015, Matrox Graphics Inc.
    All Rights Reserved.

\************************************************************************************************************/

#ifndef INC_LAUDIOPROCESSOR_H
#define INC_LAUDIOPROCESSOR_H

/* -----------------------------------------------------------------------------------------------------------
                                     I N C L U D E S   A N D   U S I N G S
   ---------------------------------------------------------------------------------------------------------*/
#include "Liberatus.h"
#include "LAudioFormat.h"

// -----------------------------------------------------------------------------------------------------------
//                                               O V E R V I E W
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

    An audio processor can be used to mix multiple audio streams, compensate for audio clock drift (WRT an
    audio output clock, an external time reference or a master audio input clock), resample audio, convert
    audio formats and adjust input/output gain.

                            +-------------------+
                            | External time ref |   <-- Timestamps for external time reference clock
                            +-------------------+        compensation.
                                      |
                                      |
                        +-------------+-------------------------------+
                        |                LAudioProcessor              |
                        |  +------------+                             |
       +-----------+    |  |            |                             |
       | LAudioIn0 |--->x--x-->|UDRB0|--x-->|ISP0|--->|PDRB0|         |
       +-----------+    |                                |            |
                        |                                v            |     +-----------+
        ...             |                               (+)-->|OSP|-->x---->| LAudioOut |
                        |  +------------+                ^      ^     |     +-----------+
       +-----------+    |  |            |                |      |     |           |
       | LAudioInN |--->x--x-->|UDRBN|--x-->|ISPN|--->|PDRBN|   +-----x<----------+
       +-----------+    |                                             |    Feedback timestamps for audio
                        +---------------------------------------------+    output clk compensation.


        [UDRB0..URDBN]: Unprocessed Data Ring Buffers 0..N. (Will be bypassed if no buffering is needed at
                         this stage).
        [ISP0..ISPN]:   Input Streams Processes 0..N.
        [PDRB0..PDRBN]: Processed Data Ring Buffers 0..N.
        OSP:            Output Stream Processes.
        (+):            Mixer.


\************************************************************************************************************/

/* -----------------------------------------------------------------------------------------------------------
                                     C O N S T A N T S   A N D   T Y P E S
   ---------------------------------------------------------------------------------------------------------*/

/**********************************************************************************************************\
                                                    Types
\**********************************************************************************************************/

typedef struct LAUDIOPROCESSOR_OBJECT*  LAudioProc_Handle;

/**********************************************************************************************************\

Enum:           LAudioProc_ProcessTypeHeader

Description:    Enumeration of the different audio process types.

Comments:       None.

\**********************************************************************************************************/
typedef enum tagLAudioProc_ProcessTypeHeader
{
    LAudioProc_ProcessTypeHeader_CLK_COMPENSATION,
    LAudioProc_ProcessTypeHeader_GAIN_ADJUSTMENT,
    LAudioProc_ProcessTypeHeader_FORMAT_CONVERSION,
    LAudioProc_ProcessTypeHeader_FORCE32 = 0x7FFFFFFF
} LAudioProc_ProcessTypeHeader;

/**********************************************************************************************************\

Enum:           LAudioProc_ClkCompCtrlMode

Description:    Enumeration of the different audio clock compensation control modes.
                 LAudioProc_ClkCompCtrlMode_TS_DRIFT:
                  Clock compensation is controlled by estimating input/output timestamps drift.
                 LAudioProc_ClkCompCtrlMode_DIRECT_CTRL:
                  Clock compensation is controlled manually.
Comments:       None.

\**********************************************************************************************************/
typedef enum tagLAudioProc_ClkCompCtrlMode
{
    LAudioProc_ClkCompCtrlMode_TS_DRIFT,
    LAudioProc_ClkCompCtrlMode_DIRECT_CTRL,
    LAudioProc_ClkCompCtrlMode_FORCE32 = 0x7FFFFFFF
} LAudioProc_ClkCompCtrlMode;

/**********************************************************************************************************\

Enum:           LAudioProc_StreamTypeHeader

Description:    Enumeration of the different audio processor stream types.

Comments:       None.

\**********************************************************************************************************/
typedef enum tagLAudioProc_StreamTypeHeader
{
    LAudioProc_StreamTypeHeader_STANDARD,
    LAudioProc_StreamTypeHeader_FORCE32 = 0x7FFFFFFF
} LAudioProc_StreamTypeHeader;

/************************************************************************************************************\

Structure:      LAudioProc_Stream

Description:    Audio processor stream descriptor.

                eType:                  Type header, must be LAudioProc_StreamTypeHeader_STANDARD.
                eFormat:                Format of the audio stream.
                apeStreamProcesses:     List of audio processes applied to the stream (passed by type header).
                uiNbProcesses:          Number of processes contained in apeStreamProcesses.
                uiTickReferenceFreq:    Frequency of the clock used to generate the timestamps of this stream.
                                        Example: set to 1000000 if timestamps are expressed in microseconds.
                                        Ignored for output stream.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLAudioProc_Stream
{
    LAudioProc_StreamTypeHeader     eType;
    LAudioFormat                    eFormat;
    LAudioProc_ProcessTypeHeader**  apeStreamProcesses;
    MUINT32                         uiNbProcesses;
    MUINT32                         uiTickReferenceFreq;
} LAudioProc_Stream;

/**********************************************************************************************************\

Enum:           LAudioProc_TimeRefTypeHeader

Description:    Enumeration of the different time reference types.

                LAudioProc_TimeRefType_EXTERNAL:        External type reference (ex: audio output clock,
                                                         video clock).
                LAudioProc_TimeRefType_INPUT:           Input time reference.

Comments:       None.

\**********************************************************************************************************/
typedef enum tagLAudioProc_TimeRefTypeHeader
{
    LAudioProc_TimeRefTypeHeader_EXTERNAL,
    LAudioProc_TimeRefTypeHeader_INPUT,
    LAudioProc_TimeRefTypeHeader_FORCE32 = 0x7FFFFFFF
} LAudioProc_TimeRefTypeHeader;

/************************************************************************************************************\

Structure:      LAudioProc_ExtTimeRefDesc

Description:    Description of external time reference.

                eType                           Type header, must be LAudioProc_TimeRefTypeHeader_EXTERNAL.
                uiId                            ID of this external time reference. Start at ID 0,
                                                 increment at every additionnal external time reference.
                uiTickReferenceFreq             Frequency of the clock used to generate the timestamps of
                                                 this external time reference. Example: set to 1000000 if
                                                 timestamps are expressed in microseconds.
                fNominalUpdatePeriod:           The nominal period, in ticks, of an update of timestamp on
                                                 this external time reference.

Comments:       Examples for fNominalUpdatePeriod: set to (1.0/60.0)*uiTickReferenceFreq if synchronization
                is done on a 60Hz video stream, set to (1024.0/48000.0)*uiTickReferenceFreq if synchronization
                is done on a 48kHz audio output playing buffers of 1024 samples.

\************************************************************************************************************/
typedef struct tagLAudioProc_ExtTimeRefDesc
{
    LAudioProc_TimeRefTypeHeader    eType;
    MUINT32                         uiId;
    MUINT32                         uiTickReferenceFreq;
    MFLOAT64                        fNominalUpdatePeriod;
} LAudioProc_ExtTimeRefDesc;

/************************************************************************************************************\

Structure:      LAudioProc_InputTimeRefDesc

Description:    Description of input time reference.

                eType                           Type header, must be LAudioProc_TimeRefTypeHeader_INPUT.
                uiInputStrmIdx                  Index of the input stream acting as time reference.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLAudioProc_InputTimeRefDesc
{
    LAudioProc_TimeRefTypeHeader    eType;
    MUINT32                         uiInputStrmIdx;
} LAudioProc_InputTimeRefDesc;

/**********************************************************************************************************\

Enum:           LAudioProc_CreateOptTypeHeader

Description:    Enumeration of the different audio processor create option types.

Comments:       None.

\**********************************************************************************************************/
typedef enum tagLAudioProc_CreateOptTypeHeader
{
    LAudioProc_CreateOptTypeHeader_STANDARD,
    LAudioProc_CreateOptTypeHeader_FORCE32 = 0x7FFFFFFF
} LAudioProc_CreateOptTypeHeader;

/************************************************************************************************************\

Structure:      LAudioProc_CreateOpt

Description:    Options for audio processor creation.

                eType:                      Type header, must be
                                             LAudioProc_CreateOptTypeHeader_STANDARD.
                hDevice:                    Liberatus device handle. Used to create output buffers.
                apeInputStreams:            List of length uiNbInputStreams containing input streams
                                             descriptors (passed by type header).
                uiNbInputStreams:           Number of streams in apeInputStreams.
                peOutputStream:             Output stream descriptor (passed by type header).
                uiNbOutputBuffers:          Number of output buffers to allocate.
                uiOutputBufferSize:         Size to allocate for output buffers.
                eOutputBufferType:          The type of LBuffer that will be created as output buffers.
                peOutputTsTimeRef:          Time reference used to generate output buffers timestamps
                                            (passed by type header). If MNULL, output timestamps
                                            will be generated using input stream 0 timestamps.
                                             For input time reference:
                                                - use LAudioProc_InputTimeRefDesc
                                                - set the index of the input used to generate output
                                                  timestamps in uiInputStrmIdx field.
                                             For external time reference:
                                                - use LAudioProc_ExtTimeRefDesc
                                                - Base time (timestamp of the first buffer to go out of the
                                                  audio processor through LAudioProc_GetNextBuffer) must be
                                                  set with LAudioProc_UpdateTimestamp before the first call to
                                                  LAudioProc_ProcessOutputStream.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLAudioProc_CreateOpt
{
    LAudioProc_CreateOptTypeHeader      eType;
    LDevice_Handle                      hDevice;
    LAudioProc_StreamTypeHeader**       apeInputStreams;
    MUINT32                             uiNbInputStreams;
    LAudioProc_StreamTypeHeader*        peOutputStream;
    MUINT32                             uiNbOutputBuffers;
    MUINT32                             uiOutputBufferSize;
    LBuffer_Type                        eOutputBufferType;
    LAudioProc_TimeRefTypeHeader*       peOutputTsTimeRef;
} LAudioProc_CreateOpt;

/************************************************************************************************************\

Structure:      LAudioProc_ClockCompensationOpt

Description:    Options clock compensation process.

                eType:                  Type header, must be LAudioProc_ProcessTypeHeader_CLK_COMPENSATION.
                peTimeRefDesc:          Time reference descriptor (passed by type header). Clock compensation
                                         is done with respect to this time reference. For clock compensation
                                         on output stream, only LAudioProc_TimeRefTypeHeader_EXTERNAL is
                                         accepted. Optional when eControlMode is
                                         LAudioProc_ClkCompCtrlMode_DIRECT_CTRL, only used for drift
                                         statistics with LAudioProc_GetInfo.
                uiMinCompensationInterval: Minimum interval, in samples, between each added/removed sample.
                                            Minimum value is 256. This can be used to balance the trade off
                                            between CPU load and compensated signal quality. As this value
                                            increases, the frequency response of the compensated signal will
                                            be less affected, at the expense of more CPU load. Since minimum
                                            value is 256, it means that the maximum clock drift that the
                                            process can compensate for is of 1/256=3906ppm. Increasing this
                                            value will therefore decrease the maximum clock drift
                                            compensation.

                eControlMode:           The mode of control of the clock compensation.
                fDirectCtrlLvl:         The level of the compensation to apply when eControlMode is
                                         LAudioProc_ClkCompCtrlMode_DIRECT_CTRL, from -1.0 to 1.0. This
                                         parameter can be updated dynamically with
                                         LAudioProc_SetDynamicProcessConfig. Ignored when eControlMode is
                                         LAudioProc_ClkCompCtrlMode_TS_DRIFT.
                                        -1.0: Max negative compensation level, for compensation of output
                                              clock slower than input clock.
                                        +1.0: Max positive compensation level, for compensation of output
                                              clock faster than input clock.
                                         0.0: Disable clock compensation.

Comments:

                peTimeRefDesc:

                Two modes of clock compensation are possible: clock compensation with respect to a master
                input (LAudioProc_InputTimeRefDesc), and clock compensation with respect to an external
                time reference like an audio output clock or a video clock (LAudioProc_ExtTimeRefDesc).
                For LAudioProc_InputTimeRefDesc:
                    - Set uiInputStrmIdx field to the index of the master input that is the time reference for
                      clock compensation.
                For LAudioProc_ExtTimeRefDesc:
                    - Set uiId to the ID of the external time reference. This ID must be used when updating
                      the timestamps with LAudioProc_UpdateTimestamp.
                    - The gap between updated timestamps (in ticks) should be close to 1/fExpectedTsUpdateRate
                      (in seconds).

Example:
                Clock compensation can be done at input and/or output stages:

                VideoOutput @ 59.999 Hz
                AudioInput1 @ 48.001 KHz
                AudioInput2 @ 47.990 KHz
                AudioInput3 @ 48.010 KHz

                              (1)       (2)
                               |         |
                AudioInput1----|---->+   |
                               |     |   |
                AudioInput2----|---->+---|---> AudioOutput(48.001 KHz)
                               |     |   |
                AudioInput3----|---->+   |
                               |         |

                To harmonize audio inputs clocks with a 60Hz external video reference, use
                LAudioProc_ExtTimeRefDesc structure as peTimeRefType and fExpectedTsUpdateRate to 60.0 on
                all input streams. Application must update the timestamp of every new video frame using
                LAudioProc_UpdateTimeStamp, passing the LAudioProc_ExtTimeRefDesc structure that describes
                the external time reference in peTimeRefType.

                Stage (1) will be the three following adjusted audio streams:
                    AudioInput1* @ (48000 * 59.999/60.0) = 47.9992 kHz --> Conversion = 48.001 -> 47.9992
                    AudioInput2* @ (48000 * 59.999/60.0) = 47.9992 kHz --> Conversion = 47.990 -> 47.9992
                    AudioInput3* @ (48000 * 59.999/60.0) = 47.9992 kHz --> Conversion = 48.001 -> 47.9992

                To compensate for audio output clock drift, use LAudioProc_ExtTimeRefDesc structure as
                peTimeRefType on output stream and set fExpectedTsUpdateRate to the
                expected buffer frequency (Ex: 1024 samples/buffer, 48kHz: set fExpectedTsUpdateRate to
                48000/1024 = 46.875Hz). Application must update the timestamp of every audio buffer played
                by the audio output (i.e. the time when the buffer began to play).

                Stage (2) will be the sum of AudioInput1*, AudioInput2* and AudioInput3* corrected from
                47.9992 kHz to 48.001 kHz.

\************************************************************************************************************/
typedef struct tagLAudioProc_ClockCompensationOpt
{
    LAudioProc_ProcessTypeHeader     eType;
    LAudioProc_TimeRefTypeHeader*    peTimeRefDesc;
    MUINT32                          uiMinCompensationInterval;
    LAudioProc_ClkCompCtrlMode       eControlMode;
    MFLOAT64                         fDirectCtrlLvl;
} LAudioProc_ClockCompensationOpt;

/************************************************************************************************************\

Structure:      LAudioProc_GainAdjustmentOpt

Description:    Options for gain adjustment process.

                eType:              Type header, must be LAudioProc_ProcessTypeHeader_GAIN_ADJUSTMENT.
                afChannelGains:     Array of gains of length uiNbChannels corresponding to the gains to apply
                                    to each channels of the stream.
                                    0.0 <= afChannelGains[N] <= 10.0
                                    afChannelGains[N] == 1.0 --> Gain of channel at index 'N' is not modified.
                uiNbChannels:       Number of channels of the stream.

Comments:       Can be updated dynamically with LAudioProc_SetDynamicConfig.

\************************************************************************************************************/
typedef struct tagLAudioProc_GainAdjustmentOpt
{
    LAudioProc_ProcessTypeHeader     eType;
    MFLOAT32*                        afChannelGains;
    MUINT32                          uiNbChannels;
} LAudioProc_GainAdjustmentOpt;

/************************************************************************************************************\

Structure:      LAudioProc_ChannelMap

Description:    Source-destination channel map.

                uiSrcChannelIdx:     Source channel index.
                uiDstChannelIdx:     Destination channel index.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLAudioProc_ChannelMap
{
    MUINT8  uiSrcChannelIdx;
    MUINT8  uiDstChannelIdx;
} LAudioProc_ChannelMap;

/************************************************************************************************************\

Structure:      LAudioProc_FormatConversionOpt

Description:    Options for format conversion process.

                eType:                  Type header, must be LAudioProc_ProcessTypeHeader_FORMAT_CONVERSION.
                eDestFormat:            Format toward which the stream will be converted. Must be coherent
                                         with paoChannelMaps. Can be used to specify resampling, bit depth
                                         conversion, etc.
                paoChannelMaps:         Pointer to an array that describes channel mapping between input and
                                         destination. Can be used to specify downmixing, channel replication
                                         or selection, etc. Must be coherent with the audio format of the
                                         stream and with eDestFormat. If MNULL, source channel mapping is
                                         preserved.
                uiChannelMapsLength:    Length of paoChannelMaps array.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLAudioProc_FormatConversionOpt
{
    LAudioProc_ProcessTypeHeader        eType;
    LAudioFormat                        eOutputFormat;
    LAudioProc_ChannelMap*              paoChannelMaps;
    MUINT32                             uiChannelMapsLength;
    // TODO: Endianness conversion ?
    // TODO: Signed / unsigned / float conversion ?
} LAudioProc_FormatConversionOpt;

/**********************************************************************************************************\

Enum:           LAudioProc_BufferDescTypeHeader

Description:    Enumeration of the different buffer descriptor types.

Comments:       None.

\**********************************************************************************************************/
typedef enum tagLAudioProc_BufferDescTypeHeader
{
    LAudioProc_BufferDescTypeHeader_STANDARD,
    LAudioProc_BufferDescTypeHeader_FORCE32 = 0x7FFFFFFF
} LAudioProc_BufferDescTypeHeader;

/************************************************************************************************************\

Structure:      LAudioProc_BufferDesc

Description:    Audio buffer descriptor.

                eType               Type header, must be LAudioProc_BufferInfoTypeHeader_STANDARD
                hBuffer             Handle to a LBuffer where the data is stored.
                uiStartOffset       The offset, in bytes, at which the data begins in the buffer.
                uiSize              The size in bytes of the data contained in the buffer after the start
                                    offset.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLAudioCodec_BufferDesc
{
    LAudioProc_BufferDescTypeHeader     eType;
    LBuffer_Handle                      hBuffer;
    MUINT32                             uiStartOffset;
    MUINT32                             uiSize;
} LAudioProc_BufferDesc;

/**********************************************************************************************************\

Enum:           LAudioProc_InfoTypeHeader

Description:    Enumeration of the different audio processor information types.

Comments:       None.

\**********************************************************************************************************/
typedef enum tagLAudioProc_InfoTypeHeader
{
    LAudioProc_InfoTypeHeader_BUFFER_INFO,
    LAudioProc_InfoTypeHeader_INTERNAL_BUFFER_INFO,
    LAudioProc_InfoTypeHeader_STREAM_STATE,
    LAudioProc_InfoTypeHeader_FORCE32 = 0x7FFFFFFF
} LAudioProc_InfoTypeHeader;

/************************************************************************************************************\

Structure:      LAudioProc_BufferInfo

Description:    Information on the input buffers provided to the LAudioProcessor module.

                eType                           Type header, must be LAudioProc_InfoTypeHeader_BUFFER_INFO.
                uiInputStreamIdx                Index of the input stream on which buffer information is
                                                 gathered. Must be set by the caller.
                uiRecommendedInputBufferSize    Recommended size, in bytes, of the buffers fed to the audio
                                                 processor with LAudioProc_ProcessInputStream.
                ePreferredInputBufferType       Preferred type of LBuffer fed to
                                                 LAudioProc_ProcessInputStream.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLAudioProc_BufferInfo
{
    LAudioProc_InfoTypeHeader    eType;
    MUINT32                      uiInputStreamIdx;
    MUINT32                      uiRecommendedInputBufferSize;
    LBuffer_Type                 ePreferredInputBufferType;
} LAudioProc_BufferInfo;


/************************************************************************************************************\

Structure:      LAudioProc_InternalBufferInfo

Description:    Information on the internal buffers used to store unprocessed and processed data for each
                 input stream.

                eType                           Type header, must be
                                                 LAudioProc_InfoTypeHeader_INTERNAL_BUFFER_INFO.
                uiInputStreamIdx                Input stream index. Must be set by the caller.
                uiUnprocessedDataBufTotSize     Total size of the unprocessed data internal buffer in bytes.
                uiProcessedDataBufTotSize       Total size of the processed data internal buffer in bytes.
                uiUnprocessedDataBufFreeSize    Free size of the unprocessed data internal buffer in bytes.
                uiProcessedDataBufFreeSize      Free size of the processed data internal buffer in bytes.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLAudioProc_InternalBufferInfo
{
    LAudioProc_InfoTypeHeader    eType;
    MUINT32                      uiInputStreamIdx;
    MUINT32                      uiUnprocessedDataBufTotSize;
    MUINT32                      uiProcessedDataBufTotSize;
    MUINT32                      uiUnprocessedDataBufFreeSize;
    MUINT32                      uiProcessedDataBufFreeSize;
} LAudioProc_InternalBufferInfo;

/************************************************************************************************************\

Structure:      LAudioProc_StreamState

Description:    Information on the state of an input/output stream. Drift is expressed with respect to 
                 the time reference of the stream (the expected clock rate of the stream) and the one of the
                 clock compensation process of the stream if any.

                eType                   Type header, must be LAudioProc_InfoTypeHeader_STREAM_STATE.
                uiStreamIdx             Index of the stream for which the state will be returned. Must be set
                                         by the caller. To get the state of the output stream, set to 
                                         numberOfInputStreams. Ex: 3 input streams, set to 3 to get the
                                         output stream state.
                fClockDrift             Clock drift: actual_clock / expected_clock
                                         = 1.0 --> Clock runs as expected.
                                         < 1.0 --> Clock slower than expected.
                                         > 1.0 --> Clock faster than expected.
                fClockDriftFromTarget   Ratio between actual clock drift and target clock drift:
                                         fClockDrift / targetClockDrift
                                         = 1.0 --> No drift, clock runs as fast as the target, or no clock
                                                    compensation on this stream.
                                         < 1.0 --> Clock slower than target.
                                         > 1.0 --> Clock faster than target.
                uiTotConsumedSamples    Total number of samples that were consumed by the processes of the
                                         stream.
                uiTotProcessedSamples   Total number of samples that were created by the processes of the
                                         stream.
                uiTotRemovedSamples     Total number of samples that were removed from the stream by the clock
.                                        compensation process.
                uiTotAddedSamples       Total number of samples that were added to the stream by the clock 
                                         compensation process.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLAudioProc_StreamState
{
    LAudioProc_InfoTypeHeader    eType;
    MUINT32                      uiStreamIdx;
    MFLOAT64                     fClockDrift;
    MFLOAT64                     fClockDriftFromTarget;
    MUINT64                      uiTotConsumedSamples;
    MUINT64                      uiTotProcessedSamples;
    MUINT64                      uiTotRemovedSamples;
    MUINT64                      uiTotAddedSamples;
} LAudioProc_StreamState;

/* -----------------------------------------------------------------------------------------------------------
                             G L O B A L   V A R I A B L E   R E F E R E N C E S
   ---------------------------------------------------------------------------------------------------------*/

/* -----------------------------------------------------------------------------------------------------------
                                                 U S A G E
   ---------------------------------------------------------------------------------------------------------*/

/* -----------------------------------------------------------------------------------------------------------
                        I N L I N E S   A N D   T E M P L A T E   D E F I N I T I O N S
   ---------------------------------------------------------------------------------------------------------*/

#if defined (__cplusplus)
extern "C" {
#endif

/************************************************************************************************************\
                                           Audio processor interface
\************************************************************************************************************/

/************************************************************************************************************\

Function:       LAudioProc_Create

Description:    Creates an audio processor.

Parameters:     IN  peCreateOpt             Options for creating an audio processor (passed by type header).
                OUT phProc                  Audio processor handle returned.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioProc_Create
(
    LAudioProc_CreateOptTypeHeader*    peCreateOpt,
    LAudioProc_Handle*                 phProc
);

/************************************************************************************************************\

Function:       LAudioProc_Destroy

Description:    Destroys an audio processor.

Parameters:     IN  hProc                   Handle to an audio processor.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.

Comments:       None.


\************************************************************************************************************/
LAPI LStatus LAudioProc_Destroy
(
    LAudioProc_Handle hProc
);

/************************************************************************************************************\

Function:       LAudioProc_ProcessInputStream

Description:    Apply input stream processes on the input stream at index 'uiStreamIdx'.
                 Input stream processes will be applied as long as:
                    1 - There is enough data to apply the stream's processes.
                    AND
                    2 - There is enough space in the processed data ring buffer to store the processed data.
                 Once these conditions are not met anymore, the remaining data, if any, will be copied in the
                 unprocessed data ring buffer. If there is not enough space in the unprocessed data ring
                 buffer to store all of the remaining data, puiConsumedBytes will be set accordingly.
                 Input buffer must contain complete audio samples, otherwise LStatus_INVALID_PARAM will be
                 returned.
                 Example for a 16-bit stereo buffer:
                        Valid:   LLRRLLRRLLRRLLRR
                        Invalid: LLRRLLRRLLRRLL
                        Invalid: LLRRLLRRLLRRLLR

Parameters:     IN  hProc                   Handle to an audio processor.
                IN  uiStreamIdx             Index of the input stream in which the data will be copied.
                IN  peBufferInfo            Information on the buffer containing audio data to process(passed
                                             by type header). Buffer can be empty, in which case only the
                                             internally buffered data will be processed, if any.
                IN  uiTs                    Timestamp, in ticks, associated with the first audio sample
                                             contained in hBuffer, after the start offset, if there is one.
                OUT puiConsumedSize         Size of the data consumed from the input buffer in bytes.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.
                LStatus_OUT_OF_RESOURCES    Function failed because internal buffer of the given stream is 
                                            full. Application needs to process data stored inside the internal  
                                            buffer by using LAudioProc_ProcessOutputStream.
                LStatus_END_OF_STREAM       End of stream has been signaled on the given stream through
                                             LAudioProc_EndStream and all the data buffered in the internal
                                             data ring buffer has been processed.
                TODO: Should it return LStatus_NO_MORE_DATA when internal buffer is empty?

Comments:       Once LAudioProc_EndStream has been called on the given stream, this call is only used to
                 process the data buffered in the unprocessed data ring buffer. Any data set in peBufferInfo
                 will be ignored. LStatus_END_OF_STREAM is only returned once the unprocessed data ring buffer
                 has been emptied.

\************************************************************************************************************/
LAPI LStatus LAudioProc_ProcessInputStream
(
    LAudioProc_Handle                   hProc,
    MUINT32                             uiStreamIdx,
    LAudioProc_BufferDescTypeHeader*    peBufferDesc,
    MUINT64                             uiTs,
    MUINT32*                            puiConsumedSize
);

/************************************************************************************************************\

Function:       LAudioProc_ProcessOutputStream

Description:    Performs a process buffer operation on the oldest internally stored data of all input streams.
                 Upon success, a new buffer is available to be collected with LAudioProc_GetNextBuffer.

Parameters:     IN  hProc                   Handle to an audio processor.
                IN  uiSamplesToProcess      Number of samples to process. Set to 0 to use the default (1024
                                             samples). This can be used to reduce the input/output latency.
                                             Output buffers must be able to hold the requested number of
                                             samples, otherwise LStatus_INVALID_PARAM will be returned.
                                             maxNbSamples = (LAudioProc_CreateOpt.uiOutputBufferSize)
                                                              / (bytes_per_sample)
                                             0 <= uiSamplesToProcess <= maxNbSamples

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.
                LStatus_NO_MORE_DATA        Function failed because not enough data is stored internally to
                                             perform a process buffer operation on the specified number of
                                             samples. Application needs to send more audio data with
                                             LAudioProc_ProcessInputStream. Application can query information
                                             on the internal buffer fullness of each input stream by using
                                             LAudioProc_GetInfo with LAudioProc_InternalBufferInfo.
                LStatus_OUT_OF_RESOURCES    Function failed because there is no more place to store the
                                             processed buffer. This is likely because application does not
                                             call LAudioProc_GetNextBuffer fast enough.
                LStatus_END_OF_STREAM       All input streams have been ended with LAudioProc_EndStream and
                                             all the internally stored audio data in the unprocessed/processed
                                             data ring buffers of every input streams have been processed.

Comments:       If an external time reference is used to generate output timestamps, the base time of this
                time reference (i.e. the first output timestamp) must be set with LAudioProc_UpdateTimestamp
                prior to the first call to LAudioProc_ProcessOutputStream. LStatus_FAIL will be returned
                otherwise.

\************************************************************************************************************/
LAPI LStatus LAudioProc_ProcessOutputStream
(
    LAudioProc_Handle                   hProc,
    MUINT32                             uiSamplesToProcess
);

/************************************************************************************************************\

Function:       LAudioProc_GetNextBuffer

Description:    Retrieves the next processed audio buffer.

Parameters:     IN  hProc                   Handle to an audio processor.
                IN  uiTimeout               Delay to wait, in milliseconds, before timeout:
                                             - 0                 = no wait
                                             - LINFINITE_TIMEOUT = wait until a processed buffer is available
                OUT peBufferDesc            Buffer descriptor (passed by type header).
                OUT puiTs                   Timestamp, in ticks, associated with the first sample
                                             contained in the collected buffer. Can be MNULL if unused.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_END_OF_STREAM       The stream is ended and no more buffers are available.
                                             This is returned once all the following conditions are met:
                                              1 - LAudioProc_EndStream has been called on every input stream.
                                              2 - All pending input data of every input stream has been
                                                   processed.
                                              3 - All processed output buffers have been collected.
                LStatus_TIMEOUT             Timeout expired. No new data is available yet.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioProc_GetNextBuffer
(
    LAudioProc_Handle                   hProc,
    MUINT32                             uiTimeout,
    LAudioProc_BufferDescTypeHeader*    peBufferDesc,
    MUINT64*                            puiTs
);

/************************************************************************************************************\

Function:       LAudioProc_ReleaseBuffer

Description:    Releases a processed buffer back to the audio processor.

Parameters:     IN  hProc                   Handle to an audio processor.
                IN  hBuffer                 Handle to a LBuffer returned by LAudioProc_GetNextBuffer to free
                                             back to the LAudioProcessor.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioProc_ReleaseBuffer
(
    LAudioProc_Handle   hProc,
    LBuffer_Handle      hBuffer
);

/************************************************************************************************************\

Function:       LAudioProc_GetInfo

Description:    Retrieves various information about the audio processor.

Parameters:     IN    hProc                 Handle to an audio processor.
                INOUT peInfo                Information to query (passed by type header).

Return Value:   LStatus_OK                  Function completed succesfully.
                LStatus_INFO_NOT_AVAILABLE  Requested information is currently unavailable.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioProc_GetInfo
(
    LAudioProc_Handle               hProc,
    LAudioProc_InfoTypeHeader*      peInfo
);

/************************************************************************************************************\

Function:       LAudioProc_SetDynamicProcessConfig

Description:    Sets process configuration dynamically.

Parameters:     IN    hProc                 Handle to an audio processor.
                IN    uiStreamIdx           Index of the stream on which to apply the dynamic process config. 
                                             Set to the number of input streams to select the output stream.
                IN    peProcessConfig       Config to update (passed by type header).

Return Value:   LStatus_OK                  Function completed succesfully.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters. 

Comments:       Not all process configurations can be updated dynamically. See comments in
                process options structures (LAudioProc_ProcessTypeHeader).

\************************************************************************************************************/
LAPI LStatus LAudioProc_SetDynamicProcessConfig
(
    LAudioProc_Handle               hProc,
    MUINT32                         uiStreamIdx,
    LAudioProc_ProcessTypeHeader*   peProcessConfig
);

/************************************************************************************************************\

Function:       LAudioProc_EndStream

Description:    Notifies the audio processor that no more data will be sent through
                 LAudioProc_ProcessInputStream for the given input stream.

Parameters:     IN  hProc                   Handle to an audio processor.
                IN  iStreamIdx              Index of the input stream to end. Set to -1 to end all streams
                                             immediately, discarding all buffered data and all processed
                                             buffers ready to be collected with LAudioProc_GetNextBuffer.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.

Comments:       If iStreamIdx is not set to -1, the given stream will produce silence as output data until
                 all other streams are ended.
                Once this is called on stream iStreamIdx, it will become impossible to send more data with
                 LAudioProc_ProcessInputStream on this stream. Application must then destroy the
                 LAudioProcessor handle and create a new one.

\************************************************************************************************************/
LAPI LStatus LAudioProc_EndStream
(
    LAudioProc_Handle               hProc,
    MINT32                          iStreamIdx
);

/************************************************************************************************************\

Function:       LAudioProc_UpdateTimeStamp

Description:    Used by application to update the timestamp of the external time references that are used in
                clock compensation processes. Also, when output timestamps are generated with an external
                time reference, this must be used to set the base time of this time reference (i.e. the first
                output timestamp) prior to the first call to LAudioProc_ProcessOutputStream.

Parameters:     IN  hProc                   Handle to an audio processor.
                IN  uiExtTimeRefId          ID of the external time reference for which the timestamp is to be
                                             updated.
                IN  uiTs                    Timestamp in ticks.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.

Comments:       Used to update the external time references timestamps for clock compensation.
                 Examples: if synchronization is done on a video clock, a timestamp should be provided
                 here for every video frame. If synchronization is done on an audio output clock, a timestamp
                 should be provided here for every audio output buffer.

\************************************************************************************************************/
LAPI LStatus LAudioProc_UpdateTimeStamp
(
    LAudioProc_Handle               hProc,
    MUINT32                         uiExtTimeRefId,
    MUINT64                         uiTs
);

#if defined (__cplusplus)
}
#endif

#endif  // INC_LAUDIOPROCESSOR_H
