/************************************************************************************************************\

Module Name:    LAudioOut.h

Description:    The LAudioOut module of the API provides the programming interface for the audio output 
                hardware and its associated output devices.

References:     LAudioOut.doc Revision 0.11.

    Copyright (c) 2016, Matrox Graphics Inc.
    All Rights Reserved.

\************************************************************************************************************/

#ifndef INC_LAUDIOOUT_H
#define INC_LAUDIOOUT_H

// -----------------------------------------------------------------------------------------------------------
//                                   I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "LAudio.h"
#include "LVideoOut.h"
#include "LAudioFormat.h"

typedef struct LAUDIOOUT_OBJECT*     LAudioOut_Handle;

// -----------------------------------------------------------------------------------------------------------
//                                                U S A G E
// -----------------------------------------------------------------------------------------------------------

/*  Playback audio buffer on an audio output.

    #include "Liberatus.h"
    #include "LBuffer.h"
    #include "LAudioOut.h"

    LStatus GetNextPCMBuffer(LDevice_Handle hDevice, LBuffer_Handle hAudioBuffer, MUINT32 uiBufferSize);
    LStatus ReleasePCMBuffer(LDevice_Handle hDevice, LBuffer_Handle hAudioBuffer);


    int main()
    {
        LDevice_Handle      hDevice         = MNULL;
        LBuffer_Handle      hAudioBuffer0   = MNULL;
        LBuffer_Handle      hAudioBuffer1   = MNULL;
        LAudioOut_Handle    hAudioOut       = MNULL;
        LAudioFormat        eAudioFormat;
        MUINT32             uiBufferSize    = 0;
        MUINT32             uiBufferIndex   = 0;

        Liberatus_Load();
        
        // Get primary device handle.
        hDevice = Liberatus_GetDevice(0);              

        // Get handle on first audio output interface.
        LAudioOut_GetHandle(hDevice, 
                            0,                                  // First interface using default signal type
                            LAccessMode_READWRITE, 
                            &hAudioOut);

        // Enum audio format.
        LAudioOut_EnumSupportedAudioFormat( hAudioOut,
                                            0,                  // Use the first available AudioFormat
                                            &eAudioFormat);

        // Set buffer attributes (2048 frames per buffer) for the playback.
        LAudioFormat_ComputeBufferSizeBasedOnFrame( eAudioFormat,
                                                    2048,                   // 2048 frames per buffer
                                                    &uiBufferSize);


        // Set buffer attributes for the audio playback session.
        LAudioOut_SetBufferAttributes(hAudioOut,
                                      eAudioFormat,
                                      uiBufferSize);

                                     
        for (uiBufferIndex = 0; uiBufferIndex < 100; uiBufferIndex++)
        {
            if (uiBufferIndex % 2)
            {
                // Get audio buffer driver audio FIFO.
                GetNextPCMBuffer(hDevice,
                                 hAudioBuffer0,
                                 uiBufferSize);

                if (uiBufferIndex == 0)
                {
                    // Send the first audio buffer for playback.
                    LAudioOut_SetBuffer(hAudioOut,
                                        hAudioBuffer0,
                                        MFALSE);            // Do NOT block CPU thread

                    // Start audio playback. Do it only once at the beginning of the playback session.
                    LAudioOut_StartPlayback(hAudioOut);
                }
                else
                {
                    // Send audio buffer for playback.
                    LAudioOut_SetBuffer(hAudioOut,
                                        hAudioBuffer0,
                                        MTRUE);             // Block CPU thread

                    // Release previous buffer.
                    ReleasePCMBuffer(hDevice,
                                     hAudioBuffer1);
                }
            }
            else
            {
                // Get audio buffer from driver audio FIFO.
                GetNextPCMBuffer(hDevice,
                                 hAudioBuffer1,
                                 uiBufferSize);

                // Send audio buffer for playback.
                LAudioOut_SetBuffer(hAudioOut,
                                    hAudioBuffer1,
                                    MTRUE);             // Block CPU thread

                // Release previous buffer.
                ReleasePCMBuffer(hDevice,
                                 hAudioBuffer0);
            }
        }
        
        // End audio playback.
        LAudioOut_StopPlayback(hAudioOut);

        // Release audio output interface.
        LAudioOut_ReleaseHandle(hAudioOut);

        Liberatus_UnLoad();
    }
*/

// -----------------------------------------------------------------------------------------------------------
//                                               O V E R V I E W
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\
    The main purpose of the Audio Output module is to provide the functionality to output (playback) an audio 
    output stream from the Matrox SV2 system.

    The module has services to:
        - enumerate audio output paths and their audio capabilities
        - configure the available audio output path in a specific mode to handle specific audio buffers
        - control the playback of audio buffers
        - retrieve the state of the interfaces
        - be notified of playback events
        - adjust dynamic controls (such as output volume level)

    Based on the hardware configuration, the module can handle analog and/or digital audio output stream 
    playback in mono, stereo, or multi-channels. It can also handle PCM (uncompressed) and/or compressed audio 
    output streams.

    The module can handle multiple audio output streams at a time.

\************************************************************************************************************/


// -----------------------------------------------------------------------------------------------------------
//                                   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Enum:           LAudioOut_ControlId

Description:    Enumeration defining all possible audio output control type values.

Comments:       - Each audio output control has a minimal, a maximal, a default value and also a step value 
                  that can be used compute valid value to modify the control.  See the LAudioOut_ControlInfo 
                  structure or the LAudioOut_SetControlValue service for more details.

\************************************************************************************************************/
typedef enum
{
    LAudioOut_ControlId_INVALID                = 0x00000000,   // Invalid value.
    LAudioOut_ControlId_FORCE32                = 0x7FFFFFFF,   // Dummy value to force enumeration to use 
                                                               //  32-bits.
} LAudioOut_ControlId;

/************************************************************************************************************\

Enum:           LAudioOut_Event

Description:    Enumerates all possible playback events.

Comments:       None.

\************************************************************************************************************/
typedef enum
{
    LAudioOut_Event_INVALID              = 0x00000000,   // Invalid value.
    LAudioOut_Event_START_PENDING        = 0x00000001,   // Playback started but it's waiting on an external 
                                                         //  event to start playing.
    LAudioOut_Event_STARTED              = 0x00000002,   // Playback already started, audio data 
                                                         //  are being played.
    LAudioOut_Event_STOPPED              = 0x00000004,   // Playback has stopped, but it's still 
                                                         //  initialized and can be restarted.
    LAudioOut_Event_STATE_CHANGED        = 0x00000008,   // Playback state has changed.
    LAudioOut_Event_OUT_OF_BUFFER        = 0x00000020,   // Application was not fast enough to call
                                                         // LAudioOut_SetBuffer in time.
    LAudioOut_Event_FORCE32              = 0x7FFFFFFF    // Dummy value to force to use 32-bits. 
} LAudioOut_Event;

/************************************************************************************************************\

Struct:         LAudioOut_ControlInfo

Description:    Structure defining a audio output control information.

Comments:       - A control is an attribute of the audio output device with a default value, a minimal value, 
                  maximal value, and a step value.
                - The minimal value and the step value should be used to compute all valid values within the 
                  range of supported control values.
                - For more information on available controls, see LAudioOut_ControlId.
                - To manipulate the controls value, see the LAudioOut_ControlInfo, LAudioOut_GetControlValues 
                  or LAudioOut_SetControlValues.
                - To get the current value of a control use the LAudioOut_GetControlValues service.


\************************************************************************************************************/
typedef struct tagLAudioOut_ControlInfo
{
    MINT32  iMinimum;   // Minimum value.
    MINT32  iMaximum;   // Maximum value.
    MINT32  iStep;      // Step value.
    MINT32  iDefault;   // Default value.
} LAudioOut_ControlInfo;

// -----------------------------------------------------------------------------------------------------------
//                         G L O B A L   F U N C T I O N   D E C L A R A T I O N S
// -----------------------------------------------------------------------------------------------------------


#if defined (__cplusplus)
extern "C" {
#endif

/************************************************************************************************************\

Function:       LAudioOut_GetCount

Description:    Gets the number of independent audio outputs available for the specified device.
                
Parameters:     IN  hDev            Device handle.
                OUT puiCount        Pointer to an MUINT32 variable to be filled with the number of independent 
                                    audio outputs available.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioOut_GetCount(
                LDevice_Handle      hDev, 
                MUINT32*            puiCount); 


/************************************************************************************************************\

Function:       LAudioOut_GetHandle

Description:    Gets the audio output handle for the specified independent audio output device.

Parameters:     IN  hDev            Device handle.
                IN  uiIndex         Index of the independent audio output for which to return the handle.
                IN  eAccessMode     Requested access mode.  Must be a value from LAccessMode.
                OUT phAudioOut      Pointer to an LAudioOut_Handle variable where the audio output device
                                    handle is written.  MNULL if unsuccessful.


Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_UNSUPPORTED         Invalid audio output device index.  No hardware associated.
                LStatus_OUT_OF_RESOURCES    Specified audio output device can't be used by the Liberatus
                                            applications because it's already being used by the system.
                LStatus_RESOURCES_BUSY      Audio output handle couldn't be returned because the audio output
                                            is busy.

Comments:       - If eAccessMode is LAccessMode_READONLY, functions that modify the audio output can't be 
                  called.
                - If eAccessMode is LAccessMode_READWRITE or LAccessMode_READWRITE_EXCLUSIVE, functions that 
                  modify the audio output can be called.
                - Only one handle with LAccessMode_READWRITE_EXCLUSIVE read/write access mode can exist at 
                  any one time. If a handle with read/write access has already been granted to a caller, 
                  any attempt to get another handle with read/write access returns a failure.

\************************************************************************************************************/
LAPI LStatus LAudioOut_GetHandle(
                LDevice_Handle          hDev, 
                MUINT32                 uiIndex,
                LAccessMode             eAccessMode,
                LAudioOut_Handle*       phAudioOut);


/************************************************************************************************************\

Function:       LAudioOut_ReleaseHandle

Description:    Releases a handle to an independent audio output device.

Parameters:     IN  hAudioOut           Handle to the audio output device.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioOut_ReleaseHandle(
                LAudioOut_Handle    hAudioOut);


/************************************************************************************************************\

Function:       LAudioOut_EnumSignalType

Description:    Enumerates all supported audio output signal types. 

Parameters:     IN  hAudioOut           Handle to the audio output device.
                IN  uiSignalIndex       Index of the signal type to enumerate.
                OUT peSignalType        Pointer to an LAudio_SignalType object where the enumerated signal
                                        types are written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_NO_MORE_DATA    No signal type value is available for the specified index.
                
Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioOut_EnumSignalType(
                LAudioOut_Handle        hAudioOut,
                MUINT32                 uiSignalIndex,
                LAudio_SignalType*      peSignalType);


/************************************************************************************************************\

Function:       LAudioOut_GetDefaultSignalType

Description:    Gets the default signal type of the audio output device.

Parameters:     IN  hAudioOut           Handle to the audio output device.
                OUT peSignalType        Pointer to an LAudio_SignalType object where the enumerated signal
                                        types are written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - The default audio output signal type device may change after an output source detection.

\************************************************************************************************************/
LAPI LStatus LAudioOut_GetDefaultSignalType(
                LAudioOut_Handle        hAudioOut,
                LAudio_SignalType*      peSignalType);


/************************************************************************************************************\

Function:       LAudioOut_SetSignalType

Description:    Sets the audio output device signal type of the audio output device.

Parameters:     IN  hAudioOut           Handle to the audio output device.
                IN  eSignalType         Audio signal type to set for the independent audio output.  Must be a 
                                        value from the LAudio_SignalType.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAudioOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - Setting the signal type may reserve and program some hardware resources that are shared 
                  with other audio output device.

\************************************************************************************************************/
LAPI LStatus LAudioOut_SetSignalType(
                LAudioOut_Handle    hAudioOut,
                LAudio_SignalType   eSignalType);


/************************************************************************************************************\

Function:       LAudioOut_GetSignalType

Description:    Gets the current audio output device signal type.

Parameters:     IN  hAudioOut           Handle to the audio output device.
                OUT peSignalType        Pointer to an LAudio_SignalType object where the enumerated signal
                                        types are written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioOut_GetSignalType(
                LAudioOut_Handle        hAudioOut,
                LAudio_SignalType*      peSignalType);


/************************************************************************************************************\

Function:       LAudioOut_EnumSupportedAudioFormat

Description:    Enumerates the audio playback formats supported by the current output device.

Parameters:     IN  hAudioOut           Handle to the audio output device.
                IN  uiIndex             Index of the audio format to be enumerated.
                OUT peAudioFormat       Pointer to an LAudioFormat object to be filled with the enumerated
                                        audio format.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_NO_MORE_DATA    No audio format value is available for the specified index.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioOut_EnumSupportedAudioFormat(
                LAudioOut_Handle    hAudioOut, 
                MUINT32             uiIndex,
                LAudioFormat*       peAudioFormat);


/************************************************************************************************************\

Function:       LAudioOut_SetBufferAttributes

Description:    Sets buffer attributes for the audio playback.

Parameters:     IN  hAudioOut           Handle to the audio output device.
                IN  eAudioFormat        Audio format to set. Must be a value from the LAudioFormat 
                                        enumeration. 
                IN  uiBufferSize        Size, in byte, of the playback buffer.
                
Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAudioOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - Buffer attributes can't be changed when playback is started.
                
\************************************************************************************************************/
LAPI LStatus LAudioOut_SetBufferAttributes(
                LAudioOut_Handle    hAudioOut,
                LAudioFormat        eAudioFormat,
                MUINT32             uiBufferSize);


/************************************************************************************************************\

Function:       LAudioOut_GetBufferAttributes

Description:    Gets the current playback audio buffer attributes.

Parameters:     IN  hAudioOut           Handle to audio output device.
                OUT peAudioFormat       Pointer to a LAudioFormat variable to be filled with the current
                                        audio format of the audio buffer for playback. 
                OUT puiBufferSize       Pointer to a MUINT32 variable to be filled with the size, in byte,
                                        of the playback buffer.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioOut_GetBufferAttributes(
                LAudioOut_Handle    hAudioOut,
                LAudioFormat*       peAudioFormat,
                MUINT32*            puiBufferSize);


/************************************************************************************************************\

Function:       LAudioOut_StartPlayback

Description:    Starts the audio playback.

Parameters:     IN  hAudioOut           Handle to the audio output device.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAudioOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - A playback audio buffer should have been previously submitted via the LAudioOut_SetBuffer 
                  service, otherwise this call will fail.

\************************************************************************************************************/
LAPI LStatus LAudioOut_StartPlayback(
                LAudioOut_Handle    hAudioOut);


/************************************************************************************************************\

Function:       LAudioOut_StartPlaybackOnVideo

Description:    Starts the audio playback session synchronized with a video playback.

Parameters:     IN  hAudioOut       Handle to the audio output device.
                IN  hVideoOut       Handle to the video output device. 

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAudioOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - Service used for the audio/video synchronization. It enables the synchronized playback of 
                  an audio and a video output to start.
                - A playback audio buffer should have been previously submitted via the LAudioOut_SetBuffer 
                  service, otherwise this call won't have any effect.

                
\************************************************************************************************************/
LAPI LStatus LAudioOut_StartPlaybackOnVideo(
                LAudioOut_Handle    hAudioOut,
                LVideoOut_Handle    hVideoOut);

/************************************************************************************************************\

Function:       LAudioOut_StopPlayback

Description:    Stops the audio playback.

Parameters:     IN  hAudioOut           Handle to the audio output device.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAudioOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - The current playback audio buffer is discarded when playback is stopped. No further data is 
                  played.

\************************************************************************************************************/
LAPI LStatus LAudioOut_StopPlayback(
                LAudioOut_Handle    hAudioOut);

/************************************************************************************************************\

Function:       LAudioOut_ScheduleStartPlayback

Description:    Schedules the start of the audio playback by adding the required commands to the passed in the 
                device thread object.

Parameters:     IN  hAudioOut           Handle to audio output device.
                IN  hDeviceThread       Device thread handle to use to start audio playback.
                
Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAudioOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_TIMEOUT         Function exited after a timeout.

Comments:       - A playback audio buffer should have been previously submitted via the 
                  LAudioOut_ScheduleSetBuffer service, otherwise this call won't have any effect.
                - LStatus_TIMEOUT is returned when the wait command can't be added to the device thread 
                  because it's full and the timeout period expired before space became available.


\************************************************************************************************************/ 
LAPI LStatus LAudioOut_ScheduleStartPlayback(
                LAudioOut_Handle        hAudioOut,
                LDeviceThread_Handle    hDeviceThread);


/************************************************************************************************************\

Function:       LAudioOut_ScheduleStartPlaybackOnVideo

Description:    Schedules the start of the synchronized audio and video playback by adding the required 
                commands to the passed in device thread object.

Parameters:     IN  hAudioOut       Handle to audio output device.
                IN  hVideoOut       Handle to video output device.
                IN  hDeviceThread   Device thread handle to use to start the audio playback.
                IN  uiTimeout_ms    Device thread full queue wait timeout.
                
Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAudioOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_TIMEOUT         Function exited after a timeout.

Comments:       - A playback audio buffer should have been previously submitted via the 
                  LAudioOut_ScheduleSetBuffer service, otherwise this call won't have any effect.

\************************************************************************************************************/ 
LAPI LStatus LAudioOut_ScheduleStartPlaybackOnVideo(
                LAudioOut_Handle        hAudioOut,
                LVideoOut_Handle        hVideoOut,
                LDeviceThread_Handle    hDeviceThread,
                MUINT32                 uiTimeout_ms);

/************************************************************************************************************\

Function:       LAudioOut_SetBuffer

Description:    Sets the next available audio buffer for playback.

Parameters:     IN  hAudioOut           Handle to audio output device.
                IN  hBuffer             Handle to the next available buffer for playback.
                IN  uiPayloadSize       The size of the effective data contained in hBuffer. The size can't
                                        be larger than the buffer size configured through
                                        LAudioOut_SetBufferAttributes.
                IN  bBlocking           MTRUE to force the service to not return until the newly set buffer
                                        is actually used (also meaning that the previous buffer is no longer
                                        used), MFALSE to return immediately after the new buffer is set even
                                        if it's not used yet.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAudioOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_CANCELED        Function exited. No playback in progress.

Comments:       - If the playback session isn't running, this service won't block even if the bBlocking flag 
                  has been set.
                - When bBlocking is MFALSE, LStatus_FAIL will be returned if LAudioOut_SetBuffer is called 
                  twice during the time that one buffer is being used.

\************************************************************************************************************/ 
LAPI LStatus LAudioOut_SetBuffer(
                LAudioOut_Handle        hAudioOut,
                LBuffer_Handle          hBuffer,
                MUINT32                 uiPayloadSize,
                MBOOL32                 bBlocking);


/************************************************************************************************************\

Function:       LAudioOut_ScheduleSetBuffer

Description:    Schedules the next available audio buffer for playback by adding the required commands to the 
                passed in device thread object.

Parameters:     IN  hAudioOut           Handle to audio output device.
                IN  hBuffer             Handle to the next available buffer for playback.
                IN  hDeviceThread       Device thread handle to use to schedule the buffer.
                IN  uiTimeout_ms        Device thread full queue wait timeout.
                IN  bBlocking           MTRUE to force the device thread block on the set buffer command until
                                        the newly set buffer is actually used (also meaning that the previous 
                                        buffer is no longer used), MFALSE to proceed to the next device thread
                                        command immediately after the new buffer is set even if it is not used
                                        yet.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAudioOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_TIMEOUT         Function exited after a timeout.

Comments:       - If the playback session isn't running, this service won't block even if the bBlocking flag 
                  has been set.

\************************************************************************************************************/ 
LAPI LStatus LAudioOut_ScheduleSetBuffer(
                LAudioOut_Handle        hAudioOut,
                LBuffer_Handle          hBuffer,
                LDeviceThread_Handle    hDeviceThread,
                MUINT32                 uiTimeout_ms,
                MBOOL32                 bBlocking);


/************************************************************************************************************\

Function:       LAudioOut_GetPlaybackState

Description:    Return the audio playback state.

Parameters:     IN  hAudioOut               Handle to the audio output device.
                OUT peState                 Pointer to an LAudioOut_State variable to be filled with 
                                            the current playback state.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioOut_GetPlaybackState(
                LAudioOut_Handle    hAudioOut,
                LAudio_State*       peState);


/************************************************************************************************************\

Function:       LAudioOut_WaitForBufferChange

Description:    Blocks a CPU thread until the next buffer playback starts.

Parameters:     IN  hAudioOut           Handle to the audio output device.
                IN  uiTimeout_ms        Time out to wait in milliseconds.  0 means no wait and 
                                        LINFINITE_TIMEOUT means forever.
                OUTO puiCounter         Pointer to an MUINT64 variable to be filled with the number of 
                                        the just started buffer. Can be MNULL if the counter isn't 
                                        requested.
                OUTO puiTickRefCounter  Pointer to an MUINT64 variable to be filled with the tick reference 
                                        counter value, in ticks since the system was started, when the 
                                        counter was updated.  Can be MNULL if this counter value is 
                                        not requested.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_TIMEOUT         Function exited after a timeout.
                LStatus_CANCELED        Function exited. No playback in progress.
                LStatus_UNSUPPORTED     Function aborted because waiting for start is unsupported 
                                        by the current hardware configuration.

Comments:       - For more information on counters, tick counters, and top audio data, see 
                  LAudioOut_GetLastBufferCounter. 

\************************************************************************************************************/
LAPI LStatus LAudioOut_WaitForBufferChange(
                LAudioOut_Handle    hAudioOut,
                MUINT32             uiTimeout_ms,
                MUINT64*            puiCounter,
                MUINT64*            puiTickRefCounter);


/************************************************************************************************************\

Function:       LAudioOut_GetLastBufferCounter

Description:    Gets the last played buffer counter value.

Parameters:     IN  hAudioOut           Handle to the audio output device.
                OUTO puiCounter          Pointer to an MUINT64 variable to be filled with the last counter
                                        number.  Can be MNULL if the played isn't requested.
                OUTO puiTickRefCounter   Pointer to an MUINT64 variable to be filled with the tick reference 
                                        counter value, in ticks since the system was started, when the 
                                        counter was last updated.  Can be MNULL if this counter value 
                                        isn't requested.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioOut_GetLastBufferCounter(
                LAudioOut_Handle    hAudioOut,
                MUINT64*            puiCounter,
                MUINT64*            puiTickRefCounter);


/************************************************************************************************************\

Function:       LAudioOut_GetCounterLUID

Description:    Gets played counters unique identifiers (LUID).

Parameters:     IN  hAudioOut           Handle to the audio output device.
                OUT puiCounterLUID      Pointer to an MUINT64 variable where the audio data counter unique 
                                        identifier is written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - Use to synchronize the device thread with the completion of a counter.

\************************************************************************************************************/
LAPI LStatus LAudioOut_GetCounterLUID(
                LAudioOut_Handle    hAudioOut,
                MUINT64*            puiCounterLUID);


/************************************************************************************************************\

Function:       LAudioOut_WaitForEvent

Description:    Blocks a CPU thread until any of the specified playback event(s) are signaled.

Parameters:     IN  hAudioOut               Handle to the audio output device.
                IN  uiTimeout_ms            Time out to wait in milliseconds.  0 means no wait and 
                                            LINFINITE_TIMEOUT means forever.
                OUT peSignaledEvent         Pointer to a LAudioOut_Event variable to be filled with one
                                            signaled playback event.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_UNSUPPORTED         Waiting for playback events isn't supported.
                LStatus_TIMEOUT             Function exited after a timeout.
                LStatus_CANCELED            Function exited. No playback in progress.

Comments:       - A playback event can be used to notify applications of important events during a playback
                   session.
                - Application can start waiting for event before the playback is started with
                   LAudioOut_StartPlayback.

\************************************************************************************************************/
LAPI LStatus LAudioOut_WaitForEvent(
                LAudioOut_Handle    hAudioOut,
                MUINT32             uiTimeout_ms,
                LAudioOut_Event*    peSignaledEvent);


/************************************************************************************************************\

Function:       LAudioOut_GetControlInfo

Description:    Returns information about an audio output device control.

Parameters:     IN  hAudioOut           Handle to the audio output device.
                IN  eControlId          ID of the control for which information should be returned.  
                                        Must be a value from the LAudioOut_ControlId.
                OUT poControlInfo       Pointer to an LAudioOut_ControlInfo variable to be filled with the 
                                        specified audio output device control information.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     The ID of a control that's unsupported by the current hardware 
                                        configuration was specified.

Comments:       - For more information on audio output device controls, see LAudioOut_ControlId and 
                  LAudioOut_ControlInfo.

\************************************************************************************************************/
LAPI LStatus LAudioOut_GetControlInfo(
                LAudioOut_Handle         hAudioOut,
                LAudioOut_ControlId      eControlId,
                LAudioOut_ControlInfo*   poControlInfo);


/************************************************************************************************************\

Function:       LAudioOut_GetControlValues

Description:    Gets the value of the specified audio output device controls.

Parameters:     IN  hAudioOut           Handle to the audio output device.
                IN  uiCount             Number of audio output device controls value to get.
                IN  paeControlIds       Pointer to an array of LAudioOut_ControlId variables containing the 
                                        IDs of all the controls for which their current value should be 
                                        returned.  The array must have at least 'uiCount' entries to avoid 
                                        memory overruns.
                OUT paiControlValues    Pointer to an array of MINT32 variables to be filled with the 
                                        specified controls current value.  The array must have at least 
                                        'uiCount' entries to avoid memory overruns.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     At least one control ID specified is unsupported by the current
                                        hardware configuration.  No control value was set.

Comments:       - For more information on audio output device controls, see LAudioOut_ControlId and 
                  LAudioOut_ControlInfo.

\************************************************************************************************************/
LAPI LStatus LAudioOut_GetControlValues(
                LAudioOut_Handle        hAudioOut,
                MUINT32                 uiCount,
                LAudioOut_ControlId*    paeControlIds,
                MINT32*                 paiControlValues);


/************************************************************************************************************\

Function:       LAudioOut_SetControlValues

Description:    Sets the value of the specified audio output device controls.

Parameters:     IN  hAudioOut           Handle to the audio output device.
                IN  uiCount             Number of audio output device controls values to set.
                IN  paeControlIds       Pointer to an array of LAudioOut_ControlId variables containing the 
                                        Ids of all the controls for which their value should be set.  The 
                                        array must have at least 'uiCount' entries to avoid memory overruns.
                OUT paiControlValues    Pointer to an array of MINT32 variables containing the new controls 
                                        value to set.  The array must have at least 'uiCount' count entries 
                                        to avoid memory overruns.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAudioOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_UNSUPPORTED     At least one control Id specified isn't supported by the current 
                                        hardware configuration.  No control value have been set.

Comments:       - For more information on audio output device controls, see LAudioOut_ControlId and 
                  LAudioOut_ControlInfo.
                - LStatus_INVALID_PARAM is returned if any of the passed in control values are invalid and 
                  the control value remains unchanged. An invalid control value is a value smaller than the 
                  minimal value, greater than the maximum value, or not a multiple of the step value.

\************************************************************************************************************/
LAPI LStatus LAudioOut_SetControlValues(
                LAudioOut_Handle        hAudioOut,
                MUINT32                 uiCount,
                LAudioOut_ControlId*    paeControlIds,
                MINT32*                 paiControlValues);


/************************************************************************************************************\

Function:       LAudioOut_SetGainLevel

Description:    Sets the gain level on the specified audio channels.

Parameters:     IN  hAudioOut           Handle to the audio output device.
                IN  uiChannelCount      The number of elements in the passed in array. The parameter should
                                        be equal or greater than the number of channels in the audio buffer.
                IN  pauiGainLevel       Pointer to the first element of the array of MUINT32 to use as new
                                        value of gain level to set on the specified audio channels. The value
                                        must be between 0 and 100.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAudioOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - To use this call, the gain level control must be supported by the hardware. If it isn't 
                  supported by the hardware, this service won't have any effect.
                - To use this call with multiple values (one per channel), it must be supported by the 
                  hardware. If it isn't supported by the hardware, only the first value of the passed in array 
                  is used and applied to all channels.

\************************************************************************************************************/
LAPI LStatus LAudioOut_SetGainLevel(
                LAudioOut_Handle    hAudioOut,
                MUINT32             uiChannelCount,
                MUINT32*            pauiGainLevel);


/************************************************************************************************************\

Function:       LAudioOut_GetGainLevel

Description:    Retrieves the current gain level for all audio channels.

Parameters:     IN  hAudioOut           Handle to the audio output device.
                IN  uiChannelCount      The number of elements in the passed in array. The parameter should 
                                        be equal or greater than the number of channels in the audio buffer.
                OUT pauiGainLevel       Pointer to the first element of the array of MUINT32 to be filled 
                                        with the gain level of all the audio channels. The values returned
                                        will be between 0 and 100.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioOut_GetGainLevel(
                LAudioOut_Handle    hAudioOut,   
                MUINT32             uiChannelCount,
                MUINT32*            pauiGainLevel);


#if defined (__cplusplus)
}
#endif

#endif  // #ifndef INC_LAUDIOOUT_H
