/************************************************************************************************************\

Module Name:    LAudioIn.h

Description:    The LAudioIn module of the API provides the programming interface for the audio input hardware 
                and its associated input devices.

References:     LAudioIn.doc Revision 0.12.

    Copyright (c) 2016, Matrox Graphics Inc.
    All Rights Reserved.

\************************************************************************************************************/

#ifndef INC_LAUDIOIN_H
#define INC_LAUDIOIN_H

// -----------------------------------------------------------------------------------------------------------
//                                   I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "LAudio.h"
#include "LVideoIn.h"
#include "LAudioFormat.h"

typedef struct LAUDIOIN_OBJECT*      LAudioIn_Handle;

// -----------------------------------------------------------------------------------------------------------
//                                                U S A G E
// -----------------------------------------------------------------------------------------------------------

/*  Capture audio stream on an Audio input.

    #include "Liberatus.h"
    #include "LBuffer.h"
    #include "LAudioIn.h"

    int main()
    {
        LDevice_Handle              hDevice         = MNULL;
        LBuffer_Handle              hCapturedBuffer = MNULL;
        LAudioIn_Handle             hAudioIn        = MNULL;
        MUINT32                     uiBufferSizeInBytes = 0;
        MUINT32                     uiIndex         = 0;
        LAudioFormat                eAudioFormat;

        Liberatus_Load();

        hDevice = Liberatus_GetDevice(0);           // Primary device handle             

        // Get handle on first usable audio input.
        LAudioIn_GetHandle(hDevice, 
            0,                                      // First audio input using default signal type
            LAccessMode_READWRITE, 
            &hAudioIn);

        // Enum audio format.
        LAudioIn_EnumSupportedAudioFormat(hAudioIn,
            0,                      // Use the first available AudioFormat
            &eAudioFormat);

        // Set buffer attributes (2048 frames per buffer) for the capture.    
        LAudioFormat_ComputeBufferSizeBasedOnFrame(
            eAudioFormat,
            2048,                   // 2048 frames per buffer
            &uiBufferSizeInBytes);

        LAudioIn_CreateBuffers(hAudioIn,
            eAudioFormat,
            uiBufferSizeInBytes,
            3);

        // Start capture audio.
        LAudioIn_StartCapture(hAudioIn);

        for(uiIndex = 0;  uiIndex < 100; uiIndex++)
        {
            LAudioIn_GetNextBuffer(
                hAudioIn,
                MTRUE,                // Wait for Ready buffer.
                200,                  // in ms
                &hCapturedBuffer,
                MNULL,
                MNULL);

            // USE AUDIO BUFFER HERE

            LAudioIn_ReleaseBuffer(
                hAudioIn,
                hCapturedBuffer);

        }   

        // End audio capture
        LAudioIn_StopCapture(hAudioIn);

        LAudioIn_DestroyBuffers(hAudioIn);

        // Release audio input.
        LAudioIn_ReleaseHandle(hAudioIn);

        Liberatus_UnLoad();
    }

*/

// -----------------------------------------------------------------------------------------------------------
//                                               O V E R V I E W
// -----------------------------------------------------------------------------------------------------------


/************************************************************************************************************\

    The main purpose of the Audio Input module is to provide the functionality to input (capture) an audio 
    input stream into the Matrox SV2 system.

    The module has services to:
        - enumerate audio input paths and their audio capabilities
        - configure the available audio input path in a specific mode to capture audio buffers
        - retrieve the captured audio buffers
        - retrieve the state of the interfaces
        - be notified of capture events
        - adjust dynamic controls (such as input gain level)

    Based on the hardware configuration, the module can handle analog and/or digital audio input stream 
    capture in mono, stereo, or multi-channels. It can also handle PCM (uncompressed) and/or compressed audio 
    input streams.

    The module can handle multiple audio input streams at a time.

\************************************************************************************************************/

/************************************************************************************************************\
    A audio inputs can be used to capture audio input stream data into LBuffer objects, named capture buffers 
    from here on, that can then be used to perform processing on the captured data.

    A audio input has an internal set of capture buffers that are used to capture input stream data and 
    that are loaned to a Liberatus application for processing.

    Each audio input internal capture buffer has 2 different attributes that describe its current state. 
    
    The first attribute is called 'Ready' and it represents whether it contains all captured stream data 
    or not. The second attribute is called 'Loaned' and it represents whether a capture buffer is loaned 
    to a Liberatus application for processing or not.

    Bellow you will find when a capture buffer 'Ready' and 'Loaned' state attributes change.

    Ready state attribute:
    ======================
        +----------------------+                                     +----------------------+
        |    Capture Buffer    |     Stream Captured Data            |    Capture Buffer    |
        |                      |------------------------------------>|                      |
        |        EMPTY         |                                     |        READY         |
        |                      |<------------------------------------|                      |
        |                      |        ReleaseBuffer() or           |                      |
        +----------------------+      DiscardReadyBuffers() or       +----------------------+
                                     ScheduleReleaseBuffer()

    Loaned state attribute:
    =======================
        +----------------------+                                     +----------------------+
        |    Capture Buffer    |           GetNextBuffer()           |    Capture Buffer    |
        |                      |------------------------------------>|                      |
        |      RELEASED        |                                     |        LOANED        |
        |                      |<------------------------------------|                      |
        |                      |           ReleaseBuffer() or        |                      |
        +----------------------+        ScheduleReleaseBuffer()      +----------------------+

    Attributes states meaning:
    ==========================
    
    +-------------+-------------+-----------------------------------------------------------+
    |   Ready     |   Loaned    | Meaning                                                   |
    +-------------+-------------+-----------------------------------------------------------+
    |   MFALSE    |   MFALSE    | Capture buffer contains no stream data and isn't loaned   |
    |             |             | to a Liberatus application.  The audio input can capture  |
    |             |             | input stream date in the buffer.                          |
    +-------------+-------------+-----------------------------------------------------------+
    |   MFALSE    |   MTRUE     | Capture buffer contains no stream data and is loaned to a |
    |             |             | Liberatus application.  The audio input can capture input |
    |             |             | stream date in the buffer and the Liberatus application   |
    |             |             | can schedule operations on the capture buffer.            |
    +-------------+-------------+-----------------------------------------------------------+
    |   MTRUE     |   MFALSE    | Capture buffer contains stream data and isn't loaned to   |
    |             |             | a Liberatus application.  The audio input can't capture   |
    |             |             | input stream date in the buffer.                          |
    +-------------+-------------+-----------------------------------------------------------+
    |   MTRUE     |   MTRUE     | Capture buffer contains stream data and is loaned to a    |
    |             |             | Liberatus application.  The audio input can't capture     |
    |             |             | input stream date in the buffer and the Liberatus         |
    |             |             | application can schedule or perform direct operations on  |
    |             |             | the capture buffer.                                       |
    +-------------+-------------+-----------------------------------------------------------+

        Capture Buffer pool definitions

            <Total>        Count of all buffers in the pool.

            <Ready>        Count of buffers containing captured data.
            <Empty>        Count of buffers containing no captured data. <Total - Ready>

            <Loaned>       Count of buffers loaned to the application
            <Released>     Count of buffers not loan to the application. <Total - Loaned>

\************************************************************************************************************/


// -----------------------------------------------------------------------------------------------------------
//                                   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Enum:           LAudioIn_ControlId

Description:    Enumeration defining all possible audio input control type values.

Comments:       - Each audio input control has a minimal, a maximal, a default value and also a step value 
                  that can be used compute valid value to modify the control.  See the LAudioIn_ControlInfo 
                  structure or the LAudioIn_SetControlValue() service for more details.

\************************************************************************************************************/
typedef enum
{
    LAudioIn_ControlId_INVALID                = 0x00000000,   // Invalid value.
    LAudioIn_ControlId_FORCE32                = 0x7FFFFFFF,   // Dummy value to force enumeration to use 
                                                              // 32-bits.
} LAudioIn_ControlId;

/************************************************************************************************************\

Enum:           LAudioIn_Event

Description:    Enumerates all possible capture events.

Comments:       None.

\************************************************************************************************************/
typedef enum
{
    LAudioIn_Event_INVALID              = 0x00000000,   // Invalid value.
    LAudioIn_Event_START_PENDING        = 0x00000001,   // Capture started but it's waiting on an external 
                                                        //  event to start capturing.
    LAudioIn_Event_STARTED              = 0x00000002,   // Capture already started, audio data 
                                                        //  are being captured.
    LAudioIn_Event_STOPPED              = 0x00000004,   // Capture has stopped, but it's still 
                                                        //  initialized and can be restarted.
    LAudioIn_Event_STATE_CHANGED        = 0x00000008,   // Capture state has changed.
    LAudioIn_Event_SOURCE_CHANGED       = 0x00000010,   // Input source has changed.  The capture was stopped 
                                                        //  and must be re-initialized to capture 
                                                        //  audio data again.
    LAudioIn_Event_OUT_OF_BUFFER        = 0x00000020,   // Capture is forced to drop audio data because no 
                                                        // empty capture buffer is available. Capture  
                                                        //  automatically resumes when capture buffers become
                                                        //  available again.
    LAudioIn_Event_ALL_BUFFER_RELEASED  = 0x00000040,   // All allocated capture buffers are currently in the 
                                                        //  Released state.  This event can be used to figure 
                                                        //  out when the handle to the audio input device can 
                                                        //  be safely released without causing a resource 
                                                        //  leaks.
    LAudioIn_Event_SOURCE_DETECTED      = 0x00000080,   // Input source has been detected.
                                                        //  LAudioIn_GetDetectedAudioParameters should return
                                                        //  new parameters.


    LAudioIn_Event_FORCE32              = 0x7FFFFFFF    // Dummy value to force to use 32-bits. 
} LAudioIn_Event;

/************************************************************************************************************\

Struct:         LAudioIn_ControlInfo

Description:    Structure defining a audio input control information.

Comments:       - A control is an attribute of the audio input device with a default value, a minimal value, 
                  maximal value, and a step value.
                - The minimal value and the step value should be used to compute all valid values within the 
                  range of supported control values.
                - For more information on available controls, see LAudioIn_ControlId.
                - To manipulate the controls value, see the LAudioIn_ControlInfo, LAudioIn_GetControlValues 
                  or LAudioIn_SetControlValues.
                - To get the current value of a control use the LAudioIn_GetControlValues service.


\************************************************************************************************************/
typedef struct tagLAudioIn_ControlInfo
{
    MINT32  iMinimum;   // Minimum value.
    MINT32  iMaximum;   // Maximum value.
    MINT32  iStep;      // Step value.
    MINT32  iDefault;   // Default value.
} LAudioIn_ControlInfo;

// -----------------------------------------------------------------------------------------------------------
//                         G L O B A L   F U N C T I O N   D E C L A R A T I O N S
// -----------------------------------------------------------------------------------------------------------


#if defined (__cplusplus)
extern "C" {
#endif

/************************************************************************************************************\

Function:       LAudioIn_GetCount

Description:    Gets number of independent audio inputs available for the specified device.
                
Parameters:     IN  hDev            Device handle.
                OUT puiCount        Pointer to an MUINT32 variable to be filled with the number of independent 
                                    audio inputs available.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioIn_GetCount(
                LDevice_Handle      hDev, 
                MUINT32*            puiCount); 


/************************************************************************************************************\

Function:       LAudioIn_GetHandle

Description:    Gets audio input handle for the specified independent audio input device.

Parameters:     IN  hDev            Device handle.
                IN  uiIndex         Index of the independent audio input for which to return the handle.
                IN  eAccessMode     Requested access mode.  Must be a value from LAccessMode.
                OUT phAudioIn       Pointer to an LAudioIn_Handle variable where the audio input device
                                    handle is written.  MNULL if unsuccessful.


Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_UNSUPPORTED         Invalid audio input device index.  No hardware associated.
                LStatus_OUT_OF_RESOURCES    Specified audio input device can't be used by the Liberatus
                                            applications because it's already being used by the system.
                LStatus_RESOURCES_BUSY      Audio input handle couldn't be returned because the audio input
                                            is busy.

Comments:       - If eAccessMode is LAccessMode_READONLY, functions that modify the audio input can't be 
                  called.
                - If eAccessMode is LAccessMode_READWRITE or LAccessMode_READWRITE_EXCLUSIVE, functions that 
                  modify the audio input can be called.
                - Only one handle with LAccessMode_READWRITE_EXCLUSIVE read/write access mode can exist at 
                  any one time. If a handle with read/write access has already been granted to a caller, 
                  any attempt to get another handle with read/write access returns a failure.

\************************************************************************************************************/
LAPI LStatus LAudioIn_GetHandle(
                LDevice_Handle          hDev, 
                MUINT32                 uiIndex,
                LAccessMode             eAccessMode,
                LAudioIn_Handle*        phAudioIn);


/************************************************************************************************************\

Function:       LAudioIn_ReleaseHandle

Description:    Releases a handle to an independent audio input device.

Parameters:     IN  hAudioIn            Handle to the audio input device.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioIn_ReleaseHandle(
                LAudioIn_Handle     hAudioIn);


/************************************************************************************************************\

Function:       LAudioIn_EnumSignalType

Description:    Enumerates all supported audio input signal types. 

Parameters:     IN  hAudioIn            Handle to the audio input device.
                IN  uiSignalIndex       Index of the signal type to enumerate.
                OUT peSignalType        Pointer to an LAudio_SignalType object where the enumerated signal
                                        types are written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_NO_MORE_DATA    No signal type value is available for the specified index.
                
Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioIn_EnumSignalType(
                LAudioIn_Handle         hAudioIn,
                MUINT32                 uiSignalIndex,
                LAudio_SignalType*      peSignalType);


/************************************************************************************************************\

Function:       LAudioIn_GetDefaultSignalType

Description:    Gets the default signal type of the audio input device.

Parameters:     IN  hAudioIn            Handle to the audio input device.
                OUT peSignalType        Pointer to an LAudio_SignalType object where the enumerated signal
                                        types are written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - The default audio input signal type device may change after an input source detection.

\************************************************************************************************************/
LAPI LStatus LAudioIn_GetDefaultSignalType(
                LAudioIn_Handle         hAudioIn,
                LAudio_SignalType*      peSignalType);


/************************************************************************************************************\

Function:       LAudioIn_SetSignalType

Description:    Sets the audio input device signal type  of the audio input device.

Parameters:     IN  hAudioIn            Handle to the audio input device.
                IN  eSignalType         Audio signal type to set for the independent audio input.  Must be a 
                                        value from the LAudio_SignalType.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAudioIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - Setting the signal type may reserve and program some hardware resources that are shared 
                  with other audio input device.

\************************************************************************************************************/
LAPI LStatus LAudioIn_SetSignalType(
                LAudioIn_Handle     hAudioIn,
                LAudio_SignalType   eSignalType);


/************************************************************************************************************\

Function:       LAudioIn_GetSignalType

Description:    Gets the current audio input device signal type.

Parameters:     IN  hAudioIn            Handle to the audio input device.
                OUT peSignalType        Pointer to an LAudio_SignalType object where the enumerated signal
                                        types are written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioIn_GetSignalType(
                LAudioIn_Handle         hAudioIn,
                LAudio_SignalType*      peSignalType);


/************************************************************************************************************\

Function:       LAudioIn_DetectSource

Description:    Performs an input source audio parameter detection based on the current signal type.

Parameters:     IN  hAudioIn            Handle to the audio input device.
                OUT pbIsCapturable      Pointer to a MBOOL32 variable to be filled with MTRUE if the detected
                                        source can be captured and MFALSE if no source was detected or the
                                        detected source is outside the range of capturable sources.
                OUT pbIsDetected        Pointer to a MBOOL32 variable to be filled with MTRUE if a source,
                                        capturable or not, was detected and MFALSE otherwise.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_UNSUPPORTED     Function failed because source detection is not supported by this 
                                        audio input.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioIn_DetectSource(
                LAudioIn_Handle         hAudioIn,
                MBOOL32*                pbIsCapturable,
                MBOOL32*                pbIsDetected);

/************************************************************************************************************\

Function:       LAudioIn_GetDetectedAudioParameters

Description:    Gets the last detected input source audio parameters.

Parameters:     IN  hAudioIn            Handle to the audio input device.
                OUT poDescriptor        Pointer to a LAudioFormat_Descriptor structure to be filled with
                                         detected audio parameters.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_NOT_OPTIMAL     Function doesn't return the latest audio parameters.

Comments:       - Can be called with a (LAccessMode_READONLY) LAudioIn_Handle.
                - LStatus_NOT_OPTIMAL: the function returns the cached audio parameters, 
                  LAudioIn_DetectSource must be called to detect new audio parameters.
                - LAudioIn_Event_SOURCE_DETECTED event indicates that new audio parameters can be retrieved.
                - poDescriptor->eAudioFormat is set to LAudioFormat_INVALID when no source is detected.
                - No guarantee that the detected source is capturable.  LAudioIn_DetectSource returns this
                  information.

\************************************************************************************************************/
LAPI LStatus LAudioIn_GetDetectedAudioParameters(
                LAudioIn_Handle             hAudioIn,
                LAudioFormat_Descriptor*    poDescriptor);

/************************************************************************************************************\

Function:       LAudioIn_EnumSupportedAudioFormat

Description:    Enumerates the audio formats supported for capture from the current input source.

Parameters:     IN  hAudioIn            Handle to the audio input device.
                IN  uiIndex             Index of the audio format to be enumerated.
                OUT peAudioFormat       Pointer to an LAudioFormat object to be filled with the enumerated
                                        audio format.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_NO_MORE_DATA    No audio format value is available for the specified index.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioIn_EnumSupportedAudioFormat(
                LAudioIn_Handle     hAudioIn, 
                MUINT32             uiIndex,
                LAudioFormat*       peAudioFormat);


/************************************************************************************************************\

Function:       LAudioIn_CreateBuffers

Description:    Creates audio capture buffers.

Parameters:     IN  hAudioIn            Handle to the audio input device.
                IN  eAudioFormat        Audio format of the audio buffer to capture.  Must be a value from
                                        LAudioFormat.
                IN  uiBufferSize        Size, in byte, of the capture buffer.  Must be a multiple of audio 
                                        frame size for PCM audio format capture buffers.
                IN  uiNbBuffers         Number of buffers allocated for the capture.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAudioIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - The audio input device allocates the specified number of buffers before the capture 
                  is started.
                - The recommended minimum number of buffers is two (2) if the time taken to process a captured 
                  buffer is less than the time taken to capture a buffer. Otherwise, a larger number of 
                  buffers should be used.
                - The recommended number of buffers depends on the time the application needs to process the 
                  captured buffers. There's no fixed maximum number of buffers; only the available memory 
                  limits the maximum.
                - Extra buffers will be created for hardware processing. The number of extra buffers created
                  may be queried with LAudioIn_GetCaptureState().


\************************************************************************************************************/
LAPI LStatus LAudioIn_CreateBuffers(
                LAudioIn_Handle hAudioIn, 
                LAudioFormat    eAudioFormat,
                MUINT32         uiBufferSize,
                MUINT32         uiNbBuffers);

/************************************************************************************************************\

Function:       LAudioIn_DestroyBuffers

Description:    Destroys the audio capture buffers.

Parameters:     IN  hAudioIn            Handle to the audio input device.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAudioIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - It's important to stop the capture and release the buffer before destroying the buffer.

\************************************************************************************************************/
LAPI LStatus LAudioIn_DestroyBuffers(
                LAudioIn_Handle     hAudioIn);


/************************************************************************************************************\

Function:       LAudioIn_GetBufferAttributes

Description:    Gets the current audio capture buffer attributes.

Parameters:     IN  hAudioIn            Handle to audio input device.
                OUT peAudioFormat       Pointer to a LAudioFormat variable to be filled with the current
                                        audio format of the audio buffer to capture. 
                OUT puiBufferSize       Pointer to a MUINT32 variable to be filled with the size, in byte,
                                        of the capture buffer.
                OUT puiNbBuffers        Pointer to an MUINT32 variable to be filled with the number of
                                        buffers allowed to be used by the audio capture device.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioIn_GetBufferAttributes(
                LAudioIn_Handle hAudioIn,
                LAudioFormat*   peAudioFormat,
                MUINT32*        puiBufferSize,
                MUINT32*        puiNbBuffers);


/************************************************************************************************************\

Function:       LAudioIn_StartCapture

Description:    Starts the audio capture.

Parameters:     IN  hAudioIn            Handle to the audio input device.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAudioIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - The number of capture buffers the audio input device is allowed to use can be set via 
                  LAudioIn_CreateBuffers.

\************************************************************************************************************/
LAPI LStatus LAudioIn_StartCapture(
                LAudioIn_Handle hAudioIn);


/************************************************************************************************************\

Function:       LAudioIn_StartCaptureOnVideo

Description:    Starts the audio capture session synchronized with a video capture.

Parameters:     IN  hAudioIn        Handle to the audio input device.
                IN  hVideoIn        Handle to the video input device. 

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAudioIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - Service used for the audio/video synchronization. It enables the synchronized capture of an 
                  audio input and a video input to start.
                
\************************************************************************************************************/
LAPI LStatus LAudioIn_StartCaptureOnVideo(
                LAudioIn_Handle hAudioIn,
                LVideoIn_Handle hVideoIn);

/************************************************************************************************************\

Function:       LAudioIn_StopCapture

Description:    Stops the audio capture.

Parameters:     IN  hAudioIn            Handle to the audio input device.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAudioIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - If capture is stopped in the middle of the capture of a buffer, this buffer is discarded.

\************************************************************************************************************/
LAPI LStatus LAudioIn_StopCapture(
                LAudioIn_Handle hAudioIn);


/************************************************************************************************************\

Function:       LAudioIn_GetNextBuffer

Description:    Gets the next audio buffer.

Parameters:     IN  hAudioIn            Handle to the audio input device.
                IN  bWaitBufferReady    MTRUE to wait for a Ready audio buffer, MFALSE to not wait. 
                IN  uiTimeout_ms        Time out to wait in milliseconds.  0 means no wait and 
                                        LINFINITE_TIMEOUT means wait forever.
                OUT phBuffer            Pointer to an LBuffer_Handle variable to be filled with the handle to 
                                        audio buffer.
                OUTO puiPayloadSize     Pointer to a MUINT32 variable to be filled with the size of the
                                        effective data contained in hBuffer. Can be MNULL if this value is
                                        unused.
                OUTO puiBufferCounter   Pointer to an MUINT64 variable to be filled with the buffer counter 
                                        value when the returned buffer was captured or will be capture 
                                        by the audio input.  Can be MNULL if this value is unused.
                OUTO puiTickRefCounter  Pointer to an MUINT64 variable to be filled with the tick reference 
                                        counter value, in ticks since the system was started, when the 
                                        returned buffer was captured by the audio input. Returns 0 when 
                                        bWaitBufferReady is MFALSE and there's no Ready buffer. 
                                        Can be MNULL if this value is unused.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_TIMEOUT         Function exited after a timeout.
                LStatus_CANCELED        Function exited. No capture in progress.
                LStatus_ACCESS_DENIED   The hAudioIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                
Comments:       - The audio capture buffers returned are allocated and owned by the audio input device. 
                  They're loaned to the application for processing. The capture buffer state changes from 
                  Released to Loaned when it's loaned to the application.
                - When bWaitBufferReady is set to MFALSE, the buffer is returned even if isn't Ready or 
                  completely filled with the audio capture data. uiTickRefCounter is then set to 0.
                - The application can schedule audio processing in advance on Empty buffers or on parts of 
                  Empty buffers in low latency mode. To get the handle of the Empty buffers, call the 
                  LAudioIn_GetNextBuffer service by setting bWaitBufferReady to MFALSE. The call will then 
                  return the handle of an Empty capture buffer on which processing can be scheduled in a 
                  device thread. To synchronize the processing and avoid using the Empty capture buffer before 
                  it actually contains captured audio data, use LAudioIn_ScheduleWaitCounter with the returned 
                  value in puiCounter.
                - A loaned buffer must be released using the LAudioIn_ReleaseBuffer or 
                  LAudioIn_ScheduleReleaseBuffer services after the application processed them to allow the 
                  audio input device to reuse the buffers to capture new audio data.
                - If the capture buffers aren't released, the audio input device will run out of Empty buffers 
                  and hold the last empty buffers available to capture until a buffer is released.
                - The counter is incremented each time a new valid signal from the input source is seen by the 
                  audio input device, whether or not it's actually captured or dropped.
                - If two consecutive buffers, returned by the LAudioIn_GetNextBuffer service, have a 
                  difference greater than one between their associated counter values, some input source audio 
                  data were dropped (i.e. have not been captured to capture buffers). This can occur if 
                  capture buffers were unavailable at the time. To know how many input source audio data 
                  haven't been captured, subtract 1 from the counter values difference.
                
\************************************************************************************************************/
LAPI LStatus LAudioIn_GetNextBuffer(
                LAudioIn_Handle hAudioIn,
                MBOOL32         bWaitBufferReady,
                MUINT32         uiTimeout_ms,
                LBuffer_Handle* phBuffer,
                MUINT32*        puiPayloadSize,
                MUINT64*        puiBufferCounter,
                MUINT64*        puiTickRefCounter);



/************************************************************************************************************\

Function:       LAudioIn_ReleaseBuffer

Description:    Releases an audio capture buffer.

Parameters:     IN  hAudioIn            Handle to the audio input device.
                IN  hAudioBuffer        Handle to the audio buffer to release.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAudioIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - Capture audio buffers loaned via the LAudioIn_GetNextBuffer service must be released after 
                  the application processes them to allow the audio input device to reuse them to capture 
                  other input source audio data. The number of capture buffers for a given audio input device 
                  is limited. Not releasing audio capture buffers starves the audio input device and will 
                  eventually force it to temporarily stop capturing input source audio data.
                - The LAudioIn_ReleaseBuffer service changes the states of a capture buffer to Empty and 
                  Released.


\************************************************************************************************************/
LAPI LStatus LAudioIn_ReleaseBuffer(
                LAudioIn_Handle     hAudioIn,
                LBuffer_Handle      hBuffer);


/************************************************************************************************************\

Function:       LAudioIn_ScheduleReleaseBuffer

Description:    Schedules the release of an audio capture buffer of the audio input device by adding required 
                commands to the passed in device thread object.

Parameters:     IN  hAudioIn            Handle to the audio input device.
                IN  hBuffer             Handle to the audio buffer to release.
                IN  hDeviceThread       Handle to the device thread to use to schedule the operation.
                IN  uiTimeout_ms        Device thread full queue wait timeout.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAudioIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_TIMEOUT         Function exited after a timeout.

Comments:       - Capture audio buffers loaned via the LAudioIn_GetNextBuffer service must be released after 
                  the application processes them to allow the audio input device to reuse them to capture 
                  other input source audio data. The number of capture buffers for a given audio input device 
                  is limited. Not releasing audio capture buffers starves the audio input device and 
                  eventually forces it to temporarily stop capturing input source audio data.
                - The LAudioIn_ScheduleReleaseBuffer service changes the state of a capture buffer from Loaned 
                  to Empty.
                - LStatus_TIMEOUT is returned when the ReleaseBuffer command can't be added to the device 
                  thread because it's full and the timeout period expired before space became available.

\************************************************************************************************************/
LAPI LStatus LAudioIn_ScheduleReleaseBuffer(
                LAudioIn_Handle         hAudioIn,
                LBuffer_Handle          hBuffer,
                LDeviceThread_Handle    hDeviceThread,
                MUINT32                 uiTimeout_ms);


/************************************************************************************************************\

Function:       LAudioIn_ScheduleWaitCounter

Description:    Schedules a wait for a counter of the audio input device by adding required commands to the 
                passed in device thread object.

Parameters:     IN  hAudioIn            Handle to the audio input device.
                IN  hDeviceThread       Handle to the device thread to use to schedule the operation.
                IN  uiTimeout_ms        Device thread full queue wait timeout.
                IN  uiCounter           Counter value when the buffer is captured by 
                                        the audio input.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_TIMEOUT         Function exited after a timeout.


Comments:       - LStatus_TIMEOUT is returned when the wait command can't be added to the device thread 
                  because it's full and the timeout period expired before space became available.

\************************************************************************************************************/
LAPI LStatus LAudioIn_ScheduleWaitCounter(
                LAudioIn_Handle         hAudioIn,
                LDeviceThread_Handle    hDeviceThread,
                MUINT32                 uiTimeout_ms,
                MUINT64                 uiCounter);


/************************************************************************************************************\

Function:       LAudioIn_DiscardReadyBuffers

Description:    Discards all capture buffers in the Ready state.

Parameters:     IN  hAudioIn            Handle to the audio input device.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAudioIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - Capture buffers in the Ready state contain captured data not yet retrieved by the 
                  application.
                - The state of all discarded capture buffers is set to Empty. The capture buffers are also 
                  marked as available to be used to capture new input source audio data.
                - The LAudioIn_DiscardReadyBuffers service can be used to discard any unprocessed capture 
                  buffers before re-starting the capture.
                - All capture buffers loaned to the application or currently in the Empty state are 
                  unaffected by this service.

\************************************************************************************************************/
LAPI LStatus LAudioIn_DiscardReadyBuffers(
                LAudioIn_Handle hAudioIn);


/************************************************************************************************************\

Function:       LAudioIn_GetCaptureState

Description:    Return the audio capture state.

Parameters:     IN  hAudioIn                Handle to the audio input device.
                OUT peState                 Pointer to an LAudioIn_State variable to be filled with 
                                            the current capture state.
                OUT puiTotalBufferCount     Pointer to an MUINT32 variable to be filled with the currently
                                            allocated capture buffers count.  
                OUT puiReadyBufferCount     Pointer to an MUINT32 variable to be filled with the currently
                                            ready capture buffers count.
                OUT puiLoanedBufferCount    Pointer to an MUINT32 variable to be filled with the currently
                                            loaned capture buffers count.
                OUT puiInternalBufferCount  Pointer to an MUINT32 variable to be filled with the number of
                                            internally created buffer to allow the hardware to run.
                                            These buffer are not part of the total buffer count.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       - For more information on BufferCount, see LAudioIn_GetNextBuffer.

\************************************************************************************************************/
LAPI LStatus LAudioIn_GetCaptureState(
                LAudioIn_Handle hAudioIn,
                LAudio_State*   peState,
                MUINT32*        puiTotalBufferCount,
                MUINT32*        puiReadyBufferCount,
                MUINT32*        puiLoanedBufferCount,
                MUINT32*        puiInternalBufferCount);

/************************************************************************************************************\

Function:       LAudioIn_WaitForBufferChange

Description:    Blocks a CPU thread until the next buffer capture starts.

Parameters:     IN  hAudioIn            Handle to the audio input device.
                IN  uiTimeout_ms        Time out to wait in milliseconds.  0 means no wait and 
                                        LINFINITE_TIMEOUT means forever.
                OUTO puiCounter          Pointer to an MUINT64 variable to be filled with the number of 
                                        the just started buffer. Can be MNULL if this value is unused.
                OUTO puiTickRefCounter   Pointer to an MUINT64 variable to be filled with the tick reference 
                                        counter value, in ticks since the system was started, when the 
                                        counter was updated.  Can be MNULL if this value is unused.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_TIMEOUT         Function exited after a timeout.
                LStatus_CANCELED        Function exited. No capture in progress.
                LStatus_UNSUPPORTED     Function aborted because waiting for start is unsupported 
                                        by the current hardware configuration.

Comments:       - For more information on counters, tick counters, and top audio data, 
                  see LAudioIn_GetLastBufferCounter. 

\************************************************************************************************************/
LAPI LStatus LAudioIn_WaitForBufferChange(
                LAudioIn_Handle hAudioIn,
                MUINT32         uiTimeout_ms,
                MUINT64*        puiCounter,
                MUINT64*        puiTickRefCounter);

/************************************************************************************************************\

Function:       LAudioIn_GetLastBufferCounter

Description:    Gets the last captured buffer counter value.

Parameters:     IN  hAudioIn            Handle to the audio input device.
                OUTO puiCounter          Pointer to an MUINT64 variable to be filled with the last counter
                                        number.  Can be MNULL if this value is unused.
                OUTO puiTickRefCounter   Pointer to an MUINT64 variable to be filled with the tick reference 
                                        counter value, in ticks since the system was started, when the 
                                        counter was last updated.  Can be MNULL if this value is unused.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - For more information on counters, tick counters, and top audio data, 
                  see LAudioIn_GetNextBuffer.

\************************************************************************************************************/
LAPI LStatus LAudioIn_GetLastBufferCounter(
                LAudioIn_Handle hAudioIn,
                MUINT64*        puiCounter,
                MUINT64*        puiTickRefCounter);

/************************************************************************************************************\

Function:       LAudioIn_GetCounterLUID

Description:    Gets captured counters unique identifiers (LUID). 

Parameters:     IN  hAudioIn            Handle to the audio input device.
                OUT puiCounterLUID      Pointer to an MUINT64 variable where the audio data counter unique 
                                        identifier is written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - Use to synchronizes the device thread with the completion of a counter.

\************************************************************************************************************/
LAPI LStatus LAudioIn_GetCounterLUID(
                LAudioIn_Handle     hAudioIn,
                MUINT64*            puiCounterLUID);


/************************************************************************************************************\

Function:       LAudioIn_WaitForEvent

Description:    Blocks a CPU thread until any of the specified capture events are signaled.

Parameters:     IN  hAudioIn                Handle to the audio input device.
                IN  uiTimeout_ms            Time out to wait in milliseconds.  0 means no wait and
                                            LINFINITE_TIMEOUT means forever.
                OUT peSignaledEvent         Pointer to a LAudioIn_Event variable to be filled with one
                                            signaled capture event.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_UNSUPPORTED         Waiting for capture events isn't supported.
                LStatus_TIMEOUT             Function exited after a timeout.
                LStatus_CANCELED            Function exited. No capture in progress.

Comments:       - A capture event can be used to notify applications of important events during a capture
                   session.
                - Application can start waiting for event before the capture is started with
                   LAudioIn_StartCapture.

\************************************************************************************************************/
LAPI LStatus LAudioIn_WaitForEvent(
                LAudioIn_Handle hAudioIn,
                MUINT32         uiTimeout_ms,
                LAudioIn_Event* peSignaledEvent);


/************************************************************************************************************\

Function:       LAudioIn_GetControlInfo

Description:    Returns information about an audio input device control.

Parameters:     IN  hAudioIn            Handle to the audio input device.
                IN  eControlId          ID of the control for which information should be returned.  
                                        Must be a value from the LAudioIn_ControlId.
                OUT poControlInfo       Pointer to an LAudioIn_ControlInfo variable to be filled with the 
                                        specified audio input device control information.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     The ID of a control that's unsupported by the current hardware 
                                        configuration was specified.

Comments:       - For more information on audio input device controls, see LAudioIn_ControlId and 
                  LAudioIn_ControlInfo.

\************************************************************************************************************/
LAPI LStatus LAudioIn_GetControlInfo(
                LAudioIn_Handle         hAudioIn,
                LAudioIn_ControlId      eControlId,
                LAudioIn_ControlInfo*   poControlInfo);


/************************************************************************************************************\

Function:       LAudioIn_GetControlValues

Description:    Gets the value of the specified audio input device controls.

Parameters:     IN  hAudioIn            Handle to the audio input device.
                IN  uiCount             Number of audio input device controls value to get.
                IN  paeControlIds       Pointer to an array of LAudioIn_ControlId variables containing the 
                                        IDs of all the controls for which their current value should be 
                                        returned.  The array must have at least 'uiCount' entries to avoid 
                                        memory overruns.
                OUT paiControlValues    Pointer to an array of MINT32 variables to be filled with the 
                                        specified controls current value.  The array must have at least 
                                        'uiCount' entries to avoid memory overruns.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     At least one control ID specified is unsupported by the current
                                        hardware configuration.  No control value was set.

Comments:       - For more information on audio input device controls, see LAudioIn_ControlId and 
                  LAudioIn_ControlInfo.

\************************************************************************************************************/
LAPI LStatus LAudioIn_GetControlValues(
                LAudioIn_Handle     hAudioIn,
                MUINT32             uiCount,
                LAudioIn_ControlId* paeControlIds,
                MINT32*             paiControlValues);


/************************************************************************************************************\

Function:       LAudioIn_SetControlValues

Description:    Sets the value of the specified audio input device controls.

Parameters:     IN  hAudioIn            Handle to the audio input device.
                IN  uiCount             Number of audio input device controls values to set.
                IN  paeControlIds       Pointer to an array of LAudioIn_ControlId variables containing the 
                                        Ids of all the controls for which their value should be set.  The 
                                        array must have at least 'uiCount' entries to avoid memory overruns.
                OUT paiControlValues    Pointer to an array of MINT32 variables containing the new controls 
                                        value to set.  The array must have at least 'uiCount' count entries 
                                        to avoid memory overruns.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAudioIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_UNSUPPORTED     At least one control Id specified isn't supported by the current 
                                        hardware configuration.  No control value have been set.

Comments:       - For more information on audio input device controls, see LAudioIn_ControlId and 
                  LAudioIn_ControlInfo.
                - LStatus_INVALID_PARAM is returned if any of the passed in control values are invalid and 
                  the control value remains unchanged. An invalid control value is a value smaller than the 
                  minimal value, greater than the maximum value, or not a multiple of the step value.

\************************************************************************************************************/
LAPI LStatus LAudioIn_SetControlValues(
                LAudioIn_Handle     hAudioIn,
                MUINT32             uiCount,
                LAudioIn_ControlId* paeControlIds,
                MINT32*             paiControlValues);


/************************************************************************************************************\

Function:       LAudioIn_SetGainLevel

Description:    Sets the gain level on the specified audio channels.

Parameters:     IN  hAudioIn            Handle to the audio input device.
                IN  uiChannelCount      The number of elements in the passed in array. The parameter should
                                        be equal or greater than the number of channels in the audio buffer.
                IN  pauiGainLevel       Pointer to the first element of the array of MUINT32 to use as new
                                        value of gain level to set on the specified audio channels. The value
                                        must be between 0 and 100.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAudioIn handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - To use this call, the gain level control must be supported by the hardware. If not 
                  supported by the hardware, this service won't do anything.
                - To use this call with multiple values (one per channel), it must be supported by the 
                  hardware. If not supported by the hardware, only the first value of the passed in array 
                  will be used and applied to all channels.

\************************************************************************************************************/
LAPI LStatus LAudioIn_SetGainLevel(
                LAudioIn_Handle hAudioIn,
                MUINT32         uiChannelCount,
                MUINT32*        pauiGainLevel);


/************************************************************************************************************\

Function:       LAudioIn_GetGainLevel

Description:    Retrieves the current gain level for all audio channels.

Parameters:     IN  hAudioIn            Handle to the audio input device.
                IN  uiChannelCount      The number of elements in the passed in array. The parameter should 
                                        be equal or greater than the number of channels in the audio buffer.
                OUT pauiGainLevel       Pointer to the first element of the array of MUINT32 to be filled 
                                        with the gain level of all the audio channels. The values returned
                                        will be between 0 and 100.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioIn_GetGainLevel(
                LAudioIn_Handle hAudioIn,   
                MUINT32         uiChannelCount,
                MUINT32*        pauiGainLevel);

/************************************************************************************************************\

Function:       LAudioIn_GetAudioInfoFrameHeader

Description:    Gets the audio info frame header.

Parameters:     IN      hAudioIn                Handle to the audio input device.
                OUT     poAudioInfoFrameHeader  Pointer to a LAudio_InfoFrameHeader structure to be filled
                                                 with the header data of the audio InfoFrame (see
                                                 LAudio_InfoFrameHeader for detail).

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     The audio InfoFrame feature is not supported by this audio input.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioIn_GetAudioInfoFrameHeader(
                LAudioIn_Handle             hAudioIn,
                LAudio_InfoFrameHeader*     poAudioInfoFrameHeader);

/************************************************************************************************************\

Function:       LAudioIn_GetAudioInfoFrameRawData

Description:    Gets the audio info frame raw data (excluding the info frame header).

Parameters:     IN      hAudioIn            Handle to the audio input device.
                IN      uiSize              Size, in bytes, of the audio InfoFrame raw data buffer.
                OUT     pauiInfoFrameData   Pointer to an audio InfoFrame raw data buffer where the audio
                                            InfoFrame raw data is written (this buffer should be allocated
                                            before passed to this function).

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_UNSUPPORTED     The audio InfoFrame feature is not supported by this audio input.
Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioIn_GetAudioInfoFrameRawData(
                LAudioIn_Handle     hAudioIn,
                MUINT32             uiSize,
                MUINT8*             pauiInfoFrameData);

#if defined (__cplusplus)
}
#endif

#endif  // #ifndef INC_LAUDIOIN_H
