/************************************************************************************************************\

Module Name:    LAudioFormat.h

Description:    Header file containing the definition of the supported Audio format.

References:     LAudioFormat.doc Revision 0.2.

    Copyright (c) 2011, Matrox Graphics Inc.
    All Rights Reserved.

\************************************************************************************************************/

#ifndef INC_LAUDIOFORMAT_H
#define INC_LAUDIOFORMAT_H

// -----------------------------------------------------------------------------------------------------------
//                                   I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "LAPI.h"

// -----------------------------------------------------------------------------------------------------------
//                                   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

#define LAFMT_SR_SHIFT                      0x0
#define LAFMT_SR_MASK                       0xFF
#define LAFMT_SR_5512HZ                     0x1
#define LAFMT_SR_8000HZ                     0x2
#define LAFMT_SR_11025HZ                    0x3
#define LAFMT_SR_12000HZ                    0x4
#define LAFMT_SR_16000HZ                    0x5
#define LAFMT_SR_22050HZ                    0x6
#define LAFMT_SR_24000HZ                    0x7
#define LAFMT_SR_32000HZ                    0x8
#define LAFMT_SR_44100HZ                    0x9
#define LAFMT_SR_48000HZ                    0xA
#define LAFMT_SR_64000HZ                    0xB
#define LAFMT_SR_88200HZ                    0xC
#define LAFMT_SR_96000HZ                    0xD
#define LAFMT_SR_176400HZ                   0xE
#define LAFMT_SR_192000HZ                   0xF
#define LAFMT_SR_384000HZ                   0x10
#define LAFMT_SR_768000HZ                   0x11

#define MAKE_LAUDIOFORMAT_SAMPLERATE(uiSampleRateIdx)   ((uiSampleRateIdx)  << LAFMT_SR_SHIFT)

#define LAFMT_SS_SHIFT                      0x8
#define LAFMT_SS_MASK                       0xF
#define LAFMT_SS_8BITS                      0x0
#define LAFMT_SS_16BITS                     0x1
#define LAFMT_SS_20BITS                     0x2
#define LAFMT_SS_24BITS                     0x3
#define LAFMT_SS_32BITS                     0x4

#define MAKE_LAUDIOFORMAT_SAMPLESIZE(uiSampleSizeIdx)   ((uiSampleSizeIdx)  << LAFMT_SS_SHIFT)

#define LAFMT_CH_SHIFT                      0xC
#define LAFMT_CH_MASK                       0xF
#define LAFMT_CH_MONO                       0x0
#define LAFMT_CH_STEREO                     0x1
#define LAFMT_CH_MULTICH_2_1                0x2
#define LAFMT_CH_MULTICH_4_0                0x5
#define LAFMT_CH_MULTICH_4_1                0x6
#define LAFMT_CH_MULTICH_5_1                0x7
#define LAFMT_CH_MULTICH_6_1                0x8
#define LAFMT_CH_MULTICH_7_1                0x9
#define LAFMT_CH_MULTICH_16                 0xF

#define MAKE_LAUDIOFORMAT_CHANNELS(uiChannelsIdx)       ((uiChannelsIdx)    << LAFMT_CH_SHIFT)

#define LAFMT_PL_FL_SHIFT                   0x10
#define LAFMT_PL_FL_MASK                    0xF
#define LAFMT_PL_1                          0x0
#define LAFMT_PL_2                          0x1
#define LAFMT_PL_3                          0x2
#define LAFMT_PL_4                          0x3
#define LAFMT_FL_NONPCM                     0x7
#define LAFMT_FL_USERDEFINED                0x8

#define MAKE_LAUDIOFORMAT_PLANS(uiPlanIdx)              ((uiPlanIdx)        << LAFMT_PL_FL_SHIFT)

#define MAKE_LAUDIOFORMAT(uiSampleRate, uiSampleSize, uiNbChannels, uiNbPlans)  \
                        (   MAKE_LAUDIOFORMAT_SAMPLERATE(uiSampleRate)          \
                        |   MAKE_LAUDIOFORMAT_SAMPLESIZE(uiSampleSize)          \
                        |   MAKE_LAUDIOFORMAT_CHANNELS(uiNbChannels)            \
                        |   MAKE_LAUDIOFORMAT_PLANS(uiNbPlans)  )

/**********************************************************************************************************\

Enum:           LAudioFormat

Description:    Supported format of audio buffer.

Comments:       - The enumeration describes the sample rate (in Hz), the sample size (in bits), 
                  the number of channels, the number of plans and the plans description.
                - The list is not complete, only a subset is presented here.
                
\**********************************************************************************************************/
typedef enum
{
    LAudioFormat_INVALID = 0x00000000,   // Invalid audio format.

    LAudioFormat_MONO_8_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_8BITS,   LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_8_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_8BITS,   LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_8_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_8_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_8_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_8_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_8_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_8_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_8_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_8_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_8_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_8_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_8_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_8_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_8_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_8_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_8_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_16_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_16BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_16_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_16BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_16_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_16_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_16_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_16_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_16_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_16_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_16_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_16_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_16_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_16_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_16_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_16_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_16_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_16_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_16_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_20_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_20BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_20_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_20BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_20_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_20_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_20_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_20_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_20_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_20_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_20_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_20_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_20_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_20_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_20_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_20_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_20_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_20_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_20_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_24_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_24BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_24_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_24BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_24_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_24_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_24_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_24_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_24_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_24_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_24_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_24_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_24_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_24_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_24_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_24_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_24_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_24_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_24_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_32_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_32BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_32_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_32BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_32_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_32_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_32_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_32_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_32_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_32_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_32_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_32_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_32_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_32_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_32_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_32_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_32_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_32_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_MONO_32_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MONO,         LAFMT_PL_1),
    LAudioFormat_STEREO_8_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_8BITS,   LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_8_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_8BITS,   LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_8_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_8BITS,   LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_8_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_8_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_8_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_8BITS,   LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_8_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_8_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_8_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_8BITS,   LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_8_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_8_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_8_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_8BITS,   LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_8_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_8_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_8BITS,   LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_8_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_8BITS,   LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_8_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_8BITS,   LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_8_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_8BITS,   LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_16_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_16BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_16_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_16BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_16_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_16BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_16_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_16_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_16_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_16BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_16_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_16_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_16_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_16BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_16_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_16_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_16_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_16BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_16_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_16_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_16BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_16_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_16BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_16_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_16BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_16_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_16BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_20_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_20BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_20_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_20BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_20_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_20BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_20_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_20_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_20_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_20BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_20_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_20_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_20_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_20BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_20_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_20_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_20_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_20BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_20_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_20_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_20BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_20_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_20BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_20_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_20BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_20_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_20BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_24_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_24BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_24_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_24BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_24_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_24BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_24_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_24_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_24_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_24BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_24_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_24_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_24_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_24BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_24_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_24_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_24_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_24BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_24_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_24_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_24BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_24_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_24BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_24_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_24BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_24_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_24BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_32_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_32BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_32_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_32BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_32_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_32BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_32_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_32_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_32_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_32BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_32_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_32_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_32_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_32BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_32_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_32_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_32_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_32BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_32_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_32_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_32BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_32_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_32BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_32_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_32BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_STEREO_32_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_32BITS,  LAFMT_CH_STEREO,       LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_8_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_8_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_8_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_8_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_8_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_8_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_8_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_8_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_8_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_8_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_8_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_8_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_8_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_8_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_8_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_8_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_8_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_16_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_16_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_16_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_16_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_16_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_16_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_16_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_16_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_16_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_16_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_16_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_16_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_16_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_16_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_16_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_16_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_16_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_20_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_20_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_20_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_20_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_20_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_20_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_20_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_20_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_20_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_20_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_20_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_20_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_20_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_20_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_20_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_20_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_20_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_24_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_24_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_24_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_24_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_24_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_24_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_24_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_24_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_24_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_24_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_24_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_24_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_24_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_24_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_24_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_24_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_24_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_32_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_32_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_32_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_32_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_32_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_32_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_32_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_32_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_32_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_32_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_32_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_32_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_32_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_32_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_32_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_32_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_2_1_32_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_2_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_8_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_8_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_8_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_8_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_8_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_8_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_8_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_8_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_8_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_8_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_8_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_8_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_8_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_8_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_8_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_8_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_8_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_16_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_16_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_16_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_16_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_16_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_16_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_16_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_16_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_16_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_16_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_16_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_16_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_16_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_16_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_16_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_16_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_16_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_20_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_20_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_20_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_20_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_20_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_20_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_20_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_20_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_20_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_20_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_20_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_20_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_20_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_20_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_20_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_20_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_20_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_24_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_24_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_24_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_24_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_24_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_24_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_24_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_24_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_24_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_24_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_24_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_24_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_24_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_24_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_24_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_24_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_24_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_32_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_32_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_32_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_32_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_32_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_32_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_32_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_32_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_32_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_32_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_32_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_32_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_32_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_32_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_32_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_32_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_0_32_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_0,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_8_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_8_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_8_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_8_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_8_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_8_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_8_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_8_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_8_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_8_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_8_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_8_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_8_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_8_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_8_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_8_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_8_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_16_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_16_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_16_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_16_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_16_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_16_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_16_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_16_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_16_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_16_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_16_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_16_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_16_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_16_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_16_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_16_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_16_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_20_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_20_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_20_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_20_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_20_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_20_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_20_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_20_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_20_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_20_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_20_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_20_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_20_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_20_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_20_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_20_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_20_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_24_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_24_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_24_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_24_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_24_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_24_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_24_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_24_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_24_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_24_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_24_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_24_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_24_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_24_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_24_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_24_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_24_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_32_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_32_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_32_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_32_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_32_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_32_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_32_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_32_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_32_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_32_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_32_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_32_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_32_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_32_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_32_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_32_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_4_1_32_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_4_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_8_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_8_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_8_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_8_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_8_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_8_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_8_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_8_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_8_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_8_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_8_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_8_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_8_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_8_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_8_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_8_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_8_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_16_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_16_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_16_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_16_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_16_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_16_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_16_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_16_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_16_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_16_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_16_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_16_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_16_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_16_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_16_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_16_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_16_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_20_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_20_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_20_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_20_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_20_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_20_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_20_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_20_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_20_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_20_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_20_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_20_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_20_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_20_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_20_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_20_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_20_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_24_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_24_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_24_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_24_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_24_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_24_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_24_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_24_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_24_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_24_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_24_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_24_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_24_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_24_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_24_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_24_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_24_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_32_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_32_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_32_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_32_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_32_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_32_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_32_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_32_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_32_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_32_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_32_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_32_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_32_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_32_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_32_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_32_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_5_1_32_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_5_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_8_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_8_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_8_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_8_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_8_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_8_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_8_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_8_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_8_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_8_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_8_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_8_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_8_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_8_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_8_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_8_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_8_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_16_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_16_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_16_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_16_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_16_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_16_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_16_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_16_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_16_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_16_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_16_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_16_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_16_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_16_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_16_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_16_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_16_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_20_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_20_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_20_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_20_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_20_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_20_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_20_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_20_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_20_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_20_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_20_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_20_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_20_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_20_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_20_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_20_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_20_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_24_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_24_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_24_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_24_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_24_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_24_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_24_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_24_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_24_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_24_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_24_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_24_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_24_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_24_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_24_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_24_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_24_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_32_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_32_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_32_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_32_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_32_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_32_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_32_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_32_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_32_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_32_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_32_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_32_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_32_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_32_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_32_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_32_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_6_1_32_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_6_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_8_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_8_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_8_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_8_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_8_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_8_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_8_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_8_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_8_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_8_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_8_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_8_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_8_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_8_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_8_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_8_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_8_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_8BITS,   LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_16_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_16_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_16_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_16_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_16_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_16_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_16_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_16_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_16_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_16_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_16_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_16_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_16_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_16_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_16_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_16_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_16_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_20_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_20_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_20_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_20_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_20_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_20_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_20_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_20_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_20_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_20_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_20_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_20_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_20_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_20_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_20_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_20_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_20_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_20BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_24_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_24_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_24_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_24_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_24_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_24_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_24_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_24_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_24_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_24_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_24_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_24_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_24_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_24_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_24_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_24_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_24_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_24BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_32_5512HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_5512HZ,    LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_32_8000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_8000HZ,    LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_32_11025HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_11025HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_32_12000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_12000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_32_16000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_16000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_32_22050HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_22050HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_32_24000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_24000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_32_32000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_32000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_32_44100HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_44100HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_32_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_32_64000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_64000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_32_88200HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_88200HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_32_96000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_96000HZ,   LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_32_176400HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_176400HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_32_192000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_192000HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_32_384000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_384000HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_7_1_32_768000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_768000HZ,  LAFMT_SS_32BITS,  LAFMT_CH_MULTICH_7_1,  LAFMT_PL_1),
    LAudioFormat_MULTICH_16_16_48000HZ
        =  MAKE_LAUDIOFORMAT(LAFMT_SR_48000HZ,   LAFMT_SS_16BITS,  LAFMT_CH_MULTICH_16,   LAFMT_PL_1),

    // Multi-Channels non-PCM audio format
    LAudioFormat_MULTICH_OGGVORBIS                      = 0x00070000,   // Compressed audio format OGGVorbis
    LAudioFormat_MULTICH_AAC                            = 0x00070001,   // Compressed audio format AAC
    LAudioFormat_IEC60958                               = 0x00070002,   // IEC-60958 stream

    // User defined audio format
    LAudioFormat_USER_DEFINED_MULTICH_14CH              = 0x00080000,   // User defined - 14 channels
    LAudioFormat_USER_DEFINED_MULTICH_16CH              = 0x00080001,   // User defined - 16 channels

    // Alias (default use 16-bits audio sample size, at 48kHz of sample rate)
    LAudioFormat_MONO                                   =   LAudioFormat_MONO_16_48000HZ,
    LAudioFormat_STEREO                                 =   LAudioFormat_STEREO_16_48000HZ,
    LAudioFormat_MULTICH_5_1                            =   LAudioFormat_MULTICH_5_1_16_48000HZ,
    LAudioFormat_MULTICH_7_1                            =   LAudioFormat_MULTICH_7_1_16_48000HZ,

    LAudioFormat_FORCE32                                = 0x7FFFFFFF,   // Dummy value to force enumeration
                                                                        // to use 32-bits.
} LAudioFormat;

/************************************************************************************************************\

Struct:         LAudioFormat_Descriptor

Description:    Structure containing informations about audio format description. Including the memory
                organization.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLAudioFormat_Descriptor
{
    LAudioFormat    eAudioFormat;               // Audio format. Must be a value from the LAudioFormat 
                                                // enumeration.
    MBOOL32         bPcmData;                   // PCM versus non-PCM data.
    MUINT32         uiSampleRate;               // Sample rate (in Hz)
    MUINT32         uiSampleSize;               // Sample size (in bits)
    MUINT32         uiNumberOfChannels;         // Number of channels.
    MUINT32         uiNumberOfPlans;            // Number of plans.
    MFLAG32         aflPlansChannelLayout[4];   // Channels layout description.
} LAudioFormat_Descriptor;


// -----------------------------------------------------------------------------------------------------------
//                      I N L I N E S   A N D   T E M P L A T E   D E F I N I T I O N S
// -----------------------------------------------------------------------------------------------------------

#if defined (__cplusplus)
extern "C" {
#endif

/************************************************************************************************************\

Function:       LAudioFormat_GetSampleRate

Description:    Gets the sample rate for the passed in audio format.

Parameters:     IN  eAudioFormat    Audio format enumeration value for which the sample rate should be
                                    return.

Return Value:   The sample rate, in Hz, of the audio format. 0 for non-PCM audio format.

Comments:       None.

\************************************************************************************************************/
LAPI MUINT32 LAudioFormat_GetSampleRate(
                LAudioFormat eAudioFormat);

/************************************************************************************************************\

Function:       LAudioFormat_GetSampleSize

Description:    Gets the number of bit per sample for the passed in audio format.

Parameters:     IN  eAudioFormat        Audio format enumeration value for which the sample size should be 
                                        return.

Return Value:   The number of bits per sample of the audio format. 0 for non-PCM audio format.

Comments:       None.

\************************************************************************************************************/
LAPI MUINT32 LAudioFormat_GetSampleSize(
                LAudioFormat eAudioFormat);

/************************************************************************************************************\

Function:       LAudioFormat_GetNumberOfChannel

Description:    Gets the number of channels for the passed in audio format.

Parameters:     IN  eAudioFormat        Audio format enumeration value for which the number of channels
                                        should be return.

Return Value:   The number of channel of the audio format. 0 for non-PCM audio format.

Comments:       None.

\************************************************************************************************************/
LAPI MUINT32 LAudioFormat_GetNumberOfChannel(
                LAudioFormat eAudioFormat);

/************************************************************************************************************\

Function:       LAudioFormat_GetNumberOfPlan

Description:    Gets the number of plans for the passed in audio format.

Parameters:     IN  eAudioFormat        Audio format enumeration value for which the number of plans should
                                        be return.

Return Value:   The number of plans of the audio format. 1 for non-PCM audio format.

Comments:       None.

\************************************************************************************************************/
LAPI MUINT32 LAudioFormat_GetNumberOfPlan(
                LAudioFormat eAudioFormat);

/************************************************************************************************************\

Function:       LAudioFormat_GetChannelLayout

Description:    Gets the channel layout for the specified plan of the passed in audio format.

Parameters:     IN  eAudioFormat        Audio format enumeration value for which the channel layout should be
                                        return.
                IN  uiPlanIndex         0-based index of the plan.
                OUT pflChannelLayout    Pointer to a MFLAG32 to be filled with the audio channel layout.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_INVALID_PARAM   The function aborted because its parameters are invalid.
                LStatus_FAIL            The function exited with a generic failure.
                LStatus_UNSUPPORTED     For non-PCM audio format.
                
Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioFormat_GetChannelLayout(
                LAudioFormat  eAudioFormat,
                MUINT32       uiPlanIndex,
                MFLAG32*      pflChannelLayout);

/************************************************************************************************************\

Function:       LAudioFormat_GetDescriptor

Description:    Gets the audio format descriptor for the passed in audio format.

Parameters:     IN  eAudioFormat        Audio format enumeration value for which the descriptor should be
                                        return.
                OUT poAudioDescriptor   Pointer to a LAudioFormat_Descriptor structure to be filled with the
                                        audio format description.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_INVALID_PARAM   The function aborted because its parameters are invalid.
                LStatus_FAIL            The function exited with a generic failure.
                
Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioFormat_GetDescriptor(
                LAudioFormat                eAudioFormat,
                LAudioFormat_Descriptor*    poAudioDescriptor);


/************************************************************************************************************\

Function:       LAudioFormat_GetAudioFormat

Description:    Retrieve an LAudioFormat based on audio parameters. Only works with PCM formats

Parameters:     IN  uiSampleRate        Sample rate of the audio stream.
                IN  uiSampleSize        Size of one audio sample in bits.
                IN  uiNbChannels        Number of channels in the audio stream.
                IN  uiNbPlans           Number of plans in the audio stream.
                OUT peAudioFormat       Pointer to a LAudioFormat to be filled with the resulting
                                        LAudioFormat

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_INVALID_PARAM   The function aborted because its parameters are invalid.
                LStatus_FAIL            The function exited with a generic failure.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioFormat_GetAudioFormat(
                MUINT32                     uiSampleRate,
                MUINT32                     uiSampleSize,
                MUINT32                     uiNbChannels,
                MUINT32                     uiNbPlans,
                LAudioFormat*               peAudioFormat);

/************************************************************************************************************\

Function:       LAudioFormat_GetBufferStrideSize

Description:    Gets the number of bytes required for each buffer stride defined by the passed in audio
                format.

Parameters:     IN  eAudioFormat            Audio format enumeration value for which the buffer stride should
                                            be return.
                OUT puiBufferStrideSize     Pointer to a MUINT32 to be filled with the number of byte
                                            required for each buffer stride defined by the passed in audio
                                            format.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_INVALID_PARAM   The function aborted because its parameters are invalid.
                LStatus_FAIL            The function exited with a generic failure.
                LStatus_UNSUPPORTED     For non-PCM audio format.
                
Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioFormat_GetBufferStrideSize(
                LAudioFormat    eAudioFormat,
                MUINT32*        puiBufferStrideSize);

/************************************************************************************************************\

Function:       LAudioFormat_ComputeBufferSizeBasedOnFrame

Description:    Computes the buffer size, in bytes, required to handle "uiNbFrame" audio frame per buffer 
                defined by the passed in audio format.

Parameters:     IN  eAudioFormat        Audio format enumeration value for which the buffer size should be
                                        computed.
                IN  uiNbFrame           Number of audio frame desired per buffer.
                OUT puiBufferSize       Pointer to a MUINT32 to be filled with the buffer size, in bytes, 
                                        required to handle "uiNbFrame" audio frame per buffer.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_INVALID_PARAM   The function aborted because its parameters are invalid.
                LStatus_FAIL            The function exited with a generic failure.
                LStatus_UNSUPPORTED     For non-PCM audio format.
                
Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioFormat_ComputeBufferSizeBasedOnFrame(
                LAudioFormat    eAudioFormat,
                MUINT32         uiNbFrame,
                MUINT32*        puiBufferSize);

#if defined (__cplusplus)
}
#endif

#endif  // INC_LAUDIOFORMAT_H
