/************************************************************************************************************\

Module Name:    LAudioCodec.h

Description:    Header declaration file of the audio codec Liberatus interface.

References:     LAudioCodec.doc

    Copyright (c) 2015, Matrox Graphics Inc.
    All Rights Reserved.

\************************************************************************************************************/

#ifndef INC_LAUDIOCODEC_H
#define INC_LAUDIOCODEC_H

/* -----------------------------------------------------------------------------------------------------------
                                     I N C L U D E S   A N D   U S I N G S
   ---------------------------------------------------------------------------------------------------------*/
#include "Liberatus.h"

/* -----------------------------------------------------------------------------------------------------------
                                     C O N S T A N T S   A N D   T Y P E S
   ---------------------------------------------------------------------------------------------------------*/

/**********************************************************************************************************\
                                             Common types
\**********************************************************************************************************/

/**********************************************************************************************************\

Enum:           LAudioCodec_PCMMode

Description:    Enumeration of the different PCM modes.

Comments:       None.

\**********************************************************************************************************/
typedef enum tagLAudioCodec_PCMMode
{
    LAudioCodec_PCMMode_LINEAR,
    LAudioCodec_PCMMode_FORCE32   = 0x7FFFFFFF
} LAudioCodec_PCMMode;

/**********************************************************************************************************\

Enum:           LAudioCodec_SamplingRate

Description:    Enumeration of the different sampling rates.

Comments:       None.

\**********************************************************************************************************/
typedef enum tagLAudioCodec_SamplingRate
{
    LAudioCodec_SamplingRate_UNKNOWN = 0,
    LAudioCodec_SamplingRate_8000Hz  = 8000,
    LAudioCodec_SamplingRate_11025Hz = 11025,
    LAudioCodec_SamplingRate_12000Hz = 12000,
    LAudioCodec_SamplingRate_16000Hz = 16000,
    LAudioCodec_SamplingRate_22050Hz = 22050,
    LAudioCodec_SamplingRate_24000Hz = 24000,
    LAudioCodec_SamplingRate_32000Hz = 32000,
    LAudioCodec_SamplingRate_44100Hz = 44100,
    LAudioCodec_SamplingRate_48000Hz = 48000,
    LAudioCodec_SamplingRate_88200Hz = 88200,
    LAudioCodec_SamplingRate_96000Hz = 96000,
    LAudioCodec_SamplingRate_FORCE32 = 0x7FFFFFFF
} LAudioCodec_SamplingRate;

/**********************************************************************************************************\

Enum:           LAudioCodec_BitDepth

Description:    Enumeration of the different bit depths.

Comments:       None.

\**********************************************************************************************************/
typedef enum tagLAudioCodec_BitDepth
{
    LAudioCodec_BitDepth_UNKNOWN = 0,
    LAudioCodec_BitDepth_16      = 16,
    LAudioCodec_BitDepth_FORCE32 = 0x7FFFFFFF
} LAudioCodec_BitDepth;

/**********************************************************************************************************\

Enum:           LAudioCodec_ChannelCfg

Description:    Enumeration of the different channel configurations.

Comments:       None.

\**********************************************************************************************************/
typedef enum tagLAudioCodec_ChannelCfg
{
    LAudioCodec_ChannelCfg_UNKNOWN  = 0,
    LAudioCodec_ChannelCfg_MONO     = 1,
    LAudioCodec_ChannelCfg_STEREO   = 2,
    LAudioCodec_ChannelCfg_FORCE32  = 0x7FFFFFFF
} LAudioCodec_ChannelCfg;

/**********************************************************************************************************\

Enum:           LAudioCodec_AudioAttributeTypeHeader

Description:    Enumeration of the different audio attribute types.

Comments:       None.

\**********************************************************************************************************/
typedef enum tagLAudioCodec_AudioAttributeTypeHeader
{
    LAudioCodec_AudioAttributeTypeHeader_STANDARD,
    LAudioCodec_AudioAttributeTypeHeader_FORCE32 = 0x7FFFFFFF
} LAudioCodec_AudioAttributeTypeHeader;

/************************************************************************************************************\

Structure:      LAudioCodec_AudioAttribute

Description:    Attributes for an audio stream.

                eType               Type header, must be LAudioCodec_AudioAttributeTypeHeader_STANDARD.
                ePCMMode            The PCM mode.
                eSamplingRate       The number of audio samples per second per channel.
                eChannelCfg         The channel configuration of the PCM stream.
                eBitDepth           The PCM bit depth (bits per PCM sample).
                flFlags             Flags, unused for now.

Comments:       Sampling rate constraints for AAC codec:
                 Profile LAudioCodec_AACProfile_LC:
                  - All available sampling rates are supported.
                 Profile LAudioCodec_AACProfile_HEV1 and LAudioCodec_AACProfile_HEV2:
                  - LAudioCodec_SamplingRate_32000Hz
                  - LAudioCodec_SamplingRate_44100Hz
                  - LAudioCodec_SamplingRate_48000Hz

                Channels constraints for AAC codec:
                 Profiles LAudioCodec_AACProfile_LC and LAudioCodec_AACProfile_HEV1:
                  - LAudioCodec_ChannelCfg_MONO
                  - LAudioCodec_ChannelCfg_STEREO
                 Profile LAudioCodec_AACProfile_HEV2:
                  - LAudioCodec_ChannelCfg_STEREO

\************************************************************************************************************/
typedef struct tagLAudioCodec_AudioAttribute
{
    LAudioCodec_AudioAttributeTypeHeader    eType;
    LAudioCodec_PCMMode                     ePCMMode;
    LAudioCodec_SamplingRate                eSamplingRate;
    LAudioCodec_ChannelCfg                  eChannelCfg;
    LAudioCodec_BitDepth                    eBitDepth;
    MFLAG32                                 flFlags;
} LAudioCodec_AudioAttribute;

/**********************************************************************************************************\

Enum:           LAudioCodec_AACStreamFormat

Description:    Enumeration of the different AAC stream formats.

Comments:       None.

\**********************************************************************************************************/
typedef enum tagLAudioCodec_AACStreamFormat
{
    LAudioCodec_AACStreamFormat_UNKNOWN,
    LAudioCodec_AACStreamFormat_RAW,
    LAudioCodec_AACStreamFormat_ADTS,
    LAudioCodec_AACStreamFormat_ADIF
} LAudioCodec_AACStreamFormat;

/**********************************************************************************************************\

Enum:           LAudioCodec_AACProfile

Description:    Enumeration of the different AAC profiles.

Comments:       None.

\**********************************************************************************************************/
typedef enum tagLAudioCodec_AACProfile
{
    LAudioCodec_AACProfile_UNKNOWN,
    LAudioCodec_AACProfile_LC,
    LAudioCodec_AACProfile_HEV1,
    LAudioCodec_AACProfile_HEV2,
    LAudioCodec_AACProfile_FORCE32 = 0x7FFFFFFF
} LAudioCodec_AACProfile;

/**********************************************************************************************************\

Enum:           LAudioCodec_AACQualityLevel

Description:    Enumeration of the different AAC quality levels.

Comments:       - Higher quality level results in more processing, but does not affect bitrate.
                - Highest quality level is LAudioCodec_AACQualityLevel_2.

\**********************************************************************************************************/
typedef enum tagLAudioCodec_AACQualityLevel
{
    LAudioCodec_AACQualityLevel_0       = 0x00000000,
    LAudioCodec_AACQualityLevel_1       = 0x00000001,
    LAudioCodec_AACQualityLevel_2       = 0x00000002,
    LAudioCodec_AACQualityLevel_FORCE32 = 0x7FFFFFFF
} LAudioCodec_AACQualityLevel;

/**********************************************************************************************************\

Enum:           LAudioCodec_BufferInfoTypeHeader

Description:    Enumeration of the different buffer info types.

Comments:       None.

\**********************************************************************************************************/
typedef enum tagLAudioCodec_BufferInfoTypeHeader
{
    LAudioCodec_BufferInfoTypeHeader_STANDARD,
    LAudioCodec_BufferInfoTypeHeader_FORCE32 = 0x7FFFFFFF
} LAudioCodec_BufferInfoTypeHeader;

/************************************************************************************************************\

Structure:      LAudioCodec_BufferInfo

Description:    Information on an audio buffer.

                eType               Type header, must be LAudioCodec_BufferInfoTypeHeader_STANDARD
                hBuffer             Handle to a LBuffer where the data is stored.
                uiStartOffset       The offset at which the data begins in the buffer.
                uiSize              The size in bytes of the data contained in the buffer after the start
                                    offset.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLAudioCodec_BufferInfo
{
    LAudioCodec_BufferInfoTypeHeader    eType;
    LBuffer_Handle                      hBuffer;
    MUINT32                             uiStartOffset;
    MUINT32                             uiSize;
} LAudioCodec_BufferInfo;

/**********************************************************************************************************\
                                           Decoder types
\**********************************************************************************************************/

typedef struct LAUDIODECODER_OBJECT*  LAudioDecoder_Handle;

/**********************************************************************************************************\

Enum:           LAudioDecoder_AACDecodingFlags

Description:    Enumeration of the different AAC decoding flags.
                 LAudioDecoder_AACFlags_INTERLEAVE_MONO_TO_STEREO   Interleave mono input to stereo output.
                                                                    Ignored when input is stereo.
                 LAudioDecoder_AACFlags_DOWNMIX_STEREO              Downmix stereo input to mono output.
                                                                    Ignored when input is mono.

Comments:       None.

\**********************************************************************************************************/
typedef enum tagLAudioDecoder_AACFlags
{
    LAudioDecoder_AACFlags_INTERLEAVE_MONO_TO_STEREO    = 0x00000001,
    LAudioDecoder_AACFlags_DOWNMIX_STEREO               = 0x00000002,
    LAudioDecoder_AACFlags_FORCE32                      = 0x7FFFFFFF
} LAudioDecoder_AACFlags;


/**********************************************************************************************************\

Enum:           LAudioDecoder_CodecOptTypeHeader

Description:    Enumeration of the different audio decoder codec specific options types.

Comments:       None.

\**********************************************************************************************************/
typedef enum tagLAudioDecoder_CodecOptTypeHeader
{
    LAudioDecoder_CodecOptTypeHeader_AAC,
    LAudioDecoder_CodecOptTypeHeader_FORCE32 = 0x7FFFFFFF
} LAudioDecoder_CodecOptTypeHeader;

/************************************************************************************************************\

Structure:      LAudioDecoder_AACOptions

Description:    AAC decoder codec specific options.

                eType            Type header, must be LAudioDecoder_CodecOptTypeHeader_AAC
                eStreamFormat    AAC Stream format. must be set to LAudioCodec_AACStreamFormat_RAW if stream
                                 is raw.
                eProfile         AAC profile. Must be set to the appropriate value if stream format is raw.
                bIsFramed        Specifies whether the input AAC stream is framed or not. Set to MTRUE if
                                 all buffers sent with LAudioDecoder_PutData contain one complete AAC frame.
                                 Set to MFALSE otherwise.
                flFlags          AAC decoding flags (see LAudioDecoder_AACFlags).

Comments:       None.

\************************************************************************************************************/
typedef struct tagLAudioDecoder_AACOptions
{
    LAudioDecoder_CodecOptTypeHeader    eType;
    LAudioCodec_AACStreamFormat         eStreamFormat;
    LAudioCodec_AACProfile              eProfile;
    MBOOL32                             bIsFramed;
    MFLAG32                             flFlags;
} LAudioDecoder_AACOptions;

/**********************************************************************************************************\

Enum:           LAudioDecoder_CreateOptTypeHeader

Description:    Enumeration of the different audio decoder create options types.

Comments:       None.

\**********************************************************************************************************/
typedef enum tagLAudioDecoder_CreateOptTypeHeader
{
    LAudioDecoder_CreateOptTypeHeader_STANDARD,
    LAudioDecoder_CreateOptTypeHeader_FORCE32 = 0x7FFFFFFF
} LAudioDecoder_CreateOptTypeHeader;

/************************************************************************************************************\

Structure:      LAudioDecoder_CreateOpt

Description:    Options for audio decoder creation.

                eType                       Type header, must be LAudioDecoder_CreateOptTypeHeader_STANDARD.
                hDevice                     Liberatus device handle. Used to create output buffers.
                oEncodedAudioAttributes     Attributes of the encoded input stream. If the input stream is
                                            raw, attribute eSamplingRate must be set to the sampling rate of
                                            the input stream.
                uiNbOutputBuffers           Number of output buffers to allocate.
                eOutputBufferType           The type of LBuffer that will be created as output buffers.
                peDecoderCodecOptions       Pointer to codec specific options (passed by type header).
Comments:       None.

\************************************************************************************************************/
typedef struct tagLAudioDecoder_CreateOpt
{
    LAudioDecoder_CreateOptTypeHeader   eType;
    LDevice_Handle                      hDevice;
    LAudioCodec_AudioAttribute          oEncodedAudioAttributes;
    MUINT32                             uiNbOutputBuffers;
    LBuffer_Type                        eOutputBufferType;
    LAudioDecoder_CodecOptTypeHeader*   peDecoderCodecOptions;
} LAudioDecoder_CreateOpt;

/**********************************************************************************************************\

Enum:           LAudioDecoder_DecodeOptTypeHeader

Description:    Enumeration of different buffer decode options types.

Comments:       None.

\**********************************************************************************************************/
typedef enum tagLAudioDecoder_DecodeOptTypeHeader
{
    LAudioDecoder_DecodeOptTypeHeader_STANDARD,
    LAudioDecoder_DecodeOptTypeHeader_INIT_DECODER,
    LAudioDecoder_DecodeOptTypeHeader_FORCE32 = 0x7FFFFFFF
} LAudioDecoder_DecodeOptTypeHeader;

/************************************************************************************************************\

Structure:      LAudioDecoder_DecodeOpt

Description:    Options for audio buffer decoding.

                eType                       Type header, must be LAudioDecoder_DecodeOptTypeHeader_STANDARD
                bReplaceOldestBuffer        If set to MTRUE and there is no more place to store the new
                                            decoded buffer, the oldest buffer waiting to be collected with
                                            LAudioDecoder_GetNextBuffer will be replaced with the new decoded
                                            buffer.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLAudioDecoder_DecodeOpt
{
    LAudioDecoder_DecodeOptTypeHeader     eType;
    MBOOL32                               bReplaceOldestBuffer;
} LAudioDecoder_DecodeOpt;

/************************************************************************************************************\

Structure:      LAudioDecoder_InitDecoderOpt

Description:    Options for audio decoder initialization.

                eType                       Type header, must be
                                            LAudioDecoder_DecodeOptTypeHeader_INIT_DECODER

Comments:       There are no parameters associated with decoder initialization, the type header by itself
                is sufficient. This structure only exists for clarity, enforcing the rule that each type 
                header is associated with one type of structure.

\************************************************************************************************************/
typedef struct tagLAudioDecoder_InitDecoderOpt
{
    LAudioDecoder_DecodeOptTypeHeader     eType;
} LAudioDecoder_InitDecoderOpt;

/**********************************************************************************************************\

Enum:           LAudioDecoder_InfoTypeHeader

Description:    Enumeration of the different decoder information types.

Comments:       None.

\**********************************************************************************************************/
typedef enum tagLAudioDecoder_InfoTypeHeader
{
    LAudioDecoder_InfoTypeHeader_GENERIC_STREAM_INFO,
    LAudioDecoder_InfoTypeHeader_AAC_STREAM_INFO,
    LAudioDecoder_InfoTypeHeader_BUFFER_INFO,
    LAudioDecoder_InfoTypeHeader_FORCE32 = 0x7FFFFFFF
} LAudioDecoder_InfoTypeHeader;

/************************************************************************************************************\

Structure:      LAudioDecoder_GenericStreamInfo

Description:    Generic information on the audio stream.

                eType               Type header, must be LAudioDecoder_InfoTypeHeader_GENERIC_STREAM_INFO.
                eSamplingRate       Sampling rate.
                eChannelCfg         Channel configuration.
                eBitDepth           Bit depth.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLAudioDecoder_GenericStreamInfo
{
    LAudioDecoder_InfoTypeHeader       eType;
    LAudioCodec_SamplingRate           eSamplingRate;
    LAudioCodec_ChannelCfg             eChannelCfg;
    LAudioCodec_BitDepth               eBitDepth;
} LAudioDecoder_GenericStreamInfo;

/************************************************************************************************************\

Structure:      LAudioDecoder_AACStreamInfo

Description:    Information on the AAC stream.

                eType               Type header, must be LAudioDecoder_InfoTypeHeader_AAC_STREAM_INFO.
                eProfile            AAC profile.
                eStreamFormat       AAC stream format.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLAudioDecoder_AACStreamInfo
{
    LAudioDecoder_InfoTypeHeader    eType;
    LAudioCodec_AACProfile          eProfile;
    LAudioCodec_AACStreamFormat     eStreamFormat;
} LAudioDecoder_AACStreamInfo;

/************************************************************************************************************\

Structure:      LAudioDecoder_BufferInfo

Description:    Information on the input/output buffers provided to/by the LAudioDecoder module.

                eType                           Type header, must be LAudioDecoder_InfoTypeHeader_BUFFER_INFO.
                uiRecommendedInputBufferSize    Recommended size of the buffers provided to the decoder with
                                                LAudioDecoder_PutData.
                uiMaximumOutputBufferSize       Maximum size of an decoded buffer collected with
                                                LAudioDecoder_GetNextBuffer.
                ePreferredInputBufferType       Preferred type of LBuffer provided to LAudioDecoder_PutData.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLAudioDecoder_BufferInfo
{
    LAudioDecoder_InfoTypeHeader eType;
    MUINT32                      uiRecommendedInputBufferSize;
    MUINT32                      uiMaximumOutputBufferSize;
    LBuffer_Type                 ePreferredInputBufferType;
} LAudioDecoder_BufferInfo;

/**********************************************************************************************************\
                                            Encoder types
\**********************************************************************************************************/

typedef struct LAUDIOENCODER_OBJECT*  LAudioEncoder_Handle;

/**********************************************************************************************************\

Enum:           LAudioEncoder_AACFlags

Description:    Enumeration of the different AAC encoding flags.
                 LAudioEncoder_AACFlags_USE_TNS         Use Temporal Noise Shaping.

Comments:       None.

\**********************************************************************************************************/
typedef enum tagLAudioEncoder_AACFlags
{
    LAudioEncoder_AACFlags_USE_TNS          = 0x00000001,
    LAudioEncoder_AACFlags_FORCE32          = 0x7FFFFFFF
} LAudioEncoder_AACFlags;

/**********************************************************************************************************\

Enum:           LAudioEncoder_CodecOptTypeHeader

Description:    Enumeration of the different audio encoder codec specific options types.

Comments:       None.

\**********************************************************************************************************/
typedef enum tagLAudioEncoder_CodecOptTypeHeader
{
    LAudioEncoder_CodecOptTypeHeader_AAC,
    LAudioEncoder_CodecOptTypeHeader_FORCE32 = 0x7FFFFFFF
} LAudioEncoder_CodecOptTypeHeader;

/************************************************************************************************************\

Structure:      LAudioEncoder_AACOptions

Description:    AAC encoder codec specific options.

                eType                   Type header, must be LAudioEncoder_CodecOptTypeHeader_AAC.
                uiBitRate               Encoding bit rate in bits per second.
                eStreamFormat           Format of the output AAC stream.
                eProfile                Encoding profile.
                flEncodingFlags         Encoding flags (LAudioEncoder_AACFlags).

Comments:       Bit rate constraints:
                 Minimum bit rate:
                   MinBitRate(bps) = 32000
                 Maximum bit rate for profile LAudioCodec_AACProfile_LC:
                   MaxBitRate(bps) = SamplingRate(Hz) * NumberOfChannels * 6
                 Maximum bit rate for profile LAudioCodec_AACProfile_HEV1:
                   MaxBitRate(bps) = SamplingRate(Hz) * NumberOfChannels * 3
                 Maximum bit rate for profile LAudioCodec_AACProfile_HEV2:
                   MaxBitRate(bps) = SamplingRate(Hz) * 3

\************************************************************************************************************/
typedef struct tagLAudioEncoder_AACOptions
{
    LAudioEncoder_CodecOptTypeHeader eType;
    MUINT32                          uiBitRate;
    LAudioCodec_AACStreamFormat      eStreamFormat;
    LAudioCodec_AACProfile           eProfile;
    LAudioCodec_AACQualityLevel      eQualityLevel;
    MFLAG32                          flEncodingFlags;
} LAudioEncoder_AACOptions;

/**********************************************************************************************************\

Enum:           LAudioEncoder_CreateOptTypeHeader

Description:    Enumeration of the different audio encoder create options types.

Comments:       None.

\**********************************************************************************************************/
typedef enum tagLAudioEncoder_CreateOptTypeHeader
{
    LAudioEncoder_CreateOptTypeHeader_STANDARD,
    LAudioEncoder_CreateOptTypeHeader_FORCE32 = 0x7FFFFFFF
} LAudioEncoder_CreateOptTypeHeader;

/************************************************************************************************************\

Structure:      LAudioEncoder_CreateOpt

Description:    Options for audio encoder creation.

                eType                       Type header, must be LAudioEncoder_CreateOptTypeHeader_STANDARD.
                hDevice                     Liberatus device handle. Used to create output buffers.
                oPCMAudioAttributes         Attributes of the input PCM stream.
                uiNbOutputBuffers           Number of output buffers to allocate.
                eOutputBufferType           The type of LBuffer that will be created as output buffers.
                peEncoderCodecOptions       Pointer to codec specific options (passed by type header).

Comments:       None.

\************************************************************************************************************/
typedef struct tagLAudioEncoder_CreateOpt
{
    LAudioEncoder_CreateOptTypeHeader     eType;
    LDevice_Handle                        hDevice;
    LAudioCodec_AudioAttribute            oPCMAudioAttributes;
    MUINT32                               uiNbOutputBuffers;
    LBuffer_Type                          eOutputBufferType;
    LAudioEncoder_CodecOptTypeHeader*     peEncoderCodecOptions;
} LAudioEncoder_CreateOpt;

/**********************************************************************************************************\

Enum:           LAudioEncoder_EncodeOptTypeHeader

Description:    Enumeration of different buffer encode options types.

Comments:       None.

\**********************************************************************************************************/
typedef enum tagLAudioEncoder_EncodeOptTypeHeader
{
    LAudioEncoder_EncodeOptTypeHeader_STANDARD,
    LAudioEncoder_EncodeOptTypeHeader_FORCE32 = 0x7FFFFFFF
} LAudioEncoder_EncodeOptTypeHeader;

/************************************************************************************************************\

Structure:      LAudioEncoder_EncodeOpt

Description:    Options for audio buffer encoding.

                eType                       Type header, must be LAudioEncoder_EncodeOptTypeHeader_STANDARD
                bReplaceOldestBuffer        If set to MTRUE and there is no more place to store the new
                                            encoded buffer, the oldest buffer waiting to be collected with
                                            LAudioEncoder_GetNextBuffer will be replaced with the new
                                            encoded buffer.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLAudioEncoder_EncodeOpt
{
    LAudioEncoder_EncodeOptTypeHeader     eType;
    MBOOL32                               bReplaceOldestBuffer;
} LAudioEncoder_EncodeOpt;

/**********************************************************************************************************\

Enum:           LAudioEncoder_InfoTypeHeader

Description:    Enumeration of the different encoder info types.

Comments:       None.

\**********************************************************************************************************/
typedef enum tagLAudioEncoder_InfoTypeHeader
{
    LAudioEncoder_InfoTypeHeader_BUFFER_INFO,
    LAudioEncoder_InfoTypeHeader_FORCE32 = 0x7FFFFFFF
} LAudioEncoder_InfoTypeHeader;

/************************************************************************************************************\

Structure:      LAudioEncoder_BufferInfo

Description:    Information on the input/output buffers provided to/by the LAudioEncoder module.

                eType                       Type header, must be LAudioEncoder_InfoTypeHeader_BUFFER_INFO.
                uiRecommendedInBufSize      Recommended size of the buffers provided to the encoder with
                                            LAudioEncoder_PutData.
                uiMaximumOutputBufferSize   Maximum size of an encoder buffer collected with
                                            LAudioEncoder_GetNextBuffer.
                ePreferredInputBufferType   Preferred type of LBuffer provided to LAudioEncoder_PutData.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLAudioEncoder_BufferInfo
{
    LAudioEncoder_InfoTypeHeader eType;
    MUINT32                      uiRecommendedInputBufferSize;
    MUINT32                      uiMaximumOutputBufferSize;
    LBuffer_Type                 ePreferredInputBufferType;
} LAudioEncoder_BufferInfo;

/* -----------------------------------------------------------------------------------------------------------
                             G L O B A L   V A R I A B L E   R E F E R E N C E S
   ---------------------------------------------------------------------------------------------------------*/

/* -----------------------------------------------------------------------------------------------------------
                                                 U S A G E
   ---------------------------------------------------------------------------------------------------------*/

/* -----------------------------------------------------------------------------------------------------------
                        I N L I N E S   A N D   T E M P L A T E   D E F I N I T I O N S
   ---------------------------------------------------------------------------------------------------------*/

#if defined (__cplusplus)
extern "C" {
#endif


/************************************************************************************************************\
                                             Decoder interface
\************************************************************************************************************/


/************************************************************************************************************\

Function:       LAudioDecoder_Create

Description:    Creates an audio decoder.

Parameters:     IN  peCreateOpt             Options for creating a decoder (passed by type header).
                OUT phAudioDec              Decoder handle returned.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioDecoder_Create
(
    LAudioDecoder_CreateOptTypeHeader* peCreateOpt,
    LAudioDecoder_Handle* phAudioDec
);

/************************************************************************************************************\

Function:       LAudioDecoder_Destroy

Description:    Destroys a decoder.

Parameters:     IN hAudioDec                Handle to a decoder.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.

Comments:       None.


\************************************************************************************************************/
LAPI LStatus LAudioDecoder_Destroy
(
    LAudioDecoder_Handle hAudioDec
);

/************************************************************************************************************\

Function:       LAudioDecoder_PutData

Description:    Sends a buffer filled with encoded data to the decoder to be copied internally.

Parameters:     IN  hAudioDec               Handle to an decoder.
                IN  peBufferInfo            Buffer information (passed by type header). must be
                                            LAudioCodec_BufferInfo_STANDARD.
                IN  uiUserData              User data associated with this chunk of encoded data, retrieved
                                            in LAudioDecoder_GetNextBuffer.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.
                LStatus_OUT_OF_RESOURCES    Function failed because internal buffer is full. Application needs
                                            to process data stored inside the internal buffer using
                                            LAudioDecoder_DecodeBuffer.

Comments:       A call to this function resets the end-of-stream status set with LAudioDecoder_EndStream.

                uiUserData:

                When input is not framed (see bIsFramed field in LAudioDecoder_AACOptions), the
                chunks of data copied with LAudioDecoder_PutData may contain more or less than the data that
                is needed to decode one audio buffer, so the following rules apply:

                  1 - If the chunk of data copied with LAudioDecoder_PutData contains more than the data that
                      is needed to decode one buffer, then the user data associated with this chunk of data
                      will be returned by LAudioDecoder_GetNextBuffer for all the decoded buffers originated,
                      partly or entirely, from this chunk of data.

                      Example:

                      PutData(UserData1)                   PutData(UserData2)     PutData(UserData3)
                      <-----------------------------------><---------------------><--------------...
                      [---BUFFER1---][---BUFFER2---][---BUFFER3---][---BUFFER4---][---BUFFER5---]...

                      BUFFER1: LAudioDecoder_GetNextBuffer() --> returns UserData1
                      BUFFER2: LAudioDecoder_GetNextBuffer() --> returns UserData1
                      BUFFER3: LAudioDecoder_GetNextBuffer() --> returns UserData1
                      BUFFER4: LAudioDecoder_GetNextBuffer() --> returns UserData2
                      BUFFER5: LAudioDecoder_GetNextBuffer() --> returns UserData3

                  2 - If the chunk of data copied with LAudioDecoder_PutData contains less than the data that
                      is needed to decode one buffer, then the user data associated with the first chunk of
                      data which the decoded buffer is originated from will be returned by
                      LAudioDecoder_GetNextBuffer.

                      Example:

                      PutData(UserData1)   PutData(UserData2)   PutData(UserData3)  PutData(UserData4)
                      <-------------------><-------------------><------------------><-----------------...
                      [-------------------BUFFER1-------------------][-------------------BUFFER2------...

                      BUFFER1: LAudioDecoder_GetNextBuffer() --> returns UserData1
                      BUFFER2: LAudioDecoder_GetNextBuffer() --> returns UserData3

\************************************************************************************************************/
LAPI LStatus LAudioDecoder_PutData
(
    LAudioDecoder_Handle                hAudioDec,
    LAudioCodec_BufferInfoTypeHeader*   peBufferInfo,
    MUINT64                             uiUserData
);

/************************************************************************************************************\

Function:       LAudioDecoder_DecodeBuffer

Description:    Performs a decode buffer operation. This is used to decode a buffer or to initialize the
                decoder by parsing a stream header, if any, and/or by decoding the first buffer of the stream.

Parameters:     IN  hAudioDec              Handle to a decoder.
                IN  peDecodeOpt            Options to decode the audio buffer (passed by type header).
                                           - LAudioDecoder_DecodeOptTypeHeader_INIT_DECODER:
                                              End initialization of the decoder by parsing stream header.
                                           - LAudioDecoder_DecodeOptTypeHeader_STANDARD:
                                              Perform a decode buffer operation.
                OUT puiNbDecodedBytes      The number of bytes that were decoded from the input buffer.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_FAIL                Function failed, likely because of corrupted stream.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.
                LStatus_NO_MORE_DATA        Function failed because not enough data is stored internally to
                                            perform a decode buffer operation. Application needs to send
                                            more encoded data with LAudioDecoder_PutData.
                LStatus_OUT_OF_RESOURCES    Function failed because there is no more place to store the
                                            decoded buffer. This is likely because the application does
                                            not call LAudioDecoder_GetNextBuffer fast enough. To avoid this,
                                            application can set the field bReplaceOldestBuffer in
                                            LAudioDecoder_DecodeOpt.
                LStatus_END_OF_STREAM       The stream has been ended with LAudioDecoder_EndStream and the
                                            internally stored encoded data has all been processed.

Comments:       When this function returns LStatus_OK for the decode buffer operation, it means that one
                buffer has been decoded and that it can now be retrieved with LAudioDecoder_GetNextBuffer.
                If the application wants to decode as much of the data copied with LAudioDecoder_PutData as
                possible, it can call LAudioDecoder_DecodeBuffer in a loop until it fails.

\************************************************************************************************************/
LAPI LStatus LAudioDecoder_DecodeBuffer
(
    LAudioDecoder_Handle                hAudioDec,
    LAudioDecoder_DecodeOptTypeHeader*  peDecodeOpt,
    MUINT32*                            puiNbDecodedBytes
);

/************************************************************************************************************\

Function:       LAudioDecoder_GetNextBuffer

Description:    Retrieves the next decoded audio buffer.

Parameters:     IN  hAudioDec               Handle to a decoder.
                OUT peBufferInfo            Information on the buffer (passed by type header).
                IN  uiTimeout               Delay to wait, in milliseconds, for a new buffer:
                                             - 0                 = no wait
                                             - LINFINITE_TIMEOUT = wait until a decoded buffer is available.
                OUT puiUserData             User data passed to the corresponding call to
                                            LAudioDecoder_PutData for this buffer. Can be MNULL if unused.
                OUT puiNbLost               The number of buffers that were overwritten by calls to
                                            LAudioDecoder_DecodeBuffer since the last call to
                                            LAudioDecoder_GetNextBuffer. Can be MNULL.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_END_OF_STREAM       The stream is ended and no more buffers are available.
                                            This is returned once all the following conditions are met:
                                             1 - LAudioDecoder_EndStream has been called.
                                             2 - All pending input data has been decoded.
                                             3 - All decoded output buffers have been collected.
                LStatus_TIMEOUT             Timeout expired. No new data is available yet.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioDecoder_GetNextBuffer
(
    LAudioDecoder_Handle                hAudioDec,
    LAudioCodec_BufferInfoTypeHeader*   peBufferInfo,
    MUINT32                             uiTimeout,
    MUINT64*                            puiUserData,
    MUINT32*                            puiNbLost
);

/************************************************************************************************************\

Function:       LAudioDecoder_ReleaseBuffer

Description:    Releases a decoded buffer back to the decoder.

Parameters:     IN  hAudioDec               Handle to a decoder.
                IN  hBuffer                 Handle to the LBuffer to release.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioDecoder_ReleaseBuffer
(
    LAudioDecoder_Handle            hAudioDec,
    LBuffer_Handle                  hBuffer
);

/************************************************************************************************************\

Function:       LAudioDecoder_GetInfo

Description:    Retrieves information on the decoder and the current stream.

Parameters:     IN  hAudioDec               Handle to a decoder.
                OUT peInfo                  Information to query (passed by type header).

Return Value:   LStatus_OK                  Function completed succesfully.
                LStatus_INFO_NOT_AVAILABLE  Requested information is currently unavailable.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioDecoder_GetInfo
(
    LAudioDecoder_Handle            hAudioDec,
    LAudioDecoder_InfoTypeHeader*   peInfo
);

/************************************************************************************************************\

Function:       LAudioDecoder_EndStream

Description:    Notifies the decoder that no more data will be sent through LAudioDecoder_PutData for
                this stream. If bFlushPendingData is set, all pending data sent with LAudioDecoder_PutData
                as well as all already decoded buffers waiting to be collected with
                LAudioDecoder_GetNextBuffer will be immediately discarded.

Parameters:     IN  hAudioDec               Handle to a decoder.
                IN  bFlushPendingData       Discard all pending data.
                
Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.
              
Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioDecoder_EndStream
(
    LAudioDecoder_Handle    hAudioDec,
    MBOOL32                 bFlushPendingData
);


/************************************************************************************************************\
                                             Encoder interface
\************************************************************************************************************/


/************************************************************************************************************\

Function:       LAudioEncoder_Create

Description:    Creates an encoder.

Parameters:     IN  peCreateOpt             Options for creating an encoder (passed by type header).
                OUT phAudioEnc              Encoder handle returned.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioEncoder_Create
(
    LAudioEncoder_CreateOptTypeHeader* peCreateOpt,
    LAudioEncoder_Handle*              phAudioEnc
);

/************************************************************************************************************\

Function:       LAudioEncoder_Destroy

Description:    Destroys an encoder.

Parameters:     IN hAudioEnc                Handle to an encoder.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioEncoder_Destroy
(
    LAudioEncoder_Handle hAudioEnc
);

/************************************************************************************************************\

Function:       LAudioEncoder_PutData

Description:    Sends a buffer filled with PCM data to the encoder to be copied internally.

Parameters:     IN  hAudioEnc               Handle to an encoder.
                IN  peBufferInfo            Buffer information (passed by type header), must be
                                            LAudioCodec_BufferInfo_STANDARD.
                IN  uiUserData              User data associated with this chunk of data, retrieved in
                                            LAudioEncoder_GetNextBuffer.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.
                LStatus_OUT_OF_RESOURCES    Function failed because internal buffer is full. Application needs
                                            to process data stored inside the internal buffer with
                                            LAudioEncoder_EncodeBuffer.

Comments:       A call to this function resets the end-of-stream status set with LAudioEncoder_EndStream.

                uiUserData:

                Since the chunks of data copied with LAudioEncoder_PutData may contain more or less than the
                data that is needed to encode one audio buffer, the following rules apply:

                  1 - If the chunk of data copied with LAudioEncoder_PutData contains more than the data that
                      is needed to encode one buffer, then the user data associated with this chunk of data
                      will be returned by LAudioEncoder_GetNextBuffer for all the encoded buffers originated,
                      partly or entirely, from this chunk of data.

                      Example:

                      PutData(UserData1)                   PutData(UserData2)     PutData(UserData3)
                      <-----------------------------------><---------------------><--------------...
                      [---BUFFER1---][---BUFFER2---][---BUFFER3---][---BUFFER4---][---BUFFER5---]...

                      BUFFER1: LAudioEncoder_GetNextBuffer() --> returns UserData1
                      BUFFER2: LAudioEncoder_GetNextBuffer() --> returns UserData1
                      BUFFER3: LAudioEncoder_GetNextBuffer() --> returns UserData1
                      BUFFER4: LAudioEncoder_GetNextBuffer() --> returns UserData2
                      BUFFER5: LAudioEncoder_GetNextBuffer() --> returns UserData3

                  2 - If the chunk of data copied with LAudioEncoder_PutData contains less than the data that
                      is needed to encode one buffer, then the user data associated with the first chunk of
                      data which the encoded buffer is originated from will be returned by
                      LAudioEncoder_GetNextBuffer.

                      Example:

                      PutData(UserData1)   PutData(UserData2)   PutData(UserData3)  PutData(UserData4)
                      <-------------------><-------------------><------------------><-----------------...
                      [-------------------BUFFER1-------------------][-------------------BUFFER2------...

                      BUFFER1: LAudioEncoder_GetNextBuffer() --> returns UserData1
                      BUFFER2: LAudioEncoder_GetNextBuffer() --> returns UserData3

\************************************************************************************************************/
LAPI LStatus LAudioEncoder_PutData
(
    LAudioEncoder_Handle                hAudioEnc,
    LAudioCodec_BufferInfoTypeHeader*   peBufferInfo,
    MUINT64                             uiUserData
);

/************************************************************************************************************\

Function:       LAudioEncoder_EncodeBuffer

Description:    Performs an encode buffer operation.

Parameters:     IN  hAudioEnc               Handle to an encoder.
                IN  peEncodeOpt             Options to encode the audio buffer (passed by type header). Must
                                            be LAudioEncoder_EncodeOptTypeHeader.
                
Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.
                LStatus_NO_MORE_DATA        Function failed because not enough data is stored internally to
                                            perform an encode buffer operation. Application needs to send
                                            more raw data with LAudioEncoder_PutData.
                LStatus_OUT_OF_RESOURCES    Function failed because there is no more place to store the
                                            encoded buffer. This is likely because the application does not
                                            call LAudioEncoder_GetNextBuffer fast enough. To avoid this,
                                            application can set the field bReplaceOldestBuffer in
                                            LAudioEncoder_EncodeOpt.
                LStatus_END_OF_STREAM       The stream has been ended with LAudioEncoder_EndStream and the
                                            internally stored raw data has all been processed.
              
Comments:       When this function returns LStatus_OK, it means that one buffer has been encoded and that it
                can now be retrieved with LAudioEncoder_GetNextBuffer. If the application wants to encode as
                much of the data copied with LAudioEncoder_PutData as possible, it can call
                LAudioEncoder_EncodeBuffer in a loop until it fails.

\************************************************************************************************************/
LAPI LStatus LAudioEncoder_EncodeBuffer
(
    LAudioEncoder_Handle                hAudioEnc,
    LAudioEncoder_EncodeOptTypeHeader*  peEncodeOpt
);

/************************************************************************************************************\

Function:       LAudioEncoder_GetNextBuffer

Description:    Retrieves the next encoded audio buffer.

Parameters:     IN  hAudioEnc               Handle to an encoder.
                OUT peBufferInfo            Information on the buffer (passed by type header).
                IN  uiTimeout               Delay to wait, in milliseconds, for a new buffer:
                                             - 0                 = no wait
                                             - LINFINITE_TIMEOUT = wait until an encoded buffer is available
                OUT puiUserData             User data passed to the corresponding call to
                                            LAudioEncoder_PutData for this buffer. Can be MNULL if unused.
                OUT puiNbLost               The number of buffers that were overwritten by calls to
                                            LAudioEncoder_EncodeBuffer since the last call to
                                            LAudioEncoder_GetNextBuffer. Can be MNULL.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.
                LStatus_END_OF_STREAM       The stream is ended and no more buffers are available.
                                            This is returned once all the following conditions are met:
                                             1 - LAudioEncoder_EndStream has been called.
                                             2 - All pending input data has been encoded.
                                             3 - All encoded output buffers have been collected.
                LStatus_TIMEOUT             Timeout expired. No new data is available yet.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioEncoder_GetNextBuffer
(
    LAudioEncoder_Handle                hAudioEnc,
    LAudioCodec_BufferInfoTypeHeader*   peBufferInfo,
    MUINT32                             uiTimeOut,
    MUINT64*                            puiUserData,
    MUINT32*                            puiNbLost
);

/************************************************************************************************************\

Function:       LAudioEncoder_ReleaseBuffer

Description:    Releases an encoded buffer to the encoder.

Parameters:     IN  hAudioEnc               Handle to an encoder.
                IN  hBuffer                 Handle to the LBuffer to release.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioEncoder_ReleaseBuffer
(
    LAudioEncoder_Handle                hAudioEnc,
    LBuffer_Handle                      hBuffer
);

/************************************************************************************************************\

Function:       LAudioEncoder_GetInfo

Description:    Retrieves information on the encoder and the current stream.

Parameters:     IN  hAudioEnc               Handle to an encoder.
                OUT peInfo                  Information to query (passed by type header).

Return Value:   LStatus_OK                  Function completed succesfully.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioEncoder_GetInfo
(
    LAudioEncoder_Handle            hAudioEnc,
    LAudioEncoder_InfoTypeHeader*   peInfo
);

/************************************************************************************************************\

Function:       LAudioEncoder_EndStream

Description:    Notifies the encoder that no more data will be sent through LAudioEncoder_PutData for this
                stream. If bFlushPendingData is set, all pending data sent with LAudioEncoder_PutData as
                well as all already encoded buffers waiting to be collected with
                LAudioEncoder_GetNextBuffer will be immediately discarded.

Parameters:     IN  hAudioEnc               Handle to an encoder.
                IN  bFlushPendingData       Discard all pending data.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAudioEncoder_EndStream
(
    LAudioEncoder_Handle    hAudioEnc,
    MBOOL32                 bFlushPendingData
);

#if defined (__cplusplus)
}
#endif

#endif
