/************************************************************************************************************\

Module Name:    LAncOut.h

Description:    The LAncOut module of the API provides the programming interface for the ancillery output
                hardware and its associated output devices.

References:     LAncOut.doc Revision 0.11.

    Copyright (c) 2016, Matrox Graphics Inc.
    All Rights Reserved.

\************************************************************************************************************/

#ifndef INC_LANCOUT_H
#define INC_LANCOUT_H

// -----------------------------------------------------------------------------------------------------------
//                                   I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "LAnc.h"
#include "LVideoOut.h"

typedef struct LANCOUT_OBJECT*     LAncOut_Handle;

// -----------------------------------------------------------------------------------------------------------
//                                                U S A G E
// -----------------------------------------------------------------------------------------------------------

/*  Playback ancillary buffer on an ancillary output.

    #include "Liberatus.h"
    #include "LBuffer.h"
    #include "LAncOut.h"

    LStatus GetNextPCMBuffer(LDevice_Handle hDevice, LBuffer_Handle hAncoutBuffer, MUINT32 uiBufferSize);
    LStatus ReleasePCMBuffer(LDevice_Handle hDevice, LBuffer_Handle hAncoutBuffer);


    int main()
    {
        LDevice_Handle      hDevice         = MNULL;
        LBuffer_Handle      hAncBuffer0     = MNULL;
        LBuffer_Handle      hAncBuffer1     = MNULL;
        LAncOut_Handle      hAncOut         = MNULL;
        MUINT32             uiBufferSize    = 0;
        MUINT32             uiBufferIndex   = 0;

        Liberatus_Load();

        // Get primary device handle.
        hDevice = Liberatus_GetDevice(0);

        // Get handle on first ancillary output interface.
        LAncOut_GetHandle(hDevice,
                            0,                                  // First interface using default signal type
                            LAccessMode_READWRITE,
                            &hAncOut);

        // Set buffer attributes for the ancillary playback session.
        LAncOut_SetBufferAttributes(hAncOut,
                                      uiBufferSize);


        for (uiBufferIndex = 0; uiBufferIndex < 100; uiBufferIndex++)
        {
            if (uiBufferIndex % 2)
            {
                // Get ancillary buffer driver ancillary FIFO.
                GetNextPCMBuffer(hDevice,
                                 hAncBuffer0,
                                 uiBufferSize);

                if (uiBufferIndex == 0)
                {
                    // Send the first ancillary buffer for playback.
                    LAncOut_SetBuffer(hAncOut,
                                        hAncBuffer0,
                                        MFALSE);            // Do NOT block CPU thread

                    // Start ancillary playback. Do it only once at the beginning of the playback session.
                    LAncOut_StartPlayback(hAncOut);
                }
                else
                {
                    // Send ancillary buffer for playback.
                    LAncOut_SetBuffer(hAncOut,
                                        hAncBuffer0,
                                        MTRUE);             // Block CPU thread

                    // Release previous buffer.
                    ReleasePCMBuffer(hDevice,
                                     hAncBuffer1);
                }
            }
            else
            {
                // Get ancillary buffer from driver ancillary FIFO.
                GetNextPCMBuffer(hDevice,
                                 hAncBuffer1,
                                 uiBufferSize);

                // Send ancillary buffer for playback.
                LAncOut_SetBuffer(hAncOut,
                                    hAncBuffer1,
                                    MTRUE);             // Block CPU thread

                // Release previous buffer.
                ReleasePCMBuffer(hDevice,
                                 hAncBuffer0);
            }
        }

        // End ancillary playback.
        LAncOut_StopPlayback(hAncOut);

        // Release ancillary output interface.
        LAncOut_ReleaseHandle(hAncOut);

        Liberatus_UnLoad();
    }
*/

// -----------------------------------------------------------------------------------------------------------
//                                               O V E R V I E W
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\
    The main purpose of the Ancillary Output module is to provide the functionality to output (playback) an
    ancillary output stream from the Matrox SV2 system.

    The module has services to:
        - enumerate ancillary output paths and their ancillary capabilities
        - configure the available ancillary output path in a specific mode to handle specific ancillary buffers
        - control the playback of ancillary buffers
        - retrieve the state of the interfaces
        - be notified of playback events
        - adjust dynamic controls (such as output volume level)

    Based on the hardware configuration, the module can handle analog and/or digital ancillary output stream
    playback in mono, stereo, or multi-channels. It can also handle PCM (uncompressed) and/or compressed
    ancillary output streams.

    The module can handle multiple ancillary output streams at a time.

\************************************************************************************************************/


// -----------------------------------------------------------------------------------------------------------
//                                   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Enum:           LAncOut_Event

Description:    Enumerates all possible playback events.

Comments:       None.

\************************************************************************************************************/
typedef enum
{
    LAncOut_Event_INVALID              = 0x00000000,   // Invalid value.
    LAncOut_Event_START_PENDING        = 0x00000001,   // Playback started but it's waiting on an external
                                                         //  event to start playing.
    LAncOut_Event_STARTED              = 0x00000002,   // Playback already started, ancillary data
                                                         //  are being played.
    LAncOut_Event_STOPPED              = 0x00000004,   // Playback has stopped, but it's still
                                                         //  initialized and can be restarted.
    LAncOut_Event_STATE_CHANGED        = 0x00000008,   // Playback state has changed.
    LAncOut_Event_OUT_OF_BUFFER        = 0x00000020,   // Application was not fast enough to call
                                                         // LAncout_SetBuffer in time.
    LAncOut_Event_FORCE32              = 0x7FFFFFFF    // Dummy value to force to use 32-bits.
} LAncOut_Event;

// -----------------------------------------------------------------------------------------------------------
//                         G L O B A L   F U N C T I O N   D E C L A R A T I O N S
// -----------------------------------------------------------------------------------------------------------


#if defined (__cplusplus)
extern "C" {
#endif

/************************************************************************************************************\

Function:       LAncOut_GetCount

Description:    Gets the number of independent ancillary outputs available for the specified device.

Parameters:     IN  hDev            Device handle.
                OUT puiCount        Pointer to an MUINT32 variable to be filled with the number of independent
                                    ancillary outputs available.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAncOut_GetCount(
                LDevice_Handle      hDev,
                MUINT32*            puiCount);


/************************************************************************************************************\

Function:       LAncOut_GetHandle

Description:    Gets the ancillary output handle for the specified independent ancillary output device.

Parameters:     IN  hDev            Device handle.
                IN  uiIndex         Index of the independent ancillary output for which to return the handle.
                IN  eAccessMode     Requested access mode.  Must be a value from LAccessMode.
                OUT phAncOut        Pointer to an LAncOut_Handle variable where the ancillary output device
                                    handle is written.  MNULL if unsuccessful.


Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_UNSUPPORTED         Invalid ancillary output device index.  No hardware associated.
                LStatus_OUT_OF_RESOURCES    Specified ancillary output device can't be used by the Liberatus
                                            applications because it's already being used by the system.
                LStatus_RESOURCES_BUSY      Ancillary output handle couldn't be returned because the ancillary
                                            output is busy.

Comments:       - If eAccessMode is LAccessMode_READONLY, functions that modify the ancillary output can't be
                  called.
                - If eAccessMode is LAccessMode_READWRITE or LAccessMode_READWRITE_EXCLUSIVE, functions that
                  modify the ancillary output can be called.
                - Only one handle with LAccessMode_READWRITE_EXCLUSIVE read/write access mode can exist at
                  any one time. If a handle with read/write access has already been granted to a caller,
                  any attempt to get another handle with read/write access returns a failure.

\************************************************************************************************************/
LAPI LStatus LAncOut_GetHandle(
                LDevice_Handle          hDev,
                MUINT32                 uiIndex,
                LAccessMode             eAccessMode,
                LAncOut_Handle*         phAncOut);


/************************************************************************************************************\

Function:       LAncOut_ReleaseHandle

Description:    Releases a handle to an independent ancillary output device.

Parameters:     IN  hAncOut             Handle to the ancillary output device.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAncOut_ReleaseHandle(
                LAncOut_Handle          hAncOut);


/************************************************************************************************************\

Function:       LAncOut_EnumSignalType

Description:    Enumerates all supported ancillary output signal types.

Parameters:     IN  hAncOut             Handle to the ancillary output device.
                IN  uiSignalIndex       Index of the signal type to enumerate.
                OUT peSignalType        Pointer to an LAnc_SignalType object where the enumerated signal
                                        types are written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_NO_MORE_DATA    No signal type value is available for the specified index.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAncOut_EnumSignalType(
                LAncOut_Handle          hAncOut,
                MUINT32                 uiSignalIndex,
                LAnc_SignalType*        peSignalType);


/************************************************************************************************************\

Function:       LAncOut_GetDefaultSignalType

Description:    Gets the default signal type of the ancillary output device.

Parameters:     IN  hAncOut             Handle to the ancillary output device.
                OUT peSignalType        Pointer to an LAnc_SignalType object where the enumerated signal
                                        types are written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - The default ancillary output signal type device may change after an output source detection.

\************************************************************************************************************/
LAPI LStatus LAncOut_GetDefaultSignalType(
                LAncOut_Handle          hAncOut,
                LAnc_SignalType*        peSignalType);


/************************************************************************************************************\

Function:       LAncOut_SetSignalType

Description:    Sets the ancillary output device signal type of the ancillary output device.

Parameters:     IN  hAncOut             Handle to the ancillary output device.
                IN  eSignalType         Ancillary signal type to set for the independent ancillary output.
                                        Must be a value from the LAnc_SignalType.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAncOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - Setting the signal type may reserve and program some hardware resources that are shared
                  with other ancillary output device.

\************************************************************************************************************/
LAPI LStatus LAncOut_SetSignalType(
                LAncOut_Handle      hAncOut,
                LAnc_SignalType     eSignalType);


/************************************************************************************************************\

Function:       LAncOut_GetSignalType

Description:    Gets the current ancillary output device signal type.

Parameters:     IN  hAncOut             Handle to the ancillary output device.
                OUT peSignalType        Pointer to an LAnc_SignalType object where the enumerated signal
                                        types are written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAncOut_GetSignalType(
                LAncOut_Handle          hAncOut,
                LAnc_SignalType*        peSignalType);


/************************************************************************************************************\

Function:       LAncOut_SetBufferAttributes

Description:    Sets buffer attributes for the ancillary playback.

Parameters:     IN  hAncOut             Handle to the ancillary output device.
                IN  uiBufferSize        Size, in byte, of the playback buffer.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAncOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - Buffer attributes can't be changed when playback is started.

\************************************************************************************************************/
LAPI LStatus LAncOut_SetBufferAttributes(
                LAncOut_Handle      hAncOut,
                MUINT32             uiBufferSize);


/************************************************************************************************************\

Function:       LAncOut_GetBufferAttributes

Description:    Gets the current playback ancillary buffer attributes.

Parameters:     IN  hAncOut             Handle to ancillary output device.
                OUT puiBufferSize       Pointer to a MUINT32 variable to be filled with the size, in byte,
                                        of the playback buffer.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAncOut_GetBufferAttributes(
                LAncOut_Handle      hAncOut,
                MUINT32*            puiBufferSize);

/************************************************************************************************************\

Function:       LAncOut_SetAncillaryParams

Description:    Set buffer attributes for the ancillary playback.

Parameters:     IN  hAncOut             Handle to the ancillary output device.
                IN  bEnableVanc         Enable vertical ancillary data.
                IN  bEnableHanc         Enable horizontal ancillary data.
                IN  bEnableVanc3G       Enable vertical ancillary 3G data.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAncOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - None

\************************************************************************************************************/
LAPI LStatus LAncOut_SetAncillaryParams(
                LAncOut_Handle      hAncOut,
                MBOOL32             bEnableVanc,
                MBOOL32             bEnableHanc,
                MBOOL32             bEnableVanc3G);

/************************************************************************************************************\

Function:       LAncOut_StartPlayback

Description:    Starts the ancillary playback.

Parameters:     IN  hAncOut             Handle to the ancillary output device.
                IN  roVidParam          Reference to a LVideo_VidParam object containing the video parameters.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAncOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - A playback ancillary buffer should have been previously submitted via the LAncOut_SetBuffer
                  service, otherwise this call will fail.

\************************************************************************************************************/
LAPI LStatus LAncOut_StartPlayback(
                LAncOut_Handle      hAncOut,
        const   LVideo_VidParam&    roVidParam);


/************************************************************************************************************\

Function:       LAncOut_StartPlaybackOnVideo

Description:    Starts the ancillary playback session synchronized with a video playback.

Parameters:     IN  hAncOut             Handle to the ancillary output device.
                IN  hVideoOut           Handle to the video output device.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAncOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - Service used for the ancillary/video synchronization. It enables the synchronized playback of
                  an ancillary and a video output to start.
                - A playback ancillary buffer should have been previously submitted via the LAncOut_SetBuffer
                  service, otherwise this call won't have any effect.


\************************************************************************************************************/
LAPI LStatus LAncOut_StartPlaybackOnVideo(
                LAncOut_Handle      hAncOut,
                LVideoOut_Handle    hVideoOut);

/************************************************************************************************************\

Function:       LAncOut_StopPlayback

Description:    Stops the ancillary playback.

Parameters:     IN  hAncOut             Handle to the ancillary output device.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAncOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.

Comments:       - The current playback ancillary buffer is discarded when playback is stopped. No further data is
                  played.

\************************************************************************************************************/
LAPI LStatus LAncOut_StopPlayback(
                LAncOut_Handle      hAncOut);

/************************************************************************************************************\

Function:       LAncOut_ScheduleStartPlayback

Description:    Schedules the start of the ancillary playback by adding the required commands to the passed in the
                device thread object.

Parameters:     IN  hAncOut             Handle to ancillary output device.
                IN  hDeviceThread       Device thread handle to use to start ancillary playback.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAncOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_TIMEOUT         Function exited after a timeout.

Comments:       - A playback ancillary buffer should have been previously submitted via the
                  LAncOut_ScheduleSetBuffer service, otherwise this call won't have any effect.
                - LStatus_TIMEOUT is returned when the wait command can't be added to the device thread
                  because it's full and the timeout period expired before space became available.


\************************************************************************************************************/
LAPI LStatus LAncOut_ScheduleStartPlayback(
                LAncOut_Handle          hAncOut,
                LDeviceThread_Handle    hDeviceThread);


/************************************************************************************************************\

Function:       LAncOut_ScheduleStartPlaybackOnVideo

Description:    Schedules the start of the synchronized ancillary and video playback by adding the required
                commands to the passed in device thread object.

Parameters:     IN  hAncOut         Handle to ancillary output device.
                IN  hVideoOut       Handle to video output device.
                IN  hDeviceThread   Device thread handle to use to start the ancillary playback.
                IN  uiTimeout_ms    Device thread full queue wait timeout.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAncOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_TIMEOUT         Function exited after a timeout.

Comments:       - A playback ancillary buffer should have been previously submitted via the
                  LAncOut_ScheduleSetBuffer service, otherwise this call won't have any effect.

\************************************************************************************************************/
LAPI LStatus LAncOut_ScheduleStartPlaybackOnVideo(
                LAncOut_Handle          hAncOut,
                LVideoOut_Handle        hVideoOut,
                LDeviceThread_Handle    hDeviceThread,
                MUINT32                 uiTimeout_ms);

/************************************************************************************************************\

Function:       LAncOut_SetBuffer

Description:    Sets the next available ancillary buffer for playback.

Parameters:     IN  hAncOut             Handle to ancillary output device.
                IN  hBuffer             Handle to the next available buffer for playback.
                IN  uiPayloadSize       The size of the effective data contained in hBuffer. The size can't
                                        be larger than the buffer size configured through
                                        LAncOut_SetBufferAttributes.
                IN  bIsTopField         MTRUE if the video buffer to display is a top field.
                                        MFALSE if the video buffer to display is a bottom field.
                IN  bBlocking           MTRUE to force the service to not return until the newly set buffer
                                        is actually used (also meaning that the previous buffer is no longer
                                        used), MFALSE to return immediately after the new buffer is set even
                                        if it's not used yet.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAncOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_CANCELED        Function exited. No playback in progress.

Comments:       - If the playback session isn't running, this service won't block even if the bBlocking flag
                  has been set.
                - When bBlocking is MFALSE, LStatus_FAIL will be returned if LAncOut_SetBuffer is called
                  twice during the time that one buffer is being used.

\************************************************************************************************************/
LAPI LStatus LAncOut_SetBuffer(
                LAncOut_Handle          hAncOut,
                LBuffer_Handle          hBuffer,
                MUINT32                 uiPayloadSize,
                MBOOL32                 bIsTopField,
                MBOOL32                 bBlocking);


/************************************************************************************************************\

Function:       LAncOut_ScheduleSetBuffer

Description:    Schedules the next available ancillary buffer for playback by adding the required commands to the
                passed in device thread object.

Parameters:     IN  hAncOut             Handle to ancillary output device.
                IN  hBuffer             Handle to the next available buffer for playback.
                IN  hDeviceThread       Device thread handle to use to schedule the buffer.
                IN  uiTimeout_ms        Device thread full queue wait timeout.
                IN  bIsTopField         MTRUE if the video buffer to display is a top field.
                                        MFALSE if the video buffer to display is a bottom field.
                IN  bBlocking           MTRUE to force the device thread block on the set buffer command until
                                        the newly set buffer is actually used (also meaning that the previous
                                        buffer is no longer used), MFALSE to proceed to the next device thread
                                        command immediately after the new buffer is set even if it is not used
                                        yet.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_ACCESS_DENIED   The hAncOut handle is READONLY.  Read/write access
                                        (LAccessMode_READWRITE) is needed to call this function.
                LStatus_TIMEOUT         Function exited after a timeout.

Comments:       - If the playback session isn't running, this service won't block even if the bBlocking flag
                  has been set.

\************************************************************************************************************/
LAPI LStatus LAncOut_ScheduleSetBuffer(
                LAncOut_Handle          hAncOut,
                LBuffer_Handle          hBuffer,
                LDeviceThread_Handle    hDeviceThread,
                MUINT32                 uiTimeout_ms,
                MBOOL32                 bIsTopField,
                MBOOL32                 bBlocking);


/************************************************************************************************************\

Function:       LAncOut_GetPlaybackState

Description:    Return the ancillary playback state.

Parameters:     IN  hAncOut                 Handle to the ancillary output device.
                OUT peState                 Pointer to an LAncOut_State variable to be filled with
                                            the current playback state.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAncOut_GetPlaybackState(
                LAncOut_Handle      hAncOut,
                LAnc_State*         peState);


/************************************************************************************************************\

Function:       LAncOut_WaitForBufferChange

Description:    Blocks a CPU thread until the next buffer playback starts.

Parameters:     IN  hAncOut             Handle to the ancillary output device.
                IN  uiTimeout_ms        Time out to wait in milliseconds.  0 means no wait and
                                        LINFINITE_TIMEOUT means forever.
                OUTO puiCounter         Pointer to an MUINT64 variable to be filled with the number of
                                        the just started buffer. Can be MNULL if the counter isn't
                                        requested.
                OUTO puiTickRefCounter  Pointer to an MUINT64 variable to be filled with the tick reference
                                        counter value, in ticks since the system was started, when the
                                        counter was updated.  Can be MNULL if this counter value is
                                        not requested.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.
                LStatus_TIMEOUT         Function exited after a timeout.
                LStatus_CANCELED        Function exited. No playback in progress.
                LStatus_UNSUPPORTED     Function aborted because waiting for start is unsupported
                                        by the current hardware configuration.

Comments:       - For more information on counters, tick counters, and top ancillary data, see
                  LAncOut_GetLastBufferCounter.

\************************************************************************************************************/
LAPI LStatus LAncOut_WaitForBufferChange(
                LAncOut_Handle      hAncOut,
                MUINT32             uiTimeout_ms,
                MUINT64*            puiCounter,
                MUINT64*            puiTickRefCounter);


/************************************************************************************************************\

Function:       LAncOut_GetLastBufferCounter

Description:    Gets the last played buffer counter value.

Parameters:     IN  hAncOut             Handle to the ancillary output device.
                OUTO puiCounter         Pointer to an MUINT64 variable to be filled with the last counter
                                        number.  Can be MNULL if the played isn't requested.
                OUTO puiTickRefCounter  Pointer to an MUINT64 variable to be filled with the tick reference
                                        counter value, in ticks since the system was started, when the
                                        counter was last updated.  Can be MNULL if this counter value
                                        isn't requested.
                OUT pbIsTopField        Pointer to an MBOOL32 variable.  MTRUE if the top field is
                                        displayed.  MFALSE if the bottom field is displayed.  MNULL if
                                        displayed field isn't requested.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LAncOut_GetLastBufferCounter(
                LAncOut_Handle      hAncOut,
                MUINT64*            puiCounter,
                MUINT64*            puiTickRefCounter,
                MBOOL32*            pbIsTopField);


/************************************************************************************************************\

Function:       LAncOut_GetCounterLUID

Description:    Gets played counters unique identifiers (LUID).

Parameters:     IN  hAncOut             Handle to the ancillary output device.
                OUT puiCounterLUID      Pointer to an MUINT64 variable where the ancillary data counter unique
                                        identifier is written.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed due to invalid parameters.
                LStatus_FAIL            Function failed.

Comments:       - Use to synchronize the device thread with the completion of a counter.

\************************************************************************************************************/
LAPI LStatus LAncOut_GetCounterLUID(
                LAncOut_Handle      hAncOut,
                MUINT64*            puiCounterLUID);


/************************************************************************************************************\

Function:       LAncOut_WaitForEvent

Description:    Blocks a CPU thread until any of the specified playback event(s) are signaled.

Parameters:     IN  hAncOut                 Handle to the ancillary output device.
                IN  uiTimeout_ms            Time out to wait in milliseconds.  0 means no wait and
                                            LINFINITE_TIMEOUT means forever.
                OUT peSignaledEvent         Pointer to a LAncOut_Event variable to be filled with one
                                            signaled playback event.

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INVALID_PARAM       Function failed due to invalid parameters.
                LStatus_FAIL                Function failed.
                LStatus_UNSUPPORTED         Waiting for playback events isn't supported.
                LStatus_TIMEOUT             Function exited after a timeout.
                LStatus_CANCELED            Function exited. No playback in progress.

Comments:       - A playback event can be used to notify applications of important events during a playback
                   session.
                - Application can start waiting for event before the playback is started with
                   LAncOut_StartPlayback.

\************************************************************************************************************/
LAPI LStatus LAncOut_WaitForEvent(
                LAncOut_Handle      hAncOut,
                MUINT32             uiTimeout_ms,
                LAncOut_Event*      peSignaledEvent);


#if defined (__cplusplus)
}
#endif

#endif  // #ifndef INC_LANCOUT_H
