#!/usr/bin/python3
# This script contains all that is required to configure and control strongswan.
import sys
import json
import os
import zipfile
import vici
import socket
import struct
from subprocess import call
from subprocess import STDOUT 
from collections import OrderedDict
from shutil import rmtree
from shutil import copytree
from shutil import copy
from configargparse import ArgumentParser
from datetime import datetime

ErrCodes = OrderedDict() 
ErrCodes['Success']                = ( 0,  'Success' )
ErrCodes['Generic']                = ( 1,  'Generic error' ) 
ErrCodes['Timeout']                = ( 2,  'Connection timeout' ) 
ErrCodes['LocalAuthFailed']        = ( 3,  'Local authentication failed' ) 
ErrCodes['PeerAuthFailed']         = ( 4,  'Peer authentication failed' ) 
ErrCodes['ParseErrorHeader']       = ( 5,  'Received IKE message with invalid header' ) 
ErrCodes['ParseErrorBody']         = ( 6,  'Received IKE message with invalid body' ) 
ErrCodes['RetransmitSendTimeout']  = ( 7,  'Sending retransmits timed out' ) 
ErrCodes['HalfOpenTimeout']        = ( 8,  'Received half-open timeout before IKE_SA established' ) 
ErrCodes['ProposalMismatchIke']    = ( 9,  'IKE proposals do not match' ) 
ErrCodes['ProposalMismatchChild']  = ( 10, 'CHILD proposals do not match' ) 
ErrCodes['TsMismatch']             = ( 11, 'Traffic selectors do not match' ) 
ErrCodes['InstallChildSaFailed']   = ( 12, 'Installation of IPsec SAs failed' ) 
ErrCodes['InstallChildPolFailed']  = ( 13, 'Installation of IPsec Policy failed' ) 
ErrCodes['VipFailure']             = ( 14, 'Allocating virtual IP failed' ) 
ErrCodes['AuthorizationFailed']    = ( 15, 'An authorize() hook failed' ) 
ErrCodes['CertExpired']            = ( 16, 'Certificate rejected; it has expired' ) 
ErrCodes['CertRevoked']            = ( 17, 'Certificate rejected; it has been revoked' ) 
ErrCodes['NoIssuerCert']           = ( 18, 'Certificate rejected; no trusted issuer found' ) 
ErrCodes['RadiusNotResponding']    = ( 19, 'A RADIUS server did not respond' ) 
ErrCodes['InvalidSecret']          = ( 20, 'Invalid secret' ) 
ErrCodes['InvalidJson']            = ( 21, 'Invalid JSON' )

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Write log message to console.
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def Log(line, level=0):
    if (level == 0) or (not args.quiet):
        print(line.rstrip())

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# stdout/file Logging helper.
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
class Logger:
#
    logFile = None

    def __init__(self, logFilePath):
        if logFilePath is not None:
            self.logFile = open(logFilePath, "w")
            self.logFile.write("%s\n" % datetime.now())

    def __del__(self):
        if self.logFile is not None:
            self.logFile.close()
        
    def Add(self, line, level=0, toStdOut=True):
        if toStdOut:
            Log(line, level)
        if self.logFile is not None:
            self.logFile.write(line.rstrip() + "\n")
#

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Exit with an error message.
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def Fail(msg, errcode = ErrCodes['Generic'], logger=None):
    errMsg = "%s (Error code %d - '%s')\n" % (msg.rstrip(), errcode[0], errcode[1])
    if logger is not None:
        logger.Add(errMsg, toStdOut=False)
    sys.stderr.write(errMsg)
    sys.exit(errcode[0])

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Parse arguments.
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
parser = ArgumentParser()

parser.add_argument(
            "--json", 
            dest="jsonFilePath", 
            help="Strongswan parameters JSON file.", 
            metavar="IN_FILE", 
            required=False)

parser.add_argument(
            "--swanctl", 
            dest="swanctlFilePath", 
            help="Destination path for the generated swanctl.conf.", 
            metavar="OUT_PATH", 
            required=False)

parser.add_argument(
            "--quiet", 
            action="store_true", 
            help="Disable verbose output.", 
            required=False)

parser.add_argument(
            "--config-package", 
            dest="configPkg", 
            help="Install strongswan config from a zip package.", 
            metavar="PKG_FILE", 
            required=False)

parser.add_argument(
            "--connect", 
            action="store_true", 
            help="Initiate the VPN connection.", 
            required=False)

parser.add_argument(
            "--conn-timeout",
            dest="connTimeout",
            help="Connection timeout in milliseconds.",
            metavar="N",
            type=int,
            default=10000,
            required=False)

parser.add_argument(
            "--disconnect", 
            action="store_true", 
            help="Terminate the VPN connection.", 
            required=False)

secretTypes = ['eap', 'ntlm', 'xauth', 'ike', 'priv-key', 'pkcs12']
parser.add_argument(
            "--load-secret", 
            dest="secretType", 
            choices=secretTypes, 
            help="Load secret of the given type (one of %s), must be used with \"--secret-id\" and \"--secret\"." % secretTypes, 
            metavar="TYPE",
            required=False)

parser.add_argument(
            "--persist",
            dest="persistSecret",
            action="store_true",
            help="Persist the loaded secret in swanctl configs.",
            required=False)

parser.add_argument(
            "--secret-id", 
            dest="secretId", 
            help="ID of the secret to load.", 
            metavar="ID", 
            required=False)

parser.add_argument(
            "--secret", 
            dest="secret", 
            help="The actual secret to load.", 
            metavar="SECRET", 
            required=False)

parser.add_argument(
            "--remove-secrets",
            dest="removeSecrets",
            action="store_true",
            help="Remove all loaded secrets (volatile and persisted).",
            required=False)

parser.add_argument(
            "--remove-volatile-secrets",
            dest="removeVolSecrets",
            action="store_true",
            help="Remove loaded volatile secrets.",
            required=False)

parser.add_argument(
            "--get-src-addr", 
            dest="getSrcAddr", 
            action="store_true",
            help="Get the IP address to use as source address to access the VPN.",
            required=False)

parser.add_argument(
            "--get-logs", 
            dest="logDest",
            metavar="DEST_FOLDER",
            help="Collect strongswan related logs.",
            required=False)

parser.add_argument(
            "--show-errcodes",
            dest="showErrcodes",
            action="store_true",
            help="Print error code specification.",
            required=False)

parser.add_argument(
            "--show-errcode-text",
            dest="showErrcodeText",
            metavar="ERRCODE",
            type=int,
            help="Print the reason for a specific error code.",
            required=False)

parser.add_argument(
            "--wait-disconnect",
            dest="waitDisconnect",
            action="store_true",
            help="Wait for disconnection and return 0. Returns 0 right away if already disconnected.",
            required=False)

args = parser.parse_args()

# List all error codes
if args.showErrcodes:
    for err in ErrCodes:
        print("  %d - %s" % (ErrCodes[err][0], ErrCodes[err][1]))
    sys.exit(0)

# Show specific error code
if args.showErrcodeText is not None:
    text = "Unknown error %d" % args.showErrcodeText
    for err in ErrCodes:
        if ErrCodes[err][0] == args.showErrcodeText:
            text = ErrCodes[err][1]
    print(text)
    sys.exit(0)

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Validate arguments.
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
if bool(args.secretType) and (args.secret is None):
    Fail("Error: must specify '--secret' with '--load-secret'.")
if bool(args.secretType) and (args.secretId is None) and (args.secretType != "priv-key") and (args.secretType != "pkcs12"):
    Fail("Error: must specify '--secret-id' with '--load-secret' (except for 'priv-key').")
if (bool(args.secretId) or bool(args.secret) or bool(args.persistSecret)) and (args.secretType is None):
    Fail("Error: must specify secret type with '--load-secret'.")
if bool(args.configPkg) and bool(args.jsonFilePath):
    Fail("Error: use only one of '--json' and '--config-package'.")
if bool(args.getSrcAddr) and (len(sys.argv) > 2):
    Fail("Error: can't use '--get-src-addr' with any other parameter.")
if bool(args.waitDisconnect) and (len(sys.argv) > 2):
    Fail("Error: can't use '--wait-disconnect' with any other parameter.")

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Remove directory recursively.
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def RmDir(path):
#
    # Remove any content in path
    try:
        rmtree(path)
    except: pass

    # Remove path if exists
    try: 
        os.rmdir(path)
    except: pass
#

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# mkdir helper.
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def MkDir(path):
    try: 
        os.mkdir(path)
    except Exception as e: 
        Fail("Error, can't create directory \'%s\': \"%s\"\n" % (path, e))

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Call process.
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def Call(cmd, printOutput=False):
    if printOutput:
        return call(cmd, shell=True)
    else:
        fnull = open(os.devnull, 'w')
        return call(cmd, shell=True, stdout=fnull, stderr=STDOUT)

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Returns whether the strongswan service is 
# loaded or not.
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def IsServiceLoaded():
    if (Call("systemctl status strongswan --no-pager") == 0):
        return True
    else:
        return False

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Write line helper.
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def WriteLine(line, outFile):
    outFile.write(line.rstrip() + "\n")
    # This is removed to avoid that a user's secret be printed in the logs if the stdout of
    # this script is ever logged somewhere.
    # Log(line, level=1)

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Write simple field.
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def FieldHandler(name, value, outFile, indent, quotes=False):
    if quotes:
        WriteLine("%s%s = \"%s\"" % (indent, name, value), outFile)
    else:
        WriteLine("%s%s = %s" % (indent, name, value), outFile)        

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Write 'local' connection section
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def ConnLocalHandler(name, content, outFile, indent):
#
    fields = [
        "round",
        "certs",
        "pubkeys",
        "auth",
        "id",
        "eap_id",
        "aaa_id",
        "xauth_id",
    ]

    WriteLine("%s%s {" % (indent, name), outFile)

    for field in content:
    #
        if field in fields:
            FieldHandler(field, content[field], outFile, indent + "    ")
        else:
            Log("Unknown connection.local field \'%s\'" % field)
    #

    WriteLine("%s}" % indent, outFile)
#

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Write 'remote' connection section
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def ConnRemoteHandler(name, content, outFile, indent):
#
    fields = [
        "round",
        "id",
        "eap_id",
        "cert_policy",
        "certs",
        "cacerts",
        "ca_id",
        "pubkeys",
        "revocation",
        "auth",
    ]

    WriteLine("%s%s {" % (indent, name), outFile)

    for field in content:
    #
        if field in fields:
            FieldHandler(field, content[field], outFile, indent + "    ")
        else:
            Log("Unknown connection.remote field \'%s\'" % field)
    #

    WriteLine("%s}" % indent, outFile)  
#

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Write 'children' connection section
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def ConnChildrenHandler(name, content, outFile, indent):
#
    fields = [
        "ah_proposals",
        "esp_proposals",
        "sha256_96",
        "local_ts",
        "remote_ts",
        "rekey_time",
        "life_time",
        "rand_time",
        "rekey_bytes",
        "life_bytes",
        "rand_bytes",
        "rekey_packets",
        "life_packets",
        "rand_packets",
        "replay_window",
        "copy_df",
        "copy_ecn",
        "copy_dscp",
    ]

    # Special case: if the 'children' section only contains a nested dictionary, assume that it 
    # must be the "child-name" sub-section. Skip this level and re-call with the contents
    # of this nested dictionary. This way we support both the following syntaxes:
    #
    # 1)  "conn" : {
    #        ...
    #        "children" : {
    #           "field1" : "value1",
    #           ...
    #        } 
    #     }
    #
    # 2) "conn" : {
    #       ...
    #       "children" : {
    #           "child-name" : {
    #              "field1" : "value1",
    #              ...
    #           }
    #       }
    #    }
    if len(content) == 1:        
        for field in content:
            if type(content[field]) is OrderedDict:
                ConnChildrenHandler(name, content[field], outFile, indent)
                return

    WriteLine("%schildren {" % indent, outFile)

    indent2 = indent + "    "

    # Support only one child: fixed name 'child'
    WriteLine("%s%s {" % (indent2, "child"), outFile)

    for field in content:
    #
        if field in fields:
            FieldHandler(field, content[field], outFile, indent2 + "    ")
        else:
            Log("Unknown connection.children field \'%s\'" % field)
    #

    # Add updown field.
    FieldHandler('updown', '/usr/libexec/ipsec/_updown iptables', outFile, indent2 + "    ")

    WriteLine("%s}" % indent2, outFile)

    WriteLine("%s}" % indent, outFile)
#

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Write single connection
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def WriteConnection(name, conn, outFile, indent):
#
    simpleFields = [
        "version",
        "local_addrs",
        "remote_addrs",
        "local_port",
        "remote_port",
        "proposals",
        "vips",
        "pull",
        "dscp",
        "encap",
        "mobike"
        "dpd_delay",
        "dpd_timeout",
        "fragmentation",
        "childless",
        "send_certreq",
        "send_cert",
        "ppk_id",
        "ppk_required",
        "keyingtries",
        "unique",
        "reauth_time",
        "rekey_time",
        "over_time",
        "rand_time" 
    ]

    WriteLine("%s%s {" % (indent, name), outFile)

    for field in conn:
    #
        if field in simpleFields:
            FieldHandler(field, conn[field], outFile, indent + "    ")
        elif field.startswith("local"):
            ConnLocalHandler(field, conn[field], outFile, indent + "    ")
        elif field.startswith("remote"):
            ConnRemoteHandler(field, conn[field], outFile, indent + "    ")
        elif (field == "children"):
            ConnChildrenHandler(field, conn[field], outFile, indent + "    ")
        else:
            Log("Unknown connection field \'%s\'." % field)
    #

    WriteLine("%s}" % indent, outFile)
#

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Write 'connections' section
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def WriteConnections(conns, outFile, logger):
#
    WriteLine("connections {", outFile)    

    # We only support one connection in swanctl.conf for now.
    if len(conns) > 1:
        Fail("Error: config must only contain one connection.", logger=logger)
    
    for connName in conns:
        WriteConnection(connName, conns[connName], outFile, "    ")

    WriteLine("}", outFile)
#

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Write secret section with 'id' parameter
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def SecretIdParamHandler(name, secret, outFile, indent):
#
    WriteLine("%s%s {" % (indent, name), outFile)

    for param in secret:
        if (param == "id") or (param == "secret"):
            FieldHandler(param, secret[param], outFile, indent + "    ", quotes=True)
        else:
            Log("Unknown secret.%s field \'%s\'" % (name, param))

    WriteLine("%s}" % indent, outFile)
#

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Write secret section with 'file' parameter
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def SecretFileParamHandler(name, secret, outFile, indent):
#
    WriteLine("%s%s {" % (indent, name), outFile)

    for param in secret:
        if (param == "file") or (param == "secret"):
            FieldHandler(param, secret[param], outFile, indent + "    ", quotes=True)
        else:
            Log("Unknown secret.%s field \'%s\'" % (name, param))

    WriteLine("%s}" % indent, outFile)
#

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Write 'secrets' section
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def WriteSecrets(secrets, outFile):
#
    fieldHandlers = {
        "eap"     : SecretIdParamHandler,
        "xauth"   : SecretIdParamHandler,
        "ntlm"    : SecretIdParamHandler,
        "ike"     : SecretIdParamHandler,
        "ppk"     : SecretIdParamHandler,
        "private" : SecretFileParamHandler,
        #"rsa"     : SecretFileParamHandler,
        #"ecdsa"   : SecretFileParamHandler,
        #"pkcs8"   : SecretFileParamHandler,
        "pkcs12"  : SecretFileParamHandler,
    }

    WriteLine("secrets {", outFile)

    for field in secrets:
        for secretType in fieldHandlers:
            if field.startswith(secretType):
                fieldHandlers[secretType](field, secrets[field], outFile, "    ")

    WriteLine("}", outFile)
#

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Generate swanctl.conf from JSON config file.
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def GenSwanctlConf(jsonPath, swanctlPath, loggerInst=None):
#
    logger = None
    if loggerInst is not None:
        logger = loggerInst
    else:
        logger = Logger(None)

    jsonFile = open(jsonPath, "r")
    jsonText = jsonFile.read() 
    swanctlFile = open(swanctlPath, "w")

    WriteLine("# swanctl config file auto-generated by %s." % sys.argv[0], swanctlFile)
    WriteLine("", swanctlFile)
    
    # Load json in dictionary
    try:
        jsonDict = json.loads(jsonText, object_pairs_hook=OrderedDict)
    except ValueError as e:
        Fail("Failed to parse %s: \"%s\"." % (jsonPath, e), ErrCodes['InvalidJson'], logger=logger)

    # Handle sections.
    for section in jsonDict.keys():
        if (section == 'connections'):
            WriteConnections(jsonDict['connections'], swanctlFile, logger)
        elif (section == 'secrets'):
            WriteSecrets(jsonDict['secrets'], swanctlFile)
        else:
            Log("Unknown section \'%s\'" % section)

    WriteLine("", swanctlFile)
    WriteLine("include conf.d/*.conf", swanctlFile)
    WriteLine("", swanctlFile)

    jsonFile.close()
    swanctlFile.close()
#

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Install configuration package from a .zip.
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def InstallConfigPackage(pkgpath):
#
    logger = Logger("/var/log/strongswan-install.log")

    try:
        fstat = os.stat(pkgpath)
    except OSError as e:
        Fail("Error: can't install config package (\"%s\")" % e, logger=logger);

    # Arbitrary maximum size of 5MB.
    maxSize = (5 * 1024 * 1024)
    if fstat.st_size > maxSize:
        Fail("File \'%s\' is too big (maximum size: %d bytes)." % (pkgpath, maxSize), logger=logger)

    try:
        zipFile = zipfile.ZipFile(pkgpath, mode='r')
    except Exception as e:
        Fail("Error, can't open zip file \'%s\': \"%s\"" % (pkgpath, e), logger=logger)

    extractPath = "/tmp/strongswan-config"

    # Remove extractPath if it already exists
    RmDir(extractPath)

    # Re-create extractPath folder
    MkDir(extractPath)
    
    try:
        zipFile.extractall(extractPath)
    except Exception as e:
        Fail("Error, can't extract package: \"%s\"" % e, logger=logger)

    configPath = extractPath

    # Detect if the top level of the zip only contains a folder (typically the base name of the zip file).
    # If so, search this folder for the config instead. 
    for root, dirs, files in os.walk(extractPath):
        if (len(files) == 0) and (len(dirs) == 1):
            newPath = os.path.join(root, dirs[0])
            logger.Add("Top-level directory detected: %s" % newPath)
            configPath = newPath
            break

    # Fail if swanctl.json is not found in the config zip.
    swanctlJson = configPath + "/swanctl.json"
    if not os.path.isfile(swanctlJson):
        Fail("Error: can't find swanctl.json in configuration zip.", logger=logger)

    # If it is running, stop strongswan service before installing the config. 
    # This is done so that the service is reloaded with the new config upon
    # subsequent connection.
    if IsServiceLoaded():
        logger.Add("Stopping strongswan service")
        if Call("systemctl stop strongswan", True) != 0:
            logger.Add("Error: can't stop strongswan service")

    # /etc/swanctl and /etc/ipsec.d should link to their respective folders in /rw/shared/etc/. 
    # Delete old configs and replace them with the default ones before overriding with the user's config.
    RmDir("/rw/shared/etc/swanctl")
    RmDir("/rw/shared/etc/ipsec.d")
    copytree("/etc/swanctl.default", "/rw/shared/etc/swanctl")
    MkDir("/rw/shared/etc/ipsec.d")

    swanctlPath = "/etc/swanctl"
    swanctlFile = swanctlPath + "/swanctl.conf"
    logger.Add("Generating %s/swanctl.conf from \'%s\'..." % (swanctlFile, swanctlJson))
    GenSwanctlConf(swanctlJson, swanctlFile, loggerInst=logger)
    copy(swanctlJson, swanctlPath)
    
    # Folders to be copied from the root of the zip to /etc/swanctl
    copyFolders = ['x509', 'x509ca', 'x509aa', 'x509ac', 'x509crl', 'x509ocsp', 'private', 'pubkey', 'pkcs12']

    for folder in copyFolders:
        srcFolder = configPath + "/" + folder
        if os.path.isdir(srcFolder):
            dstFolder = swanctlPath + "/" + folder
            logger.Add("Copying \'%s\' to \'%s\'..." % (srcFolder, dstFolder))
            RmDir(dstFolder)
            copytree(srcFolder, dstFolder)

    if os.path.isfile(configPath + "/triplets.dat"):
        logger.Add("Copying \'%s\' to \'/etc/ipsec.d/\'" % (configPath + "/triplets.dat"))
        copy(configPath + "/triplets.dat", "/etc/ipsec.d/")

    # Remove temp extraction.
    logger.Add("Removing \'%s\'..." % extractPath)
    RmDir(extractPath)    
#

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Install pcrypt
# Enable pcrypt for all possible AEAD algos, except AES-CTR / SHA*, which 
# does not seem to work with pcrypt. 
# Note: "modprobe tcrypt..." commands are expected to fail with non-zero 
# code, so we don't check return code.
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def InstallPcrypt():
#
    # First unload pcrypt to make sure to remove existing configs.
    Call('modprobe -r pcrypt')

    # Bypass if /rw/config/pcrypt.conf contains line "nopcrypt=1"
    try:
        cfg = open("/rw/config/pcrypt.conf", "r")
        lines = cfg.readlines()
        for line in lines:
            if line.rstrip() == "nopcrypt=1":
                Log("pcrypt disabled.")
                return
    except: pass

    Log("Configuring pcrypt...")

    # AES-GCM
    Call('modprobe tcrypt alg="pcrypt(rfc4106(gcm(aes)))" type=3')
    # AES-CCM
    Call('modprobe tcrypt alg="pcrypt(rfc4309(ccm(aes)))" type=3')
    # AES-GMAC
    Call('modprobe tcrypt alg="pcrypt(rfc4543(gcm_base(ctr-aes-aesni,ghash-generic)))" type=3')
    # AES-CBC / AES-XCBC
    Call('modprobe tcrypt alg="pcrypt(authenc(xcbc(aes),cbc(aes)))" type=3')
    # AES-CTR / AES-XCBC
    Call('modprobe tcrypt alg="pcrypt(authenc(xcbc(aes),rfc3686(ctr(aes))))" type=3')
    # AES-CBC / AES-CMAC
    Call('modprobe tcrypt alg="pcrypt(authenc(cmac(aes),cbc(aes)))" type=3')
    # AES-CTR / AES-CMAC
    Call('modprobe tcrypt alg="pcrypt(authenc(cmac(aes),rfc3686(ctr(aes))))" type=3')
    # AES-CBC / SHA
    Call('modprobe tcrypt alg="pcrypt(authenc(hmac(sha1),cbc(aes)))" type=3')
    Call('modprobe tcrypt alg="pcrypt(authenc(hmac(sha256),cbc(aes)))" type=3')
    Call('modprobe tcrypt alg="pcrypt(authenc(hmac(sha384),cbc(aes)))" type=3')
    Call('modprobe tcrypt alg="pcrypt(authenc(hmac(sha512),cbc(aes)))" type=3')

    # Limit pcrypt to using CPUs 1,2,3 and leave CPU 0 free to avoid stalling the whole system.
    Call('echo e > /sys/kernel/pcrypt/pdecrypt/serial_cpumask')
    Call('echo e > /sys/kernel/pcrypt/pdecrypt/parallel_cpumask')
    Call('echo e > /sys/kernel/pcrypt/pencrypt/serial_cpumask')
    Call('echo e > /sys/kernel/pcrypt/pencrypt/parallel_cpumask')
#

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Create VICI session. 
# If loadService is True, strongswan service will be loaded. If False, 
# session will only be created if the strongswan service is already running.
# In any case, fail if strongswan has not been configured yet.
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def CreateSession(loadService):
#
    # Fail if any function necessitating session is called before the configuration has been installed.
    if not os.path.isfile("/etc/swanctl/swanctl.conf"):
        Fail("Error: config has not been installed.")

    # Remove any dangling symlinks from /etc/swanctl/conf.d/volatile-*.conf to /var/run/strongswan/*
    if os.path.isdir("/etc/swanctl/conf.d"):
        for f in os.listdir("/etc/swanctl/conf.d/"):
            fullpath = os.path.join("/etc/swanctl/conf.d/", f)
            if os.path.islink(fullpath):
                targetpath = os.path.realpath(fullpath)
                if not os.path.isfile(targetpath):
                    Log("Removing dangling symlink '%s'" % fullpath)
                    os.remove(fullpath)

    # Start strongswan service if stopped.
    if loadService:
        if not IsServiceLoaded():
            # Install and configure parallel-crypto kernel module
            InstallPcrypt()

            # Start strongswan service
            if (Call("systemctl start strongswan", True) == 0):
                Log("Started strongswan service.")
            else:
                Fail("Error: can't start strongswan service")

    sess = None

    try:
        sess = vici.Session()
    except Exception as e:
        if loadService:
            Fail("Error: can't open VICI session ('%s')" % e)
        else:
            pass
    
    return sess
#

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Check connection state.
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def IsConnected():
    for conn in sess.list_sas():
        return True
    return False

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Initiate VPN connection.
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def Connect():
#
    if IsConnected():
        Log("Already connected.")
        return

    connName = None

    try:
        connName = sess.get_conns()['conns'][0].decode()
    except: pass

    logger  = Logger("/var/log/strongswan-connect.log")
    failure = None

    if connName:
        logger.Add("Connecting to \'%s\'..." % connName)

        # Connect to error-notify socket.
        errNotify = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
        errNotify.connect("/var/run/charon.enfy")
        errNotify.setblocking(0)

        errcode = ErrCodes['Generic']
    
        # Map error-notify error codes to our owns
        errCodesMap = {
            1  : ErrCodes['RadiusNotResponding'],    # ERROR_NOTIFY_RADIUS_NOT_RESPONDING
            2  : ErrCodes['LocalAuthFailed'],        # ERROR_NOTIFY_LOCAL_AUTH_FAILED
            3  : ErrCodes['PeerAuthFailed'],         # ERROR_NOTIFY_PEER_AUTH_FAILED
            4  : ErrCodes['ParseErrorHeader'],       # ERROR_NOTIFY_PARSE_ERROR_HEADER
            5  : ErrCodes['ParseErrorBody'],         # ERROR_NOTIFY_PARSE_ERROR_BODY
            6  : ErrCodes['RetransmitSendTimeout'],  # ERROR_NOTIFY_RETRANSMIT_SEND_TIMEOUT
            7  : ErrCodes['HalfOpenTimeout'],        # ERROR_NOTIFY_HALF_OPEN_TIMEOUT
            8  : ErrCodes['ProposalMismatchIke'],    # ERROR_NOTIFY_PROPOSAL_MISMATCH_IKE
            9  : ErrCodes['ProposalMismatchChild'],  # ERROR_NOTIFY_PROPOSAL_MISMATCH_CHILD
            10 : ErrCodes['TsMismatch'],             # ERROR_NOTIFY_TS_MISMATCH
            11 : ErrCodes['InstallChildSaFailed'],   # ERROR_NOTIFY_INSTALL_CHILD_SA_FAILED
            12 : ErrCodes['InstallChildPolFailed'],  # ERROR_NOTIFY_INSTALL_CHILD_POLICY_FAILED
            13 : ErrCodes['Generic'],                # ERROR_NOTIFY_UNIQUE_REPLACE
            14 : ErrCodes['Generic'],                # ERROR_NOTIFY_UNIQUE_KEEP
            15 : ErrCodes['VipFailure'],             # ERROR_NOTIFY_VIP_FAILURE
            16 : ErrCodes['AuthorizationFailed'],    # ERROR_NOTIFY_AUTHORIZATION_FAILED
            17 : ErrCodes['CertExpired'],            # ERROR_NOTIFY_CERT_EXPIRED
            18 : ErrCodes['CertRevoked'],            # ERROR_NOTIFY_CERT_REVOKED
            19 : ErrCodes['NoIssuerCert'],           # ERROR_NOTIFY_NO_ISSUER_CERT
            20 : ErrCodes['Generic'],                # ERROR_NOTIFY_RETRANSMIT_SEND
        }

        timeout = "%d" % args.connTimeout

        sa = OrderedDict()
        sa['child']   = "child"
        sa['timeout'] = timeout

        try:
            response = sess.initiate(sa)
            # Note: response must be iterated for the connection to succeed.
            for msg in response:
                # Values in the returned dict are in byte array and must be decoded to string.
                decmsg = { key: val.decode() for key, val in msg.items() }
                logger.Add("{group} | {level} | {ikesa-name} | {ikesa-uniqueid} | {msg}".format(**decmsg), level=1)
        except Exception as e:
        #
            # Recv from the error-notify socket and parse message
            # Message structure: 
            # struct error_notify_msg_t {
            #    /** message type */
            #    int type;
            #    /** string with an error description */
            #    char str[384];
            #    /** connection name, if known */
            #    char name[64];
            #    /** peer identity, if known */
            #    char id[256];
            #    /** peer address and port, if known */
            #    char ip[60];
            # } __attribute__((packed));
            try:
            #
                # Iterate and log all messages, but the error is extracted from the last one
                while True:
                #
                    data = errNotify.recv(768, socket.MSG_WAITALL)

                    if len(data) == 768:
                        # Unpack the message into an array. '>' is for big-endian.
                        unpack = struct.unpack(">i384s64s256s60s", data)
                        logger.Add("error-notify: Type(%i) | Conn(%s) | Id(%s) | Ip(%s) | %s" % (unpack[0], unpack[2].decode(), unpack[3].decode(), unpack[4].decode(), unpack[1].decode()))
                        try:
                            errcode = errCodesMap[unpack[0]]
                        except: 
                            errcode = ErrCodes['Generic']
                    else: 
                        if len(data) > 0:
                            logger.Add("error-notify: received %d bytes." % len(data))
                        break
                #
            #
            except: pass

            if (("CHILD_SA 'child' not established after %dms" % args.connTimeout) in ("%s" % e)):
                logger.Add("Reached %dms timeout: disconnecting..." % args.connTimeout)
                Disconnect(True, loggerInst=logger)
                errcode = ErrCodes['Timeout']
            failure = ("Can't connect to '%s': '%s'" % (connName, e), errcode)
        #

        errNotify.shutdown(socket.SHUT_RDWR)
        errNotify.close()
    else:
        failure = ("No connection found.", ErrCodes['Generic'])

    if failure is not None:
        Fail(failure[0], failure[1], logger=logger)
#

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Terminate VPN connection.
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def Disconnect(force=False, loggerInst=None):
#
    logger=None
    if loggerInst is not None:
        logger = loggerInst
    else:
        logger = Logger(None)

    if sess is not None:
    #
        connName = None

        for conn in sess.list_sas():
            if connName is None:
                connName = list(conn.keys())[0]

        if connName is not None:
        #
            logger.Add("Disconnecting IKE_SA '%s'..." % connName)
            sa = OrderedDict()
            sa['ike'] = connName
            if force:
                sa['force'] = 'yes'
            else:
                sa['timeout'] = '4000'
            sa['loglevel'] = '0'

            try:
                result = sess.terminate(sa)
                for msg in result:
                    # Values in the returned dict are in byte array and must be decoded to string.
                    decmsg = { key: val.decode() for key, val in msg.items() }
                    logger.Add("{group} | {level} | {ikesa-name} | {ikesa-uniqueid} | {msg}".format(**decmsg), level=1)
            except Exception as e:
                if not force:
                    logger.Add("Failed to disconnect from '%s', force disconnect..." % connName)
                    Disconnect(True, loggerInst=logger)
                else:
                    Fail("Failed to force disconnect from '%s': '%s'." % (connName, e), logger=logger)

            # If a nameserver was installed by strongswan, the original symlink
            # "/run/resolv.conf.override" -> "/run/connman/resolv.conf" should have been
            # replaced by a real file containing both the original connman config plus the
            # nameserver installed by strongswan. Remove this file and restore the original
            # symlink to make sure we are pointing to the actual version of connman's resolv.conf.
            try:
                if not os.path.islink("/run/resolv.conf.override"):
                    os.remove("/run/resolv.conf.override")
                    os.symlink("/run/connman/resolv.conf", "/run/resolv.conf.override")
            except Exception as e:
                logger.Add("Failed to restore symlink to connman's resolv.conf: '%s'" % e)

            return
        #
    #

    logger.Add("Already disconnected.")
#

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Retrieve private key files from passphrase.
# Here we retrieve the file(s) protected by the given passphrase by trying to decrypt all files
# with all supported formats (RSA / ECDSA / PKCS8, PEM / DER) with openssl command lines. Once
# we find a match, we try again but with a dummy passphrase: if this works, then it means that 
# the file is not passphrase-encrypted. If it does not, then the match is good and the file is 
# added to the returned list.
# openssl rsa -in FILE -passin pass:PASSPHRASE
# openssl rsa -in FILE -inform der -passin pass:PASSPHRASE
# openssl ec -in FILE -passin pass:PASSPHRASE
# openssl ec -in FILE -inform der -passin pass:PASSPHRASE
# openssl pkcs8 -in FILE -passin pass:PASSPHRASE
# openssl pkcs8 -in FILE -inform DER -passin pass:PASSPHRASE
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def MatchPrivateKeys(passphrase):
#
    privKeys = list()
    privDir  = '/etc/swanctl/private/'
    files    = [f for f in os.listdir(privDir) if os.path.isfile(privDir + f)]

    for f in files:
    #
        Log("Trying to decrypt '%s'..." % f, level=1)

        types = [ 'rsa', 'ec', 'pkcs8' ]

        for t in types: 
        #
            cmd     = "openssl %s -in '%s'" % (t, privDir + f)
            passStr = " -passin pass:'%s'" % passphrase
            match   = False

            Log("...Trying 'openssl %s'..." % t, level=1)

            if (Call(cmd + passStr) == 0):
                match = True
            else:
                cmd += " -inform der"
                Log("...Trying 'openssl %s -inform der'..." % t, level=1)
                if (Call(cmd + passStr) == 0):
                    match = True

            # If we were able to decrypt a file with the passphrase, try again with a dummy 
            # passphrase. If this works, it means that the file is not encrypted, if it does not,
            # it means that it is encrypted with the passphrase.
            if match:
            #
                cmd += " -passin pass:dqhbJrDBJV57ydct6DG0BV8Bi5vjbnncBN3nkjn2c"

                if (Call(cmd) != 0):
                    Log("Key '%s' matches passphrase." % f, level=1)
                    privKeys.append(f)
                else:
                    Log("Key '%s' is not encrypted." % f, level=1)

                # Go to next file.
                break
            #
        #
    #

    return privKeys            
#

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Retrieve pkcs12 files from passphrase.
# Detect pkcs12 file(s) in the pkcs12 directory that 
# can be decrypted with the given passphrase
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
def MatchPkcs12(passphrase):
#
    matches     = list()
    pkcs12Dir   = '/etc/swanctl/pkcs12/'
    files       = [f for f in os.listdir(pkcs12Dir) if os.path.isfile(pkcs12Dir + f)]

    for f in files:
    #
        Log("Trying to decrypt '%s'..." % f, level=1)

        cmd = "openssl pkcs12 -in '%s' -nokeys -passin pass:'%s'" % (pkcs12Dir + f, passphrase)

        if (Call(cmd) == 0):
            Log("PKCS#12 '%s' matches passphrase." % f, level=1)
            matches.append(f)
    #

    # Retry with legacy entryption if no p12 matches.
    if len(matches) == 0:
    #
        for f in files:
            Log("Trying to decrypt '%s' with legacy mode..." % f, level=1)

            cmd = "openssl pkcs12 -in '%s' -nokeys -passin pass:'%s' -legacy" % (pkcs12Dir + f, passphrase)

            if (Call(cmd) == 0):
                Log("PKCS#12 '%s' matches passphrase in legacy mode." % f, level=1)
                matches.append(f)
    #

    return matches
#

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Main code.
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Install user's config
if bool(args.configPkg):
    InstallConfigPackage(args.configPkg)
# Convert swanctl.json to swanctl.conf
elif bool(args.jsonFilePath):
    swanctlPath = "/dev/null"
    if bool(args.swanctlFilePath):
        swanctlPath = args.swanctlFilePath
        swanctlPath += '/swanctl.conf'
    GenSwanctlConf(args.jsonFilePath, swanctlPath)

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Load strongswan service and create VICI session if needed.
sess = None
if bool(args.secretType) \
    or bool(args.connect) \
    or bool(args.disconnect) \
    or bool(args.getSrcAddr) \
    or bool(args.removeSecrets) \
    or bool(args.removeVolSecrets) \
    or bool(args.waitDisconnect):
    sess = CreateSession(bool(args.connect))

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Load / remove secrets.

# Remove persisted/volatile secret installed on top of the config and clear
# credentials loaded in the daemon (secrets might still be present in the base config).
if bool(args.removeSecrets) or bool(args.removeVolSecrets):
#
    for t in secretTypes:
    #
        # Remove persisted secrets
        if args.removeSecrets:
            persistentConf = "/etc/swanctl/conf.d/%s.conf" % t
            if (os.path.isfile(persistentConf)):
                Log("Removing %s..." % persistentConf)
                os.remove(persistentConf)

        # Remove volatile secrets
        volatileConf = "/etc/swanctl/conf.d/volatile-%s.conf" % t

        if (os.path.islink(volatileConf)):
            targetConf = os.path.realpath(volatileConf)
            Log("Removing %s..." % volatileConf)
            os.remove(volatileConf)
            if (targetConf) and os.path.isfile(targetConf):
                 os.remove(targetConf)
    #

    if sess is not None:
        Log("Clear credentials...")
        sess.clear_creds()
        # Reload secrets that are specified in swanctl.conf (if any) 
        # and that are persisted in case only volatile secrets were removed.
        Log("Reload credentials...")    
        Call("swanctl --load-creds --noprompt", True)
#

# Load a volatile secret / store a persistent secret.
if bool(args.secretType):
#
    persistFile = "/etc/swanctl/conf.d/%s.conf" % args.secretType
    volatileLink = "/etc/swanctl/conf.d/volatile-%s.conf" % args.secretType

    # Make sure that an empty secret will be written as "secret = "" ", 
    # instead of just "secret = ", which is ignored
    if args.secret == "":
        args.secret = "\"\""

    if bool(args.persistSecret):
    #
        secretFile = open(persistFile, "w")

        # Remove volatile file for this type of secret if any 
        if os.path.islink(volatileLink):
            volatileFile = os.path.realpath(volatileLink)
            os.remove(volatileLink)
            if (volatileFile) and os.path.isfile(volatileFile):
                 os.remove(volatileFile)
    #
    else:
    #
        volatileDir  = "/var/run/strongswan/"
        volatileFile = "volatile-%s.conf" % args.secretType
        volatilePath = volatileDir + volatileFile

        if not os.path.isdir(volatileDir):
            MkDir(volatileDir)
        if not os.path.islink(volatileLink):
            os.symlink(volatilePath, volatileLink)

        secretFile = open(volatilePath, "w")

        # Remove persitent file for this type of secret if any.
        if os.path.isfile(persistFile):
            os.remove(persistFile)
    #

    if (args.secretType != "priv-key") and (args.secretType != "pkcs12"):
    #
        Log("Writing %s secret..." % ("persistent" if bool(args.persistSecret) else "volatile"))

        secret = {
            args.secretType : {
                "id"     : args.secretId,
                "secret" : args.secret
            }
        }

        WriteSecrets(secret, secretFile)
    #
    else:
    #
        # Special case for priv-key / pkcs12: must match the secret to its respective file (in case there would ever
        # be more than one private key / pkcs12 installed on a box). For the very unlikely event of having several 
        # private keys / pkcs12 encrypted with the same secret, build a list of such files and repeat the 
        # secret for each one of them.
        if args.secretType == "priv-key":
            files = MatchPrivateKeys(args.secret)
            prefix = "private"
        else:
            files = MatchPkcs12(args.secret)
            prefix = "pkcs12"

        if len(files) > 0:
            Log("Found files: %s" % files)

            count = 0
            secrets = OrderedDict()

            for f in files:
                secretName = "%s-%d" % (prefix, count)
                secrets[secretName] = { "file" : f, "secret" : args.secret }
                count = count + 1

            WriteSecrets(secrets, secretFile)
        else:
            Log("No file matching secret.")
            secretFile.close()
            os.remove(secretFile.name)
            if not args.persistSecret and os.path.islink(volatileLink):
                os.remove(volatileLink)
            Fail("No file installed in the '%s' folder matching the provided secret" % prefix, ErrCodes['InvalidSecret'])
    #

    secretFile.close()

    if sess is not None:
        Log("Reloading credentials...")
        Call("swanctl --load-creds --noprompt", True)
#

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Terminate VPN connection
if bool(args.disconnect):
    Disconnect()

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Initiate VPN connection
if bool(args.connect):
    Connect()

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Get IP address to use as source address to access VPN
if bool(args.getSrcAddr):
#
    if sess is None:
        Fail("Error: service is not loaded.")

    for conn in sess.list_sas():
    #
        for key in conn: 
        #
            for child in conn[key]['child-sas']:
            #
                ip = None

                try:
                    # Get the first listed local-ts
                    localTs = conn[key]['child-sas'][child]['local-ts'][0].decode()
                    # Remove any trailing network prefix length (ex: 192.168.0.1/24)
                    ip = localTs.split("/", 1)[0]
                except: pass
                
                if ip:
                    print(ip)
                    sys.exit(0)
            #
        #
    #

    Fail("Error: can't retrieve source IP address.")
#

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Collect strongswan related logs.
if bool(args.logDest):
#
    if not os.path.isdir(args.logDest):
        Fail("Error: '%s' is not a directory." % args.logDest)

    dst = args.logDest

    # Common commands always executed
    commandsCommon = [
        # systemctl-strongswan.log
        "systemctl status strongswan --no-pager > %s/systemctl-strongswan.log 2>&1" % dst,

        # journalctl-strongswan.log
        "journalctl -u strongswan --no-pager > %s/journalctl-strongswan.log 2>&1" % dst,

        # swanctl.conf: Replace secrets with '*****'
        "sed 's/secret[[:space:]]*=.*/secret = \*\*\*\*\*/g' /etc/swanctl/swanctl.conf > %s/swanctl.conf 2>&1" % dst,

        # Collect secret config files in conf.d, replace secrets with '*****'
        "for f in /etc/swanctl/conf.d/*.conf; do \
            [ -e $f ] || continue; \
            mkdir -p %s/conf.d; \
            name=$(basename $f); \
            sed 's/secret[[:space:]]*=.*/secret = \*\*\*\*\*/g' $f > %s/conf.d/$name; \
        done" % (dst, dst),

        # Collect logs generated by this script.
        "cp /var/log/strongswan-connect.log %s/strongswan-connect.log > /dev/null 2>&1" % dst,
        "cp /var/log/strongswan-install.log %s/strongswan-install.log > /dev/null 2>&1" % dst,

        # Collect resolv.conf. StrongSwan can add nameservers to the resolv.conf generated by connman.
        "cp /etc/resolv.conf %s/resolv.conf > /dev/null 2>&1" % dst,
    ]

    # Commands executed with service is running
    commandsWithService = [
        # swanctl.log
        "echo \"\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\" > %s/swanctl.log 2>&1" % dst,
        "echo \"swanctl --list-conns\" >> %s/swanctl.log 2>&1" % dst,
        "swanctl --list-conns >> %s/swanctl.log 2>&1" % dst,
        "echo \"\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\" >> %s/swanctl.log 2>&1" % dst,
        "echo \"swanctl --list-conns -P\" >> %s/swanctl.log 2>&1" % dst,
        "swanctl --list-conns -P >> %s/swanctl.log 2>&1" % dst,
        "echo \"\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\" >> %s/swanctl.log 2>&1" % dst,
        "echo \"swanctl --list-sas\" >> %s/swanctl.log 2>&1" % dst,
        "swanctl --list-sas >> %s/swanctl.log 2>&1" % dst,
        "echo \"\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\" >> %s/swanctl.log 2>&1" % dst,
        "echo \"swanctl --list-sas -P\" >> %s/swanctl.log 2>&1" % dst,
        "swanctl --list-sas -P >> %s/swanctl.log 2>&1" % dst,
        "echo \"\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\" >> %s/swanctl.log 2>&1" % dst,
        "echo \"swanctl --list-certs\" >> %s/swanctl.log 2>&1" % dst,
        "swanctl --list-certs >> %s/swanctl.log 2>&1" % dst,
        "echo \"\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\" >> %s/swanctl.log 2>&1" % dst,
        "echo \"swanctl --counters\" >> %s/swanctl.log 2>&1" % dst,
        "swanctl --counters >> %s/swanctl.log 2>&1" % dst,
        "echo \"\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\" >> %s/swanctl.log 2>&1" % dst,
        "echo \"swanctl --list-pols -P\" >> %s/swanctl.log 2>&1" % dst,
        "swanctl --list-pols -P >> %s/swanctl.log 2>&1" % dst,
        "echo \"\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\" >> %s/swanctl.log 2>&1" % dst,
        "echo \"swanctl --list-algs\" >> %s/swanctl.log 2>&1" % dst,
        "swanctl --list-algs >> %s/swanctl.log 2>&1" % dst,
        "echo \"\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\" >> %s/swanctl.log 2>&1" % dst,
        "echo \"swanctl --stat\" >> %s/swanctl.log" % dst,
        "swanctl --stat >> %s/swanctl.log 2>&1" % dst,
        "echo \"\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\" >> %s/swanctl.log 2>&1" % dst,
        "echo \"swanctl --stat -P\" >> %s/swanctl.log" % dst,
        "swanctl --stat -P >> %s/swanctl.log 2>&1" % dst,
        "echo \"\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\" >> %s/swanctl.log 2>&1" % dst,
        "echo \"find /etc/swanctl/\" >> %s/swanctl.log" % dst,
        "find /etc/swanctl/ >> %s/swanctl.log 2>&1" % dst,
        "echo \"\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\" >> %s/swanctl.log 2>&1" % dst,
        "echo \"find -L /etc/swanctl/{private,pubkey,x509,x509aa,x509ac,x509ca,x509crl,x509ocsp,pkcs12,conf.d} -type f -exec md5sum {} \;\" >> %s/swanctl.log" % dst,
        "find -L /etc/swanctl/{private,pubkey,x509,x509aa,x509ac,x509ca,x509crl,x509ocsp,pkcs12,conf.d} -type f -exec md5sum {} \; >> %s/swanctl.log 2>&1" % dst,

        # ip.log
        "echo \"\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\" > %s/ip.log 2>&1" % dst,
        "echo \"ip -s xfrm state\" >> %s/ip.log 2>&1" % dst,
        "ip -s xfrm state >> %s/ip.log 2>&1" % dst,
        "echo \"\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\" >> %s/ip.log 2>&1" % dst,
        "echo \"ip -s xfrm policy\" >> %s/ip.log 2>&1" % dst,
        "ip -s xfrm policy >> %s/ip.log 2>&1" % dst,
        "echo \"\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\" >> %s/ip.log 2>&1" % dst,
        "echo \"ip route show table 220\" >> %s/ip.log 2>&1" % dst,
        "ip route show table 220 >> %s/ip.log 2>&1" % dst,
    ]

    # Commands executed with service is not running
    commandsWithoutService = [
        # swanctl.log
        "echo \"\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\" > %s/swanctl.log 2>&1" % dst,
        "echo \"find /etc/swanctl/\" >> %s/swanctl.log" % dst,
        "find /etc/swanctl/ >> %s/swanctl.log 2>&1" % dst,
        "echo \"\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\" >> %s/swanctl.log 2>&1" % dst,
        "echo \"find -L /etc/swanctl/{private,pubkey,x509,x509aa,x509ac,x509ca,x509crl,x509ocsp,pkcs12,conf.d} -type f -exec md5sum {} \;\" >> %s/swanctl.log" % dst,
        "find -L /etc/swanctl/{private,pubkey,x509,x509aa,x509ac,x509ca,x509crl,x509ocsp,pkcs12,conf.d} -type f -exec md5sum {} \; >> %s/swanctl.log 2>&1" % dst,
    ]

    for cmd in commandsCommon:
        Call(cmd, True)

    if IsServiceLoaded():
        for cmd in commandsWithService: 
            Call(cmd, True)
    else:
        for cmd in commandsWithoutService:
            Call(cmd, True)  
#

# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# Wait for disconnection.
if bool(args.waitDisconnect) and (sess is not None):
#
    # To avoid a race condition where the disconnection would arrive between the calls to IsConnected() 
    # and to sess.listen(), in which case we would block in listen(), thinking we are still connected, 
    # we split the listen() function in two to be able to register to the event before calling IsConnected().
    # If the disconnection occurs between the register_events() and the listen_registered_events(), the event
    # will be received, so this eliminates the possible lock.
    sess.register_events(["child-updown"], True)

    try:
        if IsConnected():
            # listen() yields the events as they arrive
            events = sess.listen_registered_events()
            for name, event in events:
                # Make sure that this is a 'down' event. 'up' events start with ("up" : "yes"), 
                # down events don't have this entry. We should not receive any 'up' events here since
                # we checked that we were really connected before starting to listen events, this is 
                # to be 100% sure.
                if event.get("up", "") == "":
                    break
    finally:
        sess.register_events(["child-updown"], False)
#

