#!/bin/bash
# This retrieves the partition 1 device of a USB drive from the given USB bus ID by parsing the output 
# of "udevadm info" on all entries in /dev/disk/by-path containing '-usb-' in its name.
# TODO: Should we allow to select the partition?

if [ -z "$1" ]; then
    echo "Usage: $0 BUS_ID"
    exit 1
fi

# Ignore text before the first dash in the given BusID. Ex: we want to extract the "2.4" part in "1-2.4".
busid=$(echo $1 | cut -d'-' -f2)

# Note: we sort the output so that the partitions are listed in order (ex: *-part1, *-part2, *-part3)
for link in $(find /dev/disk/by-path -type l -name "*-usb-*" | sort); do

    # Extract the bus-id from the name of the symbolic link
    # Example: 
    #  1) pci-0000:00:14.0-usb-0:2.4:1.0-scsi-0:0:0:0
    #  2)                      0:2.4:1.0-scsi-0:0:0:0
    #  3)                      0:2.4:1.0
    #  4)                        2.4
    symlinkbusid=$(basename $link | awk -F '-usb-' '{print $2}' | cut -d'-' -f1 | cut -d':' -f2)
    
    udevadm_output=$(udevadm info -q property $link)

    devname=$(echo "$udevadm_output" | grep "DEVNAME=" | cut -d= -f2)
    devtype=$(echo "$udevadm_output" | grep "DEVTYPE=" | cut -d= -f2)
    idtype=$(echo "$udevadm_output" | grep "ID_TYPE=" | cut -d= -f2)
    subsystem=$(echo "$udevadm_output" | grep "SUBSYSTEM=" | cut -d= -f2)
    idbus=$(echo "$udevadm_output" | grep "ID_BUS=" | cut -d= -f2)
    idparttabletype=$(echo "$udevadm_output" | grep "ID_PART_TABLE_TYPE=" | cut -d= -f2)

    # Use the first listed partition (it should be part1 since we sort the output of find).
    # TODO: Validate filesystem type supported? (FAT32, NTFS, ext4, ...)
    if [ "$symlinkbusid" = "$busid" ] && [ "$idtype" = "disk" ] && [ "$subsystem" = "block" ] && [ "$idbus" = "usb" ]; then
        if [ "$devtype" = "disk" ] && [ "$idparttabletype" = "" ]; then
            # The device has no partition table, so don't look for a partition device (ex: mount /dev/sda instead of /dev/sda1).
            echo "$devname"
            exit 0
        elif [  "$devtype" = "partition" ]; then
            # The device has a partition table, return the first listed partition device.
            echo "$devname" 
            exit 0
        fi
    fi 

done

exit 1

