#!/bin/bash

verbose=0
disable=0
syslog_cfg_dir="/rw/config/syslog-ng"
syslog_cfg_file="$syslog_cfg_dir/syslog-ng-extra.conf"
server_cacert_path="$syslog_cfg_dir/ca.d"
client_cert_path="$syslog_cfg_dir/cert.d"
client_key_path="$syslog_cfg_dir/key.d"
server_cacert=""
server_ip=""
server_hostname=""
server_port="6514"
client_cert=""
client_key=""
ipv6=0
bsd=0
declare -a rules
facilities=("any" "kern" "user" "mail" "daemon" "auth" "syslog" "lpr" "news" "uucp" "cron" "authpriv" "ftp" "ntp" "security" "console")
levels=("any" "emerg" "alert" "crit" "err" "warning" "notice" "info" "debug")

function error()
{
    >&2 echo "$0: Error - $1"
    exit 1
}

function msg()
{
    if [ "$verbose" = "1" ]; then
        echo "$1"
    fi
}

function arraycontains()
{
    key="$1"
    shift

    while [[ $# -gt 0 ]]    
    do
        if [ "$1" = "$key" ]; then
            return 0
        else
            shift
        fi
    done

    return 1
}

echo "# syslog-ng extra config file generated by: \"$0 $@\"" > $syslog_cfg_file
echo "" >> $syslog_cfg_file

while [[ $# -gt 0 ]]
do
    case $1 in
        -h|--help)
            echo "Usage:"
            echo " $0 <--server-ip IPADDRESS || --server-hostname HOSTNAME> <--rule FACILITY LEVEL TAG> [--server-ca-cert cacert.pem] [--server-port PORT] [--ipv6] [--
            ] [--client-cert clientcert.pem clientkey.pem]"
            echo ""
            echo "Options:"
            echo " -i|--server-ip:        The IP address of the syslog server. Only one of --server-ip"
            echo "                        and --server-host must be used."
            echo " -n|--server-hostname:  The DNS hostname of the syslog server. Only one of --server-ip"
            echo "                        and --server-host must be used."
            echo " -r|--rule:             Define a rule for logs to be sent to the server. To define"
            echo "                        multiple rules, repeat this parameter for each rule."
            echo "                        FACILITY: One of the following syslog facilities:" 
            echo "                          ${facilities[*]}"
            echo "                        LEVEL: One of the following syslog log levels:"
            echo "                          ${levels[*]}"
            echo "                        MATCH: An application-defined syslog match (wildcard: \"any\")."
            echo " -s|--server-ca-cert:   The x509 CA root certificate that was used to generate the"
            echo "                        server's certificate. This enables TLS encryption."
            echo " -p|--server-port:      The TLS port used by the syslog server (default=6514)."
            echo " -6|--ipv6:             Use IPv6. This assumes that the server IPv6 address is"
            echo "                        provided with --server-ip. Link-local addresses are not"
            echo "                        supported."
            echo " -c|--client-cert:      The client x509 certificate and RSA key files. This enables"
            echo "                        mutual authentification."
            echo " -b|--bsd:              Use the legacy BSD syslog protocol (RFC3164) instead of the"
            echo "                        new IETF protocol (RFC5424)."
            echo " -d|--disable           Create the config file but the logging is disabled."
            echo " -v|--verbose:          Enable verbose output."
            exit 0
            ;;
        -s|--server-ca-cert)
            server_cacert="$2"
            shift; shift
            ;;
        -i|--server-ip)
            server_ip="$2"
            shift; shift
            ;;
        -n|--server-hostname)
            server_hostname="$2"
            shift; shift
            ;;
        -p|--server-port)
            server_port="$2"
            shift; shift
            ;;
        -c|--client-cert)
            client_cert="$2"
            client_key="$3"
            shift; shift; shift
            ;;
        -6|--ipv6)
            ipv6=1
            shift
            ;;
        -b|--bsd)
            bsd=1
            shift
            ;;
        -r|--rule)
            facility="$2"
            level="$3"
            match="$4"

            arraycontains "$facility" "${facilities[@]}"

            if [ "$?" != "0" ]; then
                error "invalid facility \"$facility\"."
            fi

            arraycontains "$level" "${levels[@]}"

            if [ "$?" != "0" ]; then
                error "invalid level \"$level\""
            fi

            rules+=("$facility $level $match")

            shift; shift; shift; shift
            ;;
        -d|--disable)
            exit 0
            ;;
        -v|--verbose)
            verbose=1
            shift
            ;;
        *)
            error "Unknown argument \"$1\""
            ;;
    esac
done

echo "options {" >> $syslog_cfg_file
echo "    keep-hostname(yes);" >> $syslog_cfg_file
echo "};" >> $syslog_cfg_file
echo "" >> $syslog_cfg_file

# Validate that at least one logging rule has been set.
if [ "${#rules[@]}" = "0" ]; then
    error "must provide at least one logging rule with \"--rule\""
fi

# Validate x509 server CA certificate.
if [ -f "$server_cacert" ]; then

    openssl x509 -in $server_cacert -text -noout > /dev/null 2>&1

    if [ "$?" != "0" ]; then
        error "invalid x509 server CA certificate \"$server_cacert\"."
    fi
fi

# Validate server address.
server_addr=""
if [ -n "$server_ip" ]; then

    if [ -n "$server_hostname" ]; then
        error "must provide only IP address or hostname for the server."
    fi

    if [ $ipv6 = 1 ]; then
        if [[ ! $server_ip =~ ^([0-9a-fA-F]{0,4}:){1,7}[0-9a-fA-F]{0,4}$ ]]; then
            error "invalid IPv6 server address \"$server_ip\"."
        fi
    else
        if [[ ! $server_ip =~ ^[0-9]+\.[0-9]+\.[0-9]+\.[0-9]+$ ]]; then
            error "invalid IPv4 server address \"$server_ip\"."
        fi
    fi

    server_addr="$server_ip"

elif [ -n "$server_hostname" ]; then
    server_addr="$server_hostname"
else
    error "must provide either an IP address or a hostname for the server."
fi

# Validate server port.
if [[ ! $server_port =~ ^[0-9]+$ ]] || [[ $server_port -gt 65535 ]]; then
    error "server port \"$server_port\" is invalid."
fi

# Validate the client certificate and key, if mutual authentification is used.
if [[ -n "$client_cert" || -n "$client_key" ]]; then

    # Validate that a valid server CA certificate has been set
    if [ -z "$server_cacert" ]; then
        error "client certificate provided but not server CA certificate."
    fi

    # Validate that both client certificate and key are set
    if [[ -z "$client_cert" || -z "$client_key" ]]; then
        error "missing client cert or key file (cert=\"$client_cert\", key=\"$client_key\")."
    fi

    # Validate that client certificate exists
    if [ ! -f "$client_cert" ]; then
        error "client certificate \"$client_cert\" does not exist."
    fi 

    # Validate that client key exists
    if [ ! -f "$client_key" ]; then
        error "client key \"$client_key\" does not exist."
    fi

    # Validate that the client certificate is a valid x509 certificate.
    openssl x509 -in $client_cert -text -noout > /dev/null 2>&1

    if [ "$?" != "0" ]; then
        error "invalid x509 client certificate \"$client_cert\""
    fi

    # Validate that the client key is a valid RSA key.
    openssl rsa -in $client_key -check > /dev/null 2>&1

    if [ "$?" != "0" ]; then
        error "invalid RSA client key \"$client_key\""
    fi

    # Validate that modulus of both certificate and key match
    client_cert_modulus=$(openssl x509 -noout -modulus -in $client_cert)
    client_cert_result=$?
    client_key_modulus=$(openssl rsa -noout -modulus -in $client_key)
    client_key_result=$?

    if [ "$client_cert_result"  != "0" ] || \
       [ "$client_key_result"   != "0" ] || \
       [ "$client_cert_modulus" != "$client_key_modulus" ]; 
    then
        error "modulus don't match for client certificate \"$client_cert\" and key \"$client_key\"."
    fi
fi

# Install the server CA root certificate.
if [ -n "$server_cacert" ]; then
    rm -f $server_cacert_path/*

    # Copy the server certificate to the syslog config folder
    cp $server_cacert $server_cacert_path

    if [ "$?" != "0" ]; then
        error "failed to copy server certificate \"$server_cacert\" to \"$server_cacert_path\"."
    fi

    # Make a link with the hash to the server certificate (ex: "bfdc1edf.0 -> servercert.pem")
    pwd=$PWD
    cd $server_cacert_path
    server_cacert_file=$(ls)
    ln -s $server_cacert_file $(openssl x509 -noout -hash -in $server_cacert_file).0
    ret=$?
    cd $pwd

    if [ "$ret" != "0" ]; then
        error "failed to create hash link to server certificate."
    fi
fi

# Install the client certificate and key files to the syslog config folders
if [[ -n "$client_cert" && -n "$client_key" ]]; then
    rm -f $client_cert_path/*
    rm -f $client_key_path/*

    cp $client_cert $client_cert_path

    if [ "$?" != "0" ]; then
        error "failed to copy client certificate \"$client_cert\" to \"$client_cert_path\"."
    fi

    # Replace the path with the syslog-ng config path
    client_cert=$client_cert_path/$(ls $client_cert_path)

    cp $client_key $client_key_path
    
    if [ "$?" != "0" ]; then
        error "failed to copy client key \"$client_key\" to \"$client_key_path\"."
    fi    

    # Replace the path with the syslog-ng config path
    client_key=$client_key_path/$(ls $client_key_path)
fi

# Write the localhost source to config file.
echo "# Localhost source:" >> $syslog_cfg_file
echo "source s_localhost { syslog(transport(udp) ip(127.0.0.1) port(514)); };" >> $syslog_cfg_file
echo "" >> $syslog_cfg_file

# Create the network destination rule.
ipvers="4" && [ $ipv6 = 1 ] && ipvers="6"

certfile=""
keyfile=""

if [ -n "$client_cert" ]; then
    certfile=" cert-file($client_cert)"
fi    

if [ -n "$client_key" ]; then
    keyfile=" key-file($client_key)"
fi

trans="tls"
tls="
    tls(ca-dir($server_cacert_path)$certfile$keyfile)"

auth="" && [ "$trans" = "tls" ] && auth=", one-way authentification" && [ -n "$client_cert" ] && auth=", mutual authentification"
msg "Network destination rule \"d_net\": \"$server_addr\" on port \"$server_port\", $trans$auth."
driver="network" && [ $bsd = 0 ]  && driver="syslog"

dest=\
"destination d_net {
  $driver(\"$server_addr\"
    port($server_port)
    transport($trans)
    ip-protocol($ipvers)
    flags($flags)$tls
  );
};"

# Write the network destination rule to config file.
echo "# Network destination:" >> $syslog_cfg_file
echo "$dest" >> $syslog_cfg_file
echo "" >> $syslog_cfg_file

# Create the filter rules and logging directives.
declare -a filters
declare -a logs
idx=0
for rule in "${rules[@]}"; do

    facility=$(echo $rule | cut -f1 -d' ')
    level=$(echo $rule | cut -f2 -d' ')
    match=$(echo $rule | cut -f3 -d' ')

    filter=""
    logfilter=""
    filtername=""
    if [[ "$facility" != "any" || "$level" != "any" || "$tag" != "any" ]]; then
    
        and=""
        facility_str="" 
        if [ "$facility" != "any" ]; then
            facility_str="facility($facility)"
            and=" and "
        fi

        level_str=""
        if [ "$level" != "any" ]; then
            level_str="${and}level($level..emerg)"
            and=" and "
        fi

        match_str=""
        if [ "$match" != "any" ]; then
            match_str="${and}match($match)"
        fi

        filtername="f_remotelog$idx"
        filter="filter $filtername { $facility_str$level_str$match_str };"
        logfilter="filter($filtername);"
    fi

    filters+=("$filter")
    
    logs+=("log { source(s_localhost); $logfilter destination(d_net); };")

    msg "Filter rule ($idx): facility($facility), level($level), tag($tag) - name = \"$filtername\"."

    idx=$((idx+1))
done

# Write the filter rules to config file.
echo "# Logging filters:" >> $syslog_cfg_file

for filter in "${filters[@]}"; do

    if [ -n "$filter" ]; then
        echo "$filter" >> $syslog_cfg_file
    fi
done

echo "" >> $syslog_cfg_file

# Write the logging directives to config file.
echo "# Logging directives:" >> $syslog_cfg_file

idx=0
for log in "${logs[@]}"; do

    filterstr="" && [[ $log == *"filter"* ]] && filterstr=" filter($idx) ->"
    msg "Logging directive ($idx): s_user ->${filterstr} d_net"
    echo "$log" >> $syslog_cfg_file
    idx=$((idx+1))
done

# Test the new config file for syntax errors.
syslog-ng -s > /dev/null 2>&1

if [ "$?" != "0" ]; then 
    error "bad syntax detected in the newly generated syslog-ng config file."
fi

systemctl restart syslog-ng@default.service

msg "Success."
exit 0

